/***************************************************************************
 *   Copyright (C) 2005 by Felipe Sateler                                  *
 *   fsateler@gmail.com                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA 02110-1301, USA.           *
 ***************************************************************************/

#include <qlayout.h>

#include <klocale.h>
#include <kglobal.h>
#include <kaboutdata.h>
#include <kparts/genericfactory.h>
#include <klistview.h>
#include <kpushbutton.h>
#include <qgroupbox.h>
#include <qvaluelist.h>
#include <kdesktopfile.h>
#include <kdirlister.h>
#include <kglobalsettings.h>
#include <qfile.h>
#include <qfileinfo.h>
#include <kmessagebox.h>

#include "appinfo.h"
#include "autostart.h"
#include "autostartwidget.h"
#include "version.h"

typedef KGenericFactory<Autostart, QWidget> AutostartFactory;
K_EXPORT_COMPONENT_FACTORY( kcm_Autostart, AutostartFactory("kcmautostart"))

extern "C" {
	KCModule* create_autostart(QWidget* parent, const char* name)
	{
		return new Autostart(parent,name);
	}
}

Autostart::Autostart(QWidget *parent, const char *name, const QStringList&)
	: KCModule(parent, name), myAboutData(0), widget(0), lister(0)
{
	myAboutData = new KAboutData("kcmautostart", I18N_NOOP("Autostart"), VERSION, 
								 "Module for configuring apps that autostart",
								 KAboutData::License_GPL_V2, NULL, NULL,
								 "fsateler@gmail.com");
	myAboutData->addAuthor("Felipe Sateler", "", "fsateler@gmail.com", "");
	
	QBoxLayout *l = new QVBoxLayout( this );
	l->setAutoAdd( TRUE );
	
	widget = new AutostartWidget(this, "AutostartWidget");
	connect( widget, SIGNAL(changed()), this, SLOT(widgetChanged()) );
	
	lister = new KDirLister();
	
	load();
};


Autostart::~Autostart()
{
	delete lister;
}

void Autostart::load()
{
	
	connect( lister, SIGNAL(completed()), this, SLOT(loadCompleted()) );
	connect( lister, SIGNAL(newItems(const KFileItemList& )), this, SLOT(loadItems(const KFileItemList& )) );
	
	lister->openURL( KGlobalSettings::autostartPath() );
}

void Autostart::save()
{
	// First remove all files in the directory
	connect( lister, SIGNAL(completed()), this, SLOT(removeCompleted()) );
	connect( lister, SIGNAL(newItems(const KFileItemList& )), this, SLOT(removeItems(const KFileItemList& )) );
	
	QString path = KGlobalSettings::autostartPath();
	lister->openURL( path );
	
	info = widget->save();
	QPtrListIterator<AppInfo> it(info);
	AppInfo* curr;
	while( ( curr = it.current() )!= NULL )
	{
		try
		{
			curr->save( path );
		}
		catch( char* err )
		{
			KMessageBox::error( this, err, i18n("Error") );
		}
		++it;
	}

	emit changed(true);
}


int Autostart::buttons()
{
	return /*KCModule::Default|*/KCModule::Apply|KCModule::Help;
}

QString Autostart::quickHelp() const
{
	return i18n("Configures applications that are automatically"
			" started during KDE startup");
}

void Autostart::loadCompleted()
{
	// Ensure no more signals are called
	disconnect( lister, SIGNAL(completed()), this, SLOT(loadCompleted()) );
	disconnect( lister, SIGNAL(newItems(const KFileItemList& )), this, SLOT(loadItems(const KFileItemList& )) );
	
	widget->load(info);
	info.clear();
}

void Autostart::removeCompleted()
{
	// Ensure no more signals are called
	disconnect( lister, SIGNAL(completed()), this, SLOT(removeCompleted()) );
	disconnect( lister, SIGNAL(newItems(const KFileItemList& )), this, SLOT(removeItems(const KFileItemList& )) );
}

void Autostart::loadItems(const KFileItemList& items)
{
	QPtrListIterator<KFileItem> it( items );
	KFileItem *curr;
	while( ( curr = it.current() ) != NULL ){
		++it;
		AppInfo* newInfo;
		if ( curr->isDir() )
			continue;
		newInfo = parseFile(curr);
		if( newInfo != NULL)
			info.append(newInfo);
	}
}

void Autostart::removeItems(const KFileItemList& items)
{
	QPtrListIterator<KFileItem> it( items );
	KFileItem *curr;
	while( ( curr = it.current() ) != NULL ){
		++it;
		if ( !curr->isFile() )
			continue;
		unlink( curr->localPath() );
	}
}

AppInfo* Autostart::parseFile(const KFileItem* item)
{
	AppInfo* app = NULL;
	if( item->isLink() )
	{
		app = new LinkAppInfo( item->name(), item->linkDest() );
	}
	else if( KDesktopFile::isDesktopFile( item->url().path() ) )
	{
		KDesktopFile file( item->url().path() );
		if( file.readType() == "Application" )
			app = new DesktopAppInfo( file.readName(), file.readEntry("Exec"), file.readComment() );
	}
	else
	{
		QFile input( item->url().path() );
		input.open(IO_ReadOnly);
		QByteArray data = input.readAll();
		input.close();
		
		mode_t perm = item->permissions();
		if( (perm & QFileInfo::ExeUser || perm & QFileInfo::ExeGroup || perm & QFileInfo::ExeOther) 
				   && item->mimeTypePtr()->is("text") )
		{
			app = new ScriptAppInfo( item->name(), data );
		}
		else
			app = new UnknownAppInfo( item->name(), data, perm );
	}
	
	return app;
}

void Autostart::widgetChanged()
{
	emit changed(true);
}

#include "autostart.moc"
