#!/bin/sh
# Copyright 2007 Google Inc. All Rights Reserved.

# Install firefox plugin for Google Destkop Linux

# Exit values
# 0: firefox plugin setup success
# 1: firefox is not found
# 2: invalid argument

PREFIX=/opt/google/desktop

#----------------------------------------------------------------------------
#   Common utility functions
DEBUG() {
  [ -z "${GDL_DEBUG_LEVEL}" ] && return 0;
  [ ${GDL_DEBUG_LEVEL} -ge $1 ] || return 0;
  shift
  echo  -e "$@" >&2
}

#--------------------------------------------------------------------------
#   Execute a command and print the command as debug information if necessary.
#   Usage:
#        ExecCmd <Debug Level> <command with args>
ExecCmd() {
  local debug_level=$1
  shift
  DEBUG $debug_level "$@"
  "$@"

  if [ $? -ne 0 ]; then
    echo "Failed to execute command: $@"
    exit 1
  fi
}

Help() {
  echo "$0 [-h] [-F DIR] [-G DIR] [--only-one] [--dry-run]"
}

#----------------------------------------------------------------------
IsFirefoxHome() {
  # Since the home of firefox must contain some specific files or dirs
  # (such as firefox, extensions, plugins, ), we use them as signature
  # to justify that $1 is the home of firefox
  has_firefox_bin=no
  for i in $1/*firefox* $1/*iceweasel*; do
    if [ -x $i ]; then
      has_firefox_bin=yes
      break
    fi
  done

  if [ x$has_firefox_bin != xyes ]; then
    return 1
  fi
  if [ ! -d $1/extensions ]; then
    return 1
  fi
  if [ ! -d $1/plugins ]; then
    return 1
  fi
  if [ ! -e $1/libxpcom.so ]; then
    return 1
  fi
  return 0
}

#-----------------------------------------------------------------------------------
DetectWithWhich() {
  local df_target_list
  local df_tmp
  local df_tmpdir


  df_target_list="`which -a firefox 2>/dev/null` \
                  `which -a mozilla-firefox 2>/dev/null` \
                  `which -a iceweasel 2>/dev/null`"

  for df_tmp in $df_target_list; do
    df_tmp=`readlink -f $df_tmp 2>/dev/null`
    if [ -n "$df_tmp" -a -e "$df_tmp" ]; then
      if IsFirefoxHome `dirname $df_tmp`; then
        echo `dirname $df_tmp`
      fi
      if [ `(grep '.*MOZ_DIST_BIN=.*' $df_tmp | wc -l) 2>/dev/null` = 1 ]; then
        df_tmpdir=`(grep '.*MOZ_DIST_BIN=.*' $df_tmp | cut -d'"' -f2 | cut -d'"' -f1) 2>/dev/null`
        if IsFirefoxHome $df_tmpdir; then
          echo $df_tmpdir
        fi
      fi
    fi
  done
}

#----------------------------------------------------------------------------------
DetectWithWhereis() {
  local df_target_dir_list
  local df_target_dir
  local df_tmp

  df_target_dir_list="`(whereis firefox | cut -d: -f2) 2>/dev/null` \
                      `(whereis mozilla-firefox | cut -d: -f2) 2>/dev/null` \
                      `(whereis iceweasel | cut -d: -f2) 2>/dev/null`"
  for df_target_dir in $df_target_dir_list; do
    df_target_dir=`readlink -f $df_target_dir 2>/dev/null`
    if [ -n "$df_target_dir" -a -e "$df_target_dir" ]; then
      if [ `(grep '.*MOZ_DIST_BIN=.*' $df_target_dir | wc -l) 2>/dev/null` = 1 ]; then
        df_tmpdir=`(grep '.*MOZ_DIST_BIN=.*' $df_target_dir | cut -d'"' -f2 | cut -d'"' -f1) 2>/dev/null`
        if IsFirefoxHome $df_tmpdir; then
          echo $df_tmpdir
        fi
      fi
      df_target_dir=`dirname $df_target_dir`
    fi
    if IsFirefoxHome $df_target_dir; then
      echo $df_target_dir
    fi
  done
}

#-----------------------------------------------------------------------------------------------
DetectInDefaultDir() {
  local df_tmp

  for df_tmp in `ls -d /usr/lib*/*firefox* \
                       /opt/*firefox* \
                       /usr/lib*/*iceweasel* 2> /dev/null`; do
    if IsFirefoxHome $df_tmp; then
      echo $df_tmp
    fi
  done
}

DetectFirefoxMain() {
  result=
  candidate_dirs="`DetectWithWhich` `DetectWithWhereis` `DetectInDefaultDir`"

  for dir in $candidate_dirs; do
    DEBUG 2 "handle candidate dir: $dir"
    if [ -z "$result" -o $df_show_all = yes ]; then
      result="$result $dir"
    fi
  done

  for dir in $result; do
    if [ ! -h $dir ]; then
      echo $dir
    fi
  done | LC_ALL="C" sort -u
}

df_show_all=yes
dry_run=no
INSTALLED_FILES_LOG=.gdl_installed_files
PLUGIN_DIR=plugin
FF_PLUGIN_SRCDIR=firefox
#--------------------------------------------------------------------
args=`getopt -u -o hF:G:D: --long help,only-one,dry-run -- "$@"`

if [ $? -ne 0 ]; then
  echo "Use $0 -h to get more information"
  exit 1
fi

set -- $args
while true; do
  case "$1" in
    -h|--help)        Help; exit 0 ;;
    -G)               PREFIX="$2"; shift 2 ;;
    -F)               FIREFOX_DIR="$2"; shift 2 ;;
    -D)               GDL_DEBUG_LEVEL=$2; shift 2 ;;
    --only-one)       df_show_all=no; shift 1 ;;
    --dry-run)        dry_run=yes; shift 1 ;;
    --)               shift; break ;;
    *)                echo "Invalid option $1, use $0 -h to get more infomation"
                      exit 1 ;;
  esac
done

if [ $dry_run = yes ]; then
  echo "Nothing will really be installed. Remove --dry-run to do real things."
fi

if [ -z "$FIREFOX_DIR" ]; then
  DEBUG 1 "Detecting where Firefox is installed ..."
  if [ `whoami` != "root" ]; then
    for dir in `(grep '^Path=' $HOME/.mozilla/firefox/profiles.ini | cut -d= -f2) 2>/dev/null`; do
      FIREFOX_DIR="$FIREFOX_DIR $HOME/.mozilla/firefox/$dir"
    done
  else
    FIREFOX_DIR="`DetectFirefoxMain`"
  fi
fi
DEBUG 2 "FIREFOX_DIR: $FIREFOX_DIR"

# Maybe command find should also be tried.
# But due to its high cost, we abandon it
if [ -z "$FIREFOX_DIR" ]; then
  exit 1
fi

DEBUG 1 "install firefox plugin now ..."
MSG=

for dir in `echo $FIREFOX_DIR | sed 's/:/ /g'`; do
  if [ ! -w $dir ]; then
    continue
  fi
  if [ $dry_run = no ]; then
    FF_PLUGIN_DIR="$dir/extensions/desktop@google.com"
    if [ ! -e "$dir/extensions" ]; then
      ExecCmd 2 mkdir -p "$dir/extensions"
    else
      ExecCmd 2 rm -rf $FF_PLUGIN_DIR
    fi
    if ln -sf "$PREFIX/$PLUGIN_DIR/$FF_PLUGIN_SRCDIR" "$FF_PLUGIN_DIR" 2>/dev/null; then
      MSG="$MSG\n\t$dir"
      # Only record the root directory.
      if [ `whoami` = "root" ]; then
        echo "[POST]$FF_PLUGIN_DIR" >> $PREFIX/$INSTALLED_FILES_LOG
      fi
    fi
  else
    MSG="$MSG\n\t$dir"
  fi
done
if [ -n "$MSG" ]; then
  DEBUG 1 "Firefox plugin has been installed to $MSG."
else
  DEBUG 1 "Firefox not found"
fi

exit 0
