#! /bin/sh
# Copyright 2007 Google Inc. All Rights Reserved.

# Install thunderbird plugin for Google Destkop Linux

# Exit values
# 0: thunderbird plugin setup success
# 1: thunderbird is not found
# 2: invalid argument

PREFIX=/opt/google/desktop

#----------------------------------------------------------------------------
#   Common utility functions
DEBUG() {
  [ -z "${GDL_DEBUG_LEVEL}" ] && return 0;
  [ ${GDL_DEBUG_LEVEL} -ge $1 ] || return 0;
  shift
  echo -e "$@" >&2
}

#--------------------------------------------------------------------------
#   Execute a command and print the command as debug information if necessary.
#   Usage:
#        ExecCmd <Debug Level> <command with args>
ExecCmd() {
  local debug_level=$1
  shift
  DEBUG $debug_level "$@"
  "$@"

  if [ $? -ne 0 ]; then
    echo "Failed to execute command: $@"
    exit 1
  fi
}

Help() {
  echo "$0 [-h] [-T DIR] [-G DIR] [--only-one] [--dry-run]"
}

#----------------------------------------------------------------------
IsThunderbirdHome() {
  # Since thunderbird home must have the following files or directories,
  # we take them as the signature of thunderbird home
  has_thunderbird_bin=no
  for i in $1/*thunderbird* $1/*icedove*; do
    if [ -x $i ]; then
      has_thunderbird_bin=yes
      break
    fi
  done

  if [ x$has_thunderbird_bin != xyes ]; then
    return 1
  fi
  if [ ! -d $1/extensions ]; then
    return 1
  fi
  if [ ! -e $1/libxpcom.so ]; then
    return 1
  fi

  return 0
}

#-----------------------------------------------------------------------------------
DetectWithWhich() {
  local dt_target_list
  local dt_tmp
  local dt_tmpdir

  dt_target_list="`which -a thunderbird 2>/dev/null` \
                  `which -a mozilla-thunderbird 2>/dev/null` \
                  `which -a icedove 2>/dev/null`"

  for dt_tmp in $dt_target_list; do
    dt_tmp=`readlink -f $dt_tmp 2>/dev/null`
    if [ -n "$dt_tmp" -a -e "$dt_tmp" ]; then
      if IsThunderbirdHome `dirname $dt_tmp`; then
        echo `dirname $dt_tmp`
      fi
      if [ `(grep '.*moz_libdir=.*' $dt_tmp | wc -l)2>/dev/null` = 1 ]; then
        dt_tmpdir=`(grep '.*moz_libdir=.*' $dt_tmp | cut -d= -f2) 2>/dev/null`
        if IsThunderbirdHome $dt_tmpdir; then
          echo $dt_tmpdir
        fi
      fi
    fi
  done
}

#----------------------------------------------------------------------------------
DetectWithWhereis() {
  local dt_target_dir_list
  local dt_target_dir
  local dt_tmp

  dt_target_dir_list="`(whereis thunderbird | cut -d: -f2) 2>/dev/null` \
                      `(whereis mozilla-thunderbird | cut -d: -f2) 2>/dev/null` \
                      `(whereis icedove | cut -d: -f2) 2>/dev/null`"
  for dt_target_dir in $dt_target_dir_list; do
    dt_target_dir=`readlink -f $dt_target_dir 2>/dev/null`
    if [ -n "dt_target_dir" -a -e "$dt_target_dir" ]; then
      if [ `(grep '.*moz_libdir=.*' $dt_target_dir | wc -l)2>/dev/null` = 1 ]; then
        dt_tmpdir=`(grep '.*moz_libdir=.*' $dt_target_dir | cut -d= -f2) 2>/dev/null`
        if IsThunderbirdHome $dt_tmpdir; then
          echo $dt_tmpdir
        fi
      fi
      dt_target_dir=`dirname $dt_target_dir`
    fi
    if IsThunderbirdHome $dt_target_dir; then
      echo $dt_target_dir
    fi
  done
}

#-----------------------------------------------------------------------------------------------
DetectInDefaultDir() {
  local dt_tmp

  for dt_tmp in `ls -d /usr/lib*/*thunderbird* \
                       /usr/lib*/*icedove* \
                       /opt/*thunderbird* 2>/dev/null`; do
    if IsThunderbirdHome $dt_tmp; then
      echo $dt_tmp
    fi
  done
}

DetectThunderbirdMain() {
  result=
  candidate_dirs="`DetectWithWhich` `DetectWithWhereis` `DetectInDefaultDir`"

  for dir in $candidate_dirs; do
    DEBUG 2 "handle candidate dir: $dir"
    if [ -z "$result" -o $dt_show_all = yes ]; then
      result="$result $dir"
    fi
  done

  for dir in $result; do
    if [ ! -h $dir ]; then
      echo $dir
    fi
  done | LC_ALL="C" sort -u
}

dt_show_all=yes
dry_run=no
INSTALLED_FILES_LOG=.gdl_installed_files
PLUGIN_DIR=plugin
TB_PLUGIN_SRCDIR=thunderbird
#--------------------------------------------------------------------
args=`getopt -u -o hT:G:D: --long help,only-one,dry-run -- "$@"`

if [ $? -ne 0 ]; then
  echo "Use $0 -h to get more information"
  exit 1
fi

set -- $args
while true; do
  case "$1" in
    -h|--help)        Help; exit 0 ;;
    -G)               PREFIX="$2"; shift 2 ;;
    -T)               THUNDERBIRD_DIR="$2"; shift 2 ;;
    -D)               GDL_DEBUG_LEVEL=$2; shift 2 ;;
    --only-one)       dt_show_all=no; shift 1 ;;
    --dry-run)        dry_run=yes; shift 1 ;;
    --)               shift; break ;;
    *)                echo "Invalid option $1, use $0 -h to get more infomation"
                      exit 1 ;;
  esac
done

if [ $dry_run = yes ]; then
  echo "Nothing will really be installed. Remove --dry-run to do real things."
fi

if [ -z "$THUNDERBIRD_DIR" ]; then
  DEBUG 1 "Detecting where Thunderbird is installed ..."
  if [ `whoami` != "root" ]; then
    for dir in `(grep '^Path=' $HOME/.mozilla-thunderbird/profiles.ini | cut -d= -f2) 2>/dev/null`; do
      THUNDERBIRD_DIR="$THUNDERBIRD_DIR $HOME/.mozilla-thunderbird/$dir"
    done
    for dir in `(grep '^Path=' $HOME/.thunderbird/profiles.ini | cut -d= -f2) 2>/dev/null`; do
      THUNDERBIRD_DIR="$THUNDERBIRD_DIR $HOME/.thunderbird/$dir"
    done
  else
    THUNDERBIRD_DIR="`DetectThunderbirdMain`"
  fi
fi
DEBUG 2 "THUNDERBIRD_DIR: $THUNDERBIRD_DIR"

# Maybe command find should also be tried.
# But due to its high cost, we abandon it
if [ -z "$THUNDERBIRD_DIR" ]; then
  exit 1
fi

DEBUG 1 "install thunderbird plugin now ..."
MSG=

for dir in `echo $THUNDERBIRD_DIR | sed 's/:/ /g'`; do
  if [ ! -w $dir ]; then
    continue
  fi
  if [ $dry_run = no ]; then
    TB_PLUGIN_DIR="$dir/extensions/desktop@google.com"
    if [ ! -e "$dir/extensions" ]; then
      ExecCmd 2 mkdir -p "$dir/extensions"
    else
      ExecCmd 2 rm -rf $TB_PLUGIN_DIR
    fi
    if ln -sf "$PREFIX/$PLUGIN_DIR/$TB_PLUGIN_SRCDIR" "$TB_PLUGIN_DIR" 2>/dev/null; then
      MSG="$MSG\n\t$dir"
      # Only record the root directory.
      if [ `whoami` = "root" ]; then
        echo "[POST]$TB_PLUGIN_DIR" >> $PREFIX/$INSTALLED_FILES_LOG
      fi
    fi
  else
    MSG="$MSG\n\t$dir"
  fi
done
if [ -n "$MSG" ]; then
  DEBUG 1 "Thunderbird plugin has been installed to $MSG."
else
  DEBUG 1 "Thunderbird not found"
fi

exit 0
