#!/bin/bash
# Copyright 2007 Google Inc. All Rights Reserved.
#
# Post install script for Google Desktop Linux
# DO NOT RUN THIS SCRIPT MANUALLY.

FF_INSTALLER='install_firefox_plugin.sh'
TB_INSTALLER='install_thunderbird_plugin.sh'

FF_PLUGIN_SRCDIR=firefox
TB_PLUGIN_SRCDIR=thunderbird

XDG_DIRECTORY_FILE="google-gdl.directory"
XDG_GDL_BOX_DESKTOP_FILE="google-gdl.desktop"
XDG_GDL_PREFERENCES_DESKTOP_FILE="google-gdl-preferences.desktop"
XDG_DESKTOP_MENU="xdg-desktop-menu"

BABY_SITTER="gdl_service"
XUI="gdl_box"
STATS="gdl_stats"
LAUNCH_CMD="gdlinux"

FIREFOX_DIR=
THUNDERBIRD_DIR=

INSTALLED_FILES_LOG=.gdl_installed_files

PREFIX="/opt/google/desktop"

XDG_DIR="xdg"
PLUGIN_DIR="plugin"
SHARELIBDIR="lib"
SUB_PLUGIN_DIR="components"
BIN_DIR="bin"

#----------------------------------------------------------------------------
#   Common utility functions included in all XDG wrapper scripts
DEBUG() {
  [ -z "${GDL_DEBUG_LEVEL}" ] && return 0;
  [ ${GDL_DEBUG_LEVEL} -ge $1 ] || return 0;
  shift
  echo "$@" >&2
}

#--------------------------------------------------------------------------
#   Execute a command and print the command as debug information if necessary.
#   Usage:
#        ExecCmd <Debug Level> <command with args>
ExecCmd() {
  local debug_level=$1
  shift
  DEBUG $debug_level "$@"
  "$@"

  if [ $? -ne 0 ]; then
    echo "Failed to execute command: $@"
    exit 1
  fi
}

#--------------------------------------------------------------------------
CheckFile() {
  if [ ! -e "$1" ]; then
    echo "File $1 is missing!"
    exit 1
  fi
}

#-------------------------------------------------------------------------
CheckFiles() {
  CheckFile $PREFIX/$INSTALLED_FILES_LOG
  CheckFile $PREFIX/$BIN_DIR/$FF_INSTALLER
  CheckFile $PREFIX/$BIN_DIR/$TB_INSTALLER
  CheckFile $PREFIX/$BIN_DIR/$LAUNCH_CMD
  CheckFile $PREFIX/$BIN_DIR/$BABY_SITTER
  CheckFile $PREFIX/$BIN_DIR/$XUI
  CheckFile $PREFIX/$BIN_DIR/$XDG_DESKTOP_MENU
  CheckFile $PREFIX/$PLUGIN_DIR/$FF_PLUGIN_SRCDIR/install.rdf
  CheckFile $PREFIX/$PLUGIN_DIR/$TB_PLUGIN_SRCDIR/install.rdf
  CheckFile $PREFIX/$XDG_DIR/$XDG_DIRECTORY_FILE
  CheckFile $PREFIX/$XDG_DIR/$XDG_GDL_BOX_DESKTOP_FILE
  CheckFile $PREFIX/$XDG_DIR/$XDG_GDL_PREFERENCES_DESKTOP_FILE
}

#----------------------------------------------------------------------
Help() {
  cat <<EOF
                Google(TM) Desktop Linux post installation tool


NAME

post_install.sh - script to do some post installation tasks after installing
Google(TM) Desktop Linux to the host system. It's intended to be used
internally by install.sh command.

SYNOPSIS

    post_install.sh [-h] [-G DIR|-F DIR|-T DIR]

DESCRIPTION

    The post_install.sh script is in charge of installing Google(TM)
    Desktop Linux's plugins of Mozilla Firefox and Thunderbird, and
    a desktop menu to the correct directories in host Linux system.

OPTIONS

    -h, --help
          Display this manual
    -G DIR
          Specify where the core of Google Desktop Linux was
          installed. If not specified, the default dir would be used
    -F DIR
          Specify where Mozilla Firefox was installed. Multiple dirs can be
          given when they are seperated be colon. If not specifed,
          efforts would be made to auto-detect Firefox root. Firefox plugin
          would not be installed if Firefox root is not specified and auto-
          detection fails
    -T DIR
          Specify where Mozilla Thunderbird was installed. If not specified,
          efforts would be made to auto-detect Thunderbird root. If not specified,
          efforts would be makde to auto-detect Thunderbird root. Thunderbird
          plugin would not be installed if Thunderbird root is not specified
          and auto-detection fails

EXAMPLES
    This script is intended to be used by install.sh internally, and shouldn't
    be invoked manually.

COPYRIGHT
    Copyright 2007 Google Inc. All Rights Reserved

EOF
}

#--------------------------------------------------------------------
InstallDesktopMenu() {
  $PREFIX/$BIN_DIR/$XDG_DESKTOP_MENU install \
    $PREFIX/$XDG_DIR/$XDG_DIRECTORY_FILE \
    $PREFIX/$XDG_DIR/$XDG_GDL_BOX_DESKTOP_FILE \
    $PREFIX/$XDG_DIR/$XDG_GDL_PREFERENCES_DESKTOP_FILE
}

#--------------------------------------------------------------------
SetSysLevelAutostart() {
  [ -n "$XDG_CONFIG_DIRS" ] || XDG_CONFIG_DIRS="/etc/xdg"

  # Make sure that XDG autostart directory is there
  for dir in `echo $XDG_CONFIG_DIRS | sed 's/:/ /g'`; do
    if [ -d "$dir" -a -w "$dir" ]; then
      ExecCmd 2 mkdir -p "$dir/autostart"
    fi
  done

  for dir in /usr/share/gnome /usr/share/kde /usr/share/kde3 \
             /opt/gnome/share /opt/kde/share /opt/kde3/share \
             /usr/share; do
    [ -d "$dir" ] && XDG_CONFIG_DIRS="$XDG_CONFIG_DIRS:$dir"
  done

  for dir in `echo $XDG_CONFIG_DIRS | sed 's/:/ /g'`; do
    if [ -d "$dir/autostart" -a -w "$dir/autostart" ]; then
      DEBUG 2 "system level autostart: $dir/autostart"
      ExecCmd 2 cp -f "$PREFIX/$XDG_DIR/$XDG_GDL_BOX_DESKTOP_FILE" "$dir/autostart/gdl_box.desktop"
      echo "[POST]$dir/autostart/gdl_box.desktop" >> $PREFIX/$INSTALLED_FILES_LOG
      chmod 644 "$dir/autostart/gdl_box.desktop"
    fi
  done
}

SetUsrLevelAutostart() {
  # Gnome Autostart
  [ -n "$XDG_CONFIG_HOME" ] || XDG_CONFIG_HOME=$HOME/.config
  DEBUG 2 "user level autostart: $XDG_CONFIG_HOME/autostart"
  ExecCmd 2 mkdir -p $XDG_CONFIG_HOME/autostart
  ExecCmd 2 cp -f $PREFIX/$XDG_DIR/$XDG_GDL_BOX_DESKTOP_FILE $XDG_CONFIG_HOME/autostart/gdl_box.desktop
  echo "[POST]$XDG_CONFIG_HOME/autostart/gdl_box.desktop" >> $PREFIX/$INSTALLED_FILES_LOG
  chmod 644 $XDG_CONFIG_HOME/autostart/gdl_box.desktop

  # KDE Autostart
  mkdir -p $HOME/.kde/Autostart

  echo "#! /bin/sh" > $HOME/.kde/Autostart/gdl_box
  echo "$PREFIX/$BIN_DIR/$LAUNCH_CMD" >> $HOME/.kde/Autostart/gdl_box
  echo "[POST]$HOME/.kde/Autostart/gdl_box" >> $PREFIX/$INSTALLED_FILES_LOG
  chmod 700 $HOME/.kde/Autostart/gdl_box
}

#--------------------------------------------------------------------
InstallInitScript() {
  [ -x $INIT_DIR/$INIT_SCRIPT ] || return 1
  [ `whoami` = "root" ] || return 1

  if [ -x /usr/sbin/update-rc.d ]; then
    # Debian like system.
    /usr/sbin/update-rc.d $INIT_SCRIPT defaults
  elif [ -x /sbin/insserv ]; then
    # SUSE like system.
    /sbin/insserv $INIT_SCRIPT
  elif [ -x /sbin/chkconfig ]; then
    # Redhat like system.
    /sbin/chkconfig --add $INIT_SCRIPT
  fi
}

#--------------------------------------------------------------------
ChangeSecureContextForShlib() {
  CHCON=`which chcon 2> /dev/null`
  if [ -n $CHCON ]; then
    for LIB in `ls $PREFIX/$PLUGIN_DIR/$FF_PLUGIN_SRCDIR/$SUB_PLUGIN_DIR/*.so`
    do
      $CHCON -t texrel_shlib_t $LIB 2> /dev/null || :
    done
    for LIB in `ls $PREFIX/$PLUGIN_DIR/$TB_PLUGIN_SRCDIR/$SUB_PLUGIN_DIR/*.so`
    do
      $CHCON -t texrel_shlib_t $LIB 2> /dev/null || :
    done
    for LIB in `ls $PREFIX/$SHARELIBDIR/*.so`
    do
      $CHCON -t texrel_shlib_t $LIB 2> /dev/null || :
    done
  else
    DEBUG 1 "this system has no command as chcon"
  fi
}

#--------------------------------------------------------------------
# Change to root directory to ensure that we have read permission for
# current directory.
cd /

args=`getopt -u -o hG:F:T:D: --long help -- "$@"`

if [ $? -ne 0 ]; then
  echo "Use $0 -h to get more information"
  exit 1
fi

set -- $args
while true; do
  case "$1" in
    -h|--help)        Help; exit 0 ;;
    -G)               PREFIX="$2"; shift 2 ;;
    -F)               FIREFOX_DIR="$2"; shift 2 ;;
    -T)               THUNDERBIRD_DIR="$2"; shift 2 ;;
    -D)               GDL_DEBUG_LEVEL=$2; shift 2 ;;
    --)               shift; break ;;
    *)                echo "Invalid option $1, use $0 -h to get more infomation"
                      exit 1 ;;
  esac
done

CheckFiles

ChangeSecureContextForShlib

# install firefox plugin
if [ -z $FIREFOX_DIR ]; then
  $PREFIX/$BIN_DIR/$FF_INSTALLER -G $PREFIX
else
  $PREFIX/$BIN_DIR/$FF_INSTALLER -F $FIREFOX_DIR -G $PREFIX
fi

# install thunderbird plugin
if [ -z $FIREFOX_DIR ]; then
  $PREFIX/$BIN_DIR/$TB_INSTALLER -G $PREFIX
else
  $PREFIX/$BIN_DIR/$TB_INSTALLER -T $THUNDERBIRD_DIR -G $PREFIX
fi

InstallDesktopMenu

if [ `whoami` != "root" ]; then
  SetUsrLevelAutostart
else
  SetSysLevelAutostart
fi

