#!/bin/bash
# Copyright 2007 Google Inc. All Rights Reserved.

# Pre uninstall script for Google Desktop Linux
# DO NOT RUN THIS SCRIPT MANUALLY.

PREFIX="/opt/google/desktop"

BIN_DIR="bin"

XUI="gdl_box"
BABY_SITTER="gdl_service"
STATS="gdl_stats"

XDG_DIRECTORY_FILE="google-gdl.directory"
XDG_GDL_BOX_DESKTOP_FILE="google-gdl.desktop"
XDG_GDL_PREFERENCES_DESKTOP_FILE="google-gdl-preferences.desktop"
XDG_DESKTOP_MENU="xdg-desktop-menu"

INSTALLED_FILES_LOG=.gdl_installed_files

#----------------------------------------------------------------------------
#   Common utility functions included in all XDG wrapper scripts
DEBUG() {
  [ -z "${GDL_DEBUG_LEVEL}" ] && return 0;
  [ ${GDL_DEBUG_LEVEL} -ge $1 ] || return 0;
  shift
  echo "$@" >&2
}

#--------------------------------------------------------------------------
#   Execute a command and print the command as debug information if necessary.
#   Usage:
#        ExecCmd <Debug Level> <command with args>
ExecCmd() {
  local debug_level=$1
  shift
  DEBUG $debug_level "$@"
  "$@"

  if [ $? -ne 0 ]; then
    echo "Failed to execute command: $@"
    exit 1
  fi
}
#--------------------------------------------------------------------------
CheckFile() {
  if [ ! -e "$1" ]; then
    echo "File $1 is missing!"
    exit 1
  fi
}

#-------------------------------------------------------------------------
# Check if all necessary files exist.
CheckFiles() {
  CheckFile $PREFIX/$BIN_DIR/$BABY_SITTER
  CheckFile $PREFIX/$BIN_DIR/$XUI
  CheckFile $PREFIX/$BIN_DIR/$XDG_DESKTOP_MENU
  CheckFile $PREFIX/$INSTALLED_FILES_LOG
}

# For future use
DeleteInitScript() {
  [ -x $INIT_DIR/$INIT_SCRIPT ] || return 1
  [ `whoami` = "root" ] || return 1

  if [ -x /usr/sbin/update-rc.d ]; then
    # Debian like system.
    /usr/sbin/update-rc.d -f $INIT_SCRIPT remove
  elif [ -x /sbin/insserv ]; then
    # SUSE like system.
    /sbin/insserv -r -f $INIT_SCRIPT
  elif [ -x /sbin/chkconfig ]; then
    # Redhat like system.
    /sbin/chkconfig --del $INIT_SCRIPT
  fi
}

DeletePostInstalledFiles() {
  for filedir in `(grep '\[POST\]' $PREFIX/$INSTALLED_FILES_LOG | sed -e 's/\[POST\]//g') 2>/dev/null`; do
    ExecCmd 1 rm -rf $filedir
  done

  ExecCmd 1 mv $PREFIX/$INSTALLED_FILES_LOG $PREFIX/$INSTALLED_FILES_LOG.old
  ExecCmd 1 grep -v '\[POST\]' $PREFIX/$INSTALLED_FILES_LOG.old > $PREFIX/$INSTALLED_FILES_LOG
  ExecCmd 1 rm -f $PREFIX/$INSTALLED_FILES_LOG.old
}

UninstallDesktopMenu() {
  $PREFIX/$BIN_DIR/$XDG_DESKTOP_MENU uninstall \
    $XDG_DIRECTORY_FILE \
    $XDG_GDL_BOX_DESKTOP_FILE \
    $XDG_GDL_PREFERENCES_DESKTOP_FILE
}

#----------------------------------------------------------------------

Help() {
  cat <<EOF
                Google(TM) Desktop Linux pre uninstallation tool


NAME

pre_uninstall.sh - script to do some pre uninstallation tasks before
uninstalling Google(TM) Desktop Linux from the host system.
It's intended to be used internally by uninstall.sh command.

SYNOPSIS

    pre_uninstall.sh [-h] [-G DIR]

DESCRIPTION

    The pre_uninstall.sh script is in charge of removing installed desktop
    files and various plugin files as well as init script.

OPTIONS

    -h, --help
          Display this manual

    -G DIR
          Specify where the core of Google Desktop Linux was
          installed. If not specified, the default dir would be used

EXAMPLES
    This script is intended to be used by uninstall.sh internally, and shouldn't
    be invoked manually.

COPYRIGHT

    Copyright 2007 Google Inc. All Rights Reserved

EOF
}


#--------------------------------------------------------------------
# Change to root directory to ensure that we have read permission for
# current directory.
cd /

args=`getopt -u -o hG:D: --long help -- "$@"`

if [ $? -ne 0 ]; then
  echo "Use $0 -h to get more information"
  exit 1
fi

set -- $args
while true; do
  case "$1" in
    -h|--help)        Help; exit 0 ;;
    -G)               PREFIX="$2"; shift 2 ;;
    -D)               GDL_DEBUG_LEVEL=$2; shift 2 ;;
    --)               shift; break ;;
    *)                echo "Invalid option $1, use $0 -h to get more infomation"
                      exit 1 ;;
  esac
done

if [ -z "$PREFIX" ]; then
  echo "-G must be used to specify the PREFIX!"
  exit 1
fi

ROOT_DIR=`grep '^\[ROOT_DIR\]' $PREFIX/$INSTALLED_FILES_LOG | sed -e 's/\[ROOT_DIR\]//g'`

if [ "$ROOT_DIR" != "$PREFIX" ]; then
  echo "PREFIX is differ than recorded ROOT_DIR!"
  exit 1
fi

CheckFiles
DeletePostInstalledFiles
UninstallDesktopMenu

