
/****************************************************************************
 *
 * All portions copyright their respective authors.  All rights reserved.
 *
 * This file is part of IVMan (ivm).
 *
 * This file may be distributed under the terms of the Q Public License
 * as defined by Troll Tech AS of Norway and appearing in the file
 * LICENSE.QPL included in the packaging of this file.
 * 
 * See http://www.troll.no/qpl for QPL licensing information.
 *
 * $Id: IvmConfigActions.c,v 1.33 2006/04/18 04:10:37 ro_han Exp $
 *****************************************************************************/

#include "common.h"
#include "IvmConfigActions.h"

#include <string.h>
#include <glib.h>
#include <libxml/tree.h>
#include <libxml/parser.h>
#include <libxml/xpath.h>
#include <libxml/xpathInternals.h>
#include <dbus/dbus.h>
#include <libhal.h>

IvmConfigActions *parseIvmConfigActions(char const *const path,
                                        char const *udi)
{
    gboolean mountable = ivm_device_is_mountable(udi);

    /* unfortunately we can't check if it's a video DVD until _after_ we
       mount it, so just assume it is for now */
    gboolean videodvd = TRUE;


    LIBXML_TEST_VERSION IvmConfigActions *ret = NULL;

    // DEBUG("parseIvmConfigActions %s", path);
    ret = (IvmConfigActions *) malloc(sizeof(IvmConfigActions));

    if (ret == NULL)
    {
        DEBUG(_("Out of memory!\n"));
        return NULL;
    }

    memset(ret, 0, sizeof(IvmConfigActions));

    ret->mount = FALSE;
    ret->need_mountpoint = FALSE;
    ret->checkOnInit = FALSE;
    ret->exec = NULL;
    int num_exec = 0;

    ret->execun = NULL;
    int num_execun = 0;

    ret->execdvd = NULL;
    int num_execdvd = 0;

    xmlDocPtr doc;
    xmlNodePtr cur;

    doc = xmlParseFile(path);
    if (doc == NULL)
    {
        DEBUG(_("Document not parsed successfully."));
        return NULL;
    }

    cur = xmlDocGetRootElement(doc);
    if (cur == NULL)
    {
        DEBUG(_("Document is empty!"));
        xmlFreeDoc(doc);
        return NULL;
    }
    if (xmlStrcmp(cur->name, (const xmlChar *) "ActionsConfig"))
    {
        DEBUG(_
              ("Incorrect document type, root node should have been %s"),
              "ActionsConfig");
        xmlFreeDoc(doc);
        return NULL;
    }
    cur = cur->children;
    while (cur != NULL)
    {
        // DEBUG("At XML node %s",cur->name);
        if (!xmlStrcmp(cur->name, (const xmlChar *) "Match"))
        {
            /* Check if our device matches - if so, continue iterating
               through siblings, otherwise jump back up to parent */

            gboolean matches = FALSE;
            xmlChar *value = 0;
            xmlChar *name = 0;
            
            if ( !udi )
                goto nextMatch;

            name = xmlGetProp(cur, "name");

            value = xmlGetProp(cur, "value");

            if (name == NULL)
            {
                DEBUG(_("Warning: %s XML tag encountered with missing or bad attributes, ignored\n"), cur->name);
                goto nextMatch;
            }

            if (!xmlStrcmp(name, (const xmlChar *) "*"))
            {
                matches = TRUE;
                goto nextMatch;
            }
            
            if (value == NULL)
            {
                DEBUG(_("Warning: %s XML tag encountered with missing or bad attributes, ignored\n"), cur->name);
                goto nextMatch;
            }


            else if (!xmlStrcmp(name, (const xmlChar *) "ivm.mountable"))
            {
                if (((!xmlStrcmp
                      (value, (const xmlChar *) "true"))
                     && mountable)
                    ||
                    ((!xmlStrcmp
                      (value, (const xmlChar *) "false")) && !mountable))
                    matches = TRUE;
            }

            else if (!xmlStrcmp(name, (const xmlChar *) "ivm.videodvd"))
            {
                if (((!xmlStrcmp
                      (value, (const xmlChar *) "true"))
                     && videodvd)
                    ||
                    ((!xmlStrcmp
                      (value, (const xmlChar *) "false")) && !videodvd))
                    matches = TRUE;
            }

            else if (!xmlStrncmp(name, (const xmlChar *) "hal.", 4))
                matches = hal_xml_property_matches(name, value, udi);
            else
                DEBUG(_("Invalid match name: %s"), (char *) name);

nextMatch:
            if (matches == TRUE && cur->children != NULL)
                cur = cur->children;
            else if (cur->next != NULL)
                cur = cur->next;
            else
                cur = cur->parent->next;

            if (value) xmlFree(value);
            if (name) xmlFree(name);
        }

        else if (!xmlStrcmp(cur->name, (const xmlChar *) "Option"))
        {
            xmlChar *name;

            name = xmlGetProp(cur, "name");

            xmlChar *value;

            value = xmlGetProp(cur, "value");
            
            if ((name == NULL) || (value == NULL))
                goto nextOption;

            if (!xmlStrcmp(name, (xmlChar *) "mount"))
            {
                if (!xmlStrcmp(value, (xmlChar *) "true"))
                    ret->mount = TRUE;
                else
                    ret->mount = FALSE;
            }

            else if (!xmlStrcmp(name, (xmlChar *) "exec"))
            {
                if (strstr(value, "$hal.volume.mount_point$") != NULL)
                    ret->need_mountpoint = TRUE;
                num_exec++;
                ret->exec = realloc(ret->exec,
                                    (num_exec + 1) * sizeof(char *));
                ret->exec[num_exec - 1] = (char *) xmlStrdup(value);
                ret->exec[num_exec] = NULL;
            }

            else if (!xmlStrcmp(name, (xmlChar *) "execun"))
            {
                if (strstr(value, "$hal.volume.mount_point$") != NULL)
                    ret->need_mountpoint = TRUE;
                num_execun++;
                ret->execun = realloc(ret->execun,
                                      (num_execun + 1) * sizeof(char *));
                ret->execun[num_execun - 1] = (char *) xmlStrdup(value);
                ret->execun[num_execun] = NULL;
            }

            else if (!xmlStrcmp(name, (xmlChar *) "execdvd"))
            {
                // always need mount point for execdvd
                ret->need_mountpoint = TRUE;

                num_execdvd++;
                ret->execdvd = realloc(ret->execdvd,
                                       (num_execdvd + 1) * sizeof(char *));
                ret->execdvd[num_execdvd - 1] = (char *) xmlStrdup(value);
                ret->execdvd[num_execdvd] = NULL;
            }
            
            else if (!xmlStrcmp(name, (xmlChar *) "checkOnInit"))
            {
                if (!xmlStrcmp(value, (xmlChar *) "true"))
                    ret->checkOnInit = TRUE;
                else
                    ret->checkOnInit = FALSE;
            }

nextOption:
            if (name) xmlFree(name);
            if (value) xmlFree(value);

            if (cur->next != NULL)
                cur = cur->next;
            else
                cur = cur->parent->next;
        }
        else if (cur->next != NULL)
            cur = cur->next;
        else
            cur = cur->parent->next;

    }
    xmlFreeDoc(doc);

    xmlCleanupParser();

    return ret;
}
