/***************************************************************************
 *   Copyright (C) 2006 by Dmitry Morozhnikov   *
 *   dmiceman@mail.ru   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "fuseisolib.h"

#include <stdio.h>
#include <fcntl.h>
#include <mntent.h>
#include <errno.h>

#include <kapplication.h>
#include <kuser.h>
#include <kprocess.h>
#include <kdirnotify_stub.h>
#include <qfile.h>
#include <qdir.h>
#include <qfileinfo.h>
#include <kmessagebox.h>
#include <klocale.h>
#include <kstandarddirs.h>
#include <kio/job.h>
#include <kio/jobclasses.h>

static const QString mtab_file = ".mtab.fuseiso";

FMountPoint::List FMountPoint::currentMountPoints() {
    FMountPoint::List result;
    
    int rc;

    KUser user;
    QString mtab_path = user.homeDir();
    mtab_path += "/";
    mtab_path += mtab_file;
    
    int fd = open(mtab_path.ascii(), O_RDWR | O_CREAT, 0644);
    if(fd < 0) {
        fprintf(stderr, "Can`t open mtab file %s: %s\n", mtab_path.ascii(), strerror(errno));
        return result;
    };
    rc = lockf(fd, F_LOCK, 0);
    if(rc != 0) {
        perror("Can`t lock mtab");
        return result;
    };
    FILE* mtab = setmntent(mtab_path.ascii(), "r");
    if(!mtab) {
        perror("Can`t open mtab");
        return result;
    };
    
    struct mntent* ent;
    while((ent = getmntent(mtab)) != NULL) {
        FMountPoint *mp = new FMountPoint();
        
        mp->m_mountedFrom = QFile::decodeName(ent->mnt_fsname);
        mp->m_mountPoint = QFile::decodeName(ent->mnt_dir);
        mp->m_mountType = QFile::decodeName(ent->mnt_type);
        
        result.append(mp);
    };
    
    endmntent(mtab);
    rc = lockf(fd, F_ULOCK, 0);
    if(rc != 0) {
        perror("Can`t unlock mtab");
        return result;
    };
    close(fd);
    
    return result;
};

FMountPoint FMountPoint::fromUrl(KURL url) {
    FMountPoint mp;

    FMountPoint::List mtab = FMountPoint::currentMountPoints();
    FMountPoint::List::iterator it;
    
    if(url.protocol() == "isomedia") {
        
        for(it = mtab.begin(); it != mtab.end(); ++it) {
            KURL to((*it)->mountPoint());
            QString fname = "/" + to.fileName();
            if(url.path().startsWith(fname)) {
                mp = **it;
                break;
            };
        };
        
    } else if(url.isLocalFile()) {
        QFileInfo fi(url.path());
        
        while(fi.exists() && fi.isReadable() && fi.isSymLink()) {
            QString fn = fi.readLink();
            if(!fn) {
                return mp; // empty
            };
            fi = QFileInfo(fn);
        };
        
        if(!fi.exists() || !fi.isReadable()) {
            return mp;
        };
     
        if(fi.isDir()) {
            
            for(it = mtab.begin(); it != mtab.end(); ++it) {
                if(url.path() == (*it)->mountPoint()) {
                    mp = **it;
                    break;
                };
            };
            
        } else {
            
            for(it = mtab.begin(); it != mtab.end(); ++it) {
                if(url.path() == (*it)->mountedFrom()) {
                    mp = **it;
                    break;
                };
            };
            
        };
        
    };
    
    return mp;
};

FuseisoLib::FuseisoLib(KURL media_dir_) {
    
    if(media_dir_.isEmpty()) {
    
        KUser user;
        media_dir = user.homeDir();
        media_dir.addPath("media");

    } else {
    
        media_dir = media_dir_;
    
    };
};

bool FuseisoLib::mount(KURL url, KURL& mount_point, QString& errstr) {

    if(!userInstall()) {
        return false;
    };
    
    if(mount_point.isEmpty()) {
        mount_point = FuseisoLib::suggestMountPoint(url);
    };

    KProcess proc;
    
    proc << "fuseiso";
    proc << "-p" << url.path() << mount_point.path();
    
    connect( &proc, SIGNAL( receivedStdout( KProcess *, char *, int ) ),
        this, SLOT( onReceivedStdout( KProcess *, char *, int ) ) );
    connect( &proc, SIGNAL( receivedStderr( KProcess *, char *, int ) ),
        this, SLOT( onReceivedStdout( KProcess *, char *, int ) ) );
    output = "";
    
    // block konqueror while fuseiso is not come to daemon state
    // may be this is not an excellent idea, but helps open 
    // mount point when it is really mounted
    proc.start(KProcess::Block, KProcess::AllOutput);
    
    if(proc.normalExit() && proc.exitStatus() == 0) {
        
        KDirNotify_stub notifier("*", "*");
        notifier.FilesAdded(KURL("isomedia:/"));
        notifier.FilesAdded(KURL("system:/isomedia/"));
        
        return true;
    } else {
        errstr = output;
        
        return false;
    };
};

bool FuseisoLib::umount(KURL url, QString &errstr) {

    KProcess proc;
    
    proc << "fusermount";
    proc << url.path() << "-u";
    
    connect( &proc, SIGNAL( receivedStdout( KProcess *, char *, int ) ),
        this, SLOT( onReceivedStdout( KProcess *, char *, int ) ) );
    connect( &proc, SIGNAL( receivedStderr( KProcess *, char *, int ) ),
        this, SLOT( onReceivedStdout( KProcess *, char *, int ) ) );
    output = "";
    
    // block konqueror while fuseiso is not come to daemon state
    // may be this is not an excellent idea, but helps open 
    // mount point when it is really mounted
    proc.start(KProcess::Block, KProcess::AllOutput);
    
    if(proc.normalExit() && proc.exitStatus() == 0) {
    
        KDirNotify_stub notifier("*", "*");
        notifier.FilesRemoved(KURL("isomedia:/" + url.fileName()));
        notifier.FilesRemoved(KURL("system:/isomedia/" + url.fileName()));
        
        return true;
    } else {
        errstr = output;
        
        return false;
    };
};

bool FuseisoLib::userInstall() {

    QDir media(media_dir.path());
    if(!media.exists()) {
        if(media.mkdir(media_dir.path(), true)) {
            KStandardDirs sdirs;
            KIO::file_copy(sdirs.findResource("data", "kfuseiso/media.directory"), 
                KURL(media_dir.path() + "/.directory"), -1, false, false, false);
            KMessageBox::information(0, 
                QString(i18n("Directory %1 created to hold image mount points")).arg(media_dir.path()));
            
            return true;
        } else {
            KMessageBox::error(0, 
                QString(i18n("Can`t create directory %1. Image file will not be mounted")).arg(media_dir.path()));
            return false;
        };
    };
    
    return true;
};

KURL FuseisoLib::suggestMountPoint(KURL url) {

    KURL mount_point(media_dir);
    
    QDir media(media_dir.path());
    int i = 0;
    while(1) {
        QString mount_dir = url.fileName();
        if(i) {
            mount_dir += QString("[%1]").arg(i);
        };
        if(!media.exists(mount_dir)) {
            mount_point.addPath(mount_dir);
            break;
        } else if(i > 100) {
            // something goes wrong
            mount_point.addPath("NULL");
            return mount_point;
        };
        i++;
    };
    
    return mount_point;
};

bool FuseisoLib::isMounted(KURL url, KURL& mount_point) {

    bool mounted = false;
    
    mtab = FMountPoint::currentMountPoints();
    for(FMountPoint::List::iterator mp = mtab.begin(); mp != mtab.end(); ++mp) {
        QString to = (*mp)->mountPoint();
        QString what = (*mp)->mountedFrom();
        if(urlcmp(url.path(), what, true, true)) {
            mount_point = KURL(to);
            mounted = isReallyMounted(mount_point, true);
            break;
        };
    };
    
    return mounted;
};

KURL FuseisoLib::findMountPoint(KURL url) {

    KURL mount_point;

    mtab = FMountPoint::currentMountPoints();
    for(FMountPoint::List::iterator mp = mtab.begin(); mp != mtab.end(); ++mp) {
        KURL to((*mp)->mountPoint());
        if(to.fileName() == url.fileName()) {
            mount_point = to;
            break;
        };
    };
    
    return mount_point;
};

bool FuseisoLib::isReallyMounted(KURL mount_point, bool cleanup) {
    bool mounted = false;
    
    KMountPoint::List mtab = KMountPoint::currentMountPoints();
    for(KMountPoint::List::iterator mp = mtab.begin(); mp != mtab.end(); ++mp) {
        QString to = (*mp)->mountPoint();
        QString what = (*mp)->mountedFrom();
        if(urlcmp(mount_point.path(), to, true, true) && what == "fuseiso") {
            mounted = true;
            break;
        };
    };
    
    if(mounted || !cleanup) {
        return mounted;
    } else if(!mounted && cleanup) {
        KUser user;
        QString mtab_path = user.homeDir();
        mtab_path += "/";
        mtab_path += mtab_file;
        
        int rc;
        
        char new_mtab_path[PATH_MAX];
        int fd = open(mtab_path.ascii(), O_RDWR | O_CREAT, 0644);
        if(fd < 0) {
            perror("Can`t open mtab");
            return false;
        };
        rc = lockf(fd, F_LOCK, 0);
        if(rc != 0) {
            perror("Can`t lock mtab");
            return false;
        };
        strncpy(new_mtab_path, mtab_path.ascii(), PATH_MAX - 16);
        new_mtab_path[PATH_MAX - 1] = 0;
        strcat(new_mtab_path, ".new");
        FILE* mtab = setmntent(mtab_path, "r");
        if(!mtab) {
            perror("Can`t open mtab");
            return false;
        };
        FILE* new_mtab = setmntent(new_mtab_path, "a+");
        if(!new_mtab) {
            perror("Can`t open new mtab");
            return false;
        };
        struct mntent* ent;
        while((ent = getmntent(mtab)) != NULL) {
            if(strcmp(ent->mnt_dir, mount_point.path().ascii()) == 0 &&
                strcmp(ent->mnt_type, "fuseiso") == 0) {
                // skip
            } else {
                rc = addmntent(new_mtab, ent);
                if(rc != 0) {
                    perror("Can`t add mtab entry");
                    return false;
                };
            };
        };
        endmntent(mtab);
        endmntent(new_mtab);
        rc = rename(new_mtab_path, mtab_path.ascii());
        if(rc != 0) {
            perror("Can`t rewrite mtab");
            return false;
        };
        rc = lockf(fd, F_ULOCK, 0);
        if(rc != 0) {
            perror("Can`t unlock mtab");
            return false;
        };
        close(fd);
        
        rc = rmdir(mount_point.path().ascii());
        if(rc != 0) {
            perror("Can`t delete mount point");
        };
        
        return false;
        
    } else {
        
        return mounted;
        
    };
};

void FuseisoLib::onReceivedStdout(KProcess *proc, char *buf, int len) {
    Q_UNUSED( proc );
    output += QString::fromLocal8Bit( buf, len );
};

#include "fuseisolib.moc"
