/***************************************************************************
 *   Copyright (C) 2006 by Dmitry Morozhnikov   *
 *   dmiceman@mail.ru   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

// heavily based on sources of ark_plugin from kdeaddons package

#include "iso_image_plugin.h"

#include <stdio.h>
#include <fcntl.h>

#include <kapplication.h>
#include <kstandarddirs.h>
#include <kaction.h>
#include <konq_popupmenu.h>
#include <kpopupmenu.h>
#include <kgenericfactory.h>
#include <kmessagebox.h>
#include <kmountpoint.h>
#include <kuser.h>
#include <kmdcodec.h>
#include <kprocess.h>
#include <krun.h>
#include <kdirnotify_stub.h>
#include <kdirnotify.h>

#include <qdir.h>

#include "fuseisolib.h"

typedef KGenericFactory<IsoImageMenu, KonqPopupMenu> IsoImageMenuFactory;
K_EXPORT_COMPONENT_FACTORY( libiso_image_plugin, IsoImageMenuFactory("iso_image_plugin") )

IsoImageMenu::IsoImageMenu( KonqPopupMenu * popupmenu, const char *name, const QStringList& /* list */ )
                : KonqPopupMenuPlugin( popupmenu, name), flib()
{
    if(KStandardDirs::findExe( "fuseiso" ).isNull() || KStandardDirs::findExe( "fusermount" ).isNull()) {
        return;
    };
    
    KGlobal::locale()->insertCatalogue("iso_image_plugin");
    
    KFileItemList itemList = popupmenu->fileItemList();
    if(itemList.count() != 1) {
        return;
    };
    
    KURL file_url = itemList.first()->url();
    
    // kdDebug() << " there at all with " << file_url.path() << endl;
    
    if(itemList.first()->mimetype() == "inode/x-iso-image-mounted") {
        KURL mount_point_url = flib.findMountPoint(file_url);
        if(mount_point_url.isEmpty()) {
            return;
        };
        
        mount_point = mount_point_url.path();
        
        KAction* action;
        
        action = new KAction(i18n("Umount image"), "cdtrack", 0, this,
            SLOT(slotUmount()), actionCollection(), "iso_image_umount_menu");
        addAction(action);
    } else if(file_url.isLocalFile()) { // local file to mount/umount/browse
        image_file = file_url.path();
        
        if(!image_file) {
            return;
        };
        
        if(!checkImage(image_file, label)) {
            return;
        };
        
        KURL mount_point_url;
        mounted = flib.isMounted(file_url, mount_point_url);
        if(mounted) {
            mount_point = mount_point_url.path();
        } else {
            mount_point_url = flib.suggestMountPoint(file_url);
            mount_point = mount_point_url.path();
        };
        
        KAction* action;
        
        if(mounted) {
            action = new KAction(i18n("Browse..."), "cdtrack", 0, this,
                SLOT(slotBrowse()), actionCollection(), "iso_image_browse_menu");
            addAction(action);
            action = new KAction(i18n("Umount image"), "cdtrack", 0, this,
                SLOT(slotUmount()), actionCollection(), "iso_image_umount_menu");
            addAction(action);
        } else {
            action = new KAction(i18n("Mount image"), "cdtrack", 0, this,
                SLOT(slotMount()), actionCollection(), "iso_image_mount_menu");
            addAction(action);
            action = new KAction(i18n("Mount image and browse"), "cdtrack", 0, this,
                SLOT(slotMountAndBrowse()), actionCollection(), "iso_image_mount_browse_menu");
            addAction(action);
        };
    };
};

IsoImageMenu::~IsoImageMenu()
{
};

void IsoImageMenu::slotMount() {

    KURL mount_point_url(mount_point);
    QString errstr;
    if(!flib.mount(KURL(image_file), mount_point_url, errstr)) {
        KMessageBox::error(0, 
            QString(i18n("Error while mounting image:\n%1")).arg(errstr));
    };

};

void IsoImageMenu::slotMountAndBrowse() {
    
    KURL mount_point_url(mount_point);
    QString errstr;
    if(flib.mount(KURL(image_file), mount_point_url, errstr)) {
        slotBrowse();
    } else {
        KMessageBox::error(0, 
            QString(i18n("Error while mounting image:\n%1")).arg(errstr));
    };
    
};

void IsoImageMenu::slotUmount() {
    
    QString errstr;
    if(!flib.umount(KURL(mount_point), errstr)) {
        KMessageBox::error(0, 
            QString(i18n("Error while unmounting image:\n%1")).arg(errstr));
    };

};

void IsoImageMenu::slotBrowse() {
    KURL mp_url(mount_point);
    new KRun(KURL(mp_url));
};

/// @todo move to fuseisolib, make use of kde file access functions
bool IsoImageMenu::checkImage(const QString fname, QString& label) {
    
    int vd_num = 0;
    int fd;
    struct iso_volume_descriptor vd;
    
    memset(&vd, 0, sizeof(struct iso_volume_descriptor));
    
    // defaults for iso
    int block_size = 2048;
    int block_offset = 0;
    int file_offset = 0;
    
    fd = open(fname.ascii(), O_RDONLY | O_LARGEFILE);
    if(fd < 0) {
        return false;
    };
    
    enum {
        IDOFF_ISO_2048 = 2048 * 16,
        IDOFF_MODE1_2352 = 2352 * 16 + 16,
        IDOFF_MODE2_2352_RAW = 2352 * 16,
        IDOFF_MODE2_2352 = 2352 * 16 + 24,
        IDOFF_MODE2_2336 = 2336 * 16 + 16,
        IDOFF_NRG = 2048 * 16 + 307200,
    };
    int iso_offsets[] = {IDOFF_ISO_2048, IDOFF_MODE2_2336, IDOFF_MODE2_2352_RAW, IDOFF_NRG};
    // try to find CD001 identifier
    int i;
    for(i = 0; i < 4; i++) {
        if(lseek(fd, iso_offsets[i], SEEK_SET) == -1) {
            kdWarning() << "can`t lseek() to next possible data start position; is it really supported file?" << endl;
            goto out;
        };
        ssize_t size = read(fd, &vd, sizeof(struct iso_volume_descriptor));
        if(size != sizeof(struct iso_volume_descriptor)) {
            kdWarning() << "only " << size << " bytes read from position " << iso_offsets[i] << 
                ", " << sizeof(struct iso_volume_descriptor) << " required; is it really supported file?" << endl;
            goto out;
        };
        if(strncmp("CD001", vd.id, 5) == 0) {
            // found CD001!
            // fill context with information about block size and block offsets
            switch(iso_offsets[i]) {
                case IDOFF_ISO_2048:
                    // normal iso file
                    // use defaults
                    break;
                case IDOFF_MODE2_2352_RAW:
                    block_size = 2352;
                    break;
                case IDOFF_MODE2_2336:
                    block_size = 2336;
                    block_offset = 16;
                    break;
                case IDOFF_NRG:
                    file_offset = 307200;
                    break;
                default:
                    break;
            };
            break;
        } else if(strncmp("CD001", vd.id + 16, 5) == 0) {
            block_size = 2352;
            block_offset = 16;
            break;
        } else if(strncmp("CD001", vd.id + 24, 5) == 0) {
            block_size = 2352;
            block_offset = 24;
            break;
        };
    };
    
    while(1) {
        if(lseek(fd, block_size * (16 + vd_num) + block_offset + file_offset, SEEK_SET) == -1) {
            kdWarning() << "can`t lseek() to next volume descriptor" << endl;
            goto out;
        };
        ssize_t size = read(fd, &vd, sizeof(struct iso_volume_descriptor));
        if(size != sizeof(struct iso_volume_descriptor)) {
            kdWarning() << "only " << size << " bytes read from volume descriptor " << vd_num << 
                ", " << sizeof(struct iso_volume_descriptor) << " required" << endl;
            goto out;
        };
        
        int vd_type = isonum_711((unsigned char *)vd.type);
        
        if(strncmp("CD001", vd.id, 5) != 0) {
            if(vd_num > 16) {
                // no more trying
                kdWarning() << "wrong standard identifier in volume descriptor " << vd_num << ", exiting.." << endl;
                goto out;
            } else {
                // try to continue
                kdWarning() << "wrong standard identifier in volume descriptor " << vd_num << ", skipping.." << endl;
            };
        } else {
            struct iso_primary_descriptor* pd = (struct iso_primary_descriptor*) &vd;
            switch(vd_type) {
                case ISO_VD_PRIMARY:
                    // that is all we need
                    
                    label = QString::fromAscii(pd->volume_id, 32).stripWhiteSpace();
                    
                    close(fd);
                    return true;
                    
                    break;
                
                case ISO_VD_SUPPLEMENTARY:
                    // not interesting
                    break;
                
                case 0:
                    // boot record, not intresting..
                    break;
                
                case ISO_VD_END:
                    goto out;
                    break;
                    
                default:
                    kdWarning() << "unsupported volume descriptor type " << vd_type << ", vd_num " << vd_num << endl;
                    break;
            };
        };
        
        vd_num += 1;
    };

out:
    
    close(fd);
    
    return false;
};

#include "iso_image_plugin.moc"
