/***************************************************************************
 *   Copyright (C) 2006 by Dmitry Morozhnikov   *
 *   dmiceman@mail.ru   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


#include <config.h>
#include "kfile_iso_image.h"

#include <kgenericfactory.h>
#include <kfile.h>

#include <qdatetime.h>

#include <stdio.h>
#include <string.h>
#include <fcntl.h>

#include "fuseisolib.h"

enum {
    IMAGE_ISO,
    IMAGE_MODE1,
    IMAGE_MODE2,
    IMAGE_MODE2_RAW, // can it contain iso9660?
    IMAGE_MODE2_2336, // can it contain iso9660?
    IMAGE_NRG,
};

int read_volume_descriptor(const char* imagefile, struct iso_volume_descriptor* vd) {
    
    int mode = 0;
    int vd_num = 0;
    int fd;
    
    // defaults for iso
    int block_size = 2048;
    int block_offset = 0;
    int file_offset = 0;
    
    fd = open(imagefile, O_RDONLY | O_LARGEFILE);
    if(fd < 0) {
        return -1;
    };
    
    enum {
        IDOFF_ISO_2048 = 2048 * 16,
        IDOFF_MODE1_2352 = 2352 * 16 + 16,
        IDOFF_MODE2_2352_RAW = 2352 * 16,
        IDOFF_MODE2_2352 = 2352 * 16 + 24,
        IDOFF_MODE2_2336 = 2336 * 16 + 16,
        IDOFF_NRG = 2048 * 16 + 307200,
    };
    int iso_offsets[] = {IDOFF_ISO_2048, IDOFF_MODE2_2336, IDOFF_MODE2_2352_RAW, IDOFF_NRG};
    // try to find CD001 identifier
    int i;
    for(i = 0; i < 4; i++) {
        if(lseek(fd, iso_offsets[i], SEEK_SET) == -1) {
            kdWarning() << "can`t lseek() to next possible data start position; is it really supported file?" << endl;
            goto out;
        };
        ssize_t size = read(fd, vd, sizeof(struct iso_volume_descriptor));
        if(size != sizeof(struct iso_volume_descriptor)) {
            kdWarning() << "only " << size << " bytes read from position " << iso_offsets[i] << 
                ", " << sizeof(struct iso_volume_descriptor) << " required; is it really supported file?" << endl;
            goto out;
        };
        if(strncmp("CD001", vd->id, 5) == 0) {
            // found CD001!
            // fill context with information about block size and block offsets
            switch(iso_offsets[i]) {
                case IDOFF_ISO_2048:
                    // normal iso file
                    // use defaults
                    break;
                case IDOFF_MODE2_2352_RAW:
                    block_size = 2352;
                    break;
                case IDOFF_MODE2_2336:
                    block_size = 2336;
                    block_offset = 16;
                    break;
                case IDOFF_NRG:
                    file_offset = 307200;
                    break;
                default:
                    break;
            };
            break;
        } else if(strncmp("CD001", vd->id + 16, 5) == 0) {
            block_size = 2352;
            block_offset = 16;
            break;
        } else if(strncmp("CD001", vd->id + 24, 5) == 0) {
            block_size = 2352;
            block_offset = 24;
            break;
        };
    };
    
    if(file_offset == 307200) {
        mode = IMAGE_NRG;
    } else if(block_size == 2048) {
        mode = IMAGE_ISO;
    } else if(block_size == 2352 && block_offset == 0) {
        mode = IMAGE_MODE2_RAW;
    } else if(block_size == 2352 && block_offset == 16) {
        mode = IMAGE_MODE1;
    } else if(block_size == 2352 && block_offset == 24) {
        mode = IMAGE_MODE2;
    } else if(block_size == 2336 && block_offset == 16) {
        mode = IMAGE_MODE2_2336;
    } else {
        mode = 0;
    };
    
    while(1) {
        if(lseek(fd, block_size * (16 + vd_num) + block_offset + file_offset, SEEK_SET) == -1) {
            kdWarning() << "can`t lseek() to next volume descriptor" << endl;
            goto out;
        };
        ssize_t size = read(fd, vd, sizeof(struct iso_volume_descriptor));
        if(size != sizeof(struct iso_volume_descriptor)) {
            kdWarning() << "only " << size << " bytes read from volume descriptor " << vd_num << 
                ", " << sizeof(struct iso_volume_descriptor) << " required" << endl;
            goto out;
        };
        
        int vd_type = isonum_711((unsigned char *)vd->type);
        
        if(strncmp("CD001", vd->id, 5) != 0) {
            if(vd_num > 16) {
                // no more trying
                kdWarning() << "wrong standard identifier in volume descriptor " << vd_num << ", exiting.." << endl;
                goto out;
            } else {
                // try to continue
                kdWarning() << "wrong standard identifier in volume descriptor " << vd_num << ", skipping.." << endl;
            };
        } else {
            switch(vd_type) {
                case ISO_VD_PRIMARY:
                    // that is all we need
                    
                    close(fd);
                    return mode;
                    
                    break;
                
                case ISO_VD_SUPPLEMENTARY:
                    // not interesting
                    break;
                
                case 0:
                    // boot record, not intresting..
                    break;
                
                case ISO_VD_END:
                    goto out;
                    break;
                    
                default:
                    kdWarning() << "unsupported volume descriptor type " << vd_type << ", vd_num " << vd_num << endl;
                    break;
            };
        };
        
        vd_num += 1;
    };

out:
    
    close(fd);
    
    return -1;
};

typedef KGenericFactory<IsoImagePlugin> IsoImageFactory;

K_EXPORT_COMPONENT_FACTORY(kfile_iso_image, IsoImageFactory( "kfile_iso_image" ))

IsoImagePlugin::IsoImagePlugin(QObject *parent, const char *name,
                       const QStringList &args)
    : KFilePlugin(parent, name, args)
{
    addMimeType( "application/x-iso-image" );
    addMimeType( "inode/x-iso-image-mounted" );
}

void IsoImagePlugin::addMimeType(QString mime_type) {
    KFileMimeTypeInfo* info = addMimeTypeInfo(mime_type);

    KFileMimeTypeInfo::GroupInfo* group = 0L;
    group = addGroupInfo(info, "IsoImageInfo", i18n("Disk image information"));

    KFileMimeTypeInfo::ItemInfo* item;

    item = addItemInfo(group, "Type", i18n("Image type"), QVariant::String);
    item = addItemInfo(group, "SystemId", i18n("System identifier"), QVariant::String);
    item = addItemInfo(group, "VolumeId", i18n("Volume identifier"), QVariant::String);
    setHint(item, KFileMimeTypeInfo::Name);
    item = addItemInfo(group, "VolumeSetId", i18n("Volume set identifier"), QVariant::String);
    setAttributes(item, KFileMimeTypeInfo::MultiLine | KFileMimeTypeInfo::SqueezeText);
    setHint(item, KFileMimeTypeInfo::Description);
    item = addItemInfo(group, "PublisherId", i18n("Publisher identifier"), QVariant::String);
    setAttributes(item, KFileMimeTypeInfo::MultiLine | KFileMimeTypeInfo::SqueezeText);
    setHint(item, KFileMimeTypeInfo::Author);
    item = addItemInfo(group, "PreparerId", i18n("Data preparer identifier"), QVariant::String);
    setAttributes(item, KFileMimeTypeInfo::MultiLine | KFileMimeTypeInfo::SqueezeText);
    item = addItemInfo(group, "ApplicationId", i18n("Application identifier"), QVariant::String);
    setAttributes(item, KFileMimeTypeInfo::MultiLine | KFileMimeTypeInfo::SqueezeText);
    item = addItemInfo(group, "CopyrightFileId", i18n("Copyright file identifier"), QVariant::String);
    item = addItemInfo(group, "AbstractFileId", i18n("Abstract file identifier"), QVariant::String);
    item = addItemInfo(group, "BibliographicFileId", i18n("Bibliographic file identifier"), QVariant::String);
    item = addItemInfo(group, "CreationDate", i18n("Volume creation date and time"), QVariant::String);
    item = addItemInfo(group, "ModificationDate", i18n("Volume modification date and time"), QVariant::String);
    item = addItemInfo(group, "ExpirationDate", i18n("Volume expiration date and time"), QVariant::String);
    item = addItemInfo(group, "EffectiveDate", i18n("Volume effective date and time"), QVariant::String);
    item = addItemInfo(group, "Thumbnail", i18n("ISO image icon"), QVariant::String);
    setHint(item, KFileMimeTypeInfo::Thumbnail);
};

static QDateTime parse_iso_date(QString date) {
    QString iso_str;
    iso_str += date.mid(0, 4);
    iso_str += "-";
    iso_str += date.mid(4, 2);
    iso_str += "-";
    iso_str += date.mid(6, 2);
    iso_str += " ";
    iso_str += date.mid(8, 2);
    iso_str += ":";
    iso_str += date.mid(10, 2);
    iso_str += ":";
    iso_str += date.mid(12, 2);
    return QDateTime::fromString(iso_str, Qt::ISODate);
};

bool IsoImagePlugin::readInfo( KFileMetaInfo& info, uint what)
{
    KURL url;
    KURL mount_point;
    bool mounted = false;
    
    if(info.mimeType() == "application/x-iso-image") {
        url = info.url();
        if(!url.isLocalFile()) {
            return false; // does not try to read iso image from network
        };
    } else if(info.mimeType() == "inode/x-iso-image-mounted") {
        FMountPoint mp = FMountPoint::fromUrl(info.url());
        if(mp.mountedFrom().isEmpty()) {
            return false; // not mounted
        };
        url = mp.mountedFrom();
        mount_point = mp.mountPoint();
        mounted = true;
    } else {
        kdDebug() << "Called with unsupported mimetype: " << info.mimeType() << endl;
        return false; // called with something strange
    };
    
    // kdDebug() << "there at all with " << url.path() << " of type " << info.mimeType() << endl;
    
    KFileMetaInfoGroup group = appendGroup(info, "IsoImageInfo");
    
    /// @todo move vd read stuff info fuseisolib and make it kde-transparent
    struct iso_primary_descriptor vd;
    memset(&vd, 0, sizeof(struct iso_volume_descriptor));
    int mode = read_volume_descriptor(url.path().ascii(), (struct iso_volume_descriptor*)&vd);
    if(mode < 0) {
        return false;
    };

    switch(mode) {
        case IMAGE_ISO:
            appendItem(group, "Type", i18n("ISO9660 image"));
            break;
        case IMAGE_MODE1:
            appendItem(group, "Type", i18n("MODE1 image"));
            break;
        case IMAGE_MODE2:
            appendItem(group, "Type", i18n("MODE2 image"));
            break;
        case IMAGE_MODE2_RAW:
            appendItem(group, "Type", i18n("MODE2 RAW image"));
            break;
        case IMAGE_MODE2_2336:
            appendItem(group, "Type", i18n("MODE2/2336 image"));
            break;
        case IMAGE_NRG:
            appendItem(group, "Type", i18n("NRG image"));
            break;
        default:
            break;
    };
    
    QString str;
    
    str = QString::fromAscii(vd.system_id, 32).stripWhiteSpace();
    if(str != "") {
        appendItem(group, "SystemId", str);
    };
    str = QString::fromAscii(vd.volume_id, 32).stripWhiteSpace();
    if(str != "") {
        appendItem(group, "VolumeId", str);
    };
    str = QString::fromAscii(vd.volume_set_id, 128).stripWhiteSpace();
    if(str != "") {
        appendItem(group, "VolumeSetId", str);
    };
    str = QString::fromAscii(vd.publisher_id, 128).stripWhiteSpace();
    if(str != "") {
        appendItem(group, "PublisherId", str);
    };
    str = QString::fromAscii(vd.preparer_id, 128).stripWhiteSpace();
    if(str != "") {
        appendItem(group, "PreparerId", str);
    };
    str = QString::fromAscii(vd.application_id, 128).stripWhiteSpace();
    if(str != "") {
        appendItem(group, "ApplicationId", str);
    };
    str = QString::fromAscii(vd.copyright_file_id, 37).stripWhiteSpace();
    if(str != "") {
        appendItem(group, "CopyrightFileId", str);
    };
    str = QString::fromAscii(vd.abstract_file_id, 37).stripWhiteSpace();
    if(str != "") {
        appendItem(group, "AbstractFileId", str);
    };
    str = QString::fromAscii(vd.bibliographic_file_id, 37).stripWhiteSpace();
    if(str != "") {
        appendItem(group, "BibliographicFileId", str);
    };
    
    QDateTime date;
    
    date = parse_iso_date(QString::fromAscii(vd.creation_date, 16).stripWhiteSpace());
    if(date.isValid()) {
        appendItem(group, "CreationDate", date);
    };
    date = parse_iso_date(QString::fromAscii(vd.modification_date, 16).stripWhiteSpace());
    if(date.isValid()) {
        appendItem(group, "ModificationDate", date);
    };
    date = parse_iso_date(QString::fromAscii(vd.expiration_date, 16).stripWhiteSpace());
    if(date.isValid()) {
        appendItem(group, "ExpirationDate", date);
    };
    date = parse_iso_date(QString::fromAscii(vd.effective_date, 16).stripWhiteSpace());
    if(date.isValid()) {
        appendItem(group, "EffectiveDate", date);
    };
    
    if(mounted && 
        (what == KFileMetaInfo::DontCare || what == KFileMetaInfo::Thumbnail || what == KFileMetaInfo::Everything)) {
        
    };
    
    return true;
}

#include "kfile_iso_image.moc"

