/***************************************************************************
 *   Copyright (C) 2006 by Dmitry Morozhnikov   *
 *   dmiceman@mail.ru   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


// heavy based on kio_media

#include <stdlib.h>

#include <kdebug.h>
#include <klocale.h>
#include <kapplication.h>
#include <dcopclient.h>
#include <kcmdlineargs.h>
#include <kuser.h>
#include <kstandarddirs.h>
#include <qfileinfo.h>

#include <qeventloop.h>

#include "kio_isomedia.h"

#include "fuseisolib.h"

static const KCmdLineOptions options[] =
{
    { "+protocol", I18N_NOOP( "Protocol name" ), 0 },
    { "+pool", I18N_NOOP( "Socket name" ), 0 },
    { "+app", I18N_NOOP( "Socket name" ), 0 },
    KCmdLineLastOption
};

extern "C" {
    int KDE_EXPORT kdemain( int argc, char **argv )
    {
        // KApplication is necessary to use other ioslaves
        putenv(strdup("SESSION_MANAGER="));
        KCmdLineArgs::init(argc, argv, "kio_isomedia", 0, 0, 0, 0);
        KCmdLineArgs::addCmdLineOptions( options );
        KApplication app( false, false );
        // We want to be anonymous even if we use DCOP
        app.dcopClient()->attach();

        KCmdLineArgs *args = KCmdLineArgs::parsedArgs();
        IsoMediaProtocol slave( args->arg(0), args->arg(1), args->arg(2) );
        slave.dispatchLoop();
        return 0;
    }
}


IsoMediaProtocol::IsoMediaProtocol(const QCString &protocol,
                             const QCString &pool, const QCString &app)
    : ForwardingSlaveBase(protocol, pool, app), mtab(), sdirs()
{
}

IsoMediaProtocol::~IsoMediaProtocol()
{
}

// borrowed from kio_media
static void addAtom(KIO::UDSEntry &entry, unsigned int ID, long l,
    const QString &s = QString::null)
{
    KIO::UDSAtom atom;
    atom.m_uds = ID;
    atom.m_long = l;
    atom.m_str = s;
    entry.append(atom);
}

void IsoMediaProtocol::createTopLevelEntry(KIO::UDSEntry &entry) {
    entry.clear();
    addAtom(entry, KIO::UDS_URL, 0, "isomedia:/");
    addAtom(entry, KIO::UDS_NAME, 0, ".");
    addAtom(entry, KIO::UDS_FILE_TYPE, S_IFDIR);
    addAtom(entry, KIO::UDS_ACCESS, 0555);
    addAtom(entry, KIO::UDS_MIME_TYPE, 0, "inode/directory");
    addAtom(entry, KIO::UDS_ICON_NAME, 0, "cdrom_unmount");
};

void IsoMediaProtocol::createMountEntry(KIO::UDSEntry &entry) {
    entry.clear();
    addAtom(entry, KIO::UDS_URL, 0, QString("file://%1").arg(sdirs.findResource("data", "kfuseiso/mount.desktop")));
    // kdDebug() << "IsoMediaProtocol::createMountEntry: mount.desktop: " << sdirs.findResource("data", "kfuseiso/mount.desktop") << endl;
    addAtom(entry, KIO::UDS_NAME, 0, i18n("Mount disk image file"));
    addAtom(entry, KIO::UDS_FILE_TYPE, S_IFREG);
    addAtom(entry, KIO::UDS_ACCESS, 0444);
    addAtom(entry, KIO::UDS_MIME_TYPE, 0, "application/x-desktop");
    addAtom(entry, KIO::UDS_ICON_NAME, 0, "add");
    addAtom(entry, KIO::UDS_SIZE, QFileInfo(sdirs.findResource("data", "kfuseiso/mount.desktop")).size());
};

bool IsoMediaProtocol::rewriteURL(const KURL &url, KURL &newUrl)
{
    if(url.path() == "/mount.desktop" || url.path() == "isomedia:/mount.desktop") {
        newUrl = QString("file://%1").arg(sdirs.findResource("data", "kfuseiso/mount.desktop"));
        return true;
    };

    if(mtab.empty()) {
        mtab = FMountPoint::currentMountPoints();
//         kdDebug() << "read " << mtab.size() << " mtab entries" << endl;
    };
    FMountPoint::List::iterator it;
    for(it = mtab.begin(); it != mtab.end(); ++it) {
        KURL mp((*it)->mountPoint());
        QString fname = "/" + mp.fileName();
        
//         kdDebug() << "trying " << fname << " for " << url.path() << endl;
        
        if(url.path().startsWith(fname)) {
            newUrl = mp.directory(true, true);
            newUrl.addPath(url.path());
            newUrl.setProtocol("file");
            
//             kdDebug() << "IsoMediaProtocol::rewriteURL: " << url << " " << newUrl << endl << flush;
            return true;
        };
    };
    
    error(KIO::ERR_DOES_NOT_EXIST, url.prettyURL());
    
    kdDebug() << "IsoMediaProtocol::rewriteURL: NOT FOUND! " << url << endl << flush;
    
//     kdDebug() << "IsoMediaProtocol::rewriteURL: " << url << " " << newUrl << endl;

    return false;
}

void IsoMediaProtocol::stat(const KURL &url)
{
    // kdDebug() << "IsoMediaProtocol::stat: " << url << endl;
    
    QString path = url.path();
    if(path.startsWith("isomedia:")) {
        path = path.mid(9);
    }
    if(path.isEmpty() || path == "/") {
        // The root is "virtual" - it's not a single physical directory
        KIO::UDSEntry entry;
        createTopLevelEntry(entry);
        statEntry(entry);
        finished();
        return;
    }
    
    // more clarification needed for stat() mechanics; disabled for now
    if(path.find('/', 1) == -1 and false) {
        KIO::UDSEntry entry;
        
        if(path.endsWith("mount.desktop")) {
        
            createMountEntry(entry);
            statEntry(entry);
            finished();
            return;
        
        } else {
        
            if(mtab.empty()) {
                mtab = FMountPoint::currentMountPoints();
            };
            FMountPoint::List::iterator it;
            for(it = mtab.begin(); it != mtab.end(); ++it) {
                KURL fname((*it)->mountPoint());
                
                if(fname.fileName() == url.fileName()) {
                    entry.clear();
                    
                    QString furl = "isomedia:/" + fname.fileName();
                    addAtom(entry, KIO::UDS_URL, 0, furl);
                    
                    QString label = KIO::encodeFileName(fname.fileName());
                    addAtom(entry, KIO::UDS_NAME, 0, label);
                    
                    addAtom(entry, KIO::UDS_FILE_TYPE, S_IFDIR);
                    
                    addAtom(entry, KIO::UDS_MIME_TYPE, 0, "inode/x-iso-image-mounted");
                    addAtom(entry, KIO::UDS_GUESSED_MIME_TYPE, 0, "inode/directory");
                    
                    addAtom(entry, KIO::UDS_ICON_NAME, 0, "cdtrack");
                    
                    addAtom(entry, KIO::UDS_LOCAL_PATH, 0, fname.path());
                    
                    statEntry(entry);
                    finished();
                    return;
                };
            };
            
            error(KIO::ERR_DOES_NOT_EXIST, url.prettyURL());
        
        };
        
    } else {
        ForwardingSlaveBase::stat(url);
    };
}

void IsoMediaProtocol::listDir(const KURL &url)
{
    // kdDebug() << "IsoMediaProtocol::listDir: " << url << endl;

    if(url.path().length() <= 1) {
        listRoot();
        return;
    };

    ForwardingSlaveBase::listDir(url);
}

void IsoMediaProtocol::listRoot()
{
    KIO::UDSEntry entry;

    KIO::UDSEntryList media_entries;
    
    createTopLevelEntry(entry);
    listEntry(entry, false);

    // to be used in stat() and rewriteURL()
    mtab = FMountPoint::currentMountPoints();
    FMountPoint::List::iterator it;
    for(it = mtab.begin(); it != mtab.end(); ++it) {
        entry.clear();
        
        KURL fname((*it)->mountPoint());
        
        QString url = "isomedia:/" + fname.fileName();
        addAtom(entry, KIO::UDS_URL, 0, url);
        
        QString label = KIO::encodeFileName(fname.fileName());
        addAtom(entry, KIO::UDS_NAME, 0, label);
        
        addAtom(entry, KIO::UDS_FILE_TYPE, S_IFDIR);
        
        addAtom(entry, KIO::UDS_MIME_TYPE, 0, "inode/x-iso-image-mounted");
        addAtom(entry, KIO::UDS_GUESSED_MIME_TYPE, 0, "inode/directory");
        
        addAtom(entry, KIO::UDS_ICON_NAME, 0, "cdtrack");
        
        addAtom(entry, KIO::UDS_LOCAL_PATH, 0, fname.path());
        
        media_entries.append(entry);
    };

    createMountEntry(entry);
    listEntry(entry, false);

    totalSize(media_entries.count());
    
    KIO::UDSEntryListIterator it1 = media_entries.begin();
    KIO::UDSEntryListIterator end = media_entries.end();

    for(; it1 != end; ++it1) {
        listEntry(*it1, false);
    };

    entry.clear();
    listEntry(entry, true);

    finished();
}

void IsoMediaProtocol::slotWarning( const QString &msg )
{
    warning( msg );
}

#include "kio_isomedia.moc"
