/*
 * Klogwatch, Netfilter log monitor
 *
 * Copyright (C) 2004 Nick Battle <nick.battle@freeuk.com>
 * Copyright (C) 2006 John Stamp <jstamp@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 */

#include <string.h>

#include "debug.h"
#include "klogIP.h"

KLogIP::KLogIP(const KLogLine *line)
{
    debug("Constructing IP\n");

    logline = line;
    protocol = line->getProto();
    date = line->getDate();
    time = line->getTime();
    datetime = line->getDateTime();
    srcIP = line->getIP("SRC");
    dstIP = line->getIP("DST");
    inIface = line->getIP("IN");
    outIface = line->getIP("OUT");
    macAddr = line->getIP("MAC");
}

KLogIP::~KLogIP()
{
    debug("Destroy IP\n");
}

const char *KLogIP::getLine(void) const
{
    return logline->getLine();
}

QString KLogIP::getHost(void) const
{
    return logline->getHost();
}

QString KLogIP::getDate() const
{
    char d[16];
    sprintf(d, "%02d/%02d/%02d", date.day(), date.month(), date.year()%100);
    return QString(d);
}

QString KLogIP::getTime(void) const
{
    return time.toString();
}

QString KLogIP::getDateTime(void) const
{
    return datetime;
}

QString KLogIP::getPrefix(void) const
{
    return logline->getPrefix();
}

QString KLogIP::getSrcIP(void) const
{
    return srcIP;
}

QString KLogIP::getSrcPort(void) const
{
    return "";
}

int KLogIP::getSrcPortNum(void) const
{
    return (-20);  // icmp uses -19 to -1 for port counters
}

QString KLogIP::getDstIP(void) const
{
    return dstIP;
}

QString KLogIP::getDstPort(void) const
{
    return "";
}

int KLogIP::getDstPortNum(void) const
{
    return (-20);  // icmp uses -19 to -1 for port counters
}

QString KLogIP::getProto(void) const
{
    return protocol;
}

QString KLogIP::getInIface(void) const
{
    return inIface;
}

QString KLogIP::getOutIface(void) const
{
    return outIface;
}

QString KLogIP::getMacAddr(void) const
{
    return macAddr;
}

QString KLogIP::getFlags(void) const
{
    return "";
}

QString KLogIP::toString(bool full) const
{
    char p[8];

    if (protocol.isEmpty())
    {
        strcpy(p, ":      ");
    }
    else
    {
        sprintf(p, ": %-4s ", protocol.latin1());
    }

    if (full)
    {
        return getDate() + " " + getTime() + p + srcIP;
    }
    else
    {
        return getTime() + p + srcIP;
    }
}

// eof
