/*
 * Klogwatch, Netfilter log monitor
 *
 * Copyright (C) 2004 Nick Battle <nick.battle@freeuk.com>
 * Copyright (C) 2006 John Stamp <jstamp@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 */

#include <netdb.h>
#include <netinet/in.h>

#include "debug.h"
#include "klogTCP.h"

KLogTCP::KLogTCP(const KLogLine *line) : KLogIP(line)
{
    debug("Constructing TCP\n");

    srcPort = line->getInt("SPT");
    dstPort = line->getInt("DPT");

    URG = line->getBool("URG");
    ACK = line->getBool("ACK");
    PSH = line->getBool("PSH");
    RST = line->getBool("RST");
    SYN = line->getBool("SYN");
    FIN = line->getBool("FIN");
}

KLogTCP::~KLogTCP()
{
    debug("Destroy TCP\n");
}

QString KLogTCP::getSrcPort(void) const
{
    return QString::number(srcPort);
}

QString KLogTCP::getDstPort(void) const
{
    char b[64];
    struct servent *service;

    if ((service = getservbyport(htons(dstPort), NULL)) == NULL)
    {
        sprintf(b, "%d (unknown)", dstPort);
    }
    else
    {
        sprintf(b, "%s (%d)", service->s_name, dstPort);
    }

    return QString(b);
}

int KLogTCP::getDstPortNum(void) const
{
    return dstPort;
}

int KLogTCP::getSrcPortNum(void) const
{
    return srcPort;
}

QString KLogTCP::getFlags(void) const
{
    char b[64];

    b[0] = '\0';

    if (URG) strcat(b, "URG ");
    if (ACK) strcat(b, "ACK ");
    if (PSH) strcat(b, "PSH ");
    if (RST) strcat(b, "RST ");
    if (SYN) strcat(b, "SYN ");
    if (FIN) strcat(b, "FIN ");

    return QString(b);
}

QString KLogTCP::toString(bool full) const
{
    char b[64];
    struct servent *service;

    if ((service = getservbyport(htons(dstPort), NULL)) == NULL)
    {
        sprintf(b, "(%d) to %d(unknown), flags ", srcPort, dstPort);
    }
    else
    {
        sprintf(b, "(%d) to %s(%d), flags ", srcPort, service->s_name, dstPort);
    }

    if (URG) strcat(b, "URG ");
    if (ACK) strcat(b, "ACK ");
    if (PSH) strcat(b, "PSH ");
    if (RST) strcat(b, "RST ");
    if (SYN) strcat(b, "SYN ");
    if (FIN) strcat(b, "FIN ");

    return KLogIP::toString(full) + b;
}

// eof
