/*
 * Klogwatch, Netfilter log monitor
 *
 * Copyright (C) 2004 Nick Battle <nick.battle@freeuk.com>
 * Copyright (C) 2006 John Stamp <jstamp@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 */

#include <qlayout.h>

#include <kglobalsettings.h>
#include <kuniqueapplication.h>

#include "debug.h"
#include "klogconfig.h"

KLogConfig::KLogConfig()
{
    file = kapp->config();
    load();
}

KLogConfig::~KLogConfig()
{
    save();
    delete file;
    me = NULL;
}

KLogConfig *KLogConfig::me;

KLogConfig *KLogConfig::getConfig()
{
    if (me == NULL)
    {
        debug("Creating config\n");
        me = new KLogConfig();
    }
    else
    {
        debug("Returning config\n");
    }

    return me;
}

void KLogConfig::load()
{
    debug("Loading config\n");

    file->setGroup("Timings");
    POLLTIME = file->readNumEntry("POLLTIME", _POLLTIME);
    PAUSETIME = file->readNumEntry("PAUSETIME", _PAUSETIME);

    file->setGroup("General");
    BUFFSIZE = file->readNumEntry("BUFFSIZE", _BUFFSIZE);
    COMMAND = file->readEntry("COMMAND", _COMMAND);
    SAVERAW = file->readBoolEntry("SAVERAW", _SAVERAW);
    FILENAME = file->readEntry("FILENAME", _FILENAME);
    COMMANDDIALOG = file->readNumEntry("COMMANDDIALOG", _COMMANDDIALOG);

    DoubleClickAction = file->readNumEntry("DOUBLECLICK_ACTION", SOURCE_HOST);

    mainWinSize = file->readSizeEntry("SIZE");
    mainWinPos = file->readPointEntry("POSITION");

    QFont def = KGlobalSettings::fixedFont();
    FONT = file->readFontEntry("FONT", &def);
    PRINTFONT = file->readFontEntry("PRINTFONT", &def);

    QFont sizingFont = FONT;
    sizingFont.setBold(true);

    QFontMetrics fm(sizingFont);

    ColumnWidths = file->readListEntry("COLUMN_WIDTHS");

    StandardWidths << QString::number(fm.width("05 NOV 88:88:88")+10)
                   << QString::number(fm.width("Log Prefix")+40)
                   << QString::number(fm.width("Protocol")+40)
                   << QString::number(fm.width("In Iface")+40)
                   << QString::number(fm.width("Out Iface")+40)
                   << QString::number(fm.width("XX:XX:XX:XX:XX:XX")+10)
                   << QString::number(fm.width("Source Address")+40)
                   << QString::number(fm.width("destination-unreachable(3), code 8")+10)
                   << QString::number(fm.width("Dest. Address")+40)
                   << QString::number(fm.width("destination-unreachable(3), code 8")+10)
                   << QString::number(fm.width("XYZ XYZ XYZ XYZ")+10);

    if ( ColumnWidths.empty() )
        ColumnWidths << QString::number(fm.width("88:88:88")+10)
                     << "0" // log prefix hidden
                     << QString::number(fm.width("Protocol")+10)
                     << "0" // in iface hidden
                     << "0" // out iface hidden
                     << "0" // mac address hidden
                     << QString::number(fm.width("888.888.888.888")+10)
                     << "0" // source port hidden
                     << "0" // dest addr hidden
                     << QString::number(fm.width("destination-unreachable(3), code 8")+10)
                     << QString::number(fm.width("XYZ XYZ XYZ XYZ")+10);

    ColumnOrder = file->readListEntry("COLUMN_ORDER");

    if (ColumnOrder.empty() )
        ColumnOrder << "0" << "1" <<  "2" << "3" << "4" << "5" <<  "6" << "7" << "8" << "9" << "10";

    SortColumn = file->readNumEntry("SORT_COLUMN", 0);
    SortAscending = file->readNumEntry("SORT_ASCENDING", Qt::Ascending);

    debug("POLLTIME = %d\n", POLLTIME);
    debug("PAUSETIME = %d\n", PAUSETIME);
    debug("BUFFSIZE = %d\n", BUFFSIZE);
    debug("COMMAND = %s\n", COMMAND.latin1());
    debug("SAVERAW = %d\n", SAVERAW);
    debug("FILENAME = %s\n", FILENAME.latin1());
    debug("FONT = %s\n", FONT.toString().latin1());
    debug("PRINTFONT = %s\n", PRINTFONT.toString().latin1());
    debug("COMMANDDIALOG = %d\n", COMMANDDIALOG);
}

void KLogConfig::save()
{
    debug("Saving config\n");

    file->setGroup("Timings");
    file->writeEntry("POLLTIME", POLLTIME);
    debug("Written POLLTIME = %d\n", POLLTIME);
    file->writeEntry("PAUSETIME", PAUSETIME);
    debug("Written PAUSETIME = %d\n", PAUSETIME);

    file->setGroup("General");
    file->writeEntry("BUFFSIZE", BUFFSIZE);
    debug("Written BUFFSIZE = %d\n", BUFFSIZE);
    file->writeEntry("COMMAND", COMMAND);
    debug("Written COMMAND = %s\n", COMMAND.latin1());
    file->writeEntry("SAVERAW", SAVERAW);
    debug("Written SAVERAW = %d\n", SAVERAW);
    file->writeEntry("FILENAME", FILENAME);
    debug("Written FILENAME = %s\n", FILENAME.latin1());
    file->writeEntry("FONT", FONT);
    debug("Written FONT = %s\n", FONT.toString().latin1());
    file->writeEntry("PRINTFONT", PRINTFONT);
    debug("Written PRINTFONT = %s\n", PRINTFONT.toString().latin1());
    file->writeEntry("COMMANDDIALOG", COMMANDDIALOG);
    debug("Written COMMANDDIALOG = %d\n", COMMANDDIALOG);

    file->writeEntry("DOUBLECLICK_ACTION", DoubleClickAction);
    file->writeEntry("SIZE", mainWinSize);
    file->writeEntry("POSITION", mainWinPos);

    file->writeEntry("COLUMN_WIDTHS", ColumnWidths);
    file->writeEntry("COLUMN_ORDER", ColumnOrder);

    file->writeEntry("SORT_COLUMN", SortColumn);
    file->writeEntry("SORT_ASCENDING", SortAscending);

    file->sync();
}

void KLogConfig::defaults()
{
    debug("Setting defaults\n");

    POLLTIME = _POLLTIME;
    debug("Set POLLTIME = %d\n", POLLTIME);
    PAUSETIME = _PAUSETIME;
    debug("Set PAUSETIME = %d\n", PAUSETIME);

    file->setGroup("General");
    BUFFSIZE = _BUFFSIZE;
    debug("Set BUFFSIZE = %d\n", BUFFSIZE);
    COMMAND = _COMMAND;
    debug("Set COMMAND = %s\n", COMMAND.latin1());
    SAVERAW = _SAVERAW;
    debug("Set SAVERAW = %d\n", SAVERAW);
    FILENAME = _FILENAME;
    debug("Set FILENAME = %s\n", FILENAME.latin1());
    COMMANDDIALOG = _COMMANDDIALOG;
    debug("Set COMMANDDIALOG = %d\n", COMMANDDIALOG);
    FONT = KGlobalSettings::fixedFont();
    debug("Set PRINTFONT = %s\n", PRINTFONT.toString().latin1());
    FONT = KGlobalSettings::fixedFont();
    debug("Set PRINTFONT = %s\n", PRINTFONT.toString().latin1());
    DoubleClickAction = 0;
}

// eof
