/*
 * Klogwatch, Netfilter log monitor
 *
 * Copyright (C) 2006 John Stamp <jstamp@users.sourceforge.net>
 *
 * with some ideas taken from:
 *   KDevelop: processlinemaker.cpp
 *   Copyright (C) 2002 John Firebaugh <jfirebaugh@kde.org>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 */

#include <arpa/inet.h>
#include <netdb.h>

#include <qcursor.h>
#include <qhbox.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qtextedit.h>
#include <qvbox.h>
#include <qwhatsthis.h>

#include <kapplication.h>
#include <kglobalsettings.h>
#include <kiconloader.h>
#include <kmessagebox.h>
#include <kprocess.h>
#include <kpushbutton.h>

#include "debug.h"
#include "klogconfig.h"
#include "klogdialog.h"


KLogDialog::KLogDialog(bool isRunCommand, QWidget *parent, const char *name, bool modal, WFlags fl)
                             : KDialog( parent, name, modal, Qt::WDestructiveClose ),
                             proc(0),
                             stdoutbuf(""),
                             stderrbuf(""),
                             priorOutput(false),
                             _showDialog(true)
{
    if (isRunCommand)
      debug("Setting up run command dialog\n");
    else
      debug("Setting up hostname dialog\n");

    // Create the layout
    QVBoxLayout *topcontents = new QVBoxLayout(this);
    topcontents->setSpacing(KDialog::spacingHint()*2);
    topcontents->setMargin(KDialog::marginHint());

    QHBoxLayout *lay = new QHBoxLayout(topcontents);
    lay->setSpacing(KDialog::spacingHint());

    QLabel *label1 = new QLabel(this);
    QPixmap icon =  KApplication::kApplication()->iconLoader()->loadIcon("messagebox_info", KIcon::NoGroup, KIcon::SizeMedium, KIcon::DefaultState, 0, true);
    label1->setPixmap(icon);

    lay->addWidget(label1, 0, Qt::AlignCenter);
    lay->addSpacing(KDialog::spacingHint());

    infoLabel = new QLabel( this );

    lay->addWidget( infoLabel );
    lay->addStretch();

    if (isRunCommand) {
        QHBoxLayout *textLay = new QHBoxLayout(topcontents);
        textLay->setSpacing(KDialog::spacingHint());

        runOutput = new QTextEdit( this);
        runOutput->setReadOnly(true);
        runOutput->setFont(KGlobalSettings::fixedFont());
        runOutput->setWordWrap(QTextEdit::NoWrap);

        QFontMetrics fm(KGlobalSettings::fixedFont());
        int fh = fm.height()*10+18;
        int fw = fm.width("1")*81+18+10;
        runOutput->setMinimumSize( QSize(fw, fh) );

        textLay->addWidget(runOutput);
        QWhatsThis::add( runOutput, "Shows the output of the currently running command." );
    }

    QHBoxLayout *buttonLay = new QHBoxLayout(topcontents);


    buttonLay->setSpacing(KDialog::spacingHint());

    QSpacerItem *leftSpacer = new QSpacerItem( 10, 20, QSizePolicy::Expanding, QSizePolicy::Minimum );
    buttonLay->addItem( leftSpacer);

    buttonCancel = new KPushButton( this);
    buttonCancel->setAutoDefault( TRUE );

    if (isRunCommand)
    {
        buttonCancel->setText("Cancel");
        buttonCancel->setIconSet(KGlobal::iconLoader()->loadIcon("button_cancel", KIcon::Toolbar, KIcon::SizeSmall, KIcon::ActiveState));
    }
    else
    {
        buttonCancel->setText("Close");
        buttonCancel->setIconSet(KGlobal::iconLoader()->loadIcon("fileclose", KIcon::Toolbar, KIcon::SizeSmall, KIcon::ActiveState));
    }

    buttonLay->addWidget( buttonCancel, 0, 0 );

    if (!isRunCommand) {
        QSpacerItem *rightSpacer = new QSpacerItem( 10, 20, QSizePolicy::Expanding, QSizePolicy::Minimum );
        buttonLay->addItem( rightSpacer);
    }

    if (!isRunCommand)
        setFixedSize( sizeHint() );
    else
        setMinimumSize( sizeHint() );

    connect( buttonCancel, SIGNAL( clicked() ), this, SLOT( close() ) );
}


KLogDialog::~KLogDialog()
{
// this is for the sake of KDE 3.1
// Because >=KDE3.2 uses KProcess(this) it doesn't need it
#if KDE_IS_VERSION(3,2,0)
    if (proc)
    {
        delete proc;
        proc=NULL;
    }
#endif
}

void KLogDialog::runCommand(const KLogIP *packet, QString command, bool showDialog)
{
    _showDialog = showDialog;
    _command = command;


#if KDE_IS_VERSION(3,2,0)
    proc = new KProcess(this);
#else
    proc = new KProcess();
#endif

    proc->setUseShell(true);

    proc->setEnvironment("DATE", packet->getDate().latin1());
    proc->setEnvironment("TIME", packet->getTime().latin1());
    proc->setEnvironment("PREFIX", packet->getPrefix().latin1());
    proc->setEnvironment("IN", packet->getInIface().latin1());
    proc->setEnvironment("OUT", packet->getOutIface().latin1());
    proc->setEnvironment("MAC", packet->getMacAddr().latin1());
    proc->setEnvironment("PROTO", packet->getProto().latin1());
    proc->setEnvironment("SRC", packet->getSrcIP().latin1());
    proc->setEnvironment("SPT", packet->getSrcPort().latin1());
    proc->setEnvironment("DST", packet->getDstIP().latin1());
    proc->setEnvironment("DPT", packet->getDstPort().latin1());
    proc->setEnvironment("FLAGS", packet->getFlags().latin1());
    proc->setEnvironment("LINE", packet->getLine());


    connect( this, SIGNAL(receivedStdoutLine(const QString&)),
             this, SLOT(insertLine(const QString&) ));

    connect( this, SIGNAL(receivedStderrLine(const QString&)),
             this, SLOT(insertLine(const QString&) ));

    connect(proc, SIGNAL(receivedStdout(KProcess*, char*, int)), 
            this, SLOT(slotReceivedStdout(KProcess*,char*,int)));

    connect(proc, SIGNAL(receivedStderr(KProcess*, char*, int)),
            this, SLOT(slotReceivedStderr(KProcess*, char*, int)));

    connect( proc, SIGNAL(processExited(KProcess *)),
             SLOT(childExited()) );

    // First we need to get the full command with the variables expanded
    // So we echo the command and capture stdout.  When the echo process
    // finishes, we know to then run the command
    QString runLine = "Run Command: ";
    infoLabel->setText(runLine);

    getRunLine = true;
    *proc << "echo" << _command;
    proc->start(KProcess::NotifyOnExit,
                KProcess::Stdout);

    if (_showDialog)
        show();
}

void KLogDialog::getHostName(const QString tempaddr, bool sourceHost)
{
    // Don't show yet.  We want to get the host name first so
    // the dialog will be properly positioned on show()
    debug("Getting hostname\n");
    QString hostType;
    QString text;
    if (sourceHost)
        hostType = "Source";
    else
        hostType = "Destination";
    setCaption(QString("%1 Host Name").arg(hostType));

    struct in_addr myaddr;
    myaddr.s_addr=inet_addr(tempaddr);
    struct hostent *host = gethostbyaddr((char*)&myaddr, sizeof(struct in_addr), AF_INET);
    if (host != NULL)
        text = host->h_name;
    else
        text = "No hostname found";
    infoLabel->setAlignment(Qt::AlignHCenter);

    infoLabel->setText(QString("Lookup for %1:\n%2").arg(tempaddr).arg(text));
    show();
}

void KLogDialog::slotReceivedStdout( const QString& s )
{
    // Flush stderr buffer
    if (!stderrbuf.isEmpty()) {
        emit receivedStderrLine(stderrbuf);
        stderrbuf = "";
    }

    stdoutbuf += s;
    int pos;
    while ( (pos = stdoutbuf.find('\n')) != -1) {
        QString line = stdoutbuf.left(pos);
        emit receivedStdoutLine(line);
        stdoutbuf.remove(0, pos+1);
    }
}


void KLogDialog::slotReceivedStdout( KProcess*, char *buffer, int buflen )
{
    slotReceivedStdout( QString::fromLocal8Bit( buffer, buflen ) );
}

void KLogDialog::slotReceivedStdout( const char* buffer )
{
    slotReceivedStdout( QString::fromLocal8Bit( buffer ) );
}

void KLogDialog::slotReceivedStderr( const QString& s )
{
    // Flush stdout buffer
    if (!stdoutbuf.isEmpty()) {
        emit receivedStdoutLine(stdoutbuf);
        stdoutbuf = "";
    }

    stderrbuf += s;
    int pos;
    while ( (pos = stderrbuf.find('\n')) != -1) {
        QString line = stderrbuf.left(pos);
        emit receivedStderrLine(line);
        stderrbuf.remove(0, pos+1);
    }
}

void KLogDialog::slotReceivedStderr( KProcess*, char *buffer, int buflen )
{
    slotReceivedStderr( QString::fromLocal8Bit( buffer, buflen ) );
}

void KLogDialog::slotReceivedStderr( const char* buffer )
{
    slotReceivedStderr( QString::fromLocal8Bit( buffer ) );
}

void KLogDialog::insertLine(const QString &line)
{
    if (getRunLine) {
        infoLabel->setText(infoLabel->text() + line.stripWhiteSpace());
    }
    else
    {
        runOutput->append(line + '\n');
        priorOutput = true;
    }
}

void KLogDialog::childExited()
{
    if (!_showDialog && !getRunLine) {
        close();
    }
    else if (getRunLine) {
        // We usually won't need this, but it's good to be on the safe side
        // Flush stdout buffer
        if (!stdoutbuf.isEmpty()) {
            // we wouldn't even be here if there was an endline right?
            stdoutbuf += '\n';
            insertLine(stdoutbuf);
            stdoutbuf = "";
        }

        // OK.  We have the expanded command for the dialog.  Now actually run the command.
        proc->clearArguments();
        getRunLine = false;
        *proc << _command;
        proc->start(KProcess::NotifyOnExit, 
                KProcess::AllOutput);
    }
    else
    {
        buttonCancel->setText("Close");
        buttonCancel->setIconSet(KGlobal::iconLoader()->loadIcon("fileclose", 
                                 KIcon::Toolbar, KIcon::SizeSmall, KIcon::ActiveState));

        // We usually won't need this, but it's good to be on the safe side
        // Flush stdout buffer
        if (!stdoutbuf.isEmpty()) {
            // we wouldn't even be here if there was an endline right?
            stdoutbuf += '\n';
            emit receivedStdoutLine(stdoutbuf);
            stdoutbuf = "";
        }
        // Flush stderr buffer
        if (!stderrbuf.isEmpty()) {
            // we wouldn't even be here if there was an endline right?
            stderrbuf += '\n';
            emit receivedStderrLine(stderrbuf);
            stderrbuf = "";
        }
#if KDE_IS_VERSION(3,2,0)
        if ( proc->signalled() )  // killed by a signal
            runOutput->append( QString("\nKLogWatch: Program killed by signal %1\n").arg(proc->exitSignal()) );
        else
#endif
            runOutput->append("\nKLogWatch: Program done.\n");
    }
}

// eof
