/*
 * Klogwatch, Netfilter log monitor
 *
 * Copyright (C) 2004 Nick Battle <nick.battle@freeuk.com>
 * Copyright (C) 2006 John Stamp <jstamp@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 */

#include <qpainter.h>

#include "debug.h"
#include "klogitem.h"

KLogItem::KLogItem(KListView *listbox, const KLogLine *line) :
          KListViewItem(listbox)
{
    this->line = line;
    packet = line->getPacket();

    this->setText(TIME_COL, packet->getDateTime());
    this->setText(PREFIX_COL, packet->getPrefix());
    this->setText(PROTOCOL_COL, (packet->getProto()).lower());
    this->setText(S_ADDR_COL, packet->getSrcIP());
    this->setText(D_PORT_COL, packet->getDstPort());

    this->setText(S_PORT_COL, packet->getSrcPort());
    this->setText(D_ADDR_COL, packet->getDstIP());
    this->setText(IN_IFACE_COL, packet->getInIface());
    this->setText(OUT_IFACE_COL, packet->getOutIface());
    this->setText(MAC_COL, packet->getMacAddr());

    this->setText(FLAGS_COL, (packet->getFlags()).lower());

    srcIpNum[0] = ((packet->getSrcIP()).section('.', 0, 0)).toInt();
    srcIpNum[1] = ((packet->getSrcIP()).section('.', 1, 1)).toInt();
    srcIpNum[2] = ((packet->getSrcIP()).section('.', 2, 2)).toInt();
    srcIpNum[3] = ((packet->getSrcIP()).section('.', 3, 3)).toInt();

    dstIpNum[0] = ((packet->getDstIP()).section('.', 0, 0)).toInt();
    dstIpNum[1] = ((packet->getDstIP()).section('.', 1, 1)).toInt();
    dstIpNum[2] = ((packet->getDstIP()).section('.', 2, 2)).toInt();
    dstIpNum[3] = ((packet->getDstIP()).section('.', 3, 3)).toInt();

    srcPortNum = packet->getSrcPortNum();
    dstPortNum = packet->getDstPortNum();

    isNew = true;
}

void KLogItem::paintCell(QPainter *p, const QColorGroup &cg,int column, int width, int alignment)
{
  if (isNew)
  {
    QFont font(p->font());
    font.setBold(true);
    p->setFont(font);
  }
  KListViewItem::paintCell(p, cg, column, width, alignment);
}

const KLogIP *KLogItem::getPacket(void) const
{
    return packet;
}

int KLogItem::compare(QListViewItem *i, int col, bool ascending ) const
{
    if (col == TIME_COL) { //sort by packet number, not clock time
      if ( packetNumber < ((KLogItem*)i)->packetNumber )
        return -1;
      else if ( packetNumber > ((KLogItem*)i)->packetNumber )
        return 1;
      else
        return 0;
    }

    else if (col == S_ADDR_COL) {  //sort source ip address numerically
      for (int count=0; count < 4; count++) {
        if (srcIpNum[count] < ((KLogItem*)i)->srcIpNum[count])
          return -1;
        else if (srcIpNum[count] > ((KLogItem*)i)->srcIpNum[count])
          return 1;
      }
      return 0;
    }

    else if (col == D_ADDR_COL) {  //sort dest ip address numerically
      for (int count=0; count < 4; count++) {
        if (dstIpNum[count] < ((KLogItem*)i)->dstIpNum[count])
          return -1;
        else if (dstIpNum[count] > ((KLogItem*)i)->dstIpNum[count])
          return 1;
      }
      return 0;
    }

    else if (col == S_PORT_COL) { //sort source port numerically
      if ( srcPortNum < ((KLogItem*)i)->srcPortNum )
        return -1;
      else if ( srcPortNum > ((KLogItem*)i)->srcPortNum )
        return 1;
      else
        return 0;
    }

    else if (col == D_PORT_COL) { //sort destination port numerically
      if ( dstPortNum < ((KLogItem*)i)->dstPortNum )
        return -1;
      else if ( dstPortNum > ((KLogItem*)i)->dstPortNum )
        return 1;
      else
        return 0;
    }

    else // return normal compare for all other columns
      return QString::localeAwareCompare( key(col, ascending), i->key(col, ascending) );
}

KLogItem::~KLogItem()
{
    debug("Destroy KLogItem\n");
    delete line;        // delete's packet too
}

