/*
 * Klogwatch, Netfilter log monitor
 *
 * Copyright (C) 2004 Nick Battle <nick.battle@freeuk.com>
 * Copyright (C) 2006 John Stamp <jstamp@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 */

#include <netdb.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>

#include <qregexp.h>

#include "debug.h"
#include "klogICMP.h"
#include "klogIP.h"
#include "klogline.h"
#include "klogTCP.h"
#include "klogUDP.h"

KLogLine::KLogLine(const QString s) : qline(s)
{
    debug("Constructing KLogLine\n");
    logline = qline.latin1();
    // don't crash on empty lines
    if (logline != NULL)
        parse();
}

KLogLine::~KLogLine()
{
    debug("Destroy KLogLine\n");
    if (packet != NULL)
    {
        delete packet;
    }
}

const char *KLogLine::getLine(void) const
{
    return logline;
}

const QString &KLogLine::toString(void) const
{
    return qline;
}

const KLogIP *KLogLine::getPacket(void) const
{
    return packet;
}

const bool KLogLine::isPacket(void) const
{
    return (packet != NULL);
}

const QDate &KLogLine::getDate(void) const
{
    return date;
}

const QTime &KLogLine::getTime(void) const
{
    return time;
}

const QString &KLogLine::getDateTime(void) const
{
    return datetime;
}

QString KLogLine::getPrefix(void) const
{
    QString returnVal = "";

    QRegExp rx( "kernel:.*IN=" );
    int pos = rx.search( logline );
    QStringList list = rx.capturedTexts();

    QStringList::Iterator it = list.begin();
    if (it != list.end()) {
        returnVal = *it;
        returnVal = returnVal.mid(7, (returnVal.length())-10);
    }

    return returnVal.stripWhiteSpace();
}


QString KLogLine::getString(const char *name) const
{
    char cstr[64];
    char pattern[64];
    const char *tag = strstr(logline, name);

    strcpy(pattern, name);
    // We need [^ ]
    // Without it, searching "OUT", for example, when there is no out interface recorded
    // will return interface "MAC=".
    strcat(pattern, "=%63[^ ]s");
    QString returnVal = "";

    if (tag != NULL && sscanf(tag, pattern, cstr) == 1)
    {
        returnVal = cstr;
    }
    else
    {
        debug("Can't find <%s>\n", name);
    }
    return returnVal;
}

QString KLogLine::getIP(const char *name) const
{
    return getString(name);
}

QString KLogLine::getHost(void) const
{
    char host[32];
    unsigned char addr[4];
    int a[4];
    QString ip = getString("SRC");

    if (!ip.isNull() &&
        sscanf(ip.latin1(), "%d.%d.%d.%d", &a[0], &a[1], &a[2], &a[3]) == 4)
    {
        for (int i=0; i<4; i++)
            addr[i] = (unsigned char)a[i];

        struct hostent *hp = gethostbyaddr(addr, 4, AF_INET);

        if (hp == NULL)
        {
            sprintf(host, "%d.%d.%d.%d", addr[0], addr[1], addr[2], addr[3]);
        }
        else
        {
            strcpy(host, hp->h_name);
        }

        debug("Formatted hostname = %s\n", host);
    }
    else
    {
        debug("Bad IP\n");
        strcpy(host, "?.?.?.?");
    }

    return QString(host);
}

int KLogLine::getInt(const char *name) const
{
    char pattern[64];
    const char *tag = strstr(logline, name);
    int value;

    strcpy(pattern, name);
    strcat(pattern, "=%d");

    if (tag == NULL || sscanf(tag, pattern, &value) != 1)
    {
        debug("Can't find int <%s>\n", name);
        value = 0;
    }

    return value;
}

bool KLogLine::getBool(const char *name) const
{
    char pattern[64];

    strcpy(pattern, " ");
    strcat(pattern, name);
    strcat(pattern, " ");

    return (strstr(logline, pattern) != NULL);
}

QString KLogLine::getProto(void) const
{
    QString protoText = getString("PROTO");
    bool ok;
    int protoNum = protoText.toInt(&ok, 10);

    if (ok)
    {
        struct protoent *protocol = getprotobynumber(protoNum);
        if (protocol)
            return (protocol->p_name);
        else
            return QString("unknown (%1)").arg(protoNum);
    }
    else
        return protoText;
}

void KLogLine::parse(void)
{
    static const char * const months[] =
    {
        "Jan", "Feb", "Mar", "Apr", "May", "Jun",
        "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
    };

    char mstr[4];
    int  year, month=0, day, hour, minute, second;

    if (sscanf(logline, "%3s %d %d:%d:%d",
                    mstr, &day, &hour, &minute, &second) == 5)
    {
        for (int m=0; m<12; m++)
        {
            if (strcmp(mstr, months[m]) == 0)
            {
                month = m + 1;
                break;
            }
        }

        if (month == 0)
        {
            debug("Bad month <%s>\n", mstr);
            month = 1; // force Jan
        }

        year = QDate::currentDate().year();
        date = QDate(year, month, day);
        time = QTime(hour, minute, second);
        datetime = QString("%1 %2 %3").arg(day).arg(months[month-1]).arg(time.toString());

        QString protocol = getProto();

        if (protocol.isEmpty())
        {
            packet = NULL;
        }
        else
        {
            if (protocol == "TCP")
            {
                debug("Parse TCP\n");
                packet = new KLogTCP(this);
            }
            else if (protocol == "UDP")
            {
                debug("Parse UDP\n");
                packet = new KLogUDP(this);
            }
            else if (protocol == "ICMP")
            {
                debug("Parse ICMP\n");
                packet = new KLogICMP(this);
            }
            else
            {
                debug("Parse unknown PROTO <%s>\n", protocol.latin1());
                packet = new KLogIP(this);
            }
        }
    }
    else
    {
        debug("KLogLine: bad date/time in log\n");
        date = QDate();
        time = QTime();
        packet = NULL;
    }
}

// eof
