/*
 * Klogwatch, Netfilter log monitor
 *
 * Copyright (C) 2004 Nick Battle <nick.battle@freeuk.com>
 * Copyright (C) 2006 John Stamp <jstamp@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 */


#include <qheader.h>
#include <qpainter.h>
#include <qpaintdevicemetrics.h>
#include <qwhatsthis.h>

#include <kglobalsettings.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kpopupmenu.h>
#include <kpushbutton.h>
#include <kuniqueapplication.h>

#include "debug.h"
#include "klogconfig.h"
#include "klogdialog.h"
#include "klogitem.h"
#include "klogIP.h"
#include "kloglist.h"
#include "klogpid.h"
#include "klogsetup.h"

KLogList::KLogList(QString file, QWidget *parent): QTlist(parent), timer(0), tailer(0)
{

    totalPacketCount = 0;
    uncheckedPacketCount = 0;
    oldestPacket = 1;

    tray = new KLogTray((KLogWatch *)parent, this);
    connect(tray, SIGNAL(quitSelected()), parent, SLOT(slotExit()));

    // header popup-menu
    listbox->header()->setClickEnabled(true);
    listbox->header()->installEventFilter(this);
    headerPopup = new KPopupMenu(this);
    headerPopup->insertTitle(i18n("View Columns"));
    headerPopup->setCheckable(true);
    headerPopup->insertItem(i18n("Time"), TIME_COL);
    headerPopup->insertItem(i18n("Log Prefix"), PREFIX_COL);
    headerPopup->insertItem(i18n("In Iface"), IN_IFACE_COL);
    headerPopup->insertItem(i18n("Out Iface"), OUT_IFACE_COL);
    headerPopup->insertItem(i18n("MAC Address"), MAC_COL);
    headerPopup->insertItem(i18n("Protocol"), PROTOCOL_COL);
    headerPopup->insertItem(i18n("Source Address"), S_ADDR_COL);
    headerPopup->insertItem(i18n("Source Port"), S_PORT_COL);
    headerPopup->insertItem(i18n("Destination Address"), D_ADDR_COL);
    headerPopup->insertItem(i18n("Destination Port"), D_PORT_COL);
    headerPopup->insertItem(i18n("Flags"), FLAGS_COL);

    listbox->setAllColumnsShowFocus(true);

    connect(listbox, SIGNAL(pressed(QListViewItem *)),
            this, SLOT(slotClick(QListViewItem *)));

    connect(headerPopup, SIGNAL(activated(int)), this, SLOT(slotToggleColumn(int)));

    connect(hideButton, SIGNAL(clicked()), this, SLOT(slotHide()));
    connect(pauseButton, SIGNAL(clicked()), this, SLOT(slotPause()));
    connect(clearButton, SIGNAL(clicked()), this, SLOT(slotClear()));

    connect( listbox, SIGNAL( contextMenuRequested( QListViewItem*, const QPoint &, int )),
             this, SLOT( rightButtonPressed( QListViewItem*, const QPoint &, int )));

    pauseButton->setToggleButton(true);
    pauseButton->setOn(false);

    clearButton->setIconSet(KGlobal::iconLoader()->loadIcon("view_remove", KIcon::Toolbar, KIcon::SizeSmall));
    pauseButton->setIconSet(KGlobal::iconLoader()->loadIcon("player_pause", KIcon::Toolbar, KIcon::SizeSmall));
    hideButton->setIconSet(KGlobal::iconLoader()->loadIcon("button_ok", KIcon::Toolbar, KIcon::SizeSmall));

    state = (debugOn ? THINKING : RUNNING);

    pollCount = 0;
    showCount = 0;
    pauseCount = 0;
    applySettings(file, parent);
}


KLogList::~KLogList()
{
    debug("Destroy KLogList\n");
}

void KLogList::applySettings(QString file, QWidget *parent)
{
    if (timer)
        timer->stop();

    if (tailer)
    {
        delete tailer;
        tailer = NULL;
    }
    klogconfig->FILENAME = file.latin1();
    tailer = new KLogTail(klogconfig->FILENAME);
    while (tailer->setup())
    {
        // A bit of a hack, but if this warning appears on startup, it will
        // muck up the main window size and position.  So store it away for the moment.
        QSize tempSize = klogconfig->mainWinSize;
        QPoint tempPos = klogconfig->mainWinPos;

        KMessageBox::sorry(0,
                           QString("Cannot open Netfilter log file '%1'!\n"
                                   "The file must be readable before KLogWatch can function properly.\n"
                                   "Please configure the location of the log file or use -f [filename] from the command line.").arg(klogconfig->FILENAME));
        delete tailer;
        tailer=NULL;

        if ( !klogconfig->mainWinSize.isEmpty() )
        {
            klogconfig->mainWinSize = tempSize;
            klogconfig->mainWinPos = tempPos;
        }

        KLogSetup *dialog = new KLogSetup(NULL);
        int choice = dialog->exec();
        delete dialog;

        if (choice == QDialog::Rejected)
        {
            debug("Give up\n");
            exit(1);
        }
        tailer = new KLogTail(klogconfig->FILENAME);
        QString caption("Netfilter Log Events from ");
        parent->setCaption(caption + klogconfig->FILENAME);
    }

    QFontMetrics fm(klogconfig->FONT);
    listbox->setFont(klogconfig->FONT);
    int fh = fm.height();
    debug("Font height = %d\n", fh);
    int im = listbox->itemMargin();
    debug("Item margin = %d\n", im);

    for (int colCount = 0; colCount < 9; colCount++)
    {
        listbox->setColumnWidthMode(colCount, QListView::Manual);
    }

    int width = restoreColLayout();

    // If the default size has never been set, let's give it something nice
    if ( klogconfig->mainWinSize.isEmpty() )
    {
        // list lines, plus header, plus bevel margins
        // Let's default to 10 lines.  A nice number.
        int height = (int)(((fh + 2*im) * 10) + (fh + 6) + 6);
        debug("Window height = %d\n", height);
        listbox->setMinimumHeight(height);

        //width += w + 5 + 15;  // + 1 per column, + scrollbar
        //we now take care of this in restoreColLayout();
        debug("Window width = %d\n", width);

        listbox->setMinimumWidth(width);

        listbox->resize(width, height);
    }

    disconnect(listbox, SIGNAL(doubleClicked(QListViewItem *)), this, 0);

    switch (klogconfig->DoubleClickAction) 
    {
        case SOURCE_HOST: // get source host name
            connect(listbox, SIGNAL(doubleClicked(QListViewItem *)),
                this, SLOT(slotSourceHostName(QListViewItem *)));
            break;
        case SOURCE_WHOIS: // get source whois info
            connect(listbox, SIGNAL(doubleClicked(QListViewItem *)),
                this, SLOT(slotSourceWhois(QListViewItem *)));
            break;
        case SOURCE_PING: // ping source host
            connect(listbox, SIGNAL(doubleClicked(QListViewItem *)),
                this, SLOT(slotSourcePing(QListViewItem *)));
            break;
        case SOURCE_TRACEROUTE: // get source traceroute info
            connect(listbox, SIGNAL(doubleClicked(QListViewItem *)),
                this, SLOT(slotSourceTraceroute(QListViewItem *)));
            break;
        case SOURCE_PORT: // get source port info
            connect(listbox, SIGNAL(doubleClicked(QListViewItem *)),
                this, SLOT(slotSANSSrcPortInfo(QListViewItem *)));
            break;
        case DEST_HOST: // get dest host name
            connect(listbox, SIGNAL(doubleClicked(QListViewItem *)),
                this, SLOT(slotDestHostName(QListViewItem *)));
            break;
        case DEST_WHOIS: // get dest whois info
            connect(listbox, SIGNAL(doubleClicked(QListViewItem *)),
                this, SLOT(slotDestWhois(QListViewItem *)));
            break;
        case DEST_PING: // ping dest host
            connect(listbox, SIGNAL(doubleClicked(QListViewItem *)),
                this, SLOT(slotDestPing(QListViewItem *)));
            break;
        case DEST_TRACEROUTE: // get dest traceroute info
            connect(listbox, SIGNAL(doubleClicked(QListViewItem *)),
                this, SLOT(slotDestTraceroute(QListViewItem *)));
            break;
        case DEST_PORT: // get dest port info
            connect(listbox, SIGNAL(doubleClicked(QListViewItem *)),
                this, SLOT(slotSANSDstPortInfo(QListViewItem *)));
            break;
        case CUSTOM_COMMAND: // run custom command
            connect(listbox, SIGNAL(doubleClicked(QListViewItem *)),
                this, SLOT(slotOpen(QListViewItem *)));
            break;
        default:
            break;
     }

    if (!timer)
    {
        timer = new QTimer(this, "timer");
        connect(timer, SIGNAL(timeout()), this, SLOT(slotTimer()));
    }
    timer->start(1000);
}

void KLogList::saveColLayout()
{
    QStringList widths, order;
    for (int i = 0; i < listbox->columns(); ++i)
    {
      widths << QString::number(listbox->columnWidth(i));
      order << QString::number(listbox->header()->mapToIndex(i));
    }
    klogconfig->ColumnWidths = widths;
    klogconfig->ColumnOrder = order;
    klogconfig->SortColumn = listbox->sortColumn();
    klogconfig->SortAscending = listbox->sortOrder();
}


int KLogList::restoreColLayout()
{
    QStringList cols = klogconfig->ColumnWidths;
    int i = 0;
    int w = 0;
    int width = 0;
    for (QStringList::ConstIterator it = cols.begin(); it != cols.end(); ++it) {
        w = slotToggleColumn(i++, (*it).toInt());
        if (w == 0)
          w++;
        width +=w;
    }
    cols = klogconfig->ColumnOrder;
    i = 0;
    for (QStringList::ConstIterator it = cols.begin(); it != cols.end(); ++it)
        listbox->header()->moveSection(i++, (*it).toInt());

    listbox->setSorting(klogconfig->SortColumn, !(klogconfig->SortAscending));

    return width + 16;
}


void KLogList::keyPressEvent(QKeyEvent *event)
{
    switch (event->key())
    {
        case Key_Enter:
        case Key_Return:
            debug("Got Enter/Return\n");
            hideButton->animateClick();
            break;

        default:
            debug("Got key, Thinking...\n");
            thinking();
            break;
    }
}


bool KLogList::eventFilter ( QObject *o, QEvent *e )
{
    if ( e->type() == QEvent::MouseButtonPress &&
        static_cast<QMouseEvent*>(e)->button() == RightButton &&
            o->isA("QHeader") )
    {
        headerPopup->popup( static_cast<QMouseEvent*>(e)->globalPos() );
        return true;
    }
    return QTlist::eventFilter(o, e);
}


//-----------------------------------------------------------------------------
// Handle RMB press, show pop up menu
void KLogList::rightButtonPressed( QListViewItem *lvi, const QPoint &, int )
{
    if (!lvi)
        return;

    if (!(lvi->isSelected())) {
        listbox->clearSelection();
    }
    listbox->setSelected( lvi, true );
    QPopupMenu *menu = new QPopupMenu(listbox);

    if(listbox->columnWidth(S_ADDR_COL))
        menu->insertItem("Get Source Host Name", SOURCE_HOST);
    if(listbox->columnWidth(S_ADDR_COL))
        menu->insertItem("Whois Source Host", SOURCE_WHOIS);
    if(listbox->columnWidth(S_ADDR_COL))
        menu->insertItem("Ping Source Host", SOURCE_PING);
    if(listbox->columnWidth(S_ADDR_COL))
        menu->insertItem("Traceroute Source Host", SOURCE_TRACEROUTE);
    if(listbox->columnWidth(S_PORT_COL))
        menu->insertItem("Check Source Port Info at SANS...", SOURCE_PORT);
    menu->insertSeparator();
    if(listbox->columnWidth(D_ADDR_COL))
        menu->insertItem("Get Dest. Host Name", DEST_HOST);
    if(listbox->columnWidth(D_ADDR_COL))
        menu->insertItem("Whois Dest. Host", DEST_WHOIS);
    if(listbox->columnWidth(D_ADDR_COL))
        menu->insertItem("Ping Dest. Host", DEST_PING);
    if(listbox->columnWidth(D_ADDR_COL))
        menu->insertItem("Traceroute Dest. Host", DEST_TRACEROUTE);
    if(listbox->columnWidth(D_PORT_COL))
        menu->insertItem("Check Dest. Port Info at SANS...", DEST_PORT);
    menu->insertSeparator();
    menu->insertItem("Run Command", CUSTOM_COMMAND);

    if ( ((KLogItem*)lvi)->srcPortNum < 0 )
        menu->setItemEnabled(SOURCE_PORT, false);
    if ( ((KLogItem*)lvi)->dstPortNum < 0 )
        menu->setItemEnabled(DEST_PORT, false);

    if (klogconfig->COMMAND.stripWhiteSpace() == "")
        menu->setItemEnabled(CUSTOM_COMMAND, false);

    switch (menu->exec(QCursor::pos(), 0) ) {
        case SOURCE_HOST:
            slotSourceHostName(lvi);
            break;
        case SOURCE_WHOIS:
            slotSourceWhois(lvi);
            break;
        case SOURCE_PING:
            slotSourcePing(lvi);
            break;
        case SOURCE_TRACEROUTE:
            slotSourceTraceroute(lvi);
            break;
        case SOURCE_PORT:
            slotSANSSrcPortInfo(lvi);
            break;
        case DEST_HOST:
            slotDestHostName(lvi);
            break;
        case DEST_WHOIS:
            slotDestWhois(lvi);
            break;
        case DEST_PING:
            slotDestPing(lvi);
            break;
        case DEST_TRACEROUTE:
            slotDestTraceroute(lvi);
            break;
        case DEST_PORT:
            slotSANSDstPortInfo(lvi);
            break;
        case CUSTOM_COMMAND:
            slotOpen(lvi);
            break;
        default:
            break;
    }
    delete menu;
}


void KLogList::slotSourceHostName(QListViewItem *lvi)
{
    if (!lvi)
        return;

    const QString tempaddr = QString("%1.%2.%3.%4")
        .arg(((KLogItem*)lvi)->srcIpNum[0])
        .arg(((KLogItem*)lvi)->srcIpNum[1])
        .arg(((KLogItem*)lvi)->srcIpNum[2])
        .arg(((KLogItem*)lvi)->srcIpNum[3]);

    KLogDialog *sourceDialog = new KLogDialog(false, this);
    sourceDialog->getHostName(tempaddr, true);
}


void KLogList::slotDestHostName(QListViewItem *lvi)
{
    if (!lvi)
        return;

    QString tempaddr = QString("%1.%2.%3.%4")
        .arg(((KLogItem*)lvi)->dstIpNum[0])
        .arg(((KLogItem*)lvi)->dstIpNum[1])
        .arg(((KLogItem*)lvi)->dstIpNum[2])
        .arg(((KLogItem*)lvi)->dstIpNum[3]);

    KLogDialog *destDialog = new KLogDialog(false, this);
    destDialog->getHostName(tempaddr, false);
}


void KLogList::slotSourceWhois(QListViewItem *lvi)
{
    if (!lvi)
        return;

    const QString tempaddr = QString("%1.%2.%3.%4")
        .arg(((KLogItem*)lvi)->srcIpNum[0])
        .arg(((KLogItem*)lvi)->srcIpNum[1])
        .arg(((KLogItem*)lvi)->srcIpNum[2])
        .arg(((KLogItem*)lvi)->srcIpNum[3]);
    const KLogIP *packet = ((KLogItem *)lvi)->getPacket();

    KLogDialog *newCommand = new KLogDialog(true, this);
    QString tempCmd = "whois " + tempaddr;
    newCommand->runCommand(packet, tempCmd, true);
}


void KLogList::slotDestWhois(QListViewItem *lvi)
{
    if (!lvi)
        return;

    const QString tempaddr = QString("%1.%2.%3.%4")
        .arg(((KLogItem*)lvi)->dstIpNum[0])
        .arg(((KLogItem*)lvi)->dstIpNum[1])
        .arg(((KLogItem*)lvi)->dstIpNum[2])
        .arg(((KLogItem*)lvi)->dstIpNum[3]);
    const KLogIP *packet = ((KLogItem *)lvi)->getPacket();

    KLogDialog *newCommand = new KLogDialog(true, this);
    QString tempCmd = "whois " + tempaddr;
    newCommand->runCommand(packet, tempCmd, true);
}


void KLogList::slotSourcePing(QListViewItem *lvi)
{
    if (!lvi)
        return;

    const QString tempaddr = QString("%1.%2.%3.%4")
        .arg(((KLogItem*)lvi)->srcIpNum[0])
        .arg(((KLogItem*)lvi)->srcIpNum[1])
        .arg(((KLogItem*)lvi)->srcIpNum[2])
        .arg(((KLogItem*)lvi)->srcIpNum[3]);
    const KLogIP *packet = ((KLogItem *)lvi)->getPacket();

    KLogDialog *newCommand = new KLogDialog(true, this);
    QString tempCmd = "ping -c 10 " + tempaddr;
    newCommand->runCommand(packet, tempCmd, true);
}


void KLogList::slotDestPing(QListViewItem *lvi)
{
    if (!lvi)
        return;

    const QString tempaddr = QString("%1.%2.%3.%4")
        .arg(((KLogItem*)lvi)->dstIpNum[0])
        .arg(((KLogItem*)lvi)->dstIpNum[1])
        .arg(((KLogItem*)lvi)->dstIpNum[2])
        .arg(((KLogItem*)lvi)->dstIpNum[3]);
    const KLogIP *packet = ((KLogItem *)lvi)->getPacket();

    KLogDialog *newCommand = new KLogDialog(true, this);
    QString tempCmd = "ping -c 10 " + tempaddr;
    newCommand->runCommand(packet, tempCmd, true);
}


void KLogList::slotSourceTraceroute(QListViewItem *lvi)
{
    if (!lvi)
        return;

    const QString tempaddr = QString("%1.%2.%3.%4")
        .arg(((KLogItem*)lvi)->srcIpNum[0])
        .arg(((KLogItem*)lvi)->srcIpNum[1])
        .arg(((KLogItem*)lvi)->srcIpNum[2])
        .arg(((KLogItem*)lvi)->srcIpNum[3]);
    const KLogIP *packet = ((KLogItem *)lvi)->getPacket();

    KLogDialog *newCommand = new KLogDialog(true, this);
    QString tempCmd = "traceroute " + tempaddr;
    newCommand->runCommand(packet, tempCmd, true);
}


void KLogList::slotDestTraceroute(QListViewItem *lvi)
{
    if (!lvi)
        return;

    const QString tempaddr = QString("%1.%2.%3.%4")
        .arg(((KLogItem*)lvi)->dstIpNum[0])
        .arg(((KLogItem*)lvi)->dstIpNum[1])
        .arg(((KLogItem*)lvi)->dstIpNum[2])
        .arg(((KLogItem*)lvi)->dstIpNum[3]);
    const KLogIP *packet = ((KLogItem *)lvi)->getPacket();

    KLogDialog *newCommand = new KLogDialog(true, this);
    QString tempCmd = "traceroute " + tempaddr;
    newCommand->runCommand(packet, tempCmd, true);
}


void KLogList::slotSANSSrcPortInfo(QListViewItem *lvi)
{
    kapp->invokeBrowser( QString("http://isc.sans.org/port_details.php?port=%1").arg(((KLogItem*)lvi)->srcPortNum));
}

void KLogList::slotSANSDstPortInfo(QListViewItem *lvi)
{
    kapp->invokeBrowser( QString("http://isc.sans.org/port_details.php?port=%1").arg(((KLogItem*)lvi)->dstPortNum));
}


bool KLogList::addNew()
{
    if ( (++pollCount < klogconfig->POLLTIME) || !tailer )
    {
        return false;  // Not time to look yet
    }

    pollCount = 0;
    bool added = false;
    mutex.lock();

    for (int count = 0; count < MAXNEWROWS; count++)
    {
        const KLogLine *line = tailer->getLogLine();

        if (line == NULL)
        {
            debug("."); fflush(stdout);
            break;
        }

        debug("++++ packet ++++\n");

        if (line->isPacket())
        {
            KLogItem *item;

            totalPacketCount++;
            uncheckedPacketCount++;

            item = new KLogItem(listbox, line);

            item->isNew = true;
            item->packetNumber = totalPacketCount;

            if (listbox->childCount() > klogconfig->BUFFSIZE)
            {
                QListViewItemIterator it( listbox );
                while ( it.current() ) {

                    if ( ((KLogItem*)it.current())->packetNumber == oldestPacket) {
                        debug("Removing oldest item\n");
                        delete(it.current());
                        oldestPacket++;
                        break;
                    }
                    ++it;
                }
            }

            if (!(listbox->isVisible()))
            {
                listbox->setCurrentItem(item);
                listbox->ensureItemVisible(item);
                debug("Made item visible\n");
            }
            else
            {
                debug("Not scrolling... window visible\n");
            }

            char label[64];
            sprintf(label, "%lu packets detected (%lu new)",
            totalPacketCount, uncheckedPacketCount);
            packetCount->setText(label);

            added = true;
        }
        else
        {
            debug("Deleting non-packet line\n");
            delete line;
            count--;
        }
    }

    if (added)
    {
        emit sigShow(false);
        tray->setTrayIcon(true);

        // This fixes the first time popup focus bug
        static bool first = true;

        if (first)
        {
            sleep(1);
            emit sigShow(false);
            first = false;
        }
    }

    mutex.unlock();
    return added;
}


int KLogList::slotToggleColumn(int id, int mode)
{
    bool show = false;
    int  width = 0;

    show  = listbox->columnWidth(id);
    QStringList cols = klogconfig->ColumnWidths;
    int i = 0;

    for (QStringList::ConstIterator it = cols.begin(); it != cols.end(); ++it)
    {
        if (id == i++)
        width = (*it).toInt();
    }

    if (mode == -1)
        if (show){
            show = false;
        }
        else {
            show = true;
        }
    else {
        show = mode;
    }

    headerPopup->setItemChecked(id, show);

    if (show) {
        listbox->header()->setResizeEnabled(true, id);
        cols = klogconfig->StandardWidths;
        i = 0;
        if (width == 0)
        {
            for (QStringList::ConstIterator it = cols.begin(); it != cols.end(); ++it)
            {
                if (id == i++)
                width = (*it).toInt();
            }
        }
        listbox->setColumnWidth(id, width);
    }
    else {
        listbox->header()->setResizeEnabled(false, id);
        listbox->header()->setStretchEnabled(false, id);
        listbox->hideColumn(id);
    }

    if (mode == -1)
        saveColLayout();
    return width;
}


void KLogList::thinking()
{
    debug("Thinking\n");
    state = THINKING;
}

void KLogList::running()
{
    debug("Running\n");
    state = RUNNING;
}

void KLogList::markOldPackets() {
    debug("clearNewEntries\n");
    QListViewItemIterator it(listbox);
    QListViewItem *item;

    for (; (item = it.current()); it++)
    {
      ((KLogItem *)item)->isNew = false;
    }
    uncheckedPacketCount = 0;

    char label[64];
    sprintf(label, "%lu packets detected (%lu new)", totalPacketCount, uncheckedPacketCount);
    packetCount->setText(label);

    tray->setTrayIcon(false);
}


void KLogList::slotTimer()
{
    if (KLogTerm())
    {
        slotExit();
        return;
    }

    if (KLogShow())
    {
        emit sigShow(true); // force popup for signal
        thinking();
        showCount = 0;
    }

    if (pauseButton->state() == QButton::On)
    {
        if (++pauseCount > klogconfig->PAUSETIME)
        {
            pauseButton->setText("Pause");
            pauseButton->setOn(false);
            pauseCount = 0;
        }
        else
        {
            return;  // No processing while paused
        }
    }

    if (state == THINKING)
    {
        addNew();
    }
    else
    {
        if (addNew())
        {
            showCount = 0;
        }

        if (!isVisible())
        {
            showCount = 0;
        }
    }

    return;
}

void KLogList::slotHide()
{
    emit sigHide();
    running();
    markOldPackets();
}

void KLogList::slotPause()
{
    debug("Pause state is %s\n",
    pauseButton->state() == QButton::On ? "ON" : "OFF");

    if (pauseButton->state() == QButton::Off)
    {
        state = THINKING;
        pauseCount = 0;
        pauseButton->setText("Pause");

        int n = listbox->childCount();

        if (n == 0)
        {
            packetCount->setText("No packets detected");
        }
        else
        {
            char label[64];
            sprintf(label, "%lu packets detected (%lu new)", totalPacketCount, uncheckedPacketCount);
            packetCount->setText(label);
        }
    }
    else
    {
        pauseCount = 0;
        pauseButton->setText("Paused");
        packetCount->setText("Packet detection paused");
    }
}

void KLogList::slotClear()
{
    thinking();
    listbox->clear();
    uncheckedPacketCount = 0;
    totalPacketCount = 0;
    oldestPacket = 1;
    tray->setTrayIcon(false);
    packetCount->setText("No packets detected");
}

void KLogList::slotClick(QListViewItem *item)
{
    debug("Clicked\n");
    thinking();
    if (item != 0) {
      if ( ((KLogItem*)item)->isNew ) {
          ((KLogItem*)item)->isNew = false;
          uncheckedPacketCount--;
          char label[64];
          sprintf(label, "%lu packets detected (%lu new)", totalPacketCount, uncheckedPacketCount);
          packetCount->setText(label);
      }
      if(uncheckedPacketCount < 1)
          tray->setTrayIcon(false);
      else
          tray->setTrayIcon(true);
    }

}

void KLogList::slotOpen(QListViewItem *item)
{
    const KLogIP *packet = ((KLogItem *)item)->getPacket();
    KLogDialog *newCommand = new KLogDialog(true, this);
    QString tempCmd = klogconfig->COMMAND.latin1();
    if (klogconfig->COMMANDDIALOG)
        newCommand->runCommand(packet, tempCmd, true);
    else
        newCommand->runCommand(packet, tempCmd, false);
}

void KLogList::saveTo(QTextStream &s)
{
    mutex.lock();
    thinking();

    QListViewItemIterator it(listbox);
    QListViewItem *item;

    for (; (item = it.current()); it++)
    {
        const KLogIP *packet = ((KLogItem *)item)->getPacket();
        if (klogconfig->SAVERAW)
            s << packet->getLine() << "\n";
        else
            s << packet->toString(true) << "\n";
    }

    mutex.unlock();
    return;
}

void KLogList::printTo(KPrinter &print)
{
    int x = 0, y = 20;
    QPainter paint;
    thinking();

    paint.begin(&print);
    paint.setFont(klogconfig->PRINTFONT);

    QPaintDeviceMetrics pm(&print);
    int maxy = pm.height();
    int maxx = pm.width();
    debug("max y offset = %d\n", maxy);
    QFontMetrics fm = paint.fontMetrics();
    int fh = fm.height();
    debug("font height = %d\n", fh);
    maxy -= fh;   // make sure we print last line!

    mutex.lock();
    QListViewItemIterator it(listbox);
    QListViewItem *item;

    for (; (item = it.current()); it++)
    {
        const KLogIP *packet = ((KLogItem *)item)->getPacket();
        QString text;

        if (klogconfig->SAVERAW)
            text = packet->getLine();
        else
            text = packet->toString(true);

        QRect textRect = paint.boundingRect(x, y, maxx, maxy, Qt::WordBreak, text);
        paint.drawText(textRect, Qt::WordBreak, text);
        y += textRect.height();

        if (y > maxy)
        {
            y = 20;
            print.newPage();
        }
    }

    mutex.unlock();
    paint.end();

    return;
}

void KLogList::slotExit()
{
    emit sigExit();
}

// eof
