/*
 * Klogwatch, Netfilter log monitor
 *
 * Copyright (C) 2004 Nick Battle <nick.battle@freeuk.com>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <signal.h>
#include <unistd.h>
#include <dirent.h>
#include <stdio.h>
#include "debug.h"

static bool klogterm = false;
static bool klogshow = false;

static void catcher(int sig)
{
    debug("Ouch! Signal %d\n", sig);

    if ((sig == SIGTERM) || (sig == SIGINT))
        klogterm = true;
    else if (sig == SIGUSR1)
        klogshow = true;
}

static int getPids(pid_t *pids, int count)
{
    DIR *d;
    struct dirent *p;
    char path[64];
    struct stat sb;
    int i = 0;

    pid_t myPid = getpid();
    sprintf(path, "/proc/%d/exe", myPid);

    if (stat(path, &sb) < 0)
        return 0;

    ino_t myInode = sb.st_ino;
    debug("MyInode = %d\n", (int)myInode);

    if ((d = opendir("/proc")) == NULL)
        return 0;

    while ((p = readdir(d)))
    {
        pid_t pid;
        if (sscanf(p->d_name, "%d", &pid) == 1 && pid != myPid)
        {
            sprintf(path, "/proc/%d/exe", pid);

            if (stat(path, &sb) == 0)
            {
                if (sb.st_ino == myInode)
                {
                    debug("pid %d = %d\n", i, pid);
                    pids[i++] = pid;

                    if (i >= count)
                        break;
                }
            }
        }
    }

    closedir(d);

    return i;
}

void KLogSigInit(void)
{
    struct sigaction actions;

    actions.sa_handler = catcher;
    actions.sa_flags = 0;
    sigemptyset(&actions.sa_mask);

    if (sigaction(SIGTERM, &actions, NULL) < 0)
        perror("Failed");

    if (sigaction(SIGINT, &actions, NULL) < 0)
        perror("Failed");

    if (sigaction(SIGUSR1, &actions, NULL) < 0)
        perror("Failed");

    debug("Signal handler set\n");
}

#define MAXPIDS 10

int KLogWatchKill(int sig)
{
    pid_t p[MAXPIDS];
    int n = getPids(p, MAXPIDS);

    for (int i=0; i<n; i++)
    {
        debug("Signalling %d with %d\n", p[i], sig);
        kill(p[i], sig);
    }

    return (n > 0) ? 0 : 1;
}

bool KLogWatchIsRunning(void)
{
    pid_t p[MAXPIDS];
    int n = getPids(p, MAXPIDS);
    return (n > 0);
}

bool KLogTerm(void)
{
    bool was = klogterm;
    klogterm = false;
    return was;
}

bool KLogShow(void)
{
    bool was = klogshow;
    klogshow = false;
    return was;
}

// eof
