/*
 * Klogwatch, Netfilter log monitor
 *
 * Copyright (C) 2004 Nick Battle <nick.battle@freeuk.com>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 */

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "debug.h"
#include "klogtail.h"

KLogTail::KLogTail(const char *file)
{
    path = QString(file);
    fd = 0;
    inode = 0;
    size = 0;
}

KLogTail::~KLogTail(void)
{
    close(fd);
}

int KLogTail::setup()
{
    if (!statFile())
    {
        return errno;
    }

    lseek(fd, 0, SEEK_END);
    return 0;
}

const KLogLine *KLogTail::getLogLine(void)
{
    QString s;
    int retries = 3;

    while (true)
    {
        char c;
        int n;

        n = read(fd, &c, 1);

        if (n < 0)
        {
            perror("KLogTail::getLogLine read:");
            return NULL;
        }

        if (n == 0)
        {
            if (s.isEmpty())
            {
                if (statFile()) 
                {
                    // We want to continue to examine the new log file
                    continue;
                }

                return NULL;  // Nothing new
            }

            if (retries--)
            {
                debug("Read partial line?\n");
                sleep(1);
            }
            else
            {
                debug("Returning partial line\n");
                s.append('\n');
                return new KLogLine(s);
            }
        }
        else if (c == '\n')
        {
            return new KLogLine(s);
        }
        else
        {
            s.append(c);
        }
    }
}

bool KLogTail::statFile(void)
{
    struct stat sb;

    if (stat(path.latin1(), &sb) < 0)
    {
        return false;
    }
    else
    {
        if ((sb.st_ino != inode) || (sb.st_size < size))
        {
            if (fd) close(fd);
            const char *p = path.latin1();
            fd = open(p, O_RDONLY);
            struct stat buf;

            if ((fd >= 0) && (fstat(fd, &buf) >= 0) && (S_ISREG(buf.st_mode)))
            {
                debug("Tailing file <%s>\n", p);

                inode = sb.st_ino;
                size = sb.st_size;
                debug("New inode %d, size = %d\n", (int)inode, (int)size);
                return true;
            }
            else
                return false;  //unable to read file
        }

        size = sb.st_size;
        return false; // Same file
    }
}

// eof
