/*
 * Klogwatch, Netfilter log monitor
 *
 * Copyright (C) 2004 Nick Battle <nick.battle@freeuk.com>
 * Copyright (C) 2006 John Stamp <jstamp@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 */

#include <kaboutdata.h>
#include <kcmdlineargs.h>
#include <kfiledialog.h>
#include <khelpmenu.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kpushbutton.h>
#include <kuniqueapplication.h>

#include "debug.h"
#include "klogsetup.h"
#include "klogwatch.h"

KLogWatch::KLogWatch(const char *logfile) :
    KMainWindow(0, 0,
    WType_TopLevel | WDestructiveClose | WStyle_ContextHelp)
{
    file = logfile;
    kloglist = new KLogList(file, this);
    setCentralWidget(kloglist);

    connect(kloglist, SIGNAL(sigHide()), this, SLOT(slotHide()));
    connect(kloglist, SIGNAL(sigShow(bool)), this, SLOT(slotShow(bool)));
    connect(kloglist, SIGNAL(sigExit()), this, SLOT(slotExit()));
    connect(kapp, SIGNAL(aboutToQuit()), this, SLOT(slotSaveConfig()));

    QString caption("Netfilter Log Events from ");
    setCaption(caption + file);
    KIconLoader *i = KGlobal::iconLoader();
    menubar = menuBar();
    QPopupMenu *filePopup = new QPopupMenu();

    filePopup->insertItem(i->loadIcon("filesave", KIcon::Small),
                "&Save", this, SLOT(slotSave()), CTRL+Key_S);

    filePopup->insertItem(i->loadIcon("configure", KIcon::Small),
                "S&etup", this, SLOT(slotSetup()), CTRL+Key_C);

    filePopup->insertItem(i->loadIcon("printer1", KIcon::Small),
                "&Print", this, SLOT(slotPrint()), CTRL+Key_P);

    filePopup->insertItem(i->loadIcon("exit", KIcon::Small),
                 "&Quit", this, SLOT(slotExit()), CTRL+Key_Q);

    menubar->insertItem("&File", filePopup);

    KAboutData *aboutData = new KAboutData(
        "klogwatch",
        I18N_NOOP("KLogWatch"),
        VERSION,
        I18N_NOOP("A simple Netfilter log file monitor"),
        KAboutData::License_GPL,
        "(c) 2003-2006, Nick Battle\n(c) 2006, John Stamp",
        I18N_NOOP(
            "Klogwatch is a monitor for displaying packets logged to\n"
            "a file by the Netfilter package. Packets logged are\n"
            "displayed in a popup window."),
        "http://klogwatch.sourceforge.net/",
        "jstamp@users.sourceforge.net");

    aboutData->addAuthor("Nick Battle",
     "Original author", "nick.battle@freeuk.com");
    aboutData->addAuthor("John Stamp",
     "Enhancements in 1.7 and 1.8\nMaintenance of 2.x", "jstamp@users.sourceforge.net"); 

    KHelpMenu *help = new KHelpMenu(this, aboutData);
    KPopupMenu *helpPopup = help->menu();
    menubar->insertItem("&Help", (QPopupMenu *)helpPopup);

    if ( klogconfig->mainWinSize.isEmpty() )
    {
        resize(minimumSizeHint());    // force minimal size
        // a bit hackish, but we need to let a person decrease the width after
        // the startup width is set.
        kloglist->listbox->setMinimumWidth(0);
    }
    else
    {
        resize(klogconfig->mainWinSize);
        move(klogconfig->mainWinPos);
    }
}

KLogWatch::~KLogWatch()
{
    debug("Destroy KLogWatch\n");
}

void KLogWatch::slotHide()
{
    debug("Hiding\n");
    hide();
}

void KLogWatch::slotShow(bool force)
{
    debug("Showing\n");

    if (force)
    {
        debug("Setting active focus\n");
        kloglist->setFocus();
        show();
        raise();
        setActiveWindow();
    }
}

void KLogWatch::resizeEvent(QResizeEvent*)
{
    klogconfig->mainWinSize = size();
}

void KLogWatch::showEvent(QShowEvent*)
{
    klogconfig->mainWinSize = size();
    klogconfig->mainWinPos = pos();

    // Make all buttons same width
    // Only does this the first time the main window is shown
    // Relies on qtlist.ui having the default text set to "Paused"
    if ( (kloglist->pauseButton->state() == QButton::Off) && (kloglist->pauseButton->text() == "Paused") )
    {
        int buttonWidth = kloglist->pauseButton->width();
        kloglist->pauseButton->setText("Pause");
        kloglist->clearButton->setMinimumWidth(buttonWidth);
        kloglist->pauseButton->setMinimumWidth(buttonWidth);
        kloglist->hideButton->setMinimumWidth(buttonWidth);
    }
}

void KLogWatch::moveEvent(QMoveEvent*)
{
    klogconfig->mainWinPos = pos();
}

void KLogWatch::slotSetup()
{
    bool repeat = true;
    debug("Setup\n");

    kloglist->saveColLayout();

    while (repeat)
    {
        KLogSetup *dialog = new KLogSetup(this);
        int choice = dialog->exec();
        delete dialog;

        if (choice == QDialog::Rejected)
            break;

        file = klogconfig->FILENAME;

        debug("got filename\n");
        QString caption("Netfilter Log Events from ");
        setCaption(caption + file);

        kloglist->applySettings(file, this);
        repeat = false;
        break;
    }
}

void KLogWatch::slotSave()
{
    debug("Saving\n");
    kloglist->thinking();

    QString filename =
    KFileDialog::getSaveFileName(QString::null, QString::null, this);

    if (!filename.isNull())
    {
        QFile file(filename);

        if (file.open(IO_WriteOnly))
        {
            QTextStream s(&file);
            kloglist->saveTo(s);
            file.close();
        }
        else
        {
            KMessageBox::sorry(this,
                QString("Cannot open file '%1'").arg(filename));
        }
    }
}

void KLogWatch::slotPrint()
{
    debug("Print\n");
    kloglist->thinking();
    KPrinter print;

    if (print.setup(this))
    {
        kloglist->printTo(print);
    }
}

void KLogWatch::slotExit()
{
    debug("Exiting\n");
    kapp->exit(0);
}

void KLogWatch::slotSaveConfig()
{
    kloglist->saveColLayout();
    klogconfig->save();
}

void KLogWatch::closeEvent( QCloseEvent *e ){
  e->ignore();
  kloglist->hideButton->animateClick();
}

// eof
