/***************************************************************************
 *   Copyright (C) 2005 by Krzysztof Zawadyl							   *
 *   k.zawadyl@gmail.com												   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "configthemes.h"
#include "popuptheme.h"
#include "mconfiguration.h"

#include <qlistbox.h>
#include <qlabel.h>
#include <qpushbutton.h>
#include <qbuttongroup.h>
#include <qstringlist.h>

#include <kfiledialog.h>
#include <klocale.h>
#include <ktar.h>
#include <kstandarddirs.h>
#include <kmessagebox.h>
#include <karchive.h>

ConfigThemes::ConfigThemes(QWidget *parent, const char *name)
    :Form1(parent, name)
{
	updateValues();
	connect( themesListBox, SIGNAL(highlighted(int)),
	         this, SLOT(showThemeInfo(int)) );
	connect( addThemeButton, SIGNAL(clicked()),
	         this, SLOT(loadTheme()) );
	connect( scanButton, SIGNAL(clicked()),
	         this, SLOT(scanForThemes()) );
	connect( deleteThemeButton, SIGNAL(clicked()),
	         this, SLOT(deleteTheme()) );
}

void ConfigThemes::updateValues()
{
	scanForThemes();
	enableThemesBox->setChecked( (Config().themesEnabled && Config().currentTheme) );
}

void ConfigThemes::updateConfig()
{
	bool themes = enableThemesBox->isChecked();

	if( themes && themesListBox->count() )
	{
		Config().themesEnabled = true;
		Config().currentTheme =
			Config().themesList.at( themesListBox->currentItem() );
		Config().currentThemeName = Config().currentTheme->themeName;
	}
	else
	{
		Config().themesEnabled = false;
		Config().currentTheme = 0;
		Config().currentThemeName = "";
	}
}


void ConfigThemes::showThemeInfo( int i )
{
	PopupTheme *theme = themesList.at( i );
	if( !theme )
	{
		qDebug( "Oops! No current theme!" );
		return;
	}

	if( !theme->themeAuthor.isEmpty() )
		authorLabel->setText( theme->themeAuthor );
	else
		authorLabel->setText( i18n("No information") );
	
	if( !theme->themeVersion.isEmpty() )
		versionLabel->setText( theme->themeVersion );
	else
		versionLabel->setText( i18n("No information") );
	
	if( !theme->themeDescription.isEmpty() )
		descriptionLabel->setText( theme->themeDescription );
	else
		descriptionLabel->setText( i18n("No information") );
	
	if( !theme->previewPix.isNull() )
	{
		previevLabel->setPixmap( theme->previewPix );
		QSize s( 300, 180 );
		previevLabel->setMaximumSize( s );
	}
	else
		previevLabel->setText( i18n("No preview available") );
}

/*
PopupTheme *getTheme( QListBoxItem *item )
{
	int i = 0;
	i = themesListBox->index( item );
	return themesList.at( i );
}*/

void ConfigThemes::loadTheme()
{
	QString fn = KFileDialog::getOpenFileName( QString::null,
	                                           "*.tgz|" + i18n("MetaMonitor theme archive") + " (*.tgz)",
	                                           this,
	                                           i18n( "Choose theme file" ) );
	if( !fn.isEmpty() )
	{

		QString localDir = ::locateLocal( "appdata", "themes/" );

		KTar archive( fn );
		if ( !archive.open( IO_ReadOnly ) )
			return;
		
		const KArchiveDirectory *archiveDir = archive.directory();
		
		QString name = *archiveDir->entries().at( 0 );
		if( name.isEmpty() )
		{
			KMessageBox::error( this, i18n("Couldn't load theme"), i18n("MetaMonitor" ) );
			return;
		}			

		if( Config().getTheme( name ) )
		{
			KMessageBox::error( this, i18n("Theme already exists"), i18n("MetaMonitor" ) );
			return;
		}
		    
		archiveDir->copyTo( localDir );
		archive.close();

		//ok, now check if the theme can be loaded
		scanForThemes();
		
		if( Config().getTheme( name ) )
		{
			//everything's fine
			int i = Config().themesList.find( Config().getTheme( name ) );
			themesListBox->setSelected( i, true );
			showThemeInfo( i );
		}
		else
		{
			KMessageBox::error( this, i18n("Couldn't load theme"), i18n("MetaMonitor" ) );

			// ok, theme loading failed, so let's remove the directory
			QString path(localDir + "/" + name);
			KIO::Job *j = KIO::del( path );

			// we don't care about the result of this operation...
			//connect( j, SIGNAL(result(KIO::Job*)),this,SLOT(deleteThemeFinished(KIO::Job*)) );

			return;
		}
	}
}

void ConfigThemes::scanForThemes()
{
	Config().scanForThemes();
	
	themesList.clear();
	themesList = Config().themesList;
	
	themesListBox->clear();
	
	PopupTheme *theme;
	QPtrListIterator<PopupTheme> it( themesList );
	while ( (theme=it.current()) )
	{
		themesListBox->insertItem( theme->themeName );
		++it;
	}
	
	if( !themesListBox->count() )
	{
		authorLabel->setText( " " );
		versionLabel->setText( " " );
		descriptionLabel->setText( " " );
		previevLabel->setText( " " );
		
		enableThemesBox->setChecked( false );
	}
	else
	{
		enableThemesBox->setChecked( true );
		int i = Config().themesList.find( Config().currentTheme );
		themesListBox->setSelected( i, true );
		showThemeInfo( i );
	}
}

void ConfigThemes::deleteTheme()
{
	QListBoxItem *it = themesListBox->selectedItem();
	if( !it )
	{
		KMessageBox::error( this, i18n("You have to select a theme"), i18n("MetaMonitor" ) );
		return;
	}

	if( KMessageBox::questionYesNo( this,
	                                i18n("Are you sure to delete this theme?"),
	                                i18n("MetaMonitor") ) ==  KMessageBox::Yes )
	{
		KIO::Job *j = Config().deleteTheme( it->text() );
		connect( j, SIGNAL(result(KIO::Job*)), this, SLOT(deleteThemeFinished(KIO::Job*)) );
	}
}

void ConfigThemes::deleteThemeFinished( KIO::Job *j )
{
	if( j->error() )
	{
		QStringList errList = j->detailedErrorStrings();
		QString s = *errList.at(1);
		//s += *errList.at( 2 );
		KMessageBox::detailedError( this, i18n("Couldn't delete theme"), s, i18n("Error" ) );
	}
	
	scanForThemes();
}
#include "configthemes.moc"
