/***************************************************************************
 *   Copyright (C) 2005 by Krzysztof Zawadyl                               *
 *   k.zawadyl@gmail.com                                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <unistd.h>

#include "filemonitor.h"
#include "message.h"
#include "mfile.h"

#include <qfile.h>
#include <qptrlist.h>
#include <qregexp.h>

#include <kmessagebox.h>
#include <klocale.h>

/**
 * @brief Constructor
 * 
 * Creates a monitor for a single file.
 * 
 * @param p pointer to the parent widget
 * @param fName file name
 */
FileMonitor::FileMonitor( MetaMonitor *p, const QString &fName, int interval )
{
	parent = p;
	mUpdateInterval = interval;
	
	mFile = new MFile( fName );
	connect( mFile, SIGNAL( reOpened() ), this, SLOT( fileReopened() ) );	
	connect( mFile, SIGNAL( lost() ), this, SLOT( fileLost() ) );	
	
	mTimer = new QTimer( this, "timer" );
	connect( mTimer, SIGNAL( timeout() ), this, SLOT( update(void) ) );
	
	init(false);
}

/**
 * @brief Destructor
 * 
 * Closes monitored filemonitor
 */
FileMonitor::~FileMonitor()
{
	if( mTimer )
		delete mTimer;
	
	if( mFile )
		delete mFile;
}

/**
 * @brief change the interval between monitor updates
 * 
 * @param interval The interval
 */
void FileMonitor::changeInterval( int interval )
{
	mUpdateInterval = interval;
	if( mTimer )
		mTimer->changeInterval(mUpdateInterval);
}

/**
 * @brief monitor's initialisation 
 * 
 * Opens a file and goes to the end of it. Pops an error message if fails
 *
 * @param quiet if true, the FileMonitor will popup the MessageBox when error 
 * @return true if success, false if not
 */
bool FileMonitor::init(bool quiet)
{
	if( !mFile->isOpen() )
		mFile->open();
	
	if( mFile->isOpen() )
	{
		mFile->goToEnd();
		mTimer->start( mUpdateInterval );
		return true;
	}
	
	if( !quiet )
		KMessageBox::detailedError( parent, 
		                            i18n( "Cannot open a file" ), 
		                            i18n( "%1: %2" ).arg( mFile->name(), mFile->errorString() ), 
		                            i18n( "Error" ) );
	return false;
}

/**
 * @brief Message's fetching
 * 
 * Fetches new messages and puts'em to the list
 *
 * @param mList list of messages to store the new message object
 * @return count of new messages
 */
void FileMonitor::update()
{
	QString line = 0;
	
	// if file is closed, try to reopen it
	if( !mFile->isOpen() )
	{
		if( !init( true ) )
			return;
		else
			fileReopened();
	}
	
	while( (line=mFile->getLogLine()) != 0 )
	{
		QString from;
		QString mess;
		QString time;
		
		if( line.contains( "Last output repeated" ) )
		{
			from = i18n( "Log daemon" );
			mess = line.stripWhiteSpace();
		}
		else
		{
			QRegExp rx;
			
			if( Config().defaultRegExp == MConfiguration::MetaLog )
				rx.setPattern( "^([^:]+(?::\\d{2}){2})\\s\\[([^\\[]+)\\](.*)$" );
			else
				if( Config().defaultRegExp == MConfiguration::SysLog )
					rx.setPattern( "^([^:]+(?::\\d{2}){2})\\s*(\\S*|\\S*\\s\\S*:)\\s*(.*)$" );
			else
				rx.setPattern( Config().customRegExp );
			
			rx.search( line );
			time = rx.cap( 1 );
			from = rx.cap( 2 );
			
			// get rid of trash
			if( from.right(1) == ":" )
				from.truncate( from.length()-1 );
			
			mess = rx.cap( 3 ).stripWhiteSpace();
		}
		
		Message *m = new Message( from, mess );
		emit messageReceived( m );
	}
}

/**
 * @brief File lost reaction
 * 
 * Sends a message about losing a file
 */
void FileMonitor::fileLost()
{
	mTimer->stop();
	
	KMessageBox::detailedError( parent, 
	                            i18n( "Log file lost!" ), 
	                            i18n( "%1: %2" ).arg( mFile->name(), mFile->errorString() ), 
	                            i18n( "Error" ) );
	
	mFile->close();
	
	Message *m = new Message( i18n("MetaMonitor"), "Log file lost!" );
	emit messageReceived( m );
}

/**
 * @brief File reopen reaction
 * 
 * Sends a message about reopening a file. Reopen can happen when
 * log daemon rotates the log files.
 */
void FileMonitor::fileReopened()
{
	mFile->goToEnd();
	Message *m = new Message( i18n("MetaMonitor"), "Log file reopened" );
	emit messageReceived( m );
}
#include "filemonitor.moc"
