/***************************************************************************
 *   Copyright (C) 2005 by Krzysztof Zawadyl							   *
 *   k.zawadyl@gmail.com												   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "mconfiguration.h"
#include "metamonitor.h"

#include <qdir.h>

#include <kapplication.h>
#include <kconfig.h>
#include <kstandarddirs.h>
#include <kurl.h>
#include <kio/job.h>

MConfiguration::MConfiguration()
{
	triggerList.clear();
}

MConfiguration::~MConfiguration()
{
	triggerList.clear();
}

void MConfiguration::read()
{
	KConfig *cfg = kapp->config();
	
	cfg->setGroup("General Options");
	groupMessages	= cfg->readBoolEntry( "GroupMessages", true );
	pausePopups		= cfg->readBoolEntry( "PausePopups", false );
	logFile 		= cfg->readEntry( "LogFile", "/var/log/everything/current" );
	updateInterval	= cfg->readNumEntry( "UpdateInterval", 2000 );
	popupsLifetime	= cfg->readNumEntry( "PopupsLifetime", 4000 );
	defaultRegExp   = (LogDaemonType)cfg->readNumEntry( "DefaultRegExpType", MetaLog );
	customRegExp    = cfg->readEntry( "CustomRegExp", "" );
	maxMessages		= cfg->readNumEntry( "MaxMessages", 0 );
	soundEnabled	= cfg->readBoolEntry( "SoundEnabled", true );
	soundPlayerCmd 	= cfg->readEntry( "SoundPlayerCommand", "artsplay" );
	resetPopupsTimers=cfg->readBoolEntry( "ResetPopupsTimers", false );
	currentThemeName= cfg->readEntry( "CurrentTheme", "" );
	themesEnabled 	= cfg->readBoolEntry( "ThemesEnabled", false );
	
	triggerList.clear();
	loadTriggers( cfg );
	
	scanForThemes();
}

void MConfiguration::write()
{
	KConfig *cfg = kapp->config();
	
	cfg->setGroup("General Options");
	cfg->writeEntry( "GroupMessages", groupMessages );
	cfg->writeEntry( "UpdateInterval", updateInterval );
	cfg->writeEntry( "PopupsLifetime", popupsLifetime );
	cfg->writeEntry( "PausePopups", pausePopups );
	cfg->writeEntry( "LogFile", logFile );
	cfg->writeEntry( "DefaultRegExpType", defaultRegExp );
	cfg->writeEntry( "CustomRegExp", customRegExp );
	cfg->writeEntry( "MaxMessages", maxMessages );
	cfg->writeEntry( "SoundEnabled", soundEnabled );
	cfg->writeEntry( "SoundPlayerCommand", soundPlayerCmd );
	cfg->writeEntry( "ResetPopupsTimers", resetPopupsTimers );
	
	if( currentTheme )
		cfg->writeEntry( "CurrentTheme", currentTheme->themeName );
	else
		cfg->writeEntry( "CurrentTheme", "" );
	
	cfg->writeEntry( "ThemesEnabled", themesEnabled );	
	writeTriggers( cfg );
	
	cfg->sync();
}

void MConfiguration::loadTriggers( KConfig *cfg )
{
	QString s;
	for( int i=0; ; i++ )
	{
		s.sprintf( "Trigger %d", i );
		if( !cfg->hasGroup( s ) )
			break;
		
		MTrigger *mt = new MTrigger();
		QFont f = mm->font();
		QColor c = mm->foregroundColor();
		
		cfg->setGroup( s );
		mt->checkSenderName = cfg->readBoolEntry( "CheckSenderName", false );
		mt->senderName 		= cfg->readEntry( "SenderName", QString::null );
		mt->senderCompType = (MTrigger::CompType)cfg->readNumEntry( "SenderCompType", MTrigger::ExactComp );
		
		mt->checkMessageText = cfg->readBoolEntry( "CheckMessageText", false );
		mt->messageText = cfg->readEntry( "MessageText", QString::null );
		mt->messageCompType = (MTrigger::CompType)cfg->readNumEntry( "MessageCompType", MTrigger::ExactComp );
		
		mt->dontShowPopups	= cfg->readBoolEntry( "DontShowPopups", false );
		
		mt->setCustomIcon 	= cfg->readBoolEntry( "SetCustomIcon", false );
		mt->setPopupsFont 	= cfg->readBoolEntry( "SetPopupsFont", false );
		mt->setPopupsLifeTime = cfg->readBoolEntry( "SetPopupsLifeTime", false );
		mt->setPopupsTextColor 	= cfg->readBoolEntry( "SetPopupsTextColor", false );
		mt->playSound		= cfg->readBoolEntry( "PlaySound", false );
		mt->runCommand		= cfg->readBoolEntry( "RunCommand", false );
		
		mt->customIcon 		= cfg->readEntry( "CustomIcon", QString::null );
		mt->popupsFont 		= cfg->readFontEntry( "PopupsFont", &f );
		mt->popupsTextColor = cfg->readColorEntry( "PopupsTextColor", &c );
		mt->popupsLifeTime 	= cfg->readNumEntry( "PopupsLifeTime", popupsLifetime );
		mt->soundFile 		= cfg->readEntry( "SoundFile", "" );
		mt->command			= cfg->readEntry( "Command", QString::null );
		mt->buttonsLabel 	= cfg->readEntry( "CmdButtonsLabel", QString::null );
		
		/**
         * @todo: trigger's sanity checking
         */
		triggerList.append( mt );
	}
}

void MConfiguration::writeTriggers( KConfig *cfg )
{
	QString s;
	
    // delete old, non existent triggers
	for( int i=0; ;i++ )
	{
		s.sprintf( "Trigger %d", i );
		if( cfg->hasGroup( s ) )
			cfg->deleteGroup( s, true );
		else
			break;
	}
	
	for( unsigned int i=0; i<triggerList.count(); i++ )
	{
		s.sprintf( "Trigger %d", i );
		cfg->setGroup( s );
		
		MTrigger *mt = triggerList.at(i);
		
		cfg->writeEntry( "CheckSenderName", mt->checkSenderName );
		if( mt->checkSenderName )
		{
			cfg->writeEntry( "SenderName", mt->senderName );
			cfg->writeEntry( "SenderCompType", mt->senderCompType );
		}
		
		cfg->writeEntry( "CheckMessageText", mt->checkMessageText );
		if( mt->checkMessageText )
		{
			cfg->writeEntry( "MessageText", mt->messageText );
			cfg->writeEntry( "MessageCompType", mt->messageCompType );
		}
		
		cfg->writeEntry( "DontShowPopups", mt->dontShowPopups );
		if( !mt->dontShowPopups )
		{
			cfg->writeEntry( "SetPopupsFont", mt->setPopupsFont );
			if( mt->setPopupsFont )
				cfg->writeEntry( "PopupsFont", mt->popupsFont );
			
			cfg->writeEntry( "SetPopupsTextColor", mt->setPopupsTextColor );
			if( mt->setPopupsTextColor )
				cfg->writeEntry( "PopupsTextColor", mt->popupsTextColor );
			
			cfg->writeEntry( "SetPopupsLifeTime", mt->setPopupsLifeTime );
			if( mt->setPopupsLifeTime )
				cfg->writeEntry( "PopupsLifeTime", mt->popupsLifeTime );
			
			cfg->writeEntry( "SetCustomIcon", mt->setCustomIcon );
			if( mt->setCustomIcon )
				cfg->writeEntry( "CustomIcon", mt->customIcon );
		}
		
		cfg->writeEntry( "PlaySound", mt->playSound );
		if( mt->playSound )
			cfg->writeEntry( "SoundFile", mt->soundFile );
		
		cfg->writeEntry( "RunCommand", mt->runCommand );
		if( mt->runCommand )
		{
			cfg->writeEntry( "Command", mt->command );
			cfg->writeEntry( "CmdButtonsLabel", mt->buttonsLabel );
		}
	}
}

void MConfiguration::scanForThemes()
{
	themesList.clear();

	// all KDE dirs
	QStringList dirList = KGlobal::dirs()->findDirs( "appdata", "themes/" );
	QStringList::Iterator it = dirList.begin();
	while( *it != 0 )
	{
		KStandardDirs::makeDir( *it );
		
		// every theme in the directory
		QDir dir( *it );
		QStringList tList = dir.entryList( QDir::Dirs );
		QStringList::Iterator themIt = tList.begin();
		while( *themIt != 0 )
		{
			QString name = *themIt;
			if( name != "." && name != ".." )
			{
				if( !getTheme( name ) )
				{
					PopupTheme *pm = new PopupTheme( name );
					if( pm->load(dir.path()+"/") )
						themesList.append( pm );
					else
						delete pm;
				}
			}
			++themIt;
		}
		++it;
	}

	currentTheme = getTheme( currentThemeName );
	if( !currentTheme )
		currentTheme = themesList.getFirst();
}

PopupTheme* MConfiguration::getTheme( QString name )
{
	QPtrListIterator<PopupTheme> it( themesList );
	PopupTheme *pm;
	while( (pm = it.current()) != 0 )
	{
		if( !pm->themeName.isEmpty() && pm->themeName == name )
			return pm;
		++it;
	}
	
	return 0;
}

KIO::Job *MConfiguration::deleteTheme( QString name )
{
	PopupTheme *pm = getTheme( name );
	if( !pm )
		return 0;

	QDir themeDir( pm->path );
	if( !themeDir.exists() )
		return 0;
	
	KURL themePath(pm->path);

	KIO::Job *j = KIO::del(themePath);
	return j;
}

MConfiguration& Config()
{
	static MConfiguration conf;
	return conf;
}

