/***************************************************************************
 *   Copyright (C) 2005 by Krzysztof Zawadyl                               *
 *   k.zawadyl@gmail.com                                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "metamonitor.h"
#include "popup.h"
#include "message.h"
#include "popupmessage.h"

#include <qtimer.h>
#include <qregexp.h>
#include <qpainter.h>
#include <qcursor.h>

#include <kdialog.h>
#include <klocale.h>
#include <kiconloader.h>
#include <kmdcodec.h>

/**
 * @brief Constructor
 * 
 * @param m Messages pointer
 * @param parent Pointer to the parent widget
 * @param name PopupMessage's name
 */
PopupMessage::PopupMessage(Message *m, Popup *parent, const char *name) :
				QWidget(parent, name, WX11BypassWM)
{
	message = m;
	p = parent;
	lifeTime = 4000;
	
	setFocusPolicy(NoFocus);
	setBackgroundOrigin( ParentOrigin );
	setMouseTracking(true);
	
	cmdProcess = new KProcess(this);
	QString anchor;
	// command's anchor
	if( !message->getCommand().isEmpty() )
	{
		QString cmd = m->getCommand();
		QString label = m->getButtonsLabel();
		
		cmd.replace( "%s", m->getFrom() );
		cmd.replace( "%t", m->getText() );
		label.replace( "%s", m->getFrom() );
		label.replace( "%t", m->getText() );
		
        // split the command but leave the commas as they are
		QRegExp rx("([^\\s'\"]+|'[^']+'|\"[^\"]+\")");
		int pos = 0;
		while ( pos >= 0 )
		{
			pos = rx.search( cmd, pos );
			
			if ( pos > -1 )
			{
				command += rx.cap( 1 );
				pos  += rx.matchedLength();
			}
		}

		anchor = "<a href=\"#\">"+label+"</a>";
		connect( this, SIGNAL( anchorClicked() ), this, SLOT( runCommand() ) );
	}
	


	/**
	 * @TODO let user to set the @CMD anchor position
	 */
	// get the message template
	QString msgTxt = parent->getBody();

	// add an anchor to message's text
	QString text = message->getText();
	if( !anchor.isEmpty() )
		text += "<br />"+anchor;
	
	// replace template tags with message's values
	msgTxt.replace( "%IMG", "<img width=22 height=22 src=\""+getIconPath( message->getIcon() )+"\" />" );
	msgTxt.replace( "%TEXT", text );
	if( parent->getSenderColor(message) != parent->getTextColor(message) )
		msgTxt.replace( "%SENDER", "<font color=\""+parent->getSenderColor(message)+"\">"+
		                			i18n( "Message from %1:" ).arg( message->getFrom() )+
		                			"</font>");
	else
		msgTxt.replace( "%SENDER", i18n( "Message from %1:" ).arg( message->getFrom() ) );
	
	// build the richText
	richText = new QSimpleRichText( "<qt text=\""+parent->getTextColor(message)+"\">"+msgTxt+"</qt>",
	                                message->getFont(),
	                                "",//context
	                                0,//QStyleSheet
	                                0,//QMimeSourceFactory
	                                -1, //nopageBreak
	                                Qt::blue,//linkColor
	                                true//underlineLinks
	                              );
	
	int w = (richText->width() > richText->widthUsed()) ? richText->width() : richText->widthUsed();
	setMinimumWidth( w );
	richText->setWidth( w );
	setMinimumHeight( richText->height() );
	
	hide();
	
	hideTimer = new QTimer( this, "hide_timer" );
	connect( hideTimer, SIGNAL( timeout() ), SLOT( timerDone() ) );
}

/**
 * @brief Destructor
 */
PopupMessage::~PopupMessage()
{
	if( hideTimer )
		delete hideTimer;
	
	if( cmdProcess )
		delete cmdProcess;
}

void PopupMessage::mouseReleaseEvent(QMouseEvent *e)
{
	if( richText->anchorAt( e->pos() ) )
		emit anchorClicked();
	else
	{
		emit clicked();
		emit clicked(this);
	}
}

void PopupMessage::msgClick()
{
	emit clicked();
	emit clicked(this);
}

void PopupMessage::setTimeout( int timeout )
{
	lifeTime = timeout;
}

/**
 * @brief Start's timer
 * 
 * Starts hide timer. If the timer is already started it does nothing.
 */
void PopupMessage::startTimer()
{
	if( lifeTime )
		hideTimer->start( lifeTime, true );
}

void PopupMessage::resetTimer()
{
	hideTimer->stop();
	startTimer();
}

/**
 * @brief Timer timeout reaction
 * 
 * Hides the PopupMessage's Widget and marks itself as expired. Then it tells parent
 * to show messages once again.
 */
void PopupMessage::timerDone()
{
	emit timeout();
	emit timeout(this);
}

void PopupMessage::runCommand()
{
	cmdProcess->clearArguments();
	*cmdProcess << command;
	
	if( !cmdProcess->start(KProcess::DontCare) )
	{
		qDebug( "Failed to run cmdProcess" );
		QValueList<QCString> a = cmdProcess->args();
		QValueList<QCString>::Iterator it; ;
		for ( it=a.begin(); it!=a.end(); ++it )
		{
			qDebug( *it );
		}
	}
}

void PopupMessage::paintEvent( QPaintEvent *e )
{
	QPainter msgPainter( this );

	richText->setWidth( e->rect().width() );
	QRect r = e->rect();
	//r.addCoords( 0, 0, 2, 2 );
	

	// draw shadow
	//QColorGroup cg = colorGroup();
	//cg.setColor(QColorGroup::Text, QColor( 128, 128, 128 ) );
	//shadowRichText->draw( &msgPainter, x+1, y+1, QRect(), cg );
	
	//msgPainter.drawRect( r );

	// draw richText
	QColorGroup cg = colorGroup();
	richText->draw( &msgPainter, 0, 0, r, cg );
	msgPainter.end();
}

QString PopupMessage::getIconPath( const QString &name )
{
	KGlobal::iconLoader()->loadIcon( name, KIcon::MainToolbar, 22 );
	QString s( KGlobal::iconLoader()->iconPath( name, KIcon::MainToolbar, false ) );
	return s;
}

void PopupMessage::resetWidth( int w )
{
	//	richText->adjustSize();
	if( w > minimumWidth() )
	{
		richText->setWidth( w );
		//shadowRichText->setWidth( w );
		int w = (richText->width() > richText->widthUsed()) ? richText->width() : richText->widthUsed();
		resize( w, richText->height() );
	}
}

void PopupMessage::mouseMoveEvent(QMouseEvent *e)
{
	//qDebug( QString( "MoveEvent: %1 %2" ).arg( e->pos().x() ).arg( e->pos().y() ) );
	
	if(richText->anchorAt( e->pos() ) && cursor().shape() != PointingHandCursor)
		setCursor( PointingHandCursor );
	else
		if( cursor().shape() != ArrowCursor )
			setCursor( ArrowCursor );
}
#include "popupmessage.moc"
