/***************************************************************************
 *                                                                         *
 *   copyright (C) 2003 by Michael Buesch                                  *
 *   email: mbuesch@freenet.de                                             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License version 2        *
 *   as published by the Free Software Foundation.                         *
 *                                                                         *
 ***************************************************************************/


#include <qlineedit.h>
#include <qlabel.h>
#include <qfiledialog.h>
#include <qmessagebox.h>
#include <qclipboard.h>
#include <qcheckbox.h>
#include <qpushbutton.h>
#include <qradiobutton.h>
#include <qtextedit.h>
#include <qfileinfo.h>
#include <qprogressbar.h>
#include <qdragobject.h>
#include <qcombobox.h>
#include <qtabwidget.h>
#include <qcombobox.h>
#include <qcursor.h>

#include <klocale.h>
#include <kaboutapplication.h>
#include <kwin.h>
#include <kurl.h>
#include <kurldrag.h>
#include <kiconloader.h>

#include "calcchecksum.h"
#include "comparewndimpl.h"
#include "err.h"
#include "csabout.h"


CalcChecksum::CalcChecksum(KCmdLineArgs *args, QWidget *parent, const char *name)
 : mainwnd(parent, name)
{
	confFile = 0;
	comm = 0;
	tray = 0;
	calls = 0;
	curRetBuf = 0;
	hf = 0;
	calculating = false;
	quitFromTray = false;
	actOnAreaSignals = false;

	keyMouseEventEater = new KeyMouseEventEater(this);

	// open/read config file
	confFile = new KConfig(CALCCHECKSUM_CONFIG_FILE, false, false);
	confFile->setGroup("GLOBAL");

	// init UI
	setFixedSize(size());
	setCaption(QString("CalcChecksum   ") +
		   i18n("version") + " " VERSION);
	// init about-tab
	CsAbout *aboutTab = new CsAbout(this);
	aboutTab->resize(aboutDlgPlaceholder->size());
	aboutTab->reparent(aboutDlgPlaceholder, 0, QPoint(0, 0));
	aboutTab->show();

	try {
		comm = new ThreadComm;
		if (!comm) {
			int e = 0;
			throw e;
		}
		calls = new NotBlockingCalls(comm->getSock());
		if (!calls) {
			int e = 0;
			throw e;
		}
	} catch (int e) {
		string err("Failed to initialize interprocess-communication!\n"
			   "You may have not enough memory to run CalcChecksum. "
			   "Quit some other programs.\n"
			   "CalcChecksum will be terminated now!");
		printError(err);
		exit(1);
	}
	connect(comm, SIGNAL(lineAvailable_signal(const string*)), this, SLOT(handleProgressRequest(const string*)));

	hf = new HashFile(calls, this);

	// initialize systray
	tray = new CalcChecksumTray(this);
	KIconLoader icons("calcchecksum");
	tray->setPixmap(icons.loadIcon("calcchecksum", KIcon::Small));

	// replace "calc-now-button". It's ugly, but it works. :)
	QPoint btnPos(calcButton->pos());
	QSize btnSize(calcButton->size());
	delete calcButton;
	calcButton = new MonitoredPushButton(i18n("CALCULATE NOW!"), this);
	calcButton->move(btnPos);
	calcButton->resize(btnSize);
	connect(calcButton, SIGNAL(clicked()), this, SLOT(calcButton_slot()));
	connect(calcButton, SIGNAL(enterBoundary()), this, SLOT(enterAllowedArea_slot()));
	connect(calcButton, SIGNAL(leaveBoundary()), this, SLOT(leaveAllowedArea_slot()));
	// replace "quit-button"
	btnPos = quitButton->pos();
	btnSize = quitButton->size();
	delete quitButton;
	quitButton = new MonitoredPushButton(i18n("Quit"), this);
	quitButton->move(btnPos);
	quitButton->resize(btnSize);
	connect(quitButton, SIGNAL(clicked()), this, SLOT(quitButton_slot()));
	connect(quitButton, SIGNAL(enterBoundary()), this, SLOT(enterAllowedArea_slot()));
	connect(quitButton, SIGNAL(leaveBoundary()), this, SLOT(leaveAllowedArea_slot()));

	readConf();
	// Note: connect the config_* slots ALWAYS after readConf()!
	connect(config_alwaysOnTop, SIGNAL(stateChanged(int)), this, SLOT(confOptionChanged_slot()));
	connect(config_showTray, SIGNAL(stateChanged(int)), this, SLOT(confOptionChanged_slot()));
	connect(config_autoCopy, SIGNAL(stateChanged(int)), this, SLOT(confOptionChanged_slot()));

	// handle commandline
	handleArgs(args);
}

CalcChecksum::~CalcChecksum()
{
	delete_ifnot_null(confFile);
	delete_ifnot_null(comm);
	delete_ifnot_null(calls);
	delete_ifnot_null(hf);
}


void MonitoredPushButton::mouseMoveEvent(QMouseEvent *e)
{
	const int y = e->y(), x = e->x();
	if (y < 0 || x < 0 ||
	    y >= height() || x >= width()) {
		emit leaveBoundary();
	} else {
		emit enterBoundary();
	}
}


void CalcChecksum::initTodo()
{
	todo.md4.calc = false;
	todo.md4.ret = "";

	todo.md5.calc = false;
	todo.md5.ret = "";

	todo.sha160.calc = false;
	todo.sha160.ret = "";

	todo.sha256.calc = false;
	todo.sha256.ret = "";

	todo.sha384.calc = false;
	todo.sha384.ret = "";

	todo.sha512.calc = false;
	todo.sha512.ret = "";

	todo.crc32.calc = false;
	todo.crc32.ret = "";

	todo.rmd160.calc = false;
	todo.rmd160.ret = "";

	todo.tiger192.calc = false;
	todo.tiger192.ret = "";

	todo.haval128_3.calc = false;
	todo.haval128_3.ret = "";

	todo.haval128_4.calc = false;
	todo.haval128_4.ret = "";

	todo.haval128_5.calc = false;
	todo.haval128_5.ret = "";

	todo.haval160_3.calc = false;
	todo.haval160_3.ret = "";

	todo.haval160_4.calc = false;
	todo.haval160_4.ret = "";

	todo.haval160_5.calc = false;
	todo.haval160_5.ret = "";

	todo.haval192_3.calc = false;
	todo.haval192_3.ret = "";

	todo.haval192_4.calc = false;
	todo.haval192_4.ret = "";

	todo.haval192_5.calc = false;
	todo.haval192_5.ret = "";

	todo.haval224_3.calc = false;
	todo.haval224_3.ret = "";

	todo.haval224_4.calc = false;
	todo.haval224_4.ret = "";

	todo.haval224_5.calc = false;
	todo.haval224_5.ret = "";

	todo.haval256_3.calc = false;
	todo.haval256_3.ret = "";

	todo.haval256_4.calc = false;
	todo.haval256_4.ret = "";

	todo.haval256_5.calc = false;
	todo.haval256_5.ret = "";
}

void CalcChecksum::fillTodo()
{
	if (md4->isChecked())
		todo.md4.calc = true;
	if (md5->isChecked())
		todo.md5.calc = true;
	if (crc32->isChecked())
		todo.crc32.calc = true;
	if (rmd160->isChecked())
		todo.rmd160.calc = true;
	if (tiger192->isChecked())
		todo.tiger192.calc = true;
	if (sha160->isChecked())
		todo.sha160.calc = true;
	if (sha256->isChecked())
		todo.sha256.calc = true;
	if (sha384->isChecked())
		todo.sha384.calc = true;
	if (sha512->isChecked())
		todo.sha512.calc = true;
	if (haval128_3->isChecked())
		todo.haval128_3.calc = true;
	if (haval128_4->isChecked())
		todo.haval128_4.calc = true;
	if (haval128_5->isChecked())
		todo.haval128_5.calc = true;
	if (haval160_3->isChecked())
		todo.haval160_3.calc = true;
	if (haval160_4->isChecked())
		todo.haval160_4.calc = true;
	if (haval160_5->isChecked())
		todo.haval160_5.calc = true;
	if (haval192_3->isChecked())
		todo.haval192_3.calc = true;
	if (haval192_4->isChecked())
		todo.haval192_4.calc = true;
	if (haval192_5->isChecked())
		todo.haval192_5.calc = true;
	if (haval224_3->isChecked())
		todo.haval224_3.calc = true;
	if (haval224_4->isChecked())
		todo.haval224_4.calc = true;
	if (haval224_5->isChecked())
		todo.haval224_5.calc = true;
	if (haval256_3->isChecked())
		todo.haval256_3.calc = true;
	if (haval256_4->isChecked())
		todo.haval256_4.calc = true;
	if (haval256_5->isChecked())
		todo.haval256_5.calc = true;
}

bool CalcChecksum::walkThroughTodo()
{
	CALCCHECKSUM_ASSERT(comm);
	CALCCHECKSUM_ASSERT(calls);

	calculating = true;
	QFile onFile;
	bool calcOnFile = false;
	if (calcOnFileRadio->isChecked()) {
		calcOnFile = true;
		onFile.setName(pathOrStringEdit->text());
	}

	QString curHash;

	if (todo.md4.calc) {
		curHash = "MD4";
		todo.md4.calc = false;
		curRetBuf = &(todo.md4.ret);
		if (calcOnFile)
			calls->calcCs(&onFile, md, 4);
		else
			calls->calcCs(pathOrStringEdit->text(), md, 4);
	} else if (todo.md5.calc) {
		curHash = "MD5";
		todo.md5.calc = false;
		curRetBuf = &(todo.md5.ret);
		if (calcOnFile)
			calls->calcCs(&onFile, md, 5);
		else
			calls->calcCs(pathOrStringEdit->text(), md, 5);
	} else if (todo.crc32.calc) {
		curHash = "CRC 32";
		todo.crc32.calc = false;
		curRetBuf = &(todo.crc32.ret);
		if (calcOnFile)
			calls->calcCs(&onFile, crc, 32);
		else
			calls->calcCs(pathOrStringEdit->text(), crc, 32);
	} else if (todo.rmd160.calc) {
		curHash = "RIPE-MD 160";
		todo.rmd160.calc = false;
		curRetBuf = &(todo.rmd160.ret);
		if (calcOnFile)
			calls->calcCs(&onFile, rmd, 160);
		else
			calls->calcCs(pathOrStringEdit->text(), rmd, 160);
	} else if (todo.tiger192.calc) {
		curHash = "TIGER 192";
		todo.tiger192.calc = false;
		curRetBuf = &(todo.tiger192.ret);
		if (calcOnFile)
			calls->calcCs(&onFile, tiger, 192);
		else
			calls->calcCs(pathOrStringEdit->text(), tiger, 192);
	} else if (todo.sha160.calc) {
		curHash = "SHA1 (160)";
		todo.sha160.calc = false;
		curRetBuf = &(todo.sha160.ret);
		if (calcOnFile)
			calls->calcCs(&onFile, sha, 160);
		else
			calls->calcCs(pathOrStringEdit->text(), sha, 160);
	} else if (todo.sha256.calc) {
		curHash = "SHA 256";
		todo.sha256.calc = false;
		curRetBuf = &(todo.sha256.ret);
		if (calcOnFile)
			calls->calcCs(&onFile, sha, 256);
		else
			calls->calcCs(pathOrStringEdit->text(), sha, 256);
	} else if (todo.sha384.calc) {
		curHash = "SHA 384";
		todo.sha384.calc = false;
		curRetBuf = &(todo.sha384.ret);
		if (calcOnFile)
			calls->calcCs(&onFile, sha, 384);
		else
			calls->calcCs(pathOrStringEdit->text(), sha, 384);
	} else if (todo.sha512.calc) {
		curHash = "SHA 512";
		todo.sha512.calc = false;
		curRetBuf = &(todo.sha512.ret);
		if (calcOnFile)
			calls->calcCs(&onFile, sha, 512);
		else
			calls->calcCs(pathOrStringEdit->text(), sha, 512);
	} else if (todo.haval128_3.calc) {
		curHash = "HAVAL 128-3";
		todo.haval128_3.calc = false;
		curRetBuf = &(todo.haval128_3.ret);
		if (calcOnFile)
			calls->calcCs(&onFile, haval, 128, 3);
		else
			calls->calcCs(pathOrStringEdit->text(), haval, 128, 3);
	} else if (todo.haval128_4.calc) {
		curHash = "HAVAL 128-4";
		todo.haval128_4.calc = false;
		curRetBuf = &(todo.haval128_4.ret);
		if (calcOnFile)
			calls->calcCs(&onFile, haval, 128, 4);
		else
			calls->calcCs(pathOrStringEdit->text(), haval, 128, 4);
	} else if (todo.haval128_5.calc) {
		curHash = "HAVAL 128-5";
		todo.haval128_5.calc = false;
		curRetBuf = &(todo.haval128_5.ret);
		if (calcOnFile)
			calls->calcCs(&onFile, haval, 128, 5);
		else
			calls->calcCs(pathOrStringEdit->text(), haval, 128, 5);
	} else if (todo.haval160_3.calc) {
		curHash = "HAVAL 160-3";
		todo.haval160_3.calc = false;
		curRetBuf = &(todo.haval160_3.ret);
		if (calcOnFile)
			calls->calcCs(&onFile, haval, 160, 3);
		else
			calls->calcCs(pathOrStringEdit->text(), haval, 160, 3);
	} else if (todo.haval160_4.calc) {
		curHash = "HAVAL 160-4";
		todo.haval160_4.calc = false;
		curRetBuf = &(todo.haval160_4.ret);
		if (calcOnFile)
			calls->calcCs(&onFile, haval, 160, 4);
		else
			calls->calcCs(pathOrStringEdit->text(), haval, 160, 4);
	} else if (todo.haval160_5.calc) {
		curHash = "HAVAL 160-5";
		todo.haval160_5.calc = false;
		curRetBuf = &(todo.haval160_5.ret);
		if (calcOnFile)
			calls->calcCs(&onFile, haval, 160, 5);
		else
			calls->calcCs(pathOrStringEdit->text(), haval, 160, 5);
	} else if (todo.haval192_3.calc) {
		curHash = "HAVAL 192-3";
		todo.haval192_3.calc = false;
		curRetBuf = &(todo.haval192_3.ret);
		if (calcOnFile)
			calls->calcCs(&onFile, haval, 192, 3);
		else
			calls->calcCs(pathOrStringEdit->text(), haval, 192, 3);
	} else if (todo.haval192_4.calc) {
		curHash = "HAVAL 192-4";
		todo.haval192_4.calc = false;
		curRetBuf = &(todo.haval192_4.ret);
		if (calcOnFile)
			calls->calcCs(&onFile, haval, 192, 4);
		else
			calls->calcCs(pathOrStringEdit->text(), haval, 192, 4);
	} else if (todo.haval192_5.calc) {
		curHash = "HAVAL 192-5";
		todo.haval192_5.calc = false;
		curRetBuf = &(todo.haval192_5.ret);
		if (calcOnFile)
			calls->calcCs(&onFile, haval, 192, 5);
		else
			calls->calcCs(pathOrStringEdit->text(), haval, 192, 5);
	} else if (todo.haval224_3.calc) {
		curHash = "HAVAL 224-3";
		todo.haval224_3.calc = false;
		curRetBuf = &(todo.haval224_3.ret);
		if (calcOnFile)
			calls->calcCs(&onFile, haval, 224, 3);
		else
			calls->calcCs(pathOrStringEdit->text(), haval, 224, 3);
	} else if (todo.haval224_4.calc) {
		curHash = "HAVAL 224-4";
		todo.haval224_4.calc = false;
		curRetBuf = &(todo.haval224_4.ret);
		if (calcOnFile)
			calls->calcCs(&onFile, haval, 224, 4);
		else
			calls->calcCs(pathOrStringEdit->text(), haval, 224, 4);
	} else if (todo.haval224_5.calc) {
		curHash = "HAVAL 224-5";
		todo.haval224_5.calc = false;
		curRetBuf = &(todo.haval224_5.ret);
		if (calcOnFile)
			calls->calcCs(&onFile, haval, 224, 5);
		else
			calls->calcCs(pathOrStringEdit->text(), haval, 224, 5);
	} else if (todo.haval256_3.calc) {
		curHash = "HAVAL 256-3";
		todo.haval256_3.calc = false;
		curRetBuf = &(todo.haval256_3.ret);
		if (calcOnFile)
			calls->calcCs(&onFile, haval, 256, 3);
		else
			calls->calcCs(pathOrStringEdit->text(), haval, 256, 3);
	} else if (todo.haval256_4.calc) {
		curHash = "HAVAL 256-4";
		todo.haval256_4.calc = false;
		curRetBuf = &(todo.haval256_4.ret);
		if (calcOnFile)
			calls->calcCs(&onFile, haval, 256, 4);
		else
			calls->calcCs(pathOrStringEdit->text(), haval, 256, 4);
	} else if (todo.haval256_5.calc) {
		curHash = "HAVAL 256-5";
		todo.haval256_5.calc = false;
		curRetBuf = &(todo.haval256_5.ret);
		if (calcOnFile)
			calls->calcCs(&onFile, haval, 256, 5);
		else
			calls->calcCs(pathOrStringEdit->text(), haval, 256, 5);
	} else {
		curRetBuf = 0;
		calculating = false;
		statusText->setText("");
		return false;
	}
	statusText->setText(i18n("calculating ") + curHash + " ...");
	return true;
}

void CalcChecksum::analyseTodoList()
{
	QString output;

	if (todo.md4.ret != "") {
		output += QString("\nMD4:		") + todo.md4.ret.c_str();
	}
	if (todo.md5.ret != "") {
		output += QString("\nMD5:		") + todo.md5.ret.c_str();
	}
	if (todo.sha160.ret != "") {
		output += QString("\nSHA1 (160):	") + todo.sha160.ret.c_str();
	}
	if (todo.sha256.ret != "") {
		output += QString("\nSHA 256:	") + todo.sha256.ret.c_str();
	}
	if (todo.sha384.ret != "") {
		output += QString("\nSHA 384:	") + todo.sha384.ret.c_str();
	}
	if (todo.sha512.ret != "") {
		output += QString("\nSHA 512:	") + todo.sha512.ret.c_str();
	}
	if (todo.crc32.ret != "") {
		output += QString("\nCRC 32:		") + todo.crc32.ret.c_str();
	}
	if (todo.rmd160.ret != "") {
		output += QString("\nRIPE-MD 160:	") + todo.rmd160.ret.c_str();
	}
	if (todo.tiger192.ret != "") {
		output += QString("\nTIGER 192:	") + todo.tiger192.ret.c_str();
	}
	if (todo.haval128_3.ret != "") {
		output += QString("\nHAVAL 128-3:	") + todo.haval128_3.ret.c_str();
	}
	if (todo.haval128_4.ret != "") {
		output += QString("\nHAVAL 128-4:	") + todo.haval128_4.ret.c_str();
	}
	if (todo.haval128_5.ret != "") {
		output += QString("\nHAVAL 128-5:	") + todo.haval128_5.ret.c_str();
	}
	if (todo.haval160_3.ret != "") {
		output += QString("\nHAVAL 160-3:	") + todo.haval160_3.ret.c_str();
	}
	if (todo.haval160_4.ret != "") {
		output += QString("\nHAVAL 160-4:	") + todo.haval160_4.ret.c_str();
	}
	if (todo.haval160_5.ret != "") {
		output += QString("\nHAVAL 160-5:	") + todo.haval160_5.ret.c_str();
	}
	if (todo.haval192_3.ret != "") {
		output += QString("\nHAVAL 192-3:	") + todo.haval192_3.ret.c_str();
	}
	if (todo.haval192_4.ret != "") {
		output += QString("\nHAVAL 192-4:	") + todo.haval192_4.ret.c_str();
	}
	if (todo.haval192_5.ret != "") {
		output += QString("\nHAVAL 192-5:	") + todo.haval192_5.ret.c_str();
	}
	if (todo.haval224_3.ret != "") {
		output += QString("\nHAVAL 224-3:	") + todo.haval224_3.ret.c_str();
	}
	if (todo.haval224_4.ret != "") {
		output += QString("\nHAVAL 224-4:	") + todo.haval224_4.ret.c_str();
	}
	if (todo.haval224_5.ret != "") {
		output += QString("\nHAVAL 224-5:	") + todo.haval224_5.ret.c_str();
	}
	if (todo.haval256_3.ret != "") {
		output += QString("\nHAVAL 256-3:	") + todo.haval256_3.ret.c_str();
	}
	if (todo.haval256_4.ret != "") {
		output += QString("\nHAVAL 256-4:	") + todo.haval256_4.ret.c_str();
	}
	if (todo.haval256_5.ret != "") {
		output += QString("\nHAVAL 256-5:	") + todo.haval256_5.ret.c_str();
	}

	CALCCHECKSUM_ASSERT(output != "");
	output.remove(0, 1);
	showOutputWnd(output, true, OutputWndImpl::led_dont_show);
}

void CalcChecksum::showOutputWnd(QString outputStr, bool showCompareButton,
				 OutputWndImpl::led_stat showLED)
{
	OutputWndImpl *wnd = new OutputWndImpl(showCompareButton, showLED);
	wnd->setOutput(outputStr);

	wnd->setGeometry(pos().x() + 10, pos().y() + 10, 500, 400);
	wnd->setActiveWindow();

	confFile->setGroup("GLOBAL");
	if(confFile->readBoolEntry("alwaysOnTop", false))
		KWin::setState(wnd->winId(), NET::StaysOnTop);

	wnd->show();
}

void CalcChecksum::handleProgressRequest(const string* data)
{
	char c = (*data)[0];
	if (unlikely(c == 'Q')) {
		// calculated checksum is returning
		if (!calculating) {
			// calc was already stopped by user
			return;
		}
		calls->termThis(); // make sure thread is terminated
		string sum(data->substr(1, data->length()));
		CALCCHECKSUM_ASSERT(curRetBuf);
		*curRetBuf = sum;

		int tabindex = mainTab->currentPageIndex();
		if (tabindex == 0) {
			// we are doing a normal calculation
			if (!walkThroughTodo()) {
				// all calculations done
				enableInterface();
				progressBar->setProgress(0);
				analyseTodoList();
			}
			return;
		} else {
			// we are doing a list-calculation
			CALCCHECKSUM_ASSERT(hf);
			progressBar->setProgress(0);
			hf->calcNext();
			return;
		}
	} else if (unlikely(c == 'S')) {
		// Progressbar top-value (total-steps value)
		progressBar->setTotalSteps(atoi(data->substr(1,
					   data->length()).c_str()));
	} else {
		// push progressBar
		progressBar->setProgress(atoi(data->c_str()));
	}
}

void CalcChecksum::calcOnFileToggled_slot(bool on)
{
	if (on) {
		currentString = pathOrStringEdit->text();
		pathOrStringEdit->setText(currentPath);
		browseFileButton->setEnabled(true);
	}
}

void CalcChecksum::calcOnStringToggled_slot(bool on)
{
	if (on) {
		currentPath = pathOrStringEdit->text();
		pathOrStringEdit->setText(currentString);
		browseFileButton->setEnabled(false);
	}
}

void CalcChecksum::quitButton_slot()
{
	enableInterface();
	if (tray->isVisible()) {
		if (QMessageBox::information(this, i18n("Quit?"),
		    i18n("Do you want to quit CalcChecksum or "
			 "just minimize it to system-tray?"),
		    i18n("Quit"), i18n("Minimize"), QString::null) == 0) {
			tray->hide();
		}
		close();
		return;
	}
	enableInterface(false);
	close();
}

void CalcChecksum::browseFile_slot()
{
	QString path = QFileDialog::getOpenFileName();
	if(path != QString::null)
		pathOrStringEdit->setText(path);
}

void CalcChecksum::browseHashFileButton_slot()
{
	QString path = QFileDialog::getOpenFileName();
	if(path != QString::null)
		hashFile->setText(path);
}

void CalcChecksum::calcButton_slot()
{
	int tabindex = mainTab->currentPageIndex();

	if (calculating) {
		// calculation is in progress. cancel it!
		if (tabindex == 0) {
			cancelCalc();
		} else if (tabindex == 1) {
			hf->cancelCalc();
			calculating = false;
			enableInterface(true);
			progressBar->setProgress(0);
		} else {
			BUG();
		}
		statusText->setText(i18n("calculation stopped by user."));
		return;
	}

	if (tabindex == 0) {
		// normal calculation requested
		initNormalCalc();
	} else if (tabindex == 1) {
		// list-file calculation requested
		initListCalc();
	} else {
		BUG();
	}
}

void CalcChecksum::initNormalCalc()
{
	if (calcOnFileRadio->isChecked() && pathOrStringEdit->text().isEmpty()) {
		QMessageBox::information(this, i18n("sorry, nothing to to"),
					 i18n("You must first select a file you "
					      "want to calculate the checksum(s) from."),
					 QMessageBox::Ok, 0, 0);
		return;
	}
	if (calcOnStringRadio->isChecked() && pathOrStringEdit->text().isEmpty()) {
		if( QMessageBox::information(this, i18n("empty string"),
					     i18n("You haven't entered a string "
						  "to calculate the checksum(s) from.\n"
						  "Do you want to calculate the checksum(s) "
						  "from an empty string?"),
					     QMessageBox::Yes, QMessageBox::No, 0)
					     == QMessageBox::No) {
			return;
		}
	}
	if (calcOnFileRadio->isChecked()) {
		// remove "file:" from path (this is if file was drag'n'dropped there)
		if (pathOrStringEdit->text().left(5) == "file:") {
			pathOrStringEdit->setText(pathOrStringEdit->text().
				mid(5, pathOrStringEdit->text().length() - 5));
		}
		QFileInfo finfo(pathOrStringEdit->text());
		if (!finfo.isFile() || !finfo.isReadable()) {
			QMessageBox::information(this, i18n("sorry, file not readable"),
						 i18n("Sorry, this file doesn't exist or "
						      "you don't have permission to read it."),
						 QMessageBox::Ok, 0, 0);
			return;
		}
	}

	enableInterface(false);
	initTodo();
	fillTodo();
	if (!walkThroughTodo()) {
		enableInterface(true);
		QMessageBox::information(this, i18n("sorry, nothing to to"),
					 i18n("You must first select the "
					      "checksum(s) to calculate."),
					 QMessageBox::Ok, 0, 0);
		return;
	}
}

void CalcChecksum::initListCalc()
{
	if (hashFile->text().isEmpty()) {
		QMessageBox::information(this, i18n("sorry, nothing to to"),
					 i18n("You must first select the file, "
					      "that contains the checksum-list."),
					 QMessageBox::Ok, 0, 0);
		return;
	}

	// remove "file:" from path (this is if file was drag'n'dropped there)
	if (hashFile->text().left(5) == "file:") {
		hashFile->setText(hashFile->text().
			mid(5, hashFile->text().length() - 5));
	}
	QFileInfo finfo(hashFile->text());
	if (!finfo.isFile() || !finfo.isReadable()) {
		QMessageBox::information(this, i18n("sorry, file not readable"),
					 i18n("Sorry, this file doesn't exist or "
					      "you don't have permission to read it."),
					 QMessageBox::Ok, 0, 0);
		return;
	}

	hf->setFn(hashFile->text());
	hf->readNow();
}

void CalcChecksum::cancelCalc()
{
	CALCCHECKSUM_ASSERT(calculating);
	CALCCHECKSUM_ASSERT(calls);
	calls->termThis();
	calculating = false;
	enableInterface(true);
	progressBar->setProgress(0);
	statusText->setText("");
}

void CalcChecksum::enableInterface(bool enable)
{
	calcButton->setText(enable ? i18n("CALCULATE NOW!") : i18n("STOP calculating."));
	if (enable) {
		actOnAreaSignals = false;
		KApplication::kApplication()->removeEventFilter(keyMouseEventEater);
		KApplication::restoreOverrideCursor();
		pathOrStringEdit->setEnabled(true);	// workaround
		hashFile->setEnabled(true);		// workaround
	} else {
		actOnAreaSignals = true;
		KApplication::kApplication()->installEventFilter(keyMouseEventEater);
		KApplication::setOverrideCursor( QCursor(Qt::WaitCursor) );
		pathOrStringEdit->setEnabled(false);	// workaround
		hashFile->setEnabled(false);		// workaround
	}
}

void CalcChecksum::enterAllowedArea_slot()
{
	if (actOnAreaSignals) {
		KApplication::kApplication()->removeEventFilter(keyMouseEventEater);
		KApplication::restoreOverrideCursor();
	}
}

void CalcChecksum::leaveAllowedArea_slot()
{
	if (actOnAreaSignals) {
		KApplication::kApplication()->installEventFilter(keyMouseEventEater);
		KApplication::setOverrideCursor( QCursor(Qt::WaitCursor) );
	}
}

bool KeyMouseEventEater::eventFilter(QObject *, QEvent *e)
{
	if (e->type() == QEvent::MouseButtonPress ||
	    e->type() == QEvent::MouseButtonRelease ||
	    e->type() == QEvent::MouseButtonDblClick ||
	    e->type() == QEvent::Wheel ||
	    e->type() == QEvent::KeyPress ||
	    e->type() == QEvent::KeyRelease ||
	    e->type() == QEvent::Drop) {
		// eat mouse button and key press events
		return true;
	}
	return false;
}

bool CalcChecksum::handleArgs(KCmdLineArgs *args)
{/* returns true, if there were some sums to calculate (and that are calculating
    immediately after return). Otherwise returns false */
	bool calcSomeCs = false;
	if (args->count()) {
		QCString file = args->arg(0);
		QFileInfo fileInfo(file);
		if (fileInfo.isReadable()) {
			if (calculating) {
				QMessageBox::information(this, i18n("Another calculation in progress"),
				i18n("You have initiated a new calculation, but another is\n"
				"already in progress. Wait until this calculation has finished\n"
				"and try again."),
				QMessageBox::Ok, 0, 0);
				return false;
			}
			pathOrStringEdit->setText(file);
			if (args->isSet("md5")) {
				md5->setChecked(true);
				calcSomeCs = true;
			}
			if (args->isSet("crc32")) {
				crc32->setChecked(true);
				calcSomeCs = true;
			}
			if (args->isSet("sha1")) {
				sha160->setChecked(true);
				calcSomeCs = true;
			}
			if (args->isSet("rmd160")) {
				rmd160->setChecked(true);
				calcSomeCs = true;
			}
			if (args->isSet("tiger")) {
				tiger192->setChecked(true);
				calcSomeCs = true;
			}
			if (calcSomeCs) {
				initTodo();
				fillTodo();
				walkThroughTodo();
			}
		}
	}

	if (args->isSet("docked") && !tray->isVisible())
		tray->show();

	return calcSomeCs;
}

void CalcChecksum::newProcCalc(KCmdLineArgs *args)
{
	/* the user started the CalcChecksum-binary, but it was already running.
	 * Now no other instance will be started. Instead the arguments given to
	 * the new instance will be transmitted to 'this' old instance (using *args).
	 * Now the window has to be brought to foreground and the args have to be handled.
	 */
	KWin::setActiveWindow(winId()); // window to foreground
	if (!isVisible())
		show();
	handleArgs(args);
}

void CalcChecksum::closeEvent(QCloseEvent *e)
{
	CALCCHECKSUM_ASSERT(tray);
	if (tray->isVisible() && !quitFromTray) {
		hide();
	} else {
		quitFromTray = false;
		if (calculating) {
			enableInterface();
			if (QMessageBox::information(this, i18n("calculating"),
					i18n("A calculation is in progress.\n"
					"Do you really want to quit CalcChecksum?"),
					QMessageBox::Yes, QMessageBox::No, 0)
					== QMessageBox::No) {
				enableInterface(false);
				return;
			}
			/* stop current calculation.
			 * Note: I know, that this is a little bit ugly
			 * if we are calculating a list, but it really
			 * doesn't matter, because we are going to shut down anyway. :)
			 */
			cancelCalc();
		}
		e->accept();    // close now
	}
}

void CalcChecksum::trayQuitButton_slot()
{
	quitFromTray = true;
	if (!isVisible()) {
		QCloseEvent e;
		closeEvent(&e);
		if (!e.isAccepted())
			return;
	}
	close();
}

void CalcChecksum::trayUndockButton_slot()
{
	if (!isVisible())
		show();
	tray->hide();
	confFile->setGroup("GLOBAL");
	confFile->writeEntry("showInTray", false);
}

void CalcChecksum::traySelectFileButton_slot()
{
	calcOnStringRadio->setChecked(false);
	calcOnFileRadio->setChecked(true);
	mainTab->setCurrentPage(0);
	if(!isVisible())
		show();
	browseFile_slot();
}

void CalcChecksum::traySelectHashListButton_slot()
{
	mainTab->setCurrentPage(1);
	if(!isVisible())
		show();
	browseHashFileButton_slot();
}

void CalcChecksum::otherTabSelected_slot()
{
	int tabindex = mainTab->currentPageIndex();
	bool enable = true;
	if (tabindex == 2 ||	// config-tab
	    tabindex == 3 ) {	// about-tab
		enable = false;
	}
	calcButton->setEnabled(enable);

	setStatusLine("");
}

void CalcChecksum::writeConf()
{
	confFile->setGroup("GLOBAL");
	confFile->writeEntry("alwaysOnTop", config_alwaysOnTop->isChecked());
	if (config_showTray->state() != QButton::NoChange) {
		confFile->writeEntry("showInTray",
				     config_showTray->isChecked());
	}
	confFile->writeEntry("autoClip", config_autoCopy->isChecked());
}

void CalcChecksum::readConf()
{
	confFile->setGroup("GLOBAL");
	config_alwaysOnTop->setChecked(confFile->readBoolEntry("alwaysOnTop", false));
	config_showTray->setChecked(confFile->readBoolEntry("showInTray", false));
	config_autoCopy->setChecked(confFile->readBoolEntry("autoClip", false));
	if (tray->isVisible() && !confFile->readBoolEntry("showInTray", false)) {
		/* tray-icon visible but not set in config-file
		 * (calcchecksum was started with --docked)
		 */
		config_showTray->setTristate(true);
		config_showTray->setNoChange();
	}
	activateConf(false);
}

void CalcChecksum::activateConf(bool writeBefore)
{
	if (writeBefore)
		writeConf();

	if (config_alwaysOnTop->isChecked())
		KWin::setState(winId(), NET::StaysOnTop);
	else
		KWin::clearState(winId(), NET::StaysOnTop);

	if (config_showTray->isChecked())
		tray->show();
	else
		tray->hide();
}

void CalcChecksum::confOptionChanged_slot()
{
	activateConf(true);
}
