/***************************************************************************
 *                                                                         *
 *   copyright (C) 2003 by Michael Buesch                                  *
 *   email: mbuesch@freenet.de                                             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License version 2        *
 *   as published by the Free Software Foundation.                         *
 *                                                                         *
 ***************************************************************************/


#ifndef CALCCHECKSUM_H
#define CALCCHECKSUM_H

#include <stdio.h>

#include <kapp.h>
#include <kconfig.h>
#include <kcmdlineargs.h>

#include <qwidget.h>
#include <qsocketnotifier.h>
#include <qpushbutton.h>

#include "mainwnd.h"
#include "notblockingcalls.h"
#include "threadcomm.h"
#include "calcchecksumtray.h"
#include "hashfile.h"
#include "outputwndimpl.h"
#include "globalstuff.h"

#include <string>
#include <iostream>

using std::string;
using std::cout;
using std::endl;

#define CALCCHECKSUM_CONFIG_FILE	"calcchecksumrc"

/** a mouse-move-event monitored QPushButton */
class MonitoredPushButton : public QPushButton
{
	Q_OBJECT
public:
	MonitoredPushButton(const QString &text, QWidget * parent, const char * name = 0)
			 : QPushButton(text, parent, name)
			{ setMouseTracking(true); }
	~MonitoredPushButton()
			{ }
protected:
	void mouseMoveEvent(QMouseEvent *e);
signals:
	void enterBoundary();
	void leaveBoundary();
};

/** an event filter for all mouse (and key) button presses */
class KeyMouseEventEater : public QObject
{
public:
	KeyMouseEventEater(QObject *parent) : QObject(parent)
			{ }
	~KeyMouseEventEater()
			{ }
protected:
	bool eventFilter(QObject *, QEvent *e);
};

/** CalcChecksum is the base class of the project */
class CalcChecksum : public mainwnd
{
	Q_OBJECT

	/** data struct to hold all checksums, that
	  * are selected for calculation, but
	  * aren't calculated yet and to hold it's
	  * returned hashes.
	  */
	struct todo_t
	{
		struct sum
		{
			bool calc;
			string ret;
		};

		sum md4;
		sum md5;

		sum sha160;
		sum sha256;
		sum sha384;
		sum sha512;

		sum crc32;
		sum rmd160;
		sum tiger192;

		sum haval128_3;
		sum haval128_4;
		sum haval128_5;
		//<
		sum haval160_3;
		sum haval160_4;
		sum haval160_5;
		//<
		sum haval192_3;
		sum haval192_4;
		sum haval192_5;
		//<
		sum haval224_3;
		sum haval224_4;
		sum haval224_5;
		//<
		sum haval256_3;
		sum haval256_4;
		sum haval256_5;
	};

public:
	/** construtor */
	CalcChecksum(KCmdLineArgs *args, QWidget* parent = 0, const char *name = 0);
	/** destructor */
	~CalcChecksum();

	/** if a new process-instance wants to initiate a new calculation, call this */
	void newProcCalc(KCmdLineArgs *args);
	/** sets the status-line to "stat" */
	void setStatusLine(QString stat)
			{ statusText->setText(stat); }
	/** set curRetBuf */
	void setRetBuf(string *buf)
			{ curRetBuf = buf; }
	/** set "calculating" status */
	void setCalculating(bool _calculating)
			{ calculating = _calculating; }
	/** initialize and show the output window */
	void showOutputWnd(QString outputStr, bool showCompareButton,
			   OutputWndImpl::led_stat showLED);
	/** enables/disables UI, call this when starting and ending calculation */
	void enableInterface(bool enable = true);

public slots:
	/** quit button in tray pressed */
	void trayQuitButton_slot();
	/** undock button in tray pressed */
	void trayUndockButton_slot();
	/** select file button in tray pressed (for normal hash-calculation) */
	void traySelectFileButton_slot();
	/** select hash-list button in tray pressed */
	void traySelectHashListButton_slot();

	/** quit button pressed */
	void quitButton_slot();
	/** Open browse-dialog */
	void browseFile_slot();
	/** open browse-dialog for hash-list */
	void browseHashFileButton_slot();
	/** "calculate now"-button pressed. Initialize calculation */
	void calcButton_slot();
	/** the mouse cursor entered an always allowed area (for example the "stop"-button) */
	void enterAllowedArea_slot();
	/** the mouse cursor left an always allowed area */
	void leaveAllowedArea_slot();
	
protected:
	/** config-file */
	KConfig *confFile;
	/** use this to call the checksum-algorithms "multithreaded" */
	NotBlockingCalls *calls;
	/** for communication between calculating thread and this thread */
	ThreadComm *comm;
	/** hash-file class */
	HashFile *hf;
	/** holds all things, that have to be done or are done */
	todo_t todo;
	/** System-tray icon */
	CalcChecksumTray *tray;
	/** current file-path entered in 'pathOrStingEdit' */
	QString currentPath;
	/** current string entered in 'pathOrStingEdit' */
	QString currentString;
	/** is a calculation in progress? */
	bool calculating;
	/** buffer where the calculated sum of the currently running
	  * calculating should go to if it's ready
	  */
	string *curRetBuf;
	/** quit-event was received from tray */
	bool quitFromTray;
	/** act on leave/enter allowed area signals */
	bool actOnAreaSignals;
	/** mouse event eater object */
	KeyMouseEventEater *keyMouseEventEater;

protected:
	/** initialize "todo" */
	void initTodo();
	/** fills all information from GUI into the todo-list */
	void fillTodo();
	/** analyse the results of the previous calculation and display it to the user */
	void analyseTodoList();
	/** walks through the todo-list and starts the next job it can find */
	bool walkThroughTodo();
	/** cancel calculation */
	void cancelCalc();
	/** handles the command-line-arguments passed to CalcChecksum*/
	bool handleArgs(KCmdLineArgs *args);
	/** widget is going to close */
	void closeEvent(QCloseEvent *e);
	/** initialize normal calculation */
	void initNormalCalc();
	/** initialize list-file calculation */
	void initListCalc();
	/** writes the config to file */
	void writeConf();
	/** reads the config from file */
	void readConf();
	/** activates the current config (call this, after the user changed a config-option) */
	void activateConf(bool writeBefore);

protected slots:
	/** handles request in 'comm' */
	void handleProgressRequest(const string* data);
	/** checkbox 'on file' toggled */
	void calcOnFileToggled_slot(bool on);
	/** checkbox 'on sting' toggled */
	void calcOnStringToggled_slot(bool on);
	/** another mainwnd-tab has been selected */
	void otherTabSelected_slot();
	/** slot for config-option changes */
	void confOptionChanged_slot();
};

#endif
