/***************************************************************************
 *                                                                         *
 *   copyright (C) 2003, 2004 by Michael Buesch                            *
 *   email: mbuesch@freenet.de                                             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License version 2        *
 *   as published by the Free Software Foundation.                         *
 *                                                                         *
 ***************************************************************************/


#include "hashfile.h"
#include "err.h"
#include "calcchecksum.h"
#include "notblockingcalls.h"
#include "md5.h"
#include "sha1.h"
#include "growval.h"

#include <qfileinfo.h>
#include <qmessagebox.h>

#include <klocale.h>

#include <string>
using std::string;

#define MD5_HASHSTRING_LEN	(MD5_HASHLEN_BYTE * 2)
#define SHA1_HASHSTRING_LEN	(SHA1_HASHLEN_BYTE * 2)

#define MAX_LINE_LENGTH		2048
#define MAX_SCAN_LINES		2048

#define GPG_EXECUTABLE		"gpg"


HashFile::HashFile(NotBlockingCalls *_calls, CalcChecksum *_parent)
{
	CALCCHECKSUM_ASSERT(_calls);
	CALCCHECKSUM_ASSERT(_parent);
	calls = _calls;
	parent = _parent;
	file_t = not_set;
	hashLen = 0;
	tmp_curPos = 0;
	gpgStat = sig_not_checked;
	gpgProc = new KProcess;
}

HashFile::~HashFile()
{
	delete_ifnot_null(gpgProc);
}

void HashFile::setFn(const QString &fileName)
{
	fn = fileName;
	QFileInfo fi(fn);
	fDir = fi.dirPath(true);
}

void HashFile::readNow()
{
	if (!parse())
		return;
	parent->enableInterface(false);
	calcNext();
}

bool HashFile::parse()
{
	QString line;
	Q_LONG readBytes;
	const Q_ULONG maxLineLen = MAX_LINE_LENGTH;

	hashList.clear();
	QFile fd(fn);
	if (!fd.open(IO_ReadOnly)) {
		QMessageBox::critical(parent, i18n("File I/O error"),
				      i18n("Couldn't open hash-file!"),
				      QMessageBox::Ok, 0, 0);
		return false;
	}
	if (!fd.size()) {
		QMessageBox::critical(parent, i18n("Wrong file-format"),
				      i18n("This file is empty."),
				      QMessageBox::Ok, 0, 0);
		return false;
	}

	if (!guessType(&fd)) {
		return false;
	}
	checkGpgSig(&fd);

	void (HashFile::*parseLine)(const QString &);
	if (file_t == md5sum_1 ||
	    file_t == sha1sum_1) {
		parseLine = &HashFile::parseLine_format1;
	} else if (file_t == md5sum_2) {
		parseLine = &HashFile::parseLine_format2;
	} else {
		BUG();
		return false;
	}
	while (!fd.atEnd()) {
		readBytes = fd.readLine(line, maxLineLen);
		if (unlikely(readBytes <= 0 ||
			     static_cast<Q_ULONG>(readBytes) != line.length())) {
			fd.close();
			return false;
		}
		line = line.stripWhiteSpace();
		(this->*parseLine)(line);
	}
	fd.close();

	if (!hashList.size()) {
		goto format_err;
	}

	return true;

      format_err:
	QMessageBox::critical(parent, i18n("Wrong file-format"),
				      i18n("This file has not been recognized "
					   "to contain a valid md5sum or sha1sum "
					   "hash list."),
				      QMessageBox::Ok, 0, 0);
	return false;
}

void HashFile::parseLine_format1(const QString &line)
{
	/* Example line for this file-format:
	 *      cdb79ca3db1f39b1940ed58aa98da2e7  debian-30r2-i386-binary-1.iso
	 */

	entry_t newEntry;

	newEntry.hash = line.mid(0, hashLen);
	if (newEntry.hash == QString::null ||
	    newEntry.hash.length() != hashLen) {
		// wrong line-format
		return;
	}
	if (!checkIfHexVal(newEntry.hash)) {
		// wrong line-format
		return;
	}

	if (line.at(hashLen) != ' ') {
		// wrong line-format
		return;
	}
#if 0
	char binChar = line.at(hashLen + 1);
	// check if binary or text-mode
	if (binChar == '*') {
		newEntry.bin = true;
	} else if (binChar == ' ') {
		newEntry.bin = false;
	} else {
		// wrong line-format
		return;
	}
#endif

	const unsigned int spaceBytes = 2;
	const unsigned int fnBeginPos = hashLen + spaceBytes;
	unsigned int fnLen = line.length() - fnBeginPos;
	if (fnBeginPos > line.length() ||
	    fnLen < 1) {
		// wrong line-format
		return;
	}
	newEntry.fn = line.mid(fnBeginPos, fnLen);
	if (newEntry.fn.length() < 1) {
		// wrong line-format
		return;
	}

	newEntry.hashStat = not_checked;
	hashList.push_back(newEntry);
}

void HashFile::parseLine_format2(const QString &line)
{
	/* Example line for this file-format:
	 *      MD5 (5.2-RELEASE-i386-disc1.iso) = cc2c9f647850df2bf96a478f0cbf18b6
	 */

	entry_t newEntry;
	const unsigned int fnBeginPos = 5;
	unsigned int fnLen;
	int pos;

	if (line.left(5) != "MD5 (") {
		// wrong line-format
		return;
	}
	pos = line.find(") = ", fnBeginPos);
	if (pos == -1) {
		// wrong line-format
		return;
	}
	fnLen = pos - fnBeginPos;
	newEntry.fn = line.mid(fnBeginPos, fnLen);
	if (newEntry.fn.length() < 1) {
		// wrong line-format
		return;
	}
	newEntry.hash = line.right(hashLen);
	if (newEntry.hash == QString::null ||
	    newEntry.hash.length() != hashLen) {
		// wrong line-format
		return;
	}
	if (!checkIfHexVal(newEntry.hash)) {
		// wrong line-format
		return;
	}

	newEntry.hashStat = not_checked;
	hashList.push_back(newEntry);
}

void HashFile::checkGpgSig(QFile *fd)
{
	QString line;
	Q_LONG readBytes;
	const Q_ULONG maxLineLen = MAX_LINE_LENGTH;
	unsigned int cnt = 0;

	fd->at(0);
	while (!fd->atEnd()) {
		if (unlikely(cnt == MAX_SCAN_LINES))
			break;
		++cnt;
		readBytes = fd->readLine(line, maxLineLen);
		if (unlikely(readBytes <= 0 ||
			     static_cast<Q_ULONG>(readBytes) != line.length())) {
			break;
		}
		if (line.find("BEGIN PGP SIGNED MESSAGE") >= 0) {
			// GnuPG signature found.
			execGpg(fn);
			return;
		}
	}

	gpgStat = sig_none;
	fd->at(0);
}

void HashFile::execGpg(const QString &filename)
{
	int stat;

	gpgProc->clearArguments();
	*gpgProc << GPG_EXECUTABLE << "--verify" << filename;
	if (!gpgProc->start(KProcess::Block)) {
		gpgStat = sig_noGpgAvailable;
		return;
	}
	stat = gpgProc->exitStatus();
	switch (stat) {
		case 0:
			gpgStat = sig_ok;
			break;
		case 1:
			gpgStat = sig_wrong;
			break;
		case 2:
			gpgStat = sig_keyNotAvailable;
			break;
		default:
			gpgStat = sig_genericGpgError;
	}
}

bool HashFile::checkIfHexVal(const QString &val)
{
	const char hexDigits[] = "0123456789ABCDEFabcdef";
	const unsigned int numDigits = sizeof(hexDigits) - 1 /* minus NULL */;
	bool found;
	unsigned int i, j, size = val.length();
	QChar cur_ch;

	for (i = 0; i < size; ++i) {
		found = false;
		cur_ch = val.at(i);
		for (j = 0; j < numDigits; ++j) {
			if (cur_ch == hexDigits[j]) {
				found = true;
				break;
			}
		}
		if (!found)
			return false;
	}

	return true;
}

void HashFile::checkResults()
{
	CALCCHECKSUM_ASSERT(parent);
	QString output;
	GrowVal globalResult;

	output = i18n("Results for list-calculation.\n");
	output += i18n("List-file: ") + fn + "\n";
	output += i18n("Hash: ");
	switch (file_t) {
		case md5sum_1:
		case md5sum_2:
			output += "MD5";
			break;
		case sha1sum_1:
			output += "SHA1 (SHA160)";
			break;
		default:
			BUG();
			return;
	}
	output += '\n';
	if (gpgStat != sig_none) {
		output += i18n("GPG-signature: ");
		switch (gpgStat) {
		case sig_ok:
			output += i18n("Good signature");
			globalResult = 0;
			break;
		case sig_wrong:
			output += i18n("BAD signature!");
			globalResult = 2;
			break;
		case sig_keyNotAvailable:
			output += i18n("Public key not found!");
			globalResult = 1;
			break;
		case sig_noGpgAvailable:
			output += i18n("Couldn't execute \"gpg\". Is it installed?");
			globalResult = 0;
			break;
		case sig_genericGpgError:
			output += i18n("Fatal error while checking the signature.");
			globalResult = 2;
			break;
		default:
			BUG();
			return;
		}
		output += '\n';
	}

	vector<entry_t>::iterator begin = hashList.begin(),
				  end = hashList.end(),
				  i = begin;
	while (i != end) {
		output += "\n";
		switch (i->hashStat) {
			case hash_ok:
				output += i18n("hash OK	: ");
				globalResult = 0;
				break;
			case hash_wrong:
				output += i18n("hash WRONG	: ");
				globalResult = 2;
				break;
			case file_err:
				output += i18n("file not found	: ");
				globalResult = 1;
				break;
			default:
				output += "CALCCHECKSUM: INTERNAL ERROR	: ";
				BUG();
		}
		output += i->fn;
		++i;
	}
	OutputWndImpl::led_stat ledStat = OutputWndImpl::led_dont_show;
	switch (globalResult.get()) {
		case 0:
			ledStat = OutputWndImpl::led_green;
			break;
		case 1:
			ledStat = OutputWndImpl::led_yellow;
			break;
		case 2:
			ledStat = OutputWndImpl::led_red;
			break;
		default:
			BUG();
	}
	parent->showOutputWnd(output, false, ledStat);
	parent->setStatusLine("");
	parent->enableInterface();
	hashList.clear();
}

void HashFile::calcNext()
{
	CALCCHECKSUM_ASSERT(calls);
	CALCCHECKSUM_ASSERT(parent);

	checksums curCs;
	int curBit;
	QFile fd;
	QFileInfo fi;
	QString curFilename;

	// check result of last calculation, it there was one.
	if (tmp_ret != "") {
		QString curRet(tmp_ret.c_str());
		curRet = curRet.lower();
		QString curGiven(hashList[tmp_curPos - 1].hash);
		curGiven = curGiven.lower();

		if (curRet == curGiven) {
			hashList[tmp_curPos - 1].hashStat = hash_ok;
			printDebug(string(hashList[tmp_curPos - 1].fn.latin1()) + " hash OK");
		} else {
			hashList[tmp_curPos - 1].hashStat = hash_wrong;
			printDebug(string(hashList[tmp_curPos - 1].fn.latin1()) + " hash WRONG");
		}
	}

	if (tmp_curPos >= hashList.size()) {
		// all calculations done
		parent->setCalculating(false);
		tmp_curPos = 0;
		tmp_ret = "";
		checkResults();
		return;
	}

	parent->setRetBuf(&tmp_ret);
	parent->setCalculating(true);

	if (file_t == md5sum_1 ||
	    file_t == md5sum_2) {
		curCs = md;
		curBit = 5;
	} else if (file_t == sha1sum_1) {
		curCs = sha;
		curBit = 160;
	} else {
		BUG();
		parent->enableInterface();
		parent->setCalculating(false);
		parent->setRetBuf(0);
		return;
	}

	curFilename = fDir + "/" + hashList[tmp_curPos].fn;
	fi.setFile(curFilename);
	if (!fi.exists() || !fi.isFile() || !fi.isReadable()) {
		// specified file doesn't exist or isn't readable
		printDebug(string("refused calc: ") + hashList[tmp_curPos].fn.latin1());
		hashList[tmp_curPos].hashStat = file_err;
		++tmp_curPos;
		tmp_ret = "";
		calcNext();
		return;
	}
	fd.setName(curFilename);
	printDebug(string("started calc: ") + hashList[tmp_curPos].fn.latin1());;
	parent->setStatusLine(QString("calculating hash for file: ") + hashList[tmp_curPos].fn);
	++tmp_curPos;
	calls->calcCs(&fd, curCs, curBit);
}

void HashFile::cancelCalc()
{
	calls->termThis();
	tmp_ret = "";
	tmp_curPos = 0;
	parent->setStatusLine("");
	hashList.clear();
}

bool HashFile::guessType(QFile *fd)
{
	QString line;
	Q_LONG readBytes;
	const Q_ULONG maxLineLen = MAX_LINE_LENGTH;
	unsigned int cnt = 0;

	parent->setStatusLine(i18n("scanning file for file-type..."));
	fd->at(0);
	while (!fd->atEnd()) {
		if (unlikely(cnt == MAX_SCAN_LINES))
			break;
		++cnt;
		readBytes = fd->readLine(line, maxLineLen);
		if (unlikely(readBytes <= 0 ||
			     static_cast<Q_ULONG>(readBytes) != line.length())) {
			break;
		}
		line = line.stripWhiteSpace();
		if (checkType1(line))
			goto found;
		if (checkType2(line))
			goto found;
	}
	QMessageBox::critical(parent, i18n("Wrong file-format"),
				      i18n("Couldn't guess the hash-list file-format."),
				      QMessageBox::Ok, 0, 0);
	fd->at(0);
	parent->setStatusLine("");
	return false;

      found:
	fd->at(0);
	parent->setStatusLine("");
	return true;
}

bool HashFile::checkType1(const QString &line)
{
	/* Example line for this file-format:
	 *      cdb79ca3db1f39b1940ed58aa98da2e7  debian-30r2-i386-binary-1.iso
	 */

	int pos;

	pos = line.find(' ', 0, true); // find the end of the hash
	if (pos == -1) {
		return false;
	}
	if (!checkIfHexVal(line.left(pos))) {
		return false;
	}
	if (pos == MD5_HASHSTRING_LEN) {
		file_t = md5sum_1;
		hashLen = MD5_HASHSTRING_LEN;
		printDebug("HashFile::checkType1(): md5sum hash list.");
		return true;
	} else if (pos == SHA1_HASHSTRING_LEN) {
		file_t = sha1sum_1;
		hashLen = SHA1_HASHSTRING_LEN;
		printDebug("HashFile::checkType1(): sha1sum hash list.");
		return true;
	}

	return false;
}

bool HashFile::checkType2(const QString &line)
{
	/* Example line for this file-format:
	 *      MD5 (5.2-RELEASE-i386-disc1.iso) = cc2c9f647850df2bf96a478f0cbf18b6
	 */

	int pos;

	if (line.left(5) != "MD5 (") {
		return false;
	}
	pos = line.find(") = ");
	if (pos == -1) {
		return false;
	}
	QString hash(line.mid(pos + 4));
	hash = hash.stripWhiteSpace();
	if (hash.length() != MD5_HASHSTRING_LEN) {
		return false;
	}
	if (!checkIfHexVal(hash)) {
		return false;
	}

	file_t = md5sum_2;
	hashLen = MD5_HASHSTRING_LEN;
	printDebug("HashFile::checkType2(): md5sum hash list.");

	return true;
}
