/***************************************************************************
 *                                                                         *
 *   copyright (C) 2003, 2004 by Michael Buesch                            *
 *   email: mbuesch@freenet.de                                             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License version 2        *
 *   as published by the Free Software Foundation.                         *
 *                                                                         *
 ***************************************************************************/

#ifndef HASHFILE_H
#define HASHFILE_H

#include <qfile.h>

#include <kprocess.h>

#include <vector>
#include <string>

using std::vector;
using std::string;

class CalcChecksum;
class NotBlockingCalls;

/** read md5sum or shasum hash-file */
class HashFile
{
	enum hf_type
	{
		not_set = 0,
		md5sum_1,
		md5sum_2,
		sha1sum_1
	};
	enum hash_status
	{
		not_checked = 0,
		file_err,
		hash_ok,
		hash_wrong
	};
	enum gpg_status
	{
		sig_not_checked = 0,
		sig_none,
		sig_noGpgAvailable,
		sig_keyNotAvailable,
		sig_ok,
		sig_wrong,
		sig_genericGpgError
	};
	struct entry_t
	{
		QString fn;
		QString hash;
		/* read file in binary-mode? (this flag is not used yet.) */
	//	bool bin;
		hash_status hashStat;
	};

public:
	HashFile(NotBlockingCalls *_calls, CalcChecksum *_parent);
	~HashFile();

	/** set file name */
	void setFn(const QString &fileName);
	/** read hash-file and do calculations. call this to START */
	void readNow();
	/** calculates the next entry. To be called from handleProgressRequest() */
	void calcNext();
	/** cancel current calculation. */
	void cancelCalc();

protected:
	/** parent widget */
	CalcChecksum *parent;
	/** file name */
	QString fn;
	/** directory where fn is */
	QString fDir;
	/** type of the hash-file */
	hf_type file_t;
	/** the GnuPG signature status of the current file */
	gpg_status gpgStat;
	/** KProcess object for GnuPG */
	KProcess *gpgProc;
	/** length of the current hash for this file */
	unsigned int hashLen;
	/** the hash-files content, read into memory by readToMem() */
	vector<entry_t> hashList;
	/** use this to call the checksum-algorithms "multithreaded" */
	NotBlockingCalls *calls;
	/** tempoary for calculation */
	string tmp_ret;
	/** tempoary for calculation */
	unsigned int tmp_curPos;

protected:
	/** reads the files contents and stores it in memory */
	bool parse();
	/** parse a single line of format 1 */
	void parseLine_format1(const QString &line);
	/** parse a single line of format 1 */
	void parseLine_format2(const QString &line);
	/** check if the value is a hex-value */
	bool checkIfHexVal(const QString &val);
	/** check the results of checkHashes */
	void checkResults();
	/** guess the hash-type of the current file */
	bool guessType(QFile *fd);
	/** check line for type1 file */
	bool checkType1(const QString &line);
	/** check line for type2 file */
	bool checkType2(const QString &line);
	/** check the gpg signature (if available) of the hash-list */
	void checkGpgSig(QFile *fd);
	/** execute GnuPG on the current hash-list file */
	void execGpg(const QString &filename);
};

#endif
