/***************************************************************************
 *                                                                         *
 *   copyright (C) 2003, 2004 by Michael Buesch                            *
 *   email: mbuesch@freenet.de                                             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License version 2        *
 *   as published by the Free Software Foundation.                         *
 *                                                                         *
 ***************************************************************************/


#include "notblockingcalls.h"
#include "threadcomm.h"
#include "crc32.h"
#include "md5.h"
#include "sha1.h"
#include "rmd160.h"
#include "tiger.h"
#include "haval.h"
#include "md4.h"
#include "sha256.h"
#include "sha512.h"
#include "err.h"


// global variables - used to transmit data to thread
static QString	_str;		// filename or string
static bool	_strIsFp;	// "str" is file-path
static int	_commSocket[2];	// interprocess-communication socket
static int	_bit;		// checksum-length (checksum-version)
static int	_passes;	// calculation passes
static checksums _useAlgorithm;	// see enum checksums


NotBlockingCalls::NotBlockingCalls(const int *commSocket)
 : QThread()
{
	_commSocket[0] = commSocket[0];
	_commSocket[1] = commSocket[1];
}

void NotBlockingCalls::run()
{
	string ret;

	switch (_useAlgorithm) {
	case md: { // calculate MD
		if (_bit == 4) {
			MD4 md4;
			if (_strIsFp) {
				ret = md4.calcMd4(static_cast<const char *>(_str.latin1()),
						  _commSocket);
			} else {
				ret = md4.calcMd4(static_cast<const char *>(_str.latin1()));
			}
		} else { // MD5
			Md5 md5;
			if(_strIsFp) {
				ret = md5.calcMd5(static_cast<const char *>(_str.latin1()),
						  _commSocket);
			} else {
				ret = md5.calcMd5(static_cast<const char *>(_str.latin1()));
			}
		}
		break;
	} case crc: { // calculate CRC
		Crc32 crc;
		if (_strIsFp) {
			ret = crc.calcCrc32(_str.latin1(), _commSocket);
		} else {
			ret = crc.calcCrc32(_str.latin1());
		}
		break;
	} case sha: { // calculate SHA
		if (_bit == 160) {
			Sha1 sha1;
			if (_strIsFp) {
				ret = sha1.calcSha1(static_cast<const char *>(_str.latin1()),
						    _commSocket);
			} else {
				ret = sha1.calcSha1(static_cast<const char *>(_str.latin1()));
			}
		} else if (_bit == 256) {
			Sha256 sha256;
			if (_strIsFp) {
				ret = sha256.calcSha256(static_cast<const char *>(_str.latin1()),
						        _commSocket);
			} else {
				ret = sha256.calcSha256(static_cast<const char *>(_str.latin1()));
			}
		} else if (_bit == 384) {
			Sha512 sha384; // yea, that's correct. :)
			if(_strIsFp) {
				ret = sha384.calcSha384(static_cast<const char *>(_str.latin1()),
							_commSocket);
			} else {
				ret = sha384.calcSha384(static_cast<const char *>(_str.latin1()));
			}
		} else { // _bit == 512
			Sha512 sha512;
			if (_strIsFp) {
				ret = sha512.calcSha512(static_cast<const char *>(_str.latin1()),
							_commSocket);
			} else {
				ret = sha512.calcSha512(static_cast<const char *>(_str.latin1()));
			}
		}
		break;
	} case rmd: { // calculate RMD-160
		Rmd160 rmd160;
		if (_strIsFp) {
			ret = rmd160.calcRmd160(static_cast<const char *>(_str.latin1()),
						_commSocket);
		} else {
			ret = rmd160.calcRmd160(static_cast<const char *>(_str.latin1()));
		}
		break;
	} case tiger: { // calculate TIGER
		Tiger tiger;
		if (_strIsFp) {
			ret = tiger.calcTiger(static_cast<const char *>(_str.latin1()),
					      _commSocket);
		} else {
			ret = tiger.calcTiger(static_cast<const char *>(_str.latin1()));
		}
		break;
	} case haval: { // calculate HAVAL
		Haval haval;
		if (_strIsFp) {
			ret = haval.calcHaval(static_cast<const char *>(_str.latin1()),
					      _bit, _passes, _commSocket);
		} else {
			ret = haval.calcHaval(static_cast<const char *>(_str.latin1()),
					      _bit, _passes);
		}
		break;
	} default: {
		// sorry, checksum not implemented
		BUG();
		return;
	}}

	retCs(ret);
}

void NotBlockingCalls::retCs(string &ret)
{
	ret = string("Q") + ret + "\n"; // format return-string
	ThreadComm comm(_commSocket);
	comm.send(&ret);
}

bool NotBlockingCalls::termThis()
{
	struct timespec timeout;
	timeout.tv_sec = 0;
	timeout.tv_nsec = 1000;

	terminate();
	while (running()) {
		// wait until thread finished
		nanosleep(&timeout, 0);
	}
	return true;
}

void NotBlockingCalls::calcCs(const QString &str, checksums algo, int bit, int passes)
{
	_str = str;
	_strIsFp = false;
	_bit = bit;
	_passes = passes;
	_useAlgorithm = algo;
	start();
}

void NotBlockingCalls::calcCs(const QFile *file, checksums algo, int bit, int passes)
{
	CALCCHECKSUM_ASSERT(file->exists());
	_str = file->name();
	_strIsFp = true;
	_passes = passes;
	_bit = bit;
	_useAlgorithm = algo;
	start();
}
