/* SHA-512 code by Jean-Luc Cooke <jlcooke@certainkey.com>
 *
 * Copyright (c) Jean-Luc Cooke <jlcooke@certainkey.com>
 * Copyright (c) Andrew McDonald <andrew@mcdonald.org.uk>
 * Copyright (c) 2003 Kyle McMartin <kyle@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 */

/* (C) 2002, 2003, 2004 Michael Buesch :
 *   derived from linux-2.5.63
 *   Modified/deleted/added some functions.
 */

/* SHA384 test vectors
 *
 * "abc"
 * 0xcb, 0x00, 0x75, 0x3f, 0x45, 0xa3, 0x5e, 0x8b,
 * 0xb5, 0xa0, 0x3d, 0x69, 0x9a, 0xc6, 0x50, 0x07,
 * 0x27, 0x2c, 0x32, 0xab, 0x0e, 0xde, 0xd1, 0x63,
 * 0x1a, 0x8b, 0x60, 0x5a, 0x43, 0xff, 0x5b, 0xed,
 * 0x80, 0x86, 0x07, 0x2b, 0xa1, 0xe7, 0xcc, 0x23,
 * 0x58, 0xba, 0xec, 0xa1, 0x34, 0xc8, 0x25, 0xa7
 *
 * "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq"
 * 0x33, 0x91, 0xfd, 0xdd, 0xfc, 0x8d, 0xc7, 0x39,
 * 0x37, 0x07, 0xa6, 0x5b, 0x1b, 0x47, 0x09, 0x39,
 * 0x7c, 0xf8, 0xb1, 0xd1, 0x62, 0xaf, 0x05, 0xab,
 * 0xfe, 0x8f, 0x45, 0x0d, 0xe5, 0xf3, 0x6b, 0xc6,
 * 0xb0, 0x45, 0x5a, 0x85, 0x20, 0xbc, 0x4e, 0x6f,
 * 0x5f, 0xe9, 0x5b, 0x1f, 0xe3, 0xc8, 0x45, 0x2b
 *
 * "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhijklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu"
 * 0x09, 0x33, 0x0c, 0x33, 0xf7, 0x11, 0x47, 0xe8,
 * 0x3d, 0x19, 0x2f, 0xc7, 0x82, 0xcd, 0x1b, 0x47,
 * 0x53, 0x11, 0x1b, 0x17, 0x3b, 0x3b, 0x05, 0xd2,
 * 0x2f, 0xa0, 0x80, 0x86, 0xe3, 0xb0, 0xf7, 0x12,
 * 0xfc, 0xc7, 0xc7, 0x1a, 0x55, 0x7e, 0x2d, 0xb9,
 * 0x66, 0xc3, 0xe9, 0xfa, 0x91, 0x74, 0x60, 0x39
 *
 * "abcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyz"
 * 0x3d, 0x20, 0x89, 0x73, 0xab, 0x35, 0x08, 0xdb,
 * 0xbd, 0x7e, 0x2c, 0x28, 0x62, 0xba, 0x29, 0x0a,
 * 0xd3, 0x01, 0x0e, 0x49, 0x78, 0xc1, 0x98, 0xdc,
 * 0x4d, 0x8f, 0xd0, 0x14, 0xe5, 0x82, 0x82, 0x3a,
 * 0x89, 0xe1, 0x6f, 0x9b, 0x2a, 0x7b, 0xbc, 0x1a,
 * 0xc9, 0x38, 0xe2, 0xd1, 0x99, 0xe8, 0xbe, 0xa4
 */

/* SHA512 test vectors from from NIST and kerneli
 *
 * "abc"
 * 0xdd, 0xaf, 0x35, 0xa1, 0x93, 0x61, 0x7a, 0xba,
 * 0xcc, 0x41, 0x73, 0x49, 0xae, 0x20, 0x41, 0x31,
 * 0x12, 0xe6, 0xfa, 0x4e, 0x89, 0xa9, 0x7e, 0xa2,
 * 0x0a, 0x9e, 0xee, 0xe6, 0x4b, 0x55, 0xd3, 0x9a,
 * 0x21, 0x92, 0x99, 0x2a, 0x27, 0x4f, 0xc1, 0xa8,
 * 0x36, 0xba, 0x3c, 0x23, 0xa3, 0xfe, 0xeb, 0xbd,
 * 0x45, 0x4d, 0x44, 0x23, 0x64, 0x3c, 0xe8, 0x0e,
 * 0x2a, 0x9a, 0xc9, 0x4f, 0xa5, 0x4c, 0xa4, 0x9f
 *
 * "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq"
 * 0x20, 0x4a, 0x8f, 0xc6, 0xdd, 0xa8, 0x2f, 0x0a,
 * 0x0c, 0xed, 0x7b, 0xeb, 0x8e, 0x08, 0xa4, 0x16,
 * 0x57, 0xc1, 0x6e, 0xf4, 0x68, 0xb2, 0x28, 0xa8,
 * 0x27, 0x9b, 0xe3, 0x31, 0xa7, 0x03, 0xc3, 0x35,
 * 0x96, 0xfd, 0x15, 0xc1, 0x3b, 0x1b, 0x07, 0xf9,
 * 0xaa, 0x1d, 0x3b, 0xea, 0x57, 0x78, 0x9c, 0xa0,
 * 0x31, 0xad, 0x85, 0xc7, 0xa7, 0x1d, 0xd7, 0x03,
 * 0x54, 0xec, 0x63, 0x12, 0x38, 0xca, 0x34, 0x45
 *
 * "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhijklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu"
 * 0x8e, 0x95, 0x9b, 0x75, 0xda, 0xe3, 0x13, 0xda,
 * 0x8c, 0xf4, 0xf7, 0x28, 0x14, 0xfc, 0x14, 0x3f,
 * 0x8f, 0x77, 0x79, 0xc6, 0xeb, 0x9f, 0x7f, 0xa1,
 * 0x72, 0x99, 0xae, 0xad, 0xb6, 0x88, 0x90, 0x18,
 * 0x50, 0x1d, 0x28, 0x9e, 0x49, 0x00, 0xf7, 0xe4,
 * 0x33, 0x1b, 0x99, 0xde, 0xc4, 0xb5, 0x43, 0x3a,
 * 0xc7, 0xd3, 0x29, 0xee, 0xb6, 0xdd, 0x26, 0x54,
 * 0x5e, 0x96, 0xe5, 0x5b, 0x87, 0x4b, 0xe9, 0x09
 *
 * "abcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyz"
 * 0x93, 0x0d, 0x0c, 0xef, 0xcb, 0x30, 0xff, 0x11,
 * 0x33, 0xb6, 0x89, 0x81, 0x21, 0xf1, 0xcf, 0x3d,
 * 0x27, 0x57, 0x8a, 0xfc, 0xaf, 0xe8, 0x67, 0x7c,
 * 0x52, 0x57, 0xcf, 0x06, 0x99, 0x11, 0xf7, 0x5d,
 * 0x8f, 0x58, 0x31, 0xb5, 0x6e, 0xbf, 0xda, 0x67,
 * 0xb2, 0x78, 0xe6, 0x6d, 0xff, 0x8b, 0x84, 0xfe,
 * 0x2b, 0x28, 0x70, 0xf7, 0x42, 0xa5, 0x80, 0xd8,
 * 0xed, 0xb4, 0x19, 0x87, 0x23, 0x28, 0x50, 0xc9
 */


#include "sha512.h"
#include "threadcomm.h"

#include <stdio.h>
#include <sys/stat.h>

#define SHA384_DIGEST_SIZE 48
#define SHA512_DIGEST_SIZE 64
#define SHA384_HMAC_BLOCK_SIZE  96
#define SHA512_HMAC_BLOCK_SIZE 128

static const uint64_t sha512_K[80] = {
        0x428a2f98d728ae22ULL, 0x7137449123ef65cdULL, 0xb5c0fbcfec4d3b2fULL,
        0xe9b5dba58189dbbcULL, 0x3956c25bf348b538ULL, 0x59f111f1b605d019ULL,
        0x923f82a4af194f9bULL, 0xab1c5ed5da6d8118ULL, 0xd807aa98a3030242ULL,
        0x12835b0145706fbeULL, 0x243185be4ee4b28cULL, 0x550c7dc3d5ffb4e2ULL,
        0x72be5d74f27b896fULL, 0x80deb1fe3b1696b1ULL, 0x9bdc06a725c71235ULL,
        0xc19bf174cf692694ULL, 0xe49b69c19ef14ad2ULL, 0xefbe4786384f25e3ULL,
        0x0fc19dc68b8cd5b5ULL, 0x240ca1cc77ac9c65ULL, 0x2de92c6f592b0275ULL,
        0x4a7484aa6ea6e483ULL, 0x5cb0a9dcbd41fbd4ULL, 0x76f988da831153b5ULL,
        0x983e5152ee66dfabULL, 0xa831c66d2db43210ULL, 0xb00327c898fb213fULL,
        0xbf597fc7beef0ee4ULL, 0xc6e00bf33da88fc2ULL, 0xd5a79147930aa725ULL,
        0x06ca6351e003826fULL, 0x142929670a0e6e70ULL, 0x27b70a8546d22ffcULL,
        0x2e1b21385c26c926ULL, 0x4d2c6dfc5ac42aedULL, 0x53380d139d95b3dfULL,
        0x650a73548baf63deULL, 0x766a0abb3c77b2a8ULL, 0x81c2c92e47edaee6ULL,
        0x92722c851482353bULL, 0xa2bfe8a14cf10364ULL, 0xa81a664bbc423001ULL,
        0xc24b8b70d0f89791ULL, 0xc76c51a30654be30ULL, 0xd192e819d6ef5218ULL,
        0xd69906245565a910ULL, 0xf40e35855771202aULL, 0x106aa07032bbd1b8ULL,
        0x19a4c116b8d2d0c8ULL, 0x1e376c085141ab53ULL, 0x2748774cdf8eeb99ULL,
        0x34b0bcb5e19b48a8ULL, 0x391c0cb3c5c95a63ULL, 0x4ed8aa4ae3418acbULL,
        0x5b9cca4f7763e373ULL, 0x682e6ff3d6b2b8a3ULL, 0x748f82ee5defb2fcULL,
        0x78a5636f43172f60ULL, 0x84c87814a1f0ab72ULL, 0x8cc702081a6439ecULL,
        0x90befffa23631e28ULL, 0xa4506cebde82bde9ULL, 0xbef9a3f7b2c67915ULL,
        0xc67178f2e372532bULL, 0xca273eceea26619cULL, 0xd186b8c721c0c207ULL,
        0xeada7dd6cde0eb1eULL, 0xf57d4f7fee6ed178ULL, 0x06f067aa72176fbaULL,
        0x0a637dc5a2c898a6ULL, 0x113f9804bef90daeULL, 0x1b710b35131c471bULL,
        0x28db77f523047d84ULL, 0x32caab7b40c72493ULL, 0x3c9ebe0a15c9bebcULL,
        0x431d67c49c100d4cULL, 0x4cc5d4becb3e42b6ULL, 0x597f299cfc657e2aULL,
        0x5fcb6fab3ad6faecULL, 0x6c44198c4a475817ULL,
};

#define SHA512e0(x)       (RORuint64_t(x,28) ^ RORuint64_t(x,34) ^ RORuint64_t(x,39))
#define SHA512e1(x)       (RORuint64_t(x,14) ^ RORuint64_t(x,18) ^ RORuint64_t(x,41))
#define SHA512s0(x)       (RORuint64_t(x, 1) ^ RORuint64_t(x, 8) ^ (x >> 7))
#define SHA512s1(x)       (RORuint64_t(x,19) ^ RORuint64_t(x,61) ^ (x >> 6))

/* H* initial state for SHA-512 */
#define SHA512H0         0x6a09e667f3bcc908ULL
#define SHA512H1         0xbb67ae8584caa73bULL
#define SHA512H2         0x3c6ef372fe94f82bULL
#define SHA512H3         0xa54ff53a5f1d36f1ULL
#define SHA512H4         0x510e527fade682d1ULL
#define SHA512H5         0x9b05688c2b3e6c1fULL
#define SHA512H6         0x1f83d9abfb41bd6bULL
#define SHA512H7         0x5be0cd19137e2179ULL

/* H'* initial state for SHA-384 */
#define SHA512HP0 0xcbbb9d5dc1059ed8ULL
#define SHA512HP1 0x629a292a367cd507ULL
#define SHA512HP2 0x9159015a3070dd17ULL
#define SHA512HP3 0x152fecd8f70e5939ULL
#define SHA512HP4 0x67332667ffc00b31ULL
#define SHA512HP5 0x8eb44a8768581511ULL
#define SHA512HP6 0xdb0c2e0d64f98fa7ULL
#define SHA512HP7 0x47b5481dbefa4fa4ULL

void Sha512::LOAD_OP(int I, uint64_t *W, const uint8_t *input)
{
        uint64_t t1  = input[(8*I)  ] & 0xff;
        t1 <<= 8;
        t1 |= input[(8*I)+1] & 0xff;
        t1 <<= 8;
        t1 |= input[(8*I)+2] & 0xff;
        t1 <<= 8;
        t1 |= input[(8*I)+3] & 0xff;
        t1 <<= 8;
        t1 |= input[(8*I)+4] & 0xff;
        t1 <<= 8;
        t1 |= input[(8*I)+5] & 0xff;
        t1 <<= 8;
        t1 |= input[(8*I)+6] & 0xff;
        t1 <<= 8;
        t1 |= input[(8*I)+7] & 0xff;
        W[I] = t1;
}

void Sha512::BLEND_OP(int I, uint64_t *W)
{
	W[I] = SHA512s1(W[I-2]) + W[I-7] + SHA512s0(W[I-15]) + W[I-16];
}

void Sha512::sha512_transform(uint64_t *state, const uint8_t *input)
{
	uint64_t a, b, c, d, e, f, g, h, t1, t2;
	uint64_t W[80];

	int i;

	/* load the input */
        for (i = 0; i < 16; i++)
                LOAD_OP(i, W, input);

        for (i = 16; i < 80; i++) {
                BLEND_OP(i, W);
        }

	/* load the state into our registers */
	a=state[0];   b=state[1];   c=state[2];   d=state[3];  
	e=state[4];   f=state[5];   g=state[6];   h=state[7];  
  
	/* now iterate */
	for (i=0; i<80; i+=8) {
		t1 = h + SHA512e1(e) + Ch(e,f,g) + sha512_K[i  ] + W[i  ];
		t2 = SHA512e0(a) + Maj(a,b,c);    d+=t1;    h=t1+t2;
		t1 = g + SHA512e1(d) + Ch(d,e,f) + sha512_K[i+1] + W[i+1];
		t2 = SHA512e0(h) + Maj(h,a,b);    c+=t1;    g=t1+t2;
		t1 = f + SHA512e1(c) + Ch(c,d,e) + sha512_K[i+2] + W[i+2];
		t2 = SHA512e0(g) + Maj(g,h,a);    b+=t1;    f=t1+t2;
		t1 = e + SHA512e1(b) + Ch(b,c,d) + sha512_K[i+3] + W[i+3];
		t2 = SHA512e0(f) + Maj(f,g,h);    a+=t1;    e=t1+t2;
		t1 = d + SHA512e1(a) + Ch(a,b,c) + sha512_K[i+4] + W[i+4];
		t2 = SHA512e0(e) + Maj(e,f,g);    h+=t1;    d=t1+t2;
		t1 = c + SHA512e1(h) + Ch(h,a,b) + sha512_K[i+5] + W[i+5];
		t2 = SHA512e0(d) + Maj(d,e,f);    g+=t1;    c=t1+t2;
		t1 = b + SHA512e1(g) + Ch(g,h,a) + sha512_K[i+6] + W[i+6];
		t2 = SHA512e0(c) + Maj(c,d,e);    f+=t1;    b=t1+t2;
		t1 = a + SHA512e1(f) + Ch(f,g,h) + sha512_K[i+7] + W[i+7];
		t2 = SHA512e0(b) + Maj(b,c,d);    e+=t1;    a=t1+t2;
	}
  
	state[0] += a; state[1] += b; state[2] += c; state[3] += d;  
	state[4] += e; state[5] += f; state[6] += g; state[7] += h;  

	/* erase our data */
	a = b = c = d = e = f = g = h = t1 = t2 = 0;
	memset(W, 0, 80 * sizeof(uint64_t));
}

void Sha512::sha512_init(struct sha512_ctx *ctx)
{
        struct sha512_ctx *sctx = ctx;
	sctx->state[0] = SHA512H0;
	sctx->state[1] = SHA512H1;
	sctx->state[2] = SHA512H2;
	sctx->state[3] = SHA512H3;
	sctx->state[4] = SHA512H4;
	sctx->state[5] = SHA512H5;
	sctx->state[6] = SHA512H6;
	sctx->state[7] = SHA512H7;
	sctx->count[0] = sctx->count[1] = sctx->count[2] = sctx->count[3] = 0;
	memset(sctx->buf, 0, sizeof(sctx->buf));
}

void Sha512::sha384_init(struct sha512_ctx *ctx)
{
        struct sha512_ctx *sctx = ctx;
        sctx->state[0] = SHA512HP0;
        sctx->state[1] = SHA512HP1;
        sctx->state[2] = SHA512HP2;
        sctx->state[3] = SHA512HP3;
        sctx->state[4] = SHA512HP4;
        sctx->state[5] = SHA512HP5;
        sctx->state[6] = SHA512HP6;
        sctx->state[7] = SHA512HP7;
        sctx->count[0] = sctx->count[1] = sctx->count[2] = sctx->count[3] = 0;
        memset(sctx->buf, 0, sizeof(sctx->buf));
}

void Sha512::sha512_update(struct sha512_ctx *ctx, const uint8_t *data, unsigned int len)
{
        struct sha512_ctx *sctx = ctx;

	unsigned int i, index, part_len;

	/* Compute number of bytes mod 128 */
	index = (unsigned int)((sctx->count[0] >> 3) & 0x7F);
	
	/* Update number of bits */
	if ((sctx->count[0] += (len << 3)) < (len << 3)) {
		if ((sctx->count[1] += 1) < 1)
			if ((sctx->count[2] += 1) < 1)
				sctx->count[3]++;
		sctx->count[1] += (len >> 29);
	}
	
        part_len = 128 - index;
	
	/* Transform as many times as possible. */
	if (len >= part_len) {
		memcpy(&sctx->buf[index], data, part_len);
		sha512_transform(sctx->state, sctx->buf);

		for (i = part_len; i + 127 < len; i+=128)
			sha512_transform(sctx->state, &data[i]);

		index = 0;
	} else {
		i = 0;
	}

	/* Buffer remaining input */
	memcpy(&sctx->buf[index], &data[i], len - i);
}

void Sha512::sha512_final(struct sha512_ctx *ctx, uint8_t *hash)
{
        struct sha512_ctx *sctx = ctx;
	
        static const uint8_t padding[128] = { 0x80, };

        uint32_t t;
	uint64_t t2;
        uint8_t bits[128];
	unsigned int index, pad_len;
	int i, j;

        index = pad_len = t = i = j = 0;
        t2 = 0;

	/* Save number of bits */
	t = sctx->count[0];
	bits[15] = t; t>>=8;
	bits[14] = t; t>>=8;
	bits[13] = t; t>>=8;
	bits[12] = t; 
	t = sctx->count[1];
	bits[11] = t; t>>=8;
	bits[10] = t; t>>=8;
	bits[9 ] = t; t>>=8;
	bits[8 ] = t; 
	t = sctx->count[2];
	bits[7 ] = t; t>>=8;
	bits[6 ] = t; t>>=8;
	bits[5 ] = t; t>>=8;
	bits[4 ] = t; 
	t = sctx->count[3];
	bits[3 ] = t; t>>=8;
	bits[2 ] = t; t>>=8;
	bits[1 ] = t; t>>=8;
	bits[0 ] = t; 

	/* Pad out to 112 mod 128. */
	index = (sctx->count[0] >> 3) & 0x7f;
	pad_len = (index < 112) ? (112 - index) : ((128+112) - index);
	sha512_update(sctx, padding, pad_len);

	/* Append length (before padding) */
	sha512_update(sctx, bits, 16);

	/* Store state in digest */
	for (i = j = 0; i < 8; i++, j += 8) {
		t2 = sctx->state[i];
		hash[j+7] = (char)t2 & 0xff; t2>>=8;
		hash[j+6] = (char)t2 & 0xff; t2>>=8;
		hash[j+5] = (char)t2 & 0xff; t2>>=8;
		hash[j+4] = (char)t2 & 0xff; t2>>=8;
		hash[j+3] = (char)t2 & 0xff; t2>>=8;
		hash[j+2] = (char)t2 & 0xff; t2>>=8;
		hash[j+1] = (char)t2 & 0xff; t2>>=8;
		hash[j  ] = (char)t2 & 0xff;
	}
	
	/* Zeroize sensitive information. */
	memset(sctx, 0, sizeof(struct sha512_ctx));
}

void Sha512::sha384_final(struct sha512_ctx *ctx, uint8_t *hash)
{
        struct sha512_ctx *sctx = ctx;
        uint8_t D[64];

        sha512_final(sctx, D);

        memcpy(hash, D, 48);
        memset(D, 0, 64);
}

bool Sha512::selfTest()
{
	Sha512 sha512;

	if (unlikely(!sha512.sha384_selfTest()))
		return false;
	if (unlikely(!sha512.sha512_selfTest()))
		return false;

	return true;
}

bool Sha512::sha384_selfTest()
{
	string test1("abc");
	string test1_md("CB00753F45A35E8BB5A03D699AC65007"
			"272C32AB0EDED1631A8B605A43FF5BED"
			"8086072BA1E7CC2358BAECA134C825A7");
	string test2("abcdbcdecdefdefgefghfghighij"
		     "hijkijkljklmklmnlmnomnopnopq");
	string test2_md("3391FDDDFC8DC7393707A65B1B470939"
			"7CF8B1D162AF05ABFE8F450DE5F36BC6"
			"B0455A8520BC4E6F5FE95B1FE3C8452B");
	string test3("abcdefghbcdefghicdefghijdefg"
		     "hijkefghijklfghijklmghijklmn"
		     "hijklmnoijklmnopjklmnopqklmn"
		     "opqrlmnopqrsmnopqrstnopqrstu");
	string test3_md("09330C33F71147E83D192FC782CD1B47"
			"53111B173B3B05D22FA08086E3B0F712"
			"FCC7C71A557E2DB966C3E9FA91746039");
	string test4("abcdefghijklmnopqrstuvwxyzab"
		     "cdefghijklmnopqrstuvwxyzabcd"
		     "efghijklmnopqrstuvwxyzabcdef"
		     "ghijklmnopqrstuvwxyz");
	string test4_md("3D208973AB3508DBBD7E2C2862BA290A"
			"D3010E4978C198DC4D8FD014E582823A"
			"89E16F9B2A7BBC1AC938E2D199E8BEA4");

	if (unlikely(calcSha384(test1) != test1_md))
		return false;
	if (unlikely(calcSha384(test2) != test2_md))
		return false;
	if (unlikely(calcSha384(test3) != test3_md))
		return false;
	if (unlikely(calcSha384(test4) != test4_md))
		return false;

	return true;
}

bool Sha512::sha512_selfTest()
{
	string test1("abc");
	string test1_md("DDAF35A193617ABACC417349AE204131"
			"12E6FA4E89A97EA20A9EEEE64B55D39A"
			"2192992A274FC1A836BA3C23A3FEEBBD"
			"454D4423643CE80E2A9AC94FA54CA49F");
	string test2("abcdbcdecdefdefgefghfghighij"
		     "hijkijkljklmklmnlmnomnopnopq");
	string test2_md("204A8FC6DDA82F0A0CED7BEB8E08A416"
			"57C16EF468B228A8279BE331A703C335"
			"96FD15C13B1B07F9AA1D3BEA57789CA0"
			"31AD85C7A71DD70354EC631238CA3445");
	string test3("abcdefghbcdefghicdefghijdefg"
		     "hijkefghijklfghijklmghijklmn"
		     "hijklmnoijklmnopjklmnopqklmn"
		     "opqrlmnopqrsmnopqrstnopqrstu");
	string test3_md("8E959B75DAE313DA8CF4F72814FC143F"
			"8F7779C6EB9F7FA17299AEADB6889018"
			"501D289E4900F7E4331B99DEC4B5433A"
			"C7D329EEB6DD26545E96E55B874BE909");
	string test4("abcdefghijklmnopqrstuvwxyzab"
		     "cdefghijklmnopqrstuvwxyzabcd"
		     "efghijklmnopqrstuvwxyzabcdef"
		     "ghijklmnopqrstuvwxyz");
	string test4_md("930D0CEFCB30FF1133B6898121F1CF3D"
			"27578AFCAFE8677C5257CF069911F75D"
			"8F5831B56EBFDA67B278E66DFF8B84FE"
			"2B2870F742A580D8EDB41987232850C9");

	if (unlikely(calcSha512(test1) != test1_md))
		return false;
	if (unlikely(calcSha512(test2) != test2_md))
		return false;
	if (unlikely(calcSha512(test3) != test3_md))
		return false;
	if (unlikely(calcSha512(test4) != test4_md))
		return false;

	return true;
}

string Sha512::calcSha384(const string &buf)
{
	uint8_t ret[SHA384_DIGEST_SIZE];
	struct sha512_ctx ctx;

	sha384_init(&ctx);
	sha512_update(&ctx, reinterpret_cast<const uint8_t *>(buf.c_str()),
		      static_cast<unsigned int>(buf.length()));
	sha384_final(&ctx, ret);

	return charToHex(reinterpret_cast<const char *>(ret),
			 SHA384_DIGEST_SIZE);
}

string Sha512::calcSha384(const string &filename, int *commSocket)
{
	FILE *fd;
	struct stat file_stat;
	size_t read;
	unsigned int bufSize;
	off_t fileSize, i;
	uint8_t ret[SHA384_DIGEST_SIZE];
	struct sha512_ctx ctx;

	if (unlikely(stat(filename.c_str(), &file_stat)))
		return "";
	fd = fopen(filename.c_str(), "r");
	if (unlikely(!fd))
		return "";

	fileSize = file_stat.st_size;
	string progrTmp(string("S") + tostr(fileSize) + '\n');
	ThreadComm comm(commSocket);
	comm.send(&progrTmp);

	sha384_init(&ctx);

	bufSize = calcBufSize(file_stat);
	char *buf = new char[bufSize];
	i = 0;
	while (i < fileSize) {
		read = fread(buf, 1, bufSize, fd);
		i += read;
		sha512_update(&ctx, reinterpret_cast<const uint8_t *>(buf),
			      read);

		progrTmp = tostr(i) + '\n';
		comm.send(&progrTmp);
	}
	fclose(fd);
	delete [] buf;

	sha384_final(&ctx, ret);

	return charToHex(reinterpret_cast<const char *>(ret),
			 SHA384_DIGEST_SIZE);
}

string Sha512::calcSha512(const string &buf)
{
	uint8_t ret[SHA512_DIGEST_SIZE];
	struct sha512_ctx ctx;

	sha512_init(&ctx);
	sha512_update(&ctx, reinterpret_cast<const uint8_t *>(buf.c_str()),
		      static_cast<unsigned int>(buf.length()));
	sha512_final(&ctx, ret);

	return charToHex(reinterpret_cast<const char *>(ret),
			 SHA512_DIGEST_SIZE);
}

string Sha512::calcSha512(const string &filename, int *commSocket)
{
	FILE *fd;
	struct stat file_stat;
	size_t read;
	unsigned int bufSize;
	off_t fileSize, i;
	uint8_t ret[SHA512_DIGEST_SIZE];
	struct sha512_ctx ctx;

	if (unlikely(stat(filename.c_str(), &file_stat)))
		return "";
	fd = fopen(filename.c_str(), "r");
	if (unlikely(!fd))
		return "";

	fileSize = file_stat.st_size;
	string progrTmp(string("S") + tostr(fileSize) + '\n');
	ThreadComm comm(commSocket);
	comm.send(&progrTmp);

	sha512_init(&ctx);

	bufSize = calcBufSize(file_stat);
	char *buf = new char[bufSize];
	i = 0;
	while (i < fileSize) {
		read = fread(buf, 1, bufSize, fd);
		i += read;
		sha512_update(&ctx, reinterpret_cast<const uint8_t *>(buf),
			      read);

		progrTmp = tostr(i) + '\n';
		comm.send(&progrTmp);
	}
	fclose(fd);
	delete [] buf;

	sha512_final(&ctx, ret);

	return charToHex(reinterpret_cast<const char *>(ret),
			 SHA512_DIGEST_SIZE);
}
