/***************************************************************************
 *   Copyright (C) 2006 by Takeshi.Hotta                                   *
 *   takeshi.hotta@gmail.com                                               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
// #include <iostream>
// #include <fstream>
// #include <string>

#include <kdebug.h>
#include <kglobal.h>
#include <klocale.h>
#include <kconfig.h>
#include <kconfigdialog.h>
#include <kuniqueapplication.h>
#include <kdirlister.h>
#include <kfileitem.h>
#include <kfilemetainfo.h>
#include <kmessagebox.h>
#include <kurl.h>
#include <kactionselector.h>
#include <kaboutapplication.h>
#include <kaboutdata.h>
#include <kpanelapplet.h>
#include <qlabel.h>
#include <qpixmap.h>
#include <qpainter.h>
#include <qstring.h>
#include <qgrid.h>
#include <qnamespace.h>
#include <qobjectlist.h>
#include <qtimer.h>
#include <qtooltip.h>
#include <qradiobutton.h>
#include <qregexp.h>
#include <qptrlist.h>
#include <qapplication.h>
#include <qevent.h>
#include <qsize.h>

#include "diskmonitor.h"
#include "base-panel-green.xpm"
#include "base-panel-black.xpm"
#include "lcdGraph-black.xpm"
#include "lcdGraph-skyblue.xpm"
#include "diskmonitorsettings.h"
#include "diskmonitorConfig.h"

// using namespace std;

static const char *version = VERSION;

class diskmonitorItem
{
public:
    diskmonitorItem() : pDeviceID(0) {}
    diskmonitorItem(const QString &deviceid, const QString &mountpoint, const QString &mountdir, const QString &freespace, const QString &scale, const double usage)
    : pDeviceID(deviceid), pMountPoint(mountpoint), pMountDir(mountdir), pFreeSpace(freespace), pScale(scale), pUsage(usage){}

    QString DeviceID() const { return pDeviceID; };
    QString MountPoint() const { return diskmonitorSettings::longFormat() ? pMountPoint : pMountDir; };
    QString FreeSpace() const { return pFreeSpace; };
    QString Scale() const { return pScale; };
    double Usage() const { return pUsage; };
    bool operator< (const diskmonitorItem item) { return pMountPoint <  item.pMountPoint; };
    bool operator> (const diskmonitorItem item) { return pMountPoint >  item.pMountPoint; };
    bool operator==(const diskmonitorItem item) { return pMountPoint == item.pMountPoint; };
private:
    QString pDeviceID;
    QString pMountPoint;
    QString pMountDir;
    QString pFreeSpace;
    QString pScale;  // It contains "GB" or "MB"
    double  pUsage;
};

template<class type> 
class QSortedList : public QPtrList<type>
{
public:
    QSortedList() {}
    QSortedList( const QSortedList<type> &l ) : QPtrList<type>(l) {}
    ~QSortedList() { this->clear(); }
    QSortedList<type> &operator=(const QSortedList<type> &l)
      { return (QSortedList<type>&)QPtrList<type>::operator=(l); }
protected:
    virtual int compareItems( QPtrCollection::Item s1, QPtrCollection::Item s2 )
      { if ( *((type*)s1) == *((type*)s2) ) return 0; return ( *((type*)s1) < *((type*)s2) ? -1 : 1 ); }
};

typedef QSortedList<diskmonitorItem> diskmonitorItemList;

class PartitionViewLabel : public QLabel
{
public:
    static const QPixmap base_panel[diskmonitorSettings::EnumColorTheme::COUNT];
    static const QPixmap bar_graph [diskmonitorSettings::EnumColorTheme::COUNT];
    static const QColor  textColor [diskmonitorSettings::EnumColorTheme::COUNT];
    static const int sizeMargin;
    static const int sizeLineWidth;
    static const int sizeMidLineWidth;
    PartitionViewLabel(QWidget *parent =0, const char *name =0, WFlags f = Qt::WNoAutoErase );
    void setItem(diskmonitorItem *partition) { pPartition = partition; }
protected:
    virtual void drawContents(QPainter *p);
private:
    diskmonitorItem *pPartition;
};

const QPixmap PartitionViewLabel::base_panel[diskmonitorSettings::EnumColorTheme::COUNT]
	={QPixmap(base_panel_green_xpm),	QPixmap(base_panel_black_xpm)};
const QPixmap PartitionViewLabel::bar_graph [diskmonitorSettings::EnumColorTheme::COUNT]
	={QPixmap(lcdGraph_black_xpm),		QPixmap(lcdGraph_skyblue_xpm)};
const QColor  PartitionViewLabel::textColor [diskmonitorSettings::EnumColorTheme::COUNT]
	={Qt::black,				QColor(208, 255, 254)};
const int PartitionViewLabel::sizeMargin(0);
const int PartitionViewLabel::sizeLineWidth(1);
const int PartitionViewLabel::sizeMidLineWidth(0);

class MainViewGrid : public QGrid
{
public:
    static const char * searchClass;
    static const int sizeMergin;
    static const int sizeSpacing;
    static uint calculateMaxRowsSize(int height);
    MainViewGrid( int n, Orientation orient, QWidget * parent = 0, const char * name = 0, WFlags f = 0 );
    ~MainViewGrid();
    void setItems(const diskmonitorItemList items);
    void linkItemToLabel();
    diskmonitorItemList &getItems() { return diskmonitorItems; };
    virtual QSize minimumSizeHint () const;
    virtual QSize sizeHint () const;
    const uint numRows() { return numrows; };
    const uint maxRows() { return maxrows; };
    const int numItems() { return diskmonitorItems.count(); };
    const int numViewLabel();
    void calculateSize(int height);
    void setLabelUpdatesEnabled( bool en = false );
protected:
    virtual void drawContents(QPainter *p);
private:
    uint numrows, numcolms, maxrows;
    int minwidth, minheight;
    diskmonitorItemList diskmonitorItems;
};

const char * MainViewGrid::searchClass("QLabel");
const int MainViewGrid::sizeMergin(0);
const int MainViewGrid::sizeSpacing(0);

PartitionViewLabel::PartitionViewLabel(QWidget *parent, const char *name, WFlags f) :
    QLabel(parent, name, f), pPartition(NULL)
{
    setSizePolicy( QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed) );
    setMargin( sizeMargin );
    setFixedSize(base_panel[diskmonitorSettings::colorTheme()].width()+2*sizeLineWidth,
                 base_panel[diskmonitorSettings::colorTheme()].height()+2*sizeLineWidth);
    setFrameStyle( QFrame::Panel | QFrame::Sunken );
    setLineWidth( sizeLineWidth );
    setMidLineWidth( sizeMidLineWidth );
    setBackgroundMode(NoBackground);
    setUpdatesEnabled( false ); // disable updates() and repaint() of PartitionViewLabel.
    show();
}

void PartitionViewLabel::drawContents(QPainter *p)
{
/*
    string fn("DiskMonitor.log");
    ofstream logs(fn.c_str(),ios::app);
    if (!logs) kdFatal() << "failed to open log"<< endl;
    else logs << "log on.PartitionViewLabel::drawContents" << endl;
*/
    if (isVisible())
    {
        QFont f(diskmonitorSettings::defaultFont());
        p->drawPixmap(1, 1, base_panel[diskmonitorSettings::colorTheme()]);
        p->setFont(f);
        p->setPen(textColor[diskmonitorSettings::colorTheme()]);
        p->drawText( 3, 24, pPartition->DeviceID());
        f.setPointSize(8);
        p->setFont(f);
        p->drawText( 3,  4, 48, 11, Qt::AlignLeft | Qt::AlignBottom, pPartition->MountPoint());
        f.setPointSize(11);
        p->setFont(f);
        p->drawText(51,  4, 37, 15, Qt::AlignRight | Qt::AlignBottom, pPartition->FreeSpace());
        f.setPointSize(7);
        p->setFont(f);
        p->drawText(88, 15, pPartition->Scale());
     /* drawing Bar graph */
        int fw = int(pPartition->Usage() / 5);
        p->drawPixmap(39, 17, bar_graph[diskmonitorSettings::colorTheme()], 0, 0, fw*3);
        int fh = int((pPartition->Usage() - 5*fw)*7/5);
        p->drawPixmap((39 + fw*3), (17 + 7 - fh), bar_graph[diskmonitorSettings::colorTheme()], 0,( 7 - fh), 3);
    };
/*
    logs << "log off.PartitionViewLabel::drawContents"<< endl;
    logs.close();
*/
}

MainViewGrid::MainViewGrid( int n, Orientation orient, QWidget * parent, const char * name, WFlags f)
    : QGrid(n, orient, parent, name, f), numrows(0), numcolms(0)

{
    setSizePolicy(QSizePolicy(QSizePolicy::Minimum, QSizePolicy::MinimumExpanding));
    setMargin(sizeMergin);
    setSpacing(sizeSpacing);
    minwidth = PartitionViewLabel::base_panel[diskmonitorSettings::colorTheme()].width()
                + 2*(PartitionViewLabel::sizeLineWidth + MainViewGrid::sizeMergin);
    minheight = n * PartitionViewLabel::base_panel[diskmonitorSettings::colorTheme()].height()
                + ( n - 1 ) * sizeSpacing
                + 2*(PartitionViewLabel::sizeLineWidth + MainViewGrid::sizeMergin);
    setMinimumSize(minwidth, minheight);
    maxrows = n;

    setUpdatesEnabled( false );  // disable updates() and repaint() of mainView.
    show();
}

uint MainViewGrid::calculateMaxRowsSize(int height)
{
    uint mr = (height - 2*MainViewGrid::sizeMergin + MainViewGrid::sizeSpacing) /
              (PartitionViewLabel::base_panel[diskmonitorSettings::colorTheme()].height()
                + 2*PartitionViewLabel::sizeLineWidth + MainViewGrid::sizeSpacing); 
    mr = mr > 0 ? mr : 1;
    return mr;
}

const int MainViewGrid::numViewLabel()
{
    QObjectList *lp = queryList( searchClass );
    int c = lp->count();
    delete(lp);
    return c;
}


void MainViewGrid::linkItemToLabel()
{
    QObjectList *lp = queryList( searchClass );
    QPtrListIterator<diskmonitorItem> iti( diskmonitorItems );

    for ( uint i =0 ; i < lp->count(); ++i )
    {
       static_cast<PartitionViewLabel *>(lp->at(i))->setItem(iti.current());
       ++iti;
    };
    delete(lp);
}

void MainViewGrid::setItems(diskmonitorItemList items)
{
/*
    string fn("DiskMonitor.log");
    ofstream logs(fn.c_str(),ios::app);
    if (!logs) kdFatal() << "failed to open log"<< endl;
    else logs << "log on.MainViewGrid::setItems" << endl;
*/
    diskmonitorItems.clear();
    diskmonitorItems = items;
    calculateSize(parentWidget()->height());

    int dif = numItems() - numViewLabel();

//    logs << "log off.MainViewGrid::setItems<numItems()>" << numItems() << "<numViewLabel()>" << numViewLabel() << endl;
    if ( dif > 0 ) { for (int i=0; i < dif; i++)  new PartitionViewLabel(this); };

    linkItemToLabel();
    setUpdatesEnabled( true );  // enable updates() and repaint() of mainView.
    update();
/*
    logs << "log off.MainViewGrid::setItems"<< endl;
    logs.close();
*/
}

void MainViewGrid::calculateSize(int height)
{
    int mr = calculateMaxRowsSize(height);
    numrows = numItems() > mr ? mr : numItems();
    numcolms = numItems() / mr;
    if ( numItems() % mr > 0 ) numcolms++ ;
    int oldwidth = minwidth;
    minwidth = (PartitionViewLabel::base_panel[diskmonitorSettings::colorTheme()].width()
                + 2*PartitionViewLabel::sizeLineWidth)*numcolms
                + MainViewGrid::sizeSpacing*(numcolms - 1)
                + 2*MainViewGrid::sizeMergin;
    minheight = (PartitionViewLabel::base_panel[diskmonitorSettings::colorTheme()].height()
                + 2*PartitionViewLabel::sizeLineWidth)*numrows
                + MainViewGrid::sizeSpacing*(numrows - 1)
                + 2*MainViewGrid::sizeMergin;
    adjustSize();
    if (minwidth != oldwidth) static_cast<diskmonitor *>(parentWidget())->reposition();
}

void MainViewGrid::drawContents(QPainter */*p*/)
{
/*
    string fn("DiskMonitor.log");
    ofstream logs(fn.c_str(),ios::app);
    if (!logs) kdFatal() << "failed to open log"<< endl;
    else logs << "log on.MainViewGrid::drawContents" << endl;
*/
    if (!isVisible())
        return;

    QObjectList *lp = queryList( searchClass );
    QObjectListIterator it(*lp);
    for ( it.toFirst(); it.current() != 0; ++it )
    {
        static_cast<PartitionViewLabel *>(it.current())->setUpdatesEnabled( true );
        static_cast<PartitionViewLabel *>(it.current())->update();
    };
/*
    logs << "log off.MainViewGrid::drawContents"<< endl;
    logs.close();
*/
}

void MainViewGrid::setLabelUpdatesEnabled( bool en )
{
    QObjectList *lp = queryList( searchClass );
    QObjectListIterator it(*lp);
    for ( it.toFirst(); it.current() != 0; ++it )
    { static_cast<PartitionViewLabel *>(it.current())->setUpdatesEnabled( en ); };
}

QSize MainViewGrid::minimumSizeHint () const
{
    return QSize(minwidth, minheight);
}

QSize MainViewGrid::sizeHint () const
{
    return QSize(minwidth, minheight);
}

MainViewGrid::~MainViewGrid()
{
/*
    string fn("DiskMonitor.log");
    ofstream logs(fn.c_str(),ios::app);
    if (!logs) kdFatal() << "failed to open log"<< endl;
    else logs << "log on. MainViewGrid::~MainViewGrid()"<< endl;
*/
    diskmonitorItems.clear();
/*
    logs << "log off.MainViewGrid::~MainViewGrid()"<< endl;
    logs.close();
*/
}

diskmonitor::diskmonitor(const QString& configFile, Type type, int actions, QWidget *parent, const char *name)
    : KPanelApplet(configFile, type, actions, parent, name), RefreshInterval(diskmonitorSettings::refreshInterval()), ExcludedDrives(diskmonitorSettings::excludedDevices())
{
    // Get the current application configuration handle
/*
    string fn("DiskMonitor.log");
    ofstream logs(fn.c_str());
    if (!logs) kdFatal() << "failed to open log"<< endl;
    else logs << "log on.diskmonitor::diskmonitor"<< endl;
*/
    ksConfig = config();

    int mr = MainViewGrid::calculateMaxRowsSize(height());
    mainView = new MainViewGrid(mr, Qt::Vertical, this);

    DirLister = new KDirLister();

    connect (DirLister, SIGNAL(completed()), this, SLOT(setDirList()));

    Timer = new QTimer(this);
    connect (Timer, SIGNAL(timeout()), this, SLOT(setDirList()));

    partitionsURL=KURL("media:/");
    IncludeFilter << "media/hdd_mounted";
    if ( diskmonitorSettings::supportRemovableMedia() ) IncludeFilter.append("media/removable_mounted");

    execDirList();
    Timer->start(RefreshInterval);
/*
    logs << "log off.diskmonitor::diskmonitor"<< endl;
    logs.close();
*/
}

void diskmonitor::execDirList()
{
    if (!DirLister->isFinished())
        return;

    DirLister->stop();
    DirLister->setDirOnlyMode(true);
    DirLister->clearMimeFilter();
    DirLister->setMimeFilter(IncludeFilter);
    DirLister->openURL( partitionsURL, false, true );
}

void diskmonitor::setDirList()
{
/*
    string fn("DiskMonitor.log");
    ofstream logs(fn.c_str(),ios::app);
    if (!logs) kdFatal() << "failed to open log"<< endl;
    else logs << "log on.diskmonitor::setDirList" << endl;
*/

    mainView->setUpdatesEnabled( false );  // disable updates() and repaint() of mainView.
    mainView->setLabelUpdatesEnabled( false ); // disable updates() and repaint() of PartitionViewLabels.

    KFileItemList Partitions = DirLister->items();
    QString deviceid;
    QString mountpoint;
    QString mountdir;
    QString freespace;
    QString scale;
    double usage;
    diskmonitorItemList tmpList;
    int len;

    for (uint i=0; i< Partitions.count(); i++)
    {
        deviceid = Partitions.at(i)->url().path().replace("/", "");
        if (diskmonitorSettings::excludedDevices().contains(deviceid))
            continue;

        mountpoint="";
        mountdir="";
        freespace="";
        scale="";
        usage=0;
//        logs << "logging.diskmonitor::setDirList<url()>" << Partitions.at(i)->url().url() << " <mimetype()>" << Partitions.at(i)->mimetype() << endl;
        KFileMetaInfo info(Partitions.at(i)->url(), Partitions.at(i)->mimetype());
        if ( not(info.isEmpty()) )
        {
            mountpoint.append(info.item("baseURL").string().replace("file://", ""));
            mountdir.append(info.item("baseURL").string().replace(QRegExp("file://.*/"), "").stripWhiteSpace());
            if (mountdir.isEmpty()) mountdir = "(root)";
            len = info.item("free").string().length();
            freespace.append(info.item("free").string().left(len-3));
            scale.append(info.item("free").string().right(2));
            usage = info.item("used").value().toDouble() / info.item("total").value().toDouble() * 100.0;
        };

        tmpList.append( new diskmonitorItem(deviceid, mountpoint.stripWhiteSpace(), mountdir.stripWhiteSpace(), freespace, scale, usage) );
//        logs << "logging.diskmonitor::setDirList<deviceid>" << deviceid << endl;
    }

    if (not(diskmonitorSettings::sortByDevice())) tmpList.sort();

    if (  tmpList.count() < uint(mainView->numViewLabel()) )
    {
        uint mr = mainView->maxRows();
        delete(mainView);
        mainView= new MainViewGrid(mr, Qt::Vertical, this);
        mainView->setItems(tmpList);
        reposition();
    }
    else mainView->setItems(tmpList);
/*
    logs << "log off.diskmonitor::setDirList<Partitions.count()>" << Partitions.count() << endl;
    logs.close();
*/
}

void diskmonitor::updateSettings()
{
//  update colorTheme
    int i = confWdg->StandardLCD->isChecked() ? 0 : 1;
    diskmonitorSettings::setColorTheme(i);

//  Update excluded devices.
    QStringList ls;
    QListBox *lb = confWdg->kcfg_ExcludedDevices->availableListBox();
    for (uint i = 0 ; i < lb->count() ; i++) ls.append(lb->text(i));

// Update SupportRemovableMedia
    if      ( diskmonitorSettings::supportRemovableMedia() and not(IncludeFilter.contains("media/removable_mounted") > 0) )
    {
        IncludeFilter.append("media/removable_mounted");
        execDirList();
    }
    else if (not(diskmonitorSettings::supportRemovableMedia()) and (IncludeFilter.contains("media/removable_mounted") > 0) )
    {
        IncludeFilter.remove("media/removable_mounted");
        execDirList();
    };

    diskmonitorSettings::setExcludedDevices( ls );
    diskmonitorSettings::writeConfig();

    setDirList();
}

diskmonitor::~diskmonitor()
{
/*
    string fn("DiskMonitor.log");
    ofstream logs(fn.c_str(),ios::app);
    if (!logs) kdFatal() << "failed to open log"<< endl;
    else logs << "log on. diskmonitor::~diskmonitor()" << endl;
*/
    Timer->stop();
    delete(Timer);
    KGlobal::locale()->removeCatalogue("diskmonitor");
    delete(mainView);
    DirLister->stop();
    delete(DirLister);
    config()->sync();
/*
    logs << "log off.diskmonitor::~diskmonitor()" << endl;
    logs.close();
*/
}

void diskmonitor::about()
{
    KAboutData data("diskmonitor", I18N_NOOP("Disk Monitor"), version,
                    I18N_NOOP("Disk Monitor : monitor free disk spaces."),
                    KAboutData::License_GPL_V2, "(c) 2006, Takeshi Hotta", "Depens on KDE 3.5.x. Tested in KDE3.5.4 on FC5",0, "takeshi.hotta@gmail.com");
    data.addAuthor("Takeshi Hotta", I18N_NOOP("Maintainer"), "takeshi.hotta@gmail.com");
    data.addCredit("Vincent Wagelaar", "Creator of Free Disk Space Applet");
    data.addCredit("Naoki Umehara", "Tester");
    KAboutApplication dialog(&data);
    dialog.exec();
}

void diskmonitor::help()
{
    KMessageBox::information(0, i18n("This is a help box"));
}

void diskmonitor::preferences()
{
    if ( KConfigDialog::showDialog( "settings" ) ) 
        return; 

    KConfigDialog* dialog = new KConfigDialog( this, "settings", diskmonitorSettings::self() ); 
    confWdg = new SettingForm( 0, "SettingForm" ); 
    dialog->addPage( confWdg, i18n("Settings"), "SettingePage" ); 

    connect( dialog, SIGNAL(settingsChanged()), this, SLOT(updateSettings()) ); 

//  set current colorTheme to radio button.
    switch (diskmonitorSettings::colorTheme())
    {
       case 0 : confWdg->StandardLCD->setChecked(true); break;
       case 1 : confWdg->ApplianceLCD->setChecked(true); break;
       default: confWdg->StandardLCD->setChecked(true);
    };

//  set Mountpoint Format button..
    if (not diskmonitorSettings::longFormat()) confWdg->shortFormat->setChecked(true);

//  set Sort by button..
    if (not diskmonitorSettings::sortByDevice()) confWdg->sortByMountpoint->setChecked(true);

//  excluded devices are stored into availableListBox.
    QListBox *lb =confWdg->kcfg_ExcludedDevices->availableListBox();
    lb->clear();
    lb->insertStringList(diskmonitorSettings::excludedDevices());

//  included devices are stored into selectedListBox.
    lb = confWdg->kcfg_ExcludedDevices->selectedListBox();
    lb->clear();
    QStringList l;

    for ( QPtrListIterator<diskmonitorItem> it( mainView->getItems() ) ; it.current() != 0 ; ++it )  l.append(it.current()->DeviceID());
    lb->insertStringList(l);

    dialog->show();
}

int diskmonitor::widthForHeight(int /*height*/) const
{
    if (mainView)
        return mainView->minimumSizeHint().width();
    else
        return 100;
}

int diskmonitor::heightForWidth(int /*width*/) const
{
    return height();
}

void diskmonitor::resizeEvent(QResizeEvent * /*e*/ )
{
/*
    string fn("DiskMonitor.log");
    ofstream logs(fn.c_str(),ios::app);
    if (!logs) kdFatal() << "failed to open log"<< endl;
    else logs << "log on.diskmonitor::resizeEvent<height()>"<< height() << endl;
*/
    uint mr = MainViewGrid::calculateMaxRowsSize(height());

//    logs << "logging.diskmonitor::resizeEvent<mr>"<< mr << "<mainView->maxRows()>" << mainView->maxRows() << "<mainView->numItems()>" << mainView->numItems()<< "<mainView->numViewLabel()>" << mainView->numViewLabel() << endl;
    if ( not( mr == mainView->maxRows() ) ) 
    {
        diskmonitorItemList items = mainView->getItems();
        delete(mainView);
        mainView= new MainViewGrid(mr, Qt::Vertical, this);
        if ( items.count() > 0) mainView->setItems(items);
//        logs << "logging.diskmonitor::resizeEvent<items.count()>"<< items.count() << endl;
    }
    else mainView->calculateSize(height());

    int dy = (height() - mainView->minimumSizeHint().height()) / 2;
    dy = dy < 0 ? 0 : dy;
    mainView->setGeometry(QRect(QPoint(0, dy), mainView->minimumSizeHint()));
/*
    logs << "log off.diskmonitor::resizeEvent<mainView->minimumSizeHint().height()>"<< mainView->minimumSizeHint().height() << endl;
    logs.close();
*/
}


extern "C"
{
    KPanelApplet* init( QWidget *parent, const QString& configFile)
    {
        KGlobal::locale()->insertCatalogue("diskmonitor");
        return new diskmonitor(configFile, KPanelApplet::Normal,
                             KPanelApplet::About | /*KPanelApplet::Help |*/ KPanelApplet::Preferences,
                             parent, "diskmonitor");
    }
}
