/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
#ifndef SEARCH_H 
#define SEARCH_H

/*Why me?
I exist to centralize search procedures, and to complicate them!

*/

#include <ctype.h>

#include "threads.h"

struct elog_srch_result
{
	int year;
	int month;
	int day;
	char *name;
	char *name2;
	char *fileName;
	char *txt;	
	int occurances;
};



struct _native_entry
{
	char *location;
	struct _native_entry *next;
};


struct _srch_incr_lst 
{
  char *fileName;
  struct _srch_incr_lst *next;
};

struct elog_srch_instr
{
  //user defined values:
  const char *term;
  const char *directory;
  void (*func)(void*) ; 
  //presently, these next three, are ignored.
  int lyrical;
  int caseSensitive;
  float accuracy;


  //internal values!
  struct _srch_incr_lst *list;
  int id;
};
struct elog_srch_instr *elog_srch_instr_new();
//NOTE:  You must call this to make these structs!

void elog_srch_instr_free(struct elog_srch_instr *instr);
//Pre:  All of its contents are already freed.
//Post:  instr is freed.
//Note:  It doesn't free any of the user defined values.


struct _native_entry *traverse(const char *directory, struct _native_entry *head);
//Pre: head is an initialized _native_entry struct, with location = directory's contents
//Post:  Will traverse the directory and infinite subdirectories and put them in a list
//		that you can go through, the returned value will be garbage.  Your head node will
//		still be the first item in the list.
bool stringMatch(char *c1, int len, int offset, const char *c2);
//Checks to see if c1+offset is equal to c2, c2 should be length len


#define ELOG_SRCH_STAT_GO 0 //you can search!
#define ELOG_SRCH_STAT_BUSY 1 //wait, if you search you'll lock up the program


struct elog_srch_result *searcher(const char *term, const char *fileName, unsigned int len);
//Reads calls reader(fileName), searches fileName for term.

void elog_srch_buffer(const char *directory);
//Pre: directory exists
//Post:  reads all files recursively and stores them for later searches.
  //these functions ^ v are to be implemented later!
void elog_srch_clearBuffer();
//Pre:
//Post:  clears the current buffer to free memory.

struct elog_srch_result *elog_srch_result_new();
//Post:  allocates and returns a srch_result

void elog_srch_result_free(struct elog_srch_result *result);
//Pre:  result is allocated and ok to free
//Post:  result and all it's contents are freed.

int elog_srch_lyrical(const char *text);
//Pre:  text is valid c string
//Post:  Tells if text contains poetry

int *elog_srch_listResults(const char *haystack, const char *needle);
//Pre:  haystack and needle are strings
//Post:  Return a list of int for all the results within haystack
//The array is terminated by a cell: -1
//Null is never returned.
	
char *elog_srch_strstr(const char *haystack, const char *needle, int allowed_diff, int len_needle);
//Pre: haystack and needle are strings
//Pre:  Returns a string within haystack where needle begins, or returns NULL
//Warning:  Does not create any memory, you are given part of haystack back!
const char *elog_srch_setDir(const char *directory);
//Pre: NULL directory returns current
//Post:  Holds this directory, and searches will be run on it with traverSearch

void elog_srch_search(const char *directory, const char *term, void (*functor)(void*));
//Pre:
//Post:  This function searches directory for term in the current thread.
//functor is called for matches.
//the functor gets an elog_srch_result struct.
//this struct is only valid for the duration of the function call


int elog_srch_incremental_search(struct elog_srch_instr *instr);
//Pre: 
//Post:  Just like _srch_search, but.  It returns after each search of a file.
//So you want to call it either by putting it on the second thread.
//or:  while (elog_srch_incremental_search(...)) ;
//it returns 0 when it's done.

//********Parameters...

float elog_srch_accuracy(float set);
//Pre:  set is between 0 and 1.0 and it will be set, else it is < 0 and the current val is returned.
//Post:

int elog_srch_caseSensitive(int set);
//Pre:  Set is 0,1,-1
//Post:  set is -1:  returns current setting; set is 0,1: sets the current value

int elog_srch_lyric(int set);
//PRe:  set is 0,1,-1
//Post:  set is -1:  returns current settings; set is 0,1: sets the current value
//Note:  Determines if the searches will only be on lyrical entries.


//************

#endif
