/***************************************************************************
 *   Copyright (C) 2006, 2007 by Niklas Knutsson                           *
 *   nq@altern.org                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef BUDGET_H
#define BUDGET_H

#include <qstring.h>
#include <qptrlist.h>
#include <qmap.h>

#include "account.h"
#include "transaction.h"
#include "security.h"

template<class type> class TransactionList : public QPtrList<type> {
	public:
		TransactionList() : QPtrList<type>() {};
	protected:
		int compareItems(QPtrCollection::Item item1, QPtrCollection::Item item2) {
			Transaction *t1 = (Transaction*) item1;
			Transaction *t2 = (Transaction*) item2;
			if(t1->date() > t2->date()) return 1;
			if(t1->date() < t2->date()) return -1;
			return 0;
		}
};

template<class type> class SplitTransactionList : public QPtrList<type> {
	public:
		SplitTransactionList() : QPtrList<type>() {};
	protected:
		int compareItems(QPtrCollection::Item item1, QPtrCollection::Item item2) {
			SplitTransaction *t1 = (SplitTransaction*) item1;
			SplitTransaction *t2 = (SplitTransaction*) item2;
			if(t1->date() > t2->date()) return 1;
			if(t1->date() < t2->date()) return -1;
			return 0;
		}
};

template<class type> class ScheduledTransactionList : public QPtrList<type> {
	public:
		ScheduledTransactionList() : QPtrList<type>() {};
	protected:
		int compareItems(QPtrCollection::Item item1, QPtrCollection::Item item2) {
			Transaction *t1 = ((ScheduledTransaction*) item1)->transaction();
			Transaction *t2 = ((ScheduledTransaction*) item2)->transaction();
			if(t1->date() > t2->date()) return 1;
			if(t1->date() < t2->date()) return -1;
			return 0;
		}
};

template<class type> class AccountList : public QPtrList<type> {
	public:
		AccountList() : QPtrList<type>() {};
	protected:
		int compareItems(QPtrCollection::Item item1, QPtrCollection::Item item2) {
			return QString::localeAwareCompare(((Account*) item1)->name(), ((Account*) item2)->name());
		}
};

template<class type> class SecurityList : public QPtrList<type> {
	public:
		SecurityList() : QPtrList<type>() {};
	protected:
		int compareItems(QPtrCollection::Item item1, QPtrCollection::Item item2) {
			return QString::localeAwareCompare(((Security*) item1)->name(), ((Security*) item2)->name());
		}
};

template<class type> class SecurityTradeList : public QPtrList<type> {
	public:
		SecurityTradeList() : QPtrList<type>() {};
	protected:
		int compareItems(QPtrCollection::Item item1, QPtrCollection::Item item2) {
			SecurityTrade *t1 = (SecurityTrade*) item1;
			SecurityTrade *t2 = (SecurityTrade*) item2;
			if(t1->date > t2->date) return 1;
			if(t1->date < t2->date) return -1;
			return 0;
		}
};

class Budget {

	public:

		Budget();
		~Budget();

		QString loadFile(QString filename, QString &errors);
		QString saveFile(QString filename, mode_t perms = 0600);
		
		void clear();

		void addTransaction(Transaction*);
		void removeTransaction(Transaction*, bool keep = false);

		void addScheduledTransaction(ScheduledTransaction*);
		void removeScheduledTransaction(ScheduledTransaction*, bool keep = false);

		void addSplitTransaction(SplitTransaction*);
		void removeSplitTransaction(SplitTransaction*, bool keep = false);
		
		void addAccount(Account*);
		void removeAccount(Account*, bool keep = false);

		bool accountHasTransactions(Account*);
		void moveTransactions(Account*, Account*);

		void addSecurity(Security*);
		void removeSecurity(Security*, bool keep = false);
		Security *findSecurity(QString name);

		bool securityHasTransactions(Security*);

		void scheduledTransactionDateModified(ScheduledTransaction*);
		void transactionDateModified(Transaction*, const QDate &olddate);
		void splitTransactionDateModified(SplitTransaction*, const QDate &olddate);
		void accountNameModified(Account*);
		void securityNameModified(Security*);
		void securityTradeDateModified(SecurityTrade*, const QDate &olddate);

		void addSecurityTrade(SecurityTrade*);
		void removeSecurityTrade(SecurityTrade*, bool keep = false);

		Account *findAccount(QString name);
		AssetsAccount *findAssetsAccount(QString name);
		IncomesAccount *findIncomesAccount(QString name);
		ExpensesAccount *findExpensesAccount(QString name);

		AccountList<IncomesAccount> incomesAccounts;
		AccountList<ExpensesAccount> expensesAccounts;
		AccountList<AssetsAccount> assetsAccounts;
		AccountList<Account> accounts;
		AssetsAccount *balancingAccount, *budgetAccount;
		TransactionList<Expense> expenses;
		TransactionList<Income> incomes;
		TransactionList<Transfer> transfers;
		TransactionList<Transaction> transactions;
		TransactionList<SecurityTransaction> securityTransactions;
		ScheduledTransactionList<ScheduledTransaction> scheduledTransactions;
		SplitTransactionList<SplitTransaction> splitTransactions;
		SecurityList<Security> securities;
		SecurityTradeList<SecurityTrade> securityTrades;

		QMap<int, IncomesAccount*> incomesAccounts_id;
		QMap<int, ExpensesAccount*> expensesAccounts_id;
		QMap<int, AssetsAccount*> assetsAccounts_id;
		QMap<int, Security*> securities_id;

};

#endif
