/*
 * GXMame
 *
 * 2002, 2003, Stéphane Pontier <shadow_walker@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include "common.h"

#include <unistd.h>
#include <string.h>
#include <stdio.h>

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <gdk-pixbuf/gdk-pixbuf-loader.h>

#include "callbacks.h"
#include "interface.h"
#include "gxmame.h"
#include "gui.h"
#include "mameio.h"
#include "icones.h"
#include "progression_window.h"
#include "unzip.h"
#include "io.h"


static void
create_toolbar (void);

static void
create_gamelist_popupmenu (void);

void
create_columns_popupmenu (void);

void
select_game (RomEntry *rom);

gboolean
set_game_history (const RomEntry *rom);

gboolean
set_game_info (const RomEntry *rom);

GtkWidget*
get_pixbuf (RomEntry *rom,
			screenshot_type sctype,
			int wwidth,
			int wheight);
void
update_screenshot_panel (RomEntry *rom);

void
change_screenshot (GtkWidget       *widget,
		        GdkEventButton  *event,
		        gpointer         user_data);

void
on_screenshot_notebook_switch_page (GtkNotebook *notebook,
                                    GtkNotebookPage *page,
                                    guint page_num,
                                    gpointer user_data);

/* these two prototype are declared here cause of the cross referencing that allows menu to be updated when drop box change and vice-versa */
/* executable changed, called when text is changed in executable dropbox (or by on_executable_selected() when changed from the menu) */
void
xecutable_changed (void);

/* executable selected from the menu */
void
on_executable_selected (GtkCheckMenuItem	*menuitem,
			 gpointer		 user_data);

void
set_game_pixbuff_from_iter (GtkTreeIter *iter, ZIP *zip, gint page_size);

gboolean
adjustement_scrolled_delayed (void);



GList *xmame_executables_combo_items = NULL;
GtkWidget *xmame_executables_entry;
GSList *exec_group_group = NULL;
guint timeout_icon;



void set_game_pixbuff_from_iter (GtkTreeIter *iter, ZIP *zip, gint page_size)
{
	RomEntry *tmprom;
	GdkRectangle rect;
	GtkTreePath *tree_path;

	gtk_tree_model_get (GTK_TREE_MODEL(main_gui.tree_model), iter, NUMBER_COLUMN+ROMENTRY, &tmprom, -1);
	tree_path = gtk_tree_model_get_path(GTK_TREE_MODEL(main_gui.tree_model),iter);
	gtk_tree_view_get_cell_area (GTK_TREE_VIEW(main_gui.displayed_list),
					tree_path,
					NULL, &rect);
	gtk_tree_path_free(tree_path);

	if (tmprom->has_roms == 1 &&
		tmprom->status &&
		(rect.y + rect.height)>0 &&
		(rect.y < page_size) &&
		!tmprom->icon_pixbuf)
	{
		tmprom->icon_pixbuf = get_icon_for_rom(tmprom, gui_prefs.ListFontHeight, zip);
		if (tmprom->icon_pixbuf)
		{
			if ((gui_prefs.current_mode==LIST_TREE) || (gui_prefs.current_mode==DETAILS_TREE))
				gtk_tree_store_set (GTK_TREE_STORE(main_gui.tree_model), iter,
					NUMBER_COLUMN+PIXBUF,tmprom->icon_pixbuf,
					-1);
			else
				gtk_list_store_set (GTK_LIST_STORE(main_gui.tree_model), iter,
					NUMBER_COLUMN+PIXBUF,tmprom->icon_pixbuf,
					-1);
		}
	}
}


gboolean adjustement_scrolled_delayed (void)
{
	GtkTreeIter iter;
	GtkTreeIter iter_child;
	GtkTreePath *tree_path;
	gint i;
	ZIP *zip;
	gchar *zipfile;
	gboolean valid;
	GtkAdjustment *vadj;

	/* open the zip file only at the begining */
	zipfile = g_build_filename(gui_prefs.IconDirectory, "icons.zip", NULL);
	zip = openzip(zipfile);

	/* Getting the vertical window area */
	vadj=gtk_scrolled_window_get_vadjustment (GTK_SCROLLED_WINDOW(main_gui.scrolled_window_games));

	/* Disable the callback */
	g_signal_handlers_block_by_func (G_OBJECT (gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(main_gui.scrolled_window_games))),
					(gpointer)adjustement_scrolled, NULL);

	if (visible_games>0)
	{
		valid = gtk_tree_model_get_iter_first (GTK_TREE_MODEL(main_gui.tree_model), &iter);
		set_game_pixbuff_from_iter(&iter,zip,(gint) (vadj->page_size));
		i=0;
			while ((i<visible_games) && (valid))
			{
				tree_path = gtk_tree_model_get_path(GTK_TREE_MODEL(main_gui.tree_model), &iter);
				if (gtk_tree_view_row_expanded(GTK_TREE_VIEW(main_gui.displayed_list), tree_path))
				{
					if (gtk_tree_model_iter_children (GTK_TREE_MODEL(main_gui.tree_model), &iter_child, &iter))
					{
						set_game_pixbuff_from_iter(&iter_child,zip, (gint) (vadj->page_size));
						while ((i<visible_games) && (gtk_tree_model_iter_next (GTK_TREE_MODEL(main_gui.tree_model), &iter_child)) )
						{
							set_game_pixbuff_from_iter(&iter_child,zip, (gint) (vadj->page_size));
							i++;
						}
					}
				}
				gtk_tree_path_free(tree_path);
				if ((i<visible_games))
				{
					valid=gtk_tree_model_iter_next (GTK_TREE_MODEL(main_gui.tree_model), &iter);
					if (valid)
					{
						set_game_pixbuff_from_iter(&iter,zip, (gint) (vadj->page_size));
						i++;
					}
				}
			}
	}

	/* Re-Enable the callback */
	g_signal_handlers_unblock_by_func (G_OBJECT (gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(main_gui.scrolled_window_games))),
					(gpointer) adjustement_scrolled, NULL);

	if (zip)
		closezip(zip);
	g_free(zipfile);
	return FALSE;
}


void adjustement_scrolled (GtkAdjustment *adjustment,
                                            gpointer user_data)
{
		if (timeout_icon)
			gtk_timeout_remove(timeout_icon);
		timeout_icon =
		    gtk_timeout_add(ICON_TIMEOUT,
				    (GtkFunction) adjustement_scrolled_delayed, NULL);
}


void init_gui ()
{
	GdkPixmap *testpix = NULL;
	GdkPixmap *testmask = NULL;
	GdkColor transparent;
	GtkWidget *screenshot_label1;
	GtkWidget *screenshot_label2;
	GtkWidget *screenshot_label3;
	GtkWidget *screenshot_label4;
	GtkWidget *screenshot_label5;
	GtkTooltips *tooltips;
	GdkPixbuf *my_appicon;
	GList *appicon_list=NULL;
	ZIP *zip=NULL;

	tooltips = gtk_tooltips_new ();

	/* Default Pixbuf once for all windows */
	my_appicon = get_icon_for_name("Gxmame", 16, zip, mamel_xpm);
	appicon_list = g_list_append (appicon_list, my_appicon);
	gtk_window_set_default_icon_list (appicon_list);

	/* Create the main window */
	MainWindow = create_MainWindow ();
	
	/* if the ListFont is empty or not loadable, use default font */
	gui_prefs.ListFontStruct = NULL;
	gui_prefs.ListFontHeight = 16;

	main_gui.filters_tree_model = NULL;
	/* Create the UI of the filter List */
	create_filterslist ();
	
	gtk_paned_set_position (main_gui.hpanedLeft, gui_prefs.Splitters[0]);
	gtk_paned_set_position (main_gui.hpanedRight, gui_prefs.Splitters[1]);

	if (!current_exec)
	{
		gtk_widget_set_sensitive(GTK_WIDGET(main_gui.audit_all_games_menu),FALSE);
	}
	gtk_widget_hide (GTK_WIDGET(main_gui.combo_progress_bar));

	/* set which items of the menu are active */
	/* need to block the call otherwise its trying to change the position but it has been already done */
	g_signal_handlers_block_by_func(G_OBJECT(main_gui.modify_the_menu),(gpointer)on_the_prefix_activate,NULL);
	gtk_check_menu_item_set_active(main_gui.modify_the_menu,gui_prefs.ModifyThe);
	g_signal_handlers_unblock_by_func(G_OBJECT(main_gui.modify_the_menu), (gpointer)on_the_prefix_activate,NULL);
	switch(gui_prefs.current_mode)
	{
		case(LIST):
			gtk_check_menu_item_set_active(main_gui.list_view_menu,TRUE);
			break;
		case(LIST_TREE):
			gtk_check_menu_item_set_active(main_gui.list_tree_view_menu,TRUE);
			break;
		case(DETAILS):
			gtk_check_menu_item_set_active(main_gui.details_view_menu,TRUE);
			break;
		case(DETAILS_TREE):
			gtk_check_menu_item_set_active(main_gui.details_tree_view_menu,TRUE);
			break;
	}
	if (!((gui_prefs.current_mode==LIST_TREE) || (gui_prefs.current_mode==DETAILS_TREE)))
	{
		gtk_widget_set_sensitive (GTK_WIDGET(main_gui.expand_all_menu),FALSE);
		gtk_widget_set_sensitive (GTK_WIDGET(main_gui.collapse_all_menu),FALSE);
	}
	gtk_check_menu_item_set_active(main_gui.toolbar_view_menu,gui_prefs.ShowToolBar);
	gtk_check_menu_item_set_active(main_gui.status_bar_view_menu,gui_prefs.ShowStatusBar);
	gtk_check_menu_item_set_active(main_gui.folder_list_menu,gui_prefs.ShowFolderList);
	gtk_check_menu_item_set_active(main_gui.screen_shot_menu,gui_prefs.ShowScreenShot);

	gtk_window_set_default_size(GTK_WINDOW(MainWindow),
				    gui_prefs.GUIWidth,
				    gui_prefs.GUIHeight);

	gtk_window_move(GTK_WINDOW(MainWindow),
				    gui_prefs.GUIPosX,
				    gui_prefs.GUIPosY);

	/* Show and hence realize mainwindow so that MainWindow->window is available */
	gtk_widget_show (MainWindow);
	/* Need to create the menu to have all button in the toolbar ??? not really needed */
	create_toolbar ();

	/* Screenshot Event Box */
	testpix = gdk_pixmap_create_from_xpm_d(MainWindow->window,
					&testmask,
					&transparent,
					gxmame_xpm);
	main_gui.main_screenshot = (GtkWidget *) gtk_image_new_from_pixmap (testpix,testmask);
	main_gui.screenshot_event_box = gtk_event_box_new();
	gtk_box_pack_start(main_gui.screenshot_hist_vbox,main_gui.screenshot_event_box,TRUE,TRUE,5);
	gtk_container_add(GTK_CONTAINER(main_gui.screenshot_event_box),GTK_WIDGET(main_gui.main_screenshot));
	gtk_widget_show (main_gui.screenshot_event_box);
	gtk_widget_show (main_gui.main_screenshot);

	/* Screenshot Notebook */
	main_gui.screenshot_notebook = gtk_notebook_new ();
	gtk_box_pack_start (main_gui.screenshot_hist_vbox, main_gui.screenshot_notebook, TRUE, TRUE, 0);
	gtk_widget_show (main_gui.screenshot_notebook);

	main_gui.screenshot1 = (GtkWidget *) gtk_image_new_from_pixmap (testpix,testmask);

	main_gui.screenshot_box1 = gtk_vbox_new (TRUE,0);
	gtk_widget_show (main_gui.screenshot_box1);
	gtk_container_add(GTK_CONTAINER(main_gui.screenshot_box1), GTK_WIDGET(main_gui.screenshot1));
	screenshot_label1 = gtk_label_new (_("Snap"));
	gtk_widget_show (screenshot_label1);
	gtk_notebook_append_page        (GTK_NOTEBOOK(main_gui.screenshot_notebook),
                                             main_gui.screenshot_box1,
                                             screenshot_label1);
	gtk_widget_show (main_gui.screenshot1);


	main_gui.screenshot2 = (GtkWidget *) gtk_image_new_from_pixmap (testpix,testmask);
	main_gui.screenshot_box2 = gtk_vbox_new (TRUE,0);
	gtk_widget_show (main_gui.screenshot_box2);
	gtk_container_add(GTK_CONTAINER(main_gui.screenshot_box2), GTK_WIDGET(main_gui.screenshot2));
	screenshot_label2 = gtk_label_new (_("Flyer"));
	gtk_widget_show (screenshot_label2);
	gtk_notebook_append_page        (GTK_NOTEBOOK(main_gui.screenshot_notebook),
                                             main_gui.screenshot_box2,
                                             screenshot_label2);
	gtk_widget_show (main_gui.screenshot2);

	main_gui.screenshot3 = (GtkWidget *) gtk_image_new_from_pixmap (testpix,testmask);
	main_gui.screenshot_box3 = gtk_vbox_new (TRUE,0);
	gtk_widget_show (main_gui.screenshot_box3);
	gtk_container_add(GTK_CONTAINER(main_gui.screenshot_box3), GTK_WIDGET(main_gui.screenshot3));
	screenshot_label3 = gtk_label_new (_("Cab"));
	gtk_widget_show (screenshot_label3);
	gtk_notebook_append_page        (GTK_NOTEBOOK(main_gui.screenshot_notebook),
                                             main_gui.screenshot_box3,
                                             screenshot_label3);
	gtk_widget_show (main_gui.screenshot3);

	main_gui.screenshot4 = (GtkWidget *) gtk_image_new_from_pixmap (testpix,testmask);
	main_gui.screenshot_box4 = gtk_vbox_new (TRUE,0);
	gtk_widget_show (main_gui.screenshot_box4);
	gtk_container_add(GTK_CONTAINER(main_gui.screenshot_box4), GTK_WIDGET(main_gui.screenshot4));
	screenshot_label4 = gtk_label_new (_("Marquee"));
	gtk_widget_show (screenshot_label4);
	gtk_notebook_append_page        (GTK_NOTEBOOK(main_gui.screenshot_notebook),
                                             main_gui.screenshot_box4,
                                             screenshot_label4);

	gtk_widget_show (main_gui.screenshot4);

	main_gui.screenshot5 = (GtkWidget *) gtk_image_new_from_pixmap (testpix,testmask);
	g_object_unref(testpix);
	if (testmask)
		g_object_unref(testmask);

	main_gui.screenshot_box5 = gtk_vbox_new (TRUE,0);
	gtk_widget_show (main_gui.screenshot_box5);
	gtk_container_add(GTK_CONTAINER(main_gui.screenshot_box5), GTK_WIDGET(main_gui.screenshot5));
	screenshot_label5 = gtk_label_new (_("Title"));
	gtk_widget_show (screenshot_label5);
	gtk_notebook_append_page        (GTK_NOTEBOOK(main_gui.screenshot_notebook),
                                             main_gui.screenshot_box5,
                                             screenshot_label5);
	gtk_widget_show (main_gui.screenshot5);

	g_signal_connect (G_OBJECT (main_gui.screenshot_notebook), "switch-page",
			    G_CALLBACK (on_screenshot_notebook_switch_page),
			    NULL);

	/* here we create the history box that will be filled later */
	main_gui.history_scrollwin = gtk_scrolled_window_new (NULL, NULL);
	gtk_container_set_border_width (GTK_CONTAINER (main_gui.history_scrollwin), 5);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (main_gui.history_scrollwin), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_box_pack_end(main_gui.screenshot_hist_vbox,main_gui.history_scrollwin,TRUE,TRUE,5);

	main_gui.history_buffer = gtk_text_buffer_new (NULL);
	main_gui.history_box = gtk_text_view_new_with_buffer (main_gui.history_buffer);
	gtk_text_view_set_cursor_visible (GTK_TEXT_VIEW(main_gui.history_box), FALSE);
	gtk_text_view_set_editable (GTK_TEXT_VIEW(main_gui.history_box), FALSE);
	gtk_text_view_set_wrap_mode (GTK_TEXT_VIEW(main_gui.history_box), GTK_WRAP_WORD);

	gtk_container_add (GTK_CONTAINER (main_gui.history_scrollwin), main_gui.history_box);

	gtk_widget_show (main_gui.history_scrollwin);
	gtk_widget_show (main_gui.history_box);

	/* Show Hide Screenshot */
	gtk_check_menu_item_set_active(main_gui.screen_shot_tab_menu,gui_prefs.ShowScreenShotTab);
	if  (gui_prefs.ShowScreenShotTab==FALSE)
		gtk_widget_hide (GTK_WIDGET(main_gui.screenshot_notebook));
	else
		gtk_widget_hide (GTK_WIDGET(main_gui.screenshot_event_box));

	/* Create the popup menu */
	create_gamelist_popupmenu ();
	create_columns_popupmenu ();
	
	/* Create the UI of the Game List */
	create_gamelist (gui_prefs.current_mode);

	/* Feed the Filters List */
	create_filterslist_content ();

	/* Feed the Game List */
	create_gamelist_content ();

	/* Need to set the size here otherwise it move when we create the gamelist */
	if (gui_prefs.ShowScreenShot)
		gtk_paned_set_position (main_gui.hpanedRight, gui_prefs.Splitters[1]);

	/* Grab focus on the game list */
	gtk_widget_grab_focus(main_gui.displayed_list);


	g_signal_connect (G_OBJECT (gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(main_gui.scrolled_window_games))), "changed",
	                  G_CALLBACK (adjustement_scrolled),
	                  NULL);
	g_signal_connect (G_OBJECT (gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(main_gui.scrolled_window_games))), "value-changed",
	                  G_CALLBACK (adjustement_scrolled),
	                  NULL);

	/* Need to set the notebook page here otherwise it segfault */
	gtk_notebook_set_current_page (GTK_NOTEBOOK(main_gui.screenshot_notebook), gui_prefs.ShowFlyer);

	if (!current_exec) {
		gxmame_message(ERROR,NULL,_("No xmame executable found"));
	} else if (list_ver.version==NULL) { /* list verification */	
		gxmame_message(ERROR,NULL,_("Problem to recognise the gamelist version.\nYou'd better build a new one.\nGo to Option and rebuild a game list"));
	} else if (!strcmp(list_ver.version,"none")) {
		gxmame_message(ERROR,NULL,_("Gamelist not available,\nyou need to build a new one.\nGo to Option and rebuild a game list"));
	} else if (!strcmp(list_ver.version,"unknown")) {
		gxmame_message(ERROR,NULL,_("Could not recognize format of gamelist.\nYou need to build a new gamelist."));
	} else if (!strcmp(list_ver.version,"too old")) {
		gxmame_message(ERROR,NULL,_("Gamelist from too old GXMame\nThe format of the gamelist has changed in this version.\nYou need to build a new gamelist."));
	} else if (gui_prefs.VersionCheck && (strcmp(current_exec->name,list_ver.name)||strcmp(current_exec->version,list_ver.version))) {
		gxmame_message(WARNING,NULL,
			_("the gamelist is from:\n%s %s\nand the current executable is:\n%s %s\nyou may want rebuild the game list"),
			list_ver.name,
			list_ver.version,
			current_exec->name,
			current_exec->version);
	}

}

/* executable changed, called when text is changed in executable dropbox (or by on_executable_selected() when changed from the menu) */
void xecutable_changed(void)
{
	gchar *used_text;
	GtkWidget *temp_menu_item;

	/* set current exec from name on xmame_executables_entry */
	used_text = gtk_editable_get_chars(GTK_EDITABLE(xmame_executables_entry),0,-1);
	temp_menu_item = GTK_WIDGET(g_object_get_data(G_OBJECT(MainWindow), used_text));
	current_exec = xmame_table_get_by_name(used_text);
	g_free(used_text);

	if (!current_exec)
		return;

	if (temp_menu_item)
	{
		g_signal_handlers_block_by_func(G_OBJECT(temp_menu_item), (gpointer)on_executable_selected,NULL);
		/* select the executable in the menu */
		gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (temp_menu_item), TRUE);	
		g_signal_handlers_unblock_by_func(G_OBJECT(temp_menu_item), (gpointer)on_executable_selected,NULL);
	}
	
	GXMAME_DEBUG("executable change to %s", current_exec->path);

	/* check if the executable is still valid */
	if (!xmame_executable_is_valid(current_exec))
	{
		gxmame_message(ERROR,NULL,_("%s is not a valid executable"), current_exec->path);
		gtk_widget_set_sensitive(GTK_WIDGET(main_gui.audit_all_games_menu),FALSE);
	} else
	{
		if (list_ver.version==NULL || !strcmp(list_ver.version,"unknown") || !strcmp(list_ver.version,"none"))
		{
			gxmame_message(ERROR,NULL,_("Gamelist not available,\nyou need to build a new one\ngo to Option and rebuild a game list"));
		} else if (gui_prefs.VersionCheck && (strcmp(current_exec->name,list_ver.name)||
					strcmp(current_exec->version,list_ver.version)))
		{
			gxmame_message(WARNING,NULL,
				_("the gamelist is from:\n%s %s\nand the current executable is:\n%s (%s) %s\nyou may want rebuild the game list"),
				list_ver.name,
				list_ver.version,
				current_exec->name,
				current_exec->target,
				current_exec->version);
		}
		gtk_widget_set_sensitive(GTK_WIDGET(main_gui.audit_all_games_menu),TRUE);
	}
}

/* executable selected from the menu */
void
on_executable_selected			(GtkCheckMenuItem	*menuitem,
					 gpointer		 user_data)
{
	const gchar *temp_text;
	if (menuitem->active)
	{
		temp_text = gtk_label_get_text(GTK_LABEL(GTK_BIN(menuitem)->child));
		GXMAME_DEBUG("selected: %s",temp_text);
		xmame_executables_entry = GTK_COMBO (main_gui.xmame_executables_combo)->entry;
		/* need to block the handler here and call the xecutable_changed() by hand otherwise, it's been called twice */
		g_signal_handlers_block_by_func(G_OBJECT(xmame_executables_entry), (gpointer)xecutable_changed,NULL);
		gtk_entry_set_text (GTK_ENTRY (xmame_executables_entry), temp_text);
		xecutable_changed();
		g_signal_handlers_unblock_by_func(G_OBJECT(xmame_executables_entry), (gpointer)xecutable_changed,NULL);
	}
	else
	{
		temp_text = gtk_label_get_text(GTK_LABEL(GTK_BIN(menuitem)->child));
		GXMAME_DEBUG("unselected: %s",temp_text);
	}

}

void list_free(gpointer element, gpointer nothing)
{
	g_free(element);
}

void add_exec_dropbox(void)
{
	gchar **exec_list;
	gchar *full_name;
	gint i;
	XmameExecutable *exec;
	GtkWidget *temp_menu_item;

	/* clean up the older menu */
	if (main_gui.executable_menu)
	{
		gtk_container_foreach(GTK_CONTAINER (main_gui.executable_menu),(GtkCallback)gtk_widget_destroy,NULL);
		/* need to set the GList to NULL otherwise segfault as soon as I set new executables
		   are all the other really needed ? */
		gtk_menu_item_remove_submenu (GTK_MENU_ITEM(main_gui.executables_title));
		gtk_widget_destroy(main_gui.executable_menu);
		exec_group_group = NULL;
	}

	/* recreate the menu */
	main_gui.executable_menu = gtk_menu_new ();
	gtk_widget_ref (GTK_WIDGET(main_gui.executable_menu));
	g_object_set_data_full (G_OBJECT (MainWindow), "executable_menu", main_gui.executable_menu,
					(GtkDestroyNotify) gtk_widget_unref);
	gtk_menu_item_set_submenu (GTK_MENU_ITEM (main_gui.executables_title), main_gui.executable_menu);

	/* recreate the combo box */
	main_gui.xmame_executables_combo = gtk_combo_new ();
	/*gtk_combo_set_value_in_list(GTK_COMBO(main_gui.xmame_executables_combo), TRUE, TRUE);*/
	gtk_widget_ref (main_gui.xmame_executables_combo);
	g_object_set_data_full (G_OBJECT (MainWindow), "xmame_executables_combo", main_gui.xmame_executables_combo,
				  (GtkDestroyNotify) gtk_widget_unref);
	/* should not use gtk_combo_set_value_in_list otherwise, is the entry has not a good value, we cant change directories*/
	/* gtk_combo_set_value_in_list (GTK_COMBO (xmame_executables_combo), TRUE, FALSE);*/
	gtk_widget_set_size_request (main_gui.xmame_executables_combo, 290, -1);
	/* need to set the glist null here otherwise, got an strange element that will crach the app */
	xmame_executables_combo_items = NULL;


	exec_list = xmame_table_get_all();

	/* Make sure we have a default executable
	   if we have anything in the table
	*/
	if (!current_exec)
		current_exec = xmame_table_get(exec_list[0]);

	for(i=0; exec_list[i];i++)
	{
		exec = xmame_table_get(exec_list[i]);

		/* will be freed after the combo box is populated */
		full_name = g_strdup_printf("%s (%s) %s", exec->name, exec->target, exec->version);
		GXMAME_DEBUG("Adding %s", full_name);
		/* to the combo box */
		xmame_executables_combo_items = g_list_append (xmame_executables_combo_items, (gpointer)full_name);
	
		/* to the menu */
		temp_menu_item = gtk_radio_menu_item_new_with_label (exec_group_group, full_name);

		exec_group_group = gtk_radio_menu_item_get_group (GTK_RADIO_MENU_ITEM (temp_menu_item));
		gtk_widget_ref (temp_menu_item);
		/* the menu is  referenced by the full_name of the executable */
		g_object_set_data_full (G_OBJECT (MainWindow), full_name, temp_menu_item,
	 			  (GtkDestroyNotify) gtk_widget_unref);
		gtk_widget_show (temp_menu_item);
		gtk_container_add (GTK_CONTAINER (main_gui.executable_menu), temp_menu_item);
		GXMAME_DEBUG("Comparing %s", exec->path);
		gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (temp_menu_item), !strcmp(exec->path, current_exec->path));

		g_signal_connect (G_OBJECT (temp_menu_item), "activate",
			G_CALLBACK (on_executable_selected),
			NULL);

		GXMAME_DEBUG("\t %s exec added checked to %i",exec->path,!strcmp(exec->path,current_exec->path));
		
	}
	if (exec_list)
		g_free(exec_list);

	/* test because we can have an existing but empty xmame_executables_hash_table,
	   that lead to a empty xmame_executables_combo_items */
	if (xmame_executables_combo_items!=NULL)
	{
		gtk_combo_set_popdown_strings (GTK_COMBO (main_gui.xmame_executables_combo), xmame_executables_combo_items);
		g_list_foreach(xmame_executables_combo_items, list_free, NULL);
		g_list_free (xmame_executables_combo_items);
	}

	gtk_toolbar_append_widget (main_gui.toolbar, main_gui.xmame_executables_combo, _("Executable"), NULL);
	gtk_widget_show (main_gui.xmame_executables_combo);

	xmame_executables_entry = GTK_COMBO (main_gui.xmame_executables_combo)->entry;
	gtk_editable_set_editable (GTK_EDITABLE (xmame_executables_entry), FALSE);
	gtk_widget_ref (xmame_executables_entry);
	g_object_set_data_full (G_OBJECT (MainWindow), "xmame_executables_entry", xmame_executables_entry,
				  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (xmame_executables_entry);

	if (current_exec)
	{
		full_name = g_strdup_printf("%s (%s) %s", current_exec->name, current_exec->target, current_exec->version);
		gtk_entry_set_text (GTK_ENTRY (xmame_executables_entry), full_name);
		g_free(full_name);
	}
	/* If i'm not letting GTK runs here, I got a GTK-warning
	   and if I try to open directory window then default option window,
	   GTK lose a reference to a window and segfault */
	while (gtk_events_pending()) gtk_main_iteration();
        g_signal_connect (G_OBJECT (xmame_executables_entry), "changed",
        		    G_CALLBACK (xecutable_changed),
        		    NULL);
}

void create_toolbar (void)
{
	GtkWidget *tmp_toolbar_icon;
	GtkWidget *filterShowButton;
	GtkWidget *snapShowButton;
	GtkWidget *list_view_button;
	GtkWidget *list_tree_view_button;
	GtkWidget *details_view_button;
	GtkWidget *details_tree_view_button;
	GtkWidget *modify_the_button;
	GtkWidget *aboutButton;

	GdkPixmap *tmp_pixmap_icon;
	GdkBitmap *tmp_mask_icon;
	GdkColor  transparent;


	/* addition of toolbar buttons */
	/* filters and snapshots buttons */
	tmp_pixmap_icon = gdk_pixmap_create_from_xpm_d(MainWindow->window,
					&tmp_mask_icon,
					&transparent,
					filtermenu_xpm);
	tmp_toolbar_icon = (GtkWidget *) gtk_image_new_from_pixmap (tmp_pixmap_icon,tmp_mask_icon);
	g_object_unref(tmp_pixmap_icon);
	if(tmp_mask_icon)
		g_object_unref(tmp_mask_icon);

	filterShowButton = gtk_toolbar_append_element (main_gui.toolbar,
        			      GTK_TOOLBAR_CHILD_TOGGLEBUTTON,
        			      NULL,
        			      _("Show Filters"),
        			      _("Toggle Folder List"), NULL,
        			      tmp_toolbar_icon, NULL, NULL);
	gtk_widget_ref (filterShowButton);
	g_object_set_data_full (G_OBJECT (MainWindow), "filterShowButton", filterShowButton,
        			  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (filterShowButton);
	main_gui.filterShowButton = GTK_TOGGLE_BUTTON(filterShowButton);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (filterShowButton), gui_prefs.ShowFolderList);

	tmp_pixmap_icon = gdk_pixmap_create_from_xpm_d(MainWindow->window,
					&tmp_mask_icon,
					&transparent,
					snapmenu_xpm);
	tmp_toolbar_icon = (GtkWidget *) gtk_image_new_from_pixmap (tmp_pixmap_icon,tmp_mask_icon);
	g_object_unref(tmp_pixmap_icon);
	if(tmp_mask_icon)
		g_object_unref(tmp_mask_icon);

	snapShowButton = gtk_toolbar_append_element (main_gui.toolbar,
        			      GTK_TOOLBAR_CHILD_TOGGLEBUTTON,
        			      NULL,
        			      _("Show Snaps"),
        			      _("Toggle Screenshot Panel"), NULL,
        			      tmp_toolbar_icon, NULL, NULL);
	gtk_widget_ref (snapShowButton);
	g_object_set_data_full (G_OBJECT (MainWindow), "snapShowButton", snapShowButton,
        			  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (snapShowButton);
	main_gui.snapShowButton = GTK_TOGGLE_BUTTON(snapShowButton);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (snapShowButton), gui_prefs.ShowScreenShot);

	gtk_toolbar_append_space (main_gui.toolbar);

	/* listing mode buttons */
	tmp_pixmap_icon = gdk_pixmap_create_from_xpm_d(MainWindow->window,
					&tmp_mask_icon,
					&transparent,
					viewdetail_xpm);
	tmp_toolbar_icon = (GtkWidget *) gtk_image_new_from_pixmap (tmp_pixmap_icon,tmp_mask_icon);
	g_object_unref(tmp_pixmap_icon);
	if(tmp_mask_icon)
		g_object_unref(tmp_mask_icon);

	list_view_button = gtk_toolbar_append_element (main_gui.toolbar,
        			      GTK_TOOLBAR_CHILD_TOGGLEBUTTON,
        			      NULL,
        			      _("List"),
        			      _("List"), NULL,
        			      tmp_toolbar_icon, NULL, NULL);
	gtk_widget_ref (list_view_button);
	g_object_set_data_full (G_OBJECT (MainWindow), "list_view_button", list_view_button,
        			  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (list_view_button);
	main_gui.list_view_button = GTK_TOGGLE_BUTTON(list_view_button);

	tmp_pixmap_icon = gdk_pixmap_create_from_xpm_d(MainWindow->window,
					&tmp_mask_icon,
					&transparent,
					viewtree_xpm);
	tmp_toolbar_icon = (GtkWidget *) gtk_image_new_from_pixmap (tmp_pixmap_icon,tmp_mask_icon);
	g_object_unref(tmp_pixmap_icon);
	if(tmp_mask_icon)
		g_object_unref(tmp_mask_icon);

	list_tree_view_button = gtk_toolbar_append_element (main_gui.toolbar,
        			      GTK_TOOLBAR_CHILD_TOGGLEBUTTON,
        			      NULL,
        			      _("List Tree"),
        			      _("List Tree"), NULL,
        			      tmp_toolbar_icon, NULL, NULL);
	gtk_widget_ref (list_tree_view_button);
	g_object_set_data_full (G_OBJECT (MainWindow), "list_tree_view_button", list_tree_view_button,
        			  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (list_tree_view_button);
	main_gui.list_tree_view_button = GTK_TOGGLE_BUTTON(list_tree_view_button);

	tmp_pixmap_icon = gdk_pixmap_create_from_xpm_d(MainWindow->window,
					&tmp_mask_icon,
					&transparent,
					viewdetail_xpm);
	tmp_toolbar_icon = (GtkWidget *) gtk_image_new_from_pixmap (tmp_pixmap_icon,tmp_mask_icon);
	g_object_unref(tmp_pixmap_icon);
	if(tmp_mask_icon)
		g_object_unref(tmp_mask_icon);

	details_view_button = gtk_toolbar_append_element (main_gui.toolbar,
        			      GTK_TOOLBAR_CHILD_TOGGLEBUTTON,
        			      NULL,
        			      _("Details"),
        			      _("Details"), NULL,
        			      tmp_toolbar_icon, NULL, NULL);
	gtk_widget_ref (details_view_button);
	g_object_set_data_full (G_OBJECT (MainWindow), "details_view_button", details_view_button,
        			  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (details_view_button);
	main_gui.details_view_button = GTK_TOGGLE_BUTTON(details_view_button);

	tmp_pixmap_icon = gdk_pixmap_create_from_xpm_d(MainWindow->window,
					&tmp_mask_icon,
					&transparent,
					viewtree_xpm);
	tmp_toolbar_icon = (GtkWidget *) gtk_image_new_from_pixmap (tmp_pixmap_icon,tmp_mask_icon);
	g_object_unref(tmp_pixmap_icon);
	if(tmp_mask_icon)
		g_object_unref(tmp_mask_icon);

	details_tree_view_button = gtk_toolbar_append_element (main_gui.toolbar,
        			      GTK_TOOLBAR_CHILD_TOGGLEBUTTON,
        			      NULL,
        			      _("Details Tree"),
        			      _("Details Tree"), NULL,
        			      tmp_toolbar_icon, NULL, NULL);
	gtk_widget_ref (details_tree_view_button);
	g_object_set_data_full (G_OBJECT (MainWindow), "details_tree_view_button", details_tree_view_button,
        			  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (details_tree_view_button);
	main_gui.details_tree_view_button = GTK_TOGGLE_BUTTON(details_tree_view_button);

	/* init the mode button */
	switch(gui_prefs.current_mode)
	{
		case(LIST):
			gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(list_view_button), TRUE);
			break;
		case(LIST_TREE):
			gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(list_tree_view_button), TRUE);
			break;
		case(DETAILS):
			gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(details_view_button), TRUE);
			break;
		case(DETAILS_TREE):
			gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(details_tree_view_button), TRUE);
			break;
	}

	gtk_toolbar_append_space (main_gui.toolbar);

	/* the trailer button */
	tmp_pixmap_icon = gdk_pixmap_create_from_xpm_d(MainWindow->window,
					&tmp_mask_icon,
					&transparent,
					thetrail_xpm);
	tmp_toolbar_icon = (GtkWidget *) gtk_image_new_from_pixmap (tmp_pixmap_icon,tmp_mask_icon);
	g_object_unref(tmp_pixmap_icon);
	if(tmp_mask_icon)
		g_object_unref(tmp_mask_icon);

	modify_the_button = gtk_toolbar_append_element (main_gui.toolbar,
        			      GTK_TOOLBAR_CHILD_TOGGLEBUTTON,
        			      NULL,
        			      _("Modify The"),
        			      _("Modify The"), NULL,
        			      tmp_toolbar_icon, NULL , NULL);
	gtk_widget_ref (modify_the_button);
	g_object_set_data_full (G_OBJECT (MainWindow), "modify_the_button", modify_the_button,
        			  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (modify_the_button);
	main_gui.modify_the_button = GTK_TOGGLE_BUTTON(modify_the_button);

	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(modify_the_button), gui_prefs.ModifyThe);

	gtk_toolbar_append_space (main_gui.toolbar);

	/* About button */
	tmp_pixmap_icon = gdk_pixmap_create_from_xpm_d(MainWindow->window,
					&tmp_mask_icon,
					&transparent,
					about_xpm);
	tmp_toolbar_icon = (GtkWidget *) gtk_image_new_from_pixmap (tmp_pixmap_icon,tmp_mask_icon);
	g_object_unref(tmp_pixmap_icon);
	if(tmp_mask_icon)
		g_object_unref(tmp_mask_icon);

	aboutButton = gtk_toolbar_append_element (main_gui.toolbar,
        			      GTK_TOOLBAR_CHILD_BUTTON,
        			      NULL,
        			      _("About"),
        			      _("About"), NULL,
        			      tmp_toolbar_icon, NULL, NULL);
	gtk_widget_ref (aboutButton);
	g_object_set_data_full (G_OBJECT (MainWindow), "aboutButton", aboutButton,
        			  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (aboutButton);


	/* executable dropbox*/
	gtk_toolbar_append_space (main_gui.toolbar);
	add_exec_dropbox();

	/* Connection of toolbar buttons signals*/
        g_signal_connect (G_OBJECT (filterShowButton), "toggled",
        		    G_CALLBACK (on_filterShowButton_toggled),
        		    NULL);
        g_signal_connect (G_OBJECT (snapShowButton), "toggled",
        		    G_CALLBACK (on_snapShowButton_toggled),
        		    NULL);
        g_signal_connect (G_OBJECT (list_view_button), "clicked",
        		    G_CALLBACK (on_mode_button_clicked),
        		    GINT_TO_POINTER(LIST));
        g_signal_connect (G_OBJECT (list_tree_view_button), "clicked",
        		    G_CALLBACK (on_mode_button_clicked),
        		    GINT_TO_POINTER(LIST_TREE));
        g_signal_connect (G_OBJECT (details_view_button), "clicked",
        		    G_CALLBACK (on_mode_button_clicked),
        		    GINT_TO_POINTER(DETAILS));
        g_signal_connect (G_OBJECT (details_tree_view_button), "clicked",
        		    G_CALLBACK (on_mode_button_clicked),
        		    GINT_TO_POINTER(DETAILS_TREE));
        g_signal_connect (G_OBJECT (modify_the_button), "toggled",
        		    G_CALLBACK(on_ModifyThe_toggled),
        		    NULL);
        g_signal_connect (G_OBJECT (aboutButton), "clicked",
        		    G_CALLBACK (on_about_activate),
        		    NULL);
}

void create_gamelist_popupmenu (void)
{
	GtkWidget *separator0;
	GtkWidget *separator1;
	GtkWidget *separator2;
	GtkWidget *separator3;
	GtkWidget *select_random_game;
	GtkWidget *play_record_menu;
	GtkWidget *playback_menu;
	GtkWidget *properties;
	GtkTooltips *tooltips;

	tooltips = gtk_tooltips_new ();

	/* build the gamelist popup menu */
	main_gui.popup_gamelist_menu = GTK_MENU(gtk_menu_new ());
	gtk_widget_ref (GTK_WIDGET(main_gui.popup_gamelist_menu));
	g_object_set_data_full (G_OBJECT (MainWindow), "popup_gamelist_menu", main_gui.popup_gamelist_menu,
				  (GtkDestroyNotify) gtk_widget_unref);


	main_gui.popup_gamelist_play = gtk_menu_item_new_with_label (_("Play"));
	gtk_widget_ref (main_gui.popup_gamelist_play);
	g_object_set_data_full (G_OBJECT (MainWindow), "popup_gamelist_play", main_gui.popup_gamelist_play,
				  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (main_gui.popup_gamelist_play);
	gtk_container_add (GTK_CONTAINER (main_gui.popup_gamelist_menu), main_gui.popup_gamelist_play);
	gtk_tooltips_set_tip (tooltips, main_gui.popup_gamelist_play, _("Play currently selected game"), NULL);
  	g_signal_connect (G_OBJECT (main_gui.popup_gamelist_play), "activate",
                      	    G_CALLBACK (on_play_activate),
                            NULL);

	separator0 = gtk_menu_item_new ();
	gtk_widget_ref (separator0);
	g_object_set_data_full (G_OBJECT (MainWindow), "separator0", separator0,
				  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (separator0);
	gtk_container_add (GTK_CONTAINER (main_gui.popup_gamelist_menu), separator0);
	gtk_widget_set_sensitive (separator0, FALSE);

	play_record_menu = gtk_image_menu_item_new_from_stock(GTK_STOCK_SAVE, NULL);
	gtk_label_set_text_with_mnemonic (GTK_LABEL (GTK_BIN (play_record_menu)->child), _("Play and Record Input..."));
	gtk_widget_ref (play_record_menu);
	gtk_widget_show (play_record_menu);
	gtk_container_add (GTK_CONTAINER (main_gui.popup_gamelist_menu), play_record_menu);
	g_object_set_data_full (G_OBJECT (MainWindow), "play_record_menu", play_record_menu,
				  (GtkDestroyNotify) gtk_widget_unref);
	gtk_tooltips_set_tip (tooltips, play_record_menu, _("Record a game for later playback"), NULL);
	g_signal_connect (G_OBJECT (play_record_menu), "activate",
			    G_CALLBACK (on_play_and_record_input_activate),
			    NULL);

	playback_menu = gtk_image_menu_item_new_from_stock(GTK_STOCK_OPEN, NULL);
	gtk_label_set_text_with_mnemonic (GTK_LABEL (GTK_BIN (playback_menu)->child), _("Playback Input..."));
	gtk_widget_ref (playback_menu);
	gtk_widget_show (playback_menu);
	gtk_container_add (GTK_CONTAINER (main_gui.popup_gamelist_menu), playback_menu);
	g_object_set_data_full (G_OBJECT (MainWindow), "playback_menu", playback_menu,
				  (GtkDestroyNotify) gtk_widget_unref);
	gtk_tooltips_set_tip (tooltips, playback_menu, _("Playback a recorded game"), NULL);
	g_signal_connect (G_OBJECT (playback_menu), "activate",
			    G_CALLBACK (on_playback_input_activate),
			    NULL);

	separator1 = gtk_menu_item_new ();
	gtk_widget_ref (separator1);
	g_object_set_data_full (G_OBJECT (MainWindow), "separator1", separator1,
				  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (separator1);
	gtk_container_add (GTK_CONTAINER (main_gui.popup_gamelist_menu), separator1);
	gtk_widget_set_sensitive (separator1, FALSE);

	main_gui.popup_gamelist_add_to_favorites = gtk_image_menu_item_new_from_stock(GTK_STOCK_ADD, NULL);
	gtk_label_set_text_with_mnemonic (GTK_LABEL (GTK_BIN (main_gui.popup_gamelist_add_to_favorites)->child), _("Add to 'Favorites'"));
	gtk_widget_ref (main_gui.popup_gamelist_add_to_favorites);
	g_object_set_data_full (G_OBJECT (MainWindow), "popup_gamelist_add_to_favorites", main_gui.popup_gamelist_add_to_favorites,
				  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (main_gui.popup_gamelist_add_to_favorites);
	gtk_container_add (GTK_CONTAINER (main_gui.popup_gamelist_menu), main_gui.popup_gamelist_add_to_favorites);
	gtk_tooltips_set_tip (tooltips, main_gui.popup_gamelist_add_to_favorites, _("Add this game to your 'Favorites' game folder"), NULL);
	g_signal_connect (G_OBJECT (main_gui.popup_gamelist_add_to_favorites), "activate",
			    G_CALLBACK (on_add_to_favorites_activate),
			    NULL);

	main_gui.popup_gamelist_remove_from_favorites = gtk_image_menu_item_new_from_stock(GTK_STOCK_REMOVE, NULL);
	gtk_label_set_text_with_mnemonic (GTK_LABEL (GTK_BIN (main_gui.popup_gamelist_remove_from_favorites)->child), _("Remove from 'Favorites'"));
	gtk_widget_ref (main_gui.popup_gamelist_remove_from_favorites);
	g_object_set_data_full (G_OBJECT (MainWindow), "popup_gamelist_remove_from_favorites", main_gui.popup_gamelist_remove_from_favorites,
				  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (main_gui.popup_gamelist_remove_from_favorites);
	gtk_container_add (GTK_CONTAINER (main_gui.popup_gamelist_menu), main_gui.popup_gamelist_remove_from_favorites);
	gtk_tooltips_set_tip (tooltips, main_gui.popup_gamelist_remove_from_favorites, _("Remove this game from your 'Favorites' game folder"), NULL);
	g_signal_connect (G_OBJECT (main_gui.popup_gamelist_remove_from_favorites), "activate",
			    G_CALLBACK (on_remove_from_favorites_activate),
			    NULL);

	separator2 = gtk_menu_item_new ();
	gtk_widget_ref (separator2);
	g_object_set_data_full (G_OBJECT (MainWindow), "separator2", separator2,
				  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (separator2);
	gtk_container_add (GTK_CONTAINER (main_gui.popup_gamelist_menu), separator2);
	gtk_widget_set_sensitive (separator2, FALSE);

	select_random_game = gtk_menu_item_new_with_label (_("Select Random Game"));
	gtk_widget_ref (select_random_game);
	g_object_set_data_full (G_OBJECT (MainWindow), "select_random_game", select_random_game,
				  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (select_random_game);
	gtk_container_add (GTK_CONTAINER (main_gui.popup_gamelist_menu), select_random_game);
	gtk_tooltips_set_tip (tooltips, select_random_game, _("Randomly select a game"), NULL);
	g_signal_connect (G_OBJECT (select_random_game), "activate",
			    G_CALLBACK (on_select_random_game_activate),
			    NULL);

	separator3 = gtk_menu_item_new ();
	gtk_widget_ref (separator3);
	g_object_set_data_full (G_OBJECT (MainWindow), "separator3", separator3,
				  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (separator3);
	gtk_container_add (GTK_CONTAINER (main_gui.popup_gamelist_menu), separator3);
	gtk_widget_set_sensitive (separator3, FALSE);

	properties = gtk_image_menu_item_new_from_stock(GTK_STOCK_PROPERTIES, NULL);
	gtk_widget_ref (properties);
	g_object_set_data_full (G_OBJECT (MainWindow), "properties", properties,
				  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (properties);
	gtk_container_add (GTK_CONTAINER (main_gui.popup_gamelist_menu), properties);
	gtk_tooltips_set_tip (tooltips, properties, _("Display the properties of the selected game"), NULL);
	g_signal_connect (G_OBJECT (properties), "activate",
			    G_CALLBACK (on_properties_activate),
			    NULL);
}

void create_columns_popupmenu (void)
{
	GtkTooltips *tooltips;
	GtkWidget *cancel;
	GtkWidget *separator1;

	tooltips = gtk_tooltips_new ();

	/* build the gamelist popup menu */
	main_gui.popup_column_menu = GTK_MENU(gtk_menu_new ());
	gtk_widget_ref (GTK_WIDGET(main_gui.popup_column_menu));
	g_object_set_data_full (G_OBJECT (MainWindow), "popup_column_menu", main_gui.popup_column_menu,
				  (GtkDestroyNotify) gtk_widget_unref);


	main_gui.popup_column_hide = gtk_menu_item_new_with_label (_("Hide Column"));
	gtk_widget_ref (main_gui.popup_column_hide);
	g_object_set_data_full (G_OBJECT (MainWindow), "hide", main_gui.popup_column_hide,
				  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (main_gui.popup_column_hide);
	gtk_container_add (GTK_CONTAINER (main_gui.popup_column_menu), main_gui.popup_column_hide);
	gtk_tooltips_set_tip (tooltips, main_gui.popup_column_hide, _("Hide Column"), NULL);
  	g_signal_connect (G_OBJECT (main_gui.popup_column_hide), "activate",
                      	    G_CALLBACK (on_column_hide_activate),
                            NULL);

	main_gui.popup_column_layout = gtk_menu_item_new_with_label (_("Column Layout..."));
	gtk_widget_ref (main_gui.popup_column_layout);
	gtk_widget_show (main_gui.popup_column_layout);
	gtk_container_add (GTK_CONTAINER (main_gui.popup_column_menu), main_gui.popup_column_layout);
	g_object_set_data_full (G_OBJECT (MainWindow), "column_layout_menu", main_gui.popup_column_layout,
				  (GtkDestroyNotify) gtk_widget_unref);
	gtk_tooltips_set_tip (tooltips, main_gui.popup_column_layout, _("Show, Hide or Order Columns"), NULL);
	g_signal_connect (G_OBJECT (main_gui.popup_column_layout), "activate",
			    G_CALLBACK (on_column_layout_activate),
			    NULL);

	separator1 = gtk_menu_item_new ();
	gtk_widget_ref (separator1);
	g_object_set_data_full (G_OBJECT (MainWindow), "separator1", separator1,
				  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (separator1);
	gtk_container_add (GTK_CONTAINER (main_gui.popup_column_menu), separator1);
	gtk_widget_set_sensitive (separator1, FALSE);

	cancel = gtk_image_menu_item_new_from_stock(GTK_STOCK_CANCEL, NULL);
	gtk_widget_ref (cancel);
	gtk_widget_show (cancel);
	gtk_container_add (GTK_CONTAINER (main_gui.popup_column_menu), cancel);
	g_object_set_data_full (G_OBJECT (MainWindow), "cancel", cancel,
				  (GtkDestroyNotify) gtk_widget_unref);
	gtk_tooltips_set_tip (tooltips, main_gui.popup_column_layout, _("Cancel"), NULL);
}

void create_filterslist (void)
{
	GtkTreeViewColumn *column;
	GtkCellRenderer *renderer;
	GtkTreeSelection *select;

	GXMAME_DEBUG ("DISPLAY FILTERS LIST");

	/* Tree View Creation */
	main_gui.filters_displayed_list = gtk_tree_view_new ();

	/* Column */
	column = gtk_tree_view_column_new ();
	gtk_tree_view_column_set_title  (GTK_TREE_VIEW_COLUMN(column), " ");
	gtk_tree_view_column_set_sort_column_id (column, 0);
	renderer = gtk_cell_renderer_text_new ();
	gtk_tree_view_column_pack_end (GTK_TREE_VIEW_COLUMN(column),
                                             renderer,
                                             FALSE);
	gtk_tree_view_column_set_attributes (GTK_TREE_VIEW_COLUMN(column), renderer,
			"text", 0,
			NULL);
	g_object_set(renderer, "ypad", 0, "yalign", 0.5, NULL);
	renderer = gtk_cell_renderer_pixbuf_new ();
	gtk_tree_view_column_pack_end (GTK_TREE_VIEW_COLUMN(column),
                                             renderer,
                                             FALSE);
	gtk_tree_view_column_add_attribute (GTK_TREE_VIEW_COLUMN(column), renderer,
			"pixbuf", 2);
	g_object_set(renderer, "xalign", 1.0, "ypad", 0, NULL);
	gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_AUTOSIZE);
	gtk_tree_view_append_column (GTK_TREE_VIEW (main_gui.filters_displayed_list), column);

	gtk_container_add (GTK_CONTAINER(main_gui.scrolled_window_filters), main_gui.filters_displayed_list);
	gtk_widget_show_all (main_gui.scrolled_window_filters);
	gtk_tree_view_set_headers_visible (GTK_TREE_VIEW(main_gui.filters_displayed_list), FALSE);

	/* Callback - Row has been selected */
	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (main_gui.filters_displayed_list));
	gtk_tree_selection_set_mode (select, GTK_SELECTION_SINGLE);
	g_signal_connect (G_OBJECT (select), "changed",
			G_CALLBACK (on_filter_row_selected),
			NULL);
	/* Callback - Row has been collapsed */
	g_signal_connect (G_OBJECT (main_gui.filters_displayed_list), "row-collapsed",
			G_CALLBACK (on_filter_row_collapsed),
			NULL);
}


static void add_filter_to_list(const gchar *name,
		simple_filter *folder_filter, GtkTreeIter *iter)
{
	gchar *filter_key;
	GdkPixbuf *icon_pixbuf;

	icon_pixbuf = get_icon_for_filter(folder_filter);
	gtk_tree_store_append (GTK_TREE_STORE(main_gui.filters_tree_model), iter, NULL);
	gtk_tree_store_set (GTK_TREE_STORE(main_gui.filters_tree_model), iter,
		0,		name,
		1,		folder_filter,
		2,		icon_pixbuf,
		3,		folder_filter->FolderID,
		-1);

	filter_key = g_strdup_printf("Filter%i", folder_filter->FolderID);
	g_object_set_data_full(G_OBJECT(main_gui.filters_tree_model), filter_key, folder_filter, g_free);
	g_free(filter_key);
}

void create_filterslist_content (void)
{
	static char *empty_value = " ";
	static char *dash_value = "-";
	static char *neogeo_value = "neogeo";
	static char *cps1_value = "cps1";
	static char *cps2_value = "cps2";
	static char *trackball_value = "trackball";
	GtkTreeIter iter;
	GtkTreeIter sub_iter;
	GtkTreeSelection *select;
	gchar *text;
	GdkPixbuf *icon_pixbuf;
	simple_filter *folder_filter;
	ZIP* zip;
	gchar *zipfile;
	GList *listpointer;
	gint folder_ID_selected;
	gint i;
	gboolean valid,is_root;
	gchar *filter_key;
	gboolean entry_found=FALSE;
	RomEntry *tmprom = NULL;

	GXMAME_DEBUG ("POPULATE FILTERS LIST");

	/* Free old tree model */
	if (main_gui.filters_tree_model)
	{
		gtk_tree_store_clear(GTK_TREE_STORE(main_gui.filters_tree_model));
		g_object_unref(main_gui.filters_tree_model);
	}

	/* Tree Model Structure */
	main_gui.filters_tree_model = (GtkTreeModel *) gtk_tree_store_new (4,
			G_TYPE_STRING,
			G_TYPE_POINTER,
			GDK_TYPE_PIXBUF,
			G_TYPE_INT);

	folder_ID_selected = NUMBER_FOLDER;

	/* ALL GAMES */
	folder_filter = g_new(simple_filter,1);
	folder_filter->type = DRIVER;
	folder_filter->is = FALSE;
	folder_filter->value = empty_value;
	folder_filter->FolderID = ALL;
	folder_filter->update_list = TRUE;
	add_filter_to_list(_("All Games"), folder_filter, &iter);

	/* AVAILABLES */
	folder_filter = g_new(simple_filter,1);
	folder_filter->type = HAS_ROMS;
	folder_filter->is = TRUE;
	folder_filter->value = NULL;
	folder_filter->int_value = 1;
	folder_filter->FolderID = AVAILABLE;
	folder_filter->update_list = TRUE;
	add_filter_to_list(_("Available"), folder_filter, &iter);

	current_filter = folder_filter;

	/* UNAVAILABLE */
	folder_filter = g_new(simple_filter,1);
	folder_filter->type = HAS_ROMS;
	folder_filter->is = FALSE;
	folder_filter->value = NULL;
	folder_filter->int_value = 1;
	folder_filter->FolderID = UNAVAILABLE;
	folder_filter->update_list = TRUE;
	add_filter_to_list(_("Unavailable"), folder_filter, &iter);

	/* NEOGEO */
	folder_filter = g_new(simple_filter,1);
	folder_filter->type = DRIVER;
	folder_filter->is = TRUE;
	folder_filter->value = neogeo_value;
	folder_filter->FolderID = NEOGEO;
	folder_filter->update_list = TRUE;
	add_filter_to_list(_("Neo-Geo"), folder_filter, &iter);

	/* CPS1 */
	folder_filter = g_new(simple_filter,1);
	folder_filter->type = DRIVER;
	folder_filter->is = TRUE;
	folder_filter->value = cps1_value;
	folder_filter->FolderID = CPS1;
	folder_filter->update_list = TRUE;
	add_filter_to_list(_("CPS1"), folder_filter, &iter);
	
	/* CPS2 */
	folder_filter = g_new(simple_filter,1);
	folder_filter->type = DRIVER;
	folder_filter->is = TRUE;
	folder_filter->value = cps2_value;
	folder_filter->FolderID = CPS2;
	folder_filter->update_list = TRUE;
	add_filter_to_list(_("CPS2"), folder_filter, &iter);

	/* MANUFACTURERS */
	folder_filter = g_new(simple_filter,1);
	folder_filter->type = DRIVER;
	folder_filter->is = FALSE;
	folder_filter->value = empty_value;
	folder_filter->FolderID = MANUFACTURERS;
	folder_filter->update_list = FALSE;
	add_filter_to_list(_("Manufacturer"), folder_filter, &iter);

	/* MANUFACTURERS - LIST */
	for(listpointer = g_list_first(manufacturers);
			(listpointer != NULL);
			listpointer = g_list_next(listpointer))
	{
		folder_filter = g_new(simple_filter,1);
		folder_filter->type = MANU;
		folder_filter->is = TRUE;
		folder_filter->value = (gchar *) listpointer->data;
		folder_filter->FolderID = folder_ID_selected++;
		folder_filter->update_list = TRUE;
		icon_pixbuf = get_icon_for_filter(folder_filter);
		gtk_tree_store_append (GTK_TREE_STORE(main_gui.filters_tree_model), &sub_iter, &iter);
		gtk_tree_store_set (GTK_TREE_STORE(main_gui.filters_tree_model), &sub_iter,
			0,		(gchar *)listpointer->data,
			1,		folder_filter,
			2,		icon_pixbuf,
			3,		folder_filter->FolderID,
			-1);

		filter_key = g_strdup_printf("Filter%i", folder_filter->FolderID);
		g_object_set_data_full(G_OBJECT(main_gui.filters_tree_model), filter_key, folder_filter, g_free);
		g_free(filter_key);
	}

	/* YEARS */
	folder_filter = g_new(simple_filter,1);
	folder_filter->type = DRIVER;
	folder_filter->is = FALSE;
	folder_filter->value = empty_value;
	folder_filter->FolderID = YEARS;
	folder_filter->update_list = FALSE;
	add_filter_to_list(_("Year"), folder_filter, &iter);

	/* YEARS - LIST */
	for(listpointer = g_list_last(years);
				(listpointer != NULL);
				listpointer = g_list_previous(listpointer))
	{
		if (!strcmp(listpointer->data, "-"))
			text = _("<unknown>");
		else
			text = (gchar* ) listpointer->data;
		folder_filter = g_new(simple_filter,1);
		folder_filter->type = YEAR;
		folder_filter->is = TRUE;
		folder_filter->value = (gchar *) listpointer->data;
		folder_filter->FolderID = folder_ID_selected++;
		folder_filter->update_list = TRUE;
		icon_pixbuf = get_icon_for_filter(folder_filter);
		gtk_tree_store_append (GTK_TREE_STORE(main_gui.filters_tree_model), &sub_iter, &iter);
		gtk_tree_store_set (GTK_TREE_STORE(main_gui.filters_tree_model), &sub_iter,
			0,		text,
			1,		folder_filter,
			2,		icon_pixbuf,
			3,		folder_filter->FolderID,
			-1);

		filter_key = g_strdup_printf("Filter%i", folder_filter->FolderID);
		g_object_set_data_full(G_OBJECT(main_gui.filters_tree_model), filter_key, folder_filter, g_free);
		g_free(filter_key);
	}


	/* WORKING */
	folder_filter = g_new(simple_filter,1);
	folder_filter->type = STATUS;
	folder_filter->is = TRUE;
	folder_filter->value = NULL;
	folder_filter->int_value = CORRECT;
	folder_filter->FolderID = WORKING;
	folder_filter->update_list = TRUE;
	add_filter_to_list(_("Working"), folder_filter, &iter);

	/* NONWORKING */
	folder_filter = g_new(simple_filter,1);
	folder_filter->type = STATUS;
	folder_filter->is = FALSE;
	folder_filter->value = NULL;
	folder_filter->int_value = CORRECT;
	folder_filter->FolderID = NONWORKING;
	folder_filter->update_list = TRUE;
	add_filter_to_list(_("Non-Working"), folder_filter, &iter);
	
	/* CUSTOM */
	folder_filter = g_new(simple_filter,1);
	folder_filter->type = DRIVER;
	folder_filter->is = FALSE;
	folder_filter->value = empty_value;
	folder_filter->FolderID = CUSTOMS;
	folder_filter->update_list = FALSE;
	add_filter_to_list(_("Custom"), folder_filter, &iter);

	/* CUSTOM - FAVORITES */
	folder_filter = g_new(simple_filter,1);
	folder_filter->type = FAVORITE;
	folder_filter->is = TRUE;
	folder_filter->value = NULL;
	folder_filter->FolderID = FAVORITES;
	folder_filter->update_list = TRUE;
	icon_pixbuf = get_icon_for_filter(folder_filter);
	gtk_tree_store_append (GTK_TREE_STORE(main_gui.filters_tree_model), &sub_iter, &iter);
	gtk_tree_store_set (GTK_TREE_STORE(main_gui.filters_tree_model), &sub_iter,
		0,		_("Favorites"),
		1,		folder_filter,
		2,		icon_pixbuf,
		3,		folder_filter->FolderID,
		-1);

	filter_key = g_strdup_printf("Filter%i", folder_filter->FolderID);
	g_object_set_data_full(G_OBJECT(main_gui.filters_tree_model), filter_key, folder_filter, g_free);
	g_free(filter_key);

	/* CUSTOM - PLAYED */
	folder_filter = g_new(simple_filter,1);
	folder_filter->type = TIMESPLAYED;
	folder_filter->is = FALSE;
	folder_filter->value = NULL;
	folder_filter->int_value = 0;
	folder_filter->FolderID = PLAYED;
	folder_filter->update_list = TRUE;
	icon_pixbuf = get_icon_for_filter(folder_filter);
	gtk_tree_store_append (GTK_TREE_STORE(main_gui.filters_tree_model), &sub_iter, &iter);
	gtk_tree_store_set (GTK_TREE_STORE(main_gui.filters_tree_model), &sub_iter,
		0,		_("Played"),
		1,		folder_filter,
		2,		icon_pixbuf,
		3,		folder_filter->FolderID,
		-1);

	filter_key = g_strdup_printf("Filter%i", folder_filter->FolderID);
	g_object_set_data_full(G_OBJECT(main_gui.filters_tree_model), filter_key, folder_filter, g_free);
	g_free(filter_key);

	/* ORIGINALS */
	folder_filter = g_new(simple_filter,1);
	folder_filter->type = CLONE;
	folder_filter->is = TRUE;
	folder_filter->value = dash_value;
	folder_filter->FolderID = ORIGINALS;
	folder_filter->update_list = TRUE;
	add_filter_to_list(_("Originals"), folder_filter, &iter);

	/* CLONES */
	folder_filter = g_new(simple_filter,1);
	folder_filter->type = CLONE;
	folder_filter->is = FALSE;
	folder_filter->value = dash_value;
	folder_filter->FolderID = CLONES;
	folder_filter->update_list = TRUE;
	add_filter_to_list(_("Clones"), folder_filter, &iter);

	/* RASTERS */
	folder_filter = g_new(simple_filter,1);
	folder_filter->type = VECTOR;
	folder_filter->is = FALSE;
	folder_filter->value = NULL;
	folder_filter->FolderID = RASTERS;
	folder_filter->update_list = TRUE;
	add_filter_to_list(_("Raster"), folder_filter, &iter);

	/* VECTORS */
	folder_filter = g_new(simple_filter,1);
	folder_filter->type = VECTOR;
	folder_filter->is = TRUE;
	folder_filter->value = NULL;
	folder_filter->FolderID = VECTORS;
	folder_filter->update_list = TRUE;
	add_filter_to_list(_("Vector"), folder_filter, &iter);

	/* TRACKBALL */
	folder_filter = g_new(simple_filter,1);
	folder_filter->type = CONTROL;
	folder_filter->is = TRUE;
	folder_filter->value = trackball_value;
	folder_filter->FolderID = TRACKBALL;
	folder_filter->update_list = TRUE;
	add_filter_to_list(_("Trackball"), folder_filter, &iter);

	/* STEREO */
	folder_filter = g_new(simple_filter,1);
	folder_filter->type = CHANNELS;
	folder_filter->is = TRUE;
	folder_filter->value = NULL;
	folder_filter->int_value = 2;
	folder_filter->FolderID = STEREO;
	folder_filter->update_list = TRUE;
	add_filter_to_list(_("Stereo"), folder_filter, &iter);

	/* Drivers */
	folder_filter = g_new(simple_filter,1);
	folder_filter->type = DRIVER;
	folder_filter->is = FALSE;
	folder_filter->value = empty_value;
	folder_filter->FolderID = DRIVERS;
	folder_filter->update_list = FALSE;
	add_filter_to_list(_("Driver"), folder_filter, &iter);

	/* Drivers - LIST*/
	for(listpointer = g_list_first(drivers);
				(listpointer != NULL);
				listpointer = g_list_next(listpointer))
	{
		if (!strcmp(listpointer->data, "-"))
			text = _("<unknown>");
		else
			text = (gchar* ) listpointer->data;
		folder_filter = g_new(simple_filter,1);
		folder_filter->type = DRIVER;
		folder_filter->is = TRUE;
		folder_filter->value = (gchar *) listpointer->data;
		folder_filter->FolderID = folder_ID_selected++;
		folder_filter->update_list = TRUE;
		icon_pixbuf = get_icon_for_filter(folder_filter);
		gtk_tree_store_append (GTK_TREE_STORE(main_gui.filters_tree_model), &sub_iter, &iter);
		gtk_tree_store_set (GTK_TREE_STORE(main_gui.filters_tree_model), &sub_iter,
			0,		text,
			1,		folder_filter,
			2,		icon_pixbuf,
			3,		folder_filter->FolderID,
			-1);

		filter_key = g_strdup_printf("Filter%i", folder_filter->FolderID);
		g_object_set_data_full(G_OBJECT(main_gui.filters_tree_model), filter_key, folder_filter, g_free);
		g_free(filter_key);
	}

	/* CATEGORY */
	folder_filter = g_new(simple_filter,1);
	folder_filter->type = CATEGORY;
	folder_filter->is = FALSE;
	folder_filter->value = empty_value;
	folder_filter->FolderID = CATEGORIES;
	folder_filter->update_list = FALSE;
	add_filter_to_list(_("Category"), folder_filter, &iter);

	/* CATEGORY - LIST */
	for(listpointer = g_list_first(categories);
				(listpointer != NULL);
				listpointer = g_list_next(listpointer))
	{
		folder_filter = g_new(simple_filter,1);
		folder_filter->type = CATEGORY;
		folder_filter->is = TRUE;
		folder_filter->value = (gchar *)listpointer->data;
		folder_filter->FolderID = folder_ID_selected++;
		folder_filter->update_list = TRUE;
		icon_pixbuf = get_icon_for_filter(folder_filter);
		gtk_tree_store_append (GTK_TREE_STORE(main_gui.filters_tree_model), &sub_iter, &iter);
		gtk_tree_store_set (GTK_TREE_STORE(main_gui.filters_tree_model), &sub_iter,
			0,		(gchar *)listpointer->data,
			1,		folder_filter,
			2,		icon_pixbuf,
			3,		folder_filter->FolderID,
			-1);

		filter_key = g_strdup_printf("Filter%i", folder_filter->FolderID);
		g_object_set_data_full(G_OBJECT(main_gui.filters_tree_model), filter_key, folder_filter, g_free);
		g_free(filter_key);
	}
	/* CATEGORY - UNKNOWN (Only if some roms don't have a category defined) */
	for(listpointer = g_list_first(roms), entry_found = FALSE;
			(listpointer != NULL);
			listpointer = g_list_next(listpointer))
	{
		tmprom = (RomEntry *) listpointer->data;
		if (!strncmp(tmprom->category,_("Unknown"),MAX_CATEGORY))
		{
			entry_found = TRUE;
			break;
		}
	}
	if (entry_found)
	{
		folder_filter = g_new(simple_filter,1);
		folder_filter->type = CATEGORY;
		folder_filter->is = TRUE;
		folder_filter->value = (gchar *) _("Unknown");
		folder_filter->FolderID = folder_ID_selected++;
		folder_filter->update_list = TRUE;
		icon_pixbuf = get_icon_for_filter(folder_filter);
		gtk_tree_store_append (GTK_TREE_STORE(main_gui.filters_tree_model), &sub_iter, &iter);
		gtk_tree_store_set (GTK_TREE_STORE(main_gui.filters_tree_model), &sub_iter,
				0,		_("Unknown"),
				1,		folder_filter,
				2,		icon_pixbuf,
				3,		folder_filter->FolderID,
				-1);

		filter_key = g_strdup_printf("Filter%i", folder_filter->FolderID);
		g_object_set_data_full(G_OBJECT(main_gui.filters_tree_model), filter_key, folder_filter, g_free);
		g_free(filter_key);
	}

	/* VERSION */
	folder_filter = g_new(simple_filter,1);
	folder_filter->type = MAMEVER;
	folder_filter->is = FALSE;
	folder_filter->value = empty_value;
	folder_filter->FolderID = VERSIONS;
	folder_filter->update_list = FALSE;
	add_filter_to_list(_("Version"), folder_filter, &iter);

	/* VERSION - LIST */
	for(listpointer = g_list_first(versions);
				(listpointer != NULL);
				listpointer = g_list_next(listpointer))
	{
		folder_filter = g_new(simple_filter,1);
		folder_filter->type = MAMEVER;
		folder_filter->is = TRUE;
		folder_filter->value = (gchar *)listpointer->data;
		folder_filter->FolderID = folder_ID_selected++;
		folder_filter->update_list = TRUE;
		icon_pixbuf = get_icon_for_filter(folder_filter);
		gtk_tree_store_append (GTK_TREE_STORE(main_gui.filters_tree_model), &sub_iter, &iter);
		gtk_tree_store_set (GTK_TREE_STORE(main_gui.filters_tree_model), &sub_iter,
			0,		(gchar *) listpointer->data,
			1,		folder_filter,
			2,		icon_pixbuf,
			3,		folder_filter->FolderID,
			-1);

		filter_key = g_strdup_printf("Filter%i", folder_filter->FolderID);
		g_object_set_data_full(G_OBJECT(main_gui.filters_tree_model), filter_key, folder_filter, g_free);
		g_free(filter_key);
	}
	/* VERSION - UNKNOWN (Only if some roms don't have a version defined) */
	for(listpointer = g_list_first(roms), entry_found = FALSE;
			(listpointer != NULL);
			listpointer = g_list_next(listpointer))
	{
		tmprom = (RomEntry *) listpointer->data;
		if (!strncmp(tmprom->mame_ver_added,_("Unknown"),MAX_MAMEVER))
		{
			entry_found = TRUE;
			break;
		}
	}
	if (entry_found)
	{
		folder_filter = g_new(simple_filter,1);
		folder_filter->type = MAMEVER;
		folder_filter->is = TRUE;
		folder_filter->value = (gchar *) _("Unknown");
		folder_filter->FolderID = folder_ID_selected++;
		folder_filter->update_list = TRUE;
		icon_pixbuf = get_icon_for_filter(folder_filter);
		gtk_tree_store_append (GTK_TREE_STORE(main_gui.filters_tree_model), &sub_iter, &iter);
		gtk_tree_store_set (GTK_TREE_STORE(main_gui.filters_tree_model), &sub_iter,
				0,		_("Unknown"),
				1,		folder_filter,
				2,		icon_pixbuf,
				3,		folder_filter->FolderID,
				-1);

		filter_key = g_strdup_printf("Filter%i", folder_filter->FolderID);
		g_object_set_data_full(G_OBJECT(main_gui.filters_tree_model), filter_key, folder_filter, g_free);
		g_free(filter_key);
	}



	/* Link Tree Model with Tree View + Sort */
	gtk_tree_view_set_model (GTK_TREE_VIEW(main_gui.filters_displayed_list), GTK_TREE_MODEL(main_gui.filters_tree_model));
	gtk_tree_sortable_set_sort_column_id (GTK_TREE_SORTABLE (main_gui.filters_tree_model), 3, GTK_SORT_ASCENDING);


	/* Select the default row */
	valid = gtk_tree_model_get_iter_first (GTK_TREE_MODEL(main_gui.filters_tree_model), &iter);
	gtk_tree_model_get (GTK_TREE_MODEL(main_gui.filters_tree_model), &iter, 1, &folder_filter, -1);
	is_root=TRUE;
	i=0;
	while ((folder_filter->FolderID!=gui_prefs.FolderID) && (valid))
	{
		if (gtk_tree_model_iter_has_child (GTK_TREE_MODEL(main_gui.filters_tree_model),&iter))
		{
			if (gtk_tree_model_iter_children (GTK_TREE_MODEL(main_gui.filters_tree_model), &sub_iter, &iter))
			{
				gtk_tree_model_get (GTK_TREE_MODEL(main_gui.filters_tree_model), &sub_iter, 1, &folder_filter, -1);
				is_root=FALSE;
				i++;
				while ((folder_filter->FolderID!=gui_prefs.FolderID) && (gtk_tree_model_iter_next (GTK_TREE_MODEL(main_gui.filters_tree_model), &sub_iter)) )
				{
					gtk_tree_model_get (GTK_TREE_MODEL(main_gui.filters_tree_model), &sub_iter, 1, &folder_filter, -1);
					i++;
				}
			}
		}
		if (folder_filter->FolderID!=gui_prefs.FolderID)
		{
			valid=gtk_tree_model_iter_next (GTK_TREE_MODEL(main_gui.filters_tree_model), &iter);
			if (valid)
			{
				gtk_tree_model_get (GTK_TREE_MODEL(main_gui.filters_tree_model), &iter, 1, &folder_filter, -1);
				is_root=TRUE;
				i++;
			}
		}
	}
	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (main_gui.filters_displayed_list));
	if (folder_filter->FolderID!=gui_prefs.FolderID)
	{
		valid = gtk_tree_model_get_iter_first (GTK_TREE_MODEL(main_gui.filters_tree_model), &iter);
		is_root=TRUE;
	}

	/* try to get icons in the zip file */
	zipfile = g_build_filename(gui_prefs.IconDirectory, "icons.zip", NULL);
	
	zip = openzip(zipfile);
	g_free(zipfile);

	if (is_root)
	{
		GtkTreePath *tree_path;
		g_signal_handlers_block_by_func (G_OBJECT (select), (gpointer)on_filter_row_selected, NULL);
		tree_path = gtk_tree_model_get_path (GTK_TREE_MODEL(main_gui.filters_tree_model), &iter);
		gtk_tree_view_set_cursor (GTK_TREE_VIEW (main_gui.filters_displayed_list),
						tree_path,
						NULL, FALSE);
		gtk_tree_path_free(tree_path);
		g_signal_handlers_unblock_by_func (G_OBJECT (select), (gpointer)on_filter_row_selected, NULL);
		gtk_tree_model_get (main_gui.filters_tree_model, &iter, 1, &folder_filter, -1);
		current_filter = folder_filter;
		icon_pixbuf = get_icon_for_name("foldopen", gui_prefs.ListFontHeight, zip, foldopen_xpm);
		gtk_tree_store_set (GTK_TREE_STORE(main_gui.filters_tree_model), &iter,
			2,		icon_pixbuf,
			-1);
		/* Scroll to selection */
		tree_path = gtk_tree_model_get_path (GTK_TREE_MODEL(main_gui.filters_tree_model), &iter);
		gtk_tree_view_scroll_to_cell (GTK_TREE_VIEW(main_gui.filters_displayed_list), tree_path, NULL, TRUE, 0.5, 0);
		gtk_tree_path_free(tree_path);
	}else
	{
		GtkTreePath *tree_path;
		tree_path = gtk_tree_model_get_path (GTK_TREE_MODEL(main_gui.filters_tree_model), &iter);
		valid = gtk_tree_view_expand_row (GTK_TREE_VIEW (main_gui.filters_displayed_list),
				tree_path,
				TRUE);
		gtk_tree_path_free(tree_path);
		g_signal_handlers_block_by_func (G_OBJECT (select), (gpointer)on_filter_row_selected, NULL);
		tree_path = gtk_tree_model_get_path (GTK_TREE_MODEL(main_gui.filters_tree_model), &sub_iter);
		gtk_tree_view_set_cursor (GTK_TREE_VIEW (main_gui.filters_displayed_list),
						tree_path,
						NULL, FALSE);
		gtk_tree_path_free(tree_path);
		g_signal_handlers_unblock_by_func (G_OBJECT (select), (gpointer)on_filter_row_selected, NULL);
		gtk_tree_model_get (main_gui.filters_tree_model, &sub_iter, 1, &folder_filter, -1);
		current_filter = folder_filter;
		icon_pixbuf = get_icon_for_name("foldopen", gui_prefs.ListFontHeight, zip, foldopen_xpm);
		gtk_tree_store_set (GTK_TREE_STORE(main_gui.filters_tree_model), &sub_iter,
			2,		icon_pixbuf,
			-1);
		/* Scroll to selection */
		tree_path = gtk_tree_model_get_path (GTK_TREE_MODEL(main_gui.filters_tree_model), &sub_iter);
		gtk_tree_view_scroll_to_cell (GTK_TREE_VIEW(main_gui.filters_displayed_list), tree_path, NULL, TRUE, 0.5, 0);
		gtk_tree_path_free(tree_path);
	}

	if (zip)
		closezip(zip);
}

void hide_filters (void)
{
	gui_prefs.ShowFolderList=0;
	gui_prefs.Splitters[0] = main_gui.scrolled_window_filters->allocation.width;
	gtk_paned_set_position (main_gui.hpanedLeft, 0);
	gtk_container_remove(GTK_CONTAINER(main_gui.hpanedLeft),GTK_WIDGET(main_gui.scrolled_window_filters));
}

void show_filters (void)
{
	gui_prefs.ShowFolderList=1;
	gtk_paned_set_position (main_gui.hpanedLeft, gui_prefs.Splitters[0]);
	gtk_paned_pack1 (main_gui.hpanedLeft, GTK_WIDGET(main_gui.scrolled_window_filters), FALSE, FALSE);
}

void hide_snaps (void)
{
	gui_prefs.ShowScreenShot=0;
	gui_prefs.Splitters[1] = main_gui.scrolled_window_games->allocation.width;
	gtk_paned_set_position (main_gui.hpanedRight, -1);
	gtk_container_remove(GTK_CONTAINER(main_gui.hpanedRight),GTK_WIDGET(main_gui.screenshot_hist_frame));
}

void show_snaps (void)
{
	gui_prefs.ShowScreenShot=1;
	gtk_paned_set_position (main_gui.hpanedRight, gui_prefs.Splitters[1]);
	gtk_paned_pack2 (main_gui.hpanedRight, GTK_WIDGET(main_gui.screenshot_hist_frame), FALSE, FALSE);
}


void hide_snaps_tab (void)
{
	gui_prefs.ShowScreenShotTab=0;
	gtk_widget_hide (GTK_WIDGET(main_gui.screenshot_notebook));
	gtk_widget_show (GTK_WIDGET(main_gui.screenshot_event_box));
	update_screenshot_panel(gui_prefs.DefaultGame_p);
}


void show_snaps_tab (void)
{
	gui_prefs.ShowScreenShotTab=1;
	gtk_widget_hide (GTK_WIDGET(main_gui.screenshot_event_box));
	gtk_widget_show (GTK_WIDGET(main_gui.screenshot_notebook));
	gtk_notebook_set_current_page (GTK_NOTEBOOK(main_gui.screenshot_notebook),gui_prefs.ShowFlyer);
	update_screenshot_panel(gui_prefs.DefaultGame_p);
}


void hide_toolbar (void)
{
	gui_prefs.ShowToolBar=0;
	gtk_widget_hide (GTK_WIDGET(main_gui.hseparator2));
	gtk_widget_hide (GTK_WIDGET(main_gui.toolbar));
	gtk_widget_hide (GTK_WIDGET(main_gui.hseparator1));
}


void show_toolbar (void)
{
	gui_prefs.ShowToolBar=1;
	gtk_widget_show (GTK_WIDGET(main_gui.hseparator2));
	gtk_widget_show (GTK_WIDGET(main_gui.toolbar));
	gtk_widget_show (GTK_WIDGET(main_gui.hseparator1));
}


void hide_status_bar (void)
{
	gui_prefs.ShowStatusBar=0;
	gtk_widget_hide (GTK_WIDGET(main_gui.tri_status_bar));
}


void show_status_bar (void)
{
	gui_prefs.ShowStatusBar=1;
	gtk_widget_show (GTK_WIDGET(main_gui.tri_status_bar));
}


/* get an icon for a rom, if not found, try the original game if the game is a clone */
GdkPixbuf *get_icon_for_rom (RomEntry *rom, guint size, ZIP *zip)
{
	GdkPixbuf *pixbuf = NULL, *scaled_pixbuf = NULL;
	gchar filename[MAX_ROMNAME +1], *filename2;
	GError **error=NULL;

	if (rom!=NULL)
	{
		filename2 = g_strdup_printf("%s/%s.ico",gui_prefs.IconDirectory,rom->romname);
		pixbuf = gdk_pixbuf_new_from_file (filename2,error);
		g_free(filename2);
		/* no picture found try parent game if any*/
		if ((pixbuf == NULL) && strcmp(rom->cloneof,"-"))
		{
			filename2 = g_strdup_printf("%s/%s.ico",gui_prefs.IconDirectory,rom->cloneof);
			pixbuf = gdk_pixbuf_new_from_file (filename,error);
			g_free(filename2);
		}
		/* we havent found the picture in the directory, maybe we could try in a zipfile */
		if (pixbuf == NULL)
		{
			struct zipent* zipent;
			char *tmp_buffer;
			if (zip != 0)
			{
				GdkPixbufLoader *loader;
				rewindzip(zip);
				g_snprintf(filename, MAX_ROMNAME +1 , "%s.",rom->romname);
				while ((zipent = readzip(zip)) != 0)
				{
					/* this should allows to find any format of picture in the zip, not only bmp */
					if (!strncmp(filename,zipent->name,strlen(rom->romname)+1))
					{
						tmp_buffer = read_zipentry(zip, zipent);
						if(tmp_buffer)
						{	/* if the file successfully uncompress, try to load it in a pixbuf loader */
							loader = gdk_pixbuf_loader_new ();
							if (!gdk_pixbuf_loader_write (loader, (guchar *)tmp_buffer, zipent->uncompressed_size,error)) {
								GXMAME_DEBUG("Error while uncompressing %s ",zipent->name);
							}
							else
							{
								gdk_pixbuf_loader_close (loader,error);
								pixbuf = gdk_pixbuf_loader_get_pixbuf (loader);
							}
							g_free(tmp_buffer);
							tmp_buffer = 0;
						}
						/* prevent to read all zip file if we have found the picture's game (uncompressed successfuly or not) */
						break;
					}
				}
				/* no picture found try parent game if any*/
				if ((pixbuf == NULL) && strcmp(rom->cloneof,"-"))
				{	
					rewindzip(zip);
					GXMAME_DEBUG("picture not found, trying clone: %s",rom->cloneof);
					g_snprintf(filename, MAX_ROMNAME +1 , "%s.",rom->cloneof);
					while ((zipent = readzip(zip)) != 0)
					{
						/* this should allows to find any format of picture in the zip, not only bmp */
						if (!strncmp(filename,zipent->name,strlen(rom->cloneof)+1))
						{
							tmp_buffer = read_zipentry(zip, zipent);
							if(tmp_buffer)
							{	/* if the file successfully uncompress, try to load it in a pixbuf loader */
								loader = gdk_pixbuf_loader_new ();
								if (!gdk_pixbuf_loader_write (loader, (guchar *)tmp_buffer, zipent->uncompressed_size,error)) {
									GXMAME_DEBUG("Error while uncompressing %s ",zipent->name);
								}
								else
								{
									gdk_pixbuf_loader_close (loader,error);
									pixbuf = gdk_pixbuf_loader_get_pixbuf (loader);
								}

								g_free(tmp_buffer);
								tmp_buffer = 0;
							}
							/* prevent to read all zip file if we have found the picture's game (uncompressed successfuly or not) */
							break;
						}
					}
				}
			}
			
		}
		if (pixbuf != NULL)
		{
			scaled_pixbuf = gdk_pixbuf_scale_simple(pixbuf,
								size,
 			             				size,
 			             				GDK_INTERP_BILINEAR);
			g_object_unref (pixbuf);
		}
	}
	return scaled_pixbuf;
}


/* get an icon with a specific name */
GdkPixbuf *get_icon_for_name(gchar *name, guint size, ZIP *zip,  char **alternative)
{
	GdkPixbuf *pixbuf = NULL, *scaled_pixbuf = NULL;
	GError **error=NULL;
	gchar *filename;

	if ((name!=NULL) && (size>0))
	{
		filename = g_strdup_printf("%s/%s.ico",gui_prefs.IconDirectory,name);
		pixbuf = gdk_pixbuf_new_from_file (filename, error);
		g_free(filename);
		/* we havent found the picture in the directory, maybe we could try in a zipfile */
		if (pixbuf == NULL)
		{
			struct zipent* zipent;
			char *tmp_buffer;
			if (zip != 0)
			{
				GdkPixbufLoader *loader;
				rewindzip(zip);
				filename = g_strdup_printf("%s.",name);
				while ((zipent = readzip(zip)) != 0)
				{
					/* this should allows to find any format of picture in the zip, not only bmp */
					if (!strncmp(filename, zipent->name, strlen(name)+1))
					{
						tmp_buffer = read_zipentry(zip, zipent);
						if(tmp_buffer)
						{	/* if the file successfully uncompress, try to load it in a pixbuf loader */
							loader = gdk_pixbuf_loader_new ();
							if (!gdk_pixbuf_loader_write (loader, (guchar *)tmp_buffer, zipent->uncompressed_size,error)) {
								GXMAME_DEBUG("Error while uncompressing %s ",zipent->name);
							}
							else
							{
								gdk_pixbuf_loader_close (loader,error);
								pixbuf = gdk_pixbuf_loader_get_pixbuf (loader);
							}
							g_free(tmp_buffer);
							tmp_buffer = 0;
						}
						/* prevent to read all zip file if we have found the picture's game (uncompressed successfuly or not) */
						break;
					}
				}
				g_free(filename);
			}
		}
		/* haven't found anything, draw back to the alternative pixmap */
		if (pixbuf == NULL)
		{
			pixbuf = gdk_pixbuf_new_from_xpm_data((const char **)alternative);
		}
		/*dont need to test since in any case we got a picture (from file or internal pixmap)*/
		scaled_pixbuf = gdk_pixbuf_scale_simple(pixbuf,
							size,
 		             				size,
 		             				GDK_INTERP_BILINEAR);
		g_object_unref (pixbuf);
	}
	return scaled_pixbuf;
}

GdkPixbuf *
get_icon_for_filter (simple_filter *current_filter)
{
	GdkPixbuf *icon_pixbuf = NULL;
	ZIP *zip;
	gchar *zipfile;

	/* try to get icons in the zip file */
	zipfile = g_build_filename(gui_prefs.IconDirectory, "icons.zip", NULL);
	zip = openzip(zipfile);
	g_free(zipfile);

	switch (current_filter->FolderID) {
		case ALL:
			icon_pixbuf = get_icon_for_name("folder", gui_prefs.ListFontHeight, zip, folder_xpm);
			break;
		case AVAILABLE:
			icon_pixbuf = get_icon_for_name("foldavail", gui_prefs.ListFontHeight, zip, foldavail_xpm);
			break;
		case UNAVAILABLE:
			icon_pixbuf = get_icon_for_name("foldunav", gui_prefs.ListFontHeight, zip, foldunav_xpm);
			break;
		case NEOGEO:
			icon_pixbuf = get_icon_for_name("neo-geo", gui_prefs.ListFontHeight, zip, neogeo_xpm);
			break;
		case CPS1:
			icon_pixbuf = get_icon_for_name("cps", gui_prefs.ListFontHeight, zip, cps_xpm);
			break;
		case CPS2:
			icon_pixbuf = get_icon_for_name("cps", gui_prefs.ListFontHeight, zip, cps_xpm);
			break;
		case MANUFACTURERS:
			icon_pixbuf = get_icon_for_name("foldmanu", gui_prefs.ListFontHeight, zip, foldmanu_xpm);
			break;
		case YEARS:
			icon_pixbuf = get_icon_for_name("foldyear", gui_prefs.ListFontHeight, zip, foldyear_xpm);
			break;
		case WORKING:
			icon_pixbuf = get_icon_for_name("working", gui_prefs.ListFontHeight, zip, working_xpm);
			break;
		case NONWORKING:
			icon_pixbuf = get_icon_for_name("nonwork", gui_prefs.ListFontHeight, zip, nonwork_xpm);
			break;
		case CUSTOMS:
			icon_pixbuf = get_icon_for_name("custom", gui_prefs.ListFontHeight, zip, custom_xpm);
			break;
		case PLAYED:
			icon_pixbuf = get_icon_for_name("custom", gui_prefs.ListFontHeight, zip, custom_xpm);
			break;
		case FAVORITES:
			icon_pixbuf = get_icon_for_name("custom", gui_prefs.ListFontHeight, zip, custom_xpm);
			break;
		case ORIGINALS:
			icon_pixbuf = get_icon_for_name("folder", gui_prefs.ListFontHeight, zip, folder_xpm);
			break;
		case CLONES:
			icon_pixbuf = get_icon_for_name("folder", gui_prefs.ListFontHeight, zip, folder_xpm);
			break;
		case RASTERS:
			icon_pixbuf = get_icon_for_name("folder", gui_prefs.ListFontHeight, zip, folder_xpm);
			break;
		case VECTORS:
			icon_pixbuf = get_icon_for_name("folder", gui_prefs.ListFontHeight, zip, folder_xpm);
			break;
		case TRACKBALL:
			icon_pixbuf = get_icon_for_name("folder", gui_prefs.ListFontHeight, zip, folder_xpm);
			break;
		case STEREO:
			icon_pixbuf = get_icon_for_name("sound", gui_prefs.ListFontHeight, zip, sound_xpm);
			break;
		case CATEGORIES:
			icon_pixbuf = get_icon_for_name("foldcat", gui_prefs.ListFontHeight, zip, foldcat_xpm);
			break;
		case VERSIONS:
			icon_pixbuf = get_icon_for_name("foldvers", gui_prefs.ListFontHeight, zip, foldvers_xpm);
			break;
		case DRIVERS:
			icon_pixbuf = get_icon_for_name("folder", gui_prefs.ListFontHeight, zip, folder_xpm);
			break;
		default:
			switch (current_filter->type) {
			case MANU:
				icon_pixbuf = get_icon_for_name("manufact", gui_prefs.ListFontHeight, zip, manufact_xpm);
				break;
			case YEAR:
				icon_pixbuf = get_icon_for_name("year", gui_prefs.ListFontHeight, zip, year_xpm);
				break;
			case CATEGORY:
				icon_pixbuf = get_icon_for_name("foldcat", gui_prefs.ListFontHeight, zip, foldcat_xpm);
				break;
			case MAMEVER:
				icon_pixbuf = get_icon_for_name("foldvers", gui_prefs.ListFontHeight, zip, foldvers_xpm);
				break;
			default:
				icon_pixbuf = get_icon_for_name("folder", gui_prefs.ListFontHeight, zip, folder_xpm);
				break;
			}
			break;
	}

	if (zip)
		closezip(zip);

	return icon_pixbuf;
}




/*this function check if an icon update is needed and get the icon from file or inlined xpm*/
void get_status_icons()
{
	ZIP *zip;
	gchar *zipfile;

	/* allocate and open zip file if needed, ie dirty cache or null icon */
	if( dirty_icon_cache ||
		!Status_Icons[INCORRECT] ||
		!Status_Icons[CORRECT] ||
		!Status_Icons[UNKNOWN] ||
		!Status_Icons[PROBLEMS])
	{
		if(dirty_icon_cache)
		{
			if(Status_Icons[CORRECT]!=NULL)
			{
					g_object_unref(Status_Icons[CORRECT]);
					Status_Icons[CORRECT] = NULL;
			}
			if(Status_Icons[INCORRECT]!=NULL)
			{
					g_object_unref(Status_Icons[INCORRECT]);
					Status_Icons[INCORRECT] = NULL;
			}
			if(Status_Icons[PROBLEMS]!=NULL)
			{
					g_object_unref(Status_Icons[PROBLEMS]);
					Status_Icons[PROBLEMS] = NULL;
			}
			if(Status_Icons[UNKNOWN]!=NULL)
			{
					g_object_unref(Status_Icons[UNKNOWN]);
					Status_Icons[UNKNOWN] = NULL;
			}
		}

		zipfile = g_build_filename(gui_prefs.IconDirectory, "icons.zip", NULL);
		zip = openzip(zipfile);
		g_free(zipfile);

	 	if(Status_Icons[CORRECT]==NULL)
			Status_Icons[CORRECT] = get_icon_for_name("roms", gui_prefs.ListFontHeight, zip, romL_xpm);
	 	if(Status_Icons[INCORRECT]==NULL)
			Status_Icons[INCORRECT] = get_icon_for_name("noroms", gui_prefs.ListFontHeight, zip, noromsL_xpm);
	 	if(Status_Icons[PROBLEMS]==NULL)
			Status_Icons[PROBLEMS] = get_icon_for_name("warning", gui_prefs.ListFontHeight, zip, redxL_xpm);
	 	if(Status_Icons[UNKNOWN]==NULL)
			Status_Icons[UNKNOWN] = get_icon_for_name("unknown", gui_prefs.ListFontHeight, zip, unknownL_xpm);

		if (zip)
			closezip(zip);
	}

}


void create_gamelist_content (void)
{
	GList *listpointer;
	RomEntry *tmprom;
	gchar *my_romname_root = NULL;
	gchar *my_hassamples;
	gchar *my_control;
	GdkColor *my_txtcolor;
	GtkTreeSelection *select;
	GtkTreeIter iter;
	GtkTreeIter iter_root;
	GtkTreeIter iter_child;
	GtkTreeModel *store;
	gboolean tree_store;   /* If the model is a tree or a list */
	gboolean is_root;
	gint i=0;
	gint j=0;
	gboolean valid;
	gchar *message;

	GXMAME_DEBUG ("POPULATE GAME LIST");

	/* Status Bar Message */
	gtk_statusbar_pop (main_gui.statusbar3, 1);
	gtk_statusbar_push (main_gui.statusbar3, 1, _("Wait..."));
	if ((main_gui.displayed_list!=NULL) && (main_gui.tree_model!=NULL))
	{
		store = NULL;
		gtk_tree_view_set_model (GTK_TREE_VIEW(main_gui.displayed_list), GTK_TREE_MODEL(store));
		/* Update UI */
		/*while (gtk_events_pending ())
			gtk_main_iteration ();*/
	}

	/* Whether the Tree Model will a tree or a list */
	if ((gui_prefs.current_mode==LIST_TREE) || (gui_prefs.current_mode==DETAILS_TREE))
		tree_store=TRUE;
	else
		tree_store=FALSE;

	/* Get the status icon */
	get_status_icons();

	/* Create a model. */
	if (tree_store)
		store = (GtkTreeModel *) gtk_tree_store_new (NUMBER_COLUMN+NUMBER_COLUMN_HIDDEN,
				G_TYPE_STRING,
				G_TYPE_STRING,
				G_TYPE_STRING,
				G_TYPE_STRING,
				G_TYPE_STRING,
				G_TYPE_STRING,
				G_TYPE_INT,
				G_TYPE_STRING,
				G_TYPE_STRING,
				G_TYPE_STRING,
				G_TYPE_STRING,
				G_TYPE_STRING,
				G_TYPE_STRING,
				G_TYPE_STRING,
				G_TYPE_INT,
				G_TYPE_INT,
				G_TYPE_STRING,
				G_TYPE_STRING,
				G_TYPE_STRING,
				G_TYPE_STRING,
				G_TYPE_STRING,
				G_TYPE_STRING,
				G_TYPE_STRING,
				G_TYPE_STRING,
				G_TYPE_STRING,
				G_TYPE_STRING,
				G_TYPE_STRING,
				G_TYPE_INT,
				G_TYPE_POINTER,     /* Rom Entry */
				GDK_TYPE_COLOR,     /* Text Color */
				GDK_TYPE_PIXBUF);   /* Pixbuf */
	else
		store = (GtkTreeModel *) gtk_list_store_new (NUMBER_COLUMN+NUMBER_COLUMN_HIDDEN,
				G_TYPE_STRING,
				G_TYPE_STRING,
				G_TYPE_STRING,
				G_TYPE_STRING,
				G_TYPE_STRING,
				G_TYPE_STRING,
				G_TYPE_INT,
				G_TYPE_STRING,
				G_TYPE_STRING,
				G_TYPE_STRING,
				G_TYPE_STRING,
				G_TYPE_STRING,
				G_TYPE_STRING,
				G_TYPE_STRING,
				G_TYPE_INT,
				G_TYPE_INT,
				G_TYPE_STRING,
				G_TYPE_STRING,
				G_TYPE_STRING,
				G_TYPE_STRING,
				G_TYPE_STRING,
				G_TYPE_STRING,
				G_TYPE_STRING,
				G_TYPE_STRING,
				G_TYPE_STRING,
				G_TYPE_STRING,
				G_TYPE_STRING,
				G_TYPE_INT,
				G_TYPE_POINTER,     /* Rom Entry */
				GDK_TYPE_COLOR,     /* Text Color */
				GDK_TYPE_PIXBUF);   /* Pixbuf */

	/* fill the model with data */
	for(listpointer = g_list_first(roms);
			(listpointer != NULL);
			 listpointer= g_list_next(listpointer))
	{
		tmprom = (RomEntry *) listpointer->data;
		if (game_filtered(tmprom))
		{
			GdkPixbuf *pixbuf = NULL;

			/* Game Name */
			if (tmprom->name_in_list)
				g_free(tmprom->name_in_list);

			if (tmprom->the_trailer && gui_prefs.ModifyThe)
				tmprom->name_in_list = g_strstrip(g_strdup_printf("%s, The %s",tmprom->gamename,tmprom->gamenameext));
			else if (tmprom->the_trailer && !gui_prefs.ModifyThe)
				tmprom->name_in_list = g_strstrip(g_strdup_printf("The %s %s",tmprom->gamename,tmprom->gamenameext));
			else
				tmprom->name_in_list = g_strstrip(g_strdup_printf("%s %s",tmprom->gamename,tmprom->gamenameext));
			/* Has Samples */
			if (tmprom->nb_samples==0)
				my_hassamples = NULL;
			else
				my_hassamples = tmprom->has_samples?_("Yes"):_("No");

			/* Control */
			if (!strcmp(tmprom->control,"trackball"))
				my_control = _("Yes");
			else
				my_control = _("No");
			
			/* Clone Color + Pixbuf width */
			if (strcmp(tmprom->cloneof,"-"))
			{  /* Clone */
				my_txtcolor = &gui_prefs.clone_color;
			}else{ /* Original */
				my_txtcolor = NULL; /* Black */
			}
			/* Pixbuf */
			if (tmprom->has_roms == 2)
			{
				pixbuf = Status_Icons[UNKNOWN];
			}
			else if (tmprom->has_roms == 0)
			{
				pixbuf = Status_Icons[INCORRECT];
			}
			else if (!tmprom->status)
			{
				pixbuf = Status_Icons[PROBLEMS];
			}
			else
			{
				if (tmprom->icon_pixbuf != NULL)
				{
					pixbuf = tmprom->icon_pixbuf;
				}else
				{
					pixbuf = Status_Icons[CORRECT];
				}
			}

			/* Determine if the row is a root */
			if ( (j==0) || !(strcmp(tmprom->cloneof,"-")) || !my_romname_root || (strcmp(tmprom->cloneof,my_romname_root)) )
			{
				is_root=TRUE;
			}else
			{
				is_root=FALSE;
			}

			/* Memorize the original names */
			if (!(strcmp(tmprom->cloneof,"-")))
			{
				if(my_romname_root)
					g_free(my_romname_root);

				my_romname_root= g_strdup(tmprom->romname);
			}
			if (tree_store)
			{
				if (is_root)
					gtk_tree_store_append (GTK_TREE_STORE(store), &iter, NULL);  /* Acquire an iterator */
				else
					gtk_tree_store_append (GTK_TREE_STORE(store), &iter, &iter_root);  /* Acquire an iterator */

				gtk_tree_store_set (GTK_TREE_STORE(store), &iter,
					GAMENAME,                   tmprom->name_in_list,
					HAS_ROMS,                   tmprom->has_roms?_("Yes"):_("No"),
					HAS_SAMPLES,                my_hassamples,
					ROMNAME,                    tmprom->romname,
					VECTOR,                     tmprom->vector?_("Vector"):_("Raster"),
					CONTROL,                    my_control,
					TIMESPLAYED,                tmprom->timesplayed,
					MANU,                       tmprom->manu,
					YEAR,                       tmprom->year,
					CLONE,                      tmprom->cloneof,
					DRIVER,                     tmprom->driver,
					STATUS,                     tmprom->has_roms?_("Available"):_("Not Available"),
					ROMOF,                      tmprom->romof,
					DRIVERSTATUS,               tmprom->status?_("Working"):_("Not Working"),
					NUMPLAYERS,                 tmprom->num_players,
					NUMBUTTONS,                 tmprom->num_buttons,
					CPU1,                       tmprom->cpu_info[0].name,
					CPU2,                       tmprom->cpu_info[1].name,
					CPU3,                       tmprom->cpu_info[2].name,
					CPU4,                       tmprom->cpu_info[3].name,
					SOUND1,                     tmprom->sound_info[0].name,
					SOUND2,                     tmprom->sound_info[1].name,
					SOUND3,                     tmprom->sound_info[2].name,
					SOUND4,                     tmprom->sound_info[3].name,
					MAMEVER,                    tmprom->mame_ver_added,
					CATEGORY,                   tmprom->category,
					FAVORITE,                   tmprom->favourite?_("Yes"):_("No"),
					CHANNELS,                   tmprom->channels,
					NUMBER_COLUMN+ROMENTRY,     tmprom,                 /* rom entry */
					NUMBER_COLUMN+TEXTCOLOR,    my_txtcolor,            /* text color */
					NUMBER_COLUMN+PIXBUF,       pixbuf,                 /* pixbuf */
					-1);
				if (is_root)
					iter_root=iter;
			}else
			{
				gtk_list_store_append (GTK_LIST_STORE(store), &iter);  /* Acquire an iterator */
				gtk_list_store_set (GTK_LIST_STORE(store), &iter,
					GAMENAME,                   tmprom->name_in_list,
					HAS_ROMS,                   tmprom->has_roms?_("Yes"):_("No"),
					HAS_SAMPLES,                my_hassamples,
					ROMNAME,                    tmprom->romname,
					VECTOR,                     tmprom->vector?_("Vector"):_("Raster"),
					CONTROL,                    my_control,
					TIMESPLAYED,                tmprom->timesplayed,
					MANU,                       tmprom->manu,
					YEAR,                       tmprom->year,
					CLONE,                      tmprom->cloneof,
					DRIVER,                     tmprom->driver,
					STATUS,                     tmprom->has_roms?_("Available"):_("Not Available"),
					ROMOF,                      tmprom->romof,
					DRIVERSTATUS,               tmprom->status?_("Working"):_("Not Working"),
					NUMPLAYERS,                 tmprom->num_players,
					NUMBUTTONS,                 tmprom->num_buttons,
					CPU1,                       tmprom->cpu_info[0].name,
					CPU2,                       tmprom->cpu_info[1].name,
					CPU3,                       tmprom->cpu_info[2].name,
					CPU4,                       tmprom->cpu_info[3].name,
					SOUND1,                     tmprom->sound_info[0].name,
					SOUND2,                     tmprom->sound_info[1].name,
					SOUND3,                     tmprom->sound_info[2].name,
					SOUND4,                     tmprom->sound_info[3].name,
					MAMEVER,                    tmprom->mame_ver_added,
					CATEGORY,                   tmprom->category,
					FAVORITE,                   tmprom->favourite?_("Yes"):_("No"),
					CHANNELS,                   tmprom->channels,
					NUMBER_COLUMN+ROMENTRY,     tmprom,                 /* rom entry */
					NUMBER_COLUMN+TEXTCOLOR,    my_txtcolor,            /* text color */
					NUMBER_COLUMN+PIXBUF,       pixbuf,                 /* pixbuf */
					-1);
			}
			tmprom->position=iter;
			tmprom->is_in_list=TRUE;
			j++;
		} else
		{
			tmprom->is_in_list=FALSE;
		}
	}
	visible_games=j;

	/* Callbacks - Sorting order has changed */
	if (main_gui.tree_model==NULL)
	{
		main_gui.tree_model = GTK_TREE_MODEL(store);
		g_signal_connect (G_OBJECT (main_gui.tree_model), "sort-column-changed",
				G_CALLBACK (on_displayed_list_sort_column_changed),
				NULL);
	}else{
		main_gui.tree_model = GTK_TREE_MODEL(store);
	}

	/* Update the corresponding tree view */
	if (main_gui.displayed_list!=NULL)
	{
		/* Link the view with the model */
		gtk_tree_view_set_model (GTK_TREE_VIEW(main_gui.displayed_list), GTK_TREE_MODEL(main_gui.tree_model));
		/* Get the selection */
		select = gtk_tree_view_get_selection (GTK_TREE_VIEW (main_gui.displayed_list));
		/* Sort the list */
		if ((gui_prefs.current_mode==DETAILS) || (gui_prefs.current_mode==DETAILS_TREE))
		{
			if (gui_prefs.SortReverse)
				gtk_tree_sortable_set_sort_column_id (GTK_TREE_SORTABLE (main_gui.tree_model), gui_prefs.SortColumn, GTK_SORT_DESCENDING);
			else
				gtk_tree_sortable_set_sort_column_id (GTK_TREE_SORTABLE (main_gui.tree_model), gui_prefs.SortColumn, GTK_SORT_ASCENDING);
		}else
		{
			g_signal_handlers_block_by_func (G_OBJECT (main_gui.tree_model), (gpointer)on_displayed_list_sort_column_changed, NULL);
			/* FIXME we sometimes have here a gtk warning why????? */
			gtk_tree_sortable_set_sort_column_id (GTK_TREE_SORTABLE (main_gui.tree_model), GAMENAME, GTK_SORT_ASCENDING);
			g_signal_handlers_unblock_by_func (G_OBJECT (main_gui.tree_model), (gpointer)on_displayed_list_sort_column_changed, NULL);
		}		/* Select the correct row */
		if (visible_games>0)
		{
			gchar *str_data;
			valid = gtk_tree_model_get_iter_first (GTK_TREE_MODEL(main_gui.tree_model), &iter);
			gtk_tree_model_get (GTK_TREE_MODEL(main_gui.tree_model), &iter, GAMENAME, &str_data, -1);
			is_root=TRUE;
			i=0;
			while ((i<visible_games) && (strncmp(str_data,gui_prefs.DefaultGame,MAX_GAMENAME)!=0) && (valid))
			{
				if (gtk_tree_model_iter_has_child (GTK_TREE_MODEL(main_gui.tree_model),&iter))
				{
					if (gtk_tree_model_iter_children (GTK_TREE_MODEL(main_gui.tree_model), &iter_child, &iter))
					{
						gtk_tree_model_get (GTK_TREE_MODEL(main_gui.tree_model), &iter_child, GAMENAME, &str_data, -1);
						is_root=FALSE;
						i++;
						while ((i<visible_games)  && (strncmp(str_data,gui_prefs.DefaultGame,MAX_GAMENAME)!=0) && (gtk_tree_model_iter_next (GTK_TREE_MODEL(main_gui.tree_model), &iter_child)) )
						{
							gtk_tree_model_get (GTK_TREE_MODEL(main_gui.tree_model), &iter_child, GAMENAME, &str_data, -1);
							i++;
						}
					}
				}
				if ((i<visible_games)  && (strncmp(str_data,gui_prefs.DefaultGame,MAX_GAMENAME)!=0))
				{
					valid=gtk_tree_model_iter_next (GTK_TREE_MODEL(main_gui.tree_model), &iter);
					if (valid)
					{
						gtk_tree_model_get (GTK_TREE_MODEL(main_gui.tree_model), &iter, GAMENAME, &str_data, -1);
						is_root=TRUE;
						i++;
					}
				}
			}
			select = gtk_tree_view_get_selection (GTK_TREE_VIEW (main_gui.displayed_list));
			if (strncmp(str_data,gui_prefs.DefaultGame,MAX_GAMENAME)!=0)
			{
				valid = gtk_tree_model_get_iter_first (GTK_TREE_MODEL(main_gui.tree_model), &iter);
				is_root=TRUE;
			}
			if (is_root)
			{
				GtkTreePath *tree_path = gtk_tree_model_get_path (GTK_TREE_MODEL(main_gui.tree_model), &iter);
				gtk_tree_view_set_cursor (GTK_TREE_VIEW (main_gui.displayed_list),
						tree_path,
						NULL, FALSE);
				/* Scroll to selection */
				gtk_tree_view_scroll_to_cell (GTK_TREE_VIEW(main_gui.displayed_list), tree_path, NULL, TRUE, 0.5, 0);
				gtk_tree_path_free(tree_path);
			}else
			{
				GtkTreePath *tree_path = gtk_tree_model_get_path (GTK_TREE_MODEL(main_gui.tree_model), &iter);
				valid=gtk_tree_view_expand_row (GTK_TREE_VIEW (main_gui.displayed_list),
						tree_path,
						TRUE);
				gtk_tree_path_free(tree_path);
				tree_path = gtk_tree_model_get_path (GTK_TREE_MODEL(main_gui.tree_model), &iter_child);
				gtk_tree_view_set_cursor (GTK_TREE_VIEW (main_gui.displayed_list),
						tree_path,
						NULL, FALSE);
				/* Scroll to selection */
				gtk_tree_view_scroll_to_cell (GTK_TREE_VIEW(main_gui.displayed_list), tree_path, NULL, TRUE, 0.5, 0);
				gtk_tree_path_free(tree_path);
			}
			g_free (str_data);
		}
		/* Header clickable. */
		if ((gui_prefs.current_mode==DETAILS) || (gui_prefs.current_mode==DETAILS_TREE))
			gtk_tree_view_set_headers_clickable
						(GTK_TREE_VIEW(main_gui.displayed_list),
						TRUE);
		else
			gtk_tree_view_set_headers_clickable
						(GTK_TREE_VIEW(main_gui.displayed_list),
						FALSE);
	}

	/* Status Bar Message */
	message = g_strdup_printf("%d %s",visible_games,visible_games>1?_("games"):_("game"));
	gtk_statusbar_pop (main_gui.statusbar3, 1);
	gtk_statusbar_push (main_gui.statusbar3, 1, message);

	/* Free Memory */
	if (message)
		g_free(message);
	if (my_romname_root)
		g_free(my_romname_root);

	if (visible_games==0)
		select_game(NULL);
}

void create_gamelist (ListMode list_mode)
{
	gint i, j, k;
	GtkTreeViewColumn *column;
	GtkTreeViewColumn *base_column;
	GtkCellRenderer *renderer;
	GtkTreeSelection *select;
	GList *col_list=NULL;
	GList *listpointer=NULL;
	static gboolean first_run = TRUE;

	GXMAME_DEBUG ("DISPLAY GAME LIST");

	/* We Create the TreeView only if it is NULL (this will occur only once) */
	if (main_gui.displayed_list==NULL)
	{
		main_gui.displayed_list = gtk_tree_view_new ();
		for(i=0;i<NUMBER_COLUMN;i++)
		{
			if (i==GAMENAME)
			{
				column = gtk_tree_view_column_new ();
				gtk_tree_view_column_set_title  (GTK_TREE_VIEW_COLUMN(column), column_title(i));
				gtk_tree_view_column_set_sort_column_id (column, i);
				/* TEXT */
				renderer = gtk_cell_renderer_text_new ();
				gtk_tree_view_column_pack_end (GTK_TREE_VIEW_COLUMN(column),
                                             renderer,
                                             FALSE);
				gtk_tree_view_column_set_attributes (GTK_TREE_VIEW_COLUMN(column), renderer,
						"text", i,
						"foreground-gdk", NUMBER_COLUMN+TEXTCOLOR,
						NULL);
				g_object_set(renderer, "ypad", 0, "yalign", 0.5, NULL);
				/* Pixbuf */
				renderer = gtk_cell_renderer_pixbuf_new ();
				gtk_tree_view_column_pack_end (GTK_TREE_VIEW_COLUMN(column),
                                             renderer,
                                             FALSE);
				gtk_tree_view_column_add_attribute (GTK_TREE_VIEW_COLUMN(column), renderer,
						"pixbuf", NUMBER_COLUMN+PIXBUF);
				g_object_set(renderer, "xalign", 1.0, "ypad", 0, NULL);
			} else
			{
				renderer = gtk_cell_renderer_text_new ();
				column = gtk_tree_view_column_new_with_attributes (column_title(i), renderer,
									   "text", i,
									   "foreground-gdk", NUMBER_COLUMN+TEXTCOLOR,
									   NULL);
				gtk_tree_view_column_set_sort_column_id (column, i);
			}
			gtk_tree_view_append_column (GTK_TREE_VIEW (main_gui.displayed_list), column);
			gtk_tree_view_column_set_min_width (GTK_TREE_VIEW_COLUMN(column), 1);
			g_signal_connect (column->button, "event",
					G_CALLBACK (on_column_click),
					column);
		}
		gtk_container_add (GTK_CONTAINER(main_gui.scrolled_window_games), main_gui.displayed_list);
		gtk_widget_show_all (main_gui.scrolled_window_games);

		/* Callback - Row has been selected */
		select = gtk_tree_view_get_selection (GTK_TREE_VIEW (main_gui.displayed_list));
		gtk_tree_selection_set_mode (select, GTK_SELECTION_SINGLE);
		g_signal_connect (G_OBJECT (select), "changed",
				G_CALLBACK (on_row_selected),
				NULL);
		/* Callback - Column Order has changed */
		g_signal_connect (G_OBJECT (main_gui.displayed_list), "columns-changed",
				G_CALLBACK (on_columns_changed),
				NULL);
		/* Callback - Click on the list */
		g_signal_connect (G_OBJECT (main_gui.displayed_list), "button-press-event",
				G_CALLBACK (on_list_clicked),
				NULL);
		/* Callback - Column size modified */
		g_signal_connect (G_OBJECT (main_gui.displayed_list), "size-request",
				G_CALLBACK (on_displayed_list_resize_column),
				NULL);
		/* Callback - Row has been collapsed */
		g_signal_connect (G_OBJECT (main_gui.displayed_list), "row-collapsed",
				G_CALLBACK (on_displayed_list_row_collapsed),
				NULL);
	}


	/* We sort the list */
	if (main_gui.tree_model!=NULL)
	{
		if ((gui_prefs.current_mode==DETAILS) || (gui_prefs.current_mode==DETAILS_TREE))
		{
			if (gui_prefs.SortReverse)
				gtk_tree_sortable_set_sort_column_id (GTK_TREE_SORTABLE (main_gui.tree_model), gui_prefs.SortColumn, GTK_SORT_DESCENDING);
			else
				gtk_tree_sortable_set_sort_column_id (GTK_TREE_SORTABLE (main_gui.tree_model), gui_prefs.SortColumn, GTK_SORT_ASCENDING);
		}else
		{
			g_signal_handlers_block_by_func (G_OBJECT (main_gui.tree_model), (gpointer)on_displayed_list_sort_column_changed, NULL);
			gtk_tree_sortable_set_sort_column_id (GTK_TREE_SORTABLE (main_gui.tree_model), GAMENAME, GTK_SORT_ASCENDING);
			g_signal_handlers_unblock_by_func (G_OBJECT (main_gui.tree_model), (gpointer)on_displayed_list_sort_column_changed, NULL);
		}
	}

	/* Header clickable Tree Model must exist. */
	if (main_gui.tree_model!=NULL)
	{
		if ((list_mode==DETAILS) || (list_mode==DETAILS_TREE))
			gtk_tree_view_set_headers_clickable
						(GTK_TREE_VIEW(main_gui.displayed_list),
						TRUE);
		else
			gtk_tree_view_set_headers_clickable
						(GTK_TREE_VIEW(main_gui.displayed_list),
						FALSE);
	}

	/* Show or hide Header */
	if ((list_mode==DETAILS) || (list_mode==DETAILS_TREE))
		gtk_tree_view_set_headers_visible (GTK_TREE_VIEW(main_gui.displayed_list), TRUE);
	else
		gtk_tree_view_set_headers_visible (GTK_TREE_VIEW(main_gui.displayed_list), FALSE);

	/* Reorder the columns */
	base_column=NULL;
	g_signal_handlers_block_by_func (G_OBJECT (main_gui.displayed_list), (gpointer)on_columns_changed, NULL);
	
	col_list = gtk_tree_view_get_columns (GTK_TREE_VIEW(main_gui.displayed_list));
	
	for (i=0;i<NUMBER_COLUMN;i++)
	{
		for (j=0;gui_prefs.ColumnOrder[j]!=i;j++)
		{}
		for (listpointer = g_list_first(col_list),k=0;
				( (listpointer!=NULL) && (k<NUMBER_COLUMN) && (j!=gtk_tree_view_column_get_sort_column_id(listpointer->data)) );
				listpointer = g_list_next(listpointer),k++)
		{}
		column = listpointer->data;
		gtk_tree_view_move_column_after (GTK_TREE_VIEW(main_gui.displayed_list),
							GTK_TREE_VIEW_COLUMN(column),
							GTK_TREE_VIEW_COLUMN(base_column));
		base_column=column;
	}
	g_signal_handlers_unblock_by_func (G_OBJECT (main_gui.displayed_list), (gpointer)on_columns_changed, NULL);
	g_list_free(col_list);
	/* Update the columns */
	/* FIXME When switching from LIST mode to DETAILS, it puts a mess in the size of the
	GAMENAME column even if I block the callback?????? */
	g_signal_handlers_block_by_func (G_OBJECT (main_gui.displayed_list), (gpointer)on_displayed_list_resize_column, NULL);
	for(i=0;i<NUMBER_COLUMN;i++)
	{
		column = gtk_tree_view_get_column (GTK_TREE_VIEW(main_gui.displayed_list), gui_prefs.ColumnOrder[i]);

		/* Font */
		j=0;
		col_list = gtk_tree_view_column_get_cell_renderers (GTK_TREE_VIEW_COLUMN(column));
		for (listpointer = g_list_first(col_list);(listpointer != NULL);listpointer = g_list_next(listpointer))
		{
			if (j==0)
			{		/* Text */
				g_object_set (G_OBJECT(listpointer->data), "font", gui_prefs.ListFont, NULL);
				j=1;
			}
		}
		g_list_free(col_list);
		/* Columns visible, Column size,... */
		if ((list_mode==DETAILS) || (list_mode==DETAILS_TREE))
		{	/* COLUMNS */
			if (gui_prefs.ColumnShown[i]==FALSE)
			{
				gtk_tree_view_column_set_visible (column, FALSE);
			}else
			{
				gtk_tree_view_column_set_visible (column, TRUE);
				if (gui_prefs.ColumnWidth[i]==0)
				{
					gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_AUTOSIZE);
				}else{
					gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_FIXED);
					gtk_tree_view_column_set_fixed_width (column, gui_prefs.ColumnWidth[i]);
				}
				gtk_tree_view_column_set_resizable (column, TRUE);
				gtk_tree_view_column_set_reorderable (column, TRUE);
			}
		}else
		{	/* NO COLUMNS */
			if (i==GAMENAME)
			{
				gtk_tree_view_column_set_visible (column, TRUE);
				gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_AUTOSIZE);
				gtk_tree_view_column_set_resizable (column, FALSE);
				gtk_tree_view_column_set_reorderable (column, FALSE);
			}else
			{
				gtk_tree_view_column_set_visible (column, FALSE);
			}
		}
		/* Reordable is disable for the time beeing because it make conflics with
		  'column popup menu' and 'sort on click header' */
		gtk_tree_view_column_set_reorderable (GTK_TREE_VIEW_COLUMN(column), FALSE);
	}
	g_signal_handlers_unblock_by_func (G_OBJECT (main_gui.displayed_list), (gpointer)on_displayed_list_resize_column, NULL);

	first_run = FALSE;
	dirty_icon_cache = FALSE;

}

void select_game(RomEntry *rom)
{

	if (rom!=NULL)
	{
		/* Maybe I should remove the string default gamename and only keep a pointer to the rom struct ? */
		gui_prefs.DefaultGame_p = rom;
		
		strncpy(gui_prefs.DefaultGame,rom->name_in_list, MAX_GAMENAME);
		g_strstrip(gui_prefs.DefaultGame);

		/* update menus */
		gtk_widget_set_sensitive(GTK_WIDGET(main_gui.play_menu),TRUE);
		gtk_widget_set_sensitive(GTK_WIDGET(main_gui.properties_menu),TRUE);
		gtk_widget_set_sensitive(GTK_WIDGET(main_gui.add_to_favorites),!rom->favourite);
		gtk_widget_set_sensitive(GTK_WIDGET(main_gui.remove_from_favorites),rom->favourite);
		gtk_widget_set_sensitive(GTK_WIDGET(main_gui.popup_gamelist_add_to_favorites),!rom->favourite);
		gtk_widget_set_sensitive(GTK_WIDGET(main_gui.popup_gamelist_remove_from_favorites),rom->favourite);

		/* update statusbar */
		gtk_statusbar_pop (main_gui.statusbar1, 1);
		gtk_statusbar_push (main_gui.statusbar1, 1, gui_prefs.DefaultGame);
		gtk_statusbar_pop (main_gui.statusbar2, 1);
		if (rom->has_roms == 2)
			gtk_statusbar_push (main_gui.statusbar2, 1, _("Unknown"));
		else if (rom->has_roms == 0)
			gtk_statusbar_push (main_gui.statusbar2, 1, _("ROMs missing"));
		else if (!rom->status)
			gtk_statusbar_push (main_gui.statusbar2, 1, _("Not working"));
		else
			gtk_statusbar_push (main_gui.statusbar2, 1, _("Working"));

		/* update screenshot panel */
		update_screenshot_panel(rom);
	}
	else /* no roms selected display the default picture */
	{
		GXMAME_DEBUG("no games selected");
		/* update menus */
		gtk_widget_set_sensitive(GTK_WIDGET(main_gui.play_menu),FALSE);
		gtk_widget_set_sensitive(GTK_WIDGET(main_gui.properties_menu),FALSE);
		gtk_widget_set_sensitive(GTK_WIDGET(main_gui.add_to_favorites),FALSE);
		gtk_widget_set_sensitive(GTK_WIDGET(main_gui.remove_from_favorites),FALSE);

		/* update statusbar */
		gtk_statusbar_pop (main_gui.statusbar1, 1);
		gtk_statusbar_push (main_gui.statusbar1, 1, _("No game selected"));
		gtk_statusbar_pop (main_gui.statusbar2, 1);
		gtk_statusbar_push (main_gui.statusbar2, 1, " ");

		/* update screenshot panel */
		update_screenshot_panel(NULL);

	}

}


void show_progress_bar(void)
{
	if(gui_prefs.ShowStatusBar)
	{
		gtk_widget_hide (GTK_WIDGET(main_gui.tri_status_bar));
		gtk_statusbar_push(main_gui.status_progress_bar, 1, _("Game search 0% complete"));
		gtk_widget_show (GTK_WIDGET(main_gui.combo_progress_bar));
	}	
}


void hide_progress_bar(void)
{
	if(gui_prefs.ShowStatusBar)
	{
		gtk_widget_hide (GTK_WIDGET(main_gui.combo_progress_bar));
		gtk_statusbar_pop(main_gui.status_progress_bar, 1);
		gtk_widget_show (GTK_WIDGET(main_gui.tri_status_bar));
	}
}

void update_progress_bar(gfloat current_value)
{
	static gint current_displayed_value;
	gchar *displayed_message;
	if (current_displayed_value!=(gint)(current_value*100))
	{
		current_displayed_value=(gint)(current_value*100);
		displayed_message = g_strdup_printf(_("Game search %i%% complete"),current_displayed_value);
		gtk_statusbar_pop(main_gui.status_progress_bar, 1);
		gtk_statusbar_push(main_gui.status_progress_bar, 1, displayed_message);
		g_free(displayed_message);
	}
	gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR(main_gui.progress_progress_bar),current_value);
}

void update_game_in_list (GtkTreeIter position)
{
	gchar *my_hassamples;
	gchar *my_control;
	GdkColor *my_txtcolor;
	RomEntry *tmprom;
	GdkPixbuf *pixbuf;
	gboolean is_tree_store;
	gint pixbuf_width=0;

	/* Whether the Tree Model will a tree or a list */
	if ((gui_prefs.current_mode==LIST_TREE) || (gui_prefs.current_mode==DETAILS_TREE))
		is_tree_store=TRUE;
	else
		is_tree_store=FALSE;

	gtk_tree_model_get (GTK_TREE_MODEL(main_gui.tree_model),
						&position,
						NUMBER_COLUMN+ROMENTRY, &tmprom,
						-1);

	/* Game Name */
	if (tmprom->name_in_list)
		g_free(tmprom->name_in_list);	

	if (tmprom->the_trailer && gui_prefs.ModifyThe)
		tmprom->name_in_list = g_strstrip(g_strdup_printf("%s, The %s",tmprom->gamename,tmprom->gamenameext));
	else if (tmprom->the_trailer && !gui_prefs.ModifyThe)
		tmprom->name_in_list = g_strstrip(g_strdup_printf("The %s %s",tmprom->gamename,tmprom->gamenameext));
	else
		tmprom->name_in_list = g_strstrip(g_strdup_printf("%s %s",tmprom->gamename,tmprom->gamenameext));

	/* Has Samples */
	if (tmprom->nb_samples==0)
		my_hassamples = NULL;
	else
	{
		my_hassamples = tmprom->has_samples?_("Yes"):_("No");
	}
	
	/* Control */
	if (!strcmp(tmprom->control,"trackball"))
		my_control = _("Yes");
	else
		my_control = _("No");
	
	/* Clone Color + Pixbuf width */
	if (strcmp(tmprom->cloneof,"-"))
	{  /* Clone */
		pixbuf_width = 50;
		my_txtcolor = &gui_prefs.clone_color;
	}else{ /* Original */
		pixbuf_width = 20;
		my_txtcolor = NULL; /* Black */
	}
	/* Pixbuf */
	if (tmprom->has_roms == 2)
	{
		pixbuf = Status_Icons[UNKNOWN];
	}
	else if (tmprom->has_roms == 0)
	{
		pixbuf = Status_Icons[INCORRECT];
	}
	else if (!tmprom->status)
	{
		pixbuf = Status_Icons[PROBLEMS];
	}
	else
	{
		if (tmprom->icon_pixbuf != NULL)
		{
			pixbuf = tmprom->icon_pixbuf;
		}else
		{
			pixbuf = Status_Icons[CORRECT];
		}
	}

	if (is_tree_store)
	{
		gtk_tree_store_set (GTK_TREE_STORE(main_gui.tree_model), &position,
			GAMENAME,                   tmprom->name_in_list,
			HAS_ROMS,                   tmprom->has_roms?_("Yes"):_("No"),
			HAS_SAMPLES,                my_hassamples,
			ROMNAME,                    tmprom->romname,
			VECTOR,                     tmprom->vector?_("Vector"):_("Raster"),
			CONTROL,                    my_control,
			TIMESPLAYED,                tmprom->timesplayed,
			MANU,                       tmprom->manu,
			YEAR,                       tmprom->year,
			CLONE,                      tmprom->cloneof,
			DRIVER,                     tmprom->driver,
			STATUS,                     tmprom->has_roms?_("Available"):_("Not Available"),
			ROMOF,                      tmprom->romof,
			DRIVERSTATUS,               tmprom->status?_("Working"):_("Not Working"),
			NUMPLAYERS,                 tmprom->num_players,
			NUMBUTTONS,                 tmprom->num_buttons,
			CPU1,                       tmprom->cpu_info[0].name,
			CPU2,                       tmprom->cpu_info[1].name,
			CPU3,                       tmprom->cpu_info[2].name,
			CPU4,                       tmprom->cpu_info[3].name,
			SOUND1,                     tmprom->sound_info[0].name,
			SOUND2,                     tmprom->sound_info[1].name,
			SOUND3,                     tmprom->sound_info[2].name,
			SOUND4,                     tmprom->sound_info[3].name,
			MAMEVER,                    tmprom->mame_ver_added,
			CATEGORY,                   tmprom->category,
			FAVORITE,                   tmprom->favourite?_("Yes"):_("No"),
			CHANNELS,                   tmprom->channels,
			NUMBER_COLUMN+TEXTCOLOR,    my_txtcolor,            /* text color */
			NUMBER_COLUMN+PIXBUF,       pixbuf,                 /* pixbuf */
			-1);
	}else
	{
		gtk_list_store_set (GTK_LIST_STORE(main_gui.tree_model), &position,
			GAMENAME,                   tmprom->name_in_list,
			HAS_ROMS,                   tmprom->has_roms?_("Yes"):_("No"),
			HAS_SAMPLES,                my_hassamples,
			ROMNAME,                    tmprom->romname,
			VECTOR,                     tmprom->vector?_("Vector"):_("Raster"),
			CONTROL,                    my_control,
			TIMESPLAYED,                tmprom->timesplayed,
			MANU,                       tmprom->manu,
			YEAR,                       tmprom->year,
			CLONE,                      tmprom->cloneof,
			DRIVER,                     tmprom->driver,
			STATUS,                     tmprom->has_roms?_("Available"):_("Not Available"),
			ROMOF,                      tmprom->romof,
			DRIVERSTATUS,               tmprom->status?_("Working"):_("Not Working"),
			NUMPLAYERS,                 tmprom->num_players,
			NUMBUTTONS,                 tmprom->num_buttons,
			CPU1,                       tmprom->cpu_info[0].name,
			CPU2,                       tmprom->cpu_info[1].name,
			CPU3,                       tmprom->cpu_info[2].name,
			CPU4,                       tmprom->cpu_info[3].name,
			SOUND1,                     tmprom->sound_info[0].name,
			SOUND2,                     tmprom->sound_info[1].name,
			SOUND3,                     tmprom->sound_info[2].name,
			SOUND4,                     tmprom->sound_info[3].name,
			MAMEVER,                    tmprom->mame_ver_added,
			CATEGORY,                   tmprom->category,
			FAVORITE,                   tmprom->favourite?_("Yes"):_("No"),
			CHANNELS,                   tmprom->channels,
			NUMBER_COLUMN+TEXTCOLOR,    my_txtcolor,            /* text color */
			NUMBER_COLUMN+PIXBUF,       pixbuf,                 /* pixbuf */
			-1);
	}

}


gboolean set_game_history(const RomEntry *rom)
{
	FILE *history_file;
	GtkTextBuffer *history_txt;
	GtkTextIter text_iter;
	gchar line[2000];
	gint i, n;
	gchar *tmp, *p;
	gchar **games;
	gboolean found_game = FALSE;
	gboolean pointer_in_info = FALSE;
	gboolean extra_newline = FALSE;

	history_txt = GTK_TEXT_BUFFER(main_gui.history_buffer);
	history_file = fopen(gui_prefs.HistoryFile, "r");

	if (!history_file)
	{
		GXMAME_DEBUG("History.dat file not found");
		return (FALSE);
	}

	/* Scan through the file. First look for '$',then $info=game, $bio and $end */
	while (fgets(line, 500, history_file)) {
		p = line;
		tmp = p;
		if (*tmp == '$' && !found_game) {
			/* Found a line */
			if (!strncmp(p, "$info", 5)) {
				p= tmp=p+6;
				/* It is an info line */
				i = 0;
				while (*tmp && (*tmp++ != '\n'))
					i++;
				/* Sometimes the list is continued on the next line */
				i--;
				do {
					p[i] = fgetc(history_file);
					if (p[i] == '\n')
						i--;
					if (p[i] == '\r')
						i--;
					i++;

				} while ((p[i - 1] != '$') && i < 2000 /* buffer size */);
				if (p[i - 1] == '$')
					ungetc('$', history_file);

				p[i - 1] = 0;
				games = g_strsplit(p, ",", 20);
				n = 0;

				while ((games[n] != NULL)) {
					games[n] = g_strchomp(games[n]);

					if (!strcmp(games[n], rom->romname)) {
						/* It is the info for the wanted game */
						found_game = TRUE;
					}
					n++;
				}
				g_strfreev(games);
			}
		} else if (found_game && (*tmp == '$')) {
			if (!strncmp(p, "$bio", 4))
				pointer_in_info = TRUE;
			if (!strncmp(p, "$end", 4)) {
				pointer_in_info = FALSE;
				break;
			}
		} else if (found_game && pointer_in_info) {
			i = 0;
			while (*tmp && (*tmp != '\r') &&(*tmp++ != '\n'))
				i++;
			if (i == 0)
			{/* and a new line but not severals*/
				if (!extra_newline)
				{
					extra_newline = TRUE;
				}
				else
					extra_newline = FALSE;
			}
			else
			{
				extra_newline = FALSE;
			}
			if (!extra_newline)
			{
				gsize bytes_read, bytes_written;
				gchar *utf8_string;

				p[i] = '\n';
				p[i + 1] = '\0';

				/* Must convert to utf-8 otherwise we get GTK-critical warnings from gtk_text_buffer_insert
				   for non-ascii characters. xmame uses ISO-8859-1 (I think).
				*/
				utf8_string = g_convert(p, (i+1), "UTF-8", "ISO-8859-1", &bytes_read, &bytes_written, NULL);
				gtk_text_buffer_get_end_iter (GTK_TEXT_BUFFER(history_txt), &text_iter);
				gtk_text_buffer_insert (GTK_TEXT_BUFFER(history_txt), &text_iter, utf8_string, -1);
				g_free(utf8_string);
			}
		}
	}

	if (!found_game)
	{
		GXMAME_DEBUG("no history info found for %s",rom->romname);
	}

	fclose(history_file);

	return (found_game);
}


gboolean set_game_info(const RomEntry * rom)
{
	FILE *mameinfo_dat;
	GtkTextBuffer *mameinfo_txt;
	GtkTextIter text_iter;
	gchar line[2000];
	gint i;
	gchar *tmp, *p;
	gboolean found_game = FALSE;
	gboolean pointer_in_info = FALSE;
	gboolean extra_newline = FALSE;

	mameinfo_txt = GTK_TEXT_BUFFER(main_gui.history_buffer);
	mameinfo_dat = fopen(gui_prefs.MameInfoFile, "r");

	if (!mameinfo_dat)
	{
		GXMAME_DEBUG("mameinfo_dat file not found");
		return (FALSE);
	}

	/* Scan through the file. First look for '$',then $info=game, $mame and $end */
	while (fgets(line, 500, mameinfo_dat)) {
		p = line;
		tmp = p;
		/* Found a line */
		if (*tmp == '$' && !found_game) {
			/* It is an info line */
			if (!strncmp(p, "$info", 5)) {
				p= tmp=p+6;
				while (*tmp && (*tmp != '\n')&& (*tmp++ != '\r'));
				tmp--;*tmp='\0';
				/* it's the god game */
				if (!strcmp(p, rom->romname) || !strcmp(p, rom->cloneof)) {
					found_game = TRUE;
				}
			}
		} else if (found_game && (*tmp == '$')) {
			if (!strncmp(p, "$mame", 5))
				pointer_in_info = TRUE;
			else if (!strncmp(p, "$end", 4)) {
				pointer_in_info = FALSE;
				break;
			}
		} else if (found_game && pointer_in_info) {
			i = 0;
			while (*tmp && (*tmp != '\r') &&(*tmp++ != '\n'))
				i++;
			if (i == 0)
			{/* and a new line but not severals*/
				if (!extra_newline)
				{
					extra_newline = TRUE;
				}
				else
					extra_newline = FALSE;
			}
			else
			{
				extra_newline = FALSE;
			}
			if (!extra_newline)
			{
				gsize bytes_read, bytes_written;
				gchar *utf8_string;
				p[i] = '\n';
				p[i + 1] = '\0';
				utf8_string = g_convert(p, (i+1), "UTF-8", "ISO-8859-1", &bytes_read, &bytes_written, NULL);
				gtk_text_buffer_get_end_iter (GTK_TEXT_BUFFER(mameinfo_txt), &text_iter);
				gtk_text_buffer_insert (GTK_TEXT_BUFFER(mameinfo_txt), &text_iter, utf8_string, -1);
				g_free(utf8_string);
			}
		}
	}

	if (!found_game)
	{
		GXMAME_DEBUG("no MameInfo info found for %s",rom->romname);
	}

	fclose(mameinfo_dat);

	return (found_game);
}


void change_screenshot (GtkWidget       *widget,
		        GdkEventButton  *event,
		        gpointer         user_data)
{	/* prevent the mouse wheel (button 4 & 5) to change the screenshot*/
	if (event && event->button <= 3)
	{
		gui_prefs.ShowFlyer = (++gui_prefs.ShowFlyer)%5;
		update_screenshot_panel(gui_prefs.DefaultGame_p);
	}
}

void on_screenshot_notebook_switch_page         (GtkNotebook *notebook,
                                                 GtkNotebookPage *page,
                                                 guint page_num,
                                                 gpointer user_data)
{
	gui_prefs.ShowFlyer = page_num;
	update_screenshot_panel(gui_prefs.DefaultGame_p);
}


GtkWidget* get_pixbuf (RomEntry *rom, screenshot_type sctype, int wwidth, int wheight)
{
	GdkPixbuf *scaled_pixbuf = NULL;
	GdkPixbuf *pixbuf;
	GdkPixmap *testpix = NULL;
	GdkPixmap *testmask = NULL;
	GtkWidget *pict = NULL;
	gchar *filename;
	GdkColor transparent;
	int width=0;
	int height=0;
	GError **error = NULL;

	GXMAME_DEBUG("width:%i  height:%i",wwidth,wheight);

	if (!rom)
		return NULL;

	/* Prevent a strange bug where wwidth=wheight=1 */
	if (wwidth<=1)
		wwidth=20;
	if (wheight<=1)
		wheight=20;

	switch (sctype)
	{
		case(SNAPSHOTS):
			filename = g_strdup_printf("%s/%s.png",gui_prefs.SnapshotDirectory,rom->romname);
			break;
		case(FLYERS):
			filename = g_strdup_printf("%s/%s.png",gui_prefs.FlyerDirectory,rom->romname);
			break;
		case(CABINETS):
			filename = g_strdup_printf("%s/%s.png",gui_prefs.CabinetDirectory,rom->romname);
			break;
		case(MARQUEES):
			filename = g_strdup_printf("%s/%s.png",gui_prefs.MarqueeDirectory,rom->romname);
			break;
		case(TITLES):
			filename = g_strdup_printf("%s/%s.png",gui_prefs.TitleDirectory,rom->romname);
			break;
		default:
			filename = g_strdup_printf("%s/%s.png",gui_prefs.SnapshotDirectory,rom->romname);
	}

	pixbuf = gdk_pixbuf_new_from_file (filename, error);
	g_free(filename);
	
	/* no picture found try parent game if any*/
	if ((!pixbuf) && strcmp(rom->cloneof,"-"))
	{
		switch (sctype)
		{
			case(SNAPSHOTS):
				filename = g_strdup_printf("%s/%s.png",gui_prefs.SnapshotDirectory,rom->cloneof);
				break;
			case(FLYERS):
				filename = g_strdup_printf("%s/%s.png",gui_prefs.FlyerDirectory,rom->cloneof);
				break;
			case(CABINETS):
				filename = g_strdup_printf("%s/%s.png",gui_prefs.CabinetDirectory,rom->cloneof);
				break;
			case(MARQUEES):
				filename = g_strdup_printf("%s/%s.png",gui_prefs.MarqueeDirectory,rom->cloneof);
				break;
			case(TITLES):
				filename = g_strdup_printf("%s/%s.png",gui_prefs.TitleDirectory,rom->cloneof);
				break;
			default:
				filename = g_strdup_printf("%s/%s.png",gui_prefs.SnapshotDirectory,rom->cloneof);
		}
		pixbuf = gdk_pixbuf_new_from_file (filename,error);
		g_free(filename);
	}

	/* we havent found the picture in the directory, maybe we could try in a zipfile */
	if (!pixbuf)
	{
		ZIP *zip;
		struct zipent* zipent;
		gchar *zipfile;
		gchar *tmp_buffer;

		switch (sctype)
		{
			case(SNAPSHOTS):
				zipfile = g_build_filename(gui_prefs.SnapshotDirectory, "snap.zip", NULL);
				break;
			case(FLYERS):
				zipfile = g_build_filename(gui_prefs.FlyerDirectory, "flyers.zip", NULL);
				break;
			case(CABINETS):
				zipfile = g_build_filename(gui_prefs.CabinetDirectory, "cabinets.zip", NULL);
				break;
			case(MARQUEES):
				zipfile = g_build_filename(gui_prefs.MarqueeDirectory, "marquees.zip", NULL);
				break;
			case(TITLES):
				zipfile = g_build_filename(gui_prefs.TitleDirectory, "titles.zip", NULL);
				break;
			default:
				zipfile = g_build_filename(gui_prefs.SnapshotDirectory, "snap.zip", NULL);
		}

		zip = openzip(zipfile);

		if (zip)
		{
			GdkPixbufLoader *loader;

			GXMAME_DEBUG("Succesfully open zip file '%s' !", zipfile);
			filename = g_strdup_printf("%s.",rom->romname);

			while ((zipent = readzip(zip)) != 0)
			{
				/* this should allows to find any format of picture in the zip, not only bmp */
				if (!strncmp(filename,zipent->name,strlen(rom->romname)+1))
				{
					GXMAME_DEBUG("found file name %s\twith CRC:%i\tsize%i",
							zipent->name,
							zipent->crc32,
							zipent->uncompressed_size);
					tmp_buffer = read_zipentry(zip, zipent);
					if(tmp_buffer)
					{	/* if the file successfully uncompress, try to load it in a pixbuf loader */
						loader = gdk_pixbuf_loader_new ();
						if (!gdk_pixbuf_loader_write (loader, (guchar *)tmp_buffer, zipent->uncompressed_size,error)) {
							GXMAME_DEBUG("Error while uncompressing %s from %s",zipent->name,zipfile);
						}
						else
						{
							gdk_pixbuf_loader_close (loader,error);
							pixbuf = gdk_pixbuf_loader_get_pixbuf (loader);
						}
						g_free(tmp_buffer);
					}
					/* prevent to read all zip file if we have found the picture's game (uncompressed successfuly or not) */
					break;
				}
			}
			g_free(filename);

			/* no picture found try parent game if any*/
			if ((!pixbuf) && strcmp(rom->cloneof,"-"))
			{
				rewindzip(zip);
				GXMAME_DEBUG("picture not found, trying clone: %s",rom->cloneof);
				filename = g_strdup_printf("%s.",rom->cloneof);
				while ((zipent = readzip(zip)) != 0)
				{
					/* this should allows to find any format of picture in the zip, not only bmp */
					if (!strncmp(filename,zipent->name,strlen(rom->cloneof)+1))
					{
						GXMAME_DEBUG("found file name %s\twith CRC:%i\tsize%i",
							zipent->name,
							zipent->crc32,
							zipent->uncompressed_size);
						tmp_buffer = read_zipentry(zip, zipent);
						if(tmp_buffer)
						{	/* if the file successfully uncompress, try to load it in a pixbuf loader */
							loader = gdk_pixbuf_loader_new ();
							if (!gdk_pixbuf_loader_write (loader, (guchar *)tmp_buffer, zipent->uncompressed_size,error)) {
								GXMAME_DEBUG("Error while uncompressing %s from %s",zipent->name,zipfile);
							}
							else
							{
								gdk_pixbuf_loader_close (loader,error);
								pixbuf = gdk_pixbuf_loader_get_pixbuf (loader);
							}
							g_free(tmp_buffer);
						}
						/* prevent to read all zip file if we have found the picture's game (uncompressed successfuly or not) */
						break;
					}
				}
				g_free(filename);
			}
		}
		if(zip)
			closezip(zip);
		else
			GXMAME_DEBUG("Error, cannot open zip file '%s' !\n", zipfile);

		g_free(zipfile);
		
	}

	if (pixbuf)
	{
		width = gdk_pixbuf_get_width (pixbuf);
		height = gdk_pixbuf_get_height (pixbuf);
		if (gui_prefs.ShowScreenShot == 1)
		{
			/* the picture is wider than the window, resize it to the window size */
			if (width>wwidth)
			{
				height = wwidth*((gdouble)height/(gdouble)width);
				width = wwidth;
			}
		}
		scaled_pixbuf = gdk_pixbuf_scale_simple(pixbuf,
							width,
							height,
							GDK_INTERP_BILINEAR);
		g_object_unref (pixbuf);
		
		gdk_pixbuf_render_pixmap_and_mask(scaled_pixbuf,
					&testpix,
					&testmask,
					127);

		
		g_object_unref (scaled_pixbuf);
	}else {
		GXMAME_DEBUG("no picture (%i), fall back to the default one",sctype);
		testpix = gdk_pixmap_create_from_xpm_d (MainWindow->window,
					&testmask,
					&transparent,
					gxmame_xpm);
	}
	pict = (GtkWidget *) gtk_image_new_from_pixmap (testpix,testmask);
	
	g_object_unref(testpix);
	if(testmask)
		g_object_unref(testmask);
	return pict;
}

void update_screenshot_panel (RomEntry *rom)
{
	GdkPixmap *testpix = NULL;
	GdkPixmap *testmask = NULL;
	gboolean had_info=FALSE,had_history=FALSE;
	GdkColor transparent;
	int wwidth, wheight;
	GtkWidget *pict = NULL;
	GtkTextIter text_iter;

	wwidth=0;wheight=0;

	if (rom!=NULL)
	{
		if (gui_prefs.ShowScreenShotTab==0)
		{
			gdk_drawable_get_size ((main_gui.screenshot_event_box)->window, &wwidth, &wheight);
		}else{
			GtkRequisition requisition;
			gtk_widget_size_request(main_gui.screenshot_box1, &requisition);
			wwidth = requisition.width;
			wheight = requisition.height;
		}

		/* erase, fill and show the history box */
		/* should freeze the history_box here rather than each function otherwise, the position of the cursor will
		    appear between mameinfo and history */
		gtk_text_buffer_set_text (GTK_TEXT_BUFFER(main_gui.history_buffer), "", -1);
		had_history = set_game_history(rom);
		if (had_history)
		{
			gtk_text_buffer_get_end_iter (GTK_TEXT_BUFFER(main_gui.history_buffer), &text_iter);
			gtk_text_buffer_insert (GTK_TEXT_BUFFER(main_gui.history_buffer), &text_iter, "\n", -1);
		}
		had_info = set_game_info(rom);

		if (gui_prefs.ShowScreenShotTab==0)
		{
			gtk_container_remove(GTK_CONTAINER(main_gui.screenshot_event_box),main_gui.main_screenshot);
			gtk_container_remove(GTK_CONTAINER(main_gui.screenshot_hist_vbox),main_gui.screenshot_event_box);
			pict = get_pixbuf (rom, gui_prefs.ShowFlyer, wwidth, wheight);
			main_gui.main_screenshot = pict;
			main_gui.screenshot_event_box = gtk_event_box_new();
		}else
		{
			switch (gui_prefs.ShowFlyer)
			{
				case (SNAPSHOTS):
					gtk_container_remove(GTK_CONTAINER(main_gui.screenshot_box1),main_gui.screenshot1);
					pict = get_pixbuf(rom, 0, wwidth, wheight);
					main_gui.screenshot1 = pict;
					break;
				case (FLYERS):
					gtk_container_remove(GTK_CONTAINER(main_gui.screenshot_box2),main_gui.screenshot2);
					pict = get_pixbuf(rom, 1, wwidth, wheight);
					main_gui.screenshot2 = pict;
					break;
				case (CABINETS):
					gtk_container_remove(GTK_CONTAINER(main_gui.screenshot_box3),main_gui.screenshot3);
					pict = get_pixbuf(rom, 2, wwidth, wheight);
					main_gui.screenshot3 = pict;
					break;
				case (MARQUEES):
					gtk_container_remove(GTK_CONTAINER(main_gui.screenshot_box4),main_gui.screenshot4);
					pict = get_pixbuf(rom, 3, wwidth, wheight);
					main_gui.screenshot4 = pict;
					break;
				case (TITLES):
					gtk_container_remove(GTK_CONTAINER(main_gui.screenshot_box5),main_gui.screenshot5);
					pict = get_pixbuf(rom, 4, wwidth, wheight);
					main_gui.screenshot5 = pict;
					break;
			}
		}

		if (had_history || had_info)
		{
			gtk_widget_show(GTK_WIDGET(main_gui.history_scrollwin));
			if (gui_prefs.ShowScreenShotTab==0)
				gtk_box_pack_end(main_gui.screenshot_hist_vbox,main_gui.screenshot_event_box,FALSE,TRUE,5);
		}
		else
		{
			gtk_widget_hide(GTK_WIDGET(main_gui.history_scrollwin));
			if (gui_prefs.ShowScreenShotTab==0)
				gtk_box_pack_end(main_gui.screenshot_hist_vbox,main_gui.screenshot_event_box,TRUE,TRUE,5);
		}
		if (gui_prefs.ShowScreenShotTab==0)
		{
			gtk_container_add(GTK_CONTAINER(main_gui.screenshot_event_box),GTK_WIDGET(main_gui.main_screenshot));
			gtk_widget_show (main_gui.screenshot_event_box);
			gtk_widget_show (main_gui.main_screenshot);
			g_signal_connect (G_OBJECT (main_gui.screenshot_event_box), "button-release-event",
				G_CALLBACK (change_screenshot),
			   	NULL);
		}else
		{
			switch (gui_prefs.ShowFlyer)
			{
				case (SNAPSHOTS):
					gtk_container_add(GTK_CONTAINER(main_gui.screenshot_box1), GTK_WIDGET(main_gui.screenshot1));
					gtk_widget_show (main_gui.screenshot_box1);
					gtk_widget_show (main_gui.screenshot1);
					break;
				case (FLYERS):
					gtk_container_add(GTK_CONTAINER(main_gui.screenshot_box2), GTK_WIDGET(main_gui.screenshot2));
					gtk_widget_show (main_gui.screenshot_box2);
					gtk_widget_show (main_gui.screenshot2);
					break;
				case (CABINETS):
					gtk_container_add(GTK_CONTAINER(main_gui.screenshot_box3), GTK_WIDGET(main_gui.screenshot3));
					gtk_widget_show (main_gui.screenshot_box3);
					gtk_widget_show (main_gui.screenshot3);
					break;
				case (MARQUEES):
					gtk_container_add(GTK_CONTAINER(main_gui.screenshot_box4), GTK_WIDGET(main_gui.screenshot4));
					gtk_widget_show (main_gui.screenshot_box4);
					gtk_widget_show (main_gui.screenshot4);
					break;
				case (TITLES):
					gtk_container_add(GTK_CONTAINER(main_gui.screenshot_box5), GTK_WIDGET(main_gui.screenshot5));
					gtk_widget_show (main_gui.screenshot_box5);
					gtk_widget_show (main_gui.screenshot5);
					break;
			}
		}
	}
	else /* no roms selected display the default picture */
	{
		if (gui_prefs.ShowScreenShotTab==0)
		{
			gtk_container_remove(GTK_CONTAINER(main_gui.screenshot_event_box),main_gui.main_screenshot);
			gtk_container_remove(GTK_CONTAINER(main_gui.screenshot_hist_vbox),main_gui.screenshot_event_box);

			testpix = gdk_pixmap_create_from_xpm_d(MainWindow->window,
						&testmask,
						&transparent,
						gxmame_xpm);
			main_gui.main_screenshot = (GtkWidget *) gtk_image_new_from_pixmap (testpix,testmask);
			g_object_unref(testpix);
			if (testmask)
				g_object_unref(testmask);

			main_gui.screenshot_event_box = gtk_event_box_new();
			gtk_box_pack_end(main_gui.screenshot_hist_vbox,main_gui.screenshot_event_box,TRUE,TRUE,5);
			gtk_container_add(GTK_CONTAINER(main_gui.screenshot_event_box),GTK_WIDGET(main_gui.main_screenshot));
			gtk_widget_show (main_gui.screenshot_event_box);
			gtk_widget_show (main_gui.main_screenshot);
		}

		/* erase and hide the history box */
		gtk_text_buffer_set_text (GTK_TEXT_BUFFER(main_gui.history_buffer), "", -1);
		gtk_widget_hide(GTK_WIDGET(main_gui.history_scrollwin));
	}
}


void precheck_for_record       (GtkButton       *button,
                                gpointer         inp_selection)
{
	const gchar *inp_file;
	inp_file = gtk_file_selection_get_filename (GTK_FILE_SELECTION(inp_selection));
	/* test if the inp file exist */
	GXMAME_DEBUG("check play file selected: {%s}",inp_file);
	if (g_file_test(inp_file, G_FILE_TEST_EXISTS))
	{	/* if yes print a message and return to the selection screen */

		GtkWidget *dialog;
		gint result;

		dialog = gtk_message_dialog_new (GTK_WINDOW(MainWindow),
							GTK_DIALOG_MODAL,
							GTK_MESSAGE_WARNING,
							GTK_BUTTONS_YES_NO,
							"A file named '%s' already exist\nDo you want to overwrite it ?",
							inp_file);
		result = gtk_dialog_run (GTK_DIALOG (dialog));
		switch (result)
		{
			case GTK_RESPONSE_YES:
				gtk_widget_hide (dialog);
				record_game (NULL,inp_selection);
				break;
			default:
				break;
		}
		gtk_widget_destroy (dialog);

	}
	else
	{
		record_game(NULL,inp_selection);
	}
}


GtkWidget* select_inp (RomEntry *Rom, gboolean play_record)
{
	GtkWidget *inp_selection;
	GtkWidget *ok_button_inp_file;
	GtkWidget *cancel_button_inp_file;
	gchar *temp_text;

	if(play_record)
		inp_selection = gtk_file_selection_new (_("Choose inp file to play"));
	else
		inp_selection = gtk_file_selection_new (_("Choose inp file to record"));

	gtk_window_set_type_hint(GTK_WINDOW (inp_selection), GDK_WINDOW_TYPE_HINT_UTILITY);
	gtk_window_set_position (GTK_WINDOW (inp_selection), GTK_WIN_POS_MOUSE);
	gtk_window_set_modal (GTK_WINDOW (inp_selection), TRUE);
	gtk_window_set_transient_for(GTK_WINDOW(inp_selection),GTK_WINDOW(MainWindow));

	temp_text = g_strdup_printf("%s/%s.inp",gui_prefs.InputDirectory, Rom->romname);
	gtk_file_selection_set_filename(GTK_FILE_SELECTION(inp_selection),temp_text);
	g_free(temp_text);

	ok_button_inp_file = GTK_FILE_SELECTION (inp_selection)->ok_button;
	cancel_button_inp_file = GTK_FILE_SELECTION (inp_selection)->cancel_button;

	if(play_record)
		g_signal_connect (G_OBJECT (ok_button_inp_file), "clicked",
				    G_CALLBACK (playback_game),
				    (gpointer) inp_selection);
	else
		g_signal_connect (G_OBJECT (ok_button_inp_file), "clicked",
				    G_CALLBACK (precheck_for_record),
				    (gpointer) inp_selection);

	/* reenable joystick, was disabled in callback.c (on_playback_input_activate/on_play_and_record_input_activate)*/
	g_signal_connect (G_OBJECT (cancel_button_inp_file), "clicked",
			    G_CALLBACK (joy_focus_on),
				NULL);

	g_signal_connect_swapped (G_OBJECT (cancel_button_inp_file), "clicked",
			    G_CALLBACK (gtk_widget_destroy),
			    (gpointer) inp_selection);
	
	gtk_widget_show(inp_selection);
	return inp_selection;
}
