/*
 * GXMame
 *
 * 2002, 2003, Stéphane Pontier <shadow_walker@users.sourceforge.net>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include "gxmame.h"
#include "io.h"
#include "gui.h"
#include "progression_window.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "gtkjoy.h"

#ifndef XMAME_ROOT
#define XMAME_ROOT "/usr/lib/games/xmame"
#endif

/*
converts a float to a string with precission 5.
this is locale independent.
buf must be of size G_ASCII_DTOSTR_BUF_SIZE
*/
const gchar* my_dtostr(char* buf, gdouble d)
{
	char *p;
	g_ascii_dtostr(buf, FLOAT_BUF_SIZE, d);

	for(p = buf; *p && *p != '.';p++); /* point p to the decimal point */

	if(*p)
	{
		p+=6;	/* precision + 1 */
		*p = '\0';
	}
	return buf;
}

#define LINE_BUF 512

void free_rom_entry(gpointer *rom_entry)
{
	RomEntry *entry = (RomEntry *)rom_entry;
	
	g_free(entry->name_in_list);
	g_free(entry->gamename);
	g_free(entry->gamenameext);
	g_free(entry->manu);
	g_free(entry->cloneof);
	g_free(entry->romof);
	g_free(entry->sampleof);
	g_free(entry);
}

gboolean load_games_list()
{
	FILE *gamelist;
	gint romindex, j;
	GList *listpointer;
	gchar *filename;
	gchar line[LINE_BUF];
	gchar **tmp_array;
	gchar *tmp, *p;
	RomEntry *rom;
	gboolean exe_version_checked = FALSE, in_list;
	gchar *manufacturer1,*manufacturer2,**manufacturer_fields = NULL;
	gint offset;

	supported_games = 0;
	supported_samples_games = 0;
	romindex = 0;
	g_message(_("Loading gamelist"));

	if (list_ver.name)
		g_free(list_ver.name);
	if (list_ver.version)
		g_free(list_ver.version);

	list_ver.version = g_strdup("none");
	list_ver.name = NULL;
	/* clean the existing list if any */
	if(roms)
	{
		g_list_foreach(roms, (GFunc) free_rom_entry, NULL);
		g_list_free(roms);
		roms = NULL;
	}
	if(years)
	{
		g_list_free(years);
		years = NULL;
	}
	if(manufacturers)
	{
		g_list_foreach(manufacturers, (GFunc) g_free, NULL);
		g_list_free(manufacturers);
		manufacturers = NULL;
	}
	if(drivers)
	{
		g_list_free(drivers);
		drivers = NULL;
	}

	filename = g_build_filename(g_get_home_dir(), ".gxmame", "gamelist", NULL);
	gamelist = fopen(filename, "r");
	g_free(filename);
	if (!gamelist) {
		list_ver.version = g_strdup("none");
		return FALSE;
	}

	while (fgets(line, LINE_BUF, gamelist)) {
		p = line;
		tmp = line;

		/* Skip comments */
		if (*tmp != '#') {
			while (*tmp && (*tmp != '\n')) {
				tmp++;
			}
			*tmp = '\0';

			tmp_array = g_strsplit(p, SEP, 23 + (NB_CPU*4));
			rom = g_malloc(sizeof(RomEntry));
			if (!rom || !tmp_array)
			{
				gxmame_message(ERROR, NULL, _("Out of memory while loading gamelist"));
				fclose(gamelist);
				g_strfreev(tmp_array);
				return FALSE;
			}
			rom->name_in_list = NULL;
			g_snprintf(rom->romname, MAX_ROMNAME,
				   "%s", tmp_array[0]);
			rom->gamename = g_strdup(tmp_array[1]);
			rom->gamenameext = g_strdup(tmp_array[2]);
			if (!strcmp(tmp_array[3], "true"))
				rom->the_trailer = TRUE;
			else
				rom->the_trailer = FALSE;

			g_snprintf(rom->year, MAX_YEAR, "%s",
				   tmp_array[4]);
			rom->manu = g_strdup(tmp_array[5]);
			rom->cloneof = g_strdup(tmp_array[6]);
			rom->romof = g_strdup(tmp_array[7]);
			g_snprintf(rom->driver, MAX_ROMNAME,
				   "%s", tmp_array[8]);

			/* Add driver to driver list */
			in_list = FALSE;
			listpointer = g_list_first(drivers);
			while ((listpointer != NULL))
			{
				if (!strcmp(listpointer->data, rom->driver))
				{
					in_list = TRUE;
					break;
				}
				listpointer = g_list_next(listpointer);
			}
			if  (!in_list)
				drivers = g_list_insert_sorted (drivers, rom->driver, (GCompareFunc)strcmp);


			if (!strcmp(tmp_array[9], "true"))
				rom->status = TRUE;
			else
				rom->status = FALSE;
			rom->colors = atoi(tmp_array[10]);


			/* offset of cpu infos in the array */
			offset = 11;
			for (j = 0; j < NB_CPU; j++)
			{
				if (!strncmp(tmp_array[(j*2) + offset], "(sound)",7)) {
					p = tmp_array[(j*2) + offset];
					p+=7;
					g_snprintf(rom->cpu_info[j].name, MAX_CPU, "%s", p);
					rom->cpu_info[j].sound_flag = TRUE;
				}
				else
				{
					g_snprintf(rom->cpu_info[j].name, MAX_CPU,
						   "%s", tmp_array[(j*2) + offset]);
					rom->cpu_info[j].sound_flag = FALSE;
				}

				rom->cpu_info[j].clock = atoi(tmp_array[(j*2) + offset + 1]);
			}

			/* calculate offset of sound cpu infos in the array */
			offset = 11 + (NB_CPU*2);

			for (j=0; j < NB_CPU; j++)
			{
				if (strcmp(tmp_array[offset + (j*2)], "")) {
					g_snprintf(rom->sound_info[j].name, MAX_CPU,
					   "%s", tmp_array[offset + (j*2)]);
				}

				rom->sound_info[j].clock = atoi(tmp_array[offset + (j*2) + 1]);
			}

			offset = 11 + (NB_CPU*4);

			rom->num_players = atoi(tmp_array[offset + 0]);
			rom->num_buttons = atoi(tmp_array[offset + 1]);
			if (strcmp(tmp_array[offset + 2], "")) {
				g_snprintf(rom->control,
					   MAX_CONTROL, "%s",
					   tmp_array[offset + 2]);
			}
			if (!strcmp(tmp_array[offset + 3], "true"))
				rom->vector = TRUE;
			else
				rom->vector = FALSE;

			rom->screen_x = atoi(tmp_array[offset + 4]);
			rom->screen_y = atoi(tmp_array[offset + 5]);
			rom->screen_freq = atoi(tmp_array[offset + 6]);
			if (!strcmp(tmp_array[offset + 7], "horizontal"))
				rom->horizontal = TRUE;
			else
				rom->horizontal = FALSE;

			rom->channels = atoi(tmp_array[offset + 8]);
			rom->nb_roms = atoi(tmp_array[offset + 9]);
			rom->nb_samples = atoi(tmp_array[offset + 10]);
			rom->sampleof = g_strdup(tmp_array[offset + 11]);
			if  (rom->nb_samples>0)
				supported_samples_games++;
			g_strfreev(tmp_array);

			/*initialize timesplayed and favorite fields*/
			rom->timesplayed = 0;
			rom->favourite = 0;

			/*generate Glist for years*/
			in_list = FALSE;
			listpointer = g_list_first(years);
			while ((listpointer != NULL))
			{
				if (!strcmp(listpointer->data, rom->year))
				{
					in_list = TRUE;
					break;
				}
				listpointer = g_list_next(listpointer);
			}
			if  (!in_list)
				years = g_list_insert_sorted (years, rom->year, (GCompareFunc)strcmp);

			/*generate glist for manufacturers*/
			manufacturer_fields = get_manufacturers (rom);
			manufacturer1 = g_strdup(manufacturer_fields[0]);
			in_list = FALSE;
			listpointer = g_list_first(manufacturers);
			while ((listpointer != NULL))
			{
				if (!strncmp(listpointer->data,manufacturer1,5))
				{
					in_list = TRUE;
					break;
				}
				listpointer = g_list_next(listpointer);
			}
			if (!in_list)
			{
				manufacturers = g_list_insert_sorted (manufacturers, manufacturer1, (GCompareFunc)strcmp);
			}
			if (manufacturer_fields[1] != NULL)
			{
				manufacturer2 = g_strdup(manufacturer_fields[1]);
				in_list = FALSE;
				listpointer = g_list_first(manufacturers);
				while ((listpointer != NULL))
				{
					if (!strncmp(listpointer->data,manufacturer2,5))
					{
						in_list = TRUE;
						break;
					}
					listpointer = g_list_next(listpointer);
				}
				if (!in_list)
				{
					manufacturers = g_list_insert_sorted (manufacturers, manufacturer2, (GCompareFunc)strcmp);
				}
			}
			g_strfreev(manufacturer_fields);

			/* fill the some fields with default value if we have problems
			   with game.ini after */
			rom->timesplayed = 0;
			rom->has_roms = 2;
			rom->has_samples = 2;
			rom->favourite = FALSE;
			rom->icon_pixbuf = NULL;

			/* fill the some fields with default value if we have problems
			   with catver after */
			g_snprintf(rom->category,MAX_CATEGORY,"Unknown");
			g_snprintf(rom->mame_ver_added,MAX_MAMEVER,"Unknown");

			/* fill the some fields in order to sort the list after */
			if (strcmp(rom->cloneof,"-"))
				g_snprintf(rom->clonesort,
					   2*MAX_ROMNAME+1, "%s-%s",
					   rom->cloneof,rom->romname);
			else
				g_snprintf(rom->clonesort,
					   2*MAX_ROMNAME, "%s",
					   rom->romname);

			roms = g_list_insert_sorted(roms, (gpointer) rom, (GCompareFunc )compare_game_name);
		} else if (!exe_version_checked) {
			/* Check for the GXMame version in the top comment line */
			while (*tmp && (*tmp != '\n')) {
				tmp++;
			}
			*tmp='\0';
			tmp_array = g_strsplit(p, " ", 3);

			GXMAME_DEBUG("Checking version of gamelist file: %s - %s",tmp_array[1], tmp_array[2]);

			if (strcmp(tmp_array[1], "GXMame") || !tmp_array[2]) {
				gxmame_message(ERROR, NULL, _("Could not recognize format of gamelist.\nYou need to build a new gamelist."));
				list_ver.version = g_strdup("unknown");
				g_strfreev(tmp_array);
				fclose(gamelist);
				return FALSE;
			}
			if (g_ascii_strtod(tmp_array[2], NULL) < 0.7)
			{
				list_ver.version = g_strdup("too old");
				g_strfreev(tmp_array);
				fclose(gamelist);
				return FALSE;
			}
			if (g_ascii_strtod(tmp_array[2], NULL) > 0.7)
			{
				list_ver.version = g_strdup("unknown");
				g_strfreev(tmp_array);
				fclose(gamelist);
				return FALSE;
			}
			exe_version_checked = TRUE;
		}else {
			while (*tmp && (*tmp != '\n')) {
				tmp++;
			}
			*tmp='\0';
			p+=2; /* Skip # */

			if (!strncmp(p, "Version", 7))
			{
				p += 8;
				list_ver.version = g_strdup(p);
			} if (!strncmp(p, "Name", 4))
			{
				p += 5;
				list_ver.name = g_strdup(p);
			}
		}
	}
	fclose(gamelist);

	GXMAME_DEBUG("List for %s %s", list_ver.name, list_ver.version);
	supported_games = g_list_length(roms);
	g_message(_("Loaded %d roms by %d manufacturers covering %d years."), g_list_length(roms),g_list_length(manufacturers),g_list_length(years));
	g_message(_("with %d games supporting samples."),supported_samples_games);

	return (TRUE);
}


gboolean load_games_ini(void)
{
	FILE *gameini_list;
	gchar *filename;
	gchar line[LINE_BUF];
	gchar **option_value = NULL;
	GList *listpointer;
	RomEntry *tmprom = NULL;
	gboolean game_found=FALSE;
	
	filename = g_build_filename(g_get_home_dir(), ".gxmame", "games.ini", NULL);
	if (!filename)
		return FALSE;

	gameini_list = fopen(filename, "r");
	g_free(filename);
	if (!gameini_list)
	{
		GXMAME_DEBUG("games.ini not found");
		not_checked_list = g_list_copy (roms);
		return FALSE;
	}

	while (fgets(line, LINE_BUF, gameini_list))
	{
		if(*line =='[')
		{	
			g_strstrip(g_strdelimit(line,"[]\r\n",' '));
			/* ok only if mame32k games,ini doesn't change */
			/* find the the rom in the glist */
			for(listpointer = g_list_first(roms), game_found = FALSE;
					(listpointer != NULL);
					listpointer = g_list_next(listpointer))
			{
				tmprom = (RomEntry *) listpointer->data;
				if (!strcmp(tmprom->romname,line))
				{
					game_found = TRUE;
					break;
				}
			}
			if (game_found)
			{
				while (fgets(line, LINE_BUF, gameini_list))
				{/* need to test the carriage return if the file come from Mame32K under windows*/
					if(*line =='\r' ||*line =='\n')
						break;
					option_value = g_strsplit(g_strstrip(g_strdelimit(line,"[]\r\n",' ')),"=",0);
					if (option_value[1] != NULL)
					{	if (!strcmp(option_value[0],"PlayCount"))
						{	tmprom->timesplayed = atoi(option_value[1]);
						} else if (!strcmp(option_value[0],"HasRoms"))
						{	tmprom->has_roms = atoi(option_value[1]);
						} else if (!strcmp(option_value[0],"HasSamples"))
						{	tmprom->has_samples = atoi(option_value[1]);
						} else if (!strcmp(option_value[0],"Favorite"))
						{	tmprom->favourite = atoi(option_value[1])?TRUE:FALSE;
						}else if (!strcmp(option_value[0],"AutofireDelay"))
						{	/*store this somewhere to be able to put it again
							  when we will save the game.ini*/
						}
					g_strfreev(option_value);
					}
				}
				if((tmprom->has_roms == 2) || ((tmprom->has_roms==0) && gui_prefs.GameCheck))
				{
					not_checked_list = g_list_append(not_checked_list, tmprom);
				}
			}
		}
		
	}
	fclose(gameini_list);
	return TRUE;
}


gboolean save_games_ini(void)
{
	FILE *game_ini_file;
	gchar *filename;
	RomEntry *rom;
	GList *list_pointer;
	
	
	filename = g_build_filename(g_get_home_dir(), ".gxmame", "games.ini", NULL);
	game_ini_file = fopen(filename, "w");
	g_free(filename);
	if (!game_ini_file)
	{
		GXMAME_DEBUG("unable to write games.ini");
		return FALSE;
	}
		
	for (list_pointer = g_list_first(roms);
			(list_pointer != NULL);
			list_pointer=g_list_next(list_pointer))
	{
		rom = (RomEntry *)list_pointer->data;
		fprintf (game_ini_file, "[%s]\r\n", rom->romname);
		fprintf (game_ini_file, "PlayCount=%i\r\n", rom->timesplayed);
		fprintf (game_ini_file, "HasRoms=%i\r\n", rom->has_roms);
		fprintf (game_ini_file, "HasSamples=%i\r\n", rom->has_samples);
		fprintf (game_ini_file, "Favorite=%s\r\n", rom->favourite?"1":"0");
		fputs ("\r\n",game_ini_file);
	}
	fclose(game_ini_file);
	
	return TRUE;
}


gboolean load_catver_ini(void)
{
	FILE *catver_file;
	gchar *filename;
	gchar line[2000];
	gchar **game_value = NULL;
	GList *listpointer;
	RomEntry *tmprom = NULL;
	gboolean game_found=FALSE;
	gchar *category;
	gchar *version;
	gboolean category_in_list, version_in_list;
	
	filename = g_build_filename(gui_prefs.catverDirectory, "catver.ini", NULL);	
	if (!filename)
		return FALSE;

	catver_file = fopen(filename, "r");
	if (!catver_file)
	{
		GXMAME_DEBUG("catver.ini not found in %s",filename);
		g_free(filename);
		filename = g_build_filename(gui_prefs.catverDirectory, "catver", NULL);	
		catver_file = fopen(filename, "r");
		if (!catver_file)
		{
			GXMAME_DEBUG("catver not found in %s",filename);
			g_free(filename);

			for(listpointer = g_list_first(roms);
					(listpointer != NULL);
					listpointer = g_list_next(listpointer))
			{
				tmprom = (RomEntry *) listpointer->data;
				g_snprintf(tmprom->category, MAX_CATEGORY, _("Unknown"));
				g_snprintf(tmprom->mame_ver_added, MAX_MAMEVER, _("Unknown"));
			}
			return FALSE;
		}
	}
	g_free(filename);

	fgets(line, 500, catver_file);
	if (*line!=';')
	{
		fclose(catver_file);
		return FALSE;
	}

	/* Jump initial comments and empty lines */
	while(*line=='\r' || *line=='\n' ||*line==';')
	{	g_strstrip(g_strdelimit(line,"\r\n",' '));
		GXMAME_DEBUG("Loading catver: %s",line);
		fgets(line, 500, catver_file);
	}
	
	if(strncmp(line,"[Category]", 10))
	{
		fclose(catver_file);
		return FALSE;
	}
		
	while (fgets(line, 500, catver_file))
	{
		/*end of categories */
		if (*line=='[')
			break;
		if (*line=='\r' ||*line=='\n' || *line==';')
			continue;
		
		g_strstrip(g_strdelimit(line,"\r\n",' '));
		game_value = g_strsplit(line,"=",2);
		for(listpointer = g_list_first(roms), game_found = FALSE;
				(listpointer != NULL);
				listpointer = g_list_next(listpointer))
		{
			tmprom = (RomEntry *) listpointer->data;
			if (!strcmp(tmprom->romname,game_value[0]))
			{
				game_found = TRUE;
				break;
			}
		}
		if (game_found)
		{
			g_snprintf(tmprom->category,MAX_CATEGORY, game_value[1]);
		}
		/* create the glist for categories */
		category_in_list = FALSE;
		listpointer = g_list_first(categories);
		while ((listpointer != NULL))
		{
			if (!strcmp(listpointer->data,game_value[1]))
			{
				category_in_list = TRUE;
				break;
			}
			listpointer = g_list_next(listpointer);
		}
		if (!category_in_list)
		{
			category = g_strdup(game_value[1]);
			categories = g_list_insert_sorted (categories, category, (GCompareFunc)strcmp);
		}
		g_strfreev(game_value);
	}

	/* and now come the version*/
	if(strncmp(line,"[VerAdded]", 10))
	{
		GXMAME_DEBUG("[VerAdded] expected but found:%s",line);
		fclose(catver_file);
		return FALSE;
	}
		
	while (fgets(line, 500, catver_file))
	{
		/* Jump initials comment and empty line */
		if (*line=='\r' ||*line=='\n' || *line==';')
			continue;

		g_strstrip(g_strdelimit(line,"\r\n",' '));
		game_value = g_strsplit(line,"=",2);
		for(listpointer = g_list_first(roms), game_found = FALSE;
				(listpointer != NULL);
				listpointer = g_list_next(listpointer))
		{
			tmprom = (RomEntry *) listpointer->data;
			if (!strcmp(tmprom->romname,game_value[0]))
			{
				game_found = TRUE;
				break;
			}
		}
		if (game_found)
		{
			g_snprintf(tmprom->mame_ver_added,MAX_MAMEVER, game_value[1]);
		}
		/* create the glist for versions */
		version_in_list = FALSE;
		listpointer = g_list_first(versions);
		while ((listpointer != NULL))
		{
			if (!strcmp(listpointer->data,game_value[1]))
			{
				version_in_list = TRUE;
				break;
			}
			listpointer = g_list_next(listpointer);
		}
		if (!version_in_list)
		{
			version = g_strdup(game_value[1]);
			versions = g_list_insert_sorted (versions, version, (GCompareFunc)strcmp);
		}
		g_strfreev(game_value);
	}
	fclose(catver_file);
	return TRUE;
}


/* preferences for the gui */
gboolean load_gxmame_ini(void)
{
	FILE *gameini_list;
	gchar *filename;
	gchar line[2000];
	gchar **option_value = NULL;
	gchar **option_value2 = NULL;
	gint i;
	
	g_snprintf(gui_prefs.DefaultGame,MAX_GAMENAME,"%s","Pac-Man (Midway)");
	gui_prefs.FolderID = 1;
	gui_prefs.ShowScreenShot = 1;
	gui_prefs.ShowScreenShotTab = 0;
	gui_prefs.ShowFlyer = 0;
	gui_prefs.ShowToolBar = 1;
	gui_prefs.ShowStatusBar = 1;
	gui_prefs.ShowFolderList = 1;
	gui_prefs.SortColumn = 0;
	gui_prefs.SortReverse = 0;
	gui_prefs.GUIPosX = 0;
	gui_prefs.GUIPosY = 0;
	gui_prefs.GUIWidth = 640;
	gui_prefs.GUIHeight = 400;
	gui_prefs.previous_mode = 3;
	gui_prefs.current_mode = 3;
	gui_prefs.Splitters[0] = 150;
	gui_prefs.Splitters[1] = 300;
	gui_prefs.ModifyThe = TRUE;
	gui_prefs.GameCheck = FALSE;
	gui_prefs.VersionCheck = TRUE;
	gui_prefs.gui_joy = TRUE;
	for (i=0; i<NUMBER_COLUMN; i++)
	{
		gui_prefs.ColumnWidth[i]=0;  /* 0 --> Auto Size */
		gui_prefs.ColumnOrder[i]=i;
		gui_prefs.ColumnShown[i]=0;
	}
	gui_prefs.ColumnShown[GAMENAME]=1;gui_prefs.ColumnShown[HAS_SAMPLES]=1;
	gui_prefs.ColumnShown[ROMNAME]=1;gui_prefs.ColumnShown[VECTOR]=1;
	gui_prefs.ColumnShown[CONTROL]=1;gui_prefs.ColumnShown[TIMESPLAYED]=1;
	gui_prefs.ColumnShown[MANU]=1;gui_prefs.ColumnShown[YEAR]=1;
	gui_prefs.ColumnShown[CLONE]=1;gui_prefs.ColumnShown[DRIVER]=1;
	gui_prefs.ColumnShown[MAMEVER]=1;gui_prefs.ColumnShown[CATEGORY]=1;

	filename = g_build_filename(g_get_home_dir(), ".gxmame", "gxmame.ini", NULL);
	gameini_list = fopen(filename, "r");
	g_free(filename);
	if (!gameini_list)
	{
		GXMAME_DEBUG("gxmame.ini not found");
		return FALSE;
	}
	/* Be careful of the \r\n terminal due to windows */
	while (fgets(line, 500, gameini_list))
	{
		if (!strncmp(line,"[default]",9))
		{	
			while (fgets(line, 500, gameini_list))
			{/* need to test the carriage return if the file come from Mame32K under windows*/
				if(*line =='\r' ||*line =='\n')
					break;
				option_value = g_strsplit(g_strstrip(g_strdelimit(line,"[]\r\n",' ')),"=",0);
				if (option_value[1] != NULL)
				{	if (!strcmp(option_value[0],"DefaultGame"))
					{	g_snprintf(gui_prefs.DefaultGame,MAX_GAMENAME,"%s",option_value[1]);
					} else if (!strcmp(option_value[0],"FolderID"))
					{	gui_prefs.FolderID = atoi(option_value[1]);
					} else if (!strcmp(option_value[0],"ShowScreenShot"))
					{	gui_prefs.ShowScreenShot = atoi(option_value[1]);
					} else if (!strcmp(option_value[0],"ShowScreenShotTab"))
					{	gui_prefs.ShowScreenShotTab = atoi(option_value[1]);
					} else if (!strcmp(option_value[0],"ShowFlyer"))
					{	gui_prefs.ShowFlyer = atoi(option_value[1]);
					} else if (!strcmp(option_value[0],"ShowToolBar"))
					{	gui_prefs.ShowToolBar = atoi(option_value[1]);
					} else if (!strcmp(option_value[0],"ShowStatusBar"))
					{	gui_prefs.ShowStatusBar = atoi(option_value[1]);
					} else if (!strcmp(option_value[0],"ModifyThe"))
					{	gui_prefs.ModifyThe = atoi(option_value[1]);
					} else if (!strcmp(option_value[0],"ShowFolderList"))
					{	gui_prefs.ShowFolderList = atoi(option_value[1]);
					} else if (!strcmp(option_value[0],"GameCheck"))
					{	gui_prefs.GameCheck = atoi(option_value[1]);
					} else if (!strcmp(option_value[0],"VersionCheck"))
					{	gui_prefs.VersionCheck = atoi(option_value[1]);
					} else if (!strcmp(option_value[0],"gui_joy"))
					{	gui_prefs.gui_joy = atoi(option_value[1]);
					} else if (!strcmp(option_value[0],"SortColumn"))
					{	gui_prefs.SortColumn = atoi(option_value[1]);
					} else if (!strcmp(option_value[0],"SortReverse"))
					{	gui_prefs.SortReverse = atoi(option_value[1]);
					} else if (!strcmp(option_value[0],"GUIPosX"))
					{	gui_prefs.GUIPosX = atoi(option_value[1]);
					} else if (!strcmp(option_value[0],"GUIPosY"))
					{	gui_prefs.GUIPosY = atoi(option_value[1]);
					} else if (!strcmp(option_value[0],"GUIWidth"))
					{	gui_prefs.GUIWidth = atoi(option_value[1]);
					}else if (!strcmp(option_value[0],"GUIHeight"))
					{	gui_prefs.GUIHeight = atoi(option_value[1]);
					}else if (!strcmp(option_value[0],"ListMode"))
					{	gui_prefs.current_mode = gui_prefs.previous_mode = atoi(option_value[1]);
					}else if (!strcmp(option_value[0],"Splitters"))
					{
						option_value2 = g_strsplit(option_value[1],",",2);
						gui_prefs.Splitters[0] = atoi(option_value2[0]);
						gui_prefs.Splitters[1] = atoi(option_value2[1]);
						g_strfreev(option_value2);
					} else if (!strcmp(option_value[0],"ListFont"))
					{	gui_prefs.ListFont = g_strdup(option_value[1]);
					} else if (!strcmp(option_value[0],"ColumnWidth"))
					{
						option_value2 = g_strsplit(option_value[1],",",NUMBER_COLUMN);
						for (i=0; i<NUMBER_COLUMN; i++)
							gui_prefs.ColumnWidth[i]=atoi(option_value2[i]);
						g_strfreev(option_value2);
					} else if (!strcmp(option_value[0],"ColumnOrder"))
					{
						option_value2 = g_strsplit(option_value[1],",",NUMBER_COLUMN);
						for (i=0; i<NUMBER_COLUMN; i++)
							gui_prefs.ColumnOrder[i]=atoi(option_value2[i]);
						g_strfreev(option_value2);
					} else if (!strcmp(option_value[0],"ColumnShown"))
					{
						option_value2 = g_strsplit(option_value[1],",",NUMBER_COLUMN);
						for (i=0; i<NUMBER_COLUMN; i++)
							gui_prefs.ColumnShown[i]=atoi(option_value2[i]);
						g_strfreev(option_value2);
					}
				g_strfreev(option_value);
				}
			}
		}
	}
	fclose(gameini_list);
	return TRUE;
}


gboolean save_gxmame_ini(void)
{
	FILE *gxmame_ini_file = NULL;
	gchar *filename;
	gint i;
	
	filename = g_build_filename(g_get_home_dir() , ".gxmame", "gxmame.ini", NULL);
	gxmame_ini_file = fopen(filename, "w");
	g_free(filename);

	if (!gxmame_ini_file)
	{
		GXMAME_DEBUG("unable to write gxmame.ini");
		return FALSE;
	}

	fputs ("[default]\r\n",gxmame_ini_file);
	fprintf (gxmame_ini_file, "DefaultGame=%s\r\n", gui_prefs.DefaultGame);
	fprintf (gxmame_ini_file, "FolderID=%i\r\n", gui_prefs.FolderID);
	fprintf (gxmame_ini_file, "ShowScreenShot=%i\r\n", gui_prefs.ShowScreenShot);
	fprintf (gxmame_ini_file, "ShowScreenShotTab=%i\r\n",gui_prefs.ShowScreenShotTab);
	fprintf (gxmame_ini_file, "ShowFlyer=%i\r\n", gui_prefs.ShowFlyer);
	fprintf (gxmame_ini_file, "ShowToolBar=%i\r\n", gui_prefs.ShowToolBar);
	fprintf (gxmame_ini_file, "ModifyThe=%i\r\n", gui_prefs.ModifyThe);
	fprintf (gxmame_ini_file, "ShowStatusBar=%i\r\n", gui_prefs.ShowStatusBar);
	fprintf (gxmame_ini_file, "ShowFolderList=%i\r\n", gui_prefs.ShowFolderList);
	fprintf (gxmame_ini_file, "GameCheck=%i\r\n", gui_prefs.GameCheck);
	fprintf (gxmame_ini_file, "VersionCheck=%i\r\n", gui_prefs.VersionCheck);
	fprintf (gxmame_ini_file, "gui_joy=%i\r\n", gui_prefs.gui_joy);
	fprintf (gxmame_ini_file, "SortColumn=%i\r\n", gui_prefs.SortColumn);
	fprintf (gxmame_ini_file, "SortReverse=%i\r\n", gui_prefs.SortReverse);
	
	gtk_window_get_position(GTK_WINDOW(MainWindow), &gui_prefs.GUIPosX, &gui_prefs.GUIPosY);
	fprintf (gxmame_ini_file, "GUIPosX=%i\r\n", gui_prefs.GUIPosX);
	fprintf (gxmame_ini_file, "GUIPosY=%i\r\n", gui_prefs.GUIPosY);
	
	gdk_drawable_get_size(MainWindow->window, &gui_prefs.GUIWidth,&gui_prefs.GUIHeight);
	fprintf (gxmame_ini_file, "GUIWidth=%i\r\n", gui_prefs.GUIWidth);
	fprintf (gxmame_ini_file, "GUIHeight=%i\r\n", gui_prefs.GUIHeight);

	fprintf (gxmame_ini_file, "ListMode=%i\r\n", gui_prefs.current_mode);
	
	fprintf (gxmame_ini_file, "Splitters=%i,%i\r\n",
			gui_prefs.ShowFolderList?main_gui.scrolled_window_filters->allocation.width:gui_prefs.Splitters[0],
			gui_prefs.ShowScreenShot?main_gui.scrolled_window_games->allocation.width:gui_prefs.Splitters[1]);
	
	fputs ("ListFont=",gxmame_ini_file);
	if (gui_prefs.ListFont)
		fputs (gui_prefs.ListFont,gxmame_ini_file);

	fputs ("\r\nColumnWidth=",gxmame_ini_file);
	for (i=0; i<NUMBER_COLUMN; i++)
	{
		fprintf(gxmame_ini_file, "%i",gui_prefs.ColumnWidth[i]);

		if (i != NUMBER_COLUMN-1)
			fputs(",",gxmame_ini_file);
	}

	fputs ("\r\nColumnOrder=",gxmame_ini_file);
	for (i=0; i<NUMBER_COLUMN; i++)
	{
		fprintf(gxmame_ini_file, "%i", gui_prefs.ColumnOrder[i]);

		if (i!=NUMBER_COLUMN-1)
			fputs(",", gxmame_ini_file);
	}

	fputs ("\r\nColumnShown=",gxmame_ini_file);
	for (i=0; i<NUMBER_COLUMN; i++)
	{
		fprintf(gxmame_ini_file, "%i",gui_prefs.ColumnShown[i]);
		if (i != NUMBER_COLUMN-1)
			fputs (",",gxmame_ini_file);
	}

	fputs ("\r\n\r\n",gxmame_ini_file);

	fclose(gxmame_ini_file);
	return TRUE;
}


/* this the master config file, it will contain the path to every other config file
   when we will share these config files with another mame32 application */
gboolean load_gxmamerc(void)
{
	FILE *gamerc_file;
	gchar *filename;
	gchar line[2000];
	gchar **option_value = NULL;
	gchar *mame_executable = NULL;
	gchar **xmame_executables;

	gui_prefs.catverDirectory = g_build_filename(g_get_home_dir(), ".gxmame", NULL);
	gui_prefs.CheatFile = g_build_filename(XMAME_ROOT, "cheat.dat", NULL);
	gui_prefs.HiscoreFile = g_build_filename(XMAME_ROOT, "hiscore.dat", NULL);
	gui_prefs.HistoryFile = g_build_filename(XMAME_ROOT, "history.dat", NULL);
	gui_prefs.MameInfoFile = g_build_filename(XMAME_ROOT, "mameinfo.dat", NULL);
	
	gui_prefs.clone_color.red=47031;
	gui_prefs.clone_color.green=47031;
	gui_prefs.clone_color.blue=47031;
	gui_prefs.Joystick_in_GUI = g_strdup(GXMAME_JOY_DEV);
	
	filename = g_build_filename(g_get_home_dir(), ".gxmame", "gxmamerc", NULL);	
	gamerc_file = fopen(filename, "r");
	g_free(filename);
	if (!gamerc_file)
	{
		xmame_table_add("xmame");
		GXMAME_DEBUG("gxmamerc not found");
		return FALSE;
	}

	while (fgets(line, 500, gamerc_file))
	{
		/* skip comments */
		if (*line !='#')
		{	
			if(*line =='\n')
				break;
			option_value = g_strsplit(g_strstrip(g_strdelimit(line,"[]\n",' ')),"=",0);
			if (option_value[1] != NULL)
			{
				if (!strcmp(option_value[0],"mame_executable"))
				{
					if (mame_executable)
						g_free(mame_executable);
					mame_executable = g_strdup(option_value[1]);
				} else if (!strcmp(option_value[0],"xmame_executables_array"))
				{
					gint i;
					xmame_executables = g_strsplit(option_value[1],";",0);
					if (xmame_executables)
					{
						for (i=0; xmame_executables[i]; i++)
							xmame_table_add(xmame_executables[i]);
						g_strfreev(xmame_executables);
					}
					
				} else if (!strcmp(option_value[0],"catverDirectory"))
				{
					g_free(gui_prefs.catverDirectory);
					gui_prefs.catverDirectory = g_strdup(option_value[1]);
				} else if (!strcmp(option_value[0],"CheatFile"))
				{	
					g_free(gui_prefs.CheatFile);
					gui_prefs.CheatFile = g_strdup(option_value[1]);
				} else if (!strcmp(option_value[0],"HiscoreFile"))
				{	
					g_free(gui_prefs.HiscoreFile);
					gui_prefs.HiscoreFile = g_strdup(option_value[1]);
				} else if (!strcmp(option_value[0],"HistoryFile"))
				{	
					g_free(gui_prefs.HistoryFile);
					gui_prefs.HistoryFile = g_strdup(option_value[1]);
				} else if (!strcmp(option_value[0],"MameInfoFile"))
				{	
					g_free(gui_prefs.MameInfoFile);
					gui_prefs.MameInfoFile = g_strdup(option_value[1]);
				} else if (!strcmp(option_value[0],"Joystick_in_GUI"))
				{	
					g_free(gui_prefs.Joystick_in_GUI);
					gui_prefs.Joystick_in_GUI = g_strdup(option_value[1]);
				} else if (!strcmp(option_value[0],"clone_color.red"))
				{	
					gui_prefs.clone_color.red = atoi(option_value[1]);
				} else if (!strcmp(option_value[0],"clone_color.green"))
				{	
					gui_prefs.clone_color.green = atoi(option_value[1]);
				} else if (!strcmp(option_value[0],"clone_color.blue"))
				{	
					gui_prefs.clone_color.blue = atoi(option_value[1]);
				}
				g_strfreev(option_value);
			}
		}
	}
	fclose(gamerc_file);

	/* set the current_executable */
	if (mame_executable)
	{
		current_exec = xmame_table_get(mame_executable);
		g_free(mame_executable);
	}
	
	if (!current_exec)
	{
		gchar **exec_list = xmame_table_get_all();
		current_exec = xmame_table_get(exec_list[0]);
		g_free(exec_list);
	}
	return TRUE;
}


gboolean save_gxmamerc(void)
{
	FILE *gamerc_file;
	gchar *filename;
	guint i;
	
	filename = g_build_filename(g_get_home_dir(), ".gxmame", "gxmamerc", NULL);	
	gamerc_file = fopen(filename, "w");
	g_free(filename);
	if (!gamerc_file)
	{
		GXMAME_DEBUG("unable to write gxmamerc");
		return (FALSE);
	}
	fputs ("# master config file of GXMame\n",gamerc_file);
	if (current_exec)
		fprintf (gamerc_file, "mame_executable=%s\n", current_exec->path);

	if (xmame_table_size() > 0)
	{
		gchar **exec_paths;
		exec_paths = xmame_table_get_all();
		fputs ("xmame_executables_array=",gamerc_file);
		for (i=0;exec_paths[i]!=NULL;i++)
		{	
			fputs (exec_paths[i],gamerc_file);
			if (exec_paths[i+1]!=NULL)
				fputs (";",gamerc_file);
		}
		g_free(exec_paths);
	}

	fprintf (gamerc_file, "\ncatverDirectory=%s", gui_prefs.catverDirectory);
	fprintf (gamerc_file, "\nCheatFile=%s", gui_prefs.CheatFile);
	fprintf (gamerc_file, "\nHiscoreFile=%s", gui_prefs.HiscoreFile);
	fprintf (gamerc_file, "\nHistoryFile=%s", gui_prefs.HistoryFile);
	fprintf (gamerc_file, "\nMameInfoFile=%s", gui_prefs.MameInfoFile);
	fprintf (gamerc_file, "\nJoystick_in_GUI=%s", gui_prefs.Joystick_in_GUI);
	fprintf (gamerc_file, "\nclone_color.red=%i", gui_prefs.clone_color.red);
	fprintf (gamerc_file, "\nclone_color.green=%i", gui_prefs.clone_color.green);
	fprintf (gamerc_file, "\nclone_color.blue=%i\n", gui_prefs.clone_color.blue);
	
	fclose(gamerc_file);
	return TRUE;
}


/* this is where directory paths are set (common with mame32k) */
gboolean load_dirs_ini(void)
{
	FILE *dirs_ini_file;
	gchar *filename;
	gchar line[2000];
	gchar **option_value = NULL;
	
	gui_prefs.RomPath = g_strsplit(XMAME_ROOT "/roms/",";",2);
	gui_prefs.SamplePath = g_strsplit(XMAME_ROOT "/samples/",";",2);
	gui_prefs.SnapshotDirectory = g_build_filename(XMAME_ROOT, "snap", NULL);
	gui_prefs.ArtworkDirectory = g_build_filename(XMAME_ROOT, "artwork", NULL);
	gui_prefs.HiscoreDirectory = g_build_filename(g_get_home_dir() , ".xmame", "hi", NULL);
	gui_prefs.FlyerDirectory = g_build_filename(XMAME_ROOT, "flyers", NULL);
	gui_prefs.CabinetDirectory = g_build_filename(XMAME_ROOT, "cabinets", NULL);
	gui_prefs.MarqueeDirectory = g_build_filename(XMAME_ROOT, "marquees", NULL);
	gui_prefs.TitleDirectory = g_build_filename(XMAME_ROOT, "titles", NULL);
	gui_prefs.IconDirectory = g_build_filename(XMAME_ROOT, "icons", NULL);
	
	gui_prefs.DiffDirectory = g_build_filename(g_get_home_dir(), ".xmame", "diff", NULL);
	gui_prefs.CtrlrDirectory = g_build_filename(XMAME_ROOT, "ctrlr", NULL);
	
	gui_prefs.InputDirectory = g_build_filename(g_get_home_dir(), ".xmame", "inp", NULL);
	gui_prefs.NVRamDirectory = g_build_filename(g_get_home_dir(), ".xmame", "nvram", NULL);
	gui_prefs.MemCardDirectory = g_build_filename(g_get_home_dir(), ".xmame", "memcard", NULL);
	gui_prefs.ConfigDirectory = g_build_filename(g_get_home_dir(), ".xmame" , "cfg", NULL);
	gui_prefs.StateDirectory = g_build_filename(g_get_home_dir(), ".xmame", "sta", NULL);
	gui_prefs.inipath = g_build_filename(g_get_home_dir(), ".xmame" , "ini", NULL);
	
	filename = g_build_filename(g_get_home_dir(), ".gxmame", "dirs.ini", NULL);	
	dirs_ini_file = fopen(filename, "r");
	g_free(filename);
	if (!dirs_ini_file)
	{
		GXMAME_DEBUG("dirs.ini not found");
		return FALSE;
	}
	/* Be careful of the \r\n terminal due to windows */
	while (fgets(line, 500, dirs_ini_file))
	{
		if (!strncmp(line,"[directories]",13))
		{	
			while (fgets(line, 500, dirs_ini_file))
			{/* need to test the carriage return if the file come from Mame32K under windows*/
				if(*line =='\r' ||*line =='\n')
					break;
				option_value = g_strsplit(g_strstrip(g_strdelimit(line,"[]\r\n",' ')),"=",0);
				if (option_value[1] != NULL)
				{	if (!strcmp(option_value[0],"RomPath"))
					{	
						g_strfreev(gui_prefs.RomPath);
						gui_prefs.RomPath = g_strsplit(option_value[1],";",0);
					} else if (!strcmp(option_value[0],"SamplePath"))
					{	
						g_strfreev(gui_prefs.SamplePath);
						gui_prefs.SamplePath = g_strsplit(option_value[1],";",0);
					} else if (!strcmp(option_value[0],"ConfigDirectory"))
					{	
						g_free(gui_prefs.ConfigDirectory);
						gui_prefs.ConfigDirectory = g_strdup(option_value[1]);
/*					} else if (!strcmp(option_value[0],"FolderDirectory"))
					{	
						g_free(gui_prefs.FolderDirectory);
						gui_prefs.FolderDirectory = g_strdup(option_value[1]);
*/					} else if (!strcmp(option_value[0],"NVRamDirectory"))
					{	
						g_free(gui_prefs.NVRamDirectory);
						gui_prefs.NVRamDirectory = g_strdup(option_value[1]);
					} else if (!strcmp(option_value[0],"CtrlrDirectory"))
					{	
						g_free(gui_prefs.CtrlrDirectory);
						gui_prefs.CtrlrDirectory = g_strdup(option_value[1]);
					} else if (!strcmp(option_value[0],"MemCardDirectory"))
					{	
						g_free(gui_prefs.MemCardDirectory);
						gui_prefs.MemCardDirectory = g_strdup(option_value[1]);
					} else if (!strcmp(option_value[0],"InputDirectory"))
					{	
						g_free(gui_prefs.InputDirectory);
						gui_prefs.InputDirectory = g_strdup(option_value[1]);
					} else if (!strcmp(option_value[0],"HiscoreDirectory"))
					{	
						g_free(gui_prefs.HiscoreDirectory);
						gui_prefs.HiscoreDirectory = g_strdup(option_value[1]);
					} else if (!strcmp(option_value[0],"StateDirectory"))
					{	
						g_free(gui_prefs.StateDirectory);
						gui_prefs.StateDirectory = g_strdup(option_value[1]);
					} else if (!strcmp(option_value[0],"ArtworkDirectory"))
					{
						g_free(gui_prefs.ArtworkDirectory);
						gui_prefs.ArtworkDirectory = g_strdup(option_value[1]);
					} else if (!strcmp(option_value[0],"SnapshotDirectory"))
					{	
						g_free(gui_prefs.SnapshotDirectory);
						gui_prefs.SnapshotDirectory = g_strdup(option_value[1]);
/*					} else if (!strcmp(option_value[0],"WavDirectory"))
					{
						g_free(gui_prefs.WavDirectory);
						gui_prefs.WavDirectory = g_strdup(option_value[1]);
*/					} else if (!strcmp(option_value[0],"FlyerDirectory"))
					{
						g_free(gui_prefs.FlyerDirectory);
						gui_prefs.FlyerDirectory = g_strdup(option_value[1]);
					} else if (!strcmp(option_value[0],"CabinetDirectory"))
					{
						g_free(gui_prefs.CabinetDirectory);
						gui_prefs.CabinetDirectory = g_strdup(option_value[1]);
					} else if (!strcmp(option_value[0],"MarqueeDirectory"))
					{	
						g_free(gui_prefs.MarqueeDirectory);
						gui_prefs.MarqueeDirectory = g_strdup(option_value[1]);
					} else if (!strcmp(option_value[0],"TitleDirectory"))
					{	
						g_free(gui_prefs.TitleDirectory);
						gui_prefs.TitleDirectory = g_strdup(option_value[1]);
					} else if (!strcmp(option_value[0],"DiffDirectory"))
					{	
						g_free(gui_prefs.DiffDirectory);
						gui_prefs.DiffDirectory = g_strdup(option_value[1]);
					} else if (!strcmp(option_value[0],"IconDirectory"))
					{	
						g_free(gui_prefs.IconDirectory);
						gui_prefs.IconDirectory = g_strdup(option_value[1]);
					} else if (!strcmp(option_value[0],"inipath"))
					{	
						g_free(gui_prefs.inipath);
						gui_prefs.inipath = g_strdup(option_value[1]);
					}
				g_strfreev(option_value);
				}
			}
		}
	}
	fclose(dirs_ini_file);
	return TRUE;
}


gboolean save_dirs_ini(void)
{
	FILE *dir_ini_file;
	gchar *filename;
	int i;
	
	filename = g_build_filename(g_get_home_dir(), ".gxmame", "dirs.ini", NULL);	
	dir_ini_file = fopen(filename, "w");
	g_free(filename);
	if (!dir_ini_file)
	{
		GXMAME_DEBUG("unable to write dirs.ini");
		return (FALSE);
	}

	fputs ("[directories]",dir_ini_file);
	fputs ("\r\nRomPath=",dir_ini_file);
	for (i=0;gui_prefs.RomPath[i]!=NULL;i++)
	{	
		fputs (gui_prefs.RomPath[i],dir_ini_file);
		if (gui_prefs.RomPath[i+1]!=NULL)
			fputs (";",dir_ini_file);
	}
	fputs ("\r\nSamplePath=",dir_ini_file);
	for (i=0;gui_prefs.SamplePath[i]!=NULL;i++)
	{
		fputs (gui_prefs.SamplePath[i],dir_ini_file);
		if (gui_prefs.SamplePath[i+1]!=NULL)
			fputs (";",dir_ini_file);
	}
	fputs ("\r\nConfigDirectory=",dir_ini_file);
	fputs (gui_prefs.ConfigDirectory,dir_ini_file);
	fputs ("\r\nNVRamDirectory=",dir_ini_file);
	fputs (gui_prefs.NVRamDirectory,dir_ini_file);
	fputs ("\r\nCtrlrDirectory=",dir_ini_file);
	fputs (gui_prefs.CtrlrDirectory,dir_ini_file);
	fputs ("\r\nMemCardDirectory=",dir_ini_file);
	fputs (gui_prefs.MemCardDirectory,dir_ini_file);
	fputs ("\r\nInputDirectory=",dir_ini_file);
	fputs (gui_prefs.InputDirectory,dir_ini_file);
	fputs ("\r\nHiscoreDirectory=",dir_ini_file);
	fputs (gui_prefs.HiscoreDirectory,dir_ini_file);
	fputs ("\r\nStateDirectory=",dir_ini_file);
	fputs (gui_prefs.StateDirectory,dir_ini_file);
	fputs ("\r\nArtworkDirectory=",dir_ini_file);
	fputs (gui_prefs.ArtworkDirectory,dir_ini_file);
	fputs ("\r\nSnapshotDirectory=",dir_ini_file);
	fputs (gui_prefs.SnapshotDirectory,dir_ini_file);
	fputs ("\r\nFlyerDirectory=",dir_ini_file);
	fputs (gui_prefs.FlyerDirectory,dir_ini_file);
	fputs ("\r\nCabinetDirectory=",dir_ini_file);
	fputs (gui_prefs.CabinetDirectory,dir_ini_file);
	fputs ("\r\nMarqueeDirectory=",dir_ini_file);
	fputs (gui_prefs.MarqueeDirectory,dir_ini_file);
	fputs ("\r\nTitleDirectory=",dir_ini_file);
	fputs (gui_prefs.TitleDirectory,dir_ini_file);
	fputs ("\r\nDiffDirectory=",dir_ini_file);
	fputs (gui_prefs.DiffDirectory,dir_ini_file);
	fputs ("\r\nIconDirectory=",dir_ini_file);
	fputs (gui_prefs.IconDirectory,dir_ini_file);
	fputs ("\r\ninipath=",dir_ini_file);
	fputs (gui_prefs.inipath,dir_ini_file);
	
	fputs ("\r\n\r\n",dir_ini_file);
	fclose(dir_ini_file);
	return TRUE;
}


/* here are the options passed to X-mame*/
gboolean load_options(RomEntry *rom)
{
	FILE *options_file;
	gchar *filename;
	gchar line[2000];
	gchar **option_value = NULL;
	xmame_game_options *target;
	GList *ctrlr_list = NULL;
	GList *my_list = NULL;

	if (rom==NULL)
	{
		target = &default_options;
		filename = g_build_filename(g_get_home_dir(), ".gxmame", "options", "default", NULL);
	}
	else
	{
		/* free a previous game_option structure (first free additional options of the game) */
		if (game_options)
		{
			g_free(((xmame_game_options *)game_options)->additional_options);
			g_free(game_options);
		}
		filename = g_build_filename(g_get_home_dir(), ".gxmame", "options", rom->romname, NULL);
		/* if there is no file with that name, exit, will use default options */
		if(!g_file_test(filename, G_FILE_TEST_EXISTS))
		{
			game_options = NULL;
			GXMAME_DEBUG("option file for %s not found",rom->romname);
			g_free(filename);
			return (FALSE);
		}
		/* otherwise, create a xmame_game_options struct for this game */
		game_options = g_malloc (sizeof(xmame_game_options));
		target = game_options;
	}
	
	/* default options */
/*** Video Related ***/
	target->bpp = 0;
	target->arbheight = 0;
	target->heightscale = 1;
	target->widthscale = 1;
/*	target->scale = 0.00000;*/
	target->effect = 0;
	target->autodouble = TRUE;		
	target->dirty = TRUE;
	target->scanlines = FALSE;
	target->artwork = TRUE;
	target->use_backdrops = TRUE;
	target->use_overlays = TRUE;
	target->use_bezels = TRUE;
	target->artwork_crop = FALSE;
	target->artwork_resolution = 0;
	target->frameskipper = 0;
	target->throttle = TRUE;
	target->sleepidle = FALSE;
	target->autoframeskip = TRUE;
	target->maxautoframeskip = 8;
	target->frameskip = 0;
	target->brightness = 1.00000;
	target->gamma_correction = 1.00000;
	target->norotate = FALSE;
	target->ror = FALSE;
	target->rol = FALSE;
	target->flipx = FALSE;
	target->flipy = FALSE;

/*** Vector Games Related ***/
	target->vectorres_flag = FALSE;
	strcpy (target->vectorres, "640x480");
	target->beam = 1.000000;
	target->flicker = 0.000000;
	target->antialias = TRUE;
	target->translucency = TRUE;
	target->intensity = 1.5;

/*display_opts (depend target)*/
/*** X11 Related ***/
	target->x11_mode = 0;
				
				
 /*** X11-window Related ***/
	target->cursor = TRUE;
	target->mitshm = TRUE;
	target->xvext = TRUE;
	target->vidix = FALSE;
	target->yuv = FALSE;
	target->yv12 = FALSE;
	target->xvfullscreen = FALSE;
	target->xsync = TRUE;
	target->privatecmap = FALSE;
	target->xil = TRUE;
	target->mtxil = FALSE;
	target->run_in_root_window = FALSE;
	target->root_window_id = 0;
	target->xvgeom_flag = FALSE;
	strcpy (target->geometry,"640x480");

 /*** X11-input related ***/
 /* (X11 & GL Glide) */
	target->grabmouse = FALSE;
	target->grabkeyboard = FALSE;
	target->winkeys = FALSE;
	strcpy (target->mapkey, " ");


/*** OpenGL Related ***/
	target->glfullscreen = FALSE;
	target->gldblbuffer = TRUE;
	target->gltexture_size = FALSE;
	target->glforceblitmode = FALSE;
	target->glext78 = TRUE;
	target->glbilinear = TRUE;
	target->gldrawbitmap = TRUE;
	target->gldrawbitmapvec = TRUE;
	target->glcolormod = TRUE;
	target->glbeam = 1.000000;
	target->glalphablending = TRUE;
	target->glantialias = TRUE;
	target->glantialiasvec = TRUE;
	strcpy (target->gllibname, "libGL.so");
	strcpy (target->glulibname, "libGLU.so");
	target->cabview = FALSE;
	strcpy (target->cabinet, "glmamejau");
	target->glres_flag = FALSE;
	strcpy (target->glres, " ");
				

/*** SDL Related ***/
/* 	target->listmodes = 0;	*/	
	target->fullscreen = TRUE;
	target->sdl_auto_mode = TRUE;
	target->modenumber = 0;
	strcpy (target->sdlmapkey, " ");

/*** GGI Related ***/
	target->ggilinear = FALSE;
	target->ggi_force_resolution = FALSE;
	target->xres = 640;
	target->yres = 480;

/*** Svgalib Related ***/
	target->tweak = FALSE;
	target->planar = TRUE;
	target->linear = FALSE;
	target->centerx = 0;
	target->centery = 0;

/*** FX (Glide) Related ***/
	strcpy (target->resolution, "640x480");
	target->fxgkeepaspect = TRUE;

/*** Photon Related ***/
	target->render_mode = 0;

 /*** Photon-window Related ***/
	target->phcursor = TRUE;
                 
 /*** Photon-input Related ***/
/*	target->phgrabmouse = FALSE;
	target->phwinkeys = FALSE;
	strcpy (target->phmapkey, " ");
*/

 /*** Video Mode Selection Related ***/
 /*** (SVGA, xf86_dga_opts, GGI) ***/
	target->keepaspect = 1;
	target->displayaspectratio = 1.330000;
	strcpy (target->disablemode, " ");

/*** Sound Related ***/
	target->sound = TRUE;
	target->samples = TRUE;
	target->fakesound = FALSE;
	target->samplefre = 22050;
	target->bufsize = 3.000000;
	target->volume = -3;
	strcpy (target->audiodevice, " ");
	strcpy (target->mixerdevice, " ");
	strcpy (target->soundfile, "xmameout.wav");

/*** Input Related ***/
	target->joytype = 0;
	target->analogstick = FALSE;
	target->mouse = FALSE;
	target->hotrod = FALSE;
	target->hotrodse = FALSE;
	target->usbpspad = FALSE;
	target->rapidfire = FALSE;
	target->ugcicoin = FALSE;
	
	strcpy (target->keymap, " ");
	strcpy (target->ctrlr, "None");
	strcpy (target->joydevname, get_joy_dev());
	target->joydevname[strlen(target->joydevname) - 1] = '\0';

 /***joy_pad_opts***  *FM townpad ***/
	strcpy (target->paddevname, "/dev/pad00");
 /***joy_x11_opts***/
	strcpy (target->x11joyname, " ");
 /***Xinput***/
	strcpy (target->XInput_trackball1, " ");
	strcpy (target->XInput_trackball2, " ");
	strcpy (target->XInput_trackball3, " ");
	strcpy (target->XInput_trackball4, " ");
/* 	strcpy (target->XInput_joystick1, " "); */
/* 	strcpy (target->XInput_joystick2, " "); */
/* 	strcpy (target->XInput_joystick3, " "); */
/* 	strcpy (target->XInput_joystick4, " "); */

/*** Network Related ***/
	target->master = 2;
	strcpy (target->slave, "localhost");
	target->netmapkey = FALSE;
	target->parallelsync = FALSE;
	target->bind_flag = FALSE; 
	target->bind = 9000;
	target->statedebug = FALSE;
/*** Digital sound related ***/
	strcpy (target->dsp_plugin, " ");
/*list-dsp-plugins  */
	target->timer = FALSE;
				

/*** Sound mixer related ***/
	strcpy (target->sound_mixer_plugin, " ");
/* list-mixer-plugins */

 /***arts***/
	target->artsBufferTime = 10;
		
 /*** Alsa Sound System 0.5***/
/* list-alsa-cards */
	target->alsacard = 0;
	target->alsadevice = 0;
        	       
        /*** Alsa Sound System 0.9 ***/
/* list-alsa-cards
list-alsa-pcm	*/
	strcpy (target->alsa_pcm, "default");
	target->alsa_buffer = 250000;
        	       
        /*** QNX Audio related ***/
	target->audio_preferred = TRUE;
				

       /*** Display  Related ***/
/*	strcpy (target->display_plugin, " ");*/
/* list-display-plugins	*/
/*	target->dwidthscale = 1;
	target->dheightscale = 1;
	target->dscanlines = FALSE;
	target->display_aspect_ratio = 1.33;
	target->keep_aspect = TRUE;
*/
/* Misc */
	target->cheat = 1;
	target->keyboard_leds = 1;
	target->debug = 0;
	strcpy (target->debug_size, "640x480");
	target->log_flag = FALSE;
	strcpy (target->log, "xmame.log");
	target->cfgname_flag = FALSE;
	strcpy (target->cfgname, "");
	target->use_additional_options = FALSE;
	target->additional_options = g_strdup ("");
	
	target->skip_disclaimer = FALSE;
	target->skip_gameinfo = FALSE;
	target->bios = 0;
	
	
	options_file = fopen(filename, "r");
	g_free(filename);
	if (!options_file)
	{
		GXMAME_DEBUG("options file not found");
		return (FALSE);
	}
	
	
	while (fgets(line, 500, options_file))
	{
	if (!strncmp(line,"[Video]",7))
		{
			while (fgets(line, 500, options_file))
			{
				if(*line =='\r' ||*line =='\n')
					break;
				option_value = g_strsplit(g_strstrip(g_strdelimit(line,"[]\r\n",' ')),"=",0);
				if (option_value[1] != NULL)
				{	if (!strcmp(option_value[0],"bpp"))
					{
						target->bpp = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"arbheight"))
					{
						target->arbheight = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"heightscale"))
					{
						target->heightscale = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"widthscale"))
					{
						target->widthscale = atoi( option_value[1]);
/*					} else if (!strcmp(option_value[0],"scale"))
					{
						target->scale = g_ascii_strtod( option_value[1]);
*/					} else if (!strcmp(option_value[0],"effect"))
					{
						target->effect = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"autodouble"))
					{
						target->autodouble = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"dirty"))
					{
						target->dirty = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"scanlines"))
					{
						target->scanlines = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"artwork"))
					{
						target->artwork = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"use_backdrops"))
					{
						target->use_backdrops = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"use_overlays"))
					{
						target->use_overlays = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"use_bezels"))
					{
						target->use_bezels = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"artwork_crop"))
					{
						target->artwork_crop = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"artwork_resolution"))
					{
						target->artwork_resolution = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"frameskipper"))
					{
						target->frameskipper = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"throttle"))
					{
						target->throttle = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"sleepidle"))
					{
						target->sleepidle = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"autoframeskip"))
					{
						target->autoframeskip = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"maxautoframeskip"))
					{
						target->maxautoframeskip = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"frameskip"))
					{
						target->frameskip = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"brightness"))
					{
						target->brightness = g_ascii_strtod(option_value[1], NULL);
						/*  converted old percentage to dosmame brightness complient 0.5-2.0 range*/
						if ((target->brightness > 2) || (target->brightness < 0.5))
							target->brightness=(target->brightness)*(1.5/100)+0.5;
					} else if (!strcmp(option_value[0],"gamma_correction"))
					{
						target->gamma_correction = g_ascii_strtod(option_value[1], NULL);
					} else if (!strcmp(option_value[0],"norotate"))
					{
						target->norotate = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"ror"))
					{
						target->ror = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"rol"))
					{
						target->rol = atoi( option_value[1]);
					}  else if (!strcmp(option_value[0],"flipx"))
					{
						target->flipx = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"flipy"))
					{
						target->flipy = atoi( option_value[1]);
					}
				g_strfreev(option_value);
				}
			}
		}
		if (!strncmp(line,"[Vector]",8))
		{	
			while (fgets(line, 500, options_file))
			{
				if(*line =='\r' ||*line =='\n')
					break;
				option_value = g_strsplit(g_strstrip(g_strdelimit(line,"[]\r\n",' ')),"=",0);
				if (option_value[1] != NULL)
				{	if (!strcmp(option_value[0],"vectorres_flag"))
					{
						target->vectorres_flag = atoi(option_value[1]);
					} else if (!strcmp(option_value[0],"vectorres"))
					{
						g_snprintf(target->vectorres,10,option_value[1]);
					} else if (!strcmp(option_value[0],"beam"))
					{
						target->beam = g_ascii_strtod(option_value[1], NULL);
					} else if (!strcmp(option_value[0],"flicker"))
					{
						target->flicker = g_ascii_strtod(option_value[1], NULL);
					} else if (!strcmp(option_value[0],"antialias"))
					{
						target->antialias = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"translucency"))
					{
						target->translucency = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"intensity"))
					{
						target->intensity = g_ascii_strtod( option_value[1], NULL);
					}
				g_strfreev(option_value);
				}
			}
		}
		if (!strncmp(line,"[X11]",5))
		{	
			while (fgets(line, 500, options_file))
			{
				if(*line =='\r' ||*line =='\n')
					break;
				option_value = g_strsplit(g_strstrip(g_strdelimit(line,"[]\r\n",' ')),"=",0);
				if (option_value[1] != NULL)
				{	if (!strcmp(option_value[0],"x11_mode"))
					{
						target->x11_mode = g_ascii_strtod(option_value[1], NULL);
					}
				g_strfreev(option_value);
				}
			}
		}
		if (!strncmp(line,"[X11-window]",12))
		{	
			while (fgets(line, 500, options_file))
			{
				if(*line =='\r' ||*line =='\n')
					break;
				option_value = g_strsplit(g_strstrip(g_strdelimit(line,"[]\r\n",' ')),"=",0);
				if (option_value[1] != NULL)
				{	if (!strcmp(option_value[0],"cursor"))
					{
						target->cursor = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"mitshm"))
					{
						target->mitshm = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"xvext"))
					{
						target->xvext = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"vidix"))
					{
						target->vidix = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"yuv"))
					{
						target->yuv = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"yv12"))
					{
						target->yv12 = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"xvfullscreen"))
					{
						target->xvfullscreen = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"xsync"))
					{
						target->xsync = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"privatecmap"))
					{
						target->privatecmap = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"xil"))
					{
						target->xil = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"mtxil"))
					{
						target->mtxil = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"run_in_root_window"))
					{
						target->run_in_root_window = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"root_window_id"))
					{
						target->root_window_id = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"xvgeom_flag"))
					{
						target->xvgeom_flag = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"geometry"))
					{
						g_snprintf(target->geometry,10,option_value[1]);
					}
				g_strfreev(option_value);
				}
			}
		}
		if (!strncmp(line,"[X11-input]",11))
		{	
			while (fgets(line, 500, options_file))
			{
				if(*line =='\r' ||*line =='\n')
					break;
				option_value = g_strsplit(g_strstrip(g_strdelimit(line,"[]\r\n",' ')),"=",0);
				if (option_value[1] != NULL)
				{	if (!strcmp(option_value[0],"grabmouse"))
					{
						target->grabmouse = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"grabkeyboard"))
					{
						target->grabkeyboard = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"winkeys"))
					{
						target->winkeys = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"mapkey"))
					{
						g_snprintf(target->mapkey,14,option_value[1]);
					}
				g_strfreev(option_value);
				}
			}
		}
		if (!strncmp(line,"[OpenGL]",8))
		{	
			while (fgets(line, 500, options_file))
			{
				if(*line =='\r' ||*line =='\n')
					break;
				option_value = g_strsplit(g_strstrip(g_strdelimit(line,"[]\r\n",' ')),"=",0);
				if (option_value[1] != NULL)
				{	if (!strcmp(option_value[0],"glfullscreen"))
					{
						target->glfullscreen = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"gldblbuffer"))
					{
						target->gldblbuffer = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"gltexture_size"))
					{
						target->gltexture_size = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"glforceblitmode"))
					{
						target->glforceblitmode = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"glext78"))
					{
						target->glext78 = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"glbilinear"))
					{
						target->glbilinear = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"gldrawbitmap"))
					{
						target->gldrawbitmap = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"gldrawbitmapvec"))
					{
						target->gldrawbitmapvec = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"glcolormod"))
					{
						target->glcolormod = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"glbeam"))
					{
						target->glbeam = g_ascii_strtod(option_value[1], NULL);
					} else if (!strcmp(option_value[0],"glalphablending"))
					{
						target->glalphablending = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"glantialias"))
					{
						target->glantialias = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"glantialiasvec"))
					{
						target->glantialiasvec = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"gllibname"))
					{
						g_snprintf(target->gllibname,20,option_value[1]);
					} else if (!strcmp(option_value[0],"glulibname"))
					{
						g_snprintf(target->glulibname,20,option_value[1]);
					} else if (!strcmp(option_value[0],"cabview"))
					{
						target->cabview = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"cabinet"))
					{
						g_snprintf(target->cabinet,20,option_value[1]);
					} else if (!strcmp(option_value[0],"glres_flag"))
					{
						target->glres_flag = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"glres"))
					{
						g_snprintf(target->glres,10,option_value[1]);
					}
				g_strfreev(option_value);
				}
			}
		}
		if (!strncmp(line,"[SDL]",5))
		{	
			while (fgets(line, 500, options_file))
			{
				if(*line =='\r' ||*line =='\n')
					break;
				option_value = g_strsplit(g_strstrip(g_strdelimit(line,"[]\r\n",' ')),"=",0);
				if (option_value[1] != NULL)
				{	if (!strcmp(option_value[0],"fullscreen"))
					{
						target->fullscreen = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"sdl_auto_mode"))
					{
						target->sdl_auto_mode = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"modenumber"))
					{
						target->modenumber = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"sdlmapkey"))
					{
						g_snprintf(target->sdlmapkey,14,option_value[1]);
					}
				g_strfreev(option_value);
				}
			}
		}
		if (!strncmp(line,"[GGI]",5))
		{	
			while (fgets(line, 500, options_file))
			{
				if(*line =='\r' ||*line =='\n')
					break;
				option_value = g_strsplit(g_strstrip(g_strdelimit(line,"[]\r\n",' ')),"=",0);
				if (option_value[1] != NULL)
				{	if (!strcmp(option_value[0],"ggilinear"))
					{
						target->ggilinear = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"ggi_force_resolution"))
					{
						target->ggi_force_resolution = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"xres"))
					{
						target->xres = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"yres"))
					{
						target->yres = atoi( option_value[1]);
					}
				g_strfreev(option_value);
				}
			}
		}
		if (!strncmp(line,"[Svgalib]",9))
		{	
			while (fgets(line, 500, options_file))
			{
				if(*line =='\r' ||*line =='\n')
					break;
				option_value = g_strsplit(g_strstrip(g_strdelimit(line,"[]\r\n",' ')),"=",0);
				if (option_value[1] != NULL)
				{	if (!strcmp(option_value[0],"tweak"))
					{
						target->tweak = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"planar"))
					{
						target->planar = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"linear"))
					{
						target->linear = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"centerx"))
					{
						target->centerx = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"centery"))
					{
						target->centery = atoi( option_value[1]);
					}
				g_strfreev(option_value);
				}
			}
		}
		if (!strncmp(line,"[FX (Glide)]",12))
		{	
			while (fgets(line, 500, options_file))
			{
				if(*line =='\r' ||*line =='\n')
					break;
				option_value = g_strsplit(g_strstrip(g_strdelimit(line,"[]\r\n",' ')),"=",0);
				if (option_value[1] != NULL)
				{	if (!strcmp(option_value[0],"resolution"))
					{
						g_snprintf(target->resolution,10,option_value[1]);
					} else if (!strcmp(option_value[0],"fxgkeepaspect"))
					{
						target->fxgkeepaspect = atoi( option_value[1]);
					}
				g_strfreev(option_value);
				}
			}
		}
		if (!strncmp(line,"[Photon]",8))
		{	
			while (fgets(line, 500, options_file))
			{
				if(*line =='\r' ||*line =='\n')
					break;
				option_value = g_strsplit(g_strstrip(g_strdelimit(line,"[]\r\n",' ')),"=",0);
				if (option_value[1] != NULL)
				{	if (!strcmp(option_value[0],"render_mode"))
					{
						target->render_mode = atoi( option_value[1]);
					}
				g_strfreev(option_value);
				}
			}
		}
		if (!strncmp(line,"[Photon-window]",15))
		{	
			while (fgets(line, 500, options_file))
			{
				if(*line =='\r' ||*line =='\n')
					break;
				option_value = g_strsplit(g_strstrip(g_strdelimit(line,"[]\r\n",' ')),"=",0);
				if (option_value[1] != NULL)
				{	if (!strcmp(option_value[0],"phcursor"))
					{
						target->phcursor = atoi( option_value[1]);
					}
				g_strfreev(option_value);
				}
			}
		}
/*		if (!strncmp(line,"[Photon-input]",14))
		{	
			while (fgets(line, 500, options_file))
			{
				if(*line =='\r' ||*line =='\n')
					break;
				option_value = g_strsplit(g_strstrip(g_strdelimit(line,"[]\r\n",' ')),"=",0);
				if (option_value[1] != NULL)
				{	if (!strcmp(option_value[0],"phgrabmouse"))
					{
						target->phgrabmouse = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"phwinkeys"))
					{
						target->phwinkeys = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"phmapkey"))
					{
						g_snprintf(target->phmapkey,14,option_value[1]);
					} 
				g_strfreev(option_value);
				}
			}
		}*/
		if (!strncmp(line,"[Video Mode]",12))
		{	
			while (fgets(line, 500, options_file))
			{
				if(*line =='\r' ||*line =='\n')
					break;
				option_value = g_strsplit(g_strstrip(g_strdelimit(line,"[]\r\n",' ')),"=",0);
				if (option_value[1] != NULL)
				{	if (!strcmp(option_value[0],"keepaspect"))
					{
						target->keepaspect = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"displayaspectratio"))
					{
						target->displayaspectratio = g_ascii_strtod(option_value[1], NULL);
					} else if (!strcmp(option_value[0],"disablemode"))
					{
						g_snprintf(target->disablemode,15,option_value[1]);
					} 
				g_strfreev(option_value);
				}
			}
		}
		if (!strncmp(line,"[Sound]",7))
		{	
			while (fgets(line, 500, options_file))
			{
				if(*line =='\r' ||*line =='\n')
					break;
				option_value = g_strsplit(g_strstrip(g_strdelimit(line,"[]\r\n",' ')),"=",0);
				if (option_value[1] != NULL)
				{	if (!strcmp(option_value[0],"sound"))
					{
						target->sound = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"samples"))
					{
						target->samples = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"fakesound"))
					{
						target->fakesound = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"samplefre"))
					{
						target->samplefre = g_ascii_strtod( option_value[1], NULL);
					} else if (!strcmp(option_value[0],"bufsize"))
					{
						target->bufsize = g_ascii_strtod( option_value[1], NULL);
					} else if (!strcmp(option_value[0],"volume"))
					{
						target->volume = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"audiodevice"))
					{
						g_snprintf(target->audiodevice,20,option_value[1]);
					} else if (!strcmp(option_value[0],"mixerdevice"))
					{
						g_snprintf(target->mixerdevice,20,option_value[1]);
					} else if (!strcmp(option_value[0],"soundfile"))
					{
						g_snprintf(target->soundfile,20,option_value[1]);
					}
				g_strfreev(option_value);
				}
			}
		}
		if (!strncmp(line,"[Input]",7))
		{
			while (fgets(line, 500, options_file))
			{
				if(*line =='\r' ||*line =='\n')
					break;
				option_value = g_strsplit(g_strstrip(g_strdelimit(line,"[]\r\n",' ')),"=",0);
				if (option_value[1] != NULL)
				{	if (!strcmp(option_value[0],"joytype"))
					{
						target->joytype = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"analogstick"))
					{
						target->analogstick = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"mouse"))
					{
						target->mouse = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"hotrod"))
					{
						target->hotrod = g_ascii_strtod(option_value[1], NULL);
					} else if (!strcmp(option_value[0],"hotrodse"))
					{
						target->hotrodse = g_ascii_strtod(option_value[1], NULL);
					} else if (!strcmp(option_value[0],"usbpspad"))
					{
						target->usbpspad = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"rapidfire"))
					{
						target->rapidfire = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"keymap"))
					{
						g_snprintf(target->keymap,4,option_value[1]);
					} else if (!strcmp(option_value[0],"ugcicoin"))
					{
						target->ugcicoin = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"ctrlr"))
					{
						g_snprintf(target->ctrlr,20,option_value[1]);
					}

				g_strfreev(option_value);
				}
			}
		}
		if (!strncmp(line,"[i386]",6))
		{
			while (fgets(line, 500, options_file))
			{
				if(*line =='\r' ||*line =='\n')
					break;
				option_value = g_strsplit(g_strstrip(g_strdelimit(line,"[]\r\n",' ')),"=",0);
				if (option_value[1] != NULL)
				{	if (!strcmp(option_value[0],"joydevname"))
					{
						g_snprintf(target->joydevname,20,option_value[1]);
					}
				g_strfreev(option_value);
				}
			}
		}
		if (!strncmp(line,"[joy_pad_opts]",14))
		{	
			while (fgets(line, 500, options_file))
			{
				if(*line =='\r' ||*line =='\n')
					break;
				option_value = g_strsplit(g_strstrip(g_strdelimit(line,"[]\r\n",' ')),"=",0);
				if (option_value[1] != NULL)
				{	if (!strcmp(option_value[0],"paddevname"))
					{
						g_snprintf(target->paddevname,20,option_value[1]);
					}
				g_strfreev(option_value);
				}
			}
		}
		if (!strncmp(line,"[joy_x11_opts]",14))
		{	
			while (fgets(line, 500, options_file))
			{
				if(*line =='\r' ||*line =='\n')
					break;
				option_value = g_strsplit(g_strstrip(g_strdelimit(line,"[]\r\n",' ')),"=",0);
				if (option_value[1] != NULL)
				{	if (!strcmp(option_value[0],"x11joyname"))
					{
						g_snprintf(target->x11joyname,20,option_value[1]);
					}
				g_strfreev(option_value);
				}
			}
		}
		if (!strncmp(line,"[Xinput]",8))
		{	
			while (fgets(line, 500, options_file))
			{
				if(*line =='\r' ||*line =='\n')
					break;
				option_value = g_strsplit(g_strstrip(g_strdelimit(line,"[]\r\n",' ')),"=",0);
				if (option_value[1] != NULL)
				{	if (!strcmp(option_value[0],"XInput_trackball1"))
					{
						g_snprintf(target->XInput_trackball1,20,option_value[1]);
					} else if (!strcmp(option_value[0],"XInput_trackball2"))
					{
						g_snprintf(target->XInput_trackball2,20,option_value[1]);
					} else if (!strcmp(option_value[0],"XInput_trackball3"))
					{
						g_snprintf(target->XInput_trackball3,20,option_value[1]);
					} else if (!strcmp(option_value[0],"XInput_trackball4"))
					{
						g_snprintf(target->XInput_trackball4,20,option_value[1]);
/*					} else if (!strcmp(option_value[0],"XInput_joystick1"))
					{
						g_snprintf(target->XInput_joystick1,20,option_value[1]);
					} else if (!strcmp(option_value[0],"XInput_joystick2"))
					{
						g_snprintf(target->XInput_joystick2,20,option_value[1]);
					} else if (!strcmp(option_value[0],"XInput_joystick3"))
					{
						g_snprintf(target->XInput_joystick3,20,option_value[1]);
					} else if (!strcmp(option_value[0],"XInput_joystick4"))
					{
						g_snprintf(target->XInput_joystick4,20,option_value[1]);
*/					}
				g_strfreev(option_value);
				}
			}
		}
		if (!strncmp(line,"[Network]",9))
		{
			while (fgets(line, 500, options_file))
			{
				if(*line =='\r' ||*line =='\n')
					break;
				option_value = g_strsplit(g_strstrip(g_strdelimit(line,"[]\r\n",' ')),"=",0);
				if (option_value[1] != NULL)
				{	if (!strcmp(option_value[0],"network_flag"))
					{
						target->network_flag = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"master"))
					{
						target->master = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"slave"))
					{
						g_snprintf(target->slave,20,option_value[1]);
					} else if (!strcmp(option_value[0],"netmapkey"))
					{
						target->netmapkey = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"parallelsync"))
					{
						target->parallelsync = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"bind_flag"))
					{
						target->bind_flag = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"bind"))
					{
						target->bind = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"statedebug"))
					{
						target->statedebug = atoi( option_value[1]);
					} 
				g_strfreev(option_value);
				}
			}
		}
		if (!strncmp(line,"[Digital sound]",15))
		{
			while (fgets(line, 500, options_file))
			{
				if(*line =='\r' ||*line =='\n')
					break;
				option_value = g_strsplit(g_strstrip(g_strdelimit(line,"[]\r\n",' ')),"=",0);
				if (option_value[1] != NULL)
				{	if (!strcmp(option_value[0],"dsp_plugin"))
					{
						g_snprintf(target->dsp_plugin,20,option_value[1]);
					} else if (!strcmp(option_value[0],"timer"))
					{
						target->timer = atoi( option_value[1]);
					} 
				g_strfreev(option_value);
				}
			}
		}
		if (!strncmp(line,"[Sound mixer]",13))
		{
			while (fgets(line, 500, options_file))
			{
				if(*line =='\r' ||*line =='\n')
					break;
				option_value = g_strsplit(g_strstrip(g_strdelimit(line,"[]\r\n",' ')),"=",0);
				if (option_value[1] != NULL)
				{
					if (!strcmp(option_value[0],"sound_mixer_plugin"))
					{
						g_snprintf(target->sound_mixer_plugin,20,option_value[1]);
					} 
				g_strfreev(option_value);
				}
			}
		}
		if (!strncmp(line,"[arts]",6))
		{
			while (fgets(line, 500, options_file))
			{
				if(*line =='\r' ||*line =='\n')
					break;
				option_value = g_strsplit(g_strstrip(g_strdelimit(line,"[]\r\n",' ')),"=",0);
				if (option_value[1] != NULL)
				{	if (!strcmp(option_value[0],"artsBufferTime"))
					{
						target->artsBufferTime = atoi( option_value[1]);
					} 
				g_strfreev(option_value);
				}
			}
		}
		if (!strncmp(line,"[Alsa Sound System 0.5]",23))
		{
			while (fgets(line, 500, options_file))
			{
				if(*line =='\r' ||*line =='\n')
					break;
				option_value = g_strsplit(g_strstrip(g_strdelimit(line,"[]\r\n",' ')),"=",0);
				if (option_value[1] != NULL)
				{	if (!strcmp(option_value[0],"alsacard"))
					{
						target->alsacard = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"alsadevice"))
					{
						target->alsadevice = atoi( option_value[1]);
					}
				g_strfreev(option_value);
				}
			}
		}
		if (!strncmp(line,"[Alsa Sound System 0.9]",23))
		{	
			while (fgets(line, 500, options_file))
			{
				if(*line =='\r' ||*line =='\n')
					break;
				option_value = g_strsplit(g_strstrip(g_strdelimit(line,"[]\r\n",' ')),"=",0);
				if (option_value[1] != NULL)
				{	if (!strcmp(option_value[0],"alsa_pcm"))
					{
						g_snprintf(target->alsa_pcm,20,option_value[1]);
					} else if (!strcmp(option_value[0],"alsa_buffer"))
					{
						target->alsa_buffer = atoi( option_value[1]);
					}
				g_strfreev(option_value);
				}
			}
		}
		if (!strncmp(line,"[QNX Audio]",11))
		{	
			while (fgets(line, 500, options_file))
			{
				if(*line =='\r' ||*line =='\n')
					break;
				option_value = g_strsplit(g_strstrip(g_strdelimit(line,"[]\r\n",' ')),"=",0);
				if (option_value[1] != NULL)
				{	if (!strcmp(option_value[0],"audio_preferred"))
					{
						target->audio_preferred = atoi( option_value[1]);
					}
				g_strfreev(option_value);
				}
			}
		}
/*		if (!strncmp(line,"[Display]",9))
		{	
			while (fgets(line, 500, options_file))
			{
				if(*line =='\r' ||*line =='\n')
					break;
				option_value = g_strsplit(g_strstrip(g_strdelimit(line,"[]\r\n",' ')),"=",0);
				if (option_value[1] != NULL)
				{	if (!strcmp(option_value[0],"display_plugin"))
					{
						g_snprintf(target->display_plugin,20,option_value[1]);
					} else if (!strcmp(option_value[0],"dwidthscale"))
					{
						target->dwidthscale = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"dheightscale"))
					{
						target->dheightscale = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"dscanlines"))
					{
						target->dscanlines = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"display_aspect_ratio"))
					{
						target->display_aspect_ratio = g_ascii_strtod(option_value[1], NULL);
					} else if (!strcmp(option_value[0],"keep_aspect"))
					{
						target->keep_aspect = atoi( option_value[1]);
					}
				g_strfreev(option_value);
				}
			}
		}*/
		if (!strncmp(line,"[Misc]",6))
		{	
			while (fgets(line, 500, options_file))
			{
				if(*line =='\r' ||*line =='\n')
					break;
				option_value = g_strsplit(g_strstrip(g_strdelimit(line,"[]\r\n",' ')),"=",0);
				if (option_value[1] != NULL)
				{	if (!strcmp(option_value[0],"cheat"))
					{
						target->cheat = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"keyboard_leds"))
					{
						target->keyboard_leds = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"debug"))
					{
						target->debug = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"debug_size"))
					{
						g_snprintf(target->debug_size,10,option_value[1]);
					} else if (!strcmp(option_value[0],"log_flag"))
					{
						target->log_flag = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"log"))
					{
						g_snprintf(target->log,20,option_value[1]);
					} else if (!strcmp(option_value[0],"cfgname_flag"))
					{
						target->cfgname_flag = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"cfgname"))
					{
						g_snprintf(target->cfgname,20,option_value[1]);
					} else if (!strcmp(option_value[0],"use_additional_options"))
					{
						target->use_additional_options = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"additional_options"))
					{
						g_free(target->additional_options);
						target->additional_options = g_strdup (option_value[1]);
					} else if (!strcmp(option_value[0],"skip_disclaimer"))
					{
						target->skip_disclaimer = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"skip_gameinfo"))
					{
						target->skip_gameinfo = atoi( option_value[1]);
					} else if (!strcmp(option_value[0],"bios"))
					{
						target->bios = atoi( option_value[1]);
					}
				g_strfreev(option_value);
				}
			}
		}
	}

	fclose(options_file);
	
	/* we should not allow xv vidix and DGA at the same time, priority to older (DGA) */
	if (target->x11_mode == 1)
	{
		target->xvext = FALSE;
		target->vidix = FALSE;
	} else if (target->xvext == 1)
		target->vidix = FALSE;
	/* select only one if YUV and YV12 made are both valid */
	if (target->yuv == TRUE)
		target->yv12 = FALSE;

	/* if ctrlr is not available in the list then -> None */
	ctrlr_list = get_ctrlr_list ();
	for(my_list = g_list_first(ctrlr_list);
			(my_list != NULL);
			my_list = g_list_next(my_list))
	{
		if (strncmp(my_list->data,target->ctrlr,20)==0)
			break;
	}
	if (my_list==NULL)
		strcpy (target->ctrlr, "None");
	g_list_free(ctrlr_list);

	return TRUE;
}

gboolean save_options(RomEntry *rom)
{
	FILE *options_file=NULL;
	gchar *filename;
	xmame_game_options *target;
	gchar double_buffer[FLOAT_BUF_SIZE];
	
	if (rom==NULL)
	{
		target = &default_options;
		filename = g_build_filename(g_get_home_dir(), ".gxmame", "options", "default", NULL);	
	}
	else
	{
		/* no game preferences to save */
		if (!game_options)
			return (TRUE);
		filename = g_build_filename(g_get_home_dir(), ".gxmame", "options", rom->romname, NULL);	
		target = game_options;
	}
	
	options_file = fopen(filename, "w");	
	g_free(filename);
	if (!options_file)
	{
		GXMAME_DEBUG("unable to write %s option", (rom)?(rom->romname):"default");
		return FALSE;
	}

	fputs ("[Video]\n",options_file);
	fprintf (options_file, "bpp=%i\n", target->bpp);
	fprintf (options_file, "arbheight=%i\n", target->arbheight);
	fprintf (options_file, "heightscale=%i\n", target->heightscale);
	fprintf (options_file, "widthscale=%i\n",target->widthscale);
/*	fprintf (options_file, "scale=%s\n", my_dtostr(double_buffer, target->scale)); */
	fprintf (options_file, "effect=%i\n", target->effect);
	fprintf (options_file, "autodouble=%i\n", target->autodouble);
	fprintf (options_file, "dirty=%i\n", target->dirty);
	fprintf (options_file, "scanlines=%i\n", target->scanlines);
	fprintf (options_file, "artwork=%i\n", target->artwork);
	fprintf (options_file, "use_backdrops=%i\n", target->use_backdrops);
	fprintf (options_file, "use_overlays=%i\n", target->use_overlays);
	fprintf (options_file, "use_bezels=%i\n", target->use_bezels);
	fprintf (options_file, "artwork_crop=%i\n", target->artwork_crop);
	fprintf (options_file, "artwork_resolution=%i\n",target->artwork_resolution);
	fprintf (options_file, "frameskipper=%i\n",target->frameskipper);
	fprintf (options_file, "throttle=%i\n",target->throttle);
	fprintf (options_file, "sleepidle=%i\n",target->sleepidle);
	fprintf (options_file, "autoframeskip=%i\n",target->autoframeskip);
	fprintf (options_file, "maxautoframeskip=%i\n",target->maxautoframeskip);
	fprintf (options_file, "frameskip=%i\n",target->frameskip);
	fprintf (options_file, "brightness=%s\n", my_dtostr(double_buffer, target->brightness));
	fprintf (options_file, "gamma_correction=%s\n", my_dtostr(double_buffer, target->gamma_correction));
	fprintf (options_file, "norotate=%i\n",target->norotate);
	fprintf (options_file, "ror=%i\n",target->ror);
	fprintf (options_file, "rol=%i\n",target->rol);
	fprintf (options_file, "flipx=%i\n",target->flipx);
	fprintf (options_file, "flipy=%i\n",target->flipy);

	fputs ("\n[Vector]\n",options_file);
	fprintf (options_file, "vectorres_flag=%i\n",target->vectorres_flag);
	fprintf (options_file, "vectorres=%s\n",target->vectorres);
	fprintf (options_file, "beam=%s\n", my_dtostr(double_buffer, target->beam));
	fprintf (options_file, "flicker=%s\n", my_dtostr(double_buffer, target->flicker));
	fprintf (options_file, "antialias=%i\n",target->antialias);
	fprintf (options_file, "translucency=%i\n",target->translucency);
	fprintf (options_file, "intensity=%s\n", my_dtostr(double_buffer, target->intensity));
	
	fputs ("\n[X11]\n",options_file);
	fprintf (options_file, "x11_mode=%i\n",target->x11_mode);
	
	fputs ("\n[X11-window]\n",options_file);
	fprintf (options_file, "cursor=%i\n",target->cursor);
	
	fprintf (options_file, "mitshm=%i\n",target->mitshm);
	fprintf (options_file, "xvext=%i\n",target->xvext);
	fprintf (options_file, "vidix=%i\n",target->vidix);
	fprintf (options_file, "yuv=%i\n",target->yuv);
	fprintf (options_file, "yv12=%i\n",target->yv12);
	fprintf (options_file, "xvfullscreen=%i\n",target->xvfullscreen);
	fprintf (options_file, "xsync=%i\n",target->xsync);
	fprintf (options_file, "privatecmap=%i\n",target->privatecmap);
	fprintf (options_file, "xil=%i\n",target->xil);
	fprintf (options_file, "mtxil=%i\n",target->mtxil);
	fprintf (options_file, "run_in_root_window=%i\n",target->run_in_root_window);
	fprintf (options_file, "root_window_id=%i\n",target->root_window_id);
	fprintf (options_file, "xvgeom_flag=%i\n",target->xvgeom_flag);
	fprintf (options_file, "geometry=%s\n",target->geometry);
	
	fputs ("\n[X11-input]\n",options_file);
	fprintf (options_file, "grabmouse=%i\n",target->grabmouse);
	fprintf (options_file, "grabkeyboard=%i\n",target->grabkeyboard);
	fprintf (options_file, "winkeys=%i\n",target->winkeys);
	fprintf (options_file, "mapkey=%s\n",target->mapkey);

	fputs ("\n[OpenGL]\n",options_file);
	fprintf (options_file, "glfullscreen=%i\n",target->glfullscreen);
	fprintf (options_file, "gldblbuffer=%i\n",target->gldblbuffer);
	fprintf (options_file, "gltexture_size=%i\n",target->gltexture_size);
	fprintf (options_file, "glforceblitmode=%i\n",target->glforceblitmode);
	fprintf (options_file, "glext78=%i\n",target->glext78);
	fprintf (options_file, "glbilinear=%i\n",target->glbilinear);
	fprintf (options_file, "gldrawbitmap=%i\n",target->gldrawbitmap);
	fprintf (options_file, "gldrawbitmapvec=%i\n",target->gldrawbitmapvec);
	fprintf (options_file, "glcolormod=%i\n",target->glcolormod);
/*	fprintf (options_file, "glbeam=%s\n", my_dtostr(double_buffer, target->glbeam)); */
	fprintf (options_file, "glalphablending=%i\n",target->glalphablending);
	fprintf (options_file, "glantialias=%i\n",target->glantialias);
	fprintf (options_file, "glantialiasvec=%i\n",target->glantialiasvec);
	fprintf (options_file, "gllibname=%s\n",target->gllibname);
	fprintf (options_file, "glulibname=%s\n",target->glulibname);
	fprintf (options_file, "cabview=%i\n",target->cabview);
	fprintf (options_file, "cabinet=%s\n",target->cabinet);
	fprintf (options_file, "glres_flag=%i\n",target->glres_flag);
	fprintf (options_file, "glres=%s\n",target->glres);

	fputs ("\n[SDL]\n",options_file);
	fprintf (options_file, "fullscreen=%i\n",target->fullscreen);
	fprintf (options_file, "sdl_auto_mode=%i\n",target->sdl_auto_mode);
	fprintf (options_file, "modenumber=%i\n",target->modenumber);
	fprintf (options_file, "sdlmapkey=%s\n",target->sdlmapkey);
	
	fputs ("\n[GGI]\n",options_file);
	fprintf (options_file, "ggilinear=%i\n",target->ggilinear);
	fprintf (options_file, "ggi_force_resolution=%i\n",target->ggi_force_resolution);
	fprintf (options_file, "xres=%i\n",target->xres);
	fprintf (options_file, "yres=%i\n",target->yres);
	
	fputs ("\n[Svgalib]\n",options_file);
	fprintf (options_file, "tweak=%i\n",target->tweak);
	fprintf (options_file, "planar=%i\n",target->planar);
	fprintf (options_file, "linear=%i\n",target->linear);
	fprintf (options_file, "centerx=%i\n",target->centerx);
	fprintf (options_file, "centery=%i\n",target->centery);

	fputs ("\n[FX (Glide)]\n",options_file);
	fprintf (options_file, "resolution=%s\n",target->resolution);
	fprintf (options_file, "fxgkeepaspect=%i\n",target->fxgkeepaspect);
	
	fputs ("\n[Photon]\n",options_file);
	fprintf (options_file, "render_mode=%i\n",target->render_mode);
	
	fputs ("\n[Photon-window]\n",options_file);
	fprintf (options_file, "phcursor=%i\n",target->phcursor);
	
/*	fputs ("\n[Photon-input]\n",options_file);
	fprintf (options_file, "phgrabmouse=%i\n",target->phgrabmouse);
	fprintf (options_file, "phwinkeys=%i\n",target->phwinkeys);
	fprintf (options_file, "phmapkey=%s\n",target->phmapkey);
*/	
	fputs ("\n[Video Mode]\n",options_file);
	fprintf (options_file, "keepaspect=%i\n",target->keepaspect);
	fprintf (options_file, "displayaspectratio=%s\n",
		my_dtostr(double_buffer, target->displayaspectratio));
	fprintf (options_file, "disablemode=%s\n",target->disablemode);
	
	fputs ("\n[Sound]\n",options_file);
	fprintf (options_file, "sound=%i\n",target->sound);
	fprintf (options_file, "samples=%i\n",target->samples);
	fprintf (options_file, "fakesound=%i\n",target->fakesound);
	fprintf (options_file, "samplefre=%i\n",target->samplefre);
	fprintf (options_file, "bufsize=%s\n", my_dtostr(double_buffer, target->bufsize));
	fprintf (options_file, "volume=%i\n",target->volume);
	fprintf (options_file, "audiodevice=%s\n",target->audiodevice);
	fprintf (options_file, "mixerdevice=%s\n",target->mixerdevice);
	fprintf (options_file, "soundfile=%s\n",target->soundfile);
	
	fputs ("\n[Input]\n",options_file);
	fprintf (options_file, "joytype=%i\n",target->joytype);
	fprintf (options_file, "analogstick=%i\n",target->analogstick);
	fprintf (options_file, "mouse=%i\n",target->mouse);
	fprintf (options_file, "hotrod=%i\n",target->hotrod);
	fprintf (options_file, "hotrodse=%i\n",target->hotrodse);
	fprintf (options_file, "usbpspad=%i\n",target->usbpspad);
	fprintf (options_file, "rapidfire=%i\n",target->rapidfire);
	fprintf (options_file, "ugcicoin=%i\n",target->ugcicoin);
	fprintf (options_file, "keymap=%s\n",target->keymap);
	fprintf (options_file, "ctrlr=%s\n",target->ctrlr);

	fputs ("\n[i386]\n",options_file);
	fprintf (options_file, "joydevname=%s\n",target->joydevname);
	
	fputs ("\n[joy_pad_opts]\n",options_file);
	fprintf (options_file, "paddevname=%s\n",target->paddevname);
	
	fputs ("\n[joy_x11_opts]\n",options_file);
	fprintf (options_file, "x11joyname=%s\n",target->x11joyname);
	
	fputs ("\n[Xinput]\n",options_file);
	fprintf (options_file, "XInput_trackball1=%s\n",target->XInput_trackball1);
	fprintf (options_file, "XInput_trackball2=%s\n",target->XInput_trackball2);
	fprintf (options_file, "XInput_trackball3=%s\n",target->XInput_trackball3);
	fprintf (options_file, "XInput_trackball4=%s\n",target->XInput_trackball4);
/*	fprintf (options_file, "XInput_joystick1=%s\n",target->XInput_joystick1);
	fprintf (options_file, "XInput_joystick2=%s\n",target->XInput_joystick2);
	fprintf (options_file, "XInput_joystick3=%s\n",target->XInput_joystick3);
	fprintf (options_file, "XInput_joystick4=%s\n",target->XInput_joystick4);
*/
	
	fputs ("\n[Network]\n",options_file);
	fprintf (options_file, "network_flag=%i\n",target->network_flag);
	fprintf (options_file, "master=%i\n",target->master);
	fprintf (options_file, "slave=%s\n",target->slave);
	fprintf (options_file, "netmapkey=%i\n",target->netmapkey);
	fprintf (options_file, "parallelsync=%i\n",target->parallelsync);
	fprintf (options_file, "bind_flag=%i\n",target->bind_flag);
	fprintf (options_file, "bind=%i\n",target->bind);
	fprintf (options_file, "statedebug=%i\n",target->statedebug);
	
	fputs ("\n[Digital sound]\n",options_file);
	fprintf (options_file, "dsp_plugin=%s\n",target->dsp_plugin);
	fprintf (options_file, "timer=%i\n",target->timer);

	fputs ("\n[Sound mixer]\n",options_file);
	fprintf (options_file, "sound_mixer_plugin=%s\n",target->sound_mixer_plugin);

	fputs ("\n[arts]\n",options_file);
	fprintf (options_file, "artsBufferTime=%i\n",target->artsBufferTime);
	
	fputs ("\n[Alsa Sound System 0.5]\n",options_file);
	fprintf (options_file, "alsacard=%i\n",target->alsacard);
	fprintf (options_file, "alsadevice=%i\n",target->alsadevice);
	
	fputs ("\n[Alsa Sound System 0.9]\n",options_file);
	fprintf (options_file, "alsa_pcm=%s\n",target->alsa_pcm);
	fprintf (options_file, "alsa_buffer=%i\n",target->alsa_buffer);
	
	fputs ("\n[QNX Audio]\n",options_file);
	fprintf (options_file, "audio_preferred=%i\n",target->audio_preferred);
	
/*	fputs ("\n[Display]\n",options_file);
	fprintf (options_file, "display_plugin=%s\n",target->display_plugin);
	fprintf (options_file, "dwidthscale=%i\n",target->dwidthscale);
	fprintf (options_file, "dheightscale=%i\n",target->dheightscale);
	fprintf (options_file, "dscanlines=%i\n",target->dscanlines);
	fprintf (options_file, "display_aspect_ratio=%s\n", my_dtostr(double_buffer, target->aspect_ratio));
	fprintf (options_file, "keep_aspect=%i\n",target->keep_aspect);
*/	
	
	fputs ("\n[Misc]\n",options_file);
	fprintf (options_file, "cheat=%i\n",target->cheat);
	fprintf (options_file, "keyboard_leds=%i\n",target->keyboard_leds);
	fprintf (options_file, "debug=%i\n",target->debug);
	fprintf (options_file, "debug_size=%s\n",target->debug_size);
	fprintf (options_file, "log_flag=%i\n",target->log_flag);
	fprintf (options_file, "log=%s\n",target->log);
	fprintf (options_file, "cfgname_flag=%i\n",target->cfgname_flag);
	fprintf (options_file, "cfgname=%s\n",target->cfgname);
	fprintf (options_file, "use_additional_options=%i\n",target->use_additional_options);
	fprintf (options_file, "additional_options=%s\n",target->additional_options);
	fprintf (options_file, "skip_disclaimer=%i\n",target->skip_disclaimer);
	fprintf (options_file, "skip_gameinfo=%i\n",target->skip_gameinfo);
	fprintf (options_file, "bios=%i\n",target->bios);
	
	fclose(options_file);
	return TRUE;
}


void quick_check(void)
{
	static gboolean quick_check_running;
	GList *list_pointer;
	gchar *filename;
	gint nb_rom_not_checked;
	gfloat done;
	int i;
	RomEntry *rom;

	/*prevent user to launch several quick check at the same time */
	if(quick_check_running)
	{
		GXMAME_DEBUG("Quick check already running");
		return;
	}
	quick_check_running = 1;
		
	show_progress_bar();
	filename = g_malloc(200);
	nb_rom_not_checked = g_list_length(not_checked_list);
	g_message(_("Performing quick check, please wait:"));
	/* Disable the callback */
	g_signal_handlers_block_by_func (G_OBJECT (gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(main_gui.scrolled_window_games))),
					(gpointer)adjustement_scrolled, NULL);

	/* need to use the last or I can find the list anymore*/
	for (list_pointer = g_list_last(not_checked_list);(list_pointer != NULL);list_pointer = g_list_last(not_checked_list))
	{
		rom = (RomEntry *)list_pointer->data;
		/* Looking for roms */
		rom->has_roms = 0;
		for (i=0;gui_prefs.RomPath[i]!=NULL;i++)
		{
			g_snprintf(filename, 200, "%s/%s.zip",gui_prefs.RomPath[i],rom->romname);
	        if(g_file_test(filename, g_file_test(filename, G_FILE_TEST_EXISTS)))
	        {
	        	rom->has_roms = 1;
	        }
	        else
	        {	/*test of directories*/
				g_snprintf(filename, 200,"%s/%s",gui_prefs.RomPath[i],rom->romname);
				if (g_file_test(filename, G_FILE_TEST_IS_DIR))
				{
					rom->has_roms = 1;
				}
				else if(strcmp(rom->cloneof,"-"))
	        	{
	        		g_snprintf(filename, 200, "%s/%s.zip",gui_prefs.RomPath[i],rom->cloneof);
	        		if(g_file_test(filename, G_FILE_TEST_EXISTS))
	       			{
						rom->has_roms = 1;
	       			}
					else
					{
						g_snprintf(filename, 200,"%s/%s",gui_prefs.RomPath[i],rom->cloneof);
						if (g_file_test(filename, G_FILE_TEST_IS_DIR))
						{
							rom->has_roms = 1;
						}
					}
				}
	        }
			/* prevent to look into other folders if we already find the rom*/
			if(rom->has_roms) break;
		}

		/* Looking for samples */
		rom->has_samples = 0;

		if (rom->nb_samples > 0)
		{
			for (i=0;gui_prefs.SamplePath[i]!=NULL;i++)
			{
				g_snprintf(filename, 200, "%s/%s.zip",gui_prefs.SamplePath[i],rom->romname);
				if(g_file_test(filename, G_FILE_TEST_EXISTS))
		        {
		        	rom->has_samples = 1;
		        }
		        else
		        {	/*test of directories*/
					g_snprintf(filename, 200,"%s/%s",gui_prefs.SamplePath[i],rom->romname);
					if (g_file_test(filename, G_FILE_TEST_IS_DIR))
					{
						rom->has_samples = 1;
					}
					else if(strcmp(rom->sampleof,"-"))
			       	{
			       		g_snprintf(filename, 200, "%s/%s.zip",gui_prefs.SamplePath[i],rom->sampleof);
			       		if(g_file_test(filename, G_FILE_TEST_EXISTS))
			       		{
			      			rom->has_samples = 1;
			       		}
						else
						{
							g_snprintf(filename, 200,"%s/%s",gui_prefs.SamplePath[i],rom->sampleof);
							if (g_file_test(filename, G_FILE_TEST_IS_DIR))
							{
								rom->has_samples = 1;
							}
						}
					}
				}
				/* prevent to look into other folders if we already find the sample*/
				if(rom->has_samples) break;
			}
		}

		if (g_list_length(not_checked_list))
			not_checked_list = g_list_remove_link(not_checked_list, list_pointer);
		else{
			list_pointer->data = NULL;
			g_list_free_1(list_pointer);
		}

		done = (gfloat)(supported_games - g_list_length(not_checked_list))/(gfloat)supported_games;
		update_progress_bar(done);
		/* allow to refresh the display during the quick check */
		while(gtk_events_pending())
			gtk_main_iteration();
		/* if the game is in the list, update it */
		if (rom->is_in_list)
			update_game_in_list(rom->position);
	}
	g_free(filename);
	hide_progress_bar();
	quick_check_running=0;
	
	/* Re-Enable the callback */
	g_signal_handlers_unblock_by_func (G_OBJECT (gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(main_gui.scrolled_window_games))),
				(gpointer)adjustement_scrolled, NULL);		
}

GList *
get_ctrlr_list (void)
{
	GList *ctrlr_list=NULL;
	GDir *di;
	const gchar *dent;
	gchar *filename;

	GXMAME_DEBUG("Getting the ctrlr list. %s",gui_prefs.CtrlrDirectory);

	di = g_dir_open(gui_prefs.CtrlrDirectory, 0, NULL);
	if (!di)
	{
		GXMAME_DEBUG ("ERROR - unable to open folder %s", gui_prefs.CtrlrDirectory);
	}
	else
	{
		while ((dent = g_dir_read_name(di)))
		{
			filename = g_build_filename(gui_prefs.CtrlrDirectory, dent, NULL);
			if (g_file_test(filename, G_FILE_TEST_IS_DIR))
				ctrlr_list = g_list_append (ctrlr_list, g_strdup(dent));

			g_free(filename);
		}
		g_dir_close(di);
	}

	return ctrlr_list;
}
