/*
 * GXMame
 *
 * 2002, 2003, Stéphane Pontier <shadow_walker@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include "xmame_executable.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>
#include "options.h"

/* Names for different options */
const gchar *artwork_directory_option_name[] = {
	"artworkpath",
	"artwork_directory"
};
const gchar	*hiscore_directory_option_name[] = {
	"spooldir",
	"hiscore_directory"
};
const gchar	*snapshot_directory_option_name[] = {
	"screenshotdir",
	"snapshot_directory"
};
const gchar	*diff_directory_option_name[] = {
	NULL,
	"diffdir",
	"diff_directory"
};

const gchar *cheat_file_option_name[] = {
	"cheatfile",
	"cheat_file"
};

const gchar *hiscore_file_option_name[] = {
	"hiscorefile",
	"hiscore_file"
};

const gchar *mameinfo_file_option_name[] = {
	"mameinfofile",
	"mameinfo_file"
};

const gchar *history_file_option_name[] = {
	"historyfile",
	"history_file"
};

#define BUFFER_SIZE 1000

static GHashTable *xmame_executables_hash = NULL;

static XmameExecutable *
xmame_executable_new(const gchar* path);

static void
xmame_executable_free(XmameExecutable *exec);


void xmame_table_init()
{
	xmame_executables_hash =
		g_hash_table_new_full (g_str_hash, g_str_equal, g_free, (GDestroyNotify) xmame_executable_free);
}

void xmame_table_free(void)
{
	if (xmame_executables_hash)
		g_hash_table_destroy(xmame_executables_hash);
	xmame_executables_hash = NULL;
}

gint xmame_table_size()
{
	return g_hash_table_size(xmame_executables_hash);
}

XmameExecutable *
xmame_table_get(const gchar *path)
{
	if (!path)
		return NULL;

	return (XmameExecutable *) g_hash_table_lookup(xmame_executables_hash, path);
}


struct lookup {
	const gchar *name;
	XmameExecutable *result;
};

static void set_if_name_match(gpointer key, gpointer value, gpointer data)
{
	struct lookup *lookup_key;
	XmameExecutable *exec;
	gchar *test_string;

	lookup_key = (struct lookup *) data;
	exec = (XmameExecutable *) value;
	test_string = g_strdup_printf("%s (%s) %s", exec->name, exec->target, exec->version);

	if (!strcmp(lookup_key->name, test_string))
		lookup_key->result = exec;

	g_free(test_string);
}

XmameExecutable *
xmame_table_get_by_name(const gchar *name)
{
	struct lookup lookup_key;

	lookup_key.result = NULL;
	lookup_key.name = name;
	GXMAME_DEBUG("xmame_table: looking for %s", name);
	g_hash_table_foreach(xmame_executables_hash, set_if_name_match, &lookup_key);
	if (lookup_key.result)
		GXMAME_DEBUG("xmame_table: found %s", lookup_key.result->path);
	return lookup_key.result;
}

struct array_count {
	gint count;
	gchar **ptr;
};

static void add_path(gpointer key, gpointer value, gpointer data)
{
	struct array_count *array;

	array = (struct array_count *) data;
	array->ptr[array->count] = (gchar *) key;
	(array->count)++;
}

gchar **xmame_table_get_all()
{
	gchar **all_paths;
	struct array_count my_array;
	guint size = g_hash_table_size(xmame_executables_hash)+1;

	all_paths = g_malloc0(size * sizeof (gchar*));
	if (size == 1)
		return all_paths;

	my_array.count = 0;
	my_array.ptr = all_paths;

	g_hash_table_foreach(xmame_executables_hash, add_path, &my_array);

	return all_paths;
}

XmameExecutable *
xmame_table_add(const gchar *path)
{
	XmameExecutable * new_exec;
	new_exec = xmame_executable_new(path);

	if (!new_exec)
		return NULL;

	if(xmame_executable_set_version(new_exec))
	{
		g_hash_table_insert(xmame_executables_hash, g_strdup(new_exec->path), new_exec);
		GXMAME_DEBUG("xmame_table: Added \"%s\", Total=%d", new_exec->path, g_hash_table_size(xmame_executables_hash));
		return new_exec;
	}
	else
	{
		xmame_executable_free(new_exec);
		return NULL;
	}
}

/* Creates a new executable.
 * Path will be set to the absolute path for the executable.
 * Accepts absolute pathnames and program names in the PATH.
 *
 * If the executable is not valid it returns NULL.
*/
XmameExecutable *
xmame_executable_new(const gchar *path)
{
	XmameExecutable *xmame_exec;

	if(!path)
		return NULL;

	xmame_exec = g_malloc0(sizeof(XmameExecutable));
	xmame_exec->path = g_find_program_in_path(path);

	if (!xmame_exec->path)
	{
		gxmame_message(WARNING, NULL, _("%s is not a valid xmame executable"), path);
		g_free(xmame_exec);
		xmame_exec= NULL;
	}

	return xmame_exec;
}

void xmame_executable_free(XmameExecutable *exec)
{
	if (!exec)
		return;

	if (exec->name)
		g_free(exec->name);
	if (exec->version)
		g_free(exec->version);
	if (exec->path)
		g_free(exec->path);
	
	xmame_executable_free_options(exec);

	g_free(exec);
}

void xmame_executable_free_options(XmameExecutable *exec)
{
	if (exec->available_options)
	{
		g_free(exec->available_options);
		exec->available_options = NULL;
	}
	if (exec->dsp_plugin_hash_table)
	{
		g_hash_table_destroy(exec->dsp_plugin_hash_table);
		exec->dsp_plugin_hash_table = NULL;
	}
	if (exec->mixer_plugin_hash_table)
	{
		g_hash_table_destroy(exec->mixer_plugin_hash_table);
		exec->mixer_plugin_hash_table = NULL;
	}
	if (exec->display_plugins_hash_table)
	{
		g_hash_table_destroy(exec->display_plugins_hash_table);
		exec->display_plugins_hash_table = NULL;
	}
	if (exec->sdl_modes_hash_table)
	{
		g_hash_table_destroy(exec->sdl_modes_hash_table);
		exec->sdl_modes_hash_table = NULL;
	}
}

/* Sets the version information for the executable.
* This is called automatically for every executable
* added to the table so there is no need to call it.
*/
gboolean xmame_executable_set_version(XmameExecutable *exec)
{
	FILE *xmame_pipe;
	gchar *opt, *p;
	gchar line[BUFFER_SIZE];
	gchar *tmp_xmame_target,*tmp_xmame_version;
	gboolean xmame_ok = FALSE;

	if(!exec || !exec->path)
		return FALSE;

	if (access(exec->path, X_OK))
		return FALSE;

	if (exec->version)
	{
		g_free(exec->version);
		exec->version = NULL;
	}
	if (exec->target)
	{
		g_free(exec->target);
		exec->target = NULL;
	}
	if (exec->name)
	{
		g_free(exec->name);
		exec->name = NULL;
	}


	opt=g_strdup_printf("%s -version -noloadconfig 2>/dev/null", exec->path);
	GXMAME_DEBUG("Trying %s", opt);
	xmame_pipe = popen(opt, "r");
	g_free(opt);
	if (!xmame_pipe)
	{
		GXMAME_DEBUG("Could not execute.");
		return FALSE;
	}
	while(fgets(line, BUFFER_SIZE, xmame_pipe) && !xmame_ok)
	{
		GXMAME_DEBUG("Reading line %s", line);
		/* that should catch any info displayed before the version name */
		if(!strncmp(line,"info",4) )
			continue;
		/* try to find the '(' */
		for(p=line;(*p && (*p != '('));p++);
		if (!*p)
			continue;
		*(p-1)='\0';
		p++;
		tmp_xmame_target = p;

		/* now try to find the ')' */
		for(;(*p && (*p != ')'));p++);
		if (!*p)
			continue;
		*p='\0';
		p+=2;
		tmp_xmame_version= p;
		
		/* do I need that ? */
		for(;(*p && (*p != '\n'));p++);
		*p='\0';
		if (strncmp(tmp_xmame_version,"version",7))
			continue;
			
		GXMAME_DEBUG("checking xmame version: %s (%s) %s",line, tmp_xmame_target, tmp_xmame_version);
			
		exec->name = g_strdup(line);
		exec->target = g_strdup(tmp_xmame_target);
		exec->version = g_strdup(tmp_xmame_version);

		xmame_ok = TRUE;
	}
		
	pclose(xmame_pipe);
		
	if (xmame_ok)
	{
		GXMAME_DEBUG("name=%s. target=%s. version=%s.",
				exec->name,
				exec->target,
				exec->version);
	}

	return xmame_ok;	
}

xmame_available_options * xmame_executable_get_options(XmameExecutable *exec)
{
	FILE *xmame_pipe;
	gchar *opt, *name, *keyword, *p;
	gint i;
	gdouble version;
	gchar *version_str;
	xmame_available_options *available_options;
	gchar line[BUFFER_SIZE];

	if (!exec)
		return NULL;

	if (exec->available_options)
		return exec->available_options;

	exec->available_options = g_malloc(sizeof(xmame_available_options));

	available_options = exec->available_options;
	/* Set default options */
	available_options->debug = FALSE;
	available_options->use_backdrops = FALSE;
	available_options->use_overlays = FALSE;
	available_options->use_bezels = FALSE;
	available_options->artwork_crop = FALSE;
	available_options->artwork_resolution = FALSE;
	available_options->artsBufferTime = FALSE;
	available_options->list_display_plugins = FALSE;
	available_options->list_alsa_cards = FALSE;
	available_options->alsacard = FALSE;
	available_options->alsadevice = FALSE;
	available_options->list_alsa_pcm = FALSE;
	available_options->alsa_pcm = FALSE;
	available_options->alsa_buffer = FALSE;
	available_options->audio_preferred = FALSE;
	available_options->network = FALSE;
	available_options->parallelsync = FALSE;
	available_options->bind = FALSE;
	available_options->statedebug = FALSE;
	available_options->list_dsp_plugins = FALSE;
	available_options->list_mixer_plugins = FALSE;
	available_options->keyboard_leds = FALSE;
	available_options->dirty = FALSE;
	available_options->xvext = FALSE;
	available_options->vidix = FALSE;
	available_options->yuv = FALSE;
	available_options->yv12 = FALSE;
	available_options->mitshm = FALSE;
	available_options->XInput_trackball = FALSE;
	available_options->paddevname = FALSE;
	available_options->x11joyname = FALSE;
	available_options->joydevname = FALSE;
	available_options->video_mode = FALSE;
	available_options->fullscreen = FALSE;
	available_options->ctrlr = FALSE;
	available_options->cfgname = FALSE;
	available_options->grabmouse = FALSE;
	available_options->grabkeyboard = FALSE;
	available_options->listmodes = FALSE;
	/* new commandline options */
	available_options->artwork_directory_option = artwork_directory_option_name[0];
	available_options->hiscore_directory_option = hiscore_directory_option_name[0];
	available_options->snapshot_directory_option = snapshot_directory_option_name[0];
	available_options->diff_directory_option = diff_directory_option_name[0];
	available_options->cheat_file_option = cheat_file_option_name[0];
	available_options->hiscore_file_option = hiscore_file_option_name[0];
	available_options->history_file_option = history_file_option_name[0];
	available_options->mameinfo_file_option = mameinfo_file_option_name[0];

	available_options->gamma_correction = FALSE;
		
	available_options->InputDirectory = FALSE;
	available_options->NVRamDirectory = FALSE;
	available_options->MemCardDirectory = FALSE;
	available_options->ConfigDirectory = FALSE;
	available_options->StateDirectory = FALSE;
	available_options->CtrlrDirectory = FALSE;
	available_options->IniPath = FALSE;

	available_options->skip_disclaimer = FALSE;
	available_options->skip_gameinfo = FALSE;

	available_options->bios = FALSE;
	available_options->ugcicoin = FALSE;


	/* first get the version number to compare it to check SDL modes */
	version_str = exec->version;

	while(*version_str && !isdigit(*version_str))
		version_str++;

	version = g_ascii_strtod(version_str, NULL);

	GXMAME_DEBUG("xmame version: %f %s\n", version, exec->version);

	/* do not test anymore if the executable is valid since its already tested previously  */
	/* to be able to load the 0.68 options */
	if (version == 0.68)
		opt=g_strdup_printf("%s -showusage -noloadconfig 2>/dev/null", exec->path);
	else
		opt=g_strdup_printf("%s -help -noloadconfig 2>/dev/null", exec->path);
	xmame_pipe = popen(opt, "r");
	g_free(opt);
	if (!xmame_pipe)
		return available_options;

	GXMAME_DEBUG("checking xmame options");
	while (fgets(line, BUFFER_SIZE, xmame_pipe))
	{
		/* got an option */
		if (line[0] == '-')
		{
			p = line + 1;
			if (!strncmp(p,"[no]debug",9))
				available_options->debug = TRUE;
			else if (!strncmp(p,"[no]dirty",9))
				available_options->dirty = TRUE;
			else if (!strncmp(p,"[no]xvext",9))
				available_options->xvext = TRUE;
			else if (!strncmp(p,"[no]vidix",9))
				available_options->vidix = TRUE;
			else if (!strncmp(p,"[no]yuv",7))
				available_options->yuv = TRUE;
			else if (!strncmp(p,"[no]yv12",8))
				available_options->yv12 = TRUE;
			else if (!strncmp(p,"[no]mitshm",10))
				available_options->mitshm = TRUE;
			else if (!strncmp(p,"[no]fullscreen",14))
				available_options->fullscreen = TRUE;

			else if (!strncmp(p,"cfgname",7))
				available_options->cfgname = TRUE;

			else if (!strncmp(p,"[no]keyboard_leds",17))
				available_options->keyboard_leds = TRUE;
			else if (!strncmp(p,"[no]use_backdrops",17))
				available_options->use_backdrops = TRUE;
			else if (!strncmp(p,"[no]use_overlays",16))
				available_options->use_overlays = TRUE;
			else if (!strncmp(p,"[no]use_bezels",14))
				available_options->use_bezels = TRUE;
			else if (!strncmp(p,"[no]artwork_crop",16))
				available_options->artwork_crop = TRUE;
			else if (!strncmp(p,"artwork_resolution",18))
				available_options->artwork_resolution = TRUE;

			else if (!strncmp(p,"intensity",9))
				available_options->intensity = TRUE;

			else if (!strncmp(p,"master",6))
				available_options->network = TRUE;
			else if (!strncmp(p,"[no]parallelsync",16))
				available_options->parallelsync = TRUE;
			else if (!strncmp(p,"bind",4))
				available_options->bind = TRUE;
			else if (!strncmp(p,"[no]statedebug",14))
				available_options->statedebug = TRUE;

			else if (!strncmp(p,"artsBufferTime",14))
				available_options->artsBufferTime = TRUE;
			else if (!strncmp(p,"list-alsa_cards",15))
				available_options->list_alsa_cards = TRUE;
			else if (!strncmp(p,"alsacard",8))
				available_options->alsacard = TRUE;
			else if (!strncmp(p,"alsadevice",10))
				available_options->alsadevice = TRUE;
			else if (!strncmp(p,"list-alsa-pcm",13))
				available_options->list_alsa_pcm = TRUE;
			else if (!strncmp(p,"alsa-pcm",8))
				available_options->alsa_pcm = TRUE;
			else if (!strncmp(p,"alsa-buffer",11))
				available_options->alsa_buffer = TRUE;
			else if (!strncmp(p,"[no]audio-preferred",19))
				available_options->audio_preferred = TRUE;

			else if (!strncmp(p,"list-display-plugins",20))
				available_options->list_display_plugins = TRUE;
			else if (!strncmp(p,"list-dsp-plugins",16))
				available_options->list_dsp_plugins = TRUE;
			else if (!strncmp(p,"list-mixer-plugins",18))
				available_options->list_mixer_plugins = TRUE;

			else if (!strncmp(p,"XInput_trackball",16))
				available_options->XInput_trackball = TRUE;
			else if (!strncmp(p,"paddevname",10))
				available_options->paddevname = TRUE;
			else if (!strncmp(p,"x11joyname",10))
				available_options->x11joyname = TRUE;
			else if (!strncmp(p,"joydevname",10))
				available_options->joydevname = TRUE;
			else if (!strncmp(p,"[no]grabmouse",13))
				available_options->grabmouse = TRUE;
			else if (!strncmp(p,"[no]grabkeyboard",16))
				available_options->grabkeyboard = TRUE;
			else if (!strncmp(p,"listmodes",9))
				available_options->listmodes = TRUE;

			else if (!strncmp(p,"diffdir",7))
				available_options->diff_directory_option = diff_directory_option_name[1];
			else if (!strncmp(p,"diff_directory",14))
				available_options->diff_directory_option = diff_directory_option_name[2];

			/* new commandline options */
			else if (!strncmp(p,"artwork_directory",17))
				available_options->artwork_directory_option = artwork_directory_option_name[1];
			else if (!strncmp(p,"hiscore_directory",17))
				available_options->hiscore_directory_option = hiscore_directory_option_name[1];
			else if (!strncmp(p,"snapshot_directory",18))
				available_options->snapshot_directory_option = snapshot_directory_option_name[1];
			
			else if (!strncmp(p,"cheat_file",10))
				available_options->cheat_file_option = cheat_file_option_name[1];
			else if (!strncmp(p,"hiscore_file",12))
				available_options->hiscore_file_option = hiscore_file_option_name[1];
			else if (!strncmp(p,"history_file",12))
				available_options->history_file_option = history_file_option_name[1];
			else if (!strncmp(p,"mameinfo_file",13))
				available_options->mameinfo_file_option = mameinfo_file_option_name[1];

			else if (!strncmp(p,"gamma-correction",16))
				available_options->gamma_correction = TRUE;


			else if (!strncmp(p,"input_directory",15))
				available_options->InputDirectory = TRUE;
			else if (!strncmp(p,"nvram_directory",14))
				available_options->NVRamDirectory = TRUE;
			else if (!strncmp(p,"memcard_directory",17))
				available_options->MemCardDirectory = TRUE;
			else if (!strncmp(p,"cfg_directory",13))
				available_options->ConfigDirectory = TRUE;
			else if (!strncmp(p,"state_directory",15))
				available_options->StateDirectory = TRUE;
			else if (!strncmp(p,"ctrlr_directory",15))
				available_options->CtrlrDirectory = TRUE;
			else if (!strncmp(p,"ctrlr",5))
				available_options->ctrlr = TRUE;

			else if (!strncmp(p,"inipath",7))
				available_options->IniPath = TRUE;

			else if (!strncmp(p,"[no]skip_disclaimer", 19))
				available_options->skip_disclaimer = TRUE;
			else if (!strncmp(p,"[no]skip_gameinfo",17))
				available_options->skip_gameinfo = TRUE;
	
			else if (!strncmp(p,"bios",4))
				available_options->bios = TRUE;
			else if (!strncmp(p,"[no]ugcicoin",12))
				available_options->ugcicoin = TRUE;
		}
		else if (line[0] == '*')
		{
			if (!strncmp(line,"*** Video Mode Selection Related ***",36))
				available_options->video_mode = TRUE;
		}
	}
	pclose(xmame_pipe);

	/* list dsp plugins */
	if (available_options->list_dsp_plugins)
	{
		gchar *key,*value;
		opt=g_strdup_printf("%s -list-dsp-plugins -noloadconfig 2>/dev/null", exec->path);
		xmame_pipe = popen(opt, "r");
		g_free(opt);
		if (exec->dsp_plugin_hash_table)
			g_hash_table_destroy(exec->dsp_plugin_hash_table);

		exec->dsp_plugin_hash_table = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, g_free);
		GXMAME_DEBUG("getting xmame dsp plugins");
		/* header : Digital sound plugins:
		   maybe should check it */
		fgets(line, BUFFER_SIZE, xmame_pipe);
		/* first empty line */
		fgets(line, BUFFER_SIZE, xmame_pipe);
		while (fgets(line, BUFFER_SIZE, xmame_pipe))
		{
			/* prevent to get other things as plugins (Open GL copyright line) */
			if (line[0] == '\n')
				break;
			else
			{
				/* remove traling \n */
				line[strlen(line) - 1] = 0;
				/* find the end of the keyword */
				for (i=1,keyword = p = line ;(*p && (*p++ != ' '));i++);
				keyword[i] = '\0';
				/* find the begining of the plugin complete name */
				for (i=0, name = ++p;(*p && (*p++ == ' '));i++);
				g_strstrip(name);
				GXMAME_DEBUG("plugin found: %s, code (%s)",name,keyword);
				key=g_strndup(keyword,strlen(keyword)-1);
				value=g_strdup(name);
				g_hash_table_insert(exec->dsp_plugin_hash_table,key,value);

			}
		}

		pclose(xmame_pipe);
	}

	/* list sound mixer plugins */
	if (available_options->list_mixer_plugins)
	{
		gchar *key,*value;
		opt=g_strdup_printf("%s -list-mixer-plugins -noloadconfig 2>/dev/null", exec->path);
		xmame_pipe = popen(opt, "r");
		g_free(opt);
		if (exec->mixer_plugin_hash_table)
			g_hash_table_destroy(exec->mixer_plugin_hash_table);

		exec->mixer_plugin_hash_table = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, g_free);
		GXMAME_DEBUG("getting xmame mixer plugins");
		/* header : Digital sound plugins:
		   maybe should check it */
		fgets(line, BUFFER_SIZE, xmame_pipe);
		/* first empty line */
		fgets(line, BUFFER_SIZE, xmame_pipe);
		while (fgets(line, BUFFER_SIZE, xmame_pipe))
		{
			/* prevent to get other things as plugins (Open GL copyright line) */
			if (line[0] == '\n')
				break;
			else
			{
				/* remove traling \n */
				line[strlen(line) - 1] = 0;
				/* find the end of the keyword */
				for (i=1,keyword = p = line ;(*p && (*p++ != ' '));i++);
				keyword[i] = '\0';
				/* find the begining of the plugin complete name */
				for (i=0, name = ++p;(*p && (*p++ == ' '));i++);
				g_strstrip(name);
				GXMAME_DEBUG("plugin found: %s, code (%s)",name,keyword);
				key=g_strndup(keyword,strlen(keyword)-1);
				value=g_strdup(name);
				g_hash_table_insert(exec->mixer_plugin_hash_table,key,value);
			}
		}
		pclose(xmame_pipe);
	}

	/* use the version number what type of brightness we use and the SDL list modes */
	if (version >= 0.66)
	{
		available_options->dos_brightness = TRUE;
	}
	else
	{
		available_options->dos_brightness = FALSE;
	}

	/* version 0.72.1 introduces new syntax for the -x11-mode option */
	if (version >= 0.72)
	{
		available_options->newx11modeusage=TRUE;
		available_options->xvext=TRUE;
		available_options->fullscreen=TRUE;
	}
	else
	{
		available_options->newx11modeusage=FALSE;
		available_options->xvext = FALSE;
		available_options->fullscreen = FALSE;
	}

	/* version 0.74.1 introduces new syntax for -hotrod[se] options */
	if (version >= 0.74)
		available_options->newhotrodusage=TRUE;
	else
		available_options->newhotrodusage=FALSE;


	/* list available modes */
	if ((available_options->listmodes) && (version >= 0.61))
	{
		gchar *key,*value;
		opt=g_strdup_printf("%s -listmodes -noloadconfig 2>/dev/null", exec->path);
		xmame_pipe = popen(opt, "r");
		g_free(opt);
		if (exec->sdl_modes_hash_table)
			g_hash_table_destroy(exec->sdl_modes_hash_table);

		exec->sdl_modes_hash_table = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, g_free);
		GXMAME_DEBUG("getting xmame SDL mode");
		while (fgets(line, BUFFER_SIZE, xmame_pipe))
		{
			if(!strncmp(line,"Modes available:",16))
			{
				GXMAME_DEBUG("begin mode listing");
				while (fgets(line, BUFFER_SIZE, xmame_pipe))
				{
					if (line[0] == '\n')
						break;
					else
					{
						/* remove traling \n */
						line[strlen(line) - 1] = 0;
						/* find the end of the keyword */
						for (i=1,keyword = p = line ;(*p && (*p++ != ')'));i++);
						keyword[i] = '\0';
						g_strstrip(keyword);
						/* find the begining of the plugin complete name */
						name = p+6;
						g_strstrip(name);
						GXMAME_DEBUG("Resolution found: %s, code (%s)",name,keyword);
						key=g_strndup(keyword,strlen(keyword)-1);
						value=g_strdup(name);
						g_hash_table_insert(exec->sdl_modes_hash_table,key,value);

					}
				}
			}
		}
		pclose(xmame_pipe);
	}
	return available_options;
}		

