/*
 *  autostart.cpp - autostart KAlarm when session restoration is complete
 *  Program:  kalarmautostart
 *  Copyright © 2001,2008 by David Jarvie <software@astrojar.org.uk>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include <qtimer.h>

#include <kcmdlineargs.h>
#include <kaboutdata.h>
#include <klocale.h>
#include <kstddirs.h>
#include <kprocess.h>
#include <dcopclient.h>
#include <kdebug.h>

#include "autostart.moc"

// Number of seconds to wait before autostarting KAlarm.
// Allow plenty of time for session restoration to happen first.
static const int AUTOSTART_DELAY = 30;

#define PROGRAM_VERSION      "1.0"
#define PROGRAM_NAME "kalarmautostart"


static KCmdLineOptions options[] =
{
	{ "!+app", I18N_NOOP("Application to autostart"), 0L },
	{ "+[arg]", I18N_NOOP("Command line arguments"), 0L },
	{ 0L, 0L, 0L }
};


int main(int argc, char *argv[])
{
	KAboutData aboutData(PROGRAM_NAME, I18N_NOOP("KAlarm Autostart"),
		PROGRAM_VERSION, I18N_NOOP("KAlarm autostart at login"),
		KAboutData::License_GPL,
		"Copyright 2002,2008 David Jarvie", 0, "http://www.astrojar.org.uk/kalarm");
	aboutData.addAuthor("David Jarvie", I18N_NOOP("Maintainer"), "software@astrojar.org.uk");
	KGlobal::locale()->insertCatalogue("kalarm");

	KCmdLineArgs::init(argc, argv, &aboutData);
	KCmdLineArgs::addCmdLineOptions(options);

	AutostartApp app;
	return app.exec();
}



AutostartApp::AutostartApp()
	: KApplication(false, false)       // initialise as non-GUI application
{
	// Login session is starting up - need to wait for it to complete
	// in order to avoid starting the client before it is restored by
	// the session (where applicable).
	QTimer::singleShot(AUTOSTART_DELAY * 1000, this, SLOT(slotAutostart()));
}

void AutostartApp::slotAutostart()
{
	KCmdLineArgs* args = KCmdLineArgs::parsedArgs();
	if (args->count() <= 0)
		kdWarning(5900) << "No command line";
	else
	{
		QCString prog = args->arg(0);
		if (kapp->dcopClient()->isApplicationRegistered(prog))
			kdDebug(5900) << "KAlarm already running" << endl;
		else
		{
			QString exe = locate("exe", QString::fromLatin1(prog));
			if (exe.isEmpty())
				kdWarning(5900) << "Executable not found: " << prog << endl;
			else
			{
				kdDebug(5900) << "Starting " << prog << endl;
				KProcess proc;
				proc << exe;
				for (int i = 1;  i < args->count();  ++i)
					proc << QString::fromLatin1(args->arg(i));
				proc.start(KProcess::DontCare);
			}
		}
	}
	exit();
}
