/*
 *  ksystemtimezone.cpp  -  kdelibs system time zone classes for KDE 3
 *  Program:  kalarm
 *  Copyright © 2006-2008 by David Jarvie <software@astrojar.org.uk>
 */
/*
   This file is part of the KDE libraries
   Copyright (c) 2005-2008 David Jarvie <djarvie@kde.org>
   Copyright (c) 2005 S.R.Haque <srhaque@iee.org>.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License version 2 as published by the Free Software Foundation.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

// This file requires HAVE_STRUCT_TM_TM_ZONE to be defined if struct tm member tm_zone is available.
// This file requires HAVE_TM_GMTOFF to be defined if struct tm member tm_gmtoff is available.

#include "ksystemtimezone.h"

#include <config.h>

#include <climits>
#include <cstdlib>
#include <unistd.h>

#include <qfile.h>
#include <qfileinfo.h>
#include <qdir.h>
#include <qregexp.h>
#include <qstringlist.h>
#include <qtextstream.h>

#include <kglobal.h>
#include <klocale.h>
#include <kmdcodec.h>
#include <kstringhandler.h>
#include <ktempfile.h>
#include <kstaticdeleter.h>
#include <kdebug.h>
#include "ktzfiletimezone.h"


/* Return the offset to UTC in the current time zone at the specified UTC time.
 * The thread-safe function localtime_r() is used in preference if available.
 */
int gmtoff(time_t t)
{
#ifdef _POSIX_THREAD_SAFE_FUNCTIONS
    tm tmtime;
    if (!localtime_r(&t, &tmtime))
        return 0;
#ifdef HAVE_TM_GMTOFF
    return tmtime.tm_gmtoff;
#else
    int lwday = tmtime.tm_wday;
    int lt = 3600*tmtime.tm_hour + 60*tmtime.tm_min + tmtime.tm_sec;
    if (!gmtime_r(&t, &tmtime))
        return 0;
    int uwday = tmtime.tm_wday;
    int ut = 3600*tmtime.tm_hour + 60*tmtime.tm_min + tmtime.tm_sec;
#endif
#else
    tm *tmtime = localtime(&t);
    if (!tmtime)
        return 0;
#ifdef HAVE_TM_GMTOFF
    return tmtime->tm_gmtoff;
#else
    int lwday = tmtime->tm_wday;
    int lt = 3600*tmtime->tm_hour + 60*tmtime->tm_min + tmtime->tm_sec;
    tmtime = gmtime(&t);
    int uwday = tmtime->tm_wday;
    int ut = 3600*tmtime->tm_hour + 60*tmtime->tm_min + tmtime->tm_sec;
#endif
#endif
#ifndef HAVE_TM_GMTOFF
    if (lwday != uwday)
    {
      // Adjust for different day
      if (lwday == uwday + 1  ||  lwday == 0 && uwday == 6)
        lt += 24*3600;
      else
        lt -= 24*3600;
    }
    return lt - ut;
#endif
}


/******************************************************************************/

class KSystemTimeZonesPrivate : public KTimeZones
{
public:
    ~KSystemTimeZonesPrivate();
    static KSystemTimeZonesPrivate *instance();
    KTimeZone local();
    static QString zoneinfoDir()   { return instance()->m_zoneinfoDir; }
    static KTimeZone readZone(const QString &name);

private:
    typedef QMap<QString, QString> MD5Map;    // zone name, checksum
    KSystemTimeZonesPrivate() {}
    bool findZoneTab( QFile& f );
    void readZoneTab();
    KTimeZone matchZoneFile(const char *path);
    bool checkChecksum(MD5Map::ConstIterator, const QString &referenceMd5Sum, uint size);
    static KTzfileTimeZoneSource *tzfileSource();
    static QString calcChecksum(const QString &zoneName, uint size);
    static float convertCoordinate(const QString &coordinate);

    static KSystemTimeZonesPrivate *m_instance;
    static KSystemTimeZoneSource *m_source;
    static KTzfileTimeZoneSource *m_tzfileSource;
    static MD5Map *m_md5Sums;
    static bool    m_haveCountryCodes;   // true if zone.tab contains any country codes
    QString m_zoneinfoDir;
};

static KStaticDeleter<KSystemTimeZonesPrivate> ksystzDeleter;  // ensure that the destructor is called
KSystemTimeZonesPrivate         *KSystemTimeZonesPrivate::m_instance = 0;
KSystemTimeZoneSource           *KSystemTimeZonesPrivate::m_source = 0;
KTzfileTimeZoneSource           *KSystemTimeZonesPrivate::m_tzfileSource = 0;
KSystemTimeZonesPrivate::MD5Map *KSystemTimeZonesPrivate::m_md5Sums = 0;
bool                             KSystemTimeZonesPrivate::m_haveCountryCodes = false;


KTimeZone KSystemTimeZones::local()
{
    static KTimeZone localzone;
    if (!localzone.isValid())
        localzone = KSystemTimeZonesPrivate::instance()->local();
    return localzone;
}

QString KSystemTimeZones::zoneinfoDir()
{
    return KSystemTimeZonesPrivate::zoneinfoDir();
}

KTimeZones *KSystemTimeZones::timeZones()
{
    return KSystemTimeZonesPrivate::instance();
}

KTimeZone KSystemTimeZones::readZone(const QString &name)
{
    return KSystemTimeZonesPrivate::readZone(name);
}

const KTimeZones::ZoneMap KSystemTimeZones::zones()
{
    return KSystemTimeZonesPrivate::instance()->zones();
}

KTimeZone KSystemTimeZones::zone(const QString& name)
{
    return KSystemTimeZonesPrivate::instance()->zone(name);
}

KTimeZone KSystemTimeZonesPrivate::readZone(const QString &name)
{
    return KTzfileTimeZone(tzfileSource(), name);
}

KTzfileTimeZoneSource *KSystemTimeZonesPrivate::tzfileSource()
{
    if (!m_tzfileSource)
        m_tzfileSource = new KTzfileTimeZoneSource(zoneinfoDir());
    return m_tzfileSource;
}

/*
 * Initialisation can be very calculation intensive, so ensure that only one
 * instance is ever constructed by making the constructor private.
 */
KSystemTimeZonesPrivate *KSystemTimeZonesPrivate::instance()
{
    if (!m_instance)
    {
        if (!m_md5Sums)
            m_md5Sums = new MD5Map;
        ksystzDeleter.setObject(m_instance, new KSystemTimeZonesPrivate);
        // Go read the database.
        //
        // On Windows, HKEY_LOCAL_MACHINE\SOFTWARE\Microsoft\Windows NT\CurrentVersion\Time Zones
        // is the place to look. The TZI binary value is the TIME_ZONE_INFORMATION structure.
        //
        // For Unix, read zone.tab.
        m_instance->readZoneTab();
    }
    return m_instance;
}

KSystemTimeZonesPrivate::~KSystemTimeZonesPrivate()
{
    delete m_tzfileSource;
    m_tzfileSource = 0;
    delete m_source;
    m_source = 0;
    delete m_md5Sums;
    m_md5Sums = 0;
    kdDebug() << "~KSystemTimeZonesPrivate(): complete" << endl;
}

bool KSystemTimeZonesPrivate::findZoneTab(QFile& f)
{
#if defined(SOLARIS) || defined(USE_SOLARIS)
    const QString ZONE_TAB_FILE = QString::fromLatin1("/tab/zone_sun.tab");
    const QString ZONE_INFO_DIR = QString::fromLatin1("/usr/share/lib/zoneinfo)";
#else
    const QString ZONE_TAB_FILE = QString::fromLatin1("/zone.tab");
    const QString ZONE_INFO_DIR = QString::fromLatin1("/usr/share/zoneinfo");
#endif

    // Find and open zone.tab - it's all easy except knowing where to look. Try the LSB location first.
    QDir dir;
    QString zoneinfoDir = ZONE_INFO_DIR;
    // make a note if the dir exists; whether it contains zone.tab or not'
    if (dir.exists(zoneinfoDir))
    {
        m_zoneinfoDir = zoneinfoDir;
        f.setName(zoneinfoDir + ZONE_TAB_FILE);
        if (f.open(IO_ReadOnly))
            return true;
        kdDebug() << "Can't open " << f.name() << endl;
    }

    zoneinfoDir = QString::fromLatin1("/usr/lib/zoneinfo");
    if (dir.exists(zoneinfoDir))
    {
        m_zoneinfoDir = zoneinfoDir;
        f.setName(zoneinfoDir + ZONE_TAB_FILE);
        if (f.open(IO_ReadOnly))
            return true;
        kdDebug() << "Can't open " << f.name() << endl;
    }

    zoneinfoDir = ::getenv("TZDIR");
    if (!zoneinfoDir.isEmpty() && dir.exists(zoneinfoDir))
    {
        m_zoneinfoDir = zoneinfoDir;
        f.setName(zoneinfoDir + ZONE_TAB_FILE);
        if (f.open(IO_ReadOnly))
            return true;
        kdDebug() << "Can't open " << f.name() << endl;
    }

    zoneinfoDir = QString::fromLatin1("/usr/share/lib/zoneinfo");
    if (dir.exists(zoneinfoDir + QString::fromLatin1("/src")))
    {
        m_zoneinfoDir = zoneinfoDir;
        // Solaris support. Synthesise something that looks like a zone.tab.
        //
        // grep -h ^Zone /usr/share/lib/zoneinfo/src/* | awk '{print "??\t+9999+99999\t" $2}'
        //
        // where the country code is set to "??" and the latitude/longitude
        // values are dummies.
        //
        QDir d(m_zoneinfoDir + QString::fromLatin1("/src"));
        d.setFilter( QDir::Files | QDir::Hidden | QDir::NoSymLinks );
        QStringList fileList = d.entryList();

        KTempFile temp;
        QFile f;
        f.setName(temp.name());
        if (!f.open(IO_WriteOnly|IO_Truncate))
        {
            kdError() << "Could not open/create temp file for writing" << endl;
            return false;
        }

        QFile zoneFile;
        QValueList<QCString> tokens;
        QString line;
        QString Zone = QString::fromLatin1("Zone");
        QTextStream tmpStream(&f);
        tmpStream.setEncoding(QTextStream::Latin1);
	for (QStringList::ConstIterator it = fileList.constBegin();  it != fileList.constEnd();  ++it)
        {
            zoneFile.setName(d.filePath((*it).latin1()));
            if (!zoneFile.open(IO_ReadOnly))
            {
                kdDebug() << "Could not open file '" << zoneFile.name().latin1() \
                         << "' for reading." << endl;
                continue;
            }
            QTextStream zoneStream(&zoneFile);
            zoneStream.setEncoding(QTextStream::Latin1);
            while (!zoneStream.atEnd())
            {
                line = zoneStream.readLine();
                if (line.startsWith(Zone))
                {
                    line.replace('\t', ' ');    // change tabs to spaces
                    for (int i = 0;  ; )
                    {
                        int next = line.find(' ', i);
                        if (next < 0)
                        {
                            tokens += QCString(line.mid(i).latin1());
                            break;
                        }
                        tokens += QCString(line.mid(i, next - i).latin1());
                        i = next + 1;
                    }
                    for (QValueList<QCString>::Iterator tit = tokens.begin();  tit != tokens.end();  ++tit)
                    {
                        int n = (*tit).length() - 1;
                        if (n >= 0  &&  (*tit)[n] == ' ')
                            (*tit).truncate(n);
                    }
                    tmpStream << "??\t+9999+99999\t" << tokens[1] << "\n";
                }
            }
            zoneFile.close();
        }
        f.close();
        if (!f.open(IO_ReadOnly))
        {
            kdError() << "Could not reopen temp file for reading." << endl;
            return false;
        }
        return true;
    }
    return false;
}

/*
 * Find the location of the zoneinfo files and store in m_zoneinfoDir.
 * Parse zone.tab and for each time zone, create a KSystemTimeZone instance.
 */
void KSystemTimeZonesPrivate::readZoneTab()
{
    QFile f;
    if ( !findZoneTab( f ) )
        return;
    // Parse the zone.tab or the fake temp file.
    QTextStream str(&f);
    QRegExp lineSeparator("[ \t]");
    QRegExp ordinateSeparator("[+-]");
    if (!m_source)
        m_source = new KSystemTimeZoneSource;
    while (!str.atEnd())
    {
        QString line = str.readLine();
        if (line.isEmpty() || line[0] == '#')
            continue;
        QStringList tokens = KStringHandler::perlSplit(lineSeparator, line, 4);
        int n = tokens.count();
        if (n < 3)
        {
            kdError() << "invalid record: " << line << endl;
            continue;
        }

        // Got three tokens. Now check for two ordinates plus first one is "".
        int i = tokens[1].find(QRegExp("[+-]"), 1);
        if (i < 0)
        {
            kdError() << "invalid coordinates: " << tokens[1] << endl;
            continue;
        }

        float latitude = convertCoordinate(tokens[1].left(i));
        float longitude = convertCoordinate(tokens[1].mid(i));

        // Add entry to list.
        if (tokens[0] == "??")
            tokens[0] = "";
        else if (!tokens[0].isEmpty())
            m_haveCountryCodes = true;
        // Solaris sets the empty Comments field to '-', making it not empty.
        // Clean it up.
        if (n > 3  &&  tokens[3] == "-")
            tokens[3] = "";
        add(KSystemTimeZone(m_source, tokens[2], tokens[0], latitude, longitude, (n > 3 ? tokens[3] : QString::null)));
    }
    f.close();
}

KTimeZone KSystemTimeZonesPrivate::local()
{
    KTimeZone local;

    instance();    // initialize data

    // SOLUTION 1: DEFINITIVE.
    // First try the simplest solution of checking for well-formed TZ setting.
    char *envZone = ::getenv("TZ");
    if (envZone)
    {
        if (envZone[0] == '\0')
            return KTimeZone::utc();
        char *TZfile = 0;
        if (envZone[0] == ':')
        {
            // TZ specifies a file name, either relative to zoneinfo/ or absolute.
            TZfile = ++envZone;
        }
        if (*envZone != '/')
        {
            local = zone(envZone);
            if (local.isValid())
                return local;
            TZfile = 0;   // relative file not found
        }
        if (TZfile)
        {
            local = matchZoneFile(TZfile);
            if (local.isValid())
                return local;
        }
    }

    // SOLUTION 2: DEFINITIVE.
    // BSD support (and Linux): local time zone id in /etc/timezone.
    QString fileZone;
    QFile f;
    f.setName("/etc/timezone");
    if (f.open(IO_ReadOnly))
    {
        QTextStream ts(&f);
        ts.setEncoding(QTextStream::Latin1);

        // Read the first line.
        if (!ts.atEnd())
        {
            fileZone = ts.readLine();

            // kdDebug() << "local=" << fileZone << endl;
            local = zone(fileZone);
        }
        f.close();
        if (local.isValid())
        {
            kdDebug() << "Local time zone=" << fileZone << " from " << f.name() << endl;
            return local;
        }
    }

    if (!m_instance->m_zoneinfoDir.isEmpty())
    {
        // SOLUTION 3: DEFINITIVE.
        // Try to follow any /etc/localtime symlink to a zoneinfo file.
        // SOLUTION 4: DEFINITIVE.
        // Try to match /etc/localtime against the list of zoneinfo files.
        local = matchZoneFile("/etc/localtime");
        if (local.isValid())
        {
            kdDebug() << "Local time zone=" << local.name() << " from /etc/localtime" << endl;
            return local;
        }
    }

    // SOLUTION 5: DEFINITIVE.
    // Solaris support using /etc/default/init.
    f.setName("/etc/default/init");
    if (f.open(IO_ReadOnly))
    {
        QTextStream ts(&f);
        ts.setEncoding(QTextStream::Latin1);

        // Read the last line starting "TZ=".
        while (!ts.atEnd() && !local.isValid())
        {
            fileZone = ts.readLine();
            if (fileZone.startsWith("TZ="))
            {
                fileZone = fileZone.mid(3);

                // kdDebug() << "local=" << fileZone << endl;
                local = zone(fileZone);
            }
        }
        f.close();
        if (local.isValid())
        {
            kdDebug() << "Local time zone=" << fileZone << " from " << f.name() << endl;
            return local;
        }
    }

    // SOLUTION 6: HEURISTIC.
    // None of the deterministic stuff above has worked: try a heuristic. We
    // try to find a pair of matching time zone abbreviations...that way, we'll
    // likely return a value in the user's own country.
    if (!m_instance->m_zoneinfoDir.isEmpty())
    {
        tzset();
        QCString tzname0(tzname[0]);   // store copies, because zone.parse() will change them
        QCString tzname1(tzname[1]);
        int bestOffset = INT_MAX;
        KSystemTimeZoneSource::startParseBlock();
        const ZoneMap zmap = zones();
        for (ZoneMap::ConstIterator it = zmap.constBegin(), end = zmap.constEnd();  it != end;  ++it)
        {
            KSystemTimeZone zone = *static_cast<const KSystemTimeZone*>(&it.data());
            int candidateOffset = QABS(zone.currentOffset(Qt::LocalTime));
            if (candidateOffset < bestOffset
            &&  zone.parse())
            {
                QValueList<QCString> abbrs = zone.abbreviations();
                if (abbrs.find(tzname0) != abbrs.end()  &&  abbrs.find(tzname1) != abbrs.end())
                {
                    // kdDebug() << "local=" << zone.name() << endl;
                    local = zone;
                    bestOffset = candidateOffset;
                    if (!bestOffset)
                        break;
                }
            }
        }
        KSystemTimeZoneSource::endParseBlock();
    }
    if (local.isValid())
        return local;

    // SOLUTION 7: FAILSAFE.
    return KTimeZone::utc();
}

// Try to find a zoneinfo/ file which matches a given file.
KTimeZone KSystemTimeZonesPrivate::matchZoneFile(const char *path)
{
    if (m_instance->m_zoneinfoDir.isEmpty())
        return KTimeZone();
 
    // SOLUTION 2: DEFINITIVE.
    // Try to follow any symlink to a zoneinfo file.
    KTimeZone local;
    QFile f;
    f.setName(path);
    QFileInfo fi(f);
    if (fi.isSymLink())
    {
        // Get the path of the file which the symlink points to
        QString zoneInfoFileName;
        char curdir[PATH_MAX+1];
        if (::getcwd(curdir, PATH_MAX))
        {
            char real[PATH_MAX+1];
            if (::realpath(static_cast<const char*>(QFile::encodeName(fi.fileName())), real))
            {
                zoneInfoFileName = QFile::decodeName(real);
                if (!QFile::exists(zoneInfoFileName))
                    zoneInfoFileName = QString::null;
            }
            ::chdir(curdir);
        }
        else
            zoneInfoFileName = fi.absFilePath();
        if (zoneInfoFileName.startsWith(m_instance->m_zoneinfoDir))
        {
            QFileInfo fiz(zoneInfoFileName);
            if (fiz.exists() && fiz.isReadable())
            {
                // We've got the zoneinfo file path.
                // The time zone name is the part of the path after the zoneinfo directory.
                QString name = zoneInfoFileName.mid(m_instance->m_zoneinfoDir.length() + 1);
                // kdDebug() << "local=" << name << endl;
                return zone(name);
            }
        }
    }
    else if (f.open(IO_ReadOnly))
    {
        // SOLUTION 3: DEFINITIVE.
        // Try to match against the list of zoneinfo files.

        // Compute the file's MD5 sum
        KMD5 context("");
        context.reset();
        context.update(f);
        uint referenceSize = f.size();
        MD5Map::ConstIterator it5, end5;
        QString referenceMd5Sum = context.hexDigest();
        f.close();

        if (m_haveCountryCodes)
        {
            /* Look for time zones with the user's country code.
             * This has two advantages: 1) it shortens the search;
             * 2) it increases the chance of the correctly titled time zone
             * being found, since multiple time zones can have identical
             * definitions. For example, Europe/Guernsey is identical to
             * Europe/London, but the latter is more likely to be the right
             * zone for a user with 'gb' country code.
             */
            QString country = KGlobal::locale()->country().upper();
            const ZoneMap zmap = zones();
            for (ZoneMap::ConstIterator zit = zmap.constBegin(), zend = zmap.constEnd();  zit != zend;  ++zit)
            {
                KTimeZone tzone = zit.data();
                if (tzone.countryCode() == country)
                {
                    QString zonename = tzone.name();
                    it5 = m_md5Sums->find(zonename);
                    if (it5 == m_md5Sums->constEnd())
                    {
                        QString candidateMd5Sum = calcChecksum(zonename, referenceSize);
                        if (candidateMd5Sum == referenceMd5Sum)
                        {
                            // kdDebug() << "local=" << zonename << endl;
                            return tzone;
                        }
                    }
                    else
                    {
                        if (it5.data() == referenceMd5Sum)
                        {
                            // The cached checksum matches. Ensure that the file hasn't changed.
                            if (checkChecksum(it5, referenceMd5Sum, referenceSize))
                            {
                                local = zone(it5.key());
                                if (local.isValid())
                                    return local;
                            }
                            break;    // cache has been cleared
                        }
                    }
                }
            }
        }

        // Look for a checksum match with the cached checksum values
        MD5Map oldChecksums = *m_md5Sums;   // save a copy of the existing checksums
        for (it5 = m_md5Sums->constBegin(), end5 = m_md5Sums->constEnd();  it5 != end5;  ++it5)
        {
            if (it5.data() == referenceMd5Sum)
            {
                // The cached checksum matches. Ensure that the file hasn't changed.
                if (checkChecksum(it5, referenceMd5Sum, referenceSize))
                {
                    local = zone(it5.key());
                    if (local.isValid())
                        return local;
                }
                oldChecksums.clear();    // the cache has been cleared
                break;
            }
        }

        // The checksum didn't match any in the cache.
        // Continue building missing entries in the cache on the assumption that
        // we haven't previously looked at the zoneinfo file which matches.
        const ZoneMap zmap = zones();
        for (ZoneMap::ConstIterator zit = zmap.constBegin(), zend = zmap.constEnd();  zit != zend;  ++zit)
        {
            KTimeZone zone = zit.data();
            QString zonename = zone.name();
            if (!m_md5Sums->contains(zonename))
            {
                QString candidateMd5Sum = calcChecksum(zonename, referenceSize);
                if (candidateMd5Sum == referenceMd5Sum)
                {
                    // kdDebug() << "local=" << zonename << endl;
                    return zone;
                }
            }
        }

        // Didn't find the file, so presumably a previously cached checksum must
        // have changed. Delete all the old checksums.
        MD5Map::ConstIterator mit;
        MD5Map::ConstIterator mend = oldChecksums.constEnd();
        for (mit = oldChecksums.constBegin();  mit != mend;  ++mit)
            m_md5Sums->remove(mit.key());

        // And recalculate the old checksums
        for (mit = oldChecksums.constBegin(); mit != mend; ++mit)
        {
            QString zonename = mit.key();
            QString candidateMd5Sum = calcChecksum(zonename, referenceSize);
            if (candidateMd5Sum == referenceMd5Sum)
            {
                // kdDebug() << "local=" << zonename << endl;
                local = zone(zonename);
                if (local.isValid())
                    return local;
            }
        }

        // The file doesn't match a zoneinfo file. If it's a TZfile, use it directly.
        if (f.open(IO_ReadOnly))
        {
            // Read the file type identifier.
            char buff[4];
            QDataStream str(&f);
            str.readRawBytes(buff, 4);
            if (buff[0] == 'T' && buff[1] == 'Z' && buff[2] == 'i' && buff[3] == 'f')
            {
                QString zonename = f.name();
                local = KTzfileTimeZone(tzfileSource(), zonename);
                if (local.isValid())
                {
                    // Add the new time zone to the list
                    KTimeZone oldzone = zone(zonename);
                    if (!oldzone.isValid() || oldzone.type() != "KTzfileTimeZone")
                    {
                        remove(oldzone);
                        add(local);
                    }
                }
            }
            f.close();
        }
    }
    return local;
}

// Check whether a checksum matches a given saved checksum.
// Returns false if file no longer matches and cache was cleared.
bool KSystemTimeZonesPrivate::checkChecksum(MD5Map::ConstIterator it5, const QString &referenceMd5Sum, uint size)
{
    // The cached checksum matches. Ensure that the file hasn't changed.
    QString zoneName = it5.key();
    QString candidateMd5Sum = calcChecksum(zoneName, size);
    if (candidateMd5Sum.isNull())
        m_md5Sums->remove(zoneName);    // no match - wrong file size
    else if (candidateMd5Sum == referenceMd5Sum)
        return true;

    // File(s) have changed, so clear the cache
    m_md5Sums->clear();
    (*m_md5Sums)[zoneName] = candidateMd5Sum;    // reinsert the newly calculated checksum
    return false;
}

// Calculate the MD5 checksum for the given zone file, provided that its size matches.
// The calculated checksum is cached.
QString KSystemTimeZonesPrivate::calcChecksum(const QString &zoneName, uint size)
{
    QString path = m_instance->m_zoneinfoDir + '/' + zoneName;
    QFileInfo fi(path);
    if (fi.size() == size)
    {
        // Only do the heavy lifting for file sizes which match.
        QFile f;
        f.setName(path);
        if (f.open(IO_ReadOnly))
        {
            KMD5 context("");
            context.reset();
            context.update(f);
            QString candidateMd5Sum = context.hexDigest();
            f.close();
            (*m_md5Sums)[zoneName] = candidateMd5Sum;    // cache the new checksum
            return candidateMd5Sum;
        }
    }
    return QString::null;
}

/**
 * Convert sHHMM or sHHMMSS to a floating point number of degrees.
 */
float KSystemTimeZonesPrivate::convertCoordinate(const QString &coordinate)
{
    int value = coordinate.toInt();
    int degrees = 0;
    int minutes = 0;
    int seconds = 0;

    if (coordinate.length() > 6)
    {
        degrees = value / 10000;
        value -= degrees * 10000;
        minutes = value / 100;
        value -= minutes * 100;
        seconds = value;
    }
    else
    {
        degrees = value / 100;
        value -= degrees * 100;
        minutes = value;
    }
    value = degrees * 3600 + minutes * 60 + seconds;
    return static_cast<float>(value / 3600.0);
}


/******************************************************************************/


KSystemTimeZoneBackend::KSystemTimeZoneBackend(KSystemTimeZoneSource *source, const QString &name,
        const QString &countryCode, float latitude, float longitude, const QString &comment)
  : KTimeZoneBackend(source, name, countryCode, latitude, longitude, comment)
{}

KSystemTimeZoneBackend::~KSystemTimeZoneBackend()
{}

KTimeZoneBackend *KSystemTimeZoneBackend::clone() const
{
    return new KSystemTimeZoneBackend(*this);
}

QCString KSystemTimeZoneBackend::type() const
{
    return "KSystemTimeZone";
}

int KSystemTimeZoneBackend::offsetAtZoneTime(const KTimeZone *caller, const QDateTime &zoneDateTime, int *secondOffset) const
{
    if (!caller->isValid()  ||  !zoneDateTime.isValid())
        return 0;
    // Make this time zone the current local time zone
    const char *originalZone = ::getenv("TZ");   // save the original local time zone
    QCString tz = caller->name().utf8();
    tz.prepend(":");
    bool change = (tz != originalZone);
    if (change)
    {
        ::setenv("TZ", tz, 1);
        ::tzset();
    }

    // Convert zone time to UTC, and then get the offset to UTC
    tm tmtime;
    tmtime.tm_sec    = zoneDateTime.time().second();
    tmtime.tm_min    = zoneDateTime.time().minute();
    tmtime.tm_hour   = zoneDateTime.time().hour();
    tmtime.tm_mday   = zoneDateTime.date().day();
    tmtime.tm_mon    = zoneDateTime.date().month() - 1;
    tmtime.tm_year   = zoneDateTime.date().year() - 1900;
    tmtime.tm_isdst  = -1;
    time_t t = mktime(&tmtime);
    int offset1 = (t == (time_t)-1) ? 0 : gmtoff(t);
    if (secondOffset)
    {
        int offset2 = offset1;
        if (t != (time_t)-1)
        {
            // Check if there is a backward DST change near to this time, by
            // checking if the UTC offset is different 1 hour later or earlier.
            // ASSUMPTION: DST SHIFTS ARE NEVER GREATER THAN 1 HOUR.
            int maxShift = 3600;
            offset2 = gmtoff(t + maxShift);
            if (offset2 < offset1)
            {
                // There is a backward DST shift during the following hour
                if (offset1 - offset2 < maxShift)
                    offset2 = gmtoff(t + (offset1 - offset2));
            }
            else if ((offset2 = gmtoff(t - maxShift)) > offset1)
            {
                // There is a backward DST shift during the previous hour
                if (offset2 - offset1 < maxShift)
                    offset2 = gmtoff(t - (offset2 - offset1));
                // Put UTC offsets into the correct order
                int o = offset1;
                offset1 = offset2;
                offset2 = o;
            }
            else offset2 = offset1;
        }
        *secondOffset = offset2;
    }

    if (change)
    {
        // Restore the original local time zone
        if (!originalZone)
            ::unsetenv("TZ");
        else
            ::setenv("TZ", originalZone, 1);
        ::tzset();
    }
    return offset1;
}

int KSystemTimeZoneBackend::offsetAtUtc(const KTimeZone *caller, const QDateTime &utcDateTime) const
{
    return offset(caller, KTimeZone::toTime_t(utcDateTime));
}

int KSystemTimeZoneBackend::offset(const KTimeZone *caller, time_t t) const
{
    if (!caller->isValid()  ||  t == KTimeZone::InvalidTime_t)
        return 0;

    // Make this time zone the current local time zone
    const char *originalZone = ::getenv("TZ");   // save the original local time zone
    QCString tz = caller->name().utf8();
    tz.prepend(":");
    bool change = (tz != originalZone);
    if (change)
    {
        ::setenv("TZ", tz, 1);
        ::tzset();
    }

    int secs = gmtoff(t);

    if (change)
    {
        // Restore the original local time zone
        if (!originalZone)
            ::unsetenv("TZ");
        else
            ::setenv("TZ", originalZone, 1);
        ::tzset();
    }
    return secs;
}

bool KSystemTimeZoneBackend::isDstAtUtc(const KTimeZone *caller, const QDateTime &utcDateTime) const
{
    return isDst(caller, KTimeZone::toTime_t(utcDateTime));
}

bool KSystemTimeZoneBackend::isDst(const KTimeZone *caller, time_t t) const
{
    Q_UNUSED(caller)
    if (t != (time_t)-1)
    {
#ifdef _POSIX_THREAD_SAFE_FUNCTIONS
        tm tmtime;
        if (localtime_r(&t, &tmtime))
            return tmtime.tm_isdst > 0;
#else
        tm *tmtime = localtime(&t);
        if (tmtime)
            return tmtime->tm_isdst > 0;
#endif
    }
    return false;
}


/******************************************************************************/

KSystemTimeZone::KSystemTimeZone(KSystemTimeZoneSource *source, const QString &name,
        const QString &countryCode, float latitude, float longitude, const QString &comment)
  : KTimeZone(new KSystemTimeZoneBackend(source, name, countryCode, latitude, longitude, comment))
{
}

KSystemTimeZone::~KSystemTimeZone()
{
}


/******************************************************************************/

class KSystemTimeZoneDataPrivate
{
public:
    QCString TZ;
    QValueList<QCString> abbreviations;
};


// N.B. KSystemTimeZoneSourcePrivate is also used by KSystemTimeZoneData
class KSystemTimeZoneSourcePrivate
{
public:
    static void setTZ(const QCString &zoneName);
    static void restoreTZ();
    static char      *savedTZ;       // temporary value of TZ environment variable saved by setTZ()
    static QCString   originalTZ;    // saved value of TZ environment variable during multiple parse() calls
    static bool       TZIsSaved;     // TZ has been saved in savedTZ
    static bool       multiParse;    // true if performing multiple parse() calls
};

char    *KSystemTimeZoneSourcePrivate::savedTZ;
QCString KSystemTimeZoneSourcePrivate::originalTZ;
bool     KSystemTimeZoneSourcePrivate::TZIsSaved = false;
bool     KSystemTimeZoneSourcePrivate::multiParse = false;


KSystemTimeZoneSource::KSystemTimeZoneSource()
    : d(0)
//  : d(new KSystemTimeZoneSourcePrivate)
{
}

KSystemTimeZoneSource::~KSystemTimeZoneSource()
{
//    delete d;
}

KTimeZoneData* KSystemTimeZoneSource::parse(const KTimeZone &zone) const
{
    QCString tz = zone.name().utf8();
    KSystemTimeZoneSourcePrivate::setTZ(tz);   // make this time zone the current local time zone

    tzset();    // initialize the tzname array
    KSystemTimeZoneData* data = new KSystemTimeZoneData;
    data->d->TZ = tz;
    data->d->abbreviations.append(tzname[0]);
    data->d->abbreviations.append(tzname[1]);

    // There is no easy means to access the sequence of daylight savings time
    // changes, or leap seconds adjustments, so leave that data empty.

    KSystemTimeZoneSourcePrivate::restoreTZ();   // restore the original local time zone if necessary
    return data;
}

void KSystemTimeZoneSource::startParseBlock()
{
    KSystemTimeZoneSourcePrivate::originalTZ = ::getenv("TZ");   // save the original local time zone
    KSystemTimeZoneSourcePrivate::multiParse = true;
}

void KSystemTimeZoneSource::endParseBlock()
{
    if (KSystemTimeZoneSourcePrivate::multiParse)
    {
        // Restore the original local time zone
        if (KSystemTimeZoneSourcePrivate::originalTZ.isEmpty())
            ::unsetenv("TZ");
        else
            ::setenv("TZ", KSystemTimeZoneSourcePrivate::originalTZ, 1);
        ::tzset();
        KSystemTimeZoneSourcePrivate::multiParse = false;
    }
}

// Set the TZ environment variable to the specified time zone,
// saving its current setting first if necessary.
void KSystemTimeZoneSourcePrivate::setTZ(const QCString &zoneName)
{
    QCString tz = zoneName;
    tz.prepend(":");
    bool setTZ = multiParse;
    if (!setTZ)
    {
        savedTZ = ::getenv("TZ");   // save the original local time zone
        TZIsSaved = true;
        setTZ = (tz != savedTZ);
    }
    if (setTZ)
    {
        ::setenv("TZ", tz, 1);
        ::tzset();
    }
}

// Restore the TZ environment variable if it was saved by setTz()
void KSystemTimeZoneSourcePrivate::restoreTZ()
{
    if (TZIsSaved)
    {
        if (!savedTZ)
            ::unsetenv("TZ");
        else
            ::setenv("TZ", savedTZ, 1);
        ::tzset();
        TZIsSaved = false;
    }
}


/******************************************************************************/

KSystemTimeZoneData::KSystemTimeZoneData()
  : d(new KSystemTimeZoneDataPrivate)
{ }

KSystemTimeZoneData::KSystemTimeZoneData(const KSystemTimeZoneData &rhs)
  : KTimeZoneData(),
    d(new KSystemTimeZoneDataPrivate)
{
    operator=(rhs);
}

KSystemTimeZoneData::~KSystemTimeZoneData()
{
    delete d;
}

KSystemTimeZoneData &KSystemTimeZoneData::operator=(const KSystemTimeZoneData &rhs)
{
    d->TZ = rhs.d->TZ;
    d->abbreviations = rhs.d->abbreviations;
    return *this;
}

KTimeZoneData *KSystemTimeZoneData::clone() const
{
    return new KSystemTimeZoneData(*this);
}

QValueList<QCString> KSystemTimeZoneData::abbreviations() const
{
    return d->abbreviations;
}

QCString KSystemTimeZoneData::abbreviation(const QDateTime &utcDateTime) const
{
    QCString abbr;
    time_t t = utcDateTime.toTime_t();
    if (t != KTimeZone::InvalidTime_t)
    {
        KSystemTimeZoneSourcePrivate::setTZ(d->TZ);   // make this time zone the current local time zone

        /* Use tm.tm_zone if available because it returns the abbreviation
         * in use at the time specified. Otherwise, use tzname[] which
         * returns the appropriate current abbreviation instead.
         */
#ifdef _POSIX_THREAD_SAFE_FUNCTIONS
        tm tmtime;
        if (localtime_r(&t, &tmtime))
#ifdef HAVE_STRUCT_TM_TM_ZONE
            abbr = tmtime.tm_zone;
#else
            abbr = tzname[(tmtime.tm_isdst > 0) ? 1 : 0];
#endif
#else
        tm *tmtime = localtime(&t);
        if (tmtime)
#ifdef HAVE_STRUCT_TM_TM_ZONE
            abbr = tmtime->tm_zone;
#else
            abbr = tzname[(tmtime->tm_isdst > 0) ? 1 : 0];
#endif
#endif
        KSystemTimeZoneSourcePrivate::restoreTZ();   // restore the original local time zone if necessary
    }
    return abbr;
}

QValueList<int> KSystemTimeZoneData::utcOffsets() const
{
    return QValueList<int>();
}
