/* This file is part of the KDE libraries
   Copyright (c) 2005,2006 David Jarvie <software@astrojar.org.uk>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#include <stdio.h>
#include <qdatetime.h>
#include <kunittest/tester.h>
#include <kunittest/module.h>
#include <kglobal.h>
#include <klocale.h>
#include <kcalendarsystem.h>
#include "ksystemtimezone.h"
#include "kdatetime.h"
#include "kdebug.h"


class KDateTimeTest : public KUnitTest::Tester
{
public:
    void allTests();
private:
    void specConstructors();
    void specSet();
    void constructors();
    void toUtc();
    void toOffsetFromUtc();
    void toLocalZone();
    void toClockTime();
    void toZone();
    void toTimeSpec();
    void set();
    void compare();
    void addSubtract();
    void addMSecs();
    void addSubtractDate();
    void dstShifts();
    void strings_iso8601();
    void strings_rfc2822();
    void strings_qttextdate();
    void strings_format();
    void cache();
    void misc();
};

KUNITTEST_MODULE( kunittest_kdatetime, "KDateTimeTest" )
KUNITTEST_MODULE_REGISTER_TESTER( KDateTimeTest )

#define CHECKD(d1, d2) if (d1 != d2) CHECK(d1.toString(Qt::ISODate), d2.toString(Qt::ISODate))
#define CHECKT(t1, t2) if (t1 != t2) CHECK(t1.toString(Qt::ISODate), t2.toString(Qt::ISODate))
#define CHECKDT(dt1, dt2) if (dt1 != dt2) CHECK(dt1.toString(Qt::ISODate), dt2.toString(Qt::ISODate))
#define CHECKTZ(tz1, tz2) if (tz1 != tz2) CHECK((tz1.isValid() ? (tz1).name() : QString("null")), (tz2.isValid() ? (tz2).name() : QString("null")))
#define CHECKSPEC(s1, s2) if (s1 != s2) CHECK("timespec1", "timespec2")
#define QCOMPARE(a,b) CHECK(a,b)

static QDateTime toUTC(const QDateTime& local)
{
    QDateTime dt;
    dt.setTime_t(local.toTime_t(), Qt::UTC);
    return dt;
}

static time_t utcToTime_t(const QDateTime& utc)
{
    tm TM;
    TM.tm_sec = utc.time().second();
    TM.tm_min = utc.time().minute();
    TM.tm_hour = utc.time().hour();
    TM.tm_mday = utc.date().day();
    TM.tm_mon = utc.date().month() - 1;
    TM.tm_year = utc.date().year() - 1900;
    TM.tm_isdst = -1;
    const char *originalZone = ::getenv("TZ");   // save the original local time zone
    ::setenv("TZ", ":UTC", 1);
    ::tzset();
    time_t t = mktime(&TM);
    if (!originalZone)
        ::unsetenv("TZ");
    else
        ::setenv("TZ", originalZone, 1);
    ::tzset();
    return t;
}

static QDateTime toLocal(const QDateTime& utc)
{
    QDateTime dtl;
    dtl.setTime_t(utcToTime_t(utc), Qt::LocalTime);
    return dtl;
}

void KDateTimeTest::allTests()
{
    specConstructors();
    specSet();
    constructors();
    toUtc();
    toOffsetFromUtc();
    toLocalZone();
    toClockTime();
    toZone();
    toTimeSpec();
    set();
    compare();
    addSubtract();
    addMSecs();
    addSubtractDate();
    dstShifts();
    strings_iso8601();
    strings_rfc2822();
    strings_qttextdate();
    strings_format();
    cache();
    misc();
}


extern int KDateTime_utcCacheHit;
extern int KDateTime_zoneCacheHit;


////////////////////////////////////////////////////////////////////////
// KDateTime::Spec constructors and basic property information methods,
// and the static convenience instances/methods.
////////////////////////////////////////////////////////////////////////

void KDateTimeTest::specConstructors()
{
    KTimeZone london = KSystemTimeZones::zone("Europe/London");
    KTimeZone losAngeles = KSystemTimeZones::zone("America/Los_Angeles");

    const char *originalZone = ::getenv("TZ");   // save the original local time zone
    ::setenv("TZ", ":Europe/London", 1);
    ::tzset();

    // Ensure that local time is different from UTC and different from 'london'
    ::setenv("TZ", ":America/Los_Angeles", 1);
    ::tzset();


    // Default constructor
    KDateTime::Spec invalid;
    CHECK(invalid.isValid(), false);
    QCOMPARE(invalid.type(), KDateTime::Invalid);
    CHECK(invalid.isLocalZone(), false);
    CHECK(invalid.isUtc(), false);
    CHECK(invalid.isOffsetFromUtc(), false);
    CHECK(invalid.isClockTime(), false);
    QCOMPARE(invalid.utcOffset(), 0);
    CHECK(invalid.timeZone().isValid(), false);


    // Time zone
    KDateTime::Spec tz(london);
    CHECK(tz.isValid(), true);
    QCOMPARE(tz.type(), KDateTime::TimeZone);
    CHECK(tz.isUtc(), false);
    CHECK(tz.isOffsetFromUtc(), false);
    CHECK(tz.isLocalZone(), false);
    CHECK(tz.isClockTime(), false);
    QCOMPARE(tz.utcOffset(), 0);
    CHECKTZ(tz.timeZone(), london);

    KDateTime::Spec tzLocal(losAngeles);
    CHECK(tzLocal.isValid(), true);
    QCOMPARE(tzLocal.type(), KDateTime::TimeZone);
    CHECKSPEC(tzLocal, KDateTime::Spec(KDateTime::LocalZone));
    CHECK(tzLocal.isUtc(), false);
    CHECK(tzLocal.isOffsetFromUtc(), false);
    CHECK(tzLocal.isLocalZone(), true);
    CHECK(tzLocal.isClockTime(), false);
    QCOMPARE(tzLocal.utcOffset(), 0);
    CHECKTZ(tzLocal.timeZone(), losAngeles);

    // ... copy constructor
    KDateTime::Spec tzCopy(tz);
    CHECK(tzCopy.isValid(), true);
    QCOMPARE(tzCopy.type(), KDateTime::TimeZone);
    CHECK(tzCopy.isUtc(), false);
    CHECK(tzCopy.isOffsetFromUtc(), false);
    CHECK(tzCopy.isLocalZone(), false);
    CHECK(tzCopy.isClockTime(), false);
    QCOMPARE(tzCopy.utcOffset(), 0);
    CHECKTZ(tzCopy.timeZone(), london);


    // Local time zone
    KDateTime::Spec local(KDateTime::LocalZone);
    CHECK(local.isValid(), true);
    QCOMPARE(local.type(), KDateTime::TimeZone);
    CHECKSPEC(local, KDateTime::Spec(KDateTime::LocalZone));
    CHECK(local.isUtc(), false);
    CHECK(local.isOffsetFromUtc(), false);
    CHECK(local.isLocalZone(), true);
    CHECK(local.isClockTime(), false);
    QCOMPARE(local.utcOffset(), 0);
    CHECKTZ(local.timeZone(), KSystemTimeZones::local());
    // Check that the local time zone is what we set at the start of these tests
    CHECK(KSystemTimeZones::local().name(), QString("America/Los_Angeles"));

    KDateTime::Spec localx(KDateTime::Spec(KDateTime::LocalZone, 2*3600));
    CHECK(localx.isValid(), true);
    QCOMPARE(localx.type(), KDateTime::TimeZone);
    CHECKSPEC(localx, KDateTime::Spec(KDateTime::LocalZone));
    CHECK(localx.isUtc(), false);
    CHECK(localx.isOffsetFromUtc(), false);
    CHECK(localx.isLocalZone(), true);
    CHECK(localx.isClockTime(), false);
    QCOMPARE(localx.utcOffset(), 0);
    CHECKTZ(localx.timeZone(), KSystemTimeZones::local());

    KDateTime::Spec local2 = KDateTime::Spec::LocalZone();
    CHECK(local2.isValid(), true);
    QCOMPARE(local2.type(), KDateTime::TimeZone);
    CHECKSPEC(local2, KDateTime::Spec(KDateTime::LocalZone));
    CHECK(local2.isUtc(), false);
    CHECK(local2.isOffsetFromUtc(), false);
    CHECK(local2.isLocalZone(), true);
    CHECK(local2.isClockTime(), false);
    QCOMPARE(local2.utcOffset(), 0);
    CHECKTZ(local2.timeZone(), KSystemTimeZones::local());

    // ... copy constructor
    KDateTime::Spec localCopy(local);
    CHECK(localCopy.isValid(), true);
    QCOMPARE(localCopy.type(), KDateTime::TimeZone);
    CHECKSPEC(localCopy, KDateTime::Spec(KDateTime::LocalZone));
    CHECK(localCopy.isUtc(), false);
    CHECK(localCopy.isOffsetFromUtc(), false);
    CHECK(localCopy.isLocalZone(), true);
    CHECK(localCopy.isClockTime(), false);
    QCOMPARE(localCopy.utcOffset(), 0);
    CHECKTZ(localCopy.timeZone(), losAngeles);


    // UTC
    KDateTime::Spec utc(KDateTime::UTC);
    CHECK(utc.isValid(), true);
    QCOMPARE(utc.type(), KDateTime::UTC);
    CHECK(utc.isUtc(), true);
    CHECK(utc.isOffsetFromUtc(), false);
    CHECK(utc.isLocalZone(), false);
    CHECK(utc.isClockTime(), false);
    QCOMPARE(utc.utcOffset(), 0);
    CHECKTZ(utc.timeZone(), KTimeZone::utc());

    KDateTime::Spec utcx(KDateTime::UTC, 2*3600);
    CHECK(utcx.isValid(), true);
    QCOMPARE(utcx.type(), KDateTime::UTC);
    CHECK(utcx.isUtc(), true);
    CHECK(utcx.isOffsetFromUtc(), false);
    CHECK(utcx.isLocalZone(), false);
    CHECK(utcx.isClockTime(), false);
    QCOMPARE(utcx.utcOffset(), 0);
    CHECKTZ(utcx.timeZone(), KTimeZone::utc());

    const KDateTime::Spec& utc2 = KDateTime::Spec::UTC();
    CHECK(utc2.isValid(), true);
    QCOMPARE(utc2.type(), KDateTime::UTC);
    CHECK(utc2.isUtc(), true);
    CHECK(utc2.isOffsetFromUtc(), false);
    CHECK(utc2.isLocalZone(), false);
    CHECK(utc2.isClockTime(), false);
    QCOMPARE(utc2.utcOffset(), 0);
    CHECKTZ(utc2.timeZone(), KTimeZone::utc());

    // ... copy constructor
    KDateTime::Spec utcCopy(utc);
    CHECK(utcCopy.isValid(), true);
    QCOMPARE(utcCopy.type(), KDateTime::UTC);
    CHECK(utcCopy.isUtc(), true);
    CHECK(utcCopy.isOffsetFromUtc(), false);
    CHECK(utcCopy.isLocalZone(), false);
    CHECK(utcCopy.isClockTime(), false);
    QCOMPARE(utcCopy.utcOffset(), 0);
    CHECKTZ(utcCopy.timeZone(), KTimeZone::utc());


    // Offset from UTC
    KDateTime::Spec offset0(KDateTime::OffsetFromUTC);
    CHECK(offset0.isValid(), true);
    QCOMPARE(offset0.type(), KDateTime::OffsetFromUTC);
    CHECK(offset0.isUtc(), true);
    CHECK(offset0.isOffsetFromUtc(), true);
    CHECK(offset0.isLocalZone(), false);
    CHECK(offset0.isClockTime(), false);
    QCOMPARE(offset0.utcOffset(), 0);
    CHECK(offset0.timeZone().isValid(), false);

    KDateTime::Spec offset(KDateTime::Spec(KDateTime::OffsetFromUTC, -2*3600));
    CHECK(offset.isValid(), true);
    QCOMPARE(offset.type(), KDateTime::OffsetFromUTC);
    CHECK(offset.isUtc(), false);
    CHECK(offset.isOffsetFromUtc(), true);
    CHECK(offset.isLocalZone(), false);
    CHECK(offset.isClockTime(), false);
    QCOMPARE(offset.utcOffset(), -2*3600);
    CHECK(offset.timeZone().isValid(), false);

    KDateTime::Spec offset2 = KDateTime::Spec::OffsetFromUTC(2*3600);
    CHECK(offset2.isValid(), true);
    QCOMPARE(offset2.type(), KDateTime::OffsetFromUTC);
    CHECK(offset2.isUtc(), false);
    CHECK(offset2.isOffsetFromUtc(), true);
    CHECK(offset2.isLocalZone(), false);
    CHECK(offset2.isClockTime(), false);
    QCOMPARE(offset2.utcOffset(), 2*3600);
    CHECK(offset2.timeZone().isValid(), false);

    // ... copy constructor
    KDateTime::Spec offsetCopy(offset);
    CHECK(offsetCopy.isValid(), true);
    QCOMPARE(offsetCopy.type(), KDateTime::OffsetFromUTC);
    CHECK(offsetCopy.isUtc(), false);
    CHECK(offsetCopy.isOffsetFromUtc(), true);
    CHECK(offsetCopy.isLocalZone(), false);
    CHECK(offsetCopy.isClockTime(), false);
    QCOMPARE(offsetCopy.utcOffset(), -2*3600);
    CHECK(offsetCopy.timeZone().isValid(), false);


    // Local clock time
    KDateTime::Spec clock(KDateTime::ClockTime);
    CHECK(clock.isValid(), true);
    QCOMPARE(clock.type(), KDateTime::ClockTime);
    CHECK(clock.isUtc(), false);
    CHECK(clock.isOffsetFromUtc(), false);
    CHECK(clock.isLocalZone(), false);
    CHECK(clock.isClockTime(), true);
    QCOMPARE(clock.utcOffset(), 0);
    CHECK(clock.timeZone().isValid(), false);

    KDateTime::Spec clockx(KDateTime::Spec(KDateTime::ClockTime, 2*3600));
    CHECK(clockx.isValid(), true);
    QCOMPARE(clockx.type(), KDateTime::ClockTime);
    CHECK(clockx.isUtc(), false);
    CHECK(clockx.isOffsetFromUtc(), false);
    CHECK(clockx.isLocalZone(), false);
    CHECK(clockx.isClockTime(), true);
    QCOMPARE(clockx.utcOffset(), 0);
    CHECK(clockx.timeZone().isValid(), false);

    const KDateTime::Spec &clock2 = KDateTime::Spec::ClockTime();
    CHECK(clock2.isValid(), true);
    QCOMPARE(clock2.type(), KDateTime::ClockTime);
    CHECK(clock2.isUtc(), false);
    CHECK(clock2.isOffsetFromUtc(), false);
    CHECK(clock2.isLocalZone(), false);
    CHECK(clock2.isClockTime(), true);
    QCOMPARE(clock2.utcOffset(), 0);
    CHECK(clock2.timeZone().isValid(), false);

    // ... copy constructor
    KDateTime::Spec clockCopy(clock);
    CHECK(clockCopy.isValid(), true);
    QCOMPARE(clockCopy.type(), KDateTime::ClockTime);
    CHECK(clockCopy.isUtc(), false);
    CHECK(clockCopy.isOffsetFromUtc(), false);
    CHECK(clockCopy.isLocalZone(), false);
    CHECK(clockCopy.isClockTime(), true);
    QCOMPARE(clockCopy.utcOffset(), 0);
    CHECK(clockCopy.timeZone().isValid(), false);


    // Restore the original local time zone
    if (!originalZone)
        ::unsetenv("TZ");
    else
        ::setenv("TZ", originalZone, 1);
    ::tzset();
}

////////////////////////////////////////////////////////////////////////
// KDateTime::Spec setType(), operator==(), etc.
////////////////////////////////////////////////////////////////////////

void KDateTimeTest::specSet()
{
    KTimeZone london = KSystemTimeZones::zone("Europe/London");
    KTimeZone losAngeles = KSystemTimeZones::zone("America/Los_Angeles");

    // Ensure that local time is different from UTC and different from 'london'
    const char *originalZone = ::getenv("TZ");   // save the original local time zone
    ::setenv("TZ", ":America/Los_Angeles", 1);
    ::tzset();

    KDateTime::Spec spec;
    QCOMPARE(spec.type(), KDateTime::Invalid);

    spec.setType(KDateTime::OffsetFromUTC, 7200);
    QCOMPARE(spec.type(), KDateTime::OffsetFromUTC);
    CHECK(spec.equivalentTo(KDateTime::Spec::OffsetFromUTC(7200)), true);
    CHECK(spec.equivalentTo(KDateTime::Spec::OffsetFromUTC(0)), false);
    CHECK(spec == KDateTime::Spec::OffsetFromUTC(7200), true);
    CHECK((spec != KDateTime::Spec::OffsetFromUTC(7200)), false);
    CHECK(spec != KDateTime::Spec::OffsetFromUTC(-7200), true);
    CHECK(spec != KDateTime::Spec(london), true);

    spec.setType(KDateTime::OffsetFromUTC, 0);
    QCOMPARE(spec.type(), KDateTime::OffsetFromUTC);
    CHECK(spec.equivalentTo(KDateTime::Spec::OffsetFromUTC(0)), true);
    CHECK(spec.equivalentTo(KDateTime::Spec::UTC()), true);
    CHECK(spec.equivalentTo(KDateTime::Spec::OffsetFromUTC(7200)), false);
    CHECK(spec == KDateTime::Spec::OffsetFromUTC(0), true);
    CHECK((spec != KDateTime::Spec::OffsetFromUTC(0)), false);
    CHECK(spec != KDateTime::Spec::OffsetFromUTC(-7200), true);
    CHECK(spec != KDateTime::Spec(london), true);

    spec.setType(london);
    QCOMPARE(spec.type(), KDateTime::TimeZone);
    CHECK(spec.equivalentTo(KDateTime::Spec(london)), true);
    CHECK(spec == KDateTime::Spec(london), true);
    CHECK((spec != KDateTime::Spec(london)), false);
    CHECK(spec != KDateTime::Spec::OffsetFromUTC(0), true);
    CHECK(spec.equivalentTo(KDateTime::Spec::OffsetFromUTC(0)), false);

    spec.setType(KDateTime::LocalZone);
    QCOMPARE(spec.type(), KDateTime::TimeZone);
    CHECK(spec.equivalentTo(KDateTime::Spec::LocalZone()), true);
    CHECK(spec == KDateTime::Spec::LocalZone(), true);
    CHECK((spec != KDateTime::Spec::LocalZone()), false);
    CHECK(spec.equivalentTo(KDateTime::Spec(losAngeles)), true);
    CHECK(spec == KDateTime::Spec(losAngeles), true);
    CHECK(spec != KDateTime::Spec(london), true);
    CHECK(spec.equivalentTo(KDateTime::Spec(london)), false);

    spec.setType(KDateTime::UTC);
    QCOMPARE(spec.type(), KDateTime::UTC);
    CHECK(spec.equivalentTo(KDateTime::Spec::UTC()), true);
    CHECK(spec == KDateTime::Spec::UTC(), true);
    CHECK((spec != KDateTime::Spec::UTC()), false);
    CHECK(spec != KDateTime::Spec::LocalZone(), true);
    CHECK(spec.equivalentTo(KDateTime::Spec::LocalZone()), false);
    CHECK(spec.equivalentTo(KDateTime::Spec::OffsetFromUTC(0)), true);

    spec.setType(KDateTime::ClockTime);
    QCOMPARE(spec.type(), KDateTime::ClockTime);
    CHECK(spec.equivalentTo(KDateTime::Spec::ClockTime()), true);
    CHECK(spec == KDateTime::Spec::ClockTime(), true);
    CHECK((spec != KDateTime::Spec::ClockTime()), false);
    CHECK(spec != KDateTime::Spec::UTC(), true);
    CHECK(spec.equivalentTo(KDateTime::Spec::UTC()), false);

    // Restore the original local time zone
    if (!originalZone)
        ::unsetenv("TZ");
    else
        ::setenv("TZ", originalZone, 1);
    ::tzset();
}


//////////////////////////////////////////////////////
// Constructors and basic property information methods
//////////////////////////////////////////////////////

void KDateTimeTest::constructors()
{
    QDate d(2001,2,13);
    QTime t(3,45,14);
    QDateTime dtLocal(d, t); // Qt::LocalTime
    QDateTime dtUTC(d, t); // Qt::UTC
    KTimeZone london = KSystemTimeZones::zone("Europe/London");

    const char *originalZone = ::getenv("TZ");   // save the original local time zone
    ::setenv("TZ", ":Europe/London", 1);
    ::tzset();
    QDateTime dtUTCtoLondon;
    dtUTCtoLondon.setTime_t(dtUTC.toTime_t(), Qt::LocalTime);

    // Ensure that local time is different from UTC and different from 'london'
    ::setenv("TZ", ":America/Los_Angeles", 1);
    ::tzset();


    // Default constructor
    KDateTime deflt;
    CHECK(deflt.isNull(), true);
    CHECK(deflt.isValid(), false);


    // No time zone or timeSpec explicitly specified
    KDateTime datetimeL(dtLocal, Qt::LocalTime);
    CHECK(datetimeL.isNull(), false);
    CHECK(datetimeL.isValid(), true);
    CHECK(datetimeL.isDateOnly(), false);
    QCOMPARE(datetimeL.timeType(), KDateTime::TimeZone);
    CHECKSPEC(datetimeL.timeSpec(), KDateTime::Spec::LocalZone());
    CHECK(datetimeL.isLocalZone(), true);
    CHECK(datetimeL.isUtc(), false);
    CHECK(datetimeL.isOffsetFromUtc(), false);
    CHECK(datetimeL.isClockTime(), false);
    QCOMPARE(datetimeL.utcOffset(), -8*3600);
    CHECKTZ(datetimeL.timeZone(), KSystemTimeZones::local());
    CHECKDT(datetimeL.dateTime(), dtLocal);

    KDateTime datetimeU(dtUTC, Qt::UTC);
    CHECK(datetimeU.isNull(), false);
    CHECK(datetimeU.isValid(), true);
    CHECK(datetimeU.isDateOnly(), false);
    QCOMPARE(datetimeU.timeType(), KDateTime::UTC);
    CHECK(datetimeU.isLocalZone(), false);
    CHECK(datetimeU.isUtc(), true);
    CHECK(datetimeU.isOffsetFromUtc(), false);
    CHECK(datetimeU.isClockTime(), false);
    QCOMPARE(datetimeU.utcOffset(), 0);
    CHECKTZ(datetimeU.timeZone(), KTimeZone::utc());
    CHECKDT(datetimeU.dateTime(), dtUTC);


    // Time zone
    KDateTime dateTz(d, london);
    CHECK(dateTz.isNull(), false);
    CHECK(dateTz.isValid(), true);
    CHECK(dateTz.isDateOnly(), true);
    QCOMPARE(dateTz.timeType(), KDateTime::TimeZone);
    CHECK(dateTz.isUtc(), false);
    CHECK(dateTz.isOffsetFromUtc(), false);
    CHECK(dateTz.isLocalZone(), false);
    CHECK(dateTz.isClockTime(), false);
    QCOMPARE(dateTz.utcOffset(), 0);
    CHECKTZ(dateTz.timeZone(), london);
    CHECKDT(dateTz.dateTime(), QDateTime(d, QTime(0,0,0))); // Qt::LocalTime));
    CHECKD(dateTz.date(), d);

    KDateTime dateTimeTz(d, QTime(3,45,14), london);
    CHECK(dateTimeTz.isNull(), false);
    CHECK(dateTimeTz.isValid(), true);
    CHECK(dateTimeTz.isDateOnly(), false);
    QCOMPARE(dateTimeTz.timeType(), KDateTime::TimeZone);
    CHECK(dateTimeTz.isUtc(), false);
    CHECK(dateTimeTz.isOffsetFromUtc(), false);
    CHECK(dateTimeTz.isLocalZone(), false);
    CHECK(dateTimeTz.isClockTime(), false);
    QCOMPARE(dateTimeTz.utcOffset(), 0);
    CHECKTZ(dateTimeTz.timeZone(), london);
    CHECKDT(dateTimeTz.dateTime(), QDateTime(d, QTime(3,45,14))); // Qt::LocalTime));
    CHECKT(dateTimeTz.time(), QTime(3,45,14));

    KDateTime datetimeTz(dtLocal, Qt::LocalTime, london);
    CHECK(datetimeTz.isNull(), false);
    CHECK(datetimeTz.isValid(), true);
    CHECK(dateTimeTz.isDateOnly(), false);
    QCOMPARE(datetimeTz.timeType(), KDateTime::TimeZone);
    CHECK(datetimeTz.isUtc(), false);
    CHECK(datetimeTz.isOffsetFromUtc(), false);
    CHECK(datetimeTz.isLocalZone(), false);
    CHECK(datetimeTz.isClockTime(), false);
    QCOMPARE(datetimeTz.utcOffset(), 0);
    CHECKTZ(datetimeTz.timeZone(), london);
    CHECKDT(datetimeTz.dateTime(), dtLocal);

    KDateTime datetimeTz2(dtUTC, Qt::UTC, london);
    CHECK(datetimeTz2.isNull(), false);
    CHECK(datetimeTz2.isValid(), true);
    CHECK(dateTimeTz.isDateOnly(), false);
    QCOMPARE(datetimeTz2.timeType(), KDateTime::TimeZone);
    CHECK(datetimeTz2.isUtc(), false);
    CHECK(datetimeTz2.isOffsetFromUtc(), false);
    CHECK(datetimeTz2.isLocalZone(), false);
    CHECK(datetimeTz2.isClockTime(), false);
    QCOMPARE(datetimeTz2.utcOffset(), 0);
    CHECKTZ(datetimeTz2.timeZone(), london);
    CHECKDT(datetimeTz2.dateTime(), dtUTCtoLondon);

    // ... copy constructor
    KDateTime datetimeTzCopy(datetimeTz);
    CHECK(datetimeTzCopy.isNull(), false);
    CHECK(datetimeTzCopy.isValid(), true);
    CHECK(dateTimeTz.isDateOnly(), false);
    QCOMPARE(datetimeTzCopy.timeType(), KDateTime::TimeZone);
    CHECK(datetimeTzCopy.isUtc(), false);
    CHECK(datetimeTzCopy.isOffsetFromUtc(), false);
    CHECK(datetimeTzCopy.isLocalZone(), false);
    CHECK(datetimeTzCopy.isClockTime(), false);
    QCOMPARE(datetimeTzCopy.utcOffset(), 0);
    CHECKTZ(datetimeTzCopy.timeZone(), datetimeTz.timeZone());
    CHECKDT(datetimeTzCopy.dateTime(), datetimeTz.dateTime());

    // UTC
    KDateTime date_UTC(d, KDateTime::Spec::UTC());
    CHECK(date_UTC.isNull(), false);
    CHECK(date_UTC.isValid(), true);
    CHECK(date_UTC.isDateOnly(), true);
    QCOMPARE(date_UTC.timeType(), KDateTime::UTC);
    CHECK(date_UTC.isUtc(), true);
    CHECK(date_UTC.isOffsetFromUtc(), false);
    CHECK(date_UTC.isLocalZone(), false);
    CHECK(date_UTC.isClockTime(), false);
    QCOMPARE(date_UTC.utcOffset(), 0);
    CHECKTZ(date_UTC.timeZone(), KTimeZone::utc());
    CHECKDT(date_UTC.dateTime(), QDateTime(d, QTime(0,0,0))); // Qt::UTC));

    KDateTime dateTime_UTC(d, t, KDateTime::UTC);
    CHECK(dateTime_UTC.isNull(), false);
    CHECK(dateTime_UTC.isValid(), true);
    CHECK(dateTime_UTC.isDateOnly(), false);
    QCOMPARE(dateTime_UTC.timeType(), KDateTime::UTC);
    CHECK(dateTime_UTC.isUtc(), true);
    CHECK(dateTime_UTC.isOffsetFromUtc(), false);
    CHECK(dateTime_UTC.isLocalZone(), false);
    CHECK(dateTime_UTC.isClockTime(), false);
    QCOMPARE(dateTime_UTC.utcOffset(), 0);
    CHECKTZ(dateTime_UTC.timeZone(), KTimeZone::utc());
    CHECKDT(dateTime_UTC.dateTime(), QDateTime(d, t)); //, Qt::UTC));

    KDateTime datetime_UTC(dtLocal, Qt::LocalTime, KDateTime::UTC);
    CHECK(datetime_UTC.isNull(), false);
    CHECK(datetime_UTC.isValid(), true);
    CHECK(datetime_UTC.isDateOnly(), false);
    QCOMPARE(datetime_UTC.timeType(), KDateTime::UTC);
    CHECK(datetime_UTC.isUtc(), true);
    CHECK(datetime_UTC.isOffsetFromUtc(), false);
    CHECK(datetime_UTC.isLocalZone(), false);
    CHECK(datetime_UTC.isClockTime(), false);
    QCOMPARE(datetime_UTC.utcOffset(), 0);
    CHECKTZ(datetime_UTC.timeZone(), KTimeZone::utc());
    CHECKDT(datetime_UTC.dateTime(), toUTC(dtLocal));

    KDateTime datetime_UTC2(dtUTC, Qt::UTC, KDateTime::UTC);
    CHECK(datetime_UTC2.isNull(), false);
    CHECK(datetime_UTC2.isValid(), true);
    CHECK(datetime_UTC2.isDateOnly(), false);
    QCOMPARE(datetime_UTC2.timeType(), KDateTime::UTC);
    CHECK(datetime_UTC2.isUtc(), true);
    CHECK(datetime_UTC2.isOffsetFromUtc(), false);
    CHECK(datetime_UTC2.isLocalZone(), false);
    CHECK(datetime_UTC2.isClockTime(), false);
    QCOMPARE(datetime_UTC2.utcOffset(), 0);
    CHECKTZ(datetime_UTC2.timeZone(), KTimeZone::utc());
    CHECKDT(datetime_UTC2.dateTime(), dtUTC);
    CHECKD(datetime_UTC2.date(), dtUTC.date());
    CHECKT(datetime_UTC2.time(), dtUTC.time());

    // ... copy constructor
    KDateTime datetime_UTCCopy(datetime_UTC);
    CHECK(datetime_UTCCopy.isNull(), false);
    CHECK(datetime_UTCCopy.isValid(), true);
    CHECK(datetime_UTCCopy.isDateOnly(), false);
    QCOMPARE(datetime_UTCCopy.timeType(), KDateTime::UTC);
    CHECK(datetime_UTCCopy.isUtc(), true);
    CHECK(datetime_UTCCopy.isOffsetFromUtc(), false);
    CHECK(datetime_UTCCopy.isLocalZone(), false);
    CHECK(datetime_UTCCopy.isClockTime(), false);
    QCOMPARE(datetime_UTCCopy.utcOffset(), 0);
    CHECKTZ(datetime_UTCCopy.timeZone(), datetime_UTC.timeZone());
    CHECKDT(datetime_UTCCopy.dateTime(), datetime_UTC.dateTime());


    // Offset from UTC
    KDateTime date_OffsetFromUTC(d, KDateTime::Spec::OffsetFromUTC(-2*3600));
    CHECK(date_OffsetFromUTC.isNull(), false);
    CHECK(date_OffsetFromUTC.isValid(), true);
    CHECK(date_OffsetFromUTC.isDateOnly(), true);
    QCOMPARE(date_OffsetFromUTC.timeType(), KDateTime::OffsetFromUTC);
    CHECK(date_OffsetFromUTC.isUtc(), false);
    CHECK(date_OffsetFromUTC.isOffsetFromUtc(), true);
    CHECK(date_OffsetFromUTC.isLocalZone(), false);
    CHECK(date_OffsetFromUTC.isClockTime(), false);
    QCOMPARE(date_OffsetFromUTC.utcOffset(), -2*3600);
    CHECK(date_OffsetFromUTC.timeZone().isValid(), false);
    CHECKDT(date_OffsetFromUTC.dateTime(), QDateTime(d, QTime(0,0,0))); //, Qt::LocalTime));

    KDateTime dateTime_OffsetFromUTC(d, t, KDateTime::Spec::OffsetFromUTC(2*3600));
    CHECK(dateTime_OffsetFromUTC.isNull(), false);
    CHECK(dateTime_OffsetFromUTC.isValid(), true);
    CHECK(dateTime_OffsetFromUTC.isDateOnly(), false);
    QCOMPARE(dateTime_OffsetFromUTC.timeType(), KDateTime::OffsetFromUTC);
    CHECK(dateTime_OffsetFromUTC.isUtc(), false);
    CHECK(dateTime_OffsetFromUTC.isOffsetFromUtc(), true);
    CHECK(dateTime_OffsetFromUTC.isLocalZone(), false);
    CHECK(dateTime_OffsetFromUTC.isClockTime(), false);
    QCOMPARE(dateTime_OffsetFromUTC.utcOffset(), 2*3600);
    CHECK(dateTime_OffsetFromUTC.timeZone().isValid(), false);
    CHECKDT(dateTime_OffsetFromUTC.dateTime(), QDateTime(d, t)); //, Qt::LocalTime));

    KDateTime datetime_OffsetFromUTC(dtLocal, Qt::LocalTime, KDateTime::Spec::OffsetFromUTC(-2*3600));
    CHECK(datetime_OffsetFromUTC.isNull(), false);
    CHECK(datetime_OffsetFromUTC.isValid(), true);
    CHECK(datetime_OffsetFromUTC.isDateOnly(), false);
    QCOMPARE(datetime_OffsetFromUTC.timeType(), KDateTime::OffsetFromUTC);
    CHECK(datetime_OffsetFromUTC.isUtc(), false);
    CHECK(datetime_OffsetFromUTC.isOffsetFromUtc(), true);
    CHECK(datetime_OffsetFromUTC.isLocalZone(), false);
    CHECK(datetime_OffsetFromUTC.isClockTime(), false);
    QCOMPARE(datetime_OffsetFromUTC.utcOffset(), -2*3600);
    CHECK(datetime_OffsetFromUTC.timeZone().isValid(), false);
    CHECKDT(datetime_OffsetFromUTC.dateTime(), dtLocal);
    CHECKD(datetime_OffsetFromUTC.date(), dtLocal.date());
    CHECKT(datetime_OffsetFromUTC.time(), dtLocal.time());

    KDateTime datetime_OffsetFromUTC2(dtUTC, Qt::UTC, KDateTime::Spec::OffsetFromUTC(2*3600));
    CHECK(datetime_OffsetFromUTC2.isNull(), false);
    CHECK(datetime_OffsetFromUTC2.isValid(), true);
    CHECK(datetime_OffsetFromUTC2.isDateOnly(), false);
    QCOMPARE(datetime_OffsetFromUTC2.timeType(), KDateTime::OffsetFromUTC);
    CHECK(datetime_OffsetFromUTC2.isUtc(), false);
    CHECK(datetime_OffsetFromUTC2.isOffsetFromUtc(), true);
    CHECK(datetime_OffsetFromUTC2.isLocalZone(), false);
    CHECK(datetime_OffsetFromUTC2.isClockTime(), false);
    QCOMPARE(datetime_OffsetFromUTC2.utcOffset(), 2*3600);
    CHECK(datetime_OffsetFromUTC2.timeZone().isValid(), false);
    QDateTime dtof = dtUTC.addSecs(2*3600);
//    dtof.setTimeSpec(Qt::LocalTime);
    CHECKDT(datetime_OffsetFromUTC2.dateTime(), dtof);

    // ... copy constructor
    KDateTime datetime_OffsetFromUTCCopy(datetime_OffsetFromUTC);
    CHECK(datetime_OffsetFromUTCCopy.isNull(), false);
    CHECK(datetime_OffsetFromUTCCopy.isValid(), true);
    CHECK(datetime_OffsetFromUTCCopy.isDateOnly(), false);
    QCOMPARE(datetime_OffsetFromUTCCopy.timeType(), KDateTime::OffsetFromUTC);
    CHECK(datetime_OffsetFromUTCCopy.isUtc(), false);
    CHECK(datetime_OffsetFromUTCCopy.isOffsetFromUtc(), true);
    CHECK(datetime_OffsetFromUTCCopy.isLocalZone(), false);
    CHECK(datetime_OffsetFromUTCCopy.isClockTime(), false);
    QCOMPARE(datetime_OffsetFromUTCCopy.utcOffset(), -2*3600);
    CHECK(datetime_OffsetFromUTCCopy.timeZone().isValid(), false);
    CHECKDT(datetime_OffsetFromUTCCopy.dateTime(), datetime_OffsetFromUTC.dateTime());


    // Local time zone
    KDateTime date_LocalZone(d, KDateTime::Spec::LocalZone());
    CHECK(date_LocalZone.isNull(), false);
    CHECK(date_LocalZone.isValid(), true);
    CHECK(date_LocalZone.isDateOnly(), true);
    QCOMPARE(date_LocalZone.timeType(), KDateTime::TimeZone);
    CHECKSPEC(date_LocalZone.timeSpec(), KDateTime::Spec::LocalZone());
    CHECK(date_LocalZone.isUtc(), false);
    CHECK(date_LocalZone.isOffsetFromUtc(), false);
    CHECK(date_LocalZone.isLocalZone(), true);
    CHECK(date_LocalZone.isClockTime(), false);
    QCOMPARE(date_LocalZone.utcOffset(), -8*3600);
    CHECKTZ(date_LocalZone.timeZone(), KSystemTimeZones::local());
    CHECKDT(date_LocalZone.dateTime(), QDateTime(d, QTime(0,0,0))); //, Qt::LocalTime));

    KDateTime dateTime_LocalZone(d, t, KDateTime::LocalZone);
    CHECK(dateTime_LocalZone.isNull(), false);
    CHECK(dateTime_LocalZone.isValid(), true);
    CHECK(dateTime_LocalZone.isDateOnly(), false);
    QCOMPARE(dateTime_LocalZone.timeType(), KDateTime::TimeZone);
    CHECKSPEC(dateTime_LocalZone.timeSpec(), KDateTime::Spec::LocalZone());
    CHECK(dateTime_LocalZone.isUtc(), false);
    CHECK(dateTime_LocalZone.isOffsetFromUtc(), false);
    CHECK(dateTime_LocalZone.isLocalZone(), true);
    CHECK(dateTime_LocalZone.isClockTime(), false);
    QCOMPARE(dateTime_LocalZone.utcOffset(), -8*3600);
    CHECKTZ(dateTime_LocalZone.timeZone(), KSystemTimeZones::local());
    CHECKDT(dateTime_LocalZone.dateTime(), QDateTime(d, t)); //, Qt::LocalTime));

    KDateTime datetime_LocalZone(dtLocal, Qt::LocalTime, KDateTime::LocalZone);
    CHECK(datetime_LocalZone.isNull(), false);
    CHECK(datetime_LocalZone.isValid(), true);
    CHECK(datetime_LocalZone.isDateOnly(), false);
    QCOMPARE(datetime_LocalZone.timeType(), KDateTime::TimeZone);
    CHECKSPEC(datetime_LocalZone.timeSpec(), KDateTime::Spec::LocalZone());
    CHECK(datetime_LocalZone.isUtc(), false);
    CHECK(datetime_LocalZone.isOffsetFromUtc(), false);
    CHECK(datetime_LocalZone.isLocalZone(), true);
    CHECK(datetime_LocalZone.isClockTime(), false);
    QCOMPARE(datetime_LocalZone.utcOffset(), -8*3600);
    CHECKTZ(datetime_LocalZone.timeZone(), KSystemTimeZones::local());
    CHECKDT(datetime_LocalZone.dateTime(), dtLocal);
    CHECKD(datetime_LocalZone.date(), dtLocal.date());
    CHECKT(datetime_LocalZone.time(), dtLocal.time());

    KDateTime datetime_LocalZone2(dtUTC, Qt::UTC, KDateTime::LocalZone);
    CHECK(datetime_LocalZone2.isNull(), false);
    CHECK(datetime_LocalZone2.isValid(), true);
    CHECK(datetime_LocalZone2.isDateOnly(), false);
    QCOMPARE(datetime_LocalZone2.timeType(), KDateTime::TimeZone);
    CHECKSPEC(datetime_LocalZone2.timeSpec(), KDateTime::Spec::LocalZone());
    CHECK(datetime_LocalZone2.isUtc(), false);
    CHECK(datetime_LocalZone2.isOffsetFromUtc(), false);
    CHECK(datetime_LocalZone2.isLocalZone(), true);
    CHECK(datetime_LocalZone2.isClockTime(), false);
    QCOMPARE(datetime_LocalZone2.utcOffset(), -8*3600);
    CHECKTZ(datetime_LocalZone2.timeZone(), KSystemTimeZones::local());
    CHECKDT(datetime_LocalZone2.dateTime(), toLocal(dtUTC));

    // ... copy constructor
    KDateTime datetime_LocalZoneCopy(datetime_LocalZone);
    CHECK(datetime_LocalZoneCopy.isNull(), false);
    CHECK(datetime_LocalZoneCopy.isValid(), true);
    CHECK(datetime_LocalZoneCopy.isDateOnly(), false);
    QCOMPARE(datetime_LocalZoneCopy.timeType(), KDateTime::TimeZone);
    CHECKSPEC(datetime_LocalZoneCopy.timeSpec(), KDateTime::Spec::LocalZone());
    CHECK(datetime_LocalZoneCopy.isUtc(), false);
    CHECK(datetime_LocalZoneCopy.isOffsetFromUtc(), false);
    CHECK(datetime_LocalZoneCopy.isLocalZone(), true);
    CHECK(datetime_LocalZoneCopy.isClockTime(), false);
    QCOMPARE(datetime_LocalZoneCopy.utcOffset(), -8*3600);
    CHECKTZ(datetime_LocalZoneCopy.timeZone(), datetime_LocalZone.timeZone());
    CHECKDT(datetime_LocalZoneCopy.dateTime(), datetime_LocalZone.dateTime());


    // Local clock time
    KDateTime date_ClockTime(d, KDateTime::Spec::ClockTime());
    CHECK(date_ClockTime.isNull(), false);
    CHECK(date_ClockTime.isValid(), true);
    CHECK(date_ClockTime.isDateOnly(), true);
    QCOMPARE(date_ClockTime.timeType(), KDateTime::ClockTime);
    CHECK(date_ClockTime.isUtc(), false);
    CHECK(date_ClockTime.isOffsetFromUtc(), false);
    CHECK(date_ClockTime.isLocalZone(), false);
    CHECK(date_ClockTime.isClockTime(), true);
    QCOMPARE(date_ClockTime.utcOffset(), 0);
    CHECK(date_ClockTime.timeZone().isValid(), false);
    CHECKDT(date_ClockTime.dateTime(), QDateTime(d, QTime(0,0,0))); //, Qt::LocalTime));

    KDateTime dateTime_ClockTime(d, t, KDateTime::ClockTime);
    CHECK(dateTime_ClockTime.isNull(), false);
    CHECK(dateTime_ClockTime.isValid(), true);
    CHECK(dateTime_ClockTime.isDateOnly(), false);
    QCOMPARE(dateTime_ClockTime.timeType(), KDateTime::ClockTime);
    CHECK(dateTime_ClockTime.isUtc(), false);
    CHECK(dateTime_ClockTime.isOffsetFromUtc(), false);
    CHECK(dateTime_ClockTime.isLocalZone(), false);
    CHECK(dateTime_ClockTime.isClockTime(), true);
    QCOMPARE(dateTime_ClockTime.utcOffset(), 0);
    CHECK(dateTime_ClockTime.timeZone().isValid(), false);
    CHECKDT(dateTime_ClockTime.dateTime(), QDateTime(d, t)); //, Qt::LocalTime));

    KDateTime datetime_ClockTime(dtLocal, Qt::LocalTime, KDateTime::ClockTime);
    CHECK(datetime_ClockTime.isNull(), false);
    CHECK(datetime_ClockTime.isValid(), true);
    CHECK(datetime_ClockTime.isDateOnly(), false);
    QCOMPARE(datetime_ClockTime.timeType(), KDateTime::ClockTime);
    CHECK(datetime_ClockTime.isUtc(), false);
    CHECK(datetime_ClockTime.isOffsetFromUtc(), false);
    CHECK(datetime_ClockTime.isLocalZone(), false);
    CHECK(datetime_ClockTime.isClockTime(), true);
    QCOMPARE(datetime_ClockTime.utcOffset(), 0);
    CHECK(datetime_ClockTime.timeZone().isValid(), false);
    CHECKDT(datetime_ClockTime.dateTime(), dtLocal);
    CHECKD(datetime_ClockTime.date(), dtLocal.date());
    CHECKT(datetime_ClockTime.time(), dtLocal.time());

    KDateTime datetime_ClockTime2(dtUTC, Qt::UTC, KDateTime::ClockTime);
    CHECK(datetime_ClockTime2.isNull(), false);
    CHECK(datetime_ClockTime2.isValid(), true);
    CHECK(datetime_ClockTime2.isDateOnly(), false);
    QCOMPARE(datetime_ClockTime2.timeType(), KDateTime::ClockTime);
    CHECK(datetime_ClockTime2.isUtc(), false);
    CHECK(datetime_ClockTime2.isOffsetFromUtc(), false);
    CHECK(datetime_ClockTime2.isLocalZone(), false);
    CHECK(datetime_ClockTime2.isClockTime(), true);
    QCOMPARE(datetime_ClockTime2.utcOffset(), 0);
    CHECK(datetime_ClockTime2.timeZone().isValid(), false);
    CHECKDT(datetime_ClockTime2.dateTime(), toLocal(dtUTC));

    // ... copy constructor
    KDateTime datetime_ClockTimeCopy(datetime_ClockTime);
    CHECK(datetime_ClockTimeCopy.isNull(), false);
    CHECK(datetime_ClockTimeCopy.isValid(), true);
    CHECK(datetime_ClockTimeCopy.isDateOnly(), false);
    QCOMPARE(datetime_ClockTimeCopy.timeType(), KDateTime::ClockTime);
    CHECK(datetime_ClockTimeCopy.isUtc(), false);
    CHECK(datetime_ClockTimeCopy.isOffsetFromUtc(), false);
    CHECK(datetime_ClockTimeCopy.isLocalZone(), false);
    CHECK(datetime_ClockTimeCopy.isClockTime(), true);
    QCOMPARE(datetime_ClockTimeCopy.utcOffset(), 0);
    CHECK(datetime_ClockTimeCopy.timeZone().isValid(), false);
    CHECKDT(datetime_ClockTimeCopy.dateTime(), datetime_ClockTime.dateTime());


    // Invalid time zone specification for a constructor
    KDateTime date_TimeZone(d, KDateTime::TimeZone);
    CHECK(date_TimeZone.isValid(), false);
    KDateTime dateTime_TimeZone(d, t, KDateTime::TimeZone);
    CHECK(dateTime_TimeZone.isValid(), false);
    KDateTime datetime_TimeZone(dtLocal, Qt::LocalTime, KDateTime::TimeZone);
    CHECK(datetime_TimeZone.isValid(), false);
    KDateTime datetime_Invalid(dtLocal, Qt::LocalTime, KDateTime::Invalid);
    CHECK(datetime_Invalid.isValid(), false);

    // Restore the original local time zone
    if (!originalZone)
        ::unsetenv("TZ");
    else
        ::setenv("TZ", originalZone, 1);
    ::tzset();
}

///////////////////////////////////
// Time conversion and operator==()
///////////////////////////////////

void KDateTimeTest::toUtc()
{
    KTimeZone london = KSystemTimeZones::zone("Europe/London");

    // Ensure that local time is different from UTC and different from 'london'
    const char *originalZone = ::getenv("TZ");   // save the original local time zone
    ::setenv("TZ", ":America/Los_Angeles", 1);
    ::tzset();

    // Zone -> UTC
    KDateTime londonWinter(QDate(2005,1,1), QTime(0,0,0), london);
    KDateTime utcWinter = londonWinter.toUtc();
    CHECK(utcWinter.isUtc(), true);
    CHECKDT(utcWinter.dateTime(), QDateTime(QDate(2005,1,1), QTime(0,0,0))); //, Qt::UTC));
    CHECK(londonWinter == utcWinter, true);
    KDateTime londonSummer(QDate(2005,6,1), QTime(0,0,0), london);
    KDateTime utcSummer = londonSummer.toUtc();
    CHECK(utcSummer.isUtc(), true);
    CHECKDT(utcSummer.dateTime(), QDateTime(QDate(2005,5,31), QTime(23,0,0))); //, Qt::UTC));
    CHECK(londonSummer == utcSummer, true);
    CHECK((londonSummer == utcWinter), false);
    CHECK((londonWinter == utcSummer), false);

    // UTC offset -> UTC
    KDateTime offset(QDate(2005,6,6), QTime(1,2,30), KDateTime::Spec::OffsetFromUTC(-5400));  // -0130
    KDateTime utcOffset = offset.toUtc();
    CHECK(utcOffset.isUtc(), true);
    CHECKDT(utcOffset.dateTime(), QDateTime(QDate(2005,6,6), QTime(2,32,30))); //, Qt::UTC));
    CHECK(offset == utcOffset, true);
    CHECK((offset == utcSummer), false);

    // Clock time -> UTC
    KDateTime clock(QDate(2005,6,6), QTime(1,2,30), KDateTime::ClockTime);
    KDateTime utcClock = clock.toUtc();
    CHECK(utcClock.isUtc(), true);
    CHECKDT(utcClock.dateTime(), QDateTime(QDate(2005,6,6), QTime(8,2,30))); //, Qt::UTC));
    CHECK(clock == utcClock, true);
    CHECK((clock == utcOffset), false);

    // UTC -> UTC
    KDateTime utc(QDate(2005,6,6), QTime(1,2,30), KDateTime::UTC);
    KDateTime utcUtc = utc.toUtc();
    CHECK(utcUtc.isUtc(), true);
    CHECKDT(utcUtc.dateTime(), QDateTime(QDate(2005,6,6), QTime(1,2,30))); //, Qt::UTC));
    CHECK(utc == utcUtc, true);
    CHECK((utc == utcClock), false);

    // ** Date only ** //

    // Zone -> UTC
    londonSummer.setDateOnly(true);
    utcSummer = londonSummer.toUtc();
    CHECK(utcSummer.isDateOnly(), true);
    CHECKDT(utcSummer.dateTime(), QDateTime(QDate(2005,6,1), QTime(0,0,0))); //, Qt::UTC));
    CHECK(utcSummer != londonSummer, true);
    CHECK((utcSummer == londonSummer), false);
    londonWinter.setDateOnly(true);
    utcWinter = londonWinter.toUtc();
    CHECK(utcWinter == londonWinter, true);
    CHECK((utcWinter != londonWinter), false);

    // UTC offset -> UTC
    offset.setDateOnly(true);
    utcOffset = offset.toUtc();
    CHECK(utcOffset.isDateOnly(), true);
    CHECKDT(utcOffset.dateTime(), QDateTime(QDate(2005,6,6), QTime(0,0,0))); //, Qt::UTC));
    CHECK(offset != utcOffset, true);
    CHECK((offset == utcOffset), false);
    KDateTime utcOffset1(QDate(2005,6,6), KDateTime::Spec::OffsetFromUTC(0));
    CHECK(utcOffset1 == utcOffset1.toUtc(), true);

    // Clock time -> UTC
    clock.setDateOnly(true);
    utcClock = clock.toUtc();
    CHECK(utcClock.isDateOnly(), true);
    CHECKDT(utcClock.dateTime(), QDateTime(QDate(2005,6,6), QTime(0,0,0))); //, Qt::UTC));
    CHECK(clock != utcClock, true);
    CHECK((clock == utcClock), false);

    // UTC -> UTC
    utc.setDateOnly(true);
    utcUtc = utc.toUtc();
    CHECK(utcUtc.isDateOnly(), true);
    CHECKDT(utcUtc.dateTime(), QDateTime(QDate(2005,6,6), QTime(0,0,0))); //, Qt::UTC));
    CHECK(utc == utcUtc, true);
    CHECK((utc != utcUtc), false);

    // Restore the original local time zone
    if (!originalZone)
        ::unsetenv("TZ");
    else
        ::setenv("TZ", originalZone, 1);
    ::tzset();
}

void KDateTimeTest::toOffsetFromUtc()
{
    KTimeZone london = KSystemTimeZones::zone("Europe/London");

    // Ensure that local time is different from UTC and different from 'london'
    const char *originalZone = ::getenv("TZ");   // save the original local time zone
    ::setenv("TZ", ":America/Los_Angeles", 1);
    ::tzset();

    // ***** toOffsetFromUtc(void) *****

    // Zone -> UTC offset
    KDateTime londonWinter(QDate(2005,1,1), QTime(2,0,0), london);
    KDateTime offsetWinter = londonWinter.toOffsetFromUtc();
    CHECK(offsetWinter.isOffsetFromUtc(), true);
    QCOMPARE(offsetWinter.utcOffset(), 0);
    CHECKDT(offsetWinter.dateTime(), QDateTime(QDate(2005,1,1), QTime(2,0,0))); //, Qt::LocalTime));
    CHECK(londonWinter == offsetWinter, true);
    KDateTime londonSummer(QDate(2005,6,1), QTime(14,0,0), london);
    KDateTime offsetSummer = londonSummer.toOffsetFromUtc();
    CHECK(offsetSummer.isOffsetFromUtc(), true);
    QCOMPARE(offsetSummer.utcOffset(), 3600);
    CHECKDT(offsetSummer.dateTime(), QDateTime(QDate(2005,6,1), QTime(14,0,0))); //, Qt::LocalTime));
    CHECK(londonSummer == offsetSummer, true);
    CHECK((londonSummer == offsetWinter), false);
    CHECK((londonWinter == offsetSummer), false);

    // UTC offset -> UTC offset
    KDateTime offset(QDate(2005,6,6), QTime(11,2,30), KDateTime::Spec::OffsetFromUTC(-5400));  // -0130
    KDateTime offsetOffset = offset.toOffsetFromUtc();
    CHECK(offsetOffset.isOffsetFromUtc(), true);
    QCOMPARE(offsetOffset.utcOffset(), -5400);
    CHECKDT(offsetOffset.dateTime(), QDateTime(QDate(2005,6,6), QTime(11,2,30))); //, Qt::LocalTime));
    CHECK(offset == offsetOffset, true);
    CHECK((offset == offsetSummer), false);

    // Clock time -> UTC offset
    KDateTime clock(QDate(2005,6,6), QTime(1,2,30), KDateTime::ClockTime);
    KDateTime offsetClock = clock.toOffsetFromUtc();
    CHECK(offsetClock.isOffsetFromUtc(), true);
    QCOMPARE(offsetClock.utcOffset(), -7*3600);
    CHECKDT(offsetClock.dateTime(), QDateTime(QDate(2005,6,6), QTime(1,2,30))); //, Qt::LocalTime));
    CHECK(clock == offsetClock, true);
    CHECK((clock == offsetOffset), false);

    // UTC -> UTC offset
    KDateTime utc(QDate(2005,6,6), QTime(11,2,30), KDateTime::UTC);
    KDateTime offsetUtc = utc.toOffsetFromUtc();
    CHECK(offsetUtc.isOffsetFromUtc(), true);
    QCOMPARE(offsetUtc.utcOffset(), 0);
    CHECKDT(offsetUtc.dateTime(), QDateTime(QDate(2005,6,6), QTime(11,2,30))); //, Qt::LocalTime));
    CHECK(utc == offsetUtc, true);
    CHECK((utc == offsetClock), false);

    // ** Date only ** //

    // Zone -> UTC offset
    londonSummer.setDateOnly(true);
    offsetSummer = londonSummer.toOffsetFromUtc();
    CHECK(offsetSummer.isDateOnly(), true);
    CHECK(offsetSummer.isOffsetFromUtc(), true);
    QCOMPARE(offsetSummer.utcOffset(), 3600);
    CHECKDT(offsetSummer.dateTime(), QDateTime(QDate(2005,6,1), QTime(0,0,0))); //, Qt::LocalTime));
    CHECK(offsetSummer == londonSummer, true);
    CHECK((offsetSummer != londonSummer), false);
    londonWinter.setDateOnly(true);
    offsetWinter = londonWinter.toUtc();
    CHECK(offsetWinter == londonWinter, true);
    CHECK((offsetWinter != londonWinter), false);

    // UTC offset -> UTC offset
    offset.setDateOnly(true);
    offsetOffset = offset.toOffsetFromUtc();
    CHECK(offsetOffset.isDateOnly(), true);
    CHECK(offsetOffset.isOffsetFromUtc(), true);
    QCOMPARE(offsetOffset.utcOffset(), -5400);
    CHECKDT(offsetOffset.dateTime(), QDateTime(QDate(2005,6,6), QTime(0,0,0))); //, Qt::LocalTime));
    CHECK(offset == offsetOffset, true);
    CHECK((offset != offsetOffset), false);

    // Clock time -> UTC offset
    clock.setDateOnly(true);
    offsetClock = clock.toOffsetFromUtc();
    CHECK(offsetClock.isDateOnly(), true);
    CHECK(offsetClock.isOffsetFromUtc(), true);
    QCOMPARE(offsetClock.utcOffset(), -7*3600);
    CHECKDT(offsetClock.dateTime(), QDateTime(QDate(2005,6,6), QTime(0,0,0))); //, Qt::LocalTime));
    CHECK(clock == offsetClock, true);
    CHECK((clock != offsetClock), false);

    // UTC -> UTC offset
    utc.setDateOnly(true);
    offsetUtc = utc.toOffsetFromUtc();
    CHECK(offsetUtc.isDateOnly(), true);
    CHECK(offsetUtc.isOffsetFromUtc(), true);
    QCOMPARE(offsetUtc.utcOffset(), 0);
    CHECKDT(offsetUtc.dateTime(), QDateTime(QDate(2005,6,6), QTime(0,0,0))); //, Qt::LocalTime));
    CHECK(utc == offsetUtc, true);
    CHECK((utc != offsetUtc), false);

    // ***** toOffsetFromUtc(int utcOffset) *****

    // Zone -> UTC offset
    KDateTime londonWinter2(QDate(2005,1,1), QTime(2,0,0), london);
    offsetWinter = londonWinter2.toOffsetFromUtc(5400);    // +1H30M
    CHECK(offsetWinter.isOffsetFromUtc(), true);
    QCOMPARE(offsetWinter.utcOffset(), 5400);
    CHECKDT(offsetWinter.dateTime(), QDateTime(QDate(2005,1,1), QTime(3,30,0))); //, Qt::LocalTime));
    CHECK(londonWinter2 == offsetWinter, true);
    KDateTime londonSummer2(QDate(2005,6,1), QTime(14,0,0), london);
    offsetSummer = londonSummer2.toOffsetFromUtc(5400);
    CHECK(offsetSummer.isOffsetFromUtc(), true);
    QCOMPARE(offsetSummer.utcOffset(), 5400);
    CHECKDT(offsetSummer.dateTime(), QDateTime(QDate(2005,6,1), QTime(14,30,0))); //, Qt::LocalTime));
    CHECK(londonSummer2 == offsetSummer, true);
    CHECK((londonSummer2 == offsetWinter), false);
    CHECK((londonWinter2 == offsetSummer), false);

    // UTC offset -> UTC offset
    KDateTime offset2(QDate(2005,6,6), QTime(11,2,30), KDateTime::Spec::OffsetFromUTC(-5400));  // -0130
    offsetOffset = offset2.toOffsetFromUtc(3600);
    CHECK(offsetOffset.isOffsetFromUtc(), true);
    QCOMPARE(offsetOffset.utcOffset(), 3600);
    CHECKDT(offsetOffset.dateTime(), QDateTime(QDate(2005,6,6), QTime(13,32,30))); //, Qt::LocalTime));
    CHECK(offset2 == offsetOffset, true);
    CHECK((offset2 == offsetSummer), false);

    // Clock time -> UTC offset
    KDateTime clock2(QDate(2005,6,6), QTime(1,2,30), KDateTime::ClockTime);
    offsetClock = clock2.toOffsetFromUtc(0);
    CHECK(offsetClock.isOffsetFromUtc(), true);
    QCOMPARE(offsetClock.utcOffset(), 0);
    CHECKDT(offsetClock.dateTime(), QDateTime(QDate(2005,6,6), QTime(8,2,30))); //, Qt::LocalTime));
    CHECK(clock2 == offsetClock, true);
    CHECK((clock2 == offsetOffset), false);

    // UTC -> UTC offset
    KDateTime utc2(QDate(2005,6,6), QTime(11,2,30), KDateTime::UTC);
    offsetUtc = utc2.toOffsetFromUtc(-3600);
    CHECK(offsetUtc.isOffsetFromUtc(), true);
    QCOMPARE(offsetUtc.utcOffset(), -3600);
    CHECKDT(offsetUtc.dateTime(), QDateTime(QDate(2005,6,6), QTime(10,2,30))); //, Qt::LocalTime));
    CHECK(utc2 == offsetUtc, true);
    CHECK((utc2 == offsetClock), false);

    // ** Date only ** //

    // Zone -> UTC offset
    londonSummer2.setDateOnly(true);
    offsetSummer = londonSummer2.toOffsetFromUtc(5400);
    CHECK(offsetSummer.isDateOnly(), true);
    CHECK(offsetSummer.isOffsetFromUtc(), true);
    QCOMPARE(offsetSummer.utcOffset(), 5400);
    CHECKDT(offsetSummer.dateTime(), QDateTime(QDate(2005,6,1), QTime(0,0,0))); //, Qt::LocalTime));
    CHECK(londonSummer2 != offsetSummer, true);
    CHECK((londonSummer2 == offsetSummer), false);
    CHECK(londonSummer2 == KDateTime(QDate(2005,6,1), KDateTime::Spec::OffsetFromUTC(3600)), true);

    // UTC offset -> UTC offset
    offset2.setDateOnly(true);
    offsetOffset = offset2.toOffsetFromUtc(-3600);
    CHECK(offsetOffset.isDateOnly(), true);
    CHECK(offsetOffset.isOffsetFromUtc(), true);
    QCOMPARE(offsetOffset.utcOffset(), -3600);
    CHECKDT(offsetOffset.dateTime(), QDateTime(QDate(2005,6,6), QTime(0,0,0))); //, Qt::LocalTime));
    CHECK(offset2 != offsetOffset, true);
    CHECK((offset2 == offsetOffset), false);

    // Clock time -> UTC offset
    clock2.setDateOnly(true);
    offsetClock = clock2.toOffsetFromUtc(6*3600);
    CHECK(offsetClock.isDateOnly(), true);
    CHECK(offsetClock.isOffsetFromUtc(), true);
    QCOMPARE(offsetClock.utcOffset(), 6*3600);
    CHECKDT(offsetClock.dateTime(), QDateTime(QDate(2005,6,6), QTime(0,0,0))); //, Qt::LocalTime));
    CHECK(clock2 != offsetClock, true);
    CHECK((clock2 == offsetClock), false);
    CHECK(clock == KDateTime(QDate(2005,6,6), KDateTime::Spec::OffsetFromUTC(-7*3600)), true);

    // UTC -> UTC offset
    utc2.setDateOnly(true);
    offsetUtc = utc2.toOffsetFromUtc(1800);
    CHECK(offsetUtc.isDateOnly(), true);
    CHECK(offsetUtc.isOffsetFromUtc(), true);
    QCOMPARE(offsetUtc.utcOffset(), 1800);
    CHECKDT(offsetUtc.dateTime(), QDateTime(QDate(2005,6,6), QTime(0,0,0))); //, Qt::LocalTime));
    CHECK(utc2 != offsetUtc, true);
    CHECK((utc2 == offsetUtc), false);
    CHECK(utc2 == KDateTime(QDate(2005,6,6), KDateTime::Spec::OffsetFromUTC(0)), true);

    // Restore the original local time zone
    if (!originalZone)
        ::unsetenv("TZ");
    else
        ::setenv("TZ", originalZone, 1);
    ::tzset();
}

void KDateTimeTest::toLocalZone()
{
    KTimeZone london = KSystemTimeZones::zone("Europe/London");

    // Ensure that local time is different from UTC and different from 'london'
    const char *originalZone = ::getenv("TZ");   // save the original local time zone
    ::setenv("TZ", ":America/Los_Angeles", 1);
    ::tzset();

    // Zone -> LocalZone
    KDateTime londonWinter(QDate(2005,1,1), QTime(0,0,0), london);
    KDateTime locWinter = londonWinter.toLocalZone();
    CHECK(locWinter.isLocalZone(), true);
    CHECKDT(locWinter.dateTime(), QDateTime(QDate(2004,12,31), QTime(16,0,0))); //, Qt::LocalTime));
    CHECK(londonWinter == locWinter, true);
    KDateTime londonSummer(QDate(2005,6,1), QTime(0,0,0), london);
    KDateTime locSummer = londonSummer.toLocalZone();
    CHECK(locSummer.isLocalZone(), true);
    CHECKDT(locSummer.dateTime(), QDateTime(QDate(2005,5,31), QTime(16,0,0))); //, Qt::LocalTime));
    CHECK(londonSummer == locSummer, true);
    CHECK((londonSummer == locWinter), false);
    CHECK((londonWinter == locSummer), false);

    // UTC offset -> LocalZone
    KDateTime offset(QDate(2005,6,6), QTime(11,2,30), KDateTime::Spec::OffsetFromUTC(-5400));  // -0130
    KDateTime locOffset = offset.toLocalZone();
    CHECK(locOffset.isLocalZone(), true);
    CHECKDT(locOffset.dateTime(), QDateTime(QDate(2005,6,6), QTime(5,32,30))); //, Qt::LocalTime));
    CHECK(offset == locOffset, true);
    CHECK((offset == locSummer), false);

    // Clock time -> LocalZone
    KDateTime clock(QDate(2005,6,6), QTime(1,2,30), KDateTime::ClockTime);
    KDateTime locClock = clock.toLocalZone();
    CHECK(locClock.isLocalZone(), true);
    CHECKDT(locClock.dateTime(), QDateTime(QDate(2005,6,6), QTime(1,2,30))); //, Qt::LocalTime));
    CHECK(clock == locClock, true);
    CHECK((clock == locOffset), false);

    // UTC -> LocalZone
    KDateTime utc(QDate(2005,6,6), QTime(11,2,30), KDateTime::UTC);
    KDateTime locUtc = utc.toLocalZone();
    CHECK(locUtc.isLocalZone(), true);
    CHECKDT(locUtc.dateTime(), QDateTime(QDate(2005,6,6), QTime(4,2,30))); //, Qt::LocalTime));
    CHECK(utc == locUtc, true);
    CHECK((utc == locClock), false);

    // ** Date only ** //

    // Zone -> LocalZone
    londonSummer.setDateOnly(true);
    locSummer = londonSummer.toLocalZone();
    CHECK(locSummer.isDateOnly(), true);
    CHECKDT(locSummer.dateTime(), QDateTime(QDate(2005,6,1), QTime(0,0,0))); //, Qt::LocalTime));
    CHECK(londonSummer != locSummer, true);
    CHECK((londonSummer == locSummer), false);

    // UTC offset -> LocalZone
    offset.setDateOnly(true);
    locOffset = offset.toLocalZone();
    CHECK(locOffset.isDateOnly(), true);
    CHECKDT(locOffset.dateTime(), QDateTime(QDate(2005,6,6), QTime(0,0,0))); //, Qt::LocalTime));
    CHECK(offset != locOffset, true);
    CHECK((offset == locOffset), false);
    CHECK(locOffset == KDateTime(QDate(2005,6,6), KDateTime::Spec::OffsetFromUTC(-7*3600)), true);

    // Clock time -> LocalZone
    clock.setDateOnly(true);
    locClock = clock.toLocalZone();
    CHECK(locClock.isDateOnly(), true);
    CHECKDT(locClock.dateTime(), QDateTime(QDate(2005,6,6), QTime(0,0,0))); //, Qt::LocalTime));
    CHECK(clock == locClock, true);
    CHECK((clock != locClock), false);

    // UTC -> LocalZone
    utc.setDateOnly(true);
    locUtc = utc.toLocalZone();
    CHECK(locUtc.isDateOnly(), true);
    CHECKDT(locUtc.dateTime(), QDateTime(QDate(2005,6,6), QTime(0,0,0))); //, Qt::LocalTime));
    CHECK(utc != locUtc, true);
    CHECK((utc == locUtc), false);

    // Restore the original local time zone
    if (!originalZone)
        ::unsetenv("TZ");
    else
        ::setenv("TZ", originalZone, 1);
    ::tzset();
}

void KDateTimeTest::toClockTime()
{
    KTimeZone london = KSystemTimeZones::zone("Europe/London");

    // Ensure that local time is different from UTC and different from 'london'
    const char *originalZone = ::getenv("TZ");   // save the original local time zone
    ::setenv("TZ", ":America/Los_Angeles", 1);
    ::tzset();

    // Zone -> ClockTime
    KDateTime londonWinter(QDate(2005,1,1), QTime(0,0,0), london);
    KDateTime locWinter = londonWinter.toClockTime();
    CHECK(locWinter.isClockTime(), true);
    CHECKDT(locWinter.dateTime(), QDateTime(QDate(2004,12,31), QTime(16,0,0))); //, Qt::LocalTime));
    CHECK(londonWinter == locWinter, true);
    KDateTime londonSummer(QDate(2005,6,1), QTime(0,0,0), london);
    KDateTime locSummer = londonSummer.toClockTime();
    CHECK(locSummer.isClockTime(), true);
    CHECKDT(locSummer.dateTime(), QDateTime(QDate(2005,5,31), QTime(16,0,0))); //, Qt::LocalTime));
    CHECK(londonSummer == locSummer, true);
    CHECK((londonSummer == locWinter), false);
    CHECK((londonWinter == locSummer), false);

    // UTC offset -> ClockTime
    KDateTime offset(QDate(2005,6,6), QTime(11,2,30), KDateTime::Spec::OffsetFromUTC(-5400));  // -0130
    KDateTime locOffset = offset.toClockTime();
    CHECK(locOffset.isClockTime(), true);
    CHECKDT(locOffset.dateTime(), QDateTime(QDate(2005,6,6), QTime(5,32,30))); //, Qt::LocalTime));
    CHECK(offset == locOffset, true);
    CHECK((offset == locSummer), false);

    // Clock time -> ClockTime
    KDateTime clock(QDate(2005,6,6), QTime(1,2,30), KDateTime::ClockTime);
    KDateTime locClock = clock.toClockTime();
    CHECK(locClock.isClockTime(), true);
    CHECKDT(locClock.dateTime(), QDateTime(QDate(2005,6,6), QTime(1,2,30))); //, Qt::LocalTime));
    CHECK(clock == locClock, true);
    CHECK((clock == locOffset), false);

    // UTC -> ClockTime
    KDateTime utc(QDate(2005,6,6), QTime(11,2,30), KDateTime::UTC);
    KDateTime locUtc = utc.toClockTime();
    CHECK(locUtc.isClockTime(), true);
    CHECKDT(locUtc.dateTime(), QDateTime(QDate(2005,6,6), QTime(4,2,30))); //, Qt::LocalTime));
    CHECK(utc == locUtc, true);
    CHECK((utc == locClock), false);

    // ** Date only ** //

    // Zone -> ClockTime
    londonSummer.setDateOnly(true);
    locSummer = londonSummer.toClockTime();
    CHECK(locSummer.isDateOnly(), true);
    CHECKDT(locSummer.dateTime(), QDateTime(QDate(2005,6,1), QTime(0,0,0))); //, Qt::LocalTime));
    CHECK(londonSummer != locSummer, true);
    CHECK((londonSummer == locSummer), false);

    // UTC offset -> ClockTime
    offset.setDateOnly(true);
    locOffset = offset.toClockTime();
    CHECK(locOffset.isDateOnly(), true);
    CHECKDT(locOffset.dateTime(), QDateTime(QDate(2005,6,6), QTime(0,0,0))); //, Qt::LocalTime));
    CHECK(offset != locOffset, true);
    CHECK((offset == locOffset), false);
    CHECK(locOffset == KDateTime(QDate(2005,6,6), KDateTime::Spec::OffsetFromUTC(-7*3600)), true);

    // Clock time -> ClockTime
    clock.setDateOnly(true);
    locClock = clock.toClockTime();
    CHECK(locClock.isDateOnly(), true);
    CHECKDT(locClock.dateTime(), QDateTime(QDate(2005,6,6), QTime(0,0,0))); //, Qt::LocalTime));
    CHECK(clock == locClock, true);
    CHECK((clock != locClock), false);

    // UTC -> ClockTime
    utc.setDateOnly(true);
    locUtc = utc.toClockTime();
    CHECK(locUtc.isDateOnly(), true);
    CHECKDT(locUtc.dateTime(), QDateTime(QDate(2005,6,6), QTime(0,0,0))); //, Qt::LocalTime));
    CHECK(utc != locUtc, true);
    CHECK((utc == locUtc), false);

    // Restore the original local time zone
    if (!originalZone)
        ::unsetenv("TZ");
    else
        ::setenv("TZ", originalZone, 1);
    ::tzset();
}

void KDateTimeTest::toZone()
{
    KTimeZone london = KSystemTimeZones::zone("Europe/London");
    KTimeZone losAngeles = KSystemTimeZones::zone("America/Los_Angeles");

    // Ensure that local time is different from UTC and different from 'london'
    const char *originalZone = ::getenv("TZ");   // save the original local time zone
    ::setenv("TZ", ":Europe/London", 1);
    ::tzset();

    // Zone -> Zone
    KDateTime londonWinter(QDate(2005,1,1), QTime(0,0,0), london);
    KDateTime locWinter = londonWinter.toZone(losAngeles);
    CHECKTZ(locWinter.timeZone(), losAngeles);
    CHECKDT(locWinter.dateTime(), QDateTime(QDate(2004,12,31), QTime(16,0,0))); //, Qt::LocalTime));
    CHECK(londonWinter == locWinter, true);
    KDateTime londonSummer(QDate(2005,6,1), QTime(0,0,0), london);
    KDateTime locSummer = londonSummer.toZone(losAngeles);
    CHECKTZ(locWinter.timeZone(), losAngeles);
    CHECKDT(locSummer.dateTime(), QDateTime(QDate(2005,5,31), QTime(16,0,0))); //, Qt::LocalTime));
    CHECK(londonSummer == locSummer, true);
    CHECK((londonSummer == locWinter), false);
    CHECK((londonWinter == locSummer), false);

    // UTC offset -> Zone
    KDateTime offset(QDate(2005,6,6), QTime(11,2,30), KDateTime::Spec::OffsetFromUTC(-5400));  // -0130
    KDateTime locOffset = offset.toZone(losAngeles);
    CHECKTZ(locOffset.timeZone(), losAngeles);
    CHECKDT(locOffset.dateTime(), QDateTime(QDate(2005,6,6), QTime(5,32,30))); //, Qt::LocalTime));
    CHECK(offset == locOffset, true);
    CHECK((offset == locSummer), false);

    // Clock time -> Zone
    KDateTime clock(QDate(2005,6,6), QTime(3,2,30), KDateTime::ClockTime);
    KDateTime locClock = clock.toZone(london);
    CHECKTZ(locClock.timeZone(), london);
    CHECKDT(locClock.dateTime(), QDateTime(QDate(2005,6,6), QTime(11,2,30))); //, Qt::LocalTime));
    CHECK(clock == locClock, true);
    CHECK((clock == locOffset), false);

    // UTC -> Zone
    KDateTime utc(QDate(2005,6,6), QTime(11,2,30), KDateTime::UTC);
    KDateTime locUtc = utc.toZone(losAngeles);
    CHECKTZ(locUtc.timeZone(), losAngeles);
    CHECKDT(locUtc.dateTime(), QDateTime(QDate(2005,6,6), QTime(4,2,30))); //, Qt::LocalTime));
    CHECK(utc == locUtc, true);
    CHECK((utc == locClock), false);

    // ** Date only ** //

    // Zone -> Zone
    londonSummer.setDateOnly(true);
    locSummer = londonSummer.toZone(losAngeles);
    CHECK(locSummer.isDateOnly(), true);
    CHECKDT(locSummer.dateTime(), QDateTime(QDate(2005,6,1), QTime(0,0,0))); //, Qt::LocalTime));
    CHECK(londonSummer != locSummer, true);
    CHECK((londonSummer == locSummer), false);

    // UTC offset -> Zone
    offset.setDateOnly(true);
    locOffset = offset.toZone(losAngeles);
    CHECK(locOffset.isDateOnly(), true);
    CHECKDT(locOffset.dateTime(), QDateTime(QDate(2005,6,6), QTime(0,0,0))); //, Qt::LocalTime));
    CHECK(offset != locOffset, true);
    CHECK((offset == locOffset), false);

    // Clock time -> Zone
    clock.setDateOnly(true);
    locClock = clock.toZone(london);
    CHECK(locClock.isDateOnly(), true);
    CHECKDT(locClock.dateTime(), QDateTime(QDate(2005,6,6), QTime(0,0,0))); //, Qt::LocalTime));
    CHECK(clock != locClock, true);
    CHECK((clock == locClock), false);

    // UTC -> Zone
    utc.setDateOnly(true);
    locUtc = utc.toZone(losAngeles);
    CHECK(locUtc.isDateOnly(), true);
    CHECKDT(locUtc.dateTime(), QDateTime(QDate(2005,6,6), QTime(0,0,0))); //, Qt::LocalTime));
    CHECK(utc != locUtc, true);
    CHECK((utc == locUtc), false);

    // Restore the original local time zone
    if (!originalZone)
        ::unsetenv("TZ");
    else
        ::setenv("TZ", originalZone, 1);
    ::tzset();
}

void KDateTimeTest::toTimeSpec()
{
    KTimeZone london = KSystemTimeZones::zone("Europe/London");
    KTimeZone cairo  = KSystemTimeZones::zone("Africa/Cairo");

    // Ensure that local time is different from UTC and different from 'london'
    const char *originalZone = ::getenv("TZ");   // save the original local time zone
    ::setenv("TZ", ":America/Los_Angeles", 1);
    ::tzset();

    KDateTime::Spec utcSpec(KDateTime::UTC);
    KDateTime::Spec cairoSpec(cairo);
    KDateTime::Spec offset1200Spec(KDateTime::OffsetFromUTC, 1200);
    KDateTime::Spec clockSpec(KDateTime::ClockTime);

    KDateTime utc1(QDate(2004,3,1), QTime(3,45,2), KDateTime::UTC);
    KDateTime zone1(QDate(2004,3,1), QTime(3,45,2), cairo);
    KDateTime offset1(QDate(2004,3,1), QTime(3,45,2), KDateTime::Spec::OffsetFromUTC(1200));    // +00:20
    KDateTime clock1(QDate(2004,3,1), QTime(3,45,2), KDateTime::ClockTime);

    KDateTime utc(QDate(2005,6,6), QTime(1,2,30), KDateTime::UTC);
    KDateTime zone(QDate(2005,7,1), QTime(2,0,0), london);
    KDateTime offset(QDate(2005,6,6), QTime(1,2,30), KDateTime::Spec::OffsetFromUTC(-5400));  // -01:30
    KDateTime clock(QDate(2005,6,6), QTime(1,2,30), KDateTime::ClockTime);

    // To UTC
    KDateTime utcZone = zone.toTimeSpec(utcSpec);
    CHECK(utcZone.isUtc(), true);
    CHECK(utcZone == KDateTime(QDate(2005,7,1), QTime(1,0,0), KDateTime::UTC), true);
    CHECK(zone.timeSpec() != utcSpec, true);
    CHECK(utcZone.timeSpec() == utcSpec, true);

    KDateTime utcOffset = offset.toTimeSpec(utcSpec);
    CHECK(utcOffset.isUtc(), true);
    CHECK(utcOffset == KDateTime(QDate(2005,6,6), QTime(2,32,30), KDateTime::UTC), true);
    CHECK(offset.timeSpec() != utcSpec, true);
    CHECK(utcOffset.timeSpec() == utcSpec, true);

    KDateTime utcClock = clock.toTimeSpec(utcSpec);
    CHECK(utcClock.isUtc(), true);
    CHECK(utcClock == KDateTime(QDate(2005,6,6), QTime(8,2,30), KDateTime::UTC), true);
    CHECK(clock.timeSpec() != utcSpec, true);
    CHECK(utcZone.timeSpec() == utcSpec, true);

    KDateTime utcUtc = utc.toTimeSpec(utcSpec);
    CHECK(utcUtc.isUtc(), true);
    CHECK(utcUtc == KDateTime(QDate(2005,6,6), QTime(1,2,30), KDateTime::UTC), true);
    CHECK(utc.timeSpec() == utcSpec, true);
    CHECK(utcUtc.timeSpec() == utcSpec, true);

    // To Zone
    KDateTime zoneZone = zone.toTimeSpec(cairoSpec);
    CHECKTZ(zoneZone.timeZone(), cairo);
    CHECK(zoneZone == KDateTime(QDate(2005,7,1), QTime(4,0,0), cairo), true);
    CHECK(zone.timeSpec() != cairoSpec, true);
    CHECK(zoneZone.timeSpec() == cairoSpec, true);

    KDateTime zoneOffset = offset.toTimeSpec(cairoSpec);
    CHECKTZ(zoneOffset.timeZone(), cairo);
    CHECK(zoneOffset == KDateTime(QDate(2005,6,6), QTime(5,32,30), cairo), true);
    CHECK(offset.timeSpec() != cairoSpec, true);
    CHECK(zoneOffset.timeSpec() == cairoSpec, true);

    KDateTime zoneClock = clock.toTimeSpec(cairoSpec);
    CHECKTZ(zoneClock.timeZone(), cairo);
    CHECK(zoneClock == KDateTime(QDate(2005,6,6), QTime(11,2,30), cairo), true);
    CHECK(clock.timeSpec() != cairoSpec, true);
    CHECK(zoneClock.timeSpec() == cairoSpec, true);

    KDateTime zoneUtc = utc.toTimeSpec(cairoSpec);
    CHECKTZ(zoneUtc.timeZone(), cairo);
    CHECK(zoneUtc == KDateTime(QDate(2005,6,6), QTime(4,2,30), cairo), true);
    CHECK(utc.timeSpec() != cairoSpec, true);
    CHECK(zoneUtc.timeSpec() == cairoSpec, true);

    // To UTC offset
    KDateTime offsetZone = zone.toTimeSpec(offset1200Spec);
    CHECK(offsetZone.isOffsetFromUtc(), true);
    QCOMPARE(offsetZone.utcOffset(), 1200);
    CHECK(offsetZone == KDateTime(QDate(2005,7,1), QTime(1,20,0), KDateTime::Spec::OffsetFromUTC(1200)), true);
    CHECK(zone.timeSpec() != offset1200Spec, true);
    CHECK(offsetZone.timeSpec() == offset1200Spec, true);

    KDateTime offsetOffset = offset.toTimeSpec(offset1200Spec);
    CHECK(offsetOffset.isOffsetFromUtc(), true);
    QCOMPARE(offsetZone.utcOffset(), 1200);
    CHECK(offsetOffset == KDateTime(QDate(2005,6,6), QTime(2,52,30), KDateTime::Spec::OffsetFromUTC(1200)), true);
    CHECK(offset.timeSpec() != offset1200Spec, true);
    CHECK(offsetOffset.timeSpec() == offset1200Spec, true);

    KDateTime offsetClock = clock.toTimeSpec(offset1200Spec);
    CHECK(offsetClock.isOffsetFromUtc(), true);
    QCOMPARE(offsetZone.utcOffset(), 1200);
    CHECK(offsetClock == KDateTime(QDate(2005,6,6), QTime(8,22,30), KDateTime::Spec::OffsetFromUTC(1200)), true);
    CHECK(clock.timeSpec() != offset1200Spec, true);
    CHECK(offsetClock.timeSpec() == offset1200Spec, true);

    KDateTime offsetUtc = utc.toTimeSpec(offset1200Spec);
    CHECK(offsetUtc.isOffsetFromUtc(), true);
    QCOMPARE(offsetZone.utcOffset(), 1200);
    CHECK(offsetUtc == KDateTime(QDate(2005,6,6), QTime(1,22,30), KDateTime::Spec::OffsetFromUTC(1200)), true);
    CHECK(utc.timeSpec() != offset1200Spec, true);
    CHECK(offsetUtc.timeSpec() == offset1200Spec, true);

    // To Clock time
    KDateTime clockZone = zone.toTimeSpec(clockSpec);
    CHECK(clockZone.isClockTime(), true);
    CHECK(clockZone == KDateTime(QDate(2005,6,30), QTime(18,0,0), KDateTime::ClockTime), true);
    CHECK(zone.timeSpec() != clockSpec, true);
    CHECK(clockZone.timeSpec() == clockSpec, true);

    KDateTime clockOffset = offset.toTimeSpec(clockSpec);
    CHECK(clockOffset.isClockTime(), true);
    CHECK(clockOffset == KDateTime(QDate(2005,6,5), QTime(19,32,30), KDateTime::ClockTime), true);
    CHECK(offset.timeSpec() != clockSpec, true);
    CHECK(clockOffset.timeSpec() == clockSpec, true);

    KDateTime clockClock = clock.toTimeSpec(clockSpec);
    CHECK(clockClock.isClockTime(), true);
    CHECK(clockClock == KDateTime(QDate(2005,6,6), QTime(1,2,30), KDateTime::ClockTime), true);
    CHECK(clock.timeSpec() == clockSpec, true);
    CHECK(clockClock.timeSpec() == clockSpec, true);

    KDateTime clockUtc = utc.toTimeSpec(clockSpec);
    CHECK(clockUtc.isClockTime(), true);
    CHECK(clockUtc == KDateTime(QDate(2005,6,5), QTime(18,2,30), KDateTime::ClockTime), true);
    CHECK(utc.timeSpec() != clockSpec, true);
    CHECK(clockUtc.timeSpec() == clockSpec, true);


    // ** Date only ** //

    KDateTime zoned(QDate(2005,7,1), london);
    KDateTime offsetd(QDate(2005,6,6), KDateTime::Spec::OffsetFromUTC(-5400));  // -01:30
    KDateTime clockd(QDate(2005,6,6), KDateTime::ClockTime);
    KDateTime utcd(QDate(2005,6,6), KDateTime::UTC);

    // To UTC
    utcZone = zoned.toTimeSpec(utcSpec);
    CHECK(utcZone.isUtc(), true);
    CHECK(utcZone.isDateOnly(), true);
    CHECK(utcZone == KDateTime(QDate(2005,7,1), KDateTime::UTC), true);
    CHECK(utcZone != zoned, true);

    utcOffset = offsetd.toTimeSpec(utcSpec);
    CHECK(utcOffset.isUtc(), true);
    CHECK(utcOffset.isDateOnly(), true);
    CHECK(utcOffset == KDateTime(QDate(2005,6,6), KDateTime::UTC), true);
    CHECK(utcOffset != offsetd, true);

    utcClock = clockd.toTimeSpec(utcSpec);
    CHECK(utcClock.isUtc(), true);
    CHECK(utcClock.isDateOnly(), true);
    CHECK(utcClock == KDateTime(QDate(2005,6,6), KDateTime::UTC), true);
    CHECK(utcClock != clockd, true);

    utcUtc = utcd.toTimeSpec(utcSpec);
    CHECK(utcUtc.isUtc(), true);
    CHECK(utcUtc.isDateOnly(), true);
    CHECK(utcUtc == KDateTime(QDate(2005,6,6), KDateTime::UTC), true);
    CHECK(utcUtc == utcd, true);

    // To Zone
    zoneZone = zoned.toTimeSpec(cairoSpec);
    CHECK(zoneZone.isDateOnly(), true);
    CHECKTZ(zoneZone.timeZone(), cairo);
    CHECK(zoneZone == KDateTime(QDate(2005,7,1), cairo), true);
    CHECK(zoneZone != zoned, true);

    zoneOffset = offsetd.toTimeSpec(cairoSpec);
    CHECK(zoneOffset.isDateOnly(), true);
    CHECKTZ(zoneOffset.timeZone(), cairo);
    CHECK(zoneOffset == KDateTime(QDate(2005,6,6), cairo), true);
    CHECK(zoneOffset != offsetd, true);

    zoneClock = clockd.toTimeSpec(cairoSpec);
    CHECK(zoneClock.isDateOnly(), true);
    CHECKTZ(zoneClock.timeZone(), cairo);
    CHECK(zoneClock == KDateTime(QDate(2005,6,6), cairo), true);
    CHECK(zoneClock != clockd, true);

    zoneUtc = utcd.toTimeSpec(cairoSpec);
    CHECK(zoneUtc.isDateOnly(), true);
    CHECKTZ(zoneUtc.timeZone(), cairo);
    CHECK(zoneUtc == KDateTime(QDate(2005,6,6), cairo), true);
    CHECK(zoneUtc != utcd, true);

    // To UTC offset
    offsetZone = zoned.toTimeSpec(offset1200Spec);
    CHECK(offsetZone.isOffsetFromUtc(), true);
    CHECK(offsetZone.isDateOnly(), true);
    QCOMPARE(offsetZone.utcOffset(), 1200);
    CHECK(offsetZone == KDateTime(QDate(2005,7,1), KDateTime::Spec::OffsetFromUTC(1200)), true);
    CHECK(offsetZone != zoned, true);

    offsetOffset = offsetd.toTimeSpec(offset1200Spec);
    CHECK(offsetOffset.isOffsetFromUtc(), true);
    CHECK(offsetOffset.isDateOnly(), true);
    QCOMPARE(offsetZone.utcOffset(), 1200);
    CHECK(offsetOffset == KDateTime(QDate(2005,6,6), KDateTime::Spec::OffsetFromUTC(1200)), true);
    CHECK(offsetOffset != offsetd, true);

    offsetClock = clockd.toTimeSpec(offset1200Spec);
    CHECK(offsetClock.isOffsetFromUtc(), true);
    CHECK(offsetClock.isDateOnly(), true);
    QCOMPARE(offsetZone.utcOffset(), 1200);
    CHECK(offsetClock == KDateTime(QDate(2005,6,6), KDateTime::Spec::OffsetFromUTC(1200)), true);
    CHECK(offsetClock != clockd, true);

    offsetUtc = utcd.toTimeSpec(offset1200Spec);
    CHECK(offsetUtc.isOffsetFromUtc(), true);
    CHECK(offsetUtc.isDateOnly(), true);
    QCOMPARE(offsetZone.utcOffset(), 1200);
    CHECK(offsetUtc == KDateTime(QDate(2005,6,6), KDateTime::Spec::OffsetFromUTC(1200)), true);
    CHECK(offsetUtc != utcd, true);

    // To Clock time
    clockZone = zoned.toTimeSpec(clockSpec);
    CHECK(clockZone.isClockTime(), true);
    CHECK(clockZone.isDateOnly(), true);
    CHECK(clockZone == KDateTime(QDate(2005,7,1), KDateTime::ClockTime), true);
    CHECK(clockZone != zoned, true);

    clockOffset = offsetd.toTimeSpec(clockSpec);
    CHECK(clockOffset.isClockTime(), true);
    CHECK(clockOffset.isDateOnly(), true);
    CHECK(clockOffset == KDateTime(QDate(2005,6,6), KDateTime::ClockTime), true);
    CHECK(clockOffset != offsetd, true);

    clockClock = clockd.toTimeSpec(clockSpec);
    CHECK(clockClock.isClockTime(), true);
    CHECK(clockClock.isDateOnly(), true);
    CHECK(clockClock == KDateTime(QDate(2005,6,6), KDateTime::ClockTime), true);
    CHECK(clockClock == clockd, true);

    clockUtc = utcd.toTimeSpec(clockSpec);
    CHECK(clockUtc.isClockTime(), true);
    CHECK(clockUtc.isDateOnly(), true);
    CHECK(clockUtc == KDateTime(QDate(2005,6,6), KDateTime::ClockTime), true);
    CHECK(clockUtc != utcd, true);


    // Restore the original local time zone
    if (!originalZone)
        ::unsetenv("TZ");
    else
        ::setenv("TZ", originalZone, 1);
    ::tzset();
}


////////////////////////////////////////////////////////////////////////
// Set methods: setDate(), setTime(), setDateTime(), setTimeSpec()
////////////////////////////////////////////////////////////////////////

void KDateTimeTest::set()
{
    KTimeZone london = KSystemTimeZones::zone("Europe/London");

    // Ensure that local time is different from UTC and different from 'london'
    const char *originalZone = ::getenv("TZ");   // save the original local time zone
    ::setenv("TZ", ":America/Los_Angeles", 1);
    ::tzset();

    // Zone
    KDateTime zoned(QDate(2005,6,1), london);
    zoned.setDate(QDate(2004,5,2));
    CHECK(zoned.isDateOnly(), true);
    CHECKDT(zoned.dateTime(), QDateTime(QDate(2004,5,2), QTime(0,0,0))); //, Qt::LocalTime));
    zoned.setTime(QTime(12,13,14));
    CHECK(zoned.isDateOnly(), false);
    CHECKDT(zoned.dateTime(), QDateTime(QDate(2004,5,2), QTime(12,13,14))); //, Qt::LocalTime));
    zoned.setDate(QDate(2004,5,4));
    CHECK(zoned.isDateOnly(), false);

    zoned.setDateOnly(false);
    CHECK(zoned.isDateOnly(), false);
    CHECKDT(zoned.dateTime(), QDateTime(QDate(2004,5,4), QTime(12,13,14))); //, Qt::LocalTime));
    zoned.setDateOnly(true);
    CHECK(zoned.isDateOnly(), true);
    CHECKDT(zoned.dateTime(), QDateTime(QDate(2004,5,4), QTime(0,0,0))); //, Qt::LocalTime));
    zoned.setDateOnly(false);
    CHECK(zoned.isDateOnly(), false);
    CHECKDT(zoned.dateTime(), QDateTime(QDate(2004,5,4), QTime(0,0,0))); //, Qt::LocalTime));

    KDateTime zone(QDate(2005,6,1), QTime(3,40,0), london);
    zone.setDate(QDate(2004,5,2));
    CHECKDT(zone.dateTime(), QDateTime(QDate(2004,5,2), QTime(3,40,0))); //, Qt::LocalTime));
    zone.setTime(QTime(12,13,14));
    CHECKDT(zone.dateTime(), QDateTime(QDate(2004,5,2), QTime(12,13,14))); //, Qt::LocalTime));
    zone.setDateTime(QDateTime(QDate(2003,6,10), QTime(5,6,7)), Qt::LocalTime);
    CHECKDT(zone.dateTime(), QDateTime(QDate(2003,6,10), QTime(5,6,7))); //, Qt::LocalTime));
    QCOMPARE(zone.utcOffset(), 3600);
    CHECKDT(zone.toUtc().dateTime(), QDateTime(QDate(2003,6,10), QTime(4,6,7))); //, Qt::UTC));

    // UTC offset
    KDateTime offsetd(QDate(2005,6,6), KDateTime::Spec::OffsetFromUTC(-5400));  // -0130
    offsetd.setDate(QDate(2004,5,2));
    CHECK(offsetd.isDateOnly(), true);
    CHECKDT(offsetd.dateTime(), QDateTime(QDate(2004,5,2), QTime(0,0,0))); //, Qt::LocalTime));
    offsetd.setTime(QTime(12,13,14));
    CHECK(offsetd.isDateOnly(), false);
    CHECKDT(offsetd.dateTime(), QDateTime(QDate(2004,5,2), QTime(12,13,14))); //, Qt::LocalTime));
    offsetd.setDate(QDate(2004,5,4));
    CHECK(offsetd.isDateOnly(), false);

    offsetd.setDateOnly(false);
    CHECK(offsetd.isDateOnly(), false);
    CHECKDT(offsetd.dateTime(), QDateTime(QDate(2004,5,4), QTime(12,13,14))); //, Qt::LocalTime));
    offsetd.setDateOnly(true);
    CHECK(offsetd.isDateOnly(), true);
    CHECKDT(offsetd.dateTime(), QDateTime(QDate(2004,5,4), QTime(0,0,0))); //, Qt::LocalTime));
    offsetd.setDateOnly(false);
    CHECK(offsetd.isDateOnly(), false);
    CHECKDT(offsetd.dateTime(), QDateTime(QDate(2004,5,4), QTime(0,0,0))); //, Qt::LocalTime));

    KDateTime offset(QDate(2005,6,6), QTime(1,2,30), KDateTime::Spec::OffsetFromUTC(-5400));  // -0130
    offset.setDate(QDate(2004,5,2));
    CHECKDT(offset.dateTime(), QDateTime(QDate(2004,5,2), QTime(1,2,30))); //, Qt::LocalTime));
    offset.setTime(QTime(12,13,14));
    CHECKDT(offset.dateTime(), QDateTime(QDate(2004,5,2), QTime(12,13,14))); //, Qt::LocalTime));
    offset.setDateTime(QDateTime(QDate(2003,12,10), QTime(5,6,7)), Qt::LocalTime);
    CHECKDT(offset.dateTime(), QDateTime(QDate(2003,12,10), QTime(5,6,7))); //, Qt::LocalTime));
    QCOMPARE(offset.utcOffset(), -5400);
    CHECKDT(offset.toUtc().dateTime(), QDateTime(QDate(2003,12,10), QTime(6,36,7))); //, Qt::UTC));

    // Clock time
    KDateTime clockd(QDate(2005,6,6), KDateTime::ClockTime);
    clockd.setDate(QDate(2004,5,2));
    CHECK(clockd.isDateOnly(), true);
    CHECKDT(clockd.dateTime(), QDateTime(QDate(2004,5,2), QTime(0,0,0))); //, Qt::LocalTime));
    clockd.setTime(QTime(12,13,14));
    CHECK(clockd.isDateOnly(), false);
    CHECKDT(clockd.dateTime(), QDateTime(QDate(2004,5,2), QTime(12,13,14))); //, Qt::LocalTime));
    clockd.setDate(QDate(2004,5,4));
    CHECK(clockd.isDateOnly(), false);

    clockd.setDateOnly(false);
    CHECK(clockd.isDateOnly(), false);
    CHECKDT(clockd.dateTime(), QDateTime(QDate(2004,5,4), QTime(12,13,14))); //, Qt::LocalTime));
    clockd.setDateOnly(true);
    CHECK(clockd.isDateOnly(), true);
    CHECKDT(clockd.dateTime(), QDateTime(QDate(2004,5,4), QTime(0,0,0))); //, Qt::LocalTime));
    clockd.setDateOnly(false);
    CHECK(clockd.isDateOnly(), false);
    CHECKDT(clockd.dateTime(), QDateTime(QDate(2004,5,4), QTime(0,0,0))); //, Qt::LocalTime));

    KDateTime clock(QDate(2005,6,6), QTime(1,2,30), KDateTime::ClockTime);
    clock.setDate(QDate(2004,5,2));
    CHECKDT(clock.dateTime(), QDateTime(QDate(2004,5,2), QTime(1,2,30))); //, Qt::LocalTime));
    clock.setTime(QTime(12,13,14));
    CHECKDT(clock.dateTime(), QDateTime(QDate(2004,5,2), QTime(12,13,14))); //, Qt::LocalTime));
    clock.setDateTime(QDateTime(QDate(2003,12,10), QTime(5,6,7)), Qt::LocalTime);
    CHECKDT(clock.dateTime(), QDateTime(QDate(2003,12,10), QTime(5,6,7))); //, Qt::LocalTime));
    QCOMPARE(clock.utcOffset(), 0);
    CHECKDT(clock.toUtc().dateTime(), QDateTime(QDate(2003,12,10), QTime(13,6,7))); //, Qt::UTC));

    // UTC
    KDateTime utcd(QDate(2005,6,6), KDateTime::UTC);
    utcd.setDate(QDate(2004,5,2));
    CHECK(utcd.isDateOnly(), true);
    CHECKDT(utcd.dateTime(), QDateTime(QDate(2004,5,2), QTime(0,0,0))); //, Qt::UTC));
    utcd.setTime(QTime(12,13,14));
    CHECK(utcd.isDateOnly(), false);
    CHECKDT(utcd.dateTime(), QDateTime(QDate(2004,5,2), QTime(12,13,14))); //, Qt::UTC));
    utcd.setDate(QDate(2004,5,4));
    CHECK(utcd.isDateOnly(), false);

    utcd.setDateOnly(false);
    CHECK(utcd.isDateOnly(), false);
    CHECKDT(utcd.dateTime(), QDateTime(QDate(2004,5,4), QTime(12,13,14))); //, Qt::UTC));
    utcd.setDateOnly(true);
    CHECK(utcd.isDateOnly(), true);
    CHECKDT(utcd.dateTime(), QDateTime(QDate(2004,5,4), QTime(0,0,0))); //, Qt::UTC));
    utcd.setDateOnly(false);
    CHECK(utcd.isDateOnly(), false);
    CHECKDT(utcd.dateTime(), QDateTime(QDate(2004,5,4), QTime(0,0,0))); //, Qt::UTC));

    KDateTime utc(QDate(2005,6,6), QTime(1,2,30), KDateTime::UTC);
    utc.setDate(QDate(2004,5,2));
    CHECKDT(utc.dateTime(), QDateTime(QDate(2004,5,2), QTime(1,2,30))); //, Qt::UTC));
    utc.setTime(QTime(12,13,14));
    CHECKDT(utc.dateTime(), QDateTime(QDate(2004,5,2), QTime(12,13,14))); //, Qt::UTC));
    utc.setDateTime(QDateTime(QDate(2003,12,10), QTime(5,6,7)), Qt::UTC);
    QCOMPARE(utc.utcOffset(), 0);
    CHECKDT(utc.dateTime(), QDateTime(QDate(2003,12,10), QTime(5,6,7))); //, Qt::UTC));

    // setTimeSpec(SpecType)
    CHECKDT(zone.dateTime(), QDateTime(QDate(2003,6,10), QTime(5,6,7))); //, Qt::LocalTime));
    zone.setTimeSpec(KDateTime::Spec::OffsetFromUTC(7200));
    CHECK(zone.isOffsetFromUtc(), true);
    QCOMPARE(zone.utcOffset(), 7200);
    CHECKDT(zone.toUtc().dateTime(), QDateTime(QDate(2003,6,10), QTime(3,6,7))); //, Qt::UTC));
    zone.setTimeSpec(KDateTime::LocalZone);
    CHECK(zone.isLocalZone(), true);
    QCOMPARE(zone.utcOffset(), -7*3600);
    CHECKDT(zone.toUtc().dateTime(), QDateTime(QDate(2003,6,10), QTime(12,6,7))); //, Qt::UTC));
    zone.setTimeSpec(KDateTime::UTC);
    CHECK(zone.isUtc(), true);
    QCOMPARE(zone.utcOffset(), 0);
    CHECKDT(zone.dateTime(), QDateTime(QDate(2003,6,10), QTime(5,6,7))); //, Qt::UTC));
    zone.setTimeSpec(KDateTime::ClockTime);
    CHECK(zone.isClockTime(), true);
    QCOMPARE(zone.utcOffset(), 0);
    CHECKDT(zone.toUtc().dateTime(), QDateTime(QDate(2003,6,10), QTime(12,6,7))); //, Qt::UTC));

    // setTimeSpec(KDateTime::Spec)
    CHECKDT(zone.dateTime(), QDateTime(QDate(2003,6,10), QTime(5,6,7))); //, Qt::LocalTime));
    zone.setTimeSpec(offset.timeSpec());
    CHECK(zone.isOffsetFromUtc(), true);
    CHECKDT(zone.toUtc().dateTime(), QDateTime(QDate(2003,6,10), QTime(6,36,7))); //, Qt::UTC));
    CHECK((zone.timeSpec() == offset.timeSpec()), true);
    zone.setTimeSpec(KDateTime::LocalZone);
    CHECK(zone.isLocalZone(), true);
    CHECKDT(zone.toUtc().dateTime(), QDateTime(QDate(2003,6,10), QTime(12,6,7))); //, Qt::UTC));
    zone.setTimeSpec(utc.timeSpec());
    CHECK(zone.isUtc(), true);
    CHECKDT(zone.dateTime(), QDateTime(QDate(2003,6,10), QTime(5,6,7))); //, Qt::UTC));
    zone.setTimeSpec(clock.timeSpec());
    CHECK(zone.isClockTime(), true);
    CHECKDT(zone.toUtc().dateTime(), QDateTime(QDate(2003,6,10), QTime(12,6,7))); //, Qt::UTC));
    zone.setTimeSpec(london);
    CHECKTZ(zone.timeZone(), london);
    QCOMPARE(zone.utcOffset(), 3600);
    CHECKDT(zone.toUtc().dateTime(), QDateTime(QDate(2003,6,10), QTime(4,6,7))); //, Qt::UTC));

    // time_t
    utcd = KDateTime(QDate(2005,6,6), QTime(12,15,20), KDateTime::UTC);
    QDateTime qtt = utcd.dateTime();
    uint secs = utcToTime_t(qtt);
    KDateTime tt;
    tt.setTime_t(static_cast<Q_INT64>(secs));
    CHECK(tt.isUtc(), true);
    CHECKDT(tt.dateTime(), utcd.dateTime());
    QCOMPARE(tt.toTime_t(), secs);

    // Restore the original local time zone
    if (!originalZone)
        ::unsetenv("TZ");
    else
        ::setenv("TZ", originalZone, 1);
    ::tzset();
}


/////////////////////////////////////////////////////////
// compare()
/////////////////////////////////////////////////////////

void KDateTimeTest::compare()
{
    KTimeZone london = KSystemTimeZones::zone("Europe/London");
    KTimeZone cairo  = KSystemTimeZones::zone("Africa/Cairo");

    // Ensure that local time is different from UTC and different from 'london'
    const char *originalZone = ::getenv("TZ");   // save the original local time zone
    ::setenv("TZ", ":America/Los_Angeles", 1);
    ::tzset();

    // Date/time values
    QCOMPARE(KDateTime(QDate(2004,3,1), QTime(3,45,2), cairo).compare(KDateTime(QDate(2004,3,1), QTime(3,45,3), cairo)), KDateTime::Before);
    QCOMPARE(KDateTime(QDate(2004,3,1), QTime(3,45,2), cairo).compare(KDateTime(QDate(2004,3,1), QTime(3,45,2), KDateTime::UTC)), KDateTime::Before);
    QCOMPARE(KDateTime(QDate(2004,3,1), QTime(3,45,2), KDateTime::UTC).compare(KDateTime(QDate(2004,3,1), QTime(3,45,2), cairo)), KDateTime::After);
    QCOMPARE(KDateTime(QDate(2004,3,1), QTime(3,45,3), cairo).compare(KDateTime(QDate(2004,3,1), QTime(3,45,2), cairo)), KDateTime::After);
    QCOMPARE(KDateTime(QDate(2004,3,2), QTime(3,45,2), cairo).compare(KDateTime(QDate(2004,3,1), QTime(17,45,2), KDateTime::LocalZone)), KDateTime::Equal);
    QCOMPARE(KDateTime(QDate(2004,3,2), QTime(3,45,2), cairo).compare(KDateTime(QDate(2004,3,2), QTime(3,45,2), cairo)), KDateTime::Equal);

    // Date/time : date-only
    QCOMPARE(KDateTime(QDate(2004,3,1), QTime(0,0,0), cairo).compare(KDateTime(QDate(2004,3,1), cairo)), KDateTime::AtStart);
    QCOMPARE(KDateTime(QDate(2004,3,1), QTime(3,45,2), cairo).compare(KDateTime(QDate(2004,3,1), cairo)), KDateTime::Inside);
    QCOMPARE(KDateTime(QDate(2004,3,1), QTime(23,59,59,999), cairo).compare(KDateTime(QDate(2004,3,1), cairo)), KDateTime::AtEnd);
    QCOMPARE(KDateTime(QDate(2004,3,1), QTime(23,59,59,999), cairo).compare(KDateTime(QDate(2004,3,2), cairo)), KDateTime::Before);
    QCOMPARE(KDateTime(QDate(2004,3,3), QTime(0,0,0), cairo).compare(KDateTime(QDate(2004,3,2), cairo)), KDateTime::After);

    QCOMPARE(KDateTime(QDate(2004,3,2), QTime(9,59,59,999), cairo).compare(KDateTime(QDate(2004,3,2), KDateTime::LocalZone)), KDateTime::Before);
    QCOMPARE(KDateTime(QDate(2004,3,2), QTime(10,0,0), cairo).compare(KDateTime(QDate(2004,3,2), KDateTime::LocalZone)), KDateTime::AtStart);
    QCOMPARE(KDateTime(QDate(2004,3,2), QTime(10,0,1), cairo).compare(KDateTime(QDate(2004,3,2), KDateTime::LocalZone)), KDateTime::Inside);
    QCOMPARE(KDateTime(QDate(2004,3,3), QTime(9,59,59,999), cairo).compare(KDateTime(QDate(2004,3,2), KDateTime::LocalZone)), KDateTime::AtEnd);
    QCOMPARE(KDateTime(QDate(2004,3,3), QTime(10,0,0), cairo).compare(KDateTime(QDate(2004,3,2), KDateTime::LocalZone)), KDateTime::After);

    // Date-only : date/time
    QCOMPARE(KDateTime(QDate(2004,3,1), cairo).compare(KDateTime(QDate(2004,3,1), QTime(0,0,0), cairo)), KDateTime::StartsAt);
    QCOMPARE(KDateTime(QDate(2004,3,1), cairo).compare(KDateTime(QDate(2004,3,1), QTime(3,45,2), cairo)), KDateTime::Outside);
    QCOMPARE(KDateTime(QDate(2004,3,1), cairo).compare(KDateTime(QDate(2004,3,1), QTime(23,59,59,999), cairo)), KDateTime::EndsAt);
    QCOMPARE(KDateTime(QDate(2004,3,2), cairo).compare(KDateTime(QDate(2004,3,1), QTime(23,59,59,999), cairo)), KDateTime::After);
    QCOMPARE(KDateTime(QDate(2004,3,2), cairo).compare(KDateTime(QDate(2004,3,3), QTime(0,0,0), cairo)), KDateTime::Before);

    QCOMPARE(KDateTime(QDate(2004,3,2), KDateTime::LocalZone).compare(KDateTime(QDate(2004,3,2), QTime(9,59,59,999), cairo)), KDateTime::After);
    QCOMPARE(KDateTime(QDate(2004,3,2), KDateTime::LocalZone).compare(KDateTime(QDate(2004,3,2), QTime(10,0,0), cairo)), KDateTime::StartsAt);
    QCOMPARE(KDateTime(QDate(2004,3,2), KDateTime::LocalZone).compare(KDateTime(QDate(2004,3,3), QTime(9,59,59,999), cairo)), KDateTime::EndsAt);
    QCOMPARE(KDateTime(QDate(2004,3,2), KDateTime::LocalZone).compare(KDateTime(QDate(2004,3,3), QTime(10,0,0), cairo)), KDateTime::Before);

    // Date-only values
    QCOMPARE(KDateTime(QDate(2004,3,1), cairo).compare(KDateTime(QDate(2004,3,2), cairo)), KDateTime::Before);
    QCOMPARE(KDateTime(QDate(2004,3,1), cairo).compare(KDateTime(QDate(2004,3,2), KDateTime::Spec::OffsetFromUTC(2*3600))), KDateTime::Before);
    QCOMPARE(KDateTime(QDate(2004,3,1), london).compare(KDateTime(QDate(2004,3,2), cairo)), static_cast<KDateTime::Comparison>(KDateTime::Before|KDateTime::AtStart|KDateTime::Inside));
    QCOMPARE(KDateTime(QDate(2004,3,1), cairo).compare(KDateTime(QDate(2004,3,2), KDateTime::Spec::OffsetFromUTC(3*3600))), static_cast<KDateTime::Comparison>(KDateTime::Before|KDateTime::AtStart|KDateTime::Inside));
    QCOMPARE(KDateTime(QDate(2004,3,1), cairo).compare(KDateTime(QDate(2004,3,1), cairo)), KDateTime::Equal);
    QCOMPARE(KDateTime(QDate(2004,3,1), cairo).compare(KDateTime(QDate(2004,3,1), KDateTime::Spec::OffsetFromUTC(2*3600))), KDateTime::Equal);
    QCOMPARE(KDateTime(QDate(2004,3,2), cairo).compare(KDateTime(QDate(2004,3,1), london)), static_cast<KDateTime::Comparison>(KDateTime::Inside|KDateTime::AtEnd|KDateTime::After));
    QCOMPARE(KDateTime(QDate(2004,3,2), KDateTime::Spec::OffsetFromUTC(3*3600)).compare(KDateTime(QDate(2004,3,1), cairo)), static_cast<KDateTime::Comparison>(KDateTime::Inside|KDateTime::AtEnd|KDateTime::After));
    QCOMPARE(KDateTime(QDate(2004,3,2), cairo).compare(KDateTime(QDate(2004,3,1), cairo)), KDateTime::After);
    QCOMPARE(KDateTime(QDate(2004,3,2), KDateTime::Spec::OffsetFromUTC(2*3600)).compare(KDateTime(QDate(2004,3,1), cairo)), KDateTime::After);
    // Compare days when daylight savings changes occur
    QCOMPARE(KDateTime(QDate(2005,3,27), london).compare(KDateTime(QDate(2005,3,27), KDateTime::Spec::OffsetFromUTC(0))), static_cast<KDateTime::Comparison>(KDateTime::AtStart|KDateTime::Inside));
    QCOMPARE(KDateTime(QDate(2005,3,27), KDateTime::Spec::OffsetFromUTC(0)).compare(KDateTime(QDate(2005,3,27), london)), KDateTime::StartsAt);
    QCOMPARE(KDateTime(QDate(2005,10,30), london).compare(KDateTime(QDate(2005,10,30), KDateTime::UTC)), KDateTime::EndsAt);
    QCOMPARE(KDateTime(QDate(2005,10,30), KDateTime::UTC).compare(KDateTime(QDate(2005,10,30), london)), static_cast<KDateTime::Comparison>(KDateTime::Inside|KDateTime::AtEnd));

    // Restore the original local time zone
    if (!originalZone)
        ::unsetenv("TZ");
    else
        ::setenv("TZ", originalZone, 1);
    ::tzset();
}


/////////////////////////////////////////////////////////
// Addition and subtraction methods, and operator<() etc.
/////////////////////////////////////////////////////////

void KDateTimeTest::addSubtract()
{
    KTimeZone london = KSystemTimeZones::zone("Europe/London");
    KTimeZone losAngeles = KSystemTimeZones::zone("America/Los_Angeles");

    // Ensure that local time is different from UTC and different from 'london'
    const char *originalZone = ::getenv("TZ");   // save the original local time zone
    ::setenv("TZ", ":America/Los_Angeles", 1);
    ::tzset();

    // UTC
    KDateTime utc1(QDate(2005,7,6), QTime(3,40,0), KDateTime::UTC);
    KDateTime utc2 = utc1.addSecs(184 * 86400);
    CHECK(utc2.isUtc(), true);
    CHECKDT(utc2.dateTime(), QDateTime(QDate(2006,1,6), QTime(3,40,0))); //, Qt::UTC));
    KDateTime utc3 = utc1.addDays(184);
    CHECK(utc3.isUtc(), true);
    CHECKDT(utc2.dateTime(), utc3.dateTime());
    KDateTime utc4 = utc1.addMonths(6);
    CHECK(utc4.isUtc(), true);
    CHECKDT(utc2.dateTime(), utc4.dateTime());
    KDateTime utc5 = utc1.addYears(4);
    CHECK(utc5.isUtc(), true);
    CHECKDT(utc5.dateTime(), QDateTime(QDate(2009,7,6), QTime(3,40,0))); //, Qt::UTC));
    QCOMPARE(utc1.secsTo(utc2), 184 * 86400);
    QCOMPARE(utc1.secsTo(utc3), 184 * 86400);
    QCOMPARE(utc1.daysTo(utc2), 184);
    CHECK(utc1 < utc2, true);
    CHECK((utc2 < utc1), false);
    CHECK(utc2 == utc3, true);

    // UTC offset
    KDateTime offset1(QDate(2005,7,6), QTime(3,40,0), KDateTime::Spec::OffsetFromUTC(-5400));  // -0130
    KDateTime offset2 = offset1.addSecs(184 * 86400);
    CHECK(offset2.isOffsetFromUtc(), true);
    QCOMPARE(offset2.utcOffset(), -5400);
    CHECKDT(offset2.dateTime(), QDateTime(QDate(2006,1,6), QTime(3,40,0))); //, Qt::LocalTime));
    KDateTime offset3 = offset1.addDays(184);
    CHECK(offset3.isOffsetFromUtc(), true);
    QCOMPARE(offset3.utcOffset(), -5400);
    CHECKDT(offset2.dateTime(), offset3.dateTime());
    KDateTime offset4 = offset1.addMonths(6);
    CHECK(offset4.isOffsetFromUtc(), true);
    QCOMPARE(offset4.utcOffset(), -5400);
    CHECKDT(offset2.dateTime(), offset4.dateTime());
    KDateTime offset5 = offset1.addYears(4);
    CHECK(offset5.isOffsetFromUtc(), true);
    QCOMPARE(offset5.utcOffset(), -5400);
    CHECKDT(offset5.dateTime(), QDateTime(QDate(2009,7,6), QTime(3,40,0))); //, Qt::LocalTime));
    QCOMPARE(offset1.secsTo(offset2), 184 * 86400);
    QCOMPARE(offset1.secsTo(offset3), 184 * 86400);
    QCOMPARE(offset1.daysTo(offset2), 184);
    CHECK(offset1 < offset2, true);
    CHECK((offset2 < offset1), false);
    CHECK(offset2 == offset3, true);

    // Zone
    KDateTime zone1(QDate(2005,7,6), QTime(3,40,0), london);
    KDateTime zone2 = zone1.addSecs(184 * 86400);
    CHECKTZ(zone2.timeZone(), london);
    CHECKDT(zone2.dateTime(), QDateTime(QDate(2006,1,6), QTime(2,40,0))); //, Qt::LocalTime));
    KDateTime zone3 = zone1.addDays(184);
    CHECKTZ(zone3.timeZone(), london);
    CHECKDT(zone3.dateTime(), QDateTime(QDate(2006,1,6), QTime(3,40,0))); //, Qt::LocalTime));
    KDateTime zone4 = zone1.addMonths(6);
    CHECKTZ(zone4.timeZone(), london);
    CHECKDT(zone4.dateTime(), zone3.dateTime());
    KDateTime zone5 = zone1.addYears(4);
    CHECKTZ(zone5.timeZone(), london);
    CHECKDT(zone5.dateTime(), QDateTime(QDate(2009,7,6), QTime(3,40,0))); //, Qt::LocalTime));
    QCOMPARE(zone1.secsTo(zone2), 184 * 86400);
    QCOMPARE(zone1.secsTo(zone3), 184 * 86400 + 3600);
    QCOMPARE(zone1.daysTo(zone2), 184);
    QCOMPARE(zone1.daysTo(zone3), 184);
    CHECK(zone1 < zone2, true);
    CHECK((zone2 < zone1), false);
    CHECK((zone2 == zone3), false);

    // Local zone
    KDateTime local1(QDate(2005,7,6), QTime(3,40,0), KDateTime::LocalZone);
    KDateTime local2 = local1.addSecs(184 * 86400);
    CHECK(local2.isLocalZone(), true);
    CHECKTZ(local2.timeZone(), losAngeles);
    CHECKDT(local2.dateTime(), QDateTime(QDate(2006,1,6), QTime(2,40,0))); //, Qt::LocalTime));
    KDateTime local3 = local1.addDays(184);
    CHECK(local3.isLocalZone(), true);
    CHECKDT(local3.dateTime(), QDateTime(QDate(2006,1,6), QTime(3,40,0))); //, Qt::LocalTime));
    KDateTime local4 = local1.addMonths(6);
    CHECK(local4.isLocalZone(), true);
    CHECKDT(local4.dateTime(), local3.dateTime());
    KDateTime local5 = local1.addYears(4);
    CHECK(local5.isLocalZone(), true);
    CHECKDT(local5.dateTime(), QDateTime(QDate(2009,7,6), QTime(3,40,0))); //, Qt::LocalTime));
    QCOMPARE(local1.secsTo(local2), 184 * 86400);
    QCOMPARE(local1.secsTo(local3), 184 * 86400 + 3600);
    QCOMPARE(local1.daysTo(local2), 184);
    QCOMPARE(local1.daysTo(local3), 184);
    CHECK(local1 < local2, true);
    CHECK((local2 < local1), false);
    CHECK((local2 == local3), false);

    // Clock time
    KDateTime clock1(QDate(2005,7,6), QTime(3,40,0), KDateTime::ClockTime);
    KDateTime clock2 = clock1.addSecs(184 * 86400);
    CHECK(clock2.isClockTime(), true);
    CHECKDT(clock2.dateTime(), QDateTime(QDate(2006,1,6), QTime(3,40,0))); //, Qt::LocalTime));
    KDateTime clock3 = clock1.addDays(184);
    CHECK(clock3.isClockTime(), true);
    CHECKDT(clock2.dateTime(), clock3.dateTime());
    KDateTime clock4 = clock1.addMonths(6);
    CHECK(clock4.isClockTime(), true);
    CHECKDT(clock2.dateTime(), clock4.dateTime());
    KDateTime clock5 = clock1.addYears(4);
    CHECK(clock5.isClockTime(), true);
    CHECKDT(clock5.dateTime(), QDateTime(QDate(2009,7,6), QTime(3,40,0))); //, Qt::LocalTime));
    QCOMPARE(clock1.secsTo(clock2), 184 * 86400);
    QCOMPARE(clock1.secsTo(clock3), 184 * 86400);
    QCOMPARE(clock1.daysTo(clock2), 184);
    QCOMPARE(clock1.daysTo(clock3), 184);
    CHECK(clock1 < clock2, true);
    CHECK((clock2 < clock1), false);
    CHECK(clock2 == clock3, true);

    // Mixed timeSpecs
    QCOMPARE(utc1.secsTo(offset1), 5400);
    QCOMPARE(utc1.secsTo(offset2), 184 * 86400 + 5400);
    QCOMPARE(offset2.secsTo(utc1), -(184 * 86400 + 5400));
    CHECK(utc1 < offset1, true);
    CHECK(utc1 <= offset1, true);
    CHECK((offset1 < utc1), false);
    CHECK((offset1 <= utc1), false);
    QCOMPARE(utc1.secsTo(zone1), -3600);
    QCOMPARE(utc1.secsTo(zone2), 184 * 86400 - 3600);
    QCOMPARE(zone2.secsTo(utc1), -(184 * 86400 - 3600));
    CHECK(utc1 > zone1, true);
    CHECK(utc1 >= zone1, true);
    CHECK((zone1 > utc1), false);
    CHECK((zone1 >= utc1), false);
    QCOMPARE(utc1.secsTo(local1), 7 * 3600);
    QCOMPARE(utc1.secsTo(local2), 184 * 86400 + 7 * 3600);
    QCOMPARE(local2.secsTo(utc1), -(184 * 86400 + 7 * 3600));
    CHECK(utc1 < local1, true);
    CHECK(utc1 <= local1, true);
    CHECK((local1 < utc1), false);
    CHECK((local1 <= utc1), false);
    QCOMPARE(utc1.secsTo(clock1), 7 * 3600);
    QCOMPARE(utc1.secsTo(clock2), 184 * 86400 + 8 * 3600);
    QCOMPARE(clock2.secsTo(utc1), -(184 * 86400 + 8 * 3600));
    CHECK(utc1 < clock1, true);
    CHECK(utc1 <= clock1, true);
    CHECK((clock1 < utc1), false);
    CHECK((clock1 <= utc1), false);

    QCOMPARE(offset1.secsTo(zone1), -9000);
    QCOMPARE(offset1.secsTo(zone2), 184 * 86400 -9000);
    QCOMPARE(zone2.secsTo(offset1), -(184 * 86400 - 9000));
    CHECK(offset1 > zone1, true);
    CHECK(offset1 >= zone1, true);
    CHECK((zone1 > offset1), false);
    CHECK((zone1 >= offset1), false);
    QCOMPARE(offset1.secsTo(local1), 7*3600 - 5400);
    QCOMPARE(offset1.secsTo(local2), 184 * 86400 + 7*3600 - 5400);
    QCOMPARE(local2.secsTo(offset1), -(184 * 86400 + 7*3600 - 5400));
    CHECK(offset1 < local1, true);
    CHECK(offset1 <= local1, true);
    CHECK((local1 < offset1), false);
    CHECK((local1 <= offset1), false);
    QCOMPARE(offset1.secsTo(clock1), 7*3600 - 5400);
    QCOMPARE(offset1.secsTo(clock2), 184 * 86400 + 8*3600 - 5400);
    QCOMPARE(clock2.secsTo(offset1), -(184 * 86400 + 8*3600 - 5400));
    CHECK(offset1 < clock1, true);
    CHECK(offset1 <= clock1, true);
    CHECK((clock1 < offset1), false);
    CHECK((clock1 <= offset1), false);

    QCOMPARE(zone1.secsTo(local1), 8*3600);
    QCOMPARE(zone1.secsTo(local2), 184 * 86400 + 8*3600);
    QCOMPARE(local2.secsTo(zone1), -(184 * 86400 + 8*3600));
    CHECK(zone1 < local1, true);
    CHECK(zone1 <= local1, true);
    CHECK((local1 < zone1), false);
    CHECK((local1 <= zone1), false);
    QCOMPARE(zone1.secsTo(clock1), 8*3600);
    QCOMPARE(zone1.secsTo(clock2), 184 * 86400 + 9*3600);
    QCOMPARE(clock2.secsTo(zone1), -(184 * 86400 + 9*3600));
    CHECK(zone1 < clock1, true);
    CHECK(zone1 <= clock1, true);
    CHECK((clock1 < zone1), false);
    CHECK((clock1 <= zone1), false);

    QCOMPARE(local1.secsTo(clock1), 0);
    QCOMPARE(local1.secsTo(clock2), 184 * 86400 + 3600);
    QCOMPARE(clock2.secsTo(local1), -(184 * 86400 + 3600));
    CHECK(local1 == clock1, true);
    CHECK(local1 <= clock1, true);
    CHECK(local1 >= clock1, true);
    CHECK((local1 < clock1), false);
    CHECK((local1 > clock1), false);

    // Restore the original local time zone
    if (!originalZone)
        ::unsetenv("TZ");
    else
        ::setenv("TZ", originalZone, 1);
    ::tzset();
}

void KDateTimeTest::addMSecs()
{
    KTimeZone london = KSystemTimeZones::zone("Europe/London");
    KTimeZone losAngeles = KSystemTimeZones::zone("America/Los_Angeles");

    // Ensure that local time is different from UTC and different from 'london'
    const char *originalZone = ::getenv("TZ");   // save the original local time zone
    ::setenv("TZ", ":America/Los_Angeles", 1);
    ::tzset();

    // UTC
    KDateTime utc1(QDate(2005,7,6), QTime(23,59,0,100), KDateTime::UTC);
    KDateTime utc2 = utc1.addMSecs(59899);
    CHECK(utc2.isUtc(), true);
    CHECKDT(utc2.dateTime(), QDateTime(QDate(2005,7,6), QTime(23,59,59,999))); //, Qt::UTC));
    utc2 = utc1.addMSecs(59900);
    CHECK(utc2.isUtc(), true);
    CHECKDT(utc2.dateTime(), QDateTime(QDate(2005,7,7), QTime(0,0,0,0))); //, Qt::UTC));
    KDateTime utc1a(QDate(2005,7,6), QTime(0,0,5,100), KDateTime::UTC);
    utc2 = utc1a.addMSecs(-5100);
    CHECK(utc2.isUtc(), true);
    CHECKDT(utc2.dateTime(), QDateTime(QDate(2005,7,6), QTime(0,0,0,0))); //, Qt::UTC));
    utc2 = utc1a.addMSecs(-5101);
    CHECK(utc2.isUtc(), true);
    CHECKDT(utc2.dateTime(), QDateTime(QDate(2005,7,5), QTime(23,59,59,999))); //, Qt::UTC));

    // UTC offset
    KDateTime offset1(QDate(2005,7,6), QTime(3,40,0,100), KDateTime::Spec::OffsetFromUTC(-5400));  // -0130
    KDateTime offset2 = offset1.addMSecs(5899);
    CHECK(offset2.isOffsetFromUtc(), true);
    QCOMPARE(offset2.utcOffset(), -5400);
    CHECKDT(offset2.dateTime(), QDateTime(QDate(2005,7,6), QTime(3,40,5,999))); //, Qt::LocalTime));
    offset2 = offset1.addMSecs(5900);
    CHECK(offset2.isOffsetFromUtc(), true);
    QCOMPARE(offset2.utcOffset(), -5400);
    CHECKDT(offset2.dateTime(), QDateTime(QDate(2005,7,6), QTime(3,40,6,0))); //, Qt::LocalTime));
    offset2 = offset1.addMSecs(-5100);
    CHECK(offset2.isOffsetFromUtc(), true);
    QCOMPARE(offset2.utcOffset(), -5400);
    CHECKDT(offset2.dateTime(), QDateTime(QDate(2005,7,6), QTime(3,39,55,0))); //, Qt::LocalTime));
    offset2 = offset1.addMSecs(-5101);
    CHECK(offset2.isOffsetFromUtc(), true);
    QCOMPARE(offset2.utcOffset(), -5400);
    CHECKDT(offset2.dateTime(), QDateTime(QDate(2005,7,6), QTime(3,39,54,999))); //, Qt::LocalTime));

    // Zone
    KDateTime zone1(QDate(2002,3,31), QTime(0,40,0,100), london);   // time changes at 01:00 UTC
    KDateTime zone2 = zone1.addMSecs(3600*1000+899);
    CHECKTZ(zone2.timeZone(), london);
    CHECKDT(zone2.dateTime(), QDateTime(QDate(2002,3,31), QTime(2,40,0,999))); //, Qt::LocalTime));
    zone2 = zone1.addMSecs(3600*1000+900);
    CHECKTZ(zone2.timeZone(), london);
    CHECKDT(zone2.dateTime(), QDateTime(QDate(2002,3,31), QTime(2,40,1,0))); //, Qt::LocalTime));
    KDateTime zone1a(QDate(2002,3,31), QTime(2,40,0,100), london);   // time changes at 01:00 UTC
    zone2 = zone1a.addMSecs(-(3600*1000+100));
    CHECKTZ(zone2.timeZone(), london);
    CHECKDT(zone2.dateTime(), QDateTime(QDate(2002,3,31), QTime(0,40,0,0))); //, Qt::LocalTime));
    zone2 = zone1a.addMSecs(-(3600*1000+101));
    CHECKTZ(zone2.timeZone(), london);
    CHECKDT(zone2.dateTime(), QDateTime(QDate(2002,3,31), QTime(0,39,59,999))); //, Qt::LocalTime));

    // Local zone
    KDateTime local1(QDate(2002,4,7), QTime(1,59,0,100), KDateTime::LocalZone);   // time changes at 02:00 local
    KDateTime local2 = local1.addMSecs(59899);
    CHECK(local2.isLocalZone(), true);
    CHECKTZ(local2.timeZone(), losAngeles);
    CHECKDT(local2.dateTime(), QDateTime(QDate(2002,4,7), QTime(1,59,59,999))); //, Qt::LocalTime));
    local2 = local1.addMSecs(59900);
    CHECK(local2.isLocalZone(), true);
    CHECKTZ(local2.timeZone(), losAngeles);
    CHECKDT(local2.dateTime(), QDateTime(QDate(2002,4,7), QTime(3,0,0,0))); //, Qt::LocalTime));
    KDateTime local1a(QDate(2002,4,7), QTime(3,0,0,100), KDateTime::LocalZone);   // time changes at 02:00 local
    local2 = local1a.addMSecs(-100);
    CHECK(local2.isLocalZone(), true);
    CHECKTZ(local2.timeZone(), losAngeles);
    CHECKDT(local2.dateTime(), QDateTime(QDate(2002,4,7), QTime(3,0,0,0))); //, Qt::LocalTime));
    local2 = local1a.addMSecs(-101);
    CHECK(local2.isLocalZone(), true);
    CHECKTZ(local2.timeZone(), losAngeles);
    CHECKDT(local2.dateTime(), QDateTime(QDate(2002,4,7), QTime(1,59,59,999))); //, Qt::LocalTime));

    // Clock time
    KDateTime clock1(QDate(2002,4,7), QTime(1,59,0,100), KDateTime::ClockTime);
    KDateTime clock2 = clock1.addMSecs(59899);
    CHECK(clock2.isClockTime(), true);
    CHECKDT(clock2.dateTime(), QDateTime(QDate(2002,4,7), QTime(1,59,59,999))); //, Qt::LocalTime));
    clock2 = clock1.addMSecs(59900);
    CHECK(clock2.isClockTime(), true);
    CHECKDT(clock2.dateTime(), QDateTime(QDate(2002,4,7), QTime(2,0,0,0))); //, Qt::LocalTime));
    KDateTime clock1a(QDate(2002,4,7), QTime(0,0,0,100), KDateTime::ClockTime);
    clock2 = clock1a.addMSecs(-100);
    CHECK(clock2.isClockTime(), true);
    CHECKDT(clock2.dateTime(), QDateTime(QDate(2002,4,7), QTime(0,0,0,0))); //, Qt::LocalTime));
    clock2 = clock1a.addMSecs(-101);
    CHECK(clock2.isClockTime(), true);
    CHECKDT(clock2.dateTime(), QDateTime(QDate(2002,4,6), QTime(23,59,59,999))); //, Qt::LocalTime));

    // Restore the original local time zone
    if (!originalZone)
        ::unsetenv("TZ");
    else
        ::setenv("TZ", originalZone, 1);
    ::tzset();
}

void KDateTimeTest::addSubtractDate()
{
    KTimeZone london = KSystemTimeZones::zone("Europe/London");
    KTimeZone losAngeles = KSystemTimeZones::zone("America/Los_Angeles");

    // Ensure that local time is different from UTC and different from 'london'
    const char *originalZone = ::getenv("TZ");   // save the original local time zone
    ::setenv("TZ", ":America/Los_Angeles", 1);
    ::tzset();

    // UTC
    KDateTime utc1(QDate(2005,7,6), KDateTime::UTC);
    KDateTime utc2 = utc1.addSecs(184 * 86400 + 100);
    CHECK(utc2.isUtc(), true);
    CHECK(utc2.isDateOnly(), true);
    CHECKDT(utc2.dateTime(), QDateTime(QDate(2006,1,6), QTime(0,0,0))); //, Qt::UTC));
    KDateTime utc3 = utc1.addDays(184);
    CHECK(utc3.isUtc(), true);
    CHECK(utc3.isDateOnly(), true);
    CHECKDT(utc2.dateTime(), utc3.dateTime());
    KDateTime utc4 = utc1.addMonths(6);
    CHECK(utc4.isUtc(), true);
    CHECK(utc4.isDateOnly(), true);
    CHECKDT(utc2.dateTime(), utc4.dateTime());
    KDateTime utc5 = utc1.addYears(4);
    CHECK(utc5.isUtc(), true);
    CHECK(utc5.isDateOnly(), true);
    CHECKDT(utc5.dateTime(), QDateTime(QDate(2009,7,6), QTime(0,0,0))); //, Qt::UTC));
    QCOMPARE(utc1.secsTo(utc2), 184 * 86400);
    QCOMPARE(utc1.secsTo(utc3), 184 * 86400);
    QCOMPARE(utc1.daysTo(utc2), 184);
    CHECK(utc1 < utc2, true);
    CHECK((utc2 < utc1), false);
    CHECK(utc2 == utc3, true);

    // UTC offset
    KDateTime offset1(QDate(2005,7,6), KDateTime::Spec::OffsetFromUTC(-5400));  // -0130
    KDateTime offset2 = offset1.addSecs(184 * 86400);
    CHECK(offset2.isDateOnly(), true);
    CHECK(offset2.isOffsetFromUtc(), true);
    QCOMPARE(offset2.utcOffset(), -5400);
    CHECKDT(offset2.dateTime(), QDateTime(QDate(2006,1,6), QTime(0,0,0))); //, Qt::LocalTime));
    KDateTime offset3 = offset1.addDays(184);
    CHECK(offset3.isDateOnly(), true);
    CHECK(offset3.isOffsetFromUtc(), true);
    QCOMPARE(offset3.utcOffset(), -5400);
    CHECKDT(offset2.dateTime(), offset3.dateTime());
    KDateTime offset4 = offset1.addMonths(6);
    CHECK(offset4.isDateOnly(), true);
    CHECK(offset4.isOffsetFromUtc(), true);
    QCOMPARE(offset4.utcOffset(), -5400);
    CHECKDT(offset2.dateTime(), offset4.dateTime());
    KDateTime offset5 = offset1.addYears(4);
    CHECK(offset5.isDateOnly(), true);
    CHECK(offset5.isOffsetFromUtc(), true);
    QCOMPARE(offset5.utcOffset(), -5400);
    CHECKDT(offset5.dateTime(), QDateTime(QDate(2009,7,6), QTime(0,0,0))); //, Qt::LocalTime));
    QCOMPARE(offset1.secsTo(offset2), 184 * 86400);
    QCOMPARE(offset1.secsTo(offset3), 184 * 86400);
    QCOMPARE(offset1.daysTo(offset2), 184);
    CHECK(offset1 < offset2, true);
    CHECK((offset2 < offset1), false);
    CHECK(offset2 == offset3, true);

    // Zone
    KDateTime zone1(QDate(2005,7,6), london);
    KDateTime zone2 = zone1.addSecs(184 * 86400);
    CHECK(zone2.isDateOnly(), true);
    CHECKTZ(zone2.timeZone(), london);
    CHECKDT(zone2.dateTime(), QDateTime(QDate(2006,1,6), QTime(0,0,0))); //, Qt::LocalTime));
    KDateTime zone3 = zone1.addDays(184);
    CHECK(zone3.isDateOnly(), true);
    CHECKTZ(zone3.timeZone(), london);
    CHECKDT(zone3.dateTime(), QDateTime(QDate(2006,1,6), QTime(0,0,0))); //, Qt::LocalTime));
    KDateTime zone4 = zone1.addMonths(6);
    CHECK(zone4.isDateOnly(), true);
    CHECKTZ(zone4.timeZone(), london);
    CHECKDT(zone4.dateTime(), zone3.dateTime());
    KDateTime zone5 = zone1.addYears(4);
    CHECK(zone5.isDateOnly(), true);
    CHECKTZ(zone5.timeZone(), london);
    CHECKDT(zone5.dateTime(), QDateTime(QDate(2009,7,6), QTime(0,0,0))); //, Qt::LocalTime));
    QCOMPARE(zone1.secsTo(zone2), 184 * 86400);
    QCOMPARE(zone1.secsTo(zone3), 184 * 86400);
    QCOMPARE(zone1.daysTo(zone2), 184);
    QCOMPARE(zone1.daysTo(zone3), 184);
    CHECK(zone1 < zone2, true);
    CHECK((zone2 < zone1), false);
    CHECK(zone2 == zone3, true);

    // Local zone
    KDateTime local1(QDate(2005,7,6), KDateTime::LocalZone);
    KDateTime local2 = local1.addSecs(184 * 86400);
    CHECK(local2.isDateOnly(), true);
    CHECK(local2.isLocalZone(), true);
    CHECKTZ(local2.timeZone(), losAngeles);
    CHECKDT(local2.dateTime(), QDateTime(QDate(2006,1,6), QTime(0,0,0))); //, Qt::LocalTime));
    KDateTime local3 = local1.addDays(184);
    CHECK(local3.isDateOnly(), true);
    CHECK(local3.isLocalZone(), true);
    CHECKDT(local3.dateTime(), QDateTime(QDate(2006,1,6), QTime(0,0,0))); //, Qt::LocalTime));
    KDateTime local4 = local1.addMonths(6);
    CHECK(local4.isDateOnly(), true);
    CHECK(local4.isLocalZone(), true);
    CHECKDT(local4.dateTime(), local3.dateTime());
    KDateTime local5 = local1.addYears(4);
    CHECK(local5.isDateOnly(), true);
    CHECK(local5.isLocalZone(), true);
    CHECKDT(local5.dateTime(), QDateTime(QDate(2009,7,6), QTime(0,0,0))); //, Qt::LocalTime));
    QCOMPARE(local1.secsTo(local2), 184 * 86400);
    QCOMPARE(local1.secsTo(local3), 184 * 86400);
    QCOMPARE(local1.daysTo(local2), 184);
    QCOMPARE(local1.daysTo(local3), 184);
    CHECK(local1 < local2, true);
    CHECK((local2 < local1), false);
    CHECK(local2 == local3, true);

    // Clock time
    KDateTime clock1(QDate(2005,7,6), KDateTime::ClockTime);
    KDateTime clock2 = clock1.addSecs(184 * 86400);
    CHECK(clock2.isDateOnly(), true);
    CHECK(clock2.isClockTime(), true);
    CHECKDT(clock2.dateTime(), QDateTime(QDate(2006,1,6), QTime(0,0,0))); //, Qt::LocalTime));
    KDateTime clock3 = clock1.addDays(184);
    CHECK(clock3.isDateOnly(), true);
    CHECK(clock3.isClockTime(), true);
    CHECKDT(clock2.dateTime(), clock3.dateTime());
    KDateTime clock4 = clock1.addMonths(6);
    CHECK(clock4.isDateOnly(), true);
    CHECK(clock4.isClockTime(), true);
    CHECKDT(clock2.dateTime(), clock4.dateTime());
    KDateTime clock5 = clock1.addYears(4);
    CHECK(clock5.isDateOnly(), true);
    CHECK(clock5.isClockTime(), true);
    CHECKDT(clock5.dateTime(), QDateTime(QDate(2009,7,6), QTime(0,0,0))); //, Qt::LocalTime));
    QCOMPARE(clock1.secsTo(clock2), 184 * 86400);
    QCOMPARE(clock1.secsTo(clock3), 184 * 86400);
    QCOMPARE(clock1.daysTo(clock2), 184);
    QCOMPARE(clock1.daysTo(clock3), 184);
    CHECK(clock1 < clock2, true);
    CHECK((clock2 < clock1), false);
    CHECK(clock2 == clock3, true);

    // Mixed timeSpecs
    QCOMPARE(utc1.secsTo(offset1), 0);
    QCOMPARE(utc1.secsTo(offset2), 184 * 86400);
    QCOMPARE(offset2.secsTo(utc1), -(184 * 86400));
    CHECK((utc1 < offset1), false);
    CHECK(utc1 <= offset1, true);
    CHECK((offset1 < utc1), false);
    CHECK(offset1 <= utc1, true);
    QCOMPARE(utc1.secsTo(zone1), 0);
    QCOMPARE(utc1.secsTo(zone2), 184 * 86400);
    QCOMPARE(zone2.secsTo(utc1), -(184 * 86400));
    CHECK((utc1 > zone1), false);
    CHECK(utc1 >= zone1, true);
    CHECK((zone1 > utc1), false);
    CHECK(zone1 >= utc1, true);
    QCOMPARE(utc1.secsTo(local1), 0);
    QCOMPARE(utc1.secsTo(local2), 184 * 86400);
    QCOMPARE(local2.secsTo(utc1), -(184 * 86400));
    CHECK((utc1 < local1), false);
    CHECK(utc1 <= local1, true);
    CHECK((local1 < utc1), false);
    CHECK(local1 <= utc1, true);
    QCOMPARE(utc1.secsTo(clock1), 0);
    QCOMPARE(utc1.secsTo(clock2), 184 * 86400);
    QCOMPARE(clock2.secsTo(utc1), -(184 * 86400));
    CHECK((utc1 < clock1), false);
    CHECK(utc1 <= clock1, true);
    CHECK((clock1 < utc1), false);
    CHECK(clock1 <= utc1, true);

    QCOMPARE(offset1.secsTo(zone1), 0);
    QCOMPARE(offset1.secsTo(zone2), 184 * 86400);
    QCOMPARE(zone2.secsTo(offset1), -(184 * 86400));
    CHECK((offset1 > zone1), false);
    CHECK(offset1 >= zone1, true);
    CHECK((zone1 > offset1), false);
    CHECK(zone1 >= offset1, true);
    QCOMPARE(offset1.secsTo(local1), 0);
    QCOMPARE(offset1.secsTo(local2), 184 * 86400);
    QCOMPARE(local2.secsTo(offset1), -(184 * 86400));
    CHECK((offset1 < local1), false);
    CHECK(offset1 <= local1, true);
    CHECK((local1 < offset1), false);
    CHECK(local1 <= offset1, true);
    QCOMPARE(offset1.secsTo(clock1), 0);
    QCOMPARE(offset1.secsTo(clock2), 184 * 86400);
    QCOMPARE(clock2.secsTo(offset1), -(184 * 86400));
    CHECK((offset1 < clock1), false);
    CHECK(offset1 <= clock1, true);
    CHECK((clock1 < offset1), false);
    CHECK(clock1 <= offset1, true);

    QCOMPARE(zone1.secsTo(local1), 0);
    QCOMPARE(zone1.secsTo(local2), 184 * 86400);
    QCOMPARE(local2.secsTo(zone1), -(184 * 86400));
    CHECK((zone1 < local1), false);
    CHECK(zone1 <= local1, true);
    CHECK((local1 < zone1), false);
    CHECK(local1 <= zone1, true);
    QCOMPARE(zone1.secsTo(clock1), 0);
    QCOMPARE(zone1.secsTo(clock2), 184 * 86400);
    QCOMPARE(clock2.secsTo(zone1), -(184 * 86400));
    CHECK((zone1 < clock1), false);
    CHECK(zone1 <= clock1, true);
    CHECK((clock1 < zone1), false);
    CHECK(clock1 <= zone1, true);

    QCOMPARE(local1.secsTo(clock1), 0);
    QCOMPARE(local1.secsTo(clock2), 184 * 86400);
    QCOMPARE(clock2.secsTo(local1), -(184 * 86400));
    CHECK(local1 == clock1, true);
    CHECK(local1 <= clock1, true);
    CHECK(local1 >= clock1, true);
    CHECK((local1 < clock1), false);
    CHECK((local1 > clock1), false);


    // Mixed date/time and date-only

    // UTC
    utc3.setTime(QTime(13,14,15));
    CHECK(utc3.isDateOnly(), false);
    CHECKT(utc3.time(), QTime(13,14,15));
    QCOMPARE(utc1.secsTo(utc3), 184 * 86400);

    KDateTime utc1t(QDate(2005,7,6), QTime(3,40,0), KDateTime::UTC);
    QCOMPARE(utc1t.secsTo(utc2), 184 * 86400);

    // UTC offset
    offset3.setTime(QTime(13,14,15));
    CHECK(offset3.isDateOnly(), false);
    CHECKT(offset3.time(), QTime(13,14,15));
    QCOMPARE(offset1.secsTo(offset3), 184 * 86400);

    KDateTime offset1t(QDate(2005,7,6), QTime(3,40,0), KDateTime::Spec::OffsetFromUTC(-5400));  // -0130
    QCOMPARE(offset1t.secsTo(offset2), 184 * 86400);

    KDateTime offset2t(QDate(2005,7,6), QTime(0,40,0), KDateTime::Spec::OffsetFromUTC(5400));  // +0130

    // Zone
    zone3.setTime(QTime(13,14,15));
    CHECK(zone3.isDateOnly(), false);
    CHECKT(zone3.time(), QTime(13,14,15));
    QCOMPARE(zone1.secsTo(zone3), 184 * 86400);

    KDateTime zone1t(QDate(2005,7,6), QTime(3,40,0), london);
    QCOMPARE(zone1t.secsTo(zone2), 184 * 86400);

    // Local zone
    local3.setTime(QTime(13,14,15));
    CHECK(local3.isDateOnly(), false);
    CHECKT(local3.time(), QTime(13,14,15));
    QCOMPARE(local1.secsTo(local3), 184 * 86400);

    KDateTime local1t(QDate(2005,7,6), QTime(3,40,0), KDateTime::LocalZone);
    QCOMPARE(local1t.secsTo(local2), 184 * 86400);

    KDateTime local2t(QDate(2005,7,5), QTime(23,40,0), KDateTime::LocalZone);

    // Clock time
    clock3.setTime(QTime(13,14,15));
    CHECK(clock3.isDateOnly(), false);
    CHECKT(clock3.time(), QTime(13,14,15));
    QCOMPARE(clock1.secsTo(clock3), 184 * 86400);

    KDateTime clock1t(QDate(2005,7,6), QTime(3,40,0), KDateTime::ClockTime);
    QCOMPARE(clock1t.secsTo(clock2), 184 * 86400);

    // Mixed timeSpecs
    QCOMPARE(utc1t.secsTo(offset1), 0);
    CHECK(utc1t != offset1, true);
    CHECK(offset1 != utc1t, true);
    CHECK((utc1t < offset1), false);
    CHECK(utc1t <= offset1, true);
    CHECK((offset1 < utc1t), false);
    CHECK(offset1 <= utc1t, true);
    QCOMPARE(utc1.secsTo(offset2t), -86400);
    QCOMPARE(offset2t.secsTo(utc1), 86400);
    CHECK(utc1 != offset2t, true);
    CHECK(offset2t != utc1, true);
    CHECK(utc1 > offset2t, true);
    CHECK(utc1 >= offset2t, true);
    CHECK(offset2t < utc1, true);
    CHECK(offset2t <= utc1, true);
    QCOMPARE(utc1t.secsTo(offset2), 184 * 86400);
    QCOMPARE(offset2.secsTo(utc1t), -(184 * 86400));
    QCOMPARE(utc1t.secsTo(zone1), 0);
    CHECK(utc1t != zone1, true);
    CHECK(zone1 != utc1t, true);
    CHECK((utc1t < zone1), false);
    CHECK((utc1t > zone1), false);
    CHECK((zone1 < utc1t), false);
    CHECK((zone1 > utc1t), false);
    QCOMPARE(utc1t.secsTo(zone2), 184 * 86400);
    QCOMPARE(zone2.secsTo(utc1t), -(184 * 86400));
    CHECK(utc1t != zone2, true);
    CHECK(zone2 != utc1t, true);
    CHECK(utc1t < zone2, true);
    CHECK(utc1t <= zone2, true);
    CHECK((zone2 < utc1t), false);
    CHECK((zone2 <= utc1t), false);
    QCOMPARE(utc1t.secsTo(local1), 86400);
    QCOMPARE(utc1t.secsTo(local2), 185 * 86400);
    QCOMPARE(local2.secsTo(utc1t), -(185 * 86400));
    CHECK(utc1t != local1, true);
    CHECK(local1 != utc1t, true);
    CHECK(utc1t < local1, true);
    CHECK(utc1t <= local1, true);
    CHECK((local1 < utc1t), false);
    CHECK((local1 <= utc1t), false);
    QCOMPARE(utc1.secsTo(local2t), 0);
    QCOMPARE(local2t.secsTo(utc1), 0);
    CHECK(utc1 != local2t, true);
    CHECK(local2t != utc1, true);
    CHECK((utc1 < local2t), false);
    CHECK(utc1 <= local2t, true);
    CHECK((local2t < utc1), false);
    CHECK(local2t <= utc1, true);
    QCOMPARE(utc1t.secsTo(clock1), 86400);
    QCOMPARE(utc1t.secsTo(clock2), 185 * 86400);
    QCOMPARE(clock2.secsTo(utc1t), -(185 * 86400));
    CHECK(utc1t != clock1, true);
    CHECK(clock1 != utc1t, true);
    CHECK(utc1t < clock1, true);
    CHECK(utc1t <= clock1, true);
    CHECK((clock1 < utc1t), false);
    CHECK((clock1 <= utc1t), false);

    QCOMPARE(offset1t.secsTo(zone1), 0);
    QCOMPARE(offset1t.secsTo(zone2), 184 * 86400);
    QCOMPARE(zone2.secsTo(offset1t), -(184 * 86400));
    CHECK(offset1t != zone1, true);
    CHECK(zone1 != offset1t, true);
    CHECK((offset1t > zone1), false);
    CHECK(offset1t >= zone1, true);
    CHECK((zone1 > offset1t), false);
    CHECK(zone1 >= offset1t, true);
    QCOMPARE(offset1t.secsTo(local1), 86400);
    QCOMPARE(offset1t.secsTo(local2), 185 * 86400);
    QCOMPARE(local2.secsTo(offset1t), -(185 * 86400));
    CHECK(offset1t != local1, true);
    CHECK(local1 != offset1t, true);
    CHECK(offset1t < local1, true);
    CHECK(offset1t <= local1, true);
    CHECK((local1 < offset1t), false);
    CHECK((local1 <= offset1t), false);
    QCOMPARE(offset1t.secsTo(clock1), 86400);
    QCOMPARE(offset1t.secsTo(clock2), 185 * 86400);
    QCOMPARE(clock2.secsTo(offset1t), -(185 * 86400));
    CHECK(offset1t != clock1, true);
    CHECK(clock1 != offset1t, true);
    CHECK(offset1t < clock1, true);
    CHECK(offset1t <= clock1, true);
    CHECK((clock1 < offset1t), false);
    CHECK((clock1 <= offset1t), false);

    QCOMPARE(zone1t.secsTo(local1), 86400);
    QCOMPARE(zone1t.secsTo(local2), 185 * 86400);
    QCOMPARE(local2.secsTo(zone1t), -(185 * 86400));
    CHECK(zone1t != local1, true);
    CHECK(local1 != zone1t, true);
    CHECK(zone1t < local1, true);
    CHECK(zone1t <= local1, true);
    CHECK((local1 < zone1t), false);
    CHECK((local1 <= zone1t), false);
    QCOMPARE(zone1t.secsTo(clock1), 86400);
    QCOMPARE(zone1t.secsTo(clock2), 185 * 86400);
    QCOMPARE(clock2.secsTo(zone1t), -(185 * 86400));
    CHECK(zone1t != clock1, true);
    CHECK(clock1 != zone1t, true);
    CHECK(zone1t < clock1, true);
    CHECK(zone1t <= clock1, true);
    CHECK((clock1 < zone1t), false);
    CHECK((clock1 <= zone1t), false);

    QCOMPARE(local1t.secsTo(clock1), 0);
    QCOMPARE(local1t.secsTo(clock2), 184 * 86400);
    QCOMPARE(clock2.secsTo(local1t), -(184 * 86400));
    CHECK(local1t != clock1, true);
    CHECK(local1t != clock1, true);
    CHECK((local1t < clock1), false);
    CHECK(local1t <= clock1, true);
    CHECK((local1t < clock1), false);
    CHECK(local1t <= clock1, true);

    // Restore the original local time zone
    if (!originalZone)
        ::unsetenv("TZ");
    else
        ::setenv("TZ", originalZone, 1);
    ::tzset();
}


///////////////////////////////////////////
// Tests around daylight saving time shifts
///////////////////////////////////////////

void KDateTimeTest::dstShifts()
{
    KTimeZone london = KSystemTimeZones::zone("Europe/London");

    // Ensure that local time is different from UTC and different from 'london'
    const char *originalZone = ::getenv("TZ");   // save the original local time zone
    ::setenv("TZ", ":America/Los_Angeles", 1);
    ::tzset();

    // Shift from DST to standard time for the UK in 2005 was at 2005-10-30 01:00 UTC.
    KDateTime dt(QDateTime(QDate(2005,10,29), QTime(23,59,59)), Qt::UTC, london);
    CHECK(dt.isSecondOccurrence(), false);
    CHECKDT(dt.dateTime(), QDateTime(QDate(2005,10,30), QTime(0,59,59))); //, Qt::LocalTime));
    dt = KDateTime(QDateTime(QDate(2005,10,30), QTime(0,0,0)), Qt::UTC, london);
    CHECK(dt.isSecondOccurrence(), false);
    CHECKDT(dt.dateTime(), QDateTime(QDate(2005,10,30), QTime(1,0,0))); //, Qt::LocalTime));
    dt = KDateTime(QDateTime(QDate(2005,10,30), QTime(0,59,59)), Qt::UTC, london);
    CHECK(dt.isSecondOccurrence(), false);
    CHECKDT(dt.dateTime(), QDateTime(QDate(2005,10,30), QTime(1,59,59))); //, Qt::LocalTime));
    dt = KDateTime(QDateTime(QDate(2005,10,30), QTime(1,0,0)), Qt::UTC, london);
    CHECK(dt.isSecondOccurrence(), true);
    CHECKDT(dt.dateTime(), QDateTime(QDate(2005,10,30), QTime(1,0,0))); //, Qt::LocalTime));
    dt = KDateTime(QDateTime(QDate(2005,10,30), QTime(1,59,59)), Qt::UTC, london);
    CHECK(dt.isSecondOccurrence(), true);
    CHECKDT(dt.dateTime(), QDateTime(QDate(2005,10,30), QTime(1,59,59))); //, Qt::LocalTime));
    dt = KDateTime(QDateTime(QDate(2005,10,30), QTime(2,0,0)), Qt::UTC, london);
    CHECK(dt.isSecondOccurrence(), false);
    CHECKDT(dt.dateTime(), QDateTime(QDate(2005,10,30), QTime(2,0,0))); //, Qt::LocalTime));

    dt = KDateTime(QDateTime(QDate(2005,10,30), QTime(0,59,59)), Qt::LocalTime, london);
    dt.setSecondOccurrence(true);   // this has no effect
    CHECKDT(dt.toUtc().dateTime(), QDateTime(QDate(2005,10,29), QTime(23,59,59))); //, Qt::UTC));
    dt = KDateTime(QDateTime(QDate(2005,10,30), QTime(1,0,0)), Qt::LocalTime, london);
    CHECK(dt.isSecondOccurrence(), false);
    CHECKDT(dt.toUtc().dateTime(), QDateTime(QDate(2005,10,30), QTime(0,0,0))); //, Qt::UTC));
    dt = KDateTime(QDateTime(QDate(2005,10,30), QTime(1,59,59)), Qt::LocalTime, london);
    CHECK(dt.isSecondOccurrence(), false);
    CHECKDT(dt.toUtc().dateTime(), QDateTime(QDate(2005,10,30), QTime(0,59,59))); //, Qt::UTC));
    dt = KDateTime(QDateTime(QDate(2005,10,30), QTime(1,0,0)), Qt::LocalTime, london);
    dt.setSecondOccurrence(true);
    CHECKDT(dt.toUtc().dateTime(), QDateTime(QDate(2005,10,30), QTime(1,0,0))); //, Qt::UTC));
    CHECK(dt.isSecondOccurrence(), true);
    dt = KDateTime(QDateTime(QDate(2005,10,30), QTime(1,59,59)), Qt::LocalTime, london);
    dt.setSecondOccurrence(true);
    CHECKDT(dt.toUtc().dateTime(), QDateTime(QDate(2005,10,30), QTime(1,59,59))); //, Qt::UTC));
    CHECK(dt.isSecondOccurrence(), true);
    dt = KDateTime(QDateTime(QDate(2005,10,30), QTime(2,0,0)), Qt::LocalTime, london);
    dt.setSecondOccurrence(true);   // this has no effect
    CHECK(dt.isSecondOccurrence(), false);
    CHECKDT(dt.toUtc().dateTime(), QDateTime(QDate(2005,10,30), QTime(2,0,0))); //, Qt::UTC));

    dt = KDateTime(QDateTime(QDate(2005,10,30), QTime(0,59,59)), Qt::LocalTime, london);
    KDateTime dt1 = dt.addSecs(1);   // local time 01:00:00
    CHECK(dt1.isSecondOccurrence(), false);
    dt1 = dt.addSecs(3600);   // local time 01:59:59
    CHECK(dt1.isSecondOccurrence(), false);
    dt1 = dt.addSecs(3601);   // local time 01:00:00
    CHECK(dt1.isSecondOccurrence(), true);
    dt1 = dt.addSecs(7200);   // local time 01:59:59
    CHECK(dt1.isSecondOccurrence(), true);
    dt1 = dt.addSecs(7201);   // local time 02:00:00
    CHECK(dt1.isSecondOccurrence(), false);

    CHECK(KDateTime(QDate(2005,10,29), london) == KDateTime(QDate(2005,10,29), KDateTime::Spec::OffsetFromUTC(3600)), true);
    CHECK(KDateTime(QDate(2005,10,30), london) != KDateTime(QDate(2005,10,30), KDateTime::Spec::OffsetFromUTC(3600)), true);
    CHECK(KDateTime(QDate(2005,10,30), london) != KDateTime(QDate(2005,10,30), KDateTime::Spec::OffsetFromUTC(0)), true);
    CHECK(KDateTime(QDate(2005,10,31), london) == KDateTime(QDate(2005,10,31), KDateTime::Spec::OffsetFromUTC(0)), true);

    // Restore the original local time zone
    if (!originalZone)
        ::unsetenv("TZ");
    else
        ::setenv("TZ", originalZone, 1);
    ::tzset();
}


////////////////////
// String conversion
////////////////////

void KDateTimeTest::strings_iso8601()
{
    KTimeZone london = KSystemTimeZones::zone("Europe/London");
    bool decpt = KGlobal::locale()->decimalSymbol() == QString::fromLatin1(".");   // whether this locale uses '.' as decimal symbol

    // Ensure that local time is different from UTC and different from 'london'
    const char *originalZone = ::getenv("TZ");   // save the original local time zone
    ::setenv("TZ", ":America/Los_Angeles", 1);
    ::tzset();

    KDateTime dtlocal(QDate(1999,12,11), QTime(3,45,06,12), KDateTime::LocalZone);
    QString s = dtlocal.toString(KDateTime::ISODate);
    if (decpt)
        QCOMPARE(s, QString("1999-12-11T03:45:06.012-08:00"));
    else
        QCOMPARE(s, QString("1999-12-11T03:45:06,012-08:00"));
    KDateTime dtlocal1 = KDateTime::fromString(s, KDateTime::ISODate);
    CHECKDT(toUTC(dtlocal1.dateTime()), toUTC(dtlocal.dateTime()));
    QCOMPARE(dtlocal1.timeType(), KDateTime::OffsetFromUTC);
    QCOMPARE(dtlocal1.utcOffset(), -8*3600);
    CHECK(dtlocal1 == dtlocal, true);
    dtlocal.setDateOnly(true);
    s = dtlocal.toString(KDateTime::ISODate);
    QCOMPARE(s, QString("1999-12-11T00:00:00-08:00"));

    KDateTime dtzone(QDate(1999,6,11), QTime(3,45,06,12), london);
    s = dtzone.toString(KDateTime::ISODate);
    if (decpt)
        QCOMPARE(s, QString("1999-06-11T03:45:06.012+01:00"));
    else
        QCOMPARE(s, QString("1999-06-11T03:45:06,012+01:00"));
    KDateTime dtzone1 = KDateTime::fromString(s, KDateTime::ISODate);
    CHECKDT(toUTC(dtzone1.dateTime()), toUTC(dtzone.dateTime()));
    QCOMPARE(dtzone1.timeType(), KDateTime::OffsetFromUTC);
    QCOMPARE(dtzone1.utcOffset(), 3600);
    CHECK(dtzone1 == dtzone, true);
    dtzone.setDateOnly(true);
    s = dtzone.toString(KDateTime::ISODate);
    QCOMPARE(s, QString("1999-06-11T00:00:00+01:00"));

    KDateTime dtclock(QDate(1999,12,11), QTime(3,45,06), KDateTime::ClockTime);
    s = dtclock.toString(KDateTime::ISODate);
    QCOMPARE(s, QString("1999-12-11T03:45:06"));
    KDateTime dtclock1 = KDateTime::fromString(s, KDateTime::ISODate);
    CHECKDT(dtclock1.dateTime(), dtclock.dateTime());
    QCOMPARE(dtclock1.timeType(), KDateTime::ClockTime);
    CHECK(dtclock1 == dtclock, true);
    dtclock.setDateOnly(true);
    s = dtclock.toString(KDateTime::ISODate);
    QCOMPARE(s, QString("1999-12-11"));
    dtclock1 = KDateTime::fromString(s, KDateTime::ISODate);
    CHECK(dtclock1.isDateOnly(), true);
    QCOMPARE(dtclock1.timeType(), KDateTime::ClockTime);
    CHECKD(dtclock1.date(), QDate(1999,12,11));

    KDateTime dtutc(QDate(1999,12,11), QTime(3,45,00), KDateTime::UTC);
    s = dtutc.toString(KDateTime::ISODate);
    QCOMPARE(s, QString("1999-12-11T03:45:00Z"));
    KDateTime dtutc1 = KDateTime::fromString(s, KDateTime::ISODate);
    CHECKDT(dtutc1.dateTime(), dtutc.dateTime());
    QCOMPARE(dtutc1.timeType(), KDateTime::UTC);
    CHECK(dtutc1 == dtutc, true);
    dtutc.setDateOnly(true);
    s = dtutc.toString(KDateTime::ISODate);
    QCOMPARE(s, QString("1999-12-11T00:00:00Z"));

    // Check signed years
    s = QString("+1999-12-11T03:45:06");
    KDateTime dtpos = KDateTime::fromString(s, KDateTime::ISODate);
    CHECKDT(dtpos.dateTime(), QDateTime(QDate(1999,12,11), QTime(3,45,06)));
    QCOMPARE(dtpos.timeType(), KDateTime::ClockTime);
    KDateTime dtpos2 = KDateTime::fromString(QString("+19991211T034506"), KDateTime::ISODate);
    CHECK(dtpos2 == dtpos, true);

    dtpos.setDateOnly(true);
    s = QString("+1999-12-11");
    dtpos = KDateTime::fromString(s, KDateTime::ISODate);
    CHECK(dtpos.isDateOnly(), true);
    QCOMPARE(dtpos.timeType(), KDateTime::ClockTime);
    CHECKD(dtpos.date(), QDate(1999,12,11));
    dtpos2 = KDateTime::fromString(QString("+19991211"), KDateTime::ISODate);
    CHECK(dtpos2 == dtpos, true);

    // Check basic format strings
    bool negZero = true;
    KDateTime dt = KDateTime::fromString(QString("20000301T1213"), KDateTime::ISODate, &negZero);
    CHECK(dt.timeType() == KDateTime::ClockTime, true);
    CHECK(dt.isDateOnly(), false);
    CHECK(negZero, false);
    CHECKDT(dt.dateTime(), QDateTime(QDate(2000,3,1), QTime(12,13,0))); //, Qt::LocalTime));
    dt = KDateTime::fromString(QString("20000301"), KDateTime::ISODate, &negZero);
    CHECK(dt.timeType() == KDateTime::ClockTime, true);
    CHECK(dt.isDateOnly(), true);
    CHECK(negZero, false);
    CHECKD(dt.date(), QDate(2000,3,1));
    KDateTime::setFromStringDefault(KDateTime::UTC);
    dt = KDateTime::fromString(QString("20000301T1213"), KDateTime::ISODate);
    CHECK(dt.timeType() == KDateTime::UTC, true);
    CHECKDT(dt.dateTime(), QDateTime(QDate(2000,3,1), QTime(12,13,0))); //, Qt::UTC));
    KDateTime::setFromStringDefault(KDateTime::LocalZone);
    dt = KDateTime::fromString(QString("20000301T1213"), KDateTime::ISODate);
    CHECK(dt.timeSpec() == KDateTime::Spec::LocalZone(), true);
    CHECKDT(dt.dateTime(), QDateTime(QDate(2000,3,1), QTime(12,13,0))); //, Qt::LocalTime));
    KDateTime::setFromStringDefault(london);
    dt = KDateTime::fromString(QString("20000301T1213"), KDateTime::ISODate);
    CHECK(dt.timeType() == KDateTime::TimeZone, true);
    CHECKDT(dt.dateTime(), QDateTime(QDate(2000,3,1), QTime(12,13,0))); //, Qt::LocalTime));
    KDateTime::setFromStringDefault(KDateTime::Spec::OffsetFromUTC(5000));  // = +01:23:20
    dt = KDateTime::fromString(QString("20000601T1213"), KDateTime::ISODate);
    CHECK(dt.timeType() == KDateTime::OffsetFromUTC, true);
    QCOMPARE(dt.utcOffset(), 5000);
    CHECKDT(dt.toUtc().dateTime(), QDateTime(QDate(2000,6,1), QTime(10,49,40))); //, Qt::UTC));
    KDateTime::setFromStringDefault(KDateTime::ClockTime);
    dt = KDateTime::fromString(QString("43210301T1213"), KDateTime::ISODate, &negZero);
    CHECK(dt.timeType() == KDateTime::ClockTime, true);
    CHECK(dt.isDateOnly(), false);
    CHECK(negZero, false);
    CHECKDT(dt.dateTime(), QDateTime(QDate(4321,3,1), QTime(12,13,0))); //, Qt::LocalTime));
    dt = KDateTime::fromString(QString("43210301"), KDateTime::ISODate, &negZero);
    CHECK(dt.isDateOnly(), true);
    CHECK(negZero, false);
    CHECKD(dt.date(), QDate(4321,3,1));

    // Check strings containing day-of-the-year
    dt = KDateTime::fromString(QString("1999-060T19:20:21.06-11:20"), KDateTime::ISODate);
    CHECK(dt.timeType() == KDateTime::OffsetFromUTC, true);
    QCOMPARE(dt.utcOffset(), -11*3600 - 20*60);
    CHECKDT(dt.dateTime(), QDateTime(QDate(1999,3,1), QTime(19,20,21,60))); //, Qt::LocalTime));
    dt = KDateTime::fromString(QString("1999-060T19:20:21,06-11:20"), KDateTime::ISODate);
    CHECK(dt.timeType() == KDateTime::OffsetFromUTC, true);
    QCOMPARE(dt.utcOffset(), -11*3600 - 20*60);
    CHECKDT(dt.dateTime(), QDateTime(QDate(1999,3,1), QTime(19,20,21,60))); //, Qt::LocalTime));
    dt = KDateTime::fromString(QString("1999060T192021.06-1120"), KDateTime::ISODate);
    CHECK(dt.timeType() == KDateTime::OffsetFromUTC, true);
    QCOMPARE(dt.utcOffset(), -11*3600 - 20*60);
    CHECKDT(dt.dateTime(), QDateTime(QDate(1999,3,1), QTime(19,20,21,60))); //, Qt::LocalTime));
    dt = KDateTime::fromString(QString("1999-060"), KDateTime::ISODate);
    CHECK(dt.timeType() == KDateTime::ClockTime, true);
    CHECK(dt.isDateOnly(), true);
    CHECKD(dt.date(), QDate(1999,3,1));

    // Check 24:00:00
    dt = KDateTime::fromString(QString("1999-06-11T24:00:00+03:00"), KDateTime::ISODate);
    CHECKDT(dt.dateTime(), QDateTime(QDate(1999,6,12), QTime(0,0,0))); //, Qt::LocalTime));
    dt = KDateTime::fromString(QString("1999-06-11T24:00:01+03:00"), KDateTime::ISODate);
    CHECK(dt.isValid(), false);

    // Check leap second
    dt = KDateTime::fromString(QString("1999-06-11T23:59:60Z"), KDateTime::ISODate);
    CHECKDT(dt.dateTime(), QDateTime(QDate(1999,6,11), QTime(23,59,59))); //, Qt::UTC));
    dt = KDateTime::fromString(QString("1999-06-11T13:59:60Z"), KDateTime::ISODate);
    CHECK(dt.isValid(), false);
    dt = KDateTime::fromString(QString("1999-06-11T13:59:60-10:00"), KDateTime::ISODate);
    CHECKDT(dt.dateTime(), QDateTime(QDate(1999,6,11), QTime(13,59,59))); //, Qt::LocalTime));
    dt = KDateTime::fromString(QString("1999-06-11T23:59:60-10:00"), KDateTime::ISODate);
    CHECK(dt.isValid(), false);

    // Check negZero
    dt = KDateTime::fromString(QString("1999-060T19:20:21.06-00:00"), KDateTime::ISODate, &negZero);
    CHECK(negZero, true);
    dt = KDateTime::fromString(QString("1999-060T19:20:21.06+00:00"), KDateTime::ISODate, &negZero);
    CHECK(negZero, false);

    // Restore the original local time zone
    if (!originalZone)
        ::unsetenv("TZ");
    else
        ::setenv("TZ", originalZone, 1);
    ::tzset();
}

void KDateTimeTest::strings_rfc2822()
{
    KTimeZone london = KSystemTimeZones::zone("Europe/London");

    // Ensure that local time is different from UTC and different from 'london'
    const char *originalZone = ::getenv("TZ");   // save the original local time zone
    ::setenv("TZ", ":America/Los_Angeles", 1);
    ::tzset();

    bool negZero = true;
    KDateTime dtlocal(QDate(1999,12,11), QTime(3,45,06), KDateTime::LocalZone);
    QString s = dtlocal.toString(KDateTime::RFCDate);
    QCOMPARE(s, QString("11 Dec 1999 03:45:06 -0800"));
    KDateTime dtlocal1 = KDateTime::fromString(s, KDateTime::RFCDate, &negZero);
    CHECKDT(toUTC(dtlocal1.dateTime()), toUTC(dtlocal.dateTime()));
    QCOMPARE(dtlocal1.timeType(), KDateTime::OffsetFromUTC);
    QCOMPARE(dtlocal1.utcOffset(), -8*3600);
    CHECK(dtlocal1 == dtlocal, true);
    CHECK(negZero, false);
    KDateTime dtlocal2 = KDateTime::fromString(s, KDateTime::RFCDateDay);
    CHECK(dtlocal2.isValid(), false);
    s = dtlocal.toString(KDateTime::RFCDateDay);
    QCOMPARE(s, QString("Sat, 11 Dec 1999 03:45:06 -0800"));
    dtlocal2 = KDateTime::fromString(s, KDateTime::RFCDate);
    CHECK(dtlocal1 == dtlocal2, true);
    CHECKDT(dtlocal1.dateTime(), dtlocal2.dateTime());
    dtlocal2 = KDateTime::fromString(s, KDateTime::RFCDateDay);
    CHECK(dtlocal1 == dtlocal2, true);
    dtlocal2 = KDateTime::fromString(QString("Saturday, 11-Dec-99 03:45:06 -0800"), KDateTime::RFCDate);
    CHECK(dtlocal1 == dtlocal2, true);
    dtlocal2 = KDateTime::fromString(QString("11 Dec 1999 03:45:06 PST"), KDateTime::RFCDate, &negZero);
    CHECK(dtlocal1 == dtlocal2, true);
    CHECK(negZero, false);
    dtlocal.setDateOnly(true);
    s = dtlocal.toString(KDateTime::RFCDate);
    QCOMPARE(s, QString("11 Dec 1999 00:00 -0800"));
    s = dtlocal.toString(KDateTime::RFCDateDay);
    QCOMPARE(s, QString("Sat, 11 Dec 1999 00:00 -0800"));

    KDateTime dtzone(QDate(1999,6,11), QTime(3,45,06), london);
    s = dtzone.toString(KDateTime::RFCDate);
    QCOMPARE(s, QString("11 Jun 1999 03:45:06 +0100"));
    KDateTime dtzone1 = KDateTime::fromString(s, KDateTime::RFCDate);
    CHECKDT(toUTC(dtzone1.dateTime()), toUTC(dtzone.dateTime()));
    QCOMPARE(dtzone1.timeType(), KDateTime::OffsetFromUTC);
    QCOMPARE(dtzone1.utcOffset(), 3600);
    CHECK(dtzone1 == dtzone, true);
    KDateTime dtzone2 = KDateTime::fromString(s, KDateTime::RFCDateDay);
    CHECK(dtzone2.isValid(), false);
    s = dtzone.toString(KDateTime::RFCDateDay);
    QCOMPARE(s, QString("Fri, 11 Jun 1999 03:45:06 +0100"));
    dtzone2 = KDateTime::fromString(s, KDateTime::RFCDate);
    CHECK(dtzone1 == dtzone2, true);
    CHECKDT(dtzone1.dateTime(), dtzone2.dateTime());
    dtzone2 = KDateTime::fromString(s, KDateTime::RFCDateDay, &negZero);
    CHECK(dtzone1 == dtzone2, true);
    CHECK(negZero, false);
    dtzone2 = KDateTime::fromString(QString("Friday, 11-Jun-99 03:45:06 +0100"), KDateTime::RFCDateDay);
    CHECK(dtzone1 == dtzone2, true);
    dtzone.setDateOnly(true);
    s = dtzone.toString(KDateTime::RFCDate);
    QCOMPARE(s, QString("11 Jun 1999 00:00 +0100"));
    s = dtzone.toString(KDateTime::RFCDateDay);
    QCOMPARE(s, QString("Fri, 11 Jun 1999 00:00 +0100"));

    KDateTime dtclock(QDate(1999,12,11), QTime(3,45,06), KDateTime::ClockTime);
    s = dtclock.toString(KDateTime::RFCDate);
    QCOMPARE(s, QString("11 Dec 1999 03:45:06 -0800"));
    KDateTime dtclock1 = KDateTime::fromString(s, KDateTime::RFCDate, &negZero);
    CHECKDT(dtclock1.dateTime(), dtclock.dateTime());
    QCOMPARE(dtclock1.timeType(), KDateTime::OffsetFromUTC);
    QCOMPARE(dtclock1.utcOffset(), -8*3600);
    CHECK(dtclock1 == dtclock, true);
    CHECK(negZero, false);
    KDateTime dtclock2 = KDateTime::fromString(s, KDateTime::RFCDateDay);
    CHECK(dtclock2.isValid(), false);
    s = dtclock.toString(KDateTime::RFCDateDay);
    QCOMPARE(s, QString("Sat, 11 Dec 1999 03:45:06 -0800"));
    dtclock2 = KDateTime::fromString(s, KDateTime::RFCDate);
    CHECK(dtclock1 == dtclock2, true);
    CHECKDT(dtclock1.dateTime(), dtclock2.dateTime());
    dtclock2 = KDateTime::fromString(s, KDateTime::RFCDateDay);
    CHECK(dtclock1 == dtclock2, true);
    dtclock2 = KDateTime::fromString(QString("Saturday, 11-Dec-99 03:45:06 -0800"), KDateTime::RFCDateDay);
    CHECK(dtclock1 == dtclock2, true);
    dtclock.setDateOnly(true);
    s = dtclock.toString(KDateTime::RFCDate);
    QCOMPARE(s, QString("11 Dec 1999 00:00 -0800"));
    s = dtclock.toString(KDateTime::RFCDateDay);
    QCOMPARE(s, QString("Sat, 11 Dec 1999 00:00 -0800"));

    KDateTime dtutc(QDate(1999,12,11), QTime(3,45,00), KDateTime::UTC);
    s = dtutc.toString(KDateTime::RFCDate);
    QCOMPARE(s, QString("11 Dec 1999 03:45 +0000"));
    KDateTime dtutc1 = KDateTime::fromString(s, KDateTime::RFCDate, &negZero);
    CHECKDT(dtutc1.dateTime(), dtutc.dateTime());
    QCOMPARE(dtutc1.timeType(), KDateTime::UTC);
    CHECK(dtutc1 == dtutc, true);
    CHECK(negZero, false);
    KDateTime dtutc2 = KDateTime::fromString(s, KDateTime::RFCDateDay);
    CHECK(dtutc2.isValid(), false);
    s = dtutc.toString(KDateTime::RFCDateDay);
    QCOMPARE(s, QString("Sat, 11 Dec 1999 03:45 +0000"));
    dtutc2 = KDateTime::fromString(s, KDateTime::RFCDate);
    CHECK(dtutc1 == dtutc2, true);
    CHECKDT(dtutc1.dateTime(), dtutc2.dateTime());
    dtutc2 = KDateTime::fromString(s, KDateTime::RFCDateDay);
    CHECK(dtutc1 == dtutc2, true);
    dtutc2 = KDateTime::fromString(QString("Saturday, 11-Dec-99 03:45 +0000"), KDateTime::RFCDate);
    CHECK(dtutc1 == dtutc2, true);
    dtutc.setDateOnly(true);
    s = dtutc.toString(KDateTime::RFCDate);
    QCOMPARE(s, QString("11 Dec 1999 00:00 +0000"));
    s = dtutc.toString(KDateTime::RFCDateDay);
    QCOMPARE(s, QString("Sat, 11 Dec 1999 00:00 +0000"));

    // Check '-0000' and unknown/invalid time zone names
    dtutc2 = KDateTime::fromString(QString("11 Dec 1999 03:45 -0000"), KDateTime::RFCDate, &negZero);
    CHECK(dtutc1 == dtutc2, true);
    CHECK(negZero, true);
    dtutc2 = KDateTime::fromString(QString("11 Dec 1999 03:45 B"), KDateTime::RFCDate, &negZero);
    CHECK(dtutc1 == dtutc2, true);
    CHECK(negZero, true);
    dtutc2 = KDateTime::fromString(QString("11 Dec 1999 03:45 BCDE"), KDateTime::RFCDate, &negZero);
    CHECK(dtutc1 == dtutc2, true);
    CHECK(negZero, true);

    // Check named time offsets
    KDateTime dtzname = KDateTime::fromString(QString("11 Dec 1999 03:45:06 UT"), KDateTime::RFCDate, &negZero);
    CHECK(dtzname.timeType() == KDateTime::UTC, true);
    CHECKDT(dtzname.dateTime(), QDateTime(QDate(1999,12,11), QTime(3,45,6))); //, Qt::UTC));
    CHECK(negZero, false);
    dtzname = KDateTime::fromString(QString("11 Dec 1999 03:45:06 GMT"), KDateTime::RFCDate, &negZero);
    CHECK(dtzname.timeType() == KDateTime::UTC, true);
    CHECKDT(dtzname.dateTime(), QDateTime(QDate(1999,12,11), QTime(3,45,6))); //, Qt::UTC));
    CHECK(negZero, false);
    dtzname = KDateTime::fromString(QString("11 Dec 1999 03:45:06 EDT"), KDateTime::RFCDate, &negZero);
    CHECK(dtzname.timeType() == KDateTime::OffsetFromUTC, true);
    QCOMPARE(dtzname.utcOffset(), -4*3600);
    CHECKDT(dtzname.dateTime(), QDateTime(QDate(1999,12,11), QTime(3,45,6))); //, Qt::LocalTime));
    CHECK(negZero, false);
    dtzname = KDateTime::fromString(QString("11 Dec 1999 03:45:06 EST"), KDateTime::RFCDate, &negZero);
    CHECK(dtzname.timeType() == KDateTime::OffsetFromUTC, true);
    QCOMPARE(dtzname.utcOffset(), -5*3600);
    CHECKDT(dtzname.dateTime(), QDateTime(QDate(1999,12,11), QTime(3,45,6))); //, Qt::LocalTime));
    CHECK(negZero, false);
    dtzname = KDateTime::fromString(QString("11 Dec 1999 03:45:06 CDT"), KDateTime::RFCDate, &negZero);
    CHECK(dtzname.timeType() == KDateTime::OffsetFromUTC, true);
    QCOMPARE(dtzname.utcOffset(), -5*3600);
    CHECKDT(dtzname.dateTime(), QDateTime(QDate(1999,12,11), QTime(3,45,6))); //, Qt::LocalTime));
    CHECK(negZero, false);
    dtzname = KDateTime::fromString(QString("11 Dec 1999 03:45:06 CST"), KDateTime::RFCDate, &negZero);
    CHECK(dtzname.timeType() == KDateTime::OffsetFromUTC, true);
    QCOMPARE(dtzname.utcOffset(), -6*3600);
    CHECKDT(dtzname.dateTime(), QDateTime(QDate(1999,12,11), QTime(3,45,6))); //, Qt::LocalTime));
    CHECK(negZero, false);
    dtzname = KDateTime::fromString(QString("11 Dec 1999 03:45:06 MDT"), KDateTime::RFCDate, &negZero);
    CHECK(dtzname.timeType() == KDateTime::OffsetFromUTC, true);
    QCOMPARE(dtzname.utcOffset(), -6*3600);
    CHECKDT(dtzname.dateTime(), QDateTime(QDate(1999,12,11), QTime(3,45,6))); //, Qt::LocalTime));
    CHECK(negZero, false);
    dtzname = KDateTime::fromString(QString("11 Dec 1999 03:45:06 MST"), KDateTime::RFCDate, &negZero);
    CHECK(dtzname.timeType() == KDateTime::OffsetFromUTC, true);
    QCOMPARE(dtzname.utcOffset(), -7*3600);
    CHECKDT(dtzname.dateTime(), QDateTime(QDate(1999,12,11), QTime(3,45,6))); //, Qt::LocalTime));
    CHECK(negZero, false);
    dtzname = KDateTime::fromString(QString("11 Dec 1999 03:45:06 PDT"), KDateTime::RFCDate, &negZero);
    CHECK(dtzname.timeType() == KDateTime::OffsetFromUTC, true);
    QCOMPARE(dtzname.utcOffset(), -7*3600);
    CHECKDT(dtzname.dateTime(), QDateTime(QDate(1999,12,11), QTime(3,45,6))); //, Qt::LocalTime));
    CHECK(negZero, false);
    dtzname = KDateTime::fromString(QString("11 Dec 1999 03:45:06 PST"), KDateTime::RFCDate, &negZero);
    CHECK(dtzname.timeType() == KDateTime::OffsetFromUTC, true);
    QCOMPARE(dtzname.utcOffset(), -8*3600);
    CHECKDT(dtzname.dateTime(), QDateTime(QDate(1999,12,11), QTime(3,45,6))); //, Qt::LocalTime));
    CHECK(negZero, false);

    // Check leap second
    KDateTime dt = KDateTime::fromString(QString("11 Dec 1999 23:59:60 -0000"), KDateTime::RFCDate);
    CHECKDT(dt.dateTime(), QDateTime(QDate(1999,12,11), QTime(23,59,59))); //, Qt::UTC));
    dt = KDateTime::fromString(QString("11 Dec 1999 13:59:60 -0000"), KDateTime::RFCDate);
    CHECK(dt.isValid(), false);
    dt = KDateTime::fromString(QString("11 Jun 1999 13:59:60 -1000"), KDateTime::RFCDate);
    CHECKDT(dt.dateTime(), QDateTime(QDate(1999,6,11), QTime(13,59,59))); //, Qt::LocalTime));
    dt = KDateTime::fromString(QString("11 Dec 1999 23:59:60 -1000"), KDateTime::RFCDate);
    CHECK(dt.isValid(), false);

    // Check erroneous strings:
    dtutc2 = KDateTime::fromString(QString("11 Dec 1999 23:59:60 -00:00"), KDateTime::RFCDate);
    CHECK(dtutc2.isValid(), false);     // colon in UTC offset
    dtutc2 = KDateTime::fromString(QString("Sun, 11 Dec 1999 03:45 +0000"), KDateTime::RFCDate);
    CHECK(dtutc2.isValid(), false);     // wrong weekday
    dtutc2 = KDateTime::fromString(QString("Satu, 11 Dec 1999 03:45 +0000"), KDateTime::RFCDate);
    CHECK(dtutc2.isValid(), false);     // bad weekday
    dtutc2 = KDateTime::fromString(QString("11 Dece 1999 03:45 +0000"), KDateTime::RFCDate);
    CHECK(dtutc2.isValid(), false);     // bad month
    dtutc2 = KDateTime::fromString(QString("11-Dec 1999 03:45 +0000"), KDateTime::RFCDate);
    CHECK(dtutc2.isValid(), false);     // only one hyphen in date

    // Restore the original local time zone
    if (!originalZone)
        ::unsetenv("TZ");
    else
        ::setenv("TZ", originalZone, 1);
    ::tzset();
}

void KDateTimeTest::strings_qttextdate()
{
    KTimeZone london = KSystemTimeZones::zone("Europe/London");

    // Ensure that local time is different from UTC and different from 'london'
    const char *originalZone = ::getenv("TZ");   // save the original local time zone
    ::setenv("TZ", ":America/Los_Angeles", 1);
    ::tzset();

    bool negZero = true;
    KDateTime dtlocal(QDate(1999,12,11), QTime(3,45,06), KDateTime::LocalZone);
    QString s = dtlocal.toString(KDateTime::QtTextDate);
    QCOMPARE(s, QString("Sat Dec 11 03:45:06 1999 -0800"));
    KDateTime dtlocal1 = KDateTime::fromString(s, KDateTime::QtTextDate, &negZero);
    CHECKDT(toUTC(dtlocal1.dateTime()), toUTC(dtlocal.dateTime()));
    QCOMPARE(dtlocal1.timeType(), KDateTime::OffsetFromUTC);
    QCOMPARE(dtlocal1.utcOffset(), -8*3600);
    CHECK(dtlocal1 == dtlocal, true);
    CHECK(dtlocal1.isDateOnly(), false);
    CHECK(negZero, false);
    dtlocal.setDateOnly(true);
    s = dtlocal.toString(KDateTime::QtTextDate);
    QCOMPARE(s, QString("Sat Dec 11 1999 -0800"));
    dtlocal1 = KDateTime::fromString(s, KDateTime::QtTextDate, &negZero);
    CHECK(dtlocal1.isDateOnly(), true);
    CHECKD(dtlocal1.date(), QDate(1999,12,11));
    QCOMPARE(dtlocal1.timeType(), KDateTime::OffsetFromUTC);
    QCOMPARE(dtlocal1.utcOffset(), -8*3600);

    KDateTime dtzone(QDate(1999,6,11), QTime(3,45,06), london);
    s = dtzone.toString(KDateTime::QtTextDate);
    QCOMPARE(s, QString("Fri Jun 11 03:45:06 1999 +0100"));
    KDateTime dtzone1 = KDateTime::fromString(s, KDateTime::QtTextDate);
    CHECKDT(toUTC(dtzone1.dateTime()), toUTC(dtzone.dateTime()));
    QCOMPARE(dtzone1.timeType(), KDateTime::OffsetFromUTC);
    QCOMPARE(dtzone1.utcOffset(), 3600);
    CHECK(dtzone1.isDateOnly(), false);
    CHECK(dtzone1 == dtzone, true);
    KDateTime dtzone2 = KDateTime::fromString(s, KDateTime::QtTextDate, &negZero);
    CHECK(dtzone1 == dtzone2, true);
    CHECK(negZero, false);
    dtzone.setDateOnly(true);
    s = dtzone.toString(KDateTime::QtTextDate);
    QCOMPARE(s, QString("Fri Jun 11 1999 +0100"));
    dtzone1 = KDateTime::fromString(s, KDateTime::QtTextDate, &negZero);
    CHECK(dtzone1.isDateOnly(), true);
    CHECKD(dtzone1.date(), QDate(1999,6,11));
    QCOMPARE(dtzone1.timeType(), KDateTime::OffsetFromUTC);
    QCOMPARE(dtzone1.utcOffset(), 3600);

    KDateTime dtclock(QDate(1999,12,11), QTime(3,45,06), KDateTime::ClockTime);
    s = dtclock.toString(KDateTime::QtTextDate);
    QCOMPARE(s, QString("Sat Dec 11 03:45:06 1999"));
    KDateTime dtclock1 = KDateTime::fromString(s, KDateTime::QtTextDate, &negZero);
    CHECKDT(dtclock1.dateTime(), dtclock.dateTime());
    QCOMPARE(dtclock1.timeType(), KDateTime::ClockTime);
    CHECK(dtclock1 == dtclock, true);
    CHECK(dtclock1.isDateOnly(), false);
    CHECK(negZero, false);
    dtclock.setDateOnly(true);
    s = dtclock.toString(KDateTime::QtTextDate);
    QCOMPARE(s, QString("Sat Dec 11 1999"));
    dtclock1 = KDateTime::fromString(s, KDateTime::QtTextDate, &negZero);
    CHECK(dtclock1.isDateOnly(), true);
    CHECKD(dtclock1.date(), QDate(1999,12,11));
    QCOMPARE(dtclock1.timeType(), KDateTime::ClockTime);

    KDateTime dtutc(QDate(1999,12,11), QTime(3,45,00), KDateTime::UTC);
    s = dtutc.toString(KDateTime::QtTextDate);
    QCOMPARE(s, QString("Sat Dec 11 03:45:00 1999 +0000"));
    KDateTime dtutc1 = KDateTime::fromString(s, KDateTime::QtTextDate, &negZero);
    CHECKDT(dtutc1.dateTime(), dtutc.dateTime());
    QCOMPARE(dtutc1.timeType(), KDateTime::UTC);
    CHECK(dtutc1 == dtutc, true);
    CHECK(dtutc1.isDateOnly(), false);
    CHECK(negZero, false);
    dtutc.setDateOnly(true);
    s = dtutc.toString(KDateTime::QtTextDate);
    QCOMPARE(s, QString("Sat Dec 11 1999 +0000"));
    dtutc1 = KDateTime::fromString(s, KDateTime::QtTextDate, &negZero);
    CHECK(dtutc1.isDateOnly(), true);
    CHECKD(dtutc1.date(), QDate(1999,12,11));
    QCOMPARE(dtutc1.timeType(), KDateTime::UTC);

    // Check '-0000'
    KDateTime dtutc2 = KDateTime::fromString(QString("Sat Dec 11 03:45:00 1999 -0000"), KDateTime::QtTextDate, &negZero);
    CHECK(dtutc1 != dtutc2, true);
    CHECK(negZero, true);

    // Check erroneous strings
    dtutc2 = KDateTime::fromString(QString("Sat Dec 11 03:45:00 1999 GMT"), KDateTime::QtTextDate, &negZero);
    CHECK(dtutc2.isValid(), false);
    dtutc2 = KDateTime::fromString(QString("Sun Dec 11 03:45:00 1999 +0000"), KDateTime::QtTextDate);
    CHECK(dtutc2.isValid(), true);     // wrong weekday: accepted by Qt!!
//    QTest::ignoreMessage(QtWarningMsg, "QDateTime::fromString: Parameter out of range");
    dtutc2 = KDateTime::fromString(QString("Satu, Dec 11 03:45:00 1999 +0000"), KDateTime::QtTextDate);
    CHECK(dtutc2.isValid(), false);     // bad weekday
    dtutc2 = KDateTime::fromString(QString("Sat Dece 11 03:45:00 1999 +0000"), KDateTime::QtTextDate);
    CHECK(dtutc2.isValid(), true);     // bad month: accepted by Qt!!

    // Restore the original local time zone
    if (!originalZone)
        ::unsetenv("TZ");
    else
        ::setenv("TZ", originalZone, 1);
    ::tzset();
}

void KDateTimeTest::strings_format()
{
    KLocale* locale = KGlobal::locale();
    KTimeZone london = KSystemTimeZones::zone("Europe/London");
    KTimeZone paris  = KSystemTimeZones::zone("Europe/Paris");
    KTimeZone berlin = KSystemTimeZones::zone("Europe/Berlin");
    KTimeZone cairo  = KSystemTimeZones::zone("Africa/Cairo");
    KTimeZones zones;
    zones.add(london);
    zones.add(paris);
    zones.add(berlin);
    zones.add(cairo);

    // Ensure that local time is different from UTC and different from 'london'
    const char *originalZone = ::getenv("TZ");   // save the original local time zone
    ::setenv("TZ", ":America/Los_Angeles", 1);
    ::tzset();

    // toString()
    QString all = QString::fromLatin1("%Y.%y.%m.%:m.%B.%b.%d.%e.%A.%a-%H.%k.%I.%l.%M.%S?%:s?%P.%p.%:u.%z.%Z.%:Z.%:A.%:a.%:B.%:b/%:S.%:z.%%.");
    KDateTime dt(QDate(1999,2,3), QTime(6,5,0), KDateTime::LocalZone);
    QString s = dt.toString(all);
    QCOMPARE(s, QString::fromLatin1("1999.99.02.2.%1.%2.03.3.%3.%4-06.6.06.6.05.00?000?am.AM.-08.-0800.PST.America/Los_Angeles.Wednesday.Wed.February.Feb/.-08:00.%.")
                                   .arg(locale->monthName(2,false)).arg(locale->monthName(2,true))
                                   .arg(locale->weekDayName(3,false)).arg(locale->weekDayName(3,true)));

    KDateTime dtzone(QDate(2006,4,30), QTime(12,45,16,25), london);
    s = dtzone.toString(all);
    QCOMPARE(s, QString::fromLatin1("2006.06.04.4.%1.%2.30.30.%3.%4-12.12.12.12.45.16?025?pm.PM.+01.+0100.BST.Europe/London.Sunday.Sun.April.Apr/:16.+01:00.%.")
                                   .arg(locale->monthName(4,false)).arg(locale->monthName(4,true))
                                   .arg(locale->weekDayName(7,false)).arg(locale->weekDayName(7,true)));

    KDateTime dtclock(QDate(2005,9,5), QTime(0,0,06,1), KDateTime::ClockTime);
    s = dtclock.toString(all);
    QCOMPARE(s, QString::fromLatin1("2005.05.09.9.%1.%2.05.5.%3.%4-00.0.12.12.00.06?001?am.AM.....Monday.Mon.September.Sep/:06..%.")
                                   .arg(locale->monthName(9,false)).arg(locale->monthName(9,true))
                                   .arg(locale->weekDayName(1,false)).arg(locale->weekDayName(1,true)));

    KDateTime dtutc(QDate(2000,12,31), QTime(13,45,16,100), KDateTime::UTC);
    s = dtutc.toString(all);
    QCOMPARE(s, QString::fromLatin1("2000.00.12.12.%1.%2.31.31.%3.%4-13.13.01.1.45.16?100?pm.PM.+00.+0000.UTC.UTC.Sunday.Sun.December.Dec/:16.+00:00.%.")
                                   .arg(locale->monthName(12,false)).arg(locale->monthName(12,true))
                                   .arg(locale->weekDayName(7,false)).arg(locale->weekDayName(7,true)));

    // fromString() without KTimeZones parameter
    dt = KDateTime::fromString(QString::fromLatin1("2005/9/05/20:2,03"), QString::fromLatin1("%Y/%:m/%d/%S:%k,%M"));
    CHECKDT(dt.dateTime(), QDateTime(QDate(2005,9,5), QTime(2,3,20))); //, Qt::LocalTime));
    QCOMPARE(dt.timeType(), KDateTime::ClockTime);

    dt = KDateTime::fromString(QString::fromLatin1("%1pm05ab%2t/052/20:2,03+10").arg(locale->weekDayName(1,false)).arg(locale->monthName(9,false)),
                               QString::fromLatin1("%a%p%yab%Bt/%e2/%S:%l,%M %z"));
    CHECKDT(dt.dateTime(), QDateTime(QDate(2005,9,5), QTime(14,3,20))); //, Qt::LocalTime));
    QCOMPARE(dt.timeType(), KDateTime::OffsetFromUTC);
    QCOMPARE(dt.utcOffset(), 10*3600);
    dt = KDateTime::fromString(QString::fromLatin1("%1pm05ab%2t/052/20:2,03+10").arg(locale->weekDayName(1,true)).arg(locale->monthName(9,true)),
                               QString::fromLatin1("%a%p%yab%Bt/%d2/%S:%l,%:M %z"));
    CHECKDT(dt.dateTime(), QDateTime(QDate(2005,9,5), QTime(14,3,20))); //, Qt::LocalTime));
    QCOMPARE(dt.timeType(), KDateTime::OffsetFromUTC);
    QCOMPARE(dt.utcOffset(), 10*3600);
    dt = KDateTime::fromString(QString::fromLatin1("monpm05absEpt/052/20:2,03+10"), QString::fromLatin1("%a%p%yab%Bt/%d2/%S:%l,%M %z"));
    CHECKDT(dt.dateTime(), QDateTime(QDate(2005,9,5), QTime(14,3,20))); //, Qt::LocalTime));
    QCOMPARE(dt.timeType(), KDateTime::OffsetFromUTC);
    QCOMPARE(dt.utcOffset(), 10*3600);
    dt = KDateTime::fromString(QString::fromLatin1("monDAYpm05absEptemBert/052/20:2,03+10"), QString::fromLatin1("%a%p%yab%Bt/%e2/%S:%l,%M %z"));
    CHECKDT(dt.dateTime(), QDateTime(QDate(2005,9,5), QTime(14,3,20))); //, Qt::LocalTime));
    QCOMPARE(dt.timeType(), KDateTime::OffsetFromUTC);
    QCOMPARE(dt.utcOffset(), 10*3600);
    dt = KDateTime::fromString(QString::fromLatin1("monDAYpm05abmzatemer/052/20:2,03+10"), QString::fromLatin1("%a%p%yab%B/%e2/%S:%l,%M %z"));
    CHECK(dt.isValid(), false);    // invalid month name
    dt = KDateTime::fromString(QString::fromLatin1("monDApm05absep/052/20:2,03+10"), QString::fromLatin1("%a%p%yab%B/%e2/%S:%l,%M %z"));
    CHECK(dt.isValid(), false);    // invalid day name
    dt = KDateTime::fromString(QString::fromLatin1("mONdAYPM2005absEpt/052/20:02,03+1000"), QString::fromLatin1("%:A%:p%Yab%Bt/%d2/%S:%I,%M %:u"));
    CHECKDT(dt.dateTime(), QDateTime(QDate(2005,9,5), QTime(14,3,20))); //, Qt::LocalTime));
    QCOMPARE(dt.utcOffset(), 10*3600);
    QCOMPARE(dt.timeType(), KDateTime::OffsetFromUTC);
    dtclock = KDateTime::fromString(QString::fromLatin1("mONdAYPM2005abSept/052/20:02,03+100"), QString::fromLatin1("%:A%:p%Yab%Bt/%e2/%S:%l,%M %:u"));
    CHECK(dtclock.isValid(), false);    // wrong number of digits in UTC offset
    dtclock = KDateTime::fromString(QString::fromLatin1("mONdAYPM2005abSept/052/20:02,03+1"), QString::fromLatin1("%:A%:p%Yab%Bt/%d2/%S:%I,%M %z"));
    CHECK(dtclock.isValid(), false);    // wrong number of digits in UTC offset
    dtclock = KDateTime::fromString(QString::fromLatin1("mONdAYPM2005absEpt/052/20:13,03+1000"), QString::fromLatin1("%:A%:p%Yab%Bt/%d2/%S:%I,%M %:u"));
    CHECK(dtclock.isValid(), false);    // hours out of range for am/pm
    dtclock = KDateTime::fromString(QString::fromLatin1("mONdAYPM2005absEpt/052/20:00,03+1000"), QString::fromLatin1("%:A%:p%Yab%Bt/%d2/%S:%I,%M %:u"));
    CHECK(dtclock.isValid(), false);    // hours out of range for am/pm

    // fromString() with KTimeZones parameter
    dt = KDateTime::fromString(QString::fromLatin1("mon 2005/9/05/20:2,03"), QString::fromLatin1("%:a %Y/%:m/%e/%S:%k,%M"), &zones);
    CHECKDT(dt.dateTime(), QDateTime(QDate(2005,9,5), QTime(2,3,20))); //, Qt::LocalTime));
    QCOMPARE(dt.timeType(), KDateTime::ClockTime);
    dt = KDateTime::fromString(QString::fromLatin1("tue 2005/9/05/20:2,03"), QString::fromLatin1("%:a %Y/%:m/%d/%S:%k,%M"), &zones);
    CHECK(dt.isValid(), false);    // wrong day-of-week

    dt = KDateTime::fromString(QString::fromLatin1("pm2005absEpt/05monday/20:2,03+03:00"), QString::fromLatin1("%p%Yab%Bt/%e%:A/%S:%l,%M %:z"), &zones);
    CHECKDT(dt.dateTime(), QDateTime(QDate(2005,9,5), QTime(14,3,20))); //, Qt::LocalTime));
    QCOMPARE(dt.timeType(), KDateTime::TimeZone);
    QCOMPARE(dt.utcOffset(), 3*3600);
    CHECKTZ(dt.timeZone(), cairo);
    dt = KDateTime::fromString(QString::fromLatin1("pm2005absEpt/05sunday/20:2,03+03:00"), QString::fromLatin1("%p%Yab%Bt/%d%A/%S:%l,%M %:z"), &zones);
    CHECK(dt.isValid(), false);    // wrong day-of-week

    dt = KDateTime::fromString(QString::fromLatin1("200509051430:01.3+0100"), QString::fromLatin1("%Y%m%d%H%M%:S%:s%z"), &zones, true);
    CHECKDT(dt.dateTime(), QDateTime(QDate(2005,9,5), QTime(14,30,01,300))); //, Qt::LocalTime));
    QCOMPARE(dt.timeType(), KDateTime::TimeZone);
    CHECKTZ(dt.timeZone(), london);
    QCOMPARE(dt.utcOffset(), 3600);

    dt = KDateTime::fromString(QString::fromLatin1("200509051430:01.3+0500"), QString::fromLatin1("%Y%m%d%H%M%:S%:s%z"), &zones, false);
    CHECKDT(dt.dateTime(), QDateTime(QDate(2005,9,5), QTime(14,30,01,300))); //, Qt::LocalTime));
    QCOMPARE(dt.timeType(), KDateTime::OffsetFromUTC);
    QCOMPARE(dt.utcOffset(), 5*3600);

    dt = KDateTime::fromString(QString::fromLatin1("200509051430:01.3+0200"), QString::fromLatin1("%Y%m%d%H%M%:S%:s%z"), &zones, true);
    CHECKDT(dt.dateTime(), QDateTime(QDate(2005,9,5), QTime(14,30,01,300))); //, Qt::LocalTime));
    QCOMPARE(dt.timeType(), KDateTime::OffsetFromUTC);
    QCOMPARE(dt.utcOffset(), 2*3600);
    dt = KDateTime::fromString(QString::fromLatin1("200509051430:01.3+0200"), QString::fromLatin1("%Y%m%d%H%M%:S%:s%z"), &zones, false);
    CHECK(dt.isValid(), false);    // matches paris and berlin

    dt = KDateTime::fromString(QString::fromLatin1("2005September051430 CEST"), QString::fromLatin1("%Y%:B%d%H%M%:S %Z"), &zones, true);
    CHECKDT(dt.dateTime(), QDateTime(QDate(2005,9,5), QTime(14,30,0))); //, Qt::LocalTime));
    QCOMPARE(dt.timeType(), KDateTime::OffsetFromUTC);
    QCOMPARE(dt.utcOffset(), 2*3600);
    dt = KDateTime::fromString(QString::fromLatin1("2005September051430 CEST"), QString::fromLatin1("%Y%:B%d%H%M%:S %Z"), &zones, false);
    CHECK(dt.isValid(), false);    // matches paris and berlin

    dt = KDateTime::fromString(QString::fromLatin1("pm05absEptembeRt/   052/   20:12,03+0100"), QString::fromLatin1("%:P%yab%:bt/  %e2/%t%S:%l,%M %z"), &zones);
    CHECKDT(dt.dateTime(), QDateTime(QDate(2005,9,5), QTime(12,3,20))); //, Qt::LocalTime));
    QCOMPARE(dt.timeType(), KDateTime::TimeZone);
    QCOMPARE(dt.utcOffset(), 3600);
    CHECKTZ(dt.timeZone(), london);

    dt = KDateTime::fromString(QString::fromLatin1("2005absEpt/042sun/20.0123456:12Am,3Africa/Cairo%"), QString::fromLatin1("%Yab%bt/%e2%a/%S%:s:%I%P,%:M %:Z%%"), &zones);
    CHECKDT(dt.dateTime(), QDateTime(QDate(2005,9,4), QTime(0,3,20,12))); //, Qt::LocalTime));
    QCOMPARE(dt.timeType(), KDateTime::TimeZone);
    CHECKTZ(dt.timeZone(), cairo);
    QCOMPARE(dt.utcOffset(), 3*3600);

    // Test large and minimum date values
    dt = KDateTime(QDate(1753,9,5), QTime(0,0,06,1), KDateTime::ClockTime);
    s = dt.toString(QString::fromLatin1("%Y"));
    QCOMPARE(s, QString::fromLatin1("1753"));

    dt = KDateTime::fromString(QString::fromLatin1("175309051430:01.3+0500"), QString::fromLatin1("%Y%m%d%H%M%:S%:s%z"));
    CHECKDT(dt.dateTime(), QDateTime(QDate(1753,9,5), QTime(14,30,1,300))); //, Qt::LocalTime));
    QCOMPARE(dt.utcOffset(), 5*3600);
    CHECK(dt.isValid(), true);
    CHECK(dt.outOfRange(), false);

    dt = KDateTime::fromString(QString::fromLatin1("799909051430:01.3+0500"), QString::fromLatin1("%Y%m%d%H%M%:S%:s%z"));
    CHECKDT(dt.dateTime(), QDateTime(QDate(7999,9,5), QTime(14,30,1,300))); //, Qt::LocalTime));
    QCOMPARE(dt.utcOffset(), 5*3600);
    CHECK(dt.isValid(), true);
    CHECK(dt.outOfRange(), false);

    dt = KDateTime::fromString(QString::fromLatin1("175112311430:01.3+0500"), QString::fromLatin1("%Y%m%d%H%M%:S%:s%z"));
    CHECK(dt.isValid(), false);    // too early
    CHECK(dt.outOfRange(), true);


    // Restore the original local time zone
    if (!originalZone)
        ::unsetenv("TZ");
    else
        ::setenv("TZ", originalZone, 1);
    ::tzset();
}

void KDateTimeTest::cache()
{
    KTimeZone london = KSystemTimeZones::zone("Europe/London");
    KTimeZone losAngeles = KSystemTimeZones::zone("America/Los_Angeles");
    KTimeZone cairo = KSystemTimeZones::zone("Africa/Cairo");

    const char *originalZone = ::getenv("TZ");   // save the original local time zone
    ::setenv("TZ", ":Europe/London", 1);
    ::tzset();

    // Ensure that local time is different from UTC and different from 'london'
    ::setenv("TZ", ":America/Los_Angeles", 1);
    ::tzset();

    int utcHit  = KDateTime_utcCacheHit;
    int zoneHit = KDateTime_zoneCacheHit;
    KDateTime local(QDate(2005,6,1), QTime(12,0,0), KDateTime::LocalZone);
    QCOMPARE(KDateTime_utcCacheHit, utcHit);
    QCOMPARE(KDateTime_zoneCacheHit, zoneHit);
    KDateTime dt1 = local.toZone(london);
    QCOMPARE(KDateTime_utcCacheHit, utcHit);
    QCOMPARE(KDateTime_zoneCacheHit, zoneHit);
    KDateTime cai = local.toZone(cairo);
    ++utcHit;
    QCOMPARE(KDateTime_utcCacheHit, utcHit);
    QCOMPARE(KDateTime_zoneCacheHit, zoneHit);
    KDateTime dt2a = local.toZone(london);
    ++utcHit;
    QCOMPARE(KDateTime_utcCacheHit, utcHit);
    QCOMPARE(KDateTime_zoneCacheHit, zoneHit);
    KDateTime dt2 = local.toZone(london);
    ++zoneHit;
    QCOMPARE(KDateTime_utcCacheHit, utcHit);
    QCOMPARE(KDateTime_zoneCacheHit, zoneHit);
    KDateTime dt3 = dt2;
    QCOMPARE(KDateTime_utcCacheHit, utcHit);
    QCOMPARE(KDateTime_zoneCacheHit, zoneHit);
    KDateTime dt4 = dt2.toZone(losAngeles);
    ++zoneHit;
    QCOMPARE(KDateTime_utcCacheHit, utcHit);
    QCOMPARE(KDateTime_zoneCacheHit, zoneHit);
    KDateTime dt4a = dt3.toZone(losAngeles);
    ++zoneHit;
    QCOMPARE(KDateTime_utcCacheHit, utcHit);
    QCOMPARE(KDateTime_zoneCacheHit, zoneHit);
    KDateTime dt5 = dt2.toZone(losAngeles);
    ++zoneHit;
    QCOMPARE(KDateTime_utcCacheHit, utcHit);
    QCOMPARE(KDateTime_zoneCacheHit, zoneHit);
    KDateTime dt5a = dt3.toZone(losAngeles);
    ++zoneHit;
    QCOMPARE(KDateTime_utcCacheHit, utcHit);
    QCOMPARE(KDateTime_zoneCacheHit, zoneHit);
    KDateTime dt6 = dt2.toZone(cairo);
    ++utcHit;
    QCOMPARE(KDateTime_utcCacheHit, utcHit);
    QCOMPARE(KDateTime_zoneCacheHit, zoneHit);
    KDateTime dt6a = dt3.toZone(cairo);
    ++zoneHit;
    QCOMPARE(KDateTime_utcCacheHit, utcHit);
    QCOMPARE(KDateTime_zoneCacheHit, zoneHit);
    dt3.detach();
    KDateTime dt7 = dt2.toZone(london);
    QCOMPARE(KDateTime_utcCacheHit, utcHit);
    QCOMPARE(KDateTime_zoneCacheHit, zoneHit);
    KDateTime dt7a = dt3.toZone(london);
    QCOMPARE(KDateTime_utcCacheHit, utcHit);
    QCOMPARE(KDateTime_zoneCacheHit, zoneHit);

    // Check that cached time zone conversions are cleared correctly
    KDateTime utc1(QDate(2005,7,6), QTime(3,40,0), KDateTime::UTC);
    KDateTime la1 = utc1.toTimeSpec(losAngeles);
    KDateTime utc2 = utc1.addDays(1);
    KDateTime la2 = utc2.toTimeSpec(losAngeles);
    CHECK(la1 != la2, true);
    QCOMPARE(la1.secsTo(la2), 86400);

    // Restore the original local time zone
    if (!originalZone)
        ::unsetenv("TZ");
    else
        ::setenv("TZ", originalZone, 1);
    ::tzset();
}

void KDateTimeTest::misc()
{
    // Ensure that local time is different from UTC and different from 'london'
    const char *originalZone = ::getenv("TZ");   // save the original local time zone
    ::setenv("TZ", ":America/Los_Angeles", 1);
    ::tzset();

    KDateTime local = KDateTime::currentLocalDateTime();
    KDateTime utc = KDateTime::currentUtcDateTime();
    QDateTime qcurrent = QDateTime::currentDateTime();
    // Because 3 calls to fetch the current time were made, they will differ slightly
    KDateTime localUtc = local.toUtc();
    int diff = localUtc.secsTo(utc);
    if (diff > 1  ||  diff < 0)
        CHECKDT(local.toUtc().dateTime(), utc.dateTime());
    diff = local.dateTime().secsTo(qcurrent);
    if (diff > 1  ||  diff < 0)
        CHECKDT(local.dateTime(), qcurrent);

    // Restore the original local time zone
    if (!originalZone)
        ::unsetenv("TZ");
    else
        ::setenv("TZ", originalZone, 1);
    ::tzset();
}
