/* This file is part of the KDE libraries
   Copyright (c) 2005 David Jarvie <software@astrojar.org.uk>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#include <stdio.h>
#include <qdatetime.h>
#include <kunittest/tester.h>
#include <kunittest/module.h>
#include <kdebug.h>
#include "ksystemtimezone.h"
#include "ktzfiletimezone.h"


class KTimeZonesTest : public KUnitTest::Tester
{
public:
    void allTests();
private:
    void ktimezones();
    void refcount();
    void utc();
    void local();
    void zone();
    void zoneinfoDir();
    void currentOffset();
    void offsetAtUtc();
    void offsetAtZoneTime();
    void abbreviation();
    void timet();
    void toUtc();
    void toZoneTime();
    void convert();
    void tzfile();
    void tzfileToZoneTime();
    void tzfileOffsetAtZoneTime();
    void tzfileUtcOffsets();
    void tzfileAbbreviation();
};

KUNITTEST_MODULE( kunittest_ktimezones, "KTimeZonesTest" )
KUNITTEST_MODULE_REGISTER_TESTER( KTimeZonesTest )

#define CHECKDT(dt1, dt2) if (dt1 != dt2) CHECK(dt1.toString(Qt::ISODate), dt2.toString(Qt::ISODate))

void KTimeZonesTest::allTests()
{
    ktimezones();
    refcount();
    utc();
    local();
    zone();
    zoneinfoDir();
    currentOffset();
    offsetAtUtc();
    offsetAtZoneTime();
    abbreviation();
    timet();
    toUtc();
    toZoneTime();
    convert();
    tzfile();
    tzfileToZoneTime();
    tzfileOffsetAtZoneTime();
    tzfileUtcOffsets();
    tzfileAbbreviation();
}


///////////////////
// KTimeZones tests
///////////////////

void KTimeZonesTest::ktimezones()
{
    KTimeZones timezones;
    KTimeZone zone1("Zone1");
    CHECK(zone1.isValid(), true);
    KTimeZone zone2("Zone2");
    CHECK(zone2.isValid(), true);
    CHECK(timezones.add(zone1), true);
    CHECK(timezones.add(zone1), false);
    CHECK(timezones.add(zone2), true);
    CHECK((int)timezones.zones().count(), 2);
    KTimeZone tz = timezones.zone("Zone1");
    CHECK((tz == zone1), true);
    tz = timezones.zone("Zone99");
    CHECK(tz.isValid(), false);
    zone1 = timezones.remove(zone1);
    CHECK(zone1.isValid(), true);
    CHECK((int)timezones.zones().count(), 1);
    CHECK(timezones.remove(zone1).isValid(), false);
    CHECK(timezones.add(zone1), true);
    CHECK(timezones.remove("Zone1").isValid(), true);
    CHECK(timezones.remove("Zone1").isValid(), false);
    CHECK(timezones.remove("Zone2").isValid(), true);
    zone1 = KTimeZone("Zone10");
    CHECK(timezones.add(zone1), true);
    CHECK((int)timezones.zones().count(), 1);
    timezones.clear();
    CHECK((int)timezones.zones().count(), 0);
}

//////////////////////////
// KTimeZone: ref counting
//////////////////////////

void KTimeZonesTest::refcount()
{
    KTimeZone *zone1 = new KTimeZone("Zone1");
    CHECK(zone1->name(), QString("Zone1"));
    KTimeZones timezones;
    timezones.add(*zone1);
    delete zone1;
    zone1 = 0;
    KTimeZone tz = timezones.zone("Zone1");
    CHECK(tz.isValid(), true);
    CHECK(tz.name(), QString("Zone1"));
    CHECK(tz.type(), QCString("KTimeZone"));
}

///////////////////
// KTimeZone: UTC
///////////////////

void KTimeZonesTest::utc()
{
    KTimeZone utc = KTimeZone::utc();
    CHECK(utc.isValid(), true);
    CHECK(utc.name(), QString("UTC"));
//    CHECK(utc.offsetAtUtc(QDateTime(QDate(2005,1,1), QTime(), Qt::LocalTime)), 0);
    CHECK(utc.offsetAtUtc(QDateTime(QDate(2005,1,1), QTime())), 0);
    CHECK(utc.offsetAtUtc(QDateTime(QDate(2005,7,1), QTime())), 0);
}

/////////////////////////
// KSystemTimeZones tests
/////////////////////////

void KTimeZonesTest::local()
{
KTimeZone loc = KSystemTimeZones::local();
kdDebug()<<"Local time zone="<<loc.name()<<endl;
    const char *originalZone = ::getenv("TZ");   // save the original local time zone
    ::setenv("TZ", ":Europe/Paris", 1);
    ::tzset();

    KTimeZone local = KSystemTimeZones::local();
    CHECK(local.isValid(), true);
    CHECK(local.name(), QString::fromLatin1("Europe/Paris"));
    CHECK(local.type(), QCString("KSystemTimeZone"));

    // Restore the original local time zone
    if (!originalZone)
        ::unsetenv("TZ");
    else
        ::setenv("TZ", originalZone, 1);
    ::tzset();
}

void KTimeZonesTest::zone()
{
    KTimeZone utc = KSystemTimeZones::zone("UTC");
    CHECK(utc.isValid(), true);
    KTimeZone losAngeles = KSystemTimeZones::zone("America/Los_Angeles");
    CHECK(losAngeles.isValid(), true);
    KTimeZone london = KSystemTimeZones::zone("Europe/London");
    CHECK(london.isValid(), true);
    CHECK(london.countryCode(), QString::fromLatin1("GB"));
    CHECK(london.latitude(), float(51*3600 + 30*60 + 30)/3600.0f);
    CHECK(london.longitude(), -float(0*3600 + 7*60 + 31)/3600.0f);
    CHECK(losAngeles.longitude(), -float(118*3600 + 14*60 + 34)/3600.0f);
}

void KTimeZonesTest::zoneinfoDir()
{
    QString zoneinfo = KSystemTimeZones::zoneinfoDir();
    CHECK(zoneinfo.isEmpty(), false);
    QString msg = QString("Please manually verify that the zoneinfo directory is \"%1\"").arg(zoneinfo);
    kdDebug() << msg << endl;
}


    ////////////////////////
    // KSystemTimeZone tests
    ////////////////////////

void KTimeZonesTest::currentOffset()
{
    // Find the current offset of a time zone
    time_t now = time(0);
    tm *tnow = localtime(&now);
#ifndef _WIN32
    int offset = tnow->tm_gmtoff;
#else
    int offset = 0;
#endif
    KTimeZone local = KSystemTimeZones::local();
    CHECK(local.isValid(), true);
    CHECK(local.currentOffset(Qt::UTC), offset);
}

void KTimeZonesTest::offsetAtUtc()
{
    // Find some offsets for Europe/London.
    KTimeZone losAngeles = KSystemTimeZones::zone("America/Los_Angeles");
    CHECK(losAngeles.isValid(), true);
    KTimeZone london = KSystemTimeZones::zone("Europe/London");
    CHECK(london.isValid(), true);
    QDateTime winter(QDate(2005,1,1), QTime(0,0,0)); //, Qt::UTC);
    QDateTime summer(QDate(2005,6,1), QTime(0,0,0)); //, Qt::UTC);
    CHECK(london.offsetAtUtc(winter), 0);
    CHECK(london.offsetAtUtc(summer), 3600);;
    CHECK(losAngeles.offsetAtUtc(winter), -28800);
    CHECK(losAngeles.offsetAtUtc(summer), -25200);;
}

void KTimeZonesTest::offsetAtZoneTime()
{
    QDateTime aGmt(QDate(2005,3,27), QTime(0,30,0)); //, Qt::LocalTime);
    QDateTime aInvalid(QDate(2005,3,27), QTime(1,30,0)); //, Qt::LocalTime);
    QDateTime aBst(QDate(2005,3,27), QTime(2,30,0)); //, Qt::LocalTime);
    QDateTime bBst(QDate(2005,10,30), QTime(0,30,0)); //, Qt::LocalTime);
    QDateTime bBstBeforeGmt(QDate(2005,10,30), QTime(1,30,0)); //, Qt::LocalTime);
    QDateTime bGmt(QDate(2005,10,30), QTime(2,30,0)); //, Qt::LocalTime);
    KTimeZone london = KSystemTimeZones::zone("Europe/London");
    CHECK(london.isValid(), true);
    int offset2;
    CHECK(london.offsetAtZoneTime(aGmt, &offset2), 0);
    CHECK(offset2, 0);
    CHECK(london.offsetAtZoneTime(aInvalid, &offset2), 3600);
    CHECK(offset2, 3600);
    CHECK(london.offsetAtZoneTime(aBst, &offset2), 3600);
    CHECK(offset2, 3600);
    CHECK(london.offsetAtZoneTime(bBst, &offset2), 3600);
    CHECK(offset2, 3600);
    CHECK(london.offsetAtZoneTime(bBstBeforeGmt, &offset2), 3600);
    CHECK(offset2, 0);
    CHECK(london.offsetAtZoneTime(bGmt, &offset2), 0);
    CHECK(offset2, 0);
}

void KTimeZonesTest::abbreviation()
{
    // Fetch time zone abbreviations
    KTimeZone losAngeles = KSystemTimeZones::zone("America/Los_Angeles");
    CHECK( losAngeles.isValid(), true);
    KTimeZone london = KSystemTimeZones::zone("Europe/London");
    CHECK( london.isValid(), true);
    QDateTime winter(QDate(2005,1,1), QTime(0,0,0)); //, Qt::UTC);
    QDateTime summer(QDate(2005,6,1), QTime(0,0,0)); //, Qt::UTC);
    QString sResult = london.abbreviation(winter);
    CHECK(london.abbreviation(winter), QCString("GMT"));
    CHECK(london.abbreviation(summer), QCString("BST"));
    CHECK(losAngeles.abbreviation(winter), QCString("PST"));
    CHECK(losAngeles.abbreviation(summer), QCString("PDT"));
}

void KTimeZonesTest::timet()
{
    QDateTime t1(QDate(1970,1,2), QTime(1,30,5)); //, Qt::UTC);
    QDateTime t2(QDate(1969,12,30), QTime(22,29,55)); //, Qt::UTC);
    time_t t1t = KTimeZone::toTime_t(t1);
    time_t t2t = KTimeZone::toTime_t(t2);
    CHECK((int)t1t, 86400 + 3600 + 30*60 + 5);
    CHECK((int)t2t, -(86400 + 3600 + 30*60 + 5));
    CHECKDT(KTimeZone::fromTime_t(t1t), t1);
    CHECKDT(KTimeZone::fromTime_t(t2t), t2);
}

void KTimeZonesTest::toUtc()
{
    // Convert to UTC.
    KTimeZone losAngeles = KSystemTimeZones::zone("America/Los_Angeles");
    CHECK( losAngeles.isValid(), true);
    KTimeZone london = KSystemTimeZones::zone("Europe/London");
    CHECK( london.isValid(), true);
    QDateTime winter(QDate(2005,1,1), QTime(0,0,0)); //, Qt::UTC);
    QDateTime summer(QDate(2005,6,1), QTime(0,0,0)); //, Qt::UTC);
    QDateTime winterLocal = winter;
//    winterLocal.setTimeSpec(Qt::LocalTime);
    QDateTime summerLocal = summer;
//    summerLocal.setTimeSpec(Qt::LocalTime);
    CHECKDT(london.toUtc(winterLocal), winter);
    CHECKDT(london.toUtc(summerLocal), summer.addSecs(-3600));;
    CHECKDT(losAngeles.toUtc(winterLocal), winter.addSecs(8*3600));
    CHECKDT(losAngeles.toUtc(summerLocal), summer.addSecs(7*3600));
}

void KTimeZonesTest::toZoneTime()
{
    // Convert from UTC.
    KTimeZone losAngeles = KSystemTimeZones::zone("America/Los_Angeles");
    CHECK( losAngeles.isValid(), true);
    KTimeZone london = KSystemTimeZones::zone("Europe/London");
    CHECK( london.isValid(), true);
    QDateTime winter(QDate(2005,1,1), QTime(0,0,0)); //, Qt::UTC);
    QDateTime summer(QDate(2005,6,1), QTime(0,0,0)); //, Qt::UTC);
    QDateTime winterLocal = winter;
//    winterLocal.setTimeSpec(Qt::LocalTime);
    QDateTime summerLocal = summer;
//    summerLocal.setTimeSpec(Qt::LocalTime);
    CHECKDT(london.toZoneTime(winter), winterLocal);
    CHECKDT(london.toZoneTime(summer), summerLocal.addSecs(3600));
    CHECKDT(losAngeles.toZoneTime(winter), winterLocal.addSecs(-8*3600));
    CHECKDT(losAngeles.toZoneTime(summer), summerLocal.addSecs(-7*3600));

    QDateTime prepre(QDate(2005,10,29), QTime(23,59,59)); //, Qt::UTC);  // before time shift (local time not repeated)
    QDateTime pre(QDate(2005,10,30), QTime(0,0,0)); //, Qt::UTC);  // before time shift (local time repeated afterwards)
    QDateTime before(QDate(2005,10,30), QTime(0,59,59)); //, Qt::UTC);  // before time shift (local time repeated afterwards)
    QDateTime at(QDate(2005,10,30), QTime(1,0,0)); //, Qt::UTC);   // at time shift (second occurrence of local time)
    QDateTime last(QDate(2005,10,30), QTime(1,59,59)); //, Qt::UTC);  // after time shift (second occurrence of local time)
    QDateTime after(QDate(2005,10,30), QTime(2,0,0)); //, Qt::UTC);   // after time shift (local time not repeated)
    bool second;
    CHECKDT(london.toZoneTime(prepre, &second), QDateTime(QDate(2005,10,30), QTime(0,59,59))); //, Qt::LocalTime));
    CHECK(second, false);
    CHECKDT(london.toZoneTime(pre, &second), QDateTime(QDate(2005,10,30), QTime(1,0,0))); //, Qt::LocalTime));
    CHECK(second, false);
    CHECKDT(london.toZoneTime(before, &second), QDateTime(QDate(2005,10,30), QTime(1,59,59))); //, Qt::LocalTime));
    CHECK(second, false);
    CHECKDT(london.toZoneTime(at, &second), QDateTime(QDate(2005,10,30), QTime(1,0,0))); //, Qt::LocalTime));
    CHECK(second, true);
    CHECKDT(london.toZoneTime(last, &second), QDateTime(QDate(2005,10,30), QTime(1,59,59))); //, Qt::LocalTime));
    CHECK(second, true);
    CHECKDT(london.toZoneTime(after, &second), QDateTime(QDate(2005,10,30), QTime(2,0,0))); //, Qt::LocalTime));
    CHECK(second, false);
}

void KTimeZonesTest::convert()
{
    // Try time zone conversions.
    KTimeZone losAngeles = KSystemTimeZones::zone("America/Los_Angeles");
    CHECK( losAngeles.isValid(), true);
    KTimeZone london = KSystemTimeZones::zone("Europe/London");
    CHECK( london.isValid(), true);
    QDateTime bstBeforePdt(QDate(2005,3,28), QTime(0,0,0)); //, Qt::LocalTime);
    QDateTime bstAfterPdt(QDate(2005,5,1), QTime(0,0,0)); //, Qt::LocalTime);
    QDateTime gmtBeforePst(QDate(2005,10,30), QTime(4,0,0)); //, Qt::LocalTime);
    QDateTime gmtAfterPst(QDate(2005,12,1), QTime(0,0,0)); //, Qt::LocalTime);
    QDateTime bstBeforePdtResult(QDate(2005,3,27), QTime(15,0,0)); //, Qt::LocalTime);
    QDateTime bstAfterPdtResult(QDate(2005,4,30), QTime(16,0,0)); //, Qt::LocalTime);
    QDateTime gmtBeforePstResult(QDate(2005,10,29), QTime(21,0,0)); //, Qt::LocalTime);
    QDateTime gmtAfterPstResult(QDate(2005,11,30), QTime(16,0,0)); //, Qt::LocalTime);

    CHECKDT(london.convert(losAngeles, bstBeforePdt), bstBeforePdtResult);
    CHECKDT(london.convert(losAngeles, bstAfterPdt), bstAfterPdtResult);
    CHECKDT(london.convert(losAngeles, gmtBeforePst), gmtBeforePstResult);
    CHECKDT(london.convert(losAngeles, gmtAfterPst), gmtAfterPstResult);
    CHECKDT(losAngeles.convert(losAngeles, bstBeforePdtResult), bstBeforePdtResult);
}

////////////////////////
// KTzfileTimeZone tests
// Plus KSystemTimeZones::readZone() tests
////////////////////////

void KTimeZonesTest::tzfile()
{
    QDateTime winter(QDate(2005,1,1), QTime(0,0,0)); //, Qt::UTC);
    QString zoneinfo = KSystemTimeZones::zoneinfoDir();
    CHECK(zoneinfo.isEmpty(), false);
    KTzfileTimeZoneSource tzsource(zoneinfo);
    KTimeZone *tzcairo = new KTzfileTimeZone(&tzsource, "Africa/Cairo");
    delete tzcairo;
    tzcairo = new KTzfileTimeZone(&tzsource, "Africa/Cairo");
    CHECK(tzcairo->offsetAtUtc(winter), 7200);
    delete tzcairo;
}

void KTimeZonesTest::tzfileToZoneTime()
{
    // Convert from UTC.
    KTzfileTimeZoneSource tzsource(KSystemTimeZones::zoneinfoDir());
    KTimeZone london = KTzfileTimeZone(&tzsource, "Europe/London");
    CHECK(london.isValid(), true);
    CHECK(london.type(), QCString("KTzfileTimeZone"));
    QDateTime prepre(QDate(2005,10,29), QTime(23,59,59)); //, Qt::UTC);  // before time shift (local time not repeated)
    QDateTime pre(QDate(2005,10,30), QTime(0,0,0)); //, Qt::UTC);  // before time shift (local time repeated afterwards)
    QDateTime before(QDate(2005,10,30), QTime(0,59,59)); //, Qt::UTC);  // before time shift (local time repeated afterwards)
    QDateTime at(QDate(2005,10,30), QTime(1,0,0)); //, Qt::UTC);    // at time shift (second occurrence of local time)
    QDateTime last(QDate(2005,10,30), QTime(1,59,59)); //, Qt::UTC);  // after time shift (second occurrence of local time)
    QDateTime after(QDate(2005,10,30), QTime(2,0,0)); //, Qt::UTC);    // after time shift (local time not repeated)
    bool second;
    CHECKDT(london.toZoneTime(prepre, &second), QDateTime(QDate(2005,10,30), QTime(0,59,59))); //, Qt::LocalTime));
    CHECK(second, false);
    CHECKDT(london.toZoneTime(pre, &second), QDateTime(QDate(2005,10,30), QTime(1,0,0))); //, Qt::LocalTime));
    CHECK(second, false);
    CHECKDT(london.toZoneTime(before, &second), QDateTime(QDate(2005,10,30), QTime(1,59,59))); //, Qt::LocalTime));
    CHECK(second, false);
    CHECKDT(london.toZoneTime(at, &second), QDateTime(QDate(2005,10,30), QTime(1,0,0))); //, Qt::LocalTime));
    CHECK(second, true);
    CHECKDT(london.toZoneTime(last, &second), QDateTime(QDate(2005,10,30), QTime(1,59,59))); //, Qt::LocalTime));
    CHECK(second, true);
    CHECKDT(london.toZoneTime(after, &second), QDateTime(QDate(2005,10,30), QTime(2,0,0))); //, Qt::LocalTime));
    CHECK(second, false);

    KTimeZone sysLondon = KSystemTimeZones::readZone("Europe/London");
    CHECK( sysLondon.isValid(), true);
    CHECKDT(sysLondon.toZoneTime(prepre, &second), QDateTime(QDate(2005,10,30), QTime(0,59,59))); //, Qt::LocalTime));
    CHECK(second, false);
    CHECKDT(sysLondon.toZoneTime(pre, &second), QDateTime(QDate(2005,10,30), QTime(1,0,0))); //, Qt::LocalTime));
    CHECK(second, false);
    CHECKDT(sysLondon.toZoneTime(before, &second), QDateTime(QDate(2005,10,30), QTime(1,59,59))); //, Qt::LocalTime));
    CHECK(second, false);
    CHECKDT(sysLondon.toZoneTime(at, &second), QDateTime(QDate(2005,10,30), QTime(1,0,0))); //, Qt::LocalTime));
    CHECK(second, true);
    CHECKDT(sysLondon.toZoneTime(last, &second), QDateTime(QDate(2005,10,30), QTime(1,59,59))); //, Qt::LocalTime));
    CHECK(second, true);
    CHECKDT(sysLondon.toZoneTime(after, &second), QDateTime(QDate(2005,10,30), QTime(2,0,0))); //, Qt::LocalTime));
    CHECK(second, false);
}

void KTimeZonesTest::tzfileOffsetAtZoneTime()
{
    QDateTime aGmt(QDate(2005,3,27), QTime(0,30,0)); //, Qt::LocalTime);
    QDateTime aInvalid(QDate(2005,3,27), QTime(1,30,0)); //, Qt::LocalTime);
    QDateTime aBst(QDate(2005,3,27), QTime(2,30,0)); //, Qt::LocalTime);
    QDateTime bBst(QDate(2005,10,30), QTime(0,30,0)); //, Qt::LocalTime);
    QDateTime bBstBeforeGmt(QDate(2005,10,30), QTime(1,30,0)); //, Qt::LocalTime);
    QDateTime bGmt(QDate(2005,10,30), QTime(2,30,0)); //, Qt::LocalTime);
    KTzfileTimeZoneSource tzsource(KSystemTimeZones::zoneinfoDir());
    KTimeZone london = KTzfileTimeZone(&tzsource, "Europe/London");
    CHECK( london.isValid(), true);
    int offset2;
    CHECK(london.offsetAtZoneTime(aGmt, &offset2), 0);
    CHECK(offset2, 0);
    CHECK(london.offsetAtZoneTime(aInvalid, &offset2), KTimeZone::InvalidOffset);
    CHECK(offset2, KTimeZone::InvalidOffset);
    CHECK(london.offsetAtZoneTime(aBst, &offset2), 3600);
    CHECK(offset2, 3600);
    CHECK(london.offsetAtZoneTime(bBst, &offset2), 3600);
    CHECK(offset2, 3600);
    CHECK(london.offsetAtZoneTime(bBstBeforeGmt, &offset2), 3600);
    CHECK(offset2, 0);
    CHECK(london.offsetAtZoneTime(bGmt, &offset2), 0);
    CHECK(offset2, 0);

    KTimeZone sysLondon = KSystemTimeZones::readZone("Europe/London");
    CHECK( sysLondon.isValid(), true);
    CHECK(sysLondon.offsetAtZoneTime(aGmt, &offset2), 0);
    CHECK(offset2, 0);
    CHECK(sysLondon.offsetAtZoneTime(aInvalid, &offset2), KTimeZone::InvalidOffset);
    CHECK(offset2, KTimeZone::InvalidOffset);
    CHECK(sysLondon.offsetAtZoneTime(aBst, &offset2), 3600);
    CHECK(offset2, 3600);
    CHECK(sysLondon.offsetAtZoneTime(bBst, &offset2), 3600);
    CHECK(offset2, 3600);
    CHECK(sysLondon.offsetAtZoneTime(bBstBeforeGmt, &offset2), 3600);
    CHECK(offset2, 0);
    CHECK(sysLondon.offsetAtZoneTime(bGmt, &offset2), 0);
    CHECK(offset2, 0);
}

void KTimeZonesTest::tzfileUtcOffsets()
{
    KTzfileTimeZoneSource tzsource(KSystemTimeZones::zoneinfoDir());
    KTimeZone london = KTzfileTimeZone(&tzsource, "Europe/London");
    CHECK( london.isValid(), true);
    QValueList<int> offsets = london.utcOffsets();
    CHECK((int)offsets.count(), 3);
    CHECK(offsets[0], 0);    // GMT
    CHECK(offsets[1], 3600); // BST
    CHECK(offsets[2], 7200); // DST

    KTimeZone sysLondon = KSystemTimeZones::readZone("Europe/London");
    CHECK( sysLondon.isValid(), true);
    offsets = sysLondon.utcOffsets();
    CHECK((int)offsets.count(), 3);
    CHECK(offsets[0], 0);    // GMT
    CHECK(offsets[1], 3600); // BST
    CHECK(offsets[2], 7200); // DST
}

void KTimeZonesTest::tzfileAbbreviation()
{
    KTzfileTimeZoneSource tzsource(KSystemTimeZones::zoneinfoDir());
    KTimeZone london = KTzfileTimeZone(&tzsource, "Europe/London");
    CHECK( london.isValid(), true);
    QDateTime winter(QDate(2005,1,1), QTime(0,0,0)); //, Qt::UTC);
    QDateTime summer(QDate(2005,6,1), QTime(0,0,0)); //, Qt::UTC);
    QDateTime standard(QDate(1970,4,30), QTime(12,45,16,25)); //, Qt::UTC);
    QString sResult = london.abbreviation(winter);
    CHECK(london.abbreviation(winter), QCString("GMT"));
    CHECK(london.abbreviation(summer), QCString("BST"));
    CHECK(london.abbreviation(standard), QCString("BST"));

    KTimeZone sysLondon = KSystemTimeZones::readZone("Europe/London");
    CHECK( sysLondon.isValid(), true);
    sResult = sysLondon.abbreviation(winter);
    CHECK(sysLondon.abbreviation(winter), QCString("GMT"));
    CHECK(sysLondon.abbreviation(summer), QCString("BST"));
    CHECK(sysLondon.abbreviation(standard), QCString("BST"));
}
