/*
 *  alarmtimewidget.cpp  -  alarm date/time entry widget
 *  Program:  kalarm
 *  Copyright © 2001-2008 by David Jarvie <software@astrojar.org.uk>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "kalarm.h"

#include <qlayout.h>
#include <qgroupbox.h>
#include <qhbox.h>
#include <qpushbutton.h>
#include <qlabel.h>
#include <qwhatsthis.h>

#include <kdialog.h>
#include <kmessagebox.h>
#include <klocale.h>

#include "checkbox.h"
#include "dateedit.h"
#include "datetime.h"
#include "preferences.h"
#include "pushbutton.h"
#include "radiobutton.h"
#include "synchtimer.h"
#include "timeedit.h"
#include "timespinbox.h"
#include "timezonecombo.h"
#include "alarmtimewidget.moc"

static const QTime time_23_59(23, 59);


const int AlarmTimeWidget::maxDelayTime = 99*60 + 59;    // < 100 hours

QString AlarmTimeWidget::i18n_w_TimeFromNow()     { return i18n("Time from no&w:"); }
QString AlarmTimeWidget::i18n_TimeAfterPeriod()
{
	return i18n("Enter the length of time (in hours and minutes) after "
	            "the current time to schedule the alarm.");
}


/******************************************************************************
* Construct a widget with a group box and title.
*/
AlarmTimeWidget::AlarmTimeWidget(const QString& groupBoxTitle, int mode, QWidget* parent, const char* name)
	: ButtonGroup(groupBoxTitle, parent, name),
	  mMinDateTimeIsNow(false),
	  mPastMax(false),
	  mMinMaxTimeSet(false)
{
	init(mode);
}

/******************************************************************************
* Construct a widget without a group box or title.
*/
AlarmTimeWidget::AlarmTimeWidget(int mode, QWidget* parent, const char* name)
	: ButtonGroup(parent, name),
	  mMinDateTimeIsNow(false),
	  mPastMax(false),
	  mMinMaxTimeSet(false)
{
	setFrameStyle(QFrame::NoFrame);
	init(mode);
}

void AlarmTimeWidget::init(int mode)
{
	static const QString recurText = i18n("If a recurrence is configured, the start date/time will be adjusted "
	                                      "to the first recurrence on or after the entered date/time."); 
	static const QString tzText = i18n("This uses KAlarm's default time zone, set in the Configuration dialog.");
	// Get alignment to use in QGridLayout (AlignAuto doesn't work correctly there)
	int alignment = QApplication::reverseLayout() ? Qt::AlignRight : Qt::AlignLeft;
	int reverseAlignment = QApplication::reverseLayout() ? Qt::AlignLeft : Qt::AlignRight;

	mDeferring = mode & DEFER_TIME;
	connect(this, SIGNAL(buttonSet(int)), SLOT(slotButtonSet(int)));
	QBoxLayout* topLayout = new QVBoxLayout(this, 0, KDialog::spacingHint());
	if (!title().isEmpty())
	{
		topLayout->setMargin(KDialog::marginHint());
		topLayout->addSpacing(fontMetrics().height() - KDialog::marginHint() + KDialog::spacingHint());
	}
	QGridLayout* grid = new QGridLayout(topLayout, 3, 2, KDialog::spacingHint());

	// At time radio button
	mAtTimeRadio = new RadioButton((mDeferring ? i18n("&Defer to date/time:") : i18n("At &date/time:")), this, "atTimeRadio");
	mAtTimeRadio->setFixedSize(mAtTimeRadio->sizeHint());
	QWhatsThis::add(mAtTimeRadio,
	                (mDeferring ? i18n("Reschedule the alarm to the specified date and time.")
	                            : i18n("Schedule the alarm at the specified date and time.")));
	grid->addWidget(mAtTimeRadio, 0, 0, alignment);

	// Date edit box
	mDateEdit = new DateEdit(this);
	mDateEdit->setFixedSize(mDateEdit->sizeHint());
	connect(mDateEdit, SIGNAL(dateEntered(const QDate&)), SLOT(dateTimeChanged()));
	static const QString enterDateText = i18n("Enter the date to schedule the alarm.");
	QWhatsThis::add(mDateEdit, QString("%1\n%2").arg(enterDateText).arg(mDeferring ? tzText : recurText));
	mAtTimeRadio->setFocusWidget(mDateEdit);
	grid->addWidget(mDateEdit, 0, 1, alignment);

	// Time edit box and Any time checkbox
	mTimeEdit = new TimeEdit(this);
	mTimeEdit->setFixedSize(mTimeEdit->sizeHint());
	connect(mTimeEdit, SIGNAL(valueChanged(int)), SLOT(dateTimeChanged()));
	static const QString enterTimeText = i18n("Enter the time to schedule the alarm.");
	QWhatsThis::add(mTimeEdit,
	      QString("%1\n%2\n\n%3").arg(enterTimeText).arg(mDeferring ? tzText : recurText).arg(TimeSpinBox::shiftWhatsThis()));
	grid->addWidget(mTimeEdit, 0, 2, alignment);

	mAnyTime = -1;    // current status is uninitialised
	if (mDeferring)
	{
		grid->setColStretch(2, 1);
		mTimeZone   = 0;
		mNoTimeZone = 0;
		mAnyTimeCheckBox = 0;
		mAnyTimeAllowed = false;
	}
	else
	{
		mAnyTimeCheckBox = new CheckBox(i18n("An&y time"), this);
		mAnyTimeCheckBox->setFixedSize(mAnyTimeCheckBox->sizeHint());
		connect(mAnyTimeCheckBox, SIGNAL(toggled(bool)), SLOT(slotAnyTimeToggled(bool)));
		QWhatsThis::add(mAnyTimeCheckBox, i18n("Schedule the alarm for any time during the day"));
		QWhatsThis::add(mAnyTimeCheckBox,
		      i18n("Check to specify only a date (without a time) for the alarm. The alarm will trigger at the first opportunity on the selected date."));
		grid->addWidget(mAnyTimeCheckBox, 0, 3, alignment);
		grid->setColStretch(3, 1);
		mAnyTimeAllowed = true;
	}

	// 'Time from now' radio button/label
	mAfterTimeRadio = new RadioButton((mDeferring ? i18n("Defer for time &interval:") : i18n_w_TimeFromNow()),
	                                  this, "afterTimeRadio");
	mAfterTimeRadio->setFixedSize(mAfterTimeRadio->sizeHint());
	QWhatsThis::add(mAfterTimeRadio,
	                (mDeferring ? i18n("Reschedule the alarm for the specified time interval after now.")
	                            : i18n("Schedule the alarm after the specified time interval from now.")));
	grid->addWidget(mAfterTimeRadio, 1, 0, alignment);

	// Delay time spin box
	mDelayTimeEdit = new TimeSpinBox(1, maxDelayTime, this);
	mDelayTimeEdit->setValue(1439);
	mDelayTimeEdit->setFixedSize(mDelayTimeEdit->sizeHint());
	connect(mDelayTimeEdit, SIGNAL(valueChanged(int)), SLOT(delayTimeChanged(int)));
	QWhatsThis::add(mDelayTimeEdit,
	                (mDeferring ? QString("%1\n\n%2").arg(i18n_TimeAfterPeriod()).arg(TimeSpinBox::shiftWhatsThis())
	                            : QString("%1\n%2\n\n%3").arg(i18n_TimeAfterPeriod()).arg(recurText).arg(TimeSpinBox::shiftWhatsThis())));
	grid->addWidget(mDelayTimeEdit, 1, 1, alignment);
	mAfterTimeRadio->setFocusWidget(mDelayTimeEdit);

	if (!mDeferring)
	{
		// Time zone selection push button
		mTimeZoneButton = new PushButton(i18n("Time Zone..."), this);
		connect(mTimeZoneButton, SIGNAL(clicked()), SLOT(showTimeZoneSelector()));
		QWhatsThis::add(mTimeZoneButton,
		      i18n("Choose a time zone for this alarm which is different from the default time zone set in KAlarm's configuration dialog."));
		grid->addMultiCellWidget(mTimeZoneButton, 1, 1, 2, 3, reverseAlignment);
		grid->setColStretch(3, 1);

		// Time zone selector
		QHBoxLayout* hlayout = new QHBoxLayout(topLayout);
		mTimeZoneBox = new QHBox(this);
		mTimeZoneBox->setSpacing(KDialog::spacingHint());
		hlayout->addWidget(mTimeZoneBox);
		QLabel* label = new QLabel(i18n("Time zone:"), mTimeZoneBox);
		label->setFixedSize(label->sizeHint());
		mTimeZone = new TimeZoneCombo(mTimeZoneBox);
		mTimeZone->setSizeLimit(15);
		connect(mTimeZone, SIGNAL(activated(int)), SLOT(slotTimeZoneChanged()));
		label->setBuddy(mTimeZone);
		QWhatsThis::add(mTimeZoneBox, i18n("Select the time zone to use for this alarm."));

		// Time zone checkbox
		mNoTimeZone = new CheckBox(i18n("Ignore time zone"), this);
		connect(mNoTimeZone, SIGNAL(toggled(bool)), SLOT(slotTimeZoneToggled(bool)));
		QWhatsThis::add(mNoTimeZone,
		      i18n("Check to use the local computer time, ignoring time zones.\n\n"
		           "You are recommended not to use this option if the alarm has a "
		           "recurrence specified in hours/minutes. If you do, the alarm may "
		           "occur at unexpected times after daylight saving time shifts."));
		hlayout->addWidget(mNoTimeZone, 1, alignment);

		// Initially show only the time zone button, not time zone selector
		mTimeZoneBox->hide();
		mNoTimeZone->hide();
	}

	// Initialise the radio button statuses
	setButton(id(mAtTimeRadio));

	// Timeout every minute to update alarm time fields.
	MinuteTimer::connect(this, SLOT(updateTimes()));
}

/******************************************************************************
* Set or clear read-only status for the controls
*/
void AlarmTimeWidget::setReadOnly(bool ro)
{
	mAtTimeRadio->setReadOnly(ro);
	mDateEdit->setReadOnly(ro);
	mTimeEdit->setReadOnly(ro);
	if (mAnyTimeCheckBox)
		mAnyTimeCheckBox->setReadOnly(ro);
	if (!mDeferring)
	{
		mTimeZoneButton->setReadOnly(ro);
		mTimeZone->setReadOnly(ro);
		mNoTimeZone->setReadOnly(ro);
	}
	mAfterTimeRadio->setReadOnly(ro);
	mDelayTimeEdit->setReadOnly(ro);
}

/******************************************************************************
* Select the "Time from now" radio button.
*/
void AlarmTimeWidget::selectTimeFromNow(int minutes)
{
	mAfterTimeRadio->setChecked(true);
	slotButtonSet(1);
	if (minutes > 0)
		mDelayTimeEdit->setValue(minutes);
}

/******************************************************************************
* Fetch the entered date/time.
* If 'checkExpired' is true and the entered value <= current time, an error occurs.
* If 'minsFromNow' is non-null, it is set to the number of minutes' delay selected,
* or to zero if a date/time was entered.
* In this case, if 'showErrorMessage' is true, output an error message.
* 'errorWidget' if non-null, is set to point to the widget containing the error.
* Reply = invalid date/time if error.
*/
KDateTime AlarmTimeWidget::getDateTime(int* minsFromNow, bool checkExpired, bool showErrorMessage, QWidget** errorWidget) const
{
	if (minsFromNow)
		*minsFromNow = 0;
	if (errorWidget)
		*errorWidget = 0;
	KDateTime now = KDateTime::currentUtcDateTime();
	now.setTime(QTime(now.time().hour(), now.time().minute(), 0));
	if (mAfterTimeRadio->isOn())
	{
		if (!mDelayTimeEdit->isValid())
		{
			if (showErrorMessage)
				KMessageBox::sorry(const_cast<AlarmTimeWidget*>(this), i18n("Invalid time"));
			if (errorWidget)
				*errorWidget = mDelayTimeEdit;
			return KDateTime();
		}
		int delayMins = mDelayTimeEdit->value();
		if (minsFromNow)
			*minsFromNow = delayMins;
		return now.addSecs(delayMins * 60).toTimeSpec(mTimeSpec);
	}
	else
	{
		bool dateOnly = mAnyTimeAllowed && mAnyTimeCheckBox && mAnyTimeCheckBox->isChecked();
		if (!mDateEdit->isValid()  ||  !mTimeEdit->isValid())
		{
			// The date and/or time is invalid
			if (!mDateEdit->isValid())
			{
				if (showErrorMessage)
					KMessageBox::sorry(const_cast<AlarmTimeWidget*>(this), i18n("Invalid date"));
				if (errorWidget)
					*errorWidget = mDateEdit;
			}
			else
			{
				if (showErrorMessage)
					KMessageBox::sorry(const_cast<AlarmTimeWidget*>(this), i18n("Invalid time"));
				if (errorWidget)
					*errorWidget = mTimeEdit;
			}
			return KDateTime();
		}

		KDateTime result;
		if (dateOnly)
		{
			result = KDateTime(mDateEdit->date(), mTimeSpec);
			if (checkExpired  &&  result.date() < now.date())
			{
				if (showErrorMessage)
					KMessageBox::sorry(const_cast<AlarmTimeWidget*>(this), i18n("Alarm date has already expired"));
				if (errorWidget)
					*errorWidget = mDateEdit;
				return KDateTime();
			}
		}
		else
		{
			result = KDateTime(mDateEdit->date(), mTimeEdit->time(), mTimeSpec);
			if (checkExpired  &&  result <= now.addSecs(1))
			{
				if (showErrorMessage)
					KMessageBox::sorry(const_cast<AlarmTimeWidget*>(this), i18n("Alarm time has already expired"));
				if (errorWidget)
					*errorWidget = mTimeEdit;
				return KDateTime();
			}
		}
		return result;
	}
}

/******************************************************************************
* Set the date/time.
*/
void AlarmTimeWidget::setDateTime(const DateTime& dt)
{
	// Set the time zone/specification first so that the call to
	// dateTimeChanged() works correctly.
	if (mDeferring)
		mTimeSpec = dt.timeSpec().isValid() ? dt.timeSpec() : KDateTime::LocalZone;
	else
	{
		KTimeZone tz = dt.timeZone();
		mNoTimeZone->setChecked(!tz.isValid());
		mTimeZone->setTimeZone(tz.isValid() ? tz : Preferences::timeZone());
		slotTimeZoneChanged();
	}

	if (dt.date().isValid())
	{
		mTimeEdit->setValue(dt.effectiveTime());
		mDateEdit->setDate(dt.date());
		dateTimeChanged();     // update the delay time edit box
	}
	else
	{
		mTimeEdit->setValid(false);
		mDateEdit->setInvalid();
		mDelayTimeEdit->setValid(false);
	}
	if (mAnyTimeCheckBox)
	{
		bool anyTime = dt.isDateOnly();
		if (anyTime)
			mAnyTimeAllowed = true;
		mAnyTimeCheckBox->setChecked(anyTime);
		setAnyTime();
	}
}

/******************************************************************************
* Set the minimum date/time to track the current time.
*/
void AlarmTimeWidget::setMinDateTimeIsCurrent()
{
	mMinDateTimeIsNow = true;
	mMinDateTime = KDateTime();
	KDateTime now = KDateTime::currentDateTime(mTimeSpec);
	mDateEdit->setMinDate(now.date());
	setMaxMinTimeIf(now);
}

/******************************************************************************
* Set the minimum date/time, adjusting the entered date/time if necessary.
* If 'dt' is invalid, any current minimum date/time is cleared.
*/
void AlarmTimeWidget::setMinDateTime(const KDateTime& dt)
{
	mMinDateTimeIsNow = false;
	mMinDateTime = dt.toTimeSpec(mTimeSpec);
	mDateEdit->setMinDate(mMinDateTime.date());
	setMaxMinTimeIf(KDateTime::currentDateTime(mTimeSpec));
}

/******************************************************************************
* Set the maximum date/time, adjusting the entered date/time if necessary.
* If 'dt' is invalid, any current maximum date/time is cleared.
*/
void AlarmTimeWidget::setMaxDateTime(const DateTime& dt)
{
	mPastMax = false;
	if (dt.isValid()  &&  dt.isDateOnly())
		mMaxDateTime = dt.effectiveKDateTime().addSecs(24*3600 - 60).toTimeSpec(mTimeSpec);
	else
		mMaxDateTime = dt.kDateTime().toTimeSpec(mTimeSpec);
	mDateEdit->setMaxDate(mMaxDateTime.date());
	KDateTime now = KDateTime::currentDateTime(mTimeSpec);
	setMaxMinTimeIf(now);
	setMaxDelayTime(now);
}

/******************************************************************************
* If the minimum and maximum date/times fall on the same date, set the minimum
* and maximum times in the time edit box.
*/
void AlarmTimeWidget::setMaxMinTimeIf(const KDateTime& now)
{
	int   mint = 0;
	QTime maxt = time_23_59;
	mMinMaxTimeSet = false;
	if (mMaxDateTime.isValid())
	{
		bool set = true;
		KDateTime minDT;
		if (mMinDateTimeIsNow)
			minDT = now.addSecs(60);
		else if (mMinDateTime.isValid())
			minDT = mMinDateTime;
		else
			set = false;
		if (set  &&  mMaxDateTime.date() == minDT.date())
		{
			// The minimum and maximum times are on the same date, so
			// constrain the time value.
			mint = minDT.time().hour()*60 + minDT.time().minute();
			maxt = mMaxDateTime.time();
			mMinMaxTimeSet = true;
		}
	}
	mTimeEdit->setMinValue(mint);
	mTimeEdit->setMaxValue(maxt);
	mTimeEdit->setWrapping(!mint  &&  maxt == time_23_59);
}

/******************************************************************************
* Set the maximum value for the delay time edit box, depending on the maximum
* value for the date/time.
*/
void AlarmTimeWidget::setMaxDelayTime(const KDateTime& now)
{
	int maxVal = maxDelayTime;
	if (mMaxDateTime.isValid())
	{
		if (now.date().daysTo(mMaxDateTime.date()) < 100)    // avoid possible 32-bit overflow on secsTo()
		{
			KDateTime dt(now);
			dt.setTime(QTime(now.time().hour(), now.time().minute(), 0));   // round down to nearest minute
			maxVal = dt.secsTo(mMaxDateTime) / 60;
			if (maxVal > maxDelayTime)
				maxVal = maxDelayTime;
		}
	}
	mDelayTimeEdit->setMaxValue(maxVal);
}

/******************************************************************************
* Set the status for whether a time is specified, or just a date.
*/
void AlarmTimeWidget::setAnyTime()
{
	int old = mAnyTime;
	mAnyTime = (mAtTimeRadio->isOn() && mAnyTimeAllowed && mAnyTimeCheckBox && mAnyTimeCheckBox->isChecked()) ? 1 : 0;
	if (mAnyTime != old)
		emit dateOnlyToggled(mAnyTime);
}

/******************************************************************************
* Enable/disable the "date only" radio button.
*/
void AlarmTimeWidget::enableAnyTime(bool enable)
{
	if (mAnyTimeCheckBox)
	{
		mAnyTimeAllowed = enable;
		bool at = mAtTimeRadio->isOn();
		mAnyTimeCheckBox->setEnabled(enable && at);
		if (at)
			mTimeEdit->setEnabled(!enable || !mAnyTimeCheckBox->isChecked());
		setAnyTime();
	}
}

/******************************************************************************
* Called every minute to update the alarm time data entry fields.
* If the maximum date/time has been reached, a 'pastMax()' signal is emitted.
*/
void AlarmTimeWidget::updateTimes()
{
	KDateTime now;
	if (mMinDateTimeIsNow)
	{
		// Make sure that the minimum date is updated when the day changes
		now = KDateTime::currentDateTime(mTimeSpec);
		mDateEdit->setMinDate(now.date());
	}
	if (mMaxDateTime.isValid())
	{
		if (!now.isValid())
			now = KDateTime::currentDateTime(mTimeSpec);
		if (!mPastMax)
		{
			// Check whether the maximum date/time has now been reached
			if (now.date() >= mMaxDateTime.date())
			{
				// The current date has reached or has passed the maximum date
				if (now.date() > mMaxDateTime.date()
				||  (!mAnyTime && now.time() > mTimeEdit->maxTime()))
				{
					mPastMax = true;
					emit pastMax();
				}
				else if (mMinDateTimeIsNow  &&  !mMinMaxTimeSet)
				{
					// The minimum date/time tracks the clock, so set the minimum
					// and maximum times
					setMaxMinTimeIf(now);
				}
			}
		}
		setMaxDelayTime(now);
	}

	if (mAfterTimeRadio->isOn())
		delayTimeChanged(mDelayTimeEdit->value());
	else
		dateTimeChanged();
}


/******************************************************************************
* Called when the radio button states have been changed.
* Updates the appropriate edit box.
*/
void AlarmTimeWidget::slotButtonSet(int)
{
	bool at = mAtTimeRadio->isOn();
	mDateEdit->setEnabled(at);
	mTimeEdit->setEnabled(at && (!mAnyTimeAllowed || !mAnyTimeCheckBox || !mAnyTimeCheckBox->isChecked()));
	if (mAnyTimeCheckBox)
		mAnyTimeCheckBox->setEnabled(at && mAnyTimeAllowed);
	// Ensure that the value of the delay edit box is > 0.
	KDateTime dt(mDateEdit->date(), mTimeEdit->time(), mTimeSpec);
	int minutes = (KDateTime::currentUtcDateTime().secsTo(dt) + 59) / 60;
	if (minutes <= 0)
		mDelayTimeEdit->setValid(true);
	mDelayTimeEdit->setEnabled(mAfterTimeRadio->isOn());
	setAnyTime();
}

/******************************************************************************
* Called after the mAnyTimeCheckBox checkbox has been toggled.
*/
void AlarmTimeWidget::slotAnyTimeToggled(bool on)
{
	on = (on && mAnyTimeAllowed);
	mTimeEdit->setEnabled(!on && mAtTimeRadio->isOn());
	setAnyTime();
	if (on)
		emit changed(KDateTime(mDateEdit->date(), mTimeSpec));
	else
		emit changed(KDateTime(mDateEdit->date(), mTimeEdit->time(), mTimeSpec));
}

/******************************************************************************
* Called after a new selection has been made in the time zone combo box.
* Re-evaluates the time specification to use.
*/
void AlarmTimeWidget::slotTimeZoneChanged()
{
	if (mNoTimeZone->isChecked())
		mTimeSpec = KDateTime::ClockTime;
	else
	{
		KTimeZone tz = mTimeZone->timeZone();
		mTimeSpec = tz.isValid() ? KDateTime::Spec(tz) : KDateTime::LocalZone;
	}
	if (!mTimeZoneBox->isVisible()  &&  mTimeSpec != Preferences::timeZone())
	{
		// The current time zone is not the default one, so
		// show the time zone selection controls
		showTimeZoneSelector();
	}
	mMinDateTime = mMinDateTime.toTimeSpec(mTimeSpec);
	mMaxDateTime = mMaxDateTime.toTimeSpec(mTimeSpec);
	updateTimes();
}

/******************************************************************************
* Called after the mNoTimeZone checkbox has been toggled.
*/
void AlarmTimeWidget::slotTimeZoneToggled(bool on)
{
	mTimeZone->setEnabled(!on && !mAfterTimeRadio->isOn());
	slotTimeZoneChanged();
}

/******************************************************************************
* Called after the mTimeZoneButton button has been clicked.
* Show the time zone selection controls, and hide the button.
*/
void AlarmTimeWidget::showTimeZoneSelector()
{
	mTimeZoneButton->hide();
	mTimeZoneBox->show();
	mNoTimeZone->show();
}

/******************************************************************************
* Show or hide the time zone button.
*/
void AlarmTimeWidget::showMoreOptions(bool more)
{
	if (more)
	{
		if (!mTimeZoneBox->isVisible())
			mTimeZoneButton->show();
	}
	else
		mTimeZoneButton->hide();
}

/******************************************************************************
* Called when the date or time edit box values have changed.
* Updates the time delay edit box accordingly.
*/
void AlarmTimeWidget::dateTimeChanged()
{
	KDateTime dt(mDateEdit->date(), mTimeEdit->time(), mTimeSpec);
	int minutes = (KDateTime::currentUtcDateTime().secsTo(dt) + 59) / 60;
	bool blocked = mDelayTimeEdit->signalsBlocked();
	mDelayTimeEdit->blockSignals(true);     // prevent infinite recursion between here and delayTimeChanged()
	if (minutes <= 0  ||  minutes > mDelayTimeEdit->maxValue())
		mDelayTimeEdit->setValid(false);
	else
		mDelayTimeEdit->setValue(minutes);
	mDelayTimeEdit->blockSignals(blocked);
	if (mAnyTimeAllowed && mAnyTimeCheckBox && mAnyTimeCheckBox->isChecked())
		emit changed(KDateTime(dt.date(), mTimeSpec));
	else
		emit changed(dt);
}

/******************************************************************************
* Called when the delay time edit box value has changed.
* Updates the Date and Time edit boxes accordingly.
*/
void AlarmTimeWidget::delayTimeChanged(int minutes)
{
	if (mDelayTimeEdit->isValid())
	{
		QDateTime dt = KDateTime::currentUtcDateTime().addSecs(minutes * 60).toTimeSpec(mTimeSpec).dateTime();
		bool blockedT = mTimeEdit->signalsBlocked();
		bool blockedD = mDateEdit->signalsBlocked();
		mTimeEdit->blockSignals(true);     // prevent infinite recursion between here and dateTimeChanged()
		mDateEdit->blockSignals(true);
		mTimeEdit->setValue(dt.time());
		mDateEdit->setDate(dt.date());
		mTimeEdit->blockSignals(blockedT);
		mDateEdit->blockSignals(blockedD);
		emit changed(KDateTime(dt.date(), dt.time(), mTimeSpec));
	}
}
