/*
 *  editdlg.cpp  -  dialogue to create or modify an alarm or alarm template
 *  Program:  kalarm
 *  Copyright © 2001-2008 by David Jarvie <software@astrojar.org.uk>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "kalarm.h"

#include <qlayout.h>
#include <qpopupmenu.h>
#include <qvbox.h>
#include <qgroupbox.h>
#include <qpushbutton.h>
#include <qdragobject.h>
#include <qlabel.h>
#include <qmessagebox.h>
#include <qvalidator.h>
#include <qwhatsthis.h>
#include <qtooltip.h>
#include <qdir.h>
#include <qstyle.h>
#include <qtimer.h>

#include <kglobal.h>
#include <klocale.h>
#include <kconfig.h>
#include <kfiledialog.h>
#include <ktabwidget.h>
#include <kmessagebox.h>
#include <kurldrag.h>
#include <kwin.h>
#include <kwinmodule.h>
#include <kdebug.h>

#include <libkdepim/maillistdrag.h>
#include <libkdepim/kvcarddrag.h>

#include "alarmcalendar.h"
#include "alarmresources.h"
#include "alarmtimewidget.h"
#include "checkbox.h"
#include "deferdlg.h"
#include "functions.h"
#include "kalarmapp.h"
#include "latecancel.h"
#include "lineedit.h"
#include "mainwindow.h"
#include "preferences.h"
#include "radiobutton.h"
#include "recurrenceedit.h"
#include "reminder.h"
#include "shellprocess.h"
#include "spinbox.h"
#include "stackedwidgets.h"
#include "templatepickdlg.h"
#include "timeedit.h"
#include "timespinbox.h"
#include "editdlg.moc"
#include "editdlgprivate.h"
#include "editdlgtypes.h"

using namespace KCal;

static const char EDIT_DIALOG_NAME[] = "EditDialog";
static const char EDIT_MORE_GROUP[] = "ShowOpts";
static const char EDIT_MORE_KEY[]   = "EditMore";
static const int  maxDelayTime = 99*60 + 59;    // < 100 hours

inline QString recurText(const KAEvent& event)
{
	QString r;
	if (event.repeatCount())
		r = QString::fromLatin1("%1 / %2").arg(event.recurrenceText()).arg(event.repetitionText());
	else
		r = event.recurrenceText();
	return i18n("Recurrence - [%1]").arg(r);
}

// Collect these widget labels together to ensure consistent wording and
// translations across different modules.
QString EditAlarmDlg::i18n_ShowInKOrganizer()   { return i18n("Show in KOrganizer"); }
QString EditAlarmDlg::i18n_g_ShowInKOrganizer() { return i18n("Show in KOr&ganizer"); }


EditAlarmDlg* EditAlarmDlg::create(bool Template, Type type, bool newAlarm, QWidget* parent, const char* name, GetResourceType getResource)
{
	kdDebug(5950) << "EditAlarmDlg::create()" << endl;
	switch (type)
	{
		case DISPLAY:  return new EditDisplayAlarmDlg(Template, newAlarm, parent, name, getResource);
		case COMMAND:  return new EditCommandAlarmDlg(Template, newAlarm, parent, name, getResource);
		case EMAIL:    return new EditEmailAlarmDlg(Template, newAlarm, parent, name, getResource);
	}
	return 0;
}

EditAlarmDlg* EditAlarmDlg::create(bool Template, const KAEvent* event, bool newAlarm, QWidget* parent,
                                   const char* name, GetResourceType getResource, bool readOnly)
{
	switch (event->action())
	{
		case KAEvent::COMMAND:
			if (!event->commandDisplay())
				return new EditCommandAlarmDlg(Template, event, newAlarm, parent, name, getResource, readOnly);
			// fall through to MESSAGE
		case KAEvent::MESSAGE:
		case KAEvent::FILE:     return new EditDisplayAlarmDlg(Template, event, newAlarm, parent, name, getResource, readOnly);
		case KAEvent::EMAIL:    return new EditEmailAlarmDlg(Template, event, newAlarm, parent, name, getResource, readOnly);
	}
	return 0;
}

/******************************************************************************
* Constructor.
* Parameters:
*   Template = true to edit/create an alarm template
*            = false to edit/create an alarm.
*   event   != to initialise the dialogue to show the specified event's data.
*   getResource = how to obtain the resource to use (if at all)
*/
EditAlarmDlg::EditAlarmDlg(bool Template, KAEvent::Action action, QWidget* parent, const char* name,
                           GetResourceType getResource)
	: KDialogBase(parent, name, true, QString::null, (Template ? Ok|Cancel|Try|Default : Ok|Cancel|Try|Help|Default), Ok),
	  mAlarmType(action),
	  mMainPageShown(false),
	  mRecurPageShown(false),
	  mRecurSetDefaultEndDate(true),
	  mTemplateName(0),
	  mDeferGroup(0),
	  mTimeWidget(0),
	  mShowInKorganizer(0),
	  mResource(0),
	  mDeferGroupHeight(0),
	  mTemplate(Template),
	  mDesiredReadOnly(false),
	  mReadOnly(false),
	  mSavedEvent(0)
{
	init(0, getResource);
}

EditAlarmDlg::EditAlarmDlg(bool Template, const KAEvent* event, QWidget* parent, const char* name,
                           GetResourceType getResource, bool readOnly)
	: KDialogBase(parent, name, true, QString::null,
	              (readOnly ? Cancel|Try|Default : Template ? Ok|Cancel|Try|Default : Ok|Cancel|Try|Help|Default),
	              (readOnly ? Cancel : Ok)),
	  mAlarmType(event->action()),
	  mMainPageShown(false),
	  mRecurPageShown(false),
	  mRecurSetDefaultEndDate(true),
	  mTemplateName(0),
	  mDeferGroup(0),
	  mTimeWidget(0),
	  mShowInKorganizer(0),
	  mResource(0),
	  mDeferGroupHeight(0),
	  mTemplate(Template),
	  mDesiredReadOnly(readOnly),
	  mReadOnly(readOnly),
	  mSavedEvent(0)
{
	init(event, getResource);
}

void EditAlarmDlg::init(const KAEvent* event, GetResourceType getResource)
{
	switch (getResource)
	{
		case RES_USE_EVENT_ID:
			if (event)
			{
				mResourceEventId = event->id();
				break;
			}
			// fall through to RES_PROMPT
		case RES_PROMPT:
			mResourceEventId = QString("");   // empty but non-null
			break;
		case RES_IGNORE:
		default:
			mResourceEventId = QString::null;
			break;
	}
}

void EditAlarmDlg::init(const KAEvent* event, bool newAlarm)
{
	if (!name())
		setName(mTemplate ? "TemplEditDlg" : "EditDlg");    // used by LikeBack
	QString caption;
	if (mReadOnly)
		caption = mTemplate ? i18n("Alarm Template [read-only]")
		        : event->expired() ? i18n("Archived Alarm [read-only]") : i18n("Alarm [read-only]");
	else
		caption = type_caption(newAlarm);
	setCaption(caption);
	setButtonText(Help, i18n("Load Template..."));
	QVBox* mainWidget = new QVBox(this);
	mainWidget->setSpacing(spacingHint());
	setMainWidget(mainWidget);
	if (mTemplate)
	{
		QHBox* box = new QHBox(mainWidget);
		box->setSpacing(spacingHint());
		QLabel* label = new QLabel(i18n("Template name:"), box);
		label->setFixedSize(label->sizeHint());
		mTemplateName = new QLineEdit(box);
		mTemplateName->setReadOnly(mReadOnly);
		label->setBuddy(mTemplateName);
		QWhatsThis::add(box, i18n("Enter the name of the alarm template"));
		box->setFixedHeight(box->sizeHint().height());
	}
	mTabs = new KTabWidget(mainWidget);
	mTabScrollGroup = new StackedScrollGroup(this, mTabs);

	StackedScrollWidget* mainScroll = new StackedScrollWidget(mTabScrollGroup);
	mTabs->addTab(mainScroll, i18n("Alarm"));
	mMainPageIndex = 0;
	PageFrame* mainPage = new PageFrame(mainScroll->viewport());
	connect(mainPage, SIGNAL(shown()), SLOT(slotShowMainPage()));
	mainScroll->addChild(mainPage);
	QVBoxLayout* topLayout = new QVBoxLayout(mainPage, marginHint(), spacingHint());

	// Recurrence tab
	StackedScrollWidget* recurScroll = new StackedScrollWidget(mTabScrollGroup);
	mTabs->addTab(recurScroll, QString::null);
	mRecurPageIndex = 1;
	QVBox* recurFrame = new QVBox(recurScroll->viewport());
	recurFrame->setMargin(marginHint());
	recurScroll->addChild(recurFrame);
	mRecurrenceEdit = new RecurrenceEdit(mReadOnly, recurFrame, "recurPage");
	connect(mRecurrenceEdit, SIGNAL(shown()), SLOT(slotShowRecurrenceEdit()));
	connect(mRecurrenceEdit, SIGNAL(typeChanged(int)), SLOT(slotRecurTypeChange(int)));
	connect(mRecurrenceEdit, SIGNAL(frequencyChanged()), SLOT(slotRecurFrequencyChange()));
	connect(mRecurrenceEdit, SIGNAL(repeatNeedsInitialisation()), SLOT(slotSetSubRepetition()));

	// Controls specific to the alarm type
	QGroupBox* actionBox = new QGroupBox(i18n("Action"), mainPage, "actionGroup");
	topLayout->addWidget(actionBox, 1);
	QVBoxLayout* layout = new QVBoxLayout(actionBox, marginHint(), spacingHint());
	layout->addSpacing(fontMetrics().height() - marginHint() + spacingHint());

	type_init(actionBox, layout);

	// Deferred date/time: visible only for a deferred recurring event.
	mDeferGroup = new QGroupBox(1, Qt::Vertical, i18n("Deferred Alarm"), mainPage, "deferGroup");
	topLayout->addWidget(mDeferGroup);
	QLabel* label = new QLabel(i18n("Deferred to:"), mDeferGroup);
	label->setFixedSize(label->sizeHint());
	mDeferTimeLabel = new QLabel(mDeferGroup);

	mDeferChangeButton = new QPushButton(i18n("C&hange..."), mDeferGroup);
	mDeferChangeButton->setFixedSize(mDeferChangeButton->sizeHint());
	connect(mDeferChangeButton, SIGNAL(clicked()), SLOT(slotEditDeferral()));
	QWhatsThis::add(mDeferChangeButton, i18n("Change the alarm's deferred time, or cancel the deferral"));
	mDeferGroup->addSpace(0);

	QHBoxLayout* hlayout = new QHBoxLayout(topLayout);

	// Date and time entry
	if (mTemplate)
	{
		mTemplateTimeGroup = new ButtonGroup(i18n("Time"), mainPage, "templateGroup");
		connect(mTemplateTimeGroup, SIGNAL(buttonSet(int)), SLOT(slotTemplateTimeType(int)));
		hlayout->addWidget(mTemplateTimeGroup);
		QGridLayout* grid = new QGridLayout(mTemplateTimeGroup, 2, 2, marginHint(), spacingHint());
		grid->addRowSpacing(0, fontMetrics().height() - marginHint() + spacingHint());
		// Get alignment to use in QGridLayout (AlignAuto doesn't work correctly there)
		int alignment = QApplication::reverseLayout() ? Qt::AlignRight : Qt::AlignLeft;

		mTemplateDefaultTime = new RadioButton(i18n("&Default time"), mTemplateTimeGroup, "templateDefTimeButton");
		mTemplateDefaultTime->setFixedSize(mTemplateDefaultTime->sizeHint());
		mTemplateDefaultTime->setReadOnly(mReadOnly);
		QWhatsThis::add(mTemplateDefaultTime,
		      i18n("Do not specify a start time for alarms based on this template. "
		           "The normal default start time will be used."));
		grid->addWidget(mTemplateDefaultTime, 1, 0, alignment);

		QHBox* box = new QHBox(mTemplateTimeGroup);
		box->setSpacing(spacingHint());
		mTemplateUseTime = new RadioButton(i18n("Time:"), box, "templateTimeButton");
		mTemplateUseTime->setFixedSize(mTemplateUseTime->sizeHint());
		mTemplateUseTime->setReadOnly(mReadOnly);
		QWhatsThis::add(mTemplateUseTime,
		      i18n("Specify a start time for alarms based on this template."));
		mTemplateTimeGroup->insert(mTemplateUseTime);
		mTemplateTime = new TimeEdit(box, "templateTimeEdit");
		mTemplateTime->setFixedSize(mTemplateTime->sizeHint());
		mTemplateTime->setReadOnly(mReadOnly);
		QWhatsThis::add(mTemplateTime,
		      QString("%1\n\n%2").arg(i18n("Enter the start time for alarms based on this template."))
		                         .arg(TimeSpinBox::shiftWhatsThis()));
		box->setStretchFactor(new QWidget(box), 1);    // left adjust the controls
		box->setFixedHeight(box->sizeHint().height());
		grid->addWidget(box, 1, 1, alignment);

		mTemplateAnyTime = new RadioButton(i18n("Date only"), mTemplateTimeGroup, "templateAnyTimeButton");
		mTemplateAnyTime->setFixedSize(mTemplateAnyTime->sizeHint());
		mTemplateAnyTime->setReadOnly(mReadOnly);
		QWhatsThis::add(mTemplateAnyTime,
		      i18n("Set the '%1' option for alarms based on this template.").arg(i18n("Any time")));
		grid->addWidget(mTemplateAnyTime, 2, 0, alignment);

		box = new QHBox(mTemplateTimeGroup);
		box->setSpacing(spacingHint());
		mTemplateUseTimeAfter = new RadioButton(AlarmTimeWidget::i18n_w_TimeFromNow(), box, "templateFromNowButton");
		mTemplateUseTimeAfter->setFixedSize(mTemplateUseTimeAfter->sizeHint());
		mTemplateUseTimeAfter->setReadOnly(mReadOnly);
		QWhatsThis::add(mTemplateUseTimeAfter,
		      i18n("Set alarms based on this template to start after the specified time "
		           "interval from when the alarm is created."));
		mTemplateTimeGroup->insert(mTemplateUseTimeAfter);
		mTemplateTimeAfter = new TimeSpinBox(1, maxDelayTime, box);
		mTemplateTimeAfter->setValue(1439);
		mTemplateTimeAfter->setFixedSize(mTemplateTimeAfter->sizeHint());
		mTemplateTimeAfter->setReadOnly(mReadOnly);
		QWhatsThis::add(mTemplateTimeAfter,
		      QString("%1\n\n%2").arg(AlarmTimeWidget::i18n_TimeAfterPeriod())
		                         .arg(TimeSpinBox::shiftWhatsThis()));
		box->setFixedHeight(box->sizeHint().height());
		grid->addWidget(box, 2, 1, alignment);

		hlayout->addStretch();
	}
	else
	{
		mTimeWidget = new AlarmTimeWidget(i18n("Time"), AlarmTimeWidget::AT_TIME, mainPage, "timeGroup");
		connect(mTimeWidget, SIGNAL(dateOnlyToggled(bool)), SLOT(slotAnyTimeToggled(bool)));
		topLayout->addWidget(mTimeWidget);
	}

	// Optional controls depending on More/Less Options button
	mMoreOptions = new QFrame(mainPage);
	mMoreOptions->setFrameStyle(QFrame::NoFrame);
	mMoreOptions->setMargin(0);
	topLayout->addWidget(mMoreOptions);
	QVBoxLayout* moreLayout = new QVBoxLayout(mMoreOptions, 0, spacingHint());

	// Reminder
	mReminder = createReminder(mMoreOptions);
	if (mReminder)
	{
		mReminder->setFixedSize(mReminder->sizeHint());
		moreLayout->addWidget(mReminder, 0, Qt::AlignAuto);
		connect(mTimeWidget, SIGNAL(changed(const KDateTime&)), mReminder, SLOT(setDefaultUnits(const KDateTime&)));
	}

	// Late cancel selector - default = allow late display
	mLateCancel = new LateCancelSelector(true, mMoreOptions);
	moreLayout->addWidget(mLateCancel, 0, Qt::AlignAuto);

	// Acknowledgement confirmation required - default = no confirmation
	hlayout = new QHBoxLayout(moreLayout);
	CheckBox* confirmAck = type_createConfirmAckCheckbox(mMoreOptions);
	if (confirmAck)
	{
		confirmAck->setFixedSize(confirmAck->sizeHint());
		hlayout->add(confirmAck);
		hlayout->addStretch(1);
	}

	if (theApp()->korganizerEnabled())
	{
		// Show in KOrganizer checkbox
		mShowInKorganizer = new CheckBox(i18n_ShowInKOrganizer(), mMoreOptions);
		mShowInKorganizer->setFixedSize(mShowInKorganizer->sizeHint());
		QWhatsThis::add(mShowInKorganizer, i18n("Check to copy the alarm into KOrganizer's calendar"));
		hlayout->add(mShowInKorganizer);
		if (!confirmAck)
			hlayout->addStretch(1);
	}

	setButtonWhatsThis(Ok, i18n("Schedule the alarm at the specified time."));

	// Hide optional controls
	KConfig* config = KGlobal::config();
	config->setGroup(EDIT_MORE_GROUP);
	showOptions(config->readBoolEntry(EDIT_MORE_KEY, false));

	// Initialise the state of all controls according to the specified event, if any
	initValues(event);
	updateGeometry();
	if (mTemplateName)
		mTemplateName->setFocus();

	// Save the initial state of all controls so that we can later tell if they have changed
	saveState((event && (mTemplate || !event->isTemplate())) ? event : 0);

	// Note the current desktop so that the dialog can be shown on it.
	// If a main window is visible, the dialog will by KDE default always appear on its
	// desktop. If the user invokes the dialog via the system tray on a different desktop,
	// that can cause confusion.
	mDesktop = KWin::currentDesktop();
}

EditAlarmDlg::~EditAlarmDlg()
{
	delete mSavedEvent;
}

/******************************************************************************
 * Initialise the dialogue controls from the specified event.
 */
void EditAlarmDlg::initValues(const KAEvent* event)
{
	setReadOnly(mDesiredReadOnly);

	mChanged           = false;
	mOnlyDeferred      = false;
	mExpiredRecurrence = false;
	mLateCancel->showAutoClose(false);
	bool deferGroupVisible = false;
	if (event)
	{
		// Set the values to those for the specified event
		if (mTemplate)
			mTemplateName->setText(event->templateName());
		bool recurs = event->recurs();
		if ((recurs || event->repeatCount())  &&  !mTemplate  &&  event->deferred())
		{
			deferGroupVisible = true;
			mDeferDateTime = event->deferDateTime();
			mDeferTimeLabel->setText(mDeferDateTime.formatLocale());
			mDeferGroup->show();
		}
		if (mTemplate)
		{
			// Editing a template
			int afterTime = event->isTemplate() ? event->templateAfterTime() : -1;
			bool noTime   = !afterTime;
			bool useTime  = !event->mainDateTime().isDateOnly();
			int button = mTemplateTimeGroup->id(noTime          ? mTemplateDefaultTime :
			                                    (afterTime > 0) ? mTemplateUseTimeAfter :
			                                    useTime         ? mTemplateUseTime : mTemplateAnyTime);
			mTemplateTimeGroup->setButton(button);
			mTemplateTimeAfter->setValue(afterTime > 0 ? afterTime : 1);
			if (!noTime && useTime)
				mTemplateTime->setValue(event->mainDateTime().kDateTime().time());
			else
				mTemplateTime->setValue(0);
		}
		else
		{
			if (event->isTemplate())
			{
				// Initialising from an alarm template: use current date
				KDateTime now = KDateTime::currentDateTime(Preferences::timeZone());
				int afterTime = event->templateAfterTime();
				if (afterTime >= 0)
				{
					mTimeWidget->setDateTime(now.addSecs(afterTime * 60));
					mTimeWidget->selectTimeFromNow();
				}
				else
				{
					KDateTime dt = event->startDateTime().kDateTime();
					dt.setTimeSpec(Preferences::timeZone());
					QDate d = now.date();
					if (!dt.isDateOnly()  &&  now.time() >= dt.time())
						d = d.addDays(1);     // alarm time has already passed, so use tomorrow
					dt.setDate(d);
					mTimeWidget->setDateTime(dt);
				}
			}
			else
			{
				mExpiredRecurrence = recurs && event->mainExpired();
				mTimeWidget->setDateTime(recurs || event->category() == KCalEvent::ARCHIVED ? event->startDateTime()
				                         : event->mainExpired() ? event->deferDateTime() : event->mainDateTime());
			}
		}

		KAEvent::Action action = event->action();
		AlarmText altext;
		if (event->commandScript())
			altext.setScript(event->cleanText());
		else
			altext.setText(event->cleanText());
		setAction(action, altext);

		mLateCancel->setMinutes(event->lateCancel(), event->startDateTime().isDateOnly(),
		                        TimePeriod::HOURS_MINUTES);
		if (mShowInKorganizer)
			mShowInKorganizer->setChecked(event->copyToKOrganizer());
		type_initValues(event);
		mRecurrenceEdit->set(*event);   // must be called after mTimeWidget is set up, to ensure correct date-only enabling
		mTabs->setTabLabel(mTabs->page(mRecurPageIndex), recurText(*event));
	}
	else
	{
		// Set the values to their defaults
		KDateTime defaultTime = KDateTime::currentUtcDateTime().addSecs(60).toTimeSpec(Preferences::timeZone());
		if (mTemplate)
		{
			mTemplateTimeGroup->setButton(mTemplateTimeGroup->id(mTemplateDefaultTime));
			mTemplateTime->setValue(0);
			mTemplateTimeAfter->setValue(1);
		}
		else
			mTimeWidget->setDateTime(defaultTime);
		mLateCancel->setMinutes((Preferences::defaultLateCancel() ? 1 : 0), false, TimePeriod::HOURS_MINUTES);
		if (mShowInKorganizer)
			mShowInKorganizer->setChecked(Preferences::defaultCopyToKOrganizer());
		type_initValues(0);
		mRecurrenceEdit->setDefaults(defaultTime);   // must be called after mTimeWidget is set up, to ensure correct date-only enabling
		slotRecurFrequencyChange();      // update the Recurrence text
	}
	mLateCancel->setFixedSize(mLateCancel->sizeHint());   // do this after type_initValues()
        if (mReminder  &&  mTimeWidget)
		mReminder->setDefaultUnits(mTimeWidget->getDateTime(0, false, false));

	if (!deferGroupVisible)
		mDeferGroup->hide();

	bool empty = AlarmCalendar::resources()->events(KCalEvent::TEMPLATE).isEmpty();
	enableButton(Help, !empty);   // Load Template button
}

/******************************************************************************
 * Set the read-only status of all non-template controls.
 */
void EditAlarmDlg::setReadOnly(bool readOnly)
{
	mReadOnly = readOnly;

	if (mTimeWidget)
		mTimeWidget->setReadOnly(readOnly);
	mLateCancel->setReadOnly(readOnly);
	if (readOnly)
		mDeferChangeButton->hide();
	else
		mDeferChangeButton->show();
	if (mShowInKorganizer)
		mShowInKorganizer->setReadOnly(readOnly);
}

/******************************************************************************
 * Save the state of all controls.
 */
void EditAlarmDlg::saveState(const KAEvent* event)
{
	delete mSavedEvent;
	mSavedEvent = 0;
	if (event)
		mSavedEvent = new KAEvent(*event);
	if (mTemplate)
	{
		mSavedTemplateName      = mTemplateName->text();
		mSavedTemplateTimeType  = mTemplateTimeGroup->selected();
		mSavedTemplateTime      = mTemplateTime->time();
		mSavedTemplateAfterTime = mTemplateTimeAfter->value();
	}
	checkText(mSavedTextFileCommandMessage, false);
	if (mTimeWidget)
		mSavedDateTime = mTimeWidget->getDateTime(0, false, false);
	mSavedLateCancel       = mLateCancel->minutes();
	if (mShowInKorganizer)
		mSavedShowInKorganizer = mShowInKorganizer->isChecked();
	mSavedRecurrenceType   = mRecurrenceEdit->repeatType();
}

/******************************************************************************
 * Check whether any of the controls has changed state since the dialog was
 * first displayed.
 * Reply = true if any non-deferral controls have changed, or if it's a new event.
 *       = false if no non-deferral controls have changed. In this case,
 *         mOnlyDeferred indicates whether deferral controls may have changed.
 */
bool EditAlarmDlg::stateChanged() const
{
	mChanged      = true;
	mOnlyDeferred = false;
	if (!mSavedEvent)
		return true;
	QString textFileCommandMessage;
	checkText(textFileCommandMessage, false);
	if (mTemplate)
	{
		if (mSavedTemplateName     != mTemplateName->text()
		||  mSavedTemplateTimeType != mTemplateTimeGroup->selected()
		||  (mTemplateUseTime->isOn()  &&  mSavedTemplateTime != mTemplateTime->time())
		||  (mTemplateUseTimeAfter->isOn()  &&  mSavedTemplateAfterTime != mTemplateTimeAfter->value()))
			return true;
	}
	else
	{
		KDateTime dt = mTimeWidget->getDateTime(0, false, false);
		if (mSavedDateTime.timeSpec() != dt.timeSpec()  ||  mSavedDateTime != dt)
			return true;
	}
	if (mSavedLateCancel       != mLateCancel->minutes()
	||  (mShowInKorganizer && mSavedShowInKorganizer != mShowInKorganizer->isChecked())
	||  textFileCommandMessage != mSavedTextFileCommandMessage
	||  mSavedRecurrenceType   != mRecurrenceEdit->repeatType())
		return true;
	if (type_stateChanged())
		return true;
	if (mRecurrenceEdit->stateChanged())
		return true;
	if (mSavedEvent  &&  mSavedEvent->deferred())
		mOnlyDeferred = true;
	mChanged = false;
	return false;
}

/******************************************************************************
 * Get the currently entered dialogue data.
 * The data is returned in the supplied KAEvent instance.
 * Reply = false if the only change has been to an existing deferral.
 */
bool EditAlarmDlg::getEvent(KAEvent& event, AlarmResource*& resource)
{
	resource = mResource;
	if (mChanged)
	{
		// It's a new event, or the edit controls have changed
		setEvent(event, mAlarmMessage, false);
		return true;
	}

	// Only the deferral time may have changed
	event = *mSavedEvent;
	if (mOnlyDeferred)
	{
		// Just modify the original event, to avoid expired recurring events
		// being returned as rubbish.
		if (mDeferDateTime.isValid())
			event.defer(mDeferDateTime, event.reminderDeferral(), false);
		else
			event.cancelDefer();
	}
	return false;
}

/******************************************************************************
*  Extract the data in the dialogue and set up a KAEvent from it.
*  If 'trial' is true, the event is set up for a simple one-off test, ignoring
*  recurrence, reminder, template etc. data.
*/
void EditAlarmDlg::setEvent(KAEvent& event, const QString& text, bool trial)
{
	KDateTime dt;
	if (!trial)
	{
		if (!mTemplate)
			dt = mAlarmDateTime.effectiveKDateTime();
		else if (mTemplateUseTime->isOn())
			dt = KDateTime(QDate(2000,1,1), mTemplateTime->time());
	}

	type_setEvent(event, dt, text, (trial ? 0 : mLateCancel->minutes()), trial);

	if (!trial)
	{
		if (mRecurrenceEdit->repeatType() != RecurrenceEdit::NO_RECUR)
		{
			mRecurrenceEdit->updateEvent(event, !mTemplate);
			KDateTime now = KDateTime::currentDateTime(mAlarmDateTime.timeSpec());
			bool dateOnly = mAlarmDateTime.isDateOnly();
			if ((dateOnly  &&  mAlarmDateTime.date() < now.date())
			||  (!dateOnly  &&  mAlarmDateTime.kDateTime() < now))
			{
				// A timed recurrence has an entered start date which has
				// already expired, so we must adjust the next repetition.
				event.setNextOccurrence(now);
			}
			mAlarmDateTime = event.startDateTime();
			if (mDeferDateTime.isValid()  &&  mDeferDateTime < mAlarmDateTime)
			{
				bool deferral = true;
				bool deferReminder = false;
				int reminder = mReminder ? mReminder->minutes() : 0;
				if (reminder)
				{
					DateTime remindTime = mAlarmDateTime.addMins(-reminder);
					if (mDeferDateTime >= remindTime)
					{
						if (remindTime > KDateTime::currentUtcDateTime())
							deferral = false;    // ignore deferral if it's after next reminder
						else if (mDeferDateTime > remindTime)
							deferReminder = true;    // it's the reminder which is being deferred
					}
				}
				if (deferral)
					event.defer(mDeferDateTime, deferReminder, false);
			}
		}
		if (mTemplate)
		{
			int afterTime = mTemplateDefaultTime->isOn() ? 0
			              : mTemplateUseTimeAfter->isOn() ? mTemplateTimeAfter->value() : -1;
			event.setTemplate(mTemplateName->text(), afterTime);
		}
	}
}

/******************************************************************************
 * Get the currently specified alarm flag bits.
 */
int EditAlarmDlg::getAlarmFlags() const
{
	return (mShowInKorganizer && mShowInKorganizer->isChecked()                  ? KAEvent::COPY_KORGANIZER : 0)
	     | (mRecurrenceEdit->repeatType() == RecurrenceEdit::AT_LOGIN            ? KAEvent::REPEAT_AT_LOGIN : 0)
	     | ((mTemplate ? mTemplateAnyTime->isOn() : mAlarmDateTime.isDateOnly()) ? KAEvent::ANY_TIME : 0);
}

/******************************************************************************
*  Called when the dialog is displayed.
*  The first time through, sets the size to the same as the last time it was
*  displayed.
*/
void EditAlarmDlg::showEvent(QShowEvent* se)
{
	KDialogBase::showEvent(se);
	if (!mDeferGroupHeight)
	{
		mDeferGroupHeight = mDeferGroup->height() + spacingHint();
		QSize s;
		if (KAlarm::readConfigWindowSize(EDIT_DIALOG_NAME, s))
		{
			bool defer = !mDeferGroup->isHidden();
			s.setHeight(s.height() + (defer ? mDeferGroupHeight : 0));
			if (!defer)
				mTabScrollGroup->setSized();
			resize(s);
		}
	}
	QTimer::singleShot(0, this, SLOT(slotResize()));
	KWin::setOnDesktop(winId(), mDesktop);    // ensure it displays on the desktop expected by the user
}

/******************************************************************************
* Update the tab sizes (again) and if the resized dialog height is greater
* than the minimum, resize it again. This is necessary because (a) resizing
* tabs doesn't always work properly the first time, and (b) resizing to the
* minimum size hint doesn't always work either.
*/
void EditAlarmDlg::slotResize()
{
	QSize s = mTabScrollGroup->adjustSize(true);
	s = minimumSizeHint();
	if (height() > s.height())
	{
		// Resize to slightly greater than the minimum height.
		// This is for some unkown reason necessary, since
		// sometimes resizing to the minimum height fails.
		resize(s.width(), s.height() + 2);
	}
}

/******************************************************************************
* Return the minimum size for the dialog.
* If the minimum size would be too high to fit the desktop, the tab contents
* are made scrollable.
*/
QSize EditAlarmDlg::minimumSizeHint() const
{
	QSize s = mTabScrollGroup->adjustSize();
	if (s.isValid())
		return s;
	return KDialogBase::minimumSizeHint();
}

/******************************************************************************
*  Called when the dialog's size has changed.
*  Records the new size (adjusted to ignore the optional height of the deferred
*  time edit widget) in the config file.
*/
void EditAlarmDlg::resizeEvent(QResizeEvent* re)
{
	if (isVisible() && mDeferGroupHeight)
	{
		QSize s = re->size();
		s.setHeight(s.height() - (mDeferGroup->isHidden() ? 0 : mDeferGroupHeight));
		KAlarm::writeConfigWindowSize(EDIT_DIALOG_NAME, s);
	}
	KDialog::resizeEvent(re);
}

/******************************************************************************
*  Called when the OK button is clicked.
*  Validate the input data.
*/
void EditAlarmDlg::slotOk()
{
	if (!stateChanged())
	{
		// No changes have been made except possibly to an existing deferral
		if (!mOnlyDeferred)
			reject();
		else
			accept();
		return;
	}
	RecurrenceEdit::RepeatType recurType = mRecurrenceEdit->repeatType();
	if (mTimeWidget
	&&  mTabs->currentPageIndex() == mRecurPageIndex  &&  recurType == RecurrenceEdit::AT_LOGIN)
		mTimeWidget->setDateTime(mRecurrenceEdit->endDateTime());
	bool timedRecurrence = mRecurrenceEdit->isTimedRepeatType();    // does it recur other than at login?
	if (mTemplate)
	{
		// Check that the template name is not blank and is unique
		QString errmsg;
		QString name = mTemplateName->text();
		if (name.isEmpty())
			errmsg = i18n("You must enter a name for the alarm template");
		else if (name != mSavedTemplateName)
		{
			if (AlarmCalendar::resources()->templateEvent(name))
				errmsg = i18n("Template name is already in use");
		}
		if (!errmsg.isEmpty())
		{
			mTemplateName->setFocus();
			KMessageBox::sorry(this, errmsg);
			return;
		}
	}
	else
	{
		QWidget* errWidget;
		mAlarmDateTime = mTimeWidget->getDateTime(0, !timedRecurrence, false, &errWidget);
		if (errWidget)
		{
			// It's more than just an existing deferral being changed, so the time matters
			mTabs->setCurrentPage(mMainPageIndex);
			errWidget->setFocus();
			mTimeWidget->getDateTime();   // display the error message now
			return;
		}
	}
	if (!type_validate(false))
		return;
	if (!mTemplate)
	{
		if (timedRecurrence)
		{
			KAEvent event;
			AlarmResource* r;
			getEvent(event, r);     // this may adjust mAlarmDateTime
			KDateTime now = KDateTime::currentDateTime(mAlarmDateTime.timeSpec());
			bool dateOnly = mAlarmDateTime.isDateOnly();
			if ((dateOnly  &&  mAlarmDateTime.date() < now.date())
			||  (!dateOnly  &&  mAlarmDateTime.kDateTime() < now))
			{
				// A timed recurrence has an entered start date which
				// has already expired, so we must adjust it.
				if (event.nextOccurrence(now, mAlarmDateTime, KAEvent::ALLOW_FOR_REPETITION) == KAEvent::NO_OCCURRENCE)
				{
					KMessageBox::sorry(this, i18n("Recurrence has already expired"));
					return;
				}
			}
			if (event.workTimeOnly()  &&  !event.nextTrigger(KAEvent::DISPLAY_TRIGGER).isValid())
			{
				if (KMessageBox::warningContinueCancel(this, i18n("The alarm will never occur during working hours"))
				    != KMessageBox::Continue)
					return;
			}
		}
		QString errmsg;
		QWidget* errWidget = mRecurrenceEdit->checkData(mAlarmDateTime.effectiveKDateTime(), errmsg);
		if (errWidget)
		{
			mTabs->setCurrentPage(mRecurPageIndex);
			errWidget->setFocus();
			KMessageBox::sorry(this, errmsg);
			return;
		}
	}
	if (recurType != RecurrenceEdit::NO_RECUR)
	{
		KAEvent recurEvent;
		int longestRecurMinutes = -1;
		int reminder = mReminder ? mReminder->minutes() : 0;
		if (reminder  &&  !mReminder->isOnceOnly())
		{
			mRecurrenceEdit->updateEvent(recurEvent, false);
			longestRecurMinutes = recurEvent.longestRecurrenceInterval().asSeconds() / 60;
			if (longestRecurMinutes  &&  reminder >= longestRecurMinutes)
			{
				mTabs->setCurrentPage(mMainPageIndex);
				mReminder->setFocusOnCount();
				KMessageBox::sorry(this, i18n("Reminder period must be less than the recurrence interval, unless '%1' is checked."
				                             ).arg(Reminder::i18n_first_recurrence_only()));
				return;
			}
		}
		if (mRecurrenceEdit->subRepeatCount())
		{
			if (longestRecurMinutes < 0)
			{
				mRecurrenceEdit->updateEvent(recurEvent, false);
				longestRecurMinutes = recurEvent.longestRecurrenceInterval().asSeconds() / 60;
			}
			if (longestRecurMinutes > 0
			&&  recurEvent.repeatInterval().asSeconds()/60 * recurEvent.repeatCount() >= longestRecurMinutes - reminder)
			{
				KMessageBox::sorry(this, i18n("The duration of a repetition within the recurrence must be less than the recurrence interval minus any reminder period"));
				mRecurrenceEdit->activateSubRepetition();   // display the alarm repetition dialog again
				return;
			}
			if (!recurEvent.repeatInterval().isDaily()
			&&  ((mTemplate && mTemplateAnyTime->isOn())  ||  (!mTemplate && mAlarmDateTime.isDateOnly())))
			{
				KMessageBox::sorry(this, i18n("For a repetition within the recurrence, its period must be in units of days or weeks for a date-only alarm"));
				mRecurrenceEdit->activateSubRepetition();   // display the alarm repetition dialog again
				return;
			}
		}
	}
	if (!checkText(mAlarmMessage))
		return;

	mResource = 0;
	// A null resource event ID indicates that the caller already
	// knows which resource to use.
	if (!mResourceEventId.isNull())
	{
		if (!mResourceEventId.isEmpty())
		{
			mResource = AlarmCalendar::resources()->resourceForEvent(mResourceEventId);
			AlarmResource::Type type = mTemplate ? AlarmResource::TEMPLATE : AlarmResource::ACTIVE;
			if (mResource->alarmType() != type)
				mResource = 0;   // event may have expired while dialogue was open
		}
		bool cancelled = false;
		if (!mResource  ||  !mResource->writable())
		{
			KCalEvent::Status type = mTemplate ? KCalEvent::TEMPLATE : KCalEvent::ACTIVE;
			mResource = AlarmResources::instance()->destination(type, this, false, &cancelled);
		}
		if (!mResource)
		{
			if (!cancelled)
				KMessageBox::sorry(this, i18n("You must select a resource to save the alarm in"));
			return;
		}
	}
	accept();
}

/******************************************************************************
*  Called when the Try button is clicked.
*  Display/execute the alarm immediately for the user to check its configuration.
*/
void EditAlarmDlg::slotTry()
{
	QString text;
	if (checkText(text))
	{
		if (!type_validate(true))
			return;
		KAEvent event;
		setEvent(event, text, true);
		void* proc = theApp()->execAlarm(event, event.firstAlarm(), false, false);
		if (proc  &&  proc != (void*)-1)
			type_trySuccessMessage((ShellProcess*)proc, text);
	}
}

/******************************************************************************
*  Called when the Cancel button is clicked.
*/
void EditAlarmDlg::slotCancel()
{
	reject();
}

/******************************************************************************
*  Called when the Load Template button is clicked.
*  Prompt to select a template and initialise the dialogue with its contents.
*/
void EditAlarmDlg::slotHelp()
{
	TemplatePickDlg dlg(this, "templPickDlg");
	if (dlg.exec() == QDialog::Accepted)
		initValues(dlg.selectedTemplate());
}

/******************************************************************************
* Called when the More Options or Less Options buttons are clicked.
* Show/hide the optional options and swap the More/Less buttons, and save the
* new setting as the default from now on.
*/
void EditAlarmDlg::slotDefault()
{
	showOptions(!mShowingMore);
	KConfig* config = KGlobal::config();
	config->setGroup(EDIT_MORE_GROUP);
	config->writeEntry(EDIT_MORE_KEY, mShowingMore);
}

/******************************************************************************
* Show/hide the optional options and swap the More/Less buttons.
*/
void EditAlarmDlg::showOptions(bool more)
{
	kdDebug(5950) << "EditAlarmDlg::showOptions(" << more << ")" << endl;
	if (more)
	{
		mMoreOptions->show();
		setButtonText(Default, i18n("Less Options <<"));
	}
	else
	{
		mMoreOptions->hide();
		setButtonText(Default, i18n("More Options >>"));
	}
	if (mTimeWidget)
		mTimeWidget->showMoreOptions(more);
	type_showOptions(more);
	mRecurrenceEdit->showMoreOptions(more);
	mShowingMore = more;
	QTimer::singleShot(0, this, SLOT(slotResize()));
}

/******************************************************************************
* Called when the Change deferral button is clicked.
*/
void EditAlarmDlg::slotEditDeferral()
{
	if (!mTimeWidget)
		return;
	bool limit = true;
	Duration repeatInterval;
	int repeatCount = mRecurrenceEdit->subRepeatCount(&repeatInterval);
	DateTime start = mSavedEvent->recurs() ? (mExpiredRecurrence ? DateTime() : mSavedEvent->mainDateTime())
	               : mTimeWidget->getDateTime(0, !repeatCount, !mExpiredRecurrence);
	if (!start.isValid())
	{
		if (!mExpiredRecurrence)
			return;
		limit = false;
	}
	KDateTime now = KDateTime::currentUtcDateTime();
	if (limit)
	{
		if (repeatCount  &&  start < now)
		{
			// Sub-repetition - find the time of the next one
			int repetition = repeatInterval.isDaily()
			               ? (start.daysTo(now) + repeatInterval.asDays() - 1) / repeatInterval.asDays()
			               : (start.secsTo(now) + repeatInterval.asSeconds() - 1) / repeatInterval.asSeconds();
			if (repetition > repeatCount)
			{
				mTimeWidget->getDateTime();    // output the appropriate error message
				return;
			}
			start = (repeatInterval * repetition).end(start.kDateTime());
		}
	}

	bool deferred = mDeferDateTime.isValid();
	DeferAlarmDlg deferDlg((deferred ? mDeferDateTime : DateTime(now.addSecs(60))),
	                       deferred, this, "EditDeferDlg");
	if (limit)
	{
		// Don't allow deferral past the next recurrence
		int reminder = mReminder ? mReminder->minutes() : 0;
		if (reminder)
		{
			DateTime remindTime = start.addMins(-reminder);
			if (KDateTime::currentUtcDateTime() < remindTime)
				start = remindTime;
		}
		deferDlg.setLimit(start.addSecs(-60));
	}
	if (deferDlg.exec() == QDialog::Accepted)
	{
		mDeferDateTime = deferDlg.getDateTime();
		mDeferTimeLabel->setText(mDeferDateTime.isValid() ? mDeferDateTime.formatLocale() : QString::null);
	}
}

/******************************************************************************
*  Called when the main page is shown.
*  Sets the focus widget to the first edit field.
*/
void EditAlarmDlg::slotShowMainPage()
{
	if (!mMainPageShown)
	{
		if (mTemplateName)
			mTemplateName->setFocus();
		mMainPageShown = true;
	}
	if (mTimeWidget)
	{
		if (!mReadOnly  &&  mRecurPageShown  &&  mRecurrenceEdit->repeatType() == RecurrenceEdit::AT_LOGIN)
			mTimeWidget->setDateTime(mRecurrenceEdit->endDateTime());
		if (mReadOnly  ||  mRecurrenceEdit->isTimedRepeatType())
			mTimeWidget->setMinDateTime();             // don't set a minimum date/time
		else
			mTimeWidget->setMinDateTimeIsCurrent();    // set the minimum date/time to track the clock
	}
}

/******************************************************************************
*  Called when the recurrence edit page is shown.
*  The recurrence defaults are set to correspond to the start date.
*  The first time, for a new alarm, the recurrence end date is set according to
*  the alarm start time.
*/
void EditAlarmDlg::slotShowRecurrenceEdit()
{
	mRecurPageIndex = mTabs->currentPageIndex();
	if (!mReadOnly  &&  !mTemplate)
	{
		mAlarmDateTime = mTimeWidget->getDateTime(0, false, false);
		KDateTime now = KDateTime::currentDateTime(mAlarmDateTime.timeSpec());
		bool expired = (mAlarmDateTime.effectiveKDateTime() < now);
		if (mRecurSetDefaultEndDate)
		{
			mRecurrenceEdit->setDefaultEndDate(expired ? now.date() : mAlarmDateTime.date());
			mRecurSetDefaultEndDate = false;
		}
		mRecurrenceEdit->setStartDate(mAlarmDateTime.date(), now.date());
		if (mRecurrenceEdit->repeatType() == RecurrenceEdit::AT_LOGIN)
			mRecurrenceEdit->setEndDateTime(expired ? now : mAlarmDateTime.kDateTime());
	}
	mRecurPageShown = true;
}

/******************************************************************************
*  Called when the recurrence type selection changes.
*  Enables/disables date-only alarms as appropriate.
*  Enables/disables controls depending on at-login setting.
*/
void EditAlarmDlg::slotRecurTypeChange(int repeatType)
{
	bool atLogin = (mRecurrenceEdit->repeatType() == RecurrenceEdit::AT_LOGIN);
	if (!mTemplate)
	{
		bool recurs = (mRecurrenceEdit->repeatType() != RecurrenceEdit::NO_RECUR);
		if (mDeferGroup)
			mDeferGroup->setEnabled(recurs);
		mTimeWidget->enableAnyTime(!recurs || repeatType != RecurrenceEdit::SUBDAILY);
		if (atLogin)
		{
			mAlarmDateTime = mTimeWidget->getDateTime(0, false, false);
			mRecurrenceEdit->setEndDateTime(mAlarmDateTime.kDateTime());
		}
		if (mReminder)
			mReminder->enableOnceOnly(recurs && !atLogin);
	}
	if (mReminder)
		mReminder->setEnabled(!atLogin);
	mLateCancel->setEnabled(!atLogin);
	if (mShowInKorganizer)
		mShowInKorganizer->setEnabled(!atLogin);
	slotRecurFrequencyChange();
}

/******************************************************************************
*  Called when the recurrence frequency selection changes, or the sub-
*  repetition interval changes.
*  Updates the recurrence frequency text.
*/
void EditAlarmDlg::slotRecurFrequencyChange()
{
	slotSetSubRepetition();
	KAEvent event;
	mRecurrenceEdit->updateEvent(event, false);
	mTabs->setTabLabel(mTabs->page(mRecurPageIndex), recurText(event));
}

/******************************************************************************
*  Called when the Repetition within Recurrence button has been pressed to
*  display the sub-repetition dialog.
*  Alarm repetition has the following restrictions:
*  1) Not allowed for a repeat-at-login alarm
*  2) For a date-only alarm, the repeat interval must be a whole number of days.
*  3) The overall repeat duration must be less than the recurrence interval.
*/
void EditAlarmDlg::slotSetSubRepetition()
{
	bool dateOnly = mTemplate ? mTemplateAnyTime->isOn() : mTimeWidget->anyTime();
	mRecurrenceEdit->setSubRepetition((mReminder ? mReminder->minutes() : 0), dateOnly);
}

/******************************************************************************
*  Called when one of the template time radio buttons is clicked,
*  to enable or disable the template time entry spin boxes.
*/
void EditAlarmDlg::slotTemplateTimeType(int)
{
	mTemplateTime->setEnabled(mTemplateUseTime->isOn());
	mTemplateTimeAfter->setEnabled(mTemplateUseTimeAfter->isOn());
}

/******************************************************************************
*  Called when the "Any time" checkbox is toggled in the date/time widget.
*  Sets the advance reminder and late cancel units to days if any time is checked.
*/
void EditAlarmDlg::slotAnyTimeToggled(bool anyTime)
{
	if (mReminder  &&  mReminder->isReminder())
		mReminder->setDateOnly(anyTime);
	mLateCancel->setDateOnly(anyTime);
}

bool EditAlarmDlg::dateOnly() const
{
	return mTimeWidget ? mTimeWidget->anyTime() : mTemplateAnyTime->isChecked();
}

bool EditAlarmDlg::isTimedRecurrence() const
{
	return mRecurrenceEdit->isTimedRepeatType();
}

void EditAlarmDlg::showMainPage()
{
	mTabs->setCurrentPage(mMainPageIndex);
}
