/*
 *  editdlgtypes.cpp  -  dialogues to create or edit alarm or template types
 *  Program:  kalarm
 *  Copyright © 2001-2008 by David Jarvie <software@astrojar.org.uk>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "kalarm.h"

#include <qlayout.h>
#include <qgroupbox.h>
#include <qlabel.h>
#include <qwhatsthis.h>
#include <qtooltip.h>
#include <qdir.h>
#include <qstyle.h>

#include <klocale.h>
#include <kfiledialog.h>
#include <kiconloader.h>
#include <kio/netaccess.h>
#include <kfileitem.h>
#include <kmessagebox.h>
#include <kabc/addresseedialog.h>
#include <kdebug.h>

#include <libkcal/icaldrag.h>

#include "buttongroup.h"
#include "checkbox.h"
#include "colourbutton.h"
#include "emailidcombo.h"
#include "fontcolourbutton.h"
#include "functions.h"
#include "kalarmapp.h"
#include "kamail.h"
#include "latecancel.h"
#include "lineedit.h"
#include "mainwindow.h"
#include "pickfileradio.h"
#include "preferences.h"
#include "radiobutton.h"
#include "reminder.h"
#include "shellprocess.h"
#include "soundpicker.h"
#include "specialactions.h"
#include "templatepickdlg.h"
#include "timespinbox.h"
#include "editdlgtypes.moc"
#include "editdlgprivate.moc"

using namespace KCal;

enum { tTEXT, tFILE, tCOMMAND };  // order of mTypeCombo items


/*=============================================================================
= Class PickLogFileRadio
=============================================================================*/
class PickLogFileRadio : public PickFileRadio
{
    public:
	PickLogFileRadio(QPushButton* b, LineEdit* e, const QString& text, QButtonGroup* parent, const char* name = 0)
		: PickFileRadio(b, e, text, parent, name) { }
	virtual QString pickFile()    // called when browse button is pressed to select a log file
	{
		return KAlarm::browseFile(i18n("Choose Log File"), mDefaultDir, fileEdit()->text(), QString::null,
		                          KFile::LocalOnly, parentWidget(), "pickLogFile");
	}
    private:
	QString mDefaultDir;   // default directory for log file browse button
};


/*=============================================================================
= Class EditDisplayAlarmDlg
= Dialog to edit display alarms.
=============================================================================*/

// Collect these widget labels together to ensure consistent wording and
// translations across different modules.
QString EditDisplayAlarmDlg::i18n_ConfirmAck()         { return i18n("Confirm acknowledgment"); }
QString EditDisplayAlarmDlg::i18n_k_ConfirmAck()       { return i18n("Confirm ac&knowledgment"); }


/******************************************************************************
* Constructor.
* Parameters:
*   Template = true to edit/create an alarm template
*            = false to edit/create an alarm.
*   event   != to initialise the dialog to show the specified event's data.
*/
EditDisplayAlarmDlg::EditDisplayAlarmDlg(bool Template, bool newAlarm, QWidget* parent, const char* name, GetResourceType getResource)
	: EditAlarmDlg(Template, KAEvent::MESSAGE, parent, name, getResource),
	  mSpecialActionsButton(0),
	  mReminderDeferral(false),
	  mReminderArchived(false)
{
	kdDebug(5950) << "EditDisplayAlarmDlg::EditDisplayAlarmDlg(new)" << endl;
	init(0, newAlarm);
}

EditDisplayAlarmDlg::EditDisplayAlarmDlg(bool Template, const KAEvent* event, bool newAlarm, QWidget* parent,
                                         const char* name, GetResourceType getResource, bool readOnly)
	: EditAlarmDlg(Template, event, parent, name, getResource, readOnly),
	  mSpecialActionsButton(0),
	  mReminderDeferral(false),
	  mReminderArchived(false)
{
	kdDebug(5950) << "EditDisplayAlarmDlg::EditDisplayAlarmDlg(event.id())" << endl;
	init(event, newAlarm);
}

/******************************************************************************
* Return the window caption.
*/
QString EditDisplayAlarmDlg::type_caption(bool newAlarm) const
{
	return isTemplate() ? (newAlarm ? i18n("New Display Alarm Template") : i18n("Edit Display Alarm Template"))
	                    : (newAlarm ? i18n("New Display Alarm") : i18n("Edit Display Alarm"));
}

/******************************************************************************
* Set up the dialog controls common to display alarms.
*/
void EditDisplayAlarmDlg::type_init(QWidget* parent, QVBoxLayout* frameLayout)
{
	// Display type combo box
	QHBox* box = new QHBox(parent);    // to group widgets for QWhatsThis text
	box->setMargin(0);
	box->setSpacing(KDialog::spacingHint());
	QLabel* label = new QLabel(i18n("Display type:"), box);
	label->setFixedSize(label->sizeHint());
	mTypeCombo = new ComboBox(box);
	QString textItem    = i18n("Text message");
	QString fileItem    = i18n("File contents");
	QString commandItem = i18n("Command output");
	mTypeCombo->insertItem(textItem, tTEXT);     // index = tTEXT
	mTypeCombo->insertItem(fileItem, tFILE);     // index = tFILE
	mTypeCombo->insertItem(commandItem, tCOMMAND);  // index = tCOMMAND
	mTypeCombo->setFixedSize(mTypeCombo->sizeHint());
	connect(mTypeCombo, SIGNAL(activated(int)), SLOT(slotAlarmTypeChanged(int)));
	label->setBuddy(mTypeCombo);
	QWhatsThis::add(box, i18n("Select what the alarm should display:\n"
	      "- '%1': the alarm will display the text message you type in.\n"
	      "- '%2': the alarm will display the contents of a text or image file.\n"
	      "- '%3': the alarm will display the output from a command.")
	      .arg(textItem).arg(fileItem).arg(commandItem));
	box->setStretchFactor(new QWidget(box), 1);    // left adjust the control
	frameLayout->addWidget(box);

	// Text message edit box
	mTextMessageEdit = new TextEdit(parent);
	mTextMessageEdit->setWordWrap(KTextEdit::NoWrap);
	QWhatsThis::add(mTextMessageEdit, i18n("Enter the text of the alarm message. It may be multi-line."));
	frameLayout->addWidget(mTextMessageEdit);

	// File name edit box
	mFileBox = new QHBox(parent);
	frameLayout->addWidget(mFileBox);
	mFileMessageEdit = new LineEdit(LineEdit::Url, mFileBox);
	mFileMessageEdit->setAcceptDrops(true);
	QWhatsThis::add(mFileMessageEdit, i18n("Enter the name or URL of a text or image file to display."));

	// File browse button
	mFileBrowseButton = new QPushButton(mFileBox);
	mFileBrowseButton->setPixmap(SmallIcon("fileopen"));
	mFileBrowseButton->setFixedSize(mFileBrowseButton->sizeHint());
	QToolTip::add(mFileBrowseButton, i18n("Choose a file"));
	QWhatsThis::add(mFileBrowseButton, i18n("Select a text or image file to display."));
	connect(mFileBrowseButton, SIGNAL(clicked()), SLOT(slotPickFile()));

	// Command type checkbox and edit box
	mCmdEdit = new CommandEdit(parent);
	frameLayout->addWidget(mCmdEdit);
	connect(mCmdEdit, SIGNAL(scriptToggled(bool)), SLOT(slotCmdScriptToggled(bool)));

	// Sound checkbox and file selector
	QHBoxLayout* hlayout = new QHBoxLayout(frameLayout);
	mSoundPicker = new SoundPicker(parent);
	mSoundPicker->setFixedSize(mSoundPicker->sizeHint());
	hlayout->addWidget(mSoundPicker);
	hlayout->addSpacing(2*spacingHint());
	hlayout->addStretch();

	// Font and colour choice button and sample text
	mFontColourButton = new FontColourButton(parent);
	mFontColourButton->setMaximumHeight(mFontColourButton->sizeHint().height());
	hlayout->addWidget(mFontColourButton);
	connect(mFontColourButton, SIGNAL(selected(const QColor&, const QColor&)), SLOT(setColours(const QColor&, const QColor&)));

	if (ShellProcess::authorised())    // don't display if shell commands not allowed (e.g. kiosk mode)
	{
		// Special actions button
		mSpecialActionsButton = new SpecialActionsButton(false, parent);
		mSpecialActionsButton->setFixedSize(mSpecialActionsButton->sizeHint());
		frameLayout->addWidget(mSpecialActionsButton, 0, (QApplication::reverseLayout() ? Qt::AlignLeft : Qt::AlignRight));
	}

	// Top-adjust the controls
	mFilePadding = new QHBox(parent);
	frameLayout->addWidget(mFilePadding);
	frameLayout->setStretchFactor(mFilePadding, 1);

	slotAlarmTypeChanged(tTEXT);
}

/******************************************************************************
* Create a reminder control.
*/
Reminder* EditDisplayAlarmDlg::createReminder(QWidget* parent, const char* name)
{
	static const QString reminderText = i18n("Enter how long in advance of the main alarm to display a reminder alarm.");
	return new Reminder(i18n("Check to additionally display a reminder in advance of the main alarm time(s)."),
	                    QString("%1\n\n%2").arg(reminderText).arg(TimeSpinBox::shiftWhatsThis()),
	                    true, true, parent, name);
}

/******************************************************************************
* Create an "acknowledgement confirmation required" checkbox.
*/
CheckBox* EditDisplayAlarmDlg::createConfirmAckCheckbox(QWidget* parent, const char* name)
{
	CheckBox* widget = new CheckBox(i18n_k_ConfirmAck(), parent, name);
	QWhatsThis::add(widget,
	      i18n("Check to be prompted for confirmation when you acknowledge the alarm."));
	return widget;
}

/******************************************************************************
* Initialise the dialog controls from the specified event.
*/
void EditDisplayAlarmDlg::type_initValues(const KAEvent* event)
{
	mKMailSerialNumber = 0;
	lateCancel()->showAutoClose(true);
	if (event)
	{
		if (mAlarmType == KAEvent::MESSAGE  &&  event->kmailSerialNumber()
		&&  AlarmText::checkIfEmail(event->cleanText()))
			mKMailSerialNumber = event->kmailSerialNumber();
		lateCancel()->setAutoClose(event->autoClose());
		if (event->defaultFont())
			mFontColourButton->setDefaultFont();
		else
			mFontColourButton->setFont(event->font());
		mFontColourButton->setBgColour(event->bgColour());
		mFontColourButton->setFgColour(event->fgColour());
		setColours(event->fgColour(), event->bgColour());
		mConfirmAck->setChecked(event->confirmAck());
		bool recurs = event->recurs();
		int reminderMins = event->reminder();
		if (!reminderMins  &&  event->reminderDeferral()  &&  !recurs)
		{
			reminderMins = event->reminderDeferral();
			mReminderDeferral = true;
		}
		if (!reminderMins  &&  event->reminderArchived()  &&  recurs)
		{
			reminderMins = event->reminderArchived();
			mReminderArchived = true;
		}
		reminder()->setMinutes(reminderMins, dateOnly());
		reminder()->setOnceOnly(event->reminderOnceOnly());
		reminder()->enableOnceOnly(recurs);
		if (mSpecialActionsButton)
			mSpecialActionsButton->setActions(event->preAction(), event->postAction(), event->cancelOnPreActionError());
		SoundPicker::Type soundType = event->speak()                ? SoundPicker::SPEAK
		                            : event->beep()                 ? SoundPicker::BEEP
		                            : !event->audioFile().isEmpty() ? SoundPicker::PLAY_FILE
		                            :                                 SoundPicker::NONE;
		mSoundPicker->set(soundType, event->audioFile(), event->soundVolume(),
		                  event->fadeVolume(), event->fadeSeconds(), event->repeatSound());
	}
	else
	{
		// Set the values to their defaults
		if (!ShellProcess::authorised())
		{
			// Don't allow shell commands in kiosk mode
			if (mSpecialActionsButton)
				mSpecialActionsButton->setEnabled(false);
		}
		lateCancel()->setAutoClose(Preferences::defaultAutoClose());
		mTypeCombo->setCurrentItem(0);
		mFontColourButton->setDefaultFont();
		mFontColourButton->setBgColour(Preferences::defaultBgColour());
		mFontColourButton->setFgColour(Preferences::defaultFgColour());
		setColours(Preferences::defaultFgColour(), Preferences::defaultBgColour());
		mConfirmAck->setChecked(Preferences::defaultConfirmAck());
		reminder()->setMinutes(0, false);
		reminder()->enableOnceOnly(isTimedRecurrence());   // must be called after mRecurrenceEdit is set up
		if (mSpecialActionsButton)
			mSpecialActionsButton->setActions(Preferences::defaultPreAction(), Preferences::defaultPostAction(), Preferences::defaultCancelOnPreActionError());
		mSoundPicker->set(Preferences::defaultSoundType(), Preferences::defaultSoundFile(),
		                  Preferences::defaultSoundVolume(), -1, 0, Preferences::defaultSoundRepeat());
	}
}

/******************************************************************************
* Called when the More/Less Options button is clicked.
* Show/hide the optional options.
*/
void EditDisplayAlarmDlg::type_showOptions(bool more)
{
	if (mSpecialActionsButton)
	{
		if (more)
		{
			switch (mTypeCombo->currentItem())
			{
				case tFILE:
				case tTEXT:
					mSpecialActionsButton->show();
					break;
				default:
					break;
			}
		}
		else
			mSpecialActionsButton->hide();
	}
}

/******************************************************************************
* Called when the font/colour button has been clicked.
* Set the colours in the message text entry control.
*/
void EditDisplayAlarmDlg::setColours(const QColor& fgColour, const QColor& bgColour)
{
	mTextMessageEdit->setPaper(bgColour);
	mTextMessageEdit->setPaletteForegroundColor(fgColour);
}

/******************************************************************************
* Set the dialog's action and the action's text.
*/
void EditDisplayAlarmDlg::setAction(KAEvent::Action action, const AlarmText& alarmText)
{
	int newItem;
	int oldItem = mTypeCombo->currentItem();
	QString text = alarmText.displayText();
	switch (action)
	{
		case KAEvent::MESSAGE:
			newItem = tTEXT;
			mTypeCombo->setCurrentItem(tTEXT);
			mTextMessageEdit->setText(text);
			mKMailSerialNumber = alarmText.isEmail() ? alarmText.kmailSerialNumber() : 0;
			break;
		case KAEvent::FILE:
			newItem = tFILE;
			mTypeCombo->setCurrentItem(tFILE);
			mFileMessageEdit->setText(text);
			break;
		case KAEvent::COMMAND:
			newItem = tCOMMAND;
			mTypeCombo->setCurrentItem(tCOMMAND);
			mCmdEdit->setText(alarmText);
			break;
		default:
			Q_ASSERT(0);
			return;
	}
	if (newItem != oldItem)
		slotAlarmTypeChanged(newItem);
}

/******************************************************************************
 * Set the read-only status of all non-template controls.
 */
void EditDisplayAlarmDlg::setReadOnly(bool readOnly)
{
	mTypeCombo->setReadOnly(readOnly);
	mTextMessageEdit->setReadOnly(readOnly);
	mFileMessageEdit->setReadOnly(readOnly);
	mCmdEdit->setReadOnly(readOnly);
	mFontColourButton->setReadOnly(readOnly);
	mSoundPicker->setReadOnly(readOnly);
	mConfirmAck->setReadOnly(readOnly);
	reminder()->setReadOnly(readOnly);
	if (mSpecialActionsButton)
		mSpecialActionsButton->setReadOnly(readOnly);
	if (readOnly)
		mFileBrowseButton->hide();
	else
		mFileBrowseButton->show();
	EditAlarmDlg::setReadOnly(readOnly);
}

/******************************************************************************
* Save the state of all controls.
*/
void EditDisplayAlarmDlg::saveState(const KAEvent* event)
{
	EditAlarmDlg::saveState(event);
	mSavedType        = mTypeCombo->currentItem();
	mSavedCmdScript   = mCmdEdit->isScript();
	mSavedSoundType   = mSoundPicker->sound();
	mSavedSoundFile   = mSoundPicker->file();
	mSavedSoundVolume = mSoundPicker->volume(mSavedSoundFadeVolume, mSavedSoundFadeSeconds);
	mSavedRepeatSound = mSoundPicker->repeat();
	mSavedConfirmAck  = mConfirmAck->isChecked();
	mSavedFont        = mFontColourButton->font();
	mSavedFgColour    = mFontColourButton->fgColour();
	mSavedBgColour    = mFontColourButton->bgColour();
	mSavedReminder    = reminder()->minutes();
	mSavedOnceOnly    = reminder()->isOnceOnly();
	mSavedAutoClose   = lateCancel()->isAutoClose();
	if (mSpecialActionsButton)
	{
		mSavedPreAction       = mSpecialActionsButton->preAction();
		mSavedPostAction      = mSpecialActionsButton->postAction();
		mSavedPreActionCancel = mSpecialActionsButton->cancelOnError();
	}
}

/******************************************************************************
* Check whether any of the controls has changed state since the dialog was
* first displayed.
* Reply = true if any controls have changed, or if it's a new event.
*       = false if no controls have changed.
*/
bool EditDisplayAlarmDlg::type_stateChanged() const
{
	if (mSavedType       != mTypeCombo->currentItem()
	||  mSavedCmdScript  != mCmdEdit->isScript()
	||  mSavedSoundType  != mSoundPicker->sound()
	||  mSavedConfirmAck != mConfirmAck->isChecked()
	||  mSavedFont       != mFontColourButton->font()
	||  mSavedFgColour   != mFontColourButton->fgColour()
	||  mSavedBgColour   != mFontColourButton->bgColour()
	||  mSavedReminder   != reminder()->minutes()
	||  mSavedOnceOnly   != reminder()->isOnceOnly()
	||  mSavedAutoClose  != lateCancel()->isAutoClose())
		return true;
	if (mSpecialActionsButton)
	{
		if (mSavedPreAction       != mSpecialActionsButton->preAction()
		||  mSavedPostAction      != mSpecialActionsButton->postAction()
		||  mSavedPreActionCancel != mSpecialActionsButton->cancelOnError())
			return true;
	}
	if (mSavedSoundType == SoundPicker::PLAY_FILE)
	{
		if (mSavedSoundFile != mSoundPicker->file())
			return true;
		if (!mSavedSoundFile.isEmpty())
		{
			float fadeVolume;
			int   fadeSecs;
			if (mSavedRepeatSound != mSoundPicker->repeat()
			||  mSavedSoundVolume != mSoundPicker->volume(fadeVolume, fadeSecs)
			||  mSavedSoundFadeVolume != fadeVolume
			||  mSavedSoundFadeSeconds != fadeSecs)
				return true;
		}
	}
	return false;
}

/******************************************************************************
* Extract the data in the dialog specific to the alarm type and set up a
* KAEvent from it.
*/
void EditDisplayAlarmDlg::type_setEvent(KAEvent& event, const KDateTime& dt, const QString& text, int lateCancel, bool trial)
{
	KAEvent::Action type;
	switch (mTypeCombo->currentItem())
	{
		case tFILE:     type = KAEvent::FILE; break;
		case tCOMMAND:  type = KAEvent::COMMAND; break;
		default:
		case tTEXT:     type = KAEvent::MESSAGE; break;
	}
	event.set(dt, text, mFontColourButton->bgColour(), mFontColourButton->fgColour(), mFontColourButton->font(),
	          type, lateCancel, getAlarmFlags());
	if (type == KAEvent::MESSAGE)
	{
		if (AlarmText::checkIfEmail(text))
			event.setKMailSerialNumber(mKMailSerialNumber);
	}
	float fadeVolume;
	int   fadeSecs;
	float volume = mSoundPicker->volume(fadeVolume, fadeSecs);
	event.setAudioFile(mSoundPicker->file(), volume, fadeVolume, fadeSecs);
	if (!trial)
		event.setReminder(reminder()->minutes(), reminder()->isOnceOnly());
	if (mSpecialActionsButton  &&  mSpecialActionsButton->isEnabled())
		event.setActions(mSpecialActionsButton->preAction(), mSpecialActionsButton->postAction(), mSpecialActionsButton->cancelOnError());
}

/******************************************************************************
* Get the currently specified alarm flag bits.
*/
int EditDisplayAlarmDlg::getAlarmFlags() const
{
	bool cmd = (mTypeCombo->currentItem() == tCOMMAND);
	return EditAlarmDlg::getAlarmFlags()
	     | (mSoundPicker->sound() == SoundPicker::BEEP  ? KAEvent::BEEP : 0)
	     | (mSoundPicker->sound() == SoundPicker::SPEAK ? KAEvent::SPEAK : 0)
	     | (mSoundPicker->repeat()                      ? KAEvent::REPEAT_SOUND : 0)
	     | (mConfirmAck->isChecked()                    ? KAEvent::CONFIRM_ACK : 0)
	     | (lateCancel()->isAutoClose()                 ? KAEvent::AUTO_CLOSE : 0)
	     | (mFontColourButton->defaultFont()            ? KAEvent::DEFAULT_FONT : 0)
	     | (cmd                                         ? KAEvent::DISPLAY_COMMAND : 0)
	     | (cmd && mCmdEdit->isScript()                 ? KAEvent::SCRIPT : 0);
}

/******************************************************************************
*  Called when one of the alarm display type combo box is changed, to display
*  the appropriate set of controls for that action type.
*/
void EditDisplayAlarmDlg::slotAlarmTypeChanged(int index)
{
	QWidget* focus = 0;
	switch (index)
	{
		case tTEXT:    // text message
			mFileBox->hide();
			mFilePadding->hide();
			mCmdEdit->hide();
			mTextMessageEdit->show();
			mSoundPicker->showSpeak(true);
			if (mSpecialActionsButton  &&  showingMore())
				mSpecialActionsButton->show();
			setButtonWhatsThis(Try, i18n("Display the alarm message now"));
			focus = mTextMessageEdit;
			break;
		case tFILE:    // file contents
			mTextMessageEdit->hide();
			mFileBox->show();
			mFilePadding->show();
			mCmdEdit->hide();
			mSoundPicker->showSpeak(false);
			if (mSpecialActionsButton  &&  showingMore())
				mSpecialActionsButton->show();
			setButtonWhatsThis(Try, i18n("Display the file now"));
			mFileMessageEdit->setNoSelect();
			focus = mFileMessageEdit;
			break;
		case tCOMMAND:    // command output
			mTextMessageEdit->hide();
			mFileBox->hide();
			slotCmdScriptToggled(mCmdEdit->isScript());  // show/hide mFilePadding
			mCmdEdit->show();
			mSoundPicker->showSpeak(true);
			if (mSpecialActionsButton)
				mSpecialActionsButton->hide();
			setButtonWhatsThis(Try, i18n("Display the command output now"));
			focus = mCmdEdit;

			break;
	}
	if (focus)
		focus->setFocus();
}

/******************************************************************************
* Called when the file browse button is pressed to select a file to display.
*/
void EditDisplayAlarmDlg::slotPickFile()
{
	static QString defaultDir;   // default directory for file browse button
	QString file = KAlarm::browseFile(i18n("Choose Text or Image File to Display"), defaultDir, mFileMessageEdit->text(),
		                          QString::null, KFile::ExistingOnly, this, "pickAlarmFile");
	mFileMessageEdit->setText(file);
}

/******************************************************************************
* Called when one of the command type radio buttons is clicked,
* to display the appropriate edit field.
*/
void EditDisplayAlarmDlg::slotCmdScriptToggled(bool on)
{
	if (on)
		mFilePadding->hide();
	else
		mFilePadding->show();
}

/******************************************************************************
*  Clean up the alarm text, and if it's a file, check whether it's valid.
*/
bool EditDisplayAlarmDlg::checkText(QString& result, bool showErrorMessage) const
{
	switch (mTypeCombo->currentItem())
	{
		case tTEXT:
			result = mTextMessageEdit->text();
			break;

		case tFILE:
		{
			QString alarmtext = mFileMessageEdit->text().stripWhiteSpace();
			// Convert any relative file path to absolute
			// (using home directory as the default)
			enum Err { NONE = 0, BLANK, NONEXISTENT, DIRECTORY, UNREADABLE, NOT_TEXT_IMAGE };
			Err err = NONE;
			KURL url;
			int i = alarmtext.find(QString::fromLatin1("/"));
			if (i > 0  &&  alarmtext[i - 1] == ':')
			{
				url = alarmtext;
				url.cleanPath();
				alarmtext = url.prettyURL();
				KIO::UDSEntry uds;
				if (!KIO::NetAccess::stat(url, uds, MainWindow::mainMainWindow()))
					err = NONEXISTENT;
				else
				{
					KFileItem fi(uds, url);
					if (fi.isDir())             err = DIRECTORY;
					else if (!fi.isReadable())  err = UNREADABLE;
				}
			}
			else if (alarmtext.isEmpty())
				err = BLANK;    // blank file name
			else
			{
				// It's a local file - convert to absolute path & check validity
				QFileInfo info(alarmtext);
				QDir::setCurrent(QDir::homeDirPath());
				alarmtext = info.absFilePath();
				url.setPath(alarmtext);
				alarmtext = QString::fromLatin1("file:") + alarmtext;
				if (!err)
				{
					if      (info.isDir())        err = DIRECTORY;
					else if (!info.exists())      err = NONEXISTENT;
					else if (!info.isReadable())  err = UNREADABLE;
				}
			}
			if (!err)
			{
				switch (KAlarm::fileType(KFileItem(KFileItem::Unknown, KFileItem::Unknown, url).mimeTypePtr()))
				{
					case KAlarm::TextFormatted:
					case KAlarm::TextPlain:
					case KAlarm::TextApplication:
					case KAlarm::Image:
						break;
					default:
						err = NOT_TEXT_IMAGE;
						break;
				}
			}
			if (err  &&  showErrorMessage)
			{
				mFileMessageEdit->setFocus();
				QString errmsg;
				switch (err)
				{
					case BLANK:
						KMessageBox::sorry(const_cast<EditDisplayAlarmDlg*>(this), i18n("Please select a file to display"));
						return false;
					case NONEXISTENT:     errmsg = i18n("%1\nnot found");  break;
					case DIRECTORY:       errmsg = i18n("%1\nis a folder");  break;
					case UNREADABLE:      errmsg = i18n("%1\nis not readable");  break;
					case NOT_TEXT_IMAGE:  errmsg = i18n("%1\nappears not to be a text or image file");  break;
					case NONE:
					default:
						break;
				}
				if (KMessageBox::warningContinueCancel(const_cast<EditDisplayAlarmDlg*>(this), errmsg.arg(alarmtext))
				    == KMessageBox::Cancel)
					return false;
			}
			result = alarmtext;
			break;
		}
		case tCOMMAND:
			result = mCmdEdit->text();
			break;
	}
	return true;
}


/*=============================================================================
= Class EditCommandAlarmDlg
= Dialog to edit command alarms.
=============================================================================*/

// Collect these widget labels together to ensure consistent wording and
// translations across different modules.
QString EditCommandAlarmDlg::i18n_EnterScript()        { return i18n("Enter a script"); }
QString EditCommandAlarmDlg::i18n_p_EnterScript()      { return i18n("Enter a scri&pt"); }
QString EditCommandAlarmDlg::i18n_ExecInTermWindow()   { return i18n("Execute in terminal window"); }
QString EditCommandAlarmDlg::i18n_w_ExecInTermWindow() { return i18n("Execute in terminal &window"); }
QString EditCommandAlarmDlg::i18n_u_ExecInTermWindow() { return i18n("Exec&ute in terminal window"); }


/******************************************************************************
* Constructor.
* Parameters:
*   Template = true to edit/create an alarm template
*            = false to edit/create an alarm.
*   event   != to initialise the dialog to show the specified event's data.
*/
EditCommandAlarmDlg::EditCommandAlarmDlg(bool Template, bool newAlarm, QWidget* parent, const char* name, GetResourceType getResource)
	: EditAlarmDlg(Template, KAEvent::COMMAND, parent, name, getResource)
{
	kdDebug(5950) << "EditCommandAlarmDlg::EditCommandAlarmDlg(new)" << endl;
	init(0, newAlarm);
}

EditCommandAlarmDlg::EditCommandAlarmDlg(bool Template, const KAEvent* event, bool newAlarm, QWidget* parent,
                                         const char* name, GetResourceType getResource, bool readOnly)
	: EditAlarmDlg(Template, event, parent, name, getResource, readOnly)
{
	kdDebug(5950) << "EditCommandAlarmDlg::EditCommandAlarmDlg(event.id())" << endl;
	init(event, newAlarm);
}

/******************************************************************************
* Return the window caption.
*/
QString EditCommandAlarmDlg::type_caption(bool newAlarm) const
{
	return isTemplate() ? (newAlarm ? i18n("New Command Alarm Template") : i18n("Edit Command Alarm Template"))
	                    : (newAlarm ? i18n("New Command Alarm") : i18n("Edit Command Alarm"));
}

/******************************************************************************
* Set up the command alarm dialog controls.
*/
void EditCommandAlarmDlg::type_init(QWidget* parent, QVBoxLayout* frameLayout)
{
	setButtonWhatsThis(Try, i18n("Execute the specified command now"));

	mCmdEdit = new CommandEdit(parent);
	connect(mCmdEdit, SIGNAL(scriptToggled(bool)), SLOT(slotCmdScriptToggled(bool)));
	frameLayout->addWidget(mCmdEdit, 0, Qt::AlignAuto);

	// What to do with command output

	mCmdOutputGroup = new ButtonGroup(i18n("Command Output"), parent);
	frameLayout->addWidget(mCmdOutputGroup);
	QBoxLayout* layout = new QVBoxLayout(mCmdOutputGroup, marginHint(), spacingHint());
	layout->addSpacing(fontMetrics().height() - marginHint() + spacingHint());

	// Execute in terminal window
	mCmdExecInTerm = new RadioButton(i18n_u_ExecInTermWindow(), mCmdOutputGroup, "execInTerm");
	mCmdExecInTerm->setFixedSize(mCmdExecInTerm->sizeHint());
	QWhatsThis::add(mCmdExecInTerm, i18n("Check to execute the command in a terminal window"));
	mCmdOutputGroup->insert(mCmdExecInTerm, Preferences::Log_Terminal);
	layout->addWidget(mCmdExecInTerm, 0, Qt::AlignAuto);

	// Log file name edit box
	QHBox* box = new QHBox(mCmdOutputGroup);
	(new QWidget(box))->setFixedWidth(mCmdExecInTerm->style().subRect(QStyle::SR_RadioButtonIndicator, mCmdExecInTerm).width());   // indent the edit box
//	(new QWidget(box))->setFixedWidth(mCmdExecInTerm->style().pixelMetric(QStyle::PM_ExclusiveIndicatorWidth));   // indent the edit box
	mCmdLogFileEdit = new LineEdit(LineEdit::Url, box);
	mCmdLogFileEdit->setAcceptDrops(true);
	QWhatsThis::add(mCmdLogFileEdit, i18n("Enter the name or path of the log file."));

	// Log file browse button.
	// The file browser dialogue is activated by the PickLogFileRadio class.
	QPushButton* browseButton = new QPushButton(box);
	browseButton->setPixmap(SmallIcon("fileopen"));
	browseButton->setFixedSize(browseButton->sizeHint());
	QToolTip::add(browseButton, i18n("Choose a file"));
	QWhatsThis::add(browseButton, i18n("Select a log file."));

	// Log output to file
	mCmdLogToFile = new PickLogFileRadio(browseButton, mCmdLogFileEdit, i18n("Lo&g to file"), mCmdOutputGroup, "cmdLog");
	mCmdLogToFile->setFixedSize(mCmdLogToFile->sizeHint());
	QWhatsThis::add(mCmdLogToFile,
	      i18n("Check to log the command output to a local file. The output will be appended to any existing contents of the file."));
	mCmdOutputGroup->insert(mCmdLogToFile, Preferences::Log_File);
	layout->addWidget(mCmdLogToFile, 0, Qt::AlignAuto);
	layout->addWidget(box);

	// Discard output
	mCmdDiscardOutput = new RadioButton(i18n("Discard"), mCmdOutputGroup, "cmdDiscard");
	mCmdDiscardOutput->setFixedSize(mCmdDiscardOutput->sizeHint());
	QWhatsThis::add(mCmdDiscardOutput, i18n("Check to discard command output."));
	mCmdOutputGroup->insert(mCmdDiscardOutput, Preferences::Log_Discard);
	layout->addWidget(mCmdDiscardOutput, 0, Qt::AlignAuto);

	// Top-adjust the controls
	mCmdPadding = new QHBox(parent);
	frameLayout->addWidget(mCmdPadding);
	frameLayout->setStretchFactor(mCmdPadding, 1);
}

/******************************************************************************
* Initialise the dialog controls from the specified event.
*/
void EditCommandAlarmDlg::type_initValues(const KAEvent* event)
{
	if (event)
	{
		// Set the values to those for the specified event
		RadioButton* logType = event->commandXterm()       ? mCmdExecInTerm
		                     : !event->logFile().isEmpty() ? mCmdLogToFile
		                     :                               mCmdDiscardOutput;
		if (logType == mCmdLogToFile)
			mCmdLogFileEdit->setText(event->logFile());    // set file name before setting radio button
		logType->setChecked(true);
	}
	else
	{
		// Set the values to their defaults
		mCmdEdit->setScript(Preferences::defaultCmdScript());
		mCmdLogFileEdit->setText(Preferences::defaultCmdLogFile());    // set file name before setting radio button
		mCmdOutputGroup->setButton(Preferences::defaultCmdLogType());
	}
	slotCmdScriptToggled(mCmdEdit->isScript());
}

/******************************************************************************
* Called when the More/Less Options button is clicked.
* Show/hide the optional options.
*/
void EditCommandAlarmDlg::type_showOptions(bool more)
{
	if (more)
		mCmdOutputGroup->show();
	else
		mCmdOutputGroup->hide();
}

/******************************************************************************
* Set the dialog's action and the action's text.
*/
void EditCommandAlarmDlg::setAction(KAEvent::Action action, const AlarmText& alarmText)
{
	Q_ASSERT(action == KAEvent::COMMAND);
	mCmdEdit->setText(alarmText);
}

/******************************************************************************
* Set the read-only status of all non-template controls.
*/
void EditCommandAlarmDlg::setReadOnly(bool readOnly)
{
	if (!isTemplate()  &&  !ShellProcess::authorised())
		readOnly = true;     // don't allow editing of existing command alarms in kiosk mode
	mCmdEdit->setReadOnly(readOnly);
	mCmdExecInTerm->setReadOnly(readOnly);
	mCmdLogToFile->setReadOnly(readOnly);
	mCmdDiscardOutput->setReadOnly(readOnly);
	EditAlarmDlg::setReadOnly(readOnly);
}

/******************************************************************************
* Save the state of all controls.
*/
void EditCommandAlarmDlg::saveState(const KAEvent* event)
{
	EditAlarmDlg::saveState(event);
	mSavedCmdScript      = mCmdEdit->isScript();
	mSavedCmdOutputRadio = mCmdOutputGroup->selected();
	mSavedCmdLogFile     = mCmdLogFileEdit->text();
}

/******************************************************************************
* Check whether any of the controls has changed state since the dialog was
* first displayed.
* Reply = true if any controls have changed, or if it's a new event.
*       = false if no controls have changed.
*/
bool EditCommandAlarmDlg::type_stateChanged() const
{
	if (mSavedCmdScript      != mCmdEdit->isScript()
	||  mSavedCmdOutputRadio != mCmdOutputGroup->selected())
		return true;
	if (mCmdOutputGroup->selected() == mSavedCmdOutputRadio)
	{
		if (mSavedCmdLogFile != mCmdLogFileEdit->text())
			return true;
	}
	return false;
}

/******************************************************************************
* Extract the data in the dialog specific to the alarm type and set up a
* KAEvent from it.
*/
void EditCommandAlarmDlg::type_setEvent(KAEvent& event, const KDateTime& dt, const QString& text, int lateCancel, bool trial)
{
	Q_UNUSED(trial);
	event.set(dt, text, QColor(), QColor(), QFont(), KAEvent::COMMAND, lateCancel, getAlarmFlags());
	if (mCmdOutputGroup->selected() == mCmdLogToFile)
		event.setLogFile(mCmdLogFileEdit->text());
}

/******************************************************************************
* Get the currently specified alarm flag bits.
*/
int EditCommandAlarmDlg::getAlarmFlags() const
{
	return EditAlarmDlg::getAlarmFlags()
	     | (mCmdEdit->isScript()                          ? KAEvent::SCRIPT : 0)
	     | (mCmdOutputGroup->selected() == mCmdExecInTerm ? KAEvent::EXEC_IN_XTERM : 0);
}

/******************************************************************************
* Validate and convert command alarm data.
*/
bool EditCommandAlarmDlg::type_validate(bool trial)
{
	Q_UNUSED(trial);
	if (mCmdOutputGroup->selected() == mCmdLogToFile)
	{
		// Validate the log file name
		QString file = mCmdLogFileEdit->text();
		QFileInfo info(file);
		QDir::setCurrent(QDir::homeDirPath());
		bool err = file.isEmpty()  ||  info.isDir();
		if (!err)
		{
			if (info.exists())
			{
				err = !info.isWritable();
			}
			else
			{
				QFileInfo dirinfo(info.dirPath(true));    // get absolute directory path
				err = (!dirinfo.isDir()  ||  !dirinfo.isWritable());
			}
		}
		if (err)
		{
			mCmdLogFileEdit->setFocus();
			KMessageBox::sorry(this, i18n("Log file must be the name or path of a local file, with write permission."));
			return false;
		}
		// Convert the log file to an absolute path
		mCmdLogFileEdit->setText(info.absFilePath());
	}
	return true;
}

/******************************************************************************
* Tell the user the result of the Try action.
*/
void EditCommandAlarmDlg::type_trySuccessMessage(ShellProcess* proc, const QString& text)
{
	if (mCmdOutputGroup->selected() != mCmdExecInTerm)
	{
		theApp()->commandMessage(proc, this);
		KMessageBox::information(this, i18n("Command executed:\n%1").arg(text));
		theApp()->commandMessage(proc, 0);
	}
}

/******************************************************************************
* Called when one of the command type radio buttons is clicked,
* to display the appropriate edit field.
*/
void EditCommandAlarmDlg::slotCmdScriptToggled(bool on)
{
	if (on)
		mCmdPadding->hide();
	else
		mCmdPadding->show();
}

/******************************************************************************
* Clean up the alarm text.
*/
bool EditCommandAlarmDlg::checkText(QString& result, bool showErrorMessage) const
{
	Q_UNUSED(showErrorMessage);
	result = mCmdEdit->text();
	return true;
}


/*=============================================================================
= Class EditEmailAlarmDlg
= Dialog to edit email alarms.
=============================================================================*/

// Collect these widget labels together to ensure consistent wording and
// translations across different modules.
QString EditEmailAlarmDlg::i18n_EmailFrom()          { return i18n("'From' email address", "From:"); }
QString EditEmailAlarmDlg::i18n_f_EmailFrom()        { return i18n("'From' email address", "&From:"); }
QString EditEmailAlarmDlg::i18n_EmailTo()            { return i18n("Email addressee", "To:"); }
QString EditEmailAlarmDlg::i18n_EmailSubject()       { return i18n("Email subject", "Subject:"); }
QString EditEmailAlarmDlg::i18n_j_EmailSubject()     { return i18n("Email subject", "Sub&ject:"); }
QString EditEmailAlarmDlg::i18n_CopyEmailToSelf()    { return i18n("Copy email to self"); }
QString EditEmailAlarmDlg::i18n_e_CopyEmailToSelf()  { return i18n("Copy &email to self"); }
QString EditEmailAlarmDlg::i18n_s_CopyEmailToSelf()  { return i18n("Copy email to &self"); }


/******************************************************************************
* Constructor.
* Parameters:
*   Template = true to edit/create an alarm template
*            = false to edit/create an alarm.
*   event   != to initialise the dialog to show the specified event's data.
*/
EditEmailAlarmDlg::EditEmailAlarmDlg(bool Template, bool newAlarm, QWidget* parent, const char* name, GetResourceType getResource)
	: EditAlarmDlg(Template, KAEvent::EMAIL, parent, name, getResource),
	  mEmailRemoveButton(0)
{
	kdDebug(5950) << "EditEmailAlarmDlg::EditEmailAlarmDlg(new)" << endl;
	init(0, newAlarm);
}

EditEmailAlarmDlg::EditEmailAlarmDlg(bool Template, const KAEvent* event, bool newAlarm, QWidget* parent,
                                     const char* name, GetResourceType getResource, bool readOnly)
	: EditAlarmDlg(Template, event, parent, name, getResource, readOnly),
	  mEmailRemoveButton(0)
{
	kdDebug(5950) << "EditEmailAlarmDlg::EditEmailAlarmDlg(event.id())" << endl;
	init(event, newAlarm);
}

/******************************************************************************
* Return the window caption.
*/
QString EditEmailAlarmDlg::type_caption(bool newAlarm) const
{
	return isTemplate() ? (newAlarm ? i18n("New Email Alarm Template") : i18n("Edit Email Alarm Template"))
	                    : (newAlarm ? i18n("New Email Alarm") : i18n("Edit Email Alarm"));
}

/******************************************************************************
* Set up the email alarm dialog controls.
*/
void EditEmailAlarmDlg::type_init(QWidget* parent, QVBoxLayout* frameLayout)
{
	setButtonWhatsThis(Try, i18n("Send the email to the specified addressees now"));

	QGridLayout* grid = new QGridLayout(frameLayout, 3, 3, spacingHint());
	grid->setColStretch(1, 1);

	mEmailFromList = 0;
	if (Preferences::emailFrom() == Preferences::MAIL_FROM_KMAIL)
	{
		// Email sender identity
		QLabel* label = new QLabel(i18n_EmailFrom(), parent);
		label->setFixedSize(label->sizeHint());
		grid->addWidget(label, 0, 0);

		mEmailFromList = new EmailIdCombo(KAMail::identityManager(), parent);
		mEmailFromList->setMinimumSize(mEmailFromList->sizeHint());
		label->setBuddy(mEmailFromList);
		QWhatsThis::add(mEmailFromList,
		      i18n("Your email identity, used to identify you as the sender when sending email alarms."));
		grid->addMultiCellWidget(mEmailFromList, 0, 0, 1, 2);
	}

	// Email recipients
	QLabel* label = new QLabel(i18n_EmailTo(), parent);
	label->setFixedSize(label->sizeHint());
	grid->addWidget(label, 1, 0);

	mEmailToEdit = new LineEdit(LineEdit::Emails, parent);
	mEmailToEdit->setMinimumSize(mEmailToEdit->sizeHint());
	QWhatsThis::add(mEmailToEdit,
	      i18n("Enter the addresses of the email recipients. Separate multiple addresses by "
	           "commas or semicolons."));
	grid->addWidget(mEmailToEdit, 1, 1);

	mEmailAddressButton = new QPushButton(parent);
	mEmailAddressButton->setPixmap(SmallIcon("contents"));
	mEmailAddressButton->setFixedSize(mEmailAddressButton->sizeHint());
	connect(mEmailAddressButton, SIGNAL(clicked()), SLOT(openAddressBook()));
	QToolTip::add(mEmailAddressButton, i18n("Open address book"));
	QWhatsThis::add(mEmailAddressButton, i18n("Select email addresses from your address book."));
	grid->addWidget(mEmailAddressButton, 1, 2);

	// Email subject
	label = new QLabel(i18n_j_EmailSubject(), parent);
	label->setFixedSize(label->sizeHint());
	grid->addWidget(label, 2, 0);

	mEmailSubjectEdit = new LineEdit(parent);
	mEmailSubjectEdit->setMinimumSize(mEmailSubjectEdit->sizeHint());
	label->setBuddy(mEmailSubjectEdit);
	QWhatsThis::add(mEmailSubjectEdit, i18n("Enter the email subject."));
	grid->addMultiCellWidget(mEmailSubjectEdit, 2, 2, 1, 2);

	// Email body
	mEmailMessageEdit = new TextEdit(parent);
	QWhatsThis::add(mEmailMessageEdit, i18n("Enter the email message."));
	frameLayout->addWidget(mEmailMessageEdit);

	// Email attachments
	grid = new QGridLayout(frameLayout, 2, 3, spacingHint());
	label = new QLabel(i18n("Attachment&s:"), parent);
	label->setFixedSize(label->sizeHint());
	grid->addWidget(label, 0, 0);

	mEmailAttachList = new QComboBox(true, parent);
	mEmailAttachList->setMinimumSize(mEmailAttachList->sizeHint());
	mEmailAttachList->lineEdit()->setReadOnly(true);
QListBox* list = mEmailAttachList->listBox();
QRect rect = list->geometry();
list->setGeometry(rect.left() - 50, rect.top(), rect.width(), rect.height());
	label->setBuddy(mEmailAttachList);
	QWhatsThis::add(mEmailAttachList,
	      i18n("Files to send as attachments to the email."));
	grid->addWidget(mEmailAttachList, 0, 1);
	grid->setColStretch(1, 1);

	mEmailAddAttachButton = new QPushButton(i18n("Add..."), parent);
	connect(mEmailAddAttachButton, SIGNAL(clicked()), SLOT(slotAddAttachment()));
	QWhatsThis::add(mEmailAddAttachButton, i18n("Add an attachment to the email."));
	grid->addWidget(mEmailAddAttachButton, 0, 2);

	mEmailRemoveButton = new QPushButton(i18n("Remo&ve"), parent);
	connect(mEmailRemoveButton, SIGNAL(clicked()), SLOT(slotRemoveAttachment()));
	QWhatsThis::add(mEmailRemoveButton, i18n("Remove the highlighted attachment from the email."));
	grid->addWidget(mEmailRemoveButton, 1, 2);

	// BCC email to sender
	mEmailBcc = new CheckBox(i18n_s_CopyEmailToSelf(), parent);
	mEmailBcc->setFixedSize(mEmailBcc->sizeHint());
	QWhatsThis::add(mEmailBcc,
	      i18n("If checked, the email will be blind copied to you."));
	grid->addMultiCellWidget(mEmailBcc, 1, 1, 0, 1, Qt::AlignAuto);
}

/******************************************************************************
* Initialise the dialog controls from the specified event.
*/
void EditEmailAlarmDlg::type_initValues(const KAEvent* event)
{
	if (event)
	{
		// Set the values to those for the specified event
		mEmailAttachList->insertStringList(event->emailAttachments());
		mEmailToEdit->setText(event->emailAddresses(", "));
		mEmailSubjectEdit->setText(event->emailSubject());
		mEmailBcc->setChecked(event->emailBcc());
		if (mEmailFromList)
			mEmailFromList->setCurrentIdentity(event->emailFromId());
	}
	else
	{
		// Set the values to their defaults
		mEmailBcc->setChecked(Preferences::defaultEmailBcc());
	}
	bool enable = !!mEmailAttachList->count();
	mEmailAttachList->setEnabled(enable);
	if (mEmailRemoveButton)
		mEmailRemoveButton->setEnabled(enable);
}

/******************************************************************************
* Set the dialog's action and the action's text.
*/
void EditEmailAlarmDlg::setAction(KAEvent::Action action, const AlarmText& alarmText)
{
	Q_ASSERT(action == KAEvent::EMAIL);
	if (alarmText.isEmail())
	{
		mEmailToEdit->setText(alarmText.to());
		mEmailSubjectEdit->setText(alarmText.subject());
		mEmailMessageEdit->setText(alarmText.body());
	}
	else
		mEmailMessageEdit->setText(alarmText.displayText());
}

/******************************************************************************
* Set the read-only status of all non-template controls.
*/
void EditEmailAlarmDlg::setReadOnly(bool readOnly)
{
	mEmailToEdit->setReadOnly(readOnly);
	mEmailSubjectEdit->setReadOnly(readOnly);
	mEmailMessageEdit->setReadOnly(readOnly);
	mEmailBcc->setReadOnly(readOnly);
	if (mEmailFromList)
		mEmailFromList->setReadOnly(readOnly);
	if (readOnly)
	{
		mEmailAddressButton->hide();
		mEmailAddAttachButton->hide();
		mEmailRemoveButton->hide();
	}
	else
	{
		mEmailAddressButton->show();
		mEmailAddAttachButton->show();
		mEmailRemoveButton->show();
	}
	EditAlarmDlg::setReadOnly(readOnly);
}

/******************************************************************************
* Save the state of all controls.
*/
void EditEmailAlarmDlg::saveState(const KAEvent* event)
{
	EditAlarmDlg::saveState(event);
	if (mEmailFromList)
		mSavedEmailFrom = mEmailFromList->currentIdentityName();
	mSavedEmailTo      = mEmailToEdit->text();
	mSavedEmailSubject = mEmailSubjectEdit->text();
	mSavedEmailAttach.clear();
	for (int i = 0;  i < mEmailAttachList->count();  ++i)
		mSavedEmailAttach += mEmailAttachList->text(i);
	mSavedEmailBcc     = mEmailBcc->isChecked();
}

/******************************************************************************
* Check whether any of the controls has changed state since the dialog was
* first displayed.
* Reply = true if any controls have changed, or if it's a new event.
*       = false if no controls have changed.
*/
bool EditEmailAlarmDlg::type_stateChanged() const
{
	QStringList emailAttach;
	for (int i = 0;  i < mEmailAttachList->count();  ++i)
		emailAttach += mEmailAttachList->text(i);
	if ((mEmailFromList  &&  mSavedEmailFrom != mEmailFromList->currentIdentityName())
	||  mSavedEmailTo      != mEmailToEdit->text()
	||  mSavedEmailSubject != mEmailSubjectEdit->text()
	||  mSavedEmailAttach  != emailAttach
	||  mSavedEmailBcc     != mEmailBcc->isChecked())
		return true;
	return false;
}

/******************************************************************************
* Extract the data in the dialog specific to the alarm type and set up a
* KAEvent from it.
*/
void EditEmailAlarmDlg::type_setEvent(KAEvent& event, const KDateTime& dt, const QString& text, int lateCancel, bool trial)
{
	Q_UNUSED(trial);
	event.set(dt, text, QColor(), QColor(), QFont(), KAEvent::EMAIL, lateCancel, getAlarmFlags());
	uint from = mEmailFromList ? mEmailFromList->currentIdentity() : 0;
	event.setEmail(from, mEmailAddresses, mEmailSubjectEdit->text(), mEmailAttachments);
}

/******************************************************************************
* Get the currently specified alarm flag bits.
*/
int EditEmailAlarmDlg::getAlarmFlags() const
{
	return EditAlarmDlg::getAlarmFlags()
	     | (mEmailBcc->isChecked() ? KAEvent::EMAIL_BCC : 0);
}

/******************************************************************************
* Convert the email addresses to a list, and validate them. Convert the email
* attachments to a list.
*/
bool EditEmailAlarmDlg::type_validate(bool trial)
{
	QString addrs = mEmailToEdit->text();
	if (addrs.isEmpty())
		mEmailAddresses.clear();
	else
	{
		QString bad = KAMail::convertAddresses(addrs, mEmailAddresses);
		if (!bad.isEmpty())
		{
			mEmailToEdit->setFocus();
			KMessageBox::error(this, i18n("Invalid email address:\n%1").arg(bad));
			return false;
		}
	}
	if (mEmailAddresses.isEmpty())
	{
		mEmailToEdit->setFocus();
		KMessageBox::error(this, i18n("No email address specified"));
		return false;
	}

	mEmailAttachments.clear();
	for (int i = 0;  i < mEmailAttachList->count();  ++i)
	{
		QString att = mEmailAttachList->text(i);
		switch (KAMail::checkAttachment(att))
		{
			case 1:
				mEmailAttachments.append(att);
				break;
			case 0:
				break;      // empty
			case -1:
				mEmailAttachList->setFocus();
				KMessageBox::error(this, i18n("Invalid email attachment:\n%1").arg(att));
				return false;
		}
	}
	if (trial  &&  KMessageBox::warningContinueCancel(this, i18n("Do you really want to send the email now to the specified recipient(s)?"),
		                                          i18n("Confirm Email"), i18n("&Send")) != KMessageBox::Continue)
		return false;
	return true;
}

/******************************************************************************
* Tell the user the result of the Try action.
*/
void EditEmailAlarmDlg::type_trySuccessMessage(ShellProcess*, const QString&)
{
	QString bcc;
	if (mEmailBcc->isChecked())
		bcc = i18n("\nBcc: %1").arg(Preferences::emailBccAddress());
	KMessageBox::information(this, i18n("Email sent to:\n%1%2").arg(mEmailAddresses.join("\n")).arg(bcc));
}

/******************************************************************************
 * Get a selection from the Address Book.
 */
void EditEmailAlarmDlg::openAddressBook()
{
	KABC::Addressee a = KABC::AddresseeDialog::getAddressee(this);
	if (a.isEmpty())
		return;
	Person person(a.realName(), a.preferredEmail());
	QString addrs = mEmailToEdit->text().stripWhiteSpace();
	if (!addrs.isEmpty())
		addrs += ", ";
	addrs += person.fullName();
	mEmailToEdit->setText(addrs);
}

/******************************************************************************
 * Select a file to attach to the email.
 */
void EditEmailAlarmDlg::slotAddAttachment()
{
	QString url = KAlarm::browseFile(i18n("Choose File to Attach"), mAttachDefaultDir, QString::null,
	                                 QString::null, KFile::ExistingOnly, this, "pickAttachFile");
	if (!url.isEmpty())
	{
		mEmailAttachList->insertItem(url);
		mEmailAttachList->setCurrentItem(mEmailAttachList->count() - 1);   // select the new item
		mEmailRemoveButton->setEnabled(true);
		mEmailAttachList->setEnabled(true);
	}
}

/******************************************************************************
 * Remove the currently selected attachment from the email.
 */
void EditEmailAlarmDlg::slotRemoveAttachment()
{
	int item = mEmailAttachList->currentItem();
	mEmailAttachList->removeItem(item);
	int count = mEmailAttachList->count();
	if (item >= count)
		mEmailAttachList->setCurrentItem(count - 1);
	if (!count)
	{
		mEmailRemoveButton->setEnabled(false);
		mEmailAttachList->setEnabled(false);
	}
}

/******************************************************************************
*  Clean up the alarm text.
*/
bool EditEmailAlarmDlg::checkText(QString& result, bool showErrorMessage) const
{
	Q_UNUSED(showErrorMessage);
	result = mEmailMessageEdit->text();
	return true;
}


/*=============================================================================
= Class CommandEdit
= A widget to allow entry of a command or a command script.
=============================================================================*/
CommandEdit::CommandEdit(QWidget* parent, const char* name)
	: QWidget(parent, name)
{
	QVBoxLayout* vlayout = new QVBoxLayout(this, 0, KDialog::spacingHint());
	mTypeScript = new CheckBox(EditCommandAlarmDlg::i18n_p_EnterScript(), this);
	mTypeScript->setFixedSize(mTypeScript->sizeHint());
	connect(mTypeScript, SIGNAL(toggled(bool)), SLOT(slotCmdScriptToggled(bool)));
	QWhatsThis::add(mTypeScript, i18n("Check to enter the contents of a script instead of a shell command line"));
	vlayout->addWidget(mTypeScript, 0, Qt::AlignAuto);

	mCommandEdit = new LineEdit(LineEdit::Url, this);
	QWhatsThis::add(mCommandEdit, i18n("Enter a shell command to execute."));
	vlayout->addWidget(mCommandEdit);

	mScriptEdit = new TextEdit(this);
	QWhatsThis::add(mScriptEdit, i18n("Enter the contents of a script to execute"));
	vlayout->addWidget(mScriptEdit);

	slotCmdScriptToggled(mTypeScript->isChecked());
}

/******************************************************************************
* Initialise the widget controls from the specified event.
*/
void CommandEdit::setScript(bool script)
{
	mTypeScript->setChecked(script);
}

bool CommandEdit::isScript() const
{
	return mTypeScript->isChecked();
}

/******************************************************************************
* Set the widget's text.
*/
void CommandEdit::setText(const AlarmText& alarmText)
{
	QString text = alarmText.displayText();
	bool script = alarmText.isScript();
	mTypeScript->setChecked(script);
	if (script)
		mScriptEdit->setText(text);
	else
		mCommandEdit->setText(text);
}

/******************************************************************************
* Return the widget's text.
*/
QString CommandEdit::text() const
{
	QString result;
	if (mTypeScript->isChecked())
		result = mScriptEdit->text();
	else
		result = mCommandEdit->text();
	return result.stripWhiteSpace();
}

/******************************************************************************
* Set the read-only status of all controls.
*/
void CommandEdit::setReadOnly(bool readOnly)
{
	mTypeScript->setReadOnly(readOnly);
	mCommandEdit->setReadOnly(readOnly);
	mScriptEdit->setReadOnly(readOnly);
}

/******************************************************************************
* Called when one of the command type radio buttons is clicked,
* to display the appropriate edit field.
*/
void CommandEdit::slotCmdScriptToggled(bool on)
{
	if (on)
	{
		mCommandEdit->hide();
		mScriptEdit->show();
		mScriptEdit->setFocus();
	}
	else
	{
		mScriptEdit->hide();
		mCommandEdit->show();
		mCommandEdit->setFocus();
	}
	emit scriptToggled(on);
}

/******************************************************************************
* Returns the minimum size of the widget.
*/
QSize CommandEdit::minimumSizeHint() const
{
	QSize t(mTypeScript->minimumSizeHint());
	QSize s(mCommandEdit->minimumSizeHint().expandedTo(mScriptEdit->minimumSizeHint()));
	s.setHeight(s.height() + KDialog::spacingHint() + t.height());
	if (s.width() < t.width())
		s.setWidth(t.width());
	return s;
}


/*=============================================================================
= Class TextEdit
= A text edit field with a minimum height of 3 text lines.
= Provides KDE 2 compatibility.
=============================================================================*/
TextEdit::TextEdit(QWidget* parent, const char* name)
	: KTextEdit(parent, name)
{
	QSize tsize = sizeHint();
	tsize.setHeight(fontMetrics().lineSpacing()*13/4 + 2*frameWidth());
	setMinimumSize(tsize);
}

void TextEdit::dragEnterEvent(QDragEnterEvent* e)
{
	if (KCal::ICalDrag::canDecode(e))
		e->accept(false);   // don't accept "text/calendar" objects
	KTextEdit::dragEnterEvent(e);
}
