/*
 *  eventlistviewbase.h  -  base classes for widget showing list of events
 *  Program:  kalarm
 *  Copyright © 2004-2008 by David Jarvie <software@astrojar.org.uk>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#ifndef EVENTLISTVIEWBASE_H
#define EVENTLISTVIEWBASE_H

#include "kalarm.h"

#include <qvaluelist.h>
#include <klistview.h>

#include "alarmevent.h"
#include "alarmresources.h"

class QPixmap;
class EventListViewItemBase;
class Find;


class EventListViewBase : public KListView
{
		Q_OBJECT
	public:
		typedef QValueList<EventListViewBase*>              InstanceList;
		typedef QValueListIterator<EventListViewBase*>      InstanceListIterator;
		typedef QValueListConstIterator<EventListViewBase*> InstanceListConstIterator;

		explicit EventListViewBase(KCalEvent::Status, QWidget* parent = 0, const char* name = 0);
		virtual ~EventListViewBase()  { }
		static void            refresh(const InstanceList&);
		void                   repaint(AlarmResource*);
		EventListViewItemBase* getEntry(const QString& eventID) const;
		void                   addEvent(KAEvent* e)   { addEvent(e, instances(), this); }
		void                   modifyEvent(KAEvent* e) { modifyEvent(e->id(), e, instances(), this); }
		void                   modifyEvent(const QString& oldEventID, KAEvent* newEvent)
		                                              { modifyEvent(oldEventID, newEvent, instances(), this); }
		void                   deleteEvent(const QString& eventID)  { deleteEvent(eventID, instances()); }
		static void            addEvent(KAEvent*, const InstanceList&, EventListViewBase* selectionView);
		static void            modifyEvent(KAEvent* e, const InstanceList& list, EventListViewBase* selectionView)
		                                              { modifyEvent(e->id(), e, list, selectionView); }
		static void            modifyEvent(const QString& oldEventID, KAEvent* newEvent, const InstanceList&, EventListViewBase* selectionView);
		static void            deleteEvent(const QString& eventID, const InstanceList&);
		static void            undeleteEvent(const QString& oldEventID, KAEvent* event, const InstanceList& list, EventListViewBase* selectionView)
		                                              { modifyEvent(oldEventID, event, list, selectionView); }
		static void            deleteResource(AlarmResource*, const InstanceList&);
		void                   deleteResource(AlarmResource*, bool setSize = true);
		void                   setCurrentStatus(KCalEvent::Status s)  { mCurrentStatus = static_cast<KCalEvent::Status>(s & mStatus); }
		void                   resizeLastColumn();
		int                    itemHeight();
		EventListViewItemBase* currentItem() const    { return (EventListViewItemBase*)KListView::currentItem(); }
		EventListViewItemBase* firstChild() const     { return (EventListViewItemBase*)KListView::firstChild(); }
		void                   select(const QString& eventID);
		bool                   anySelected() const;    // are any items selected?
		KAEvent*               selectedEvent() const;
		KAEvent::List          selectedEvents() const;
		EventListViewItemBase* selectedItem() const;
		QValueList<EventListViewItemBase*> selectedItems() const;
		int                    selectedCount() const;
		int                    lastColumn() const     { return mLastColumn; }
		virtual QString        whatsThisText(int column) const = 0;
		virtual InstanceList   instances() = 0; // return all instances

	public slots:
		void                   refresh();
		virtual void           slotFind();
		virtual void           slotFindNext()         { findNext(true); }
		virtual void           slotFindPrev()         { findNext(false); }
		virtual void           slotSelectAll();
		virtual void           slotDeselect();

	signals:
		void                   itemDeleted();
		void                   findActive(bool);

	protected:
		virtual void           populate() = 0;         // populate the list with all desired events
		virtual EventListViewItemBase* createItem(KAEvent*) = 0;   // only used by default addEntry() method
		virtual bool           shouldShowEvent(const KAEvent*) const  { return true; }
		EventListViewItemBase* addEntry(KAEvent*, bool setSize = false, bool reselect = false);
		EventListViewItemBase* addEntry(EventListViewItemBase*, bool setSize, bool reselect);
		EventListViewItemBase* updateEntry(EventListViewItemBase*, KAEvent* newEvent, bool setSize = false, bool reselect = false);
		void                   deleteEntry(EventListViewItemBase*, bool setSize = false);
		void                   addLastColumn(const QString& title);
		virtual void           showEvent(QShowEvent*);
		virtual void           resizeEvent(QResizeEvent*);

	private slots:
		void                   slotResourceLoaded(AlarmResource*, bool active);
		void                   slotResourceStatusChanged(AlarmResource*, AlarmResources::Change);

	private:
		void                   findNext(bool forward);
		void                   emitItemDeleted();

		Find*                  mFind;                 // alarm search object
		int                    mLastColumn;           // index to last column
		int                    mLastColumnHeaderWidth;
		KCalEvent::Status      mStatus;         // types of events contained in this list
		KCalEvent::Status      mCurrentStatus;  // types of events currently displayed (subset of mStatus)
};


class EventListViewItemBase : public QListViewItem
{
	public:
		EventListViewItemBase(EventListViewBase* parent, KAEvent*);
		KAEvent*               event() const             { return mEvent; }
		AlarmResource*         resource() const          { return mResource; }
		QPixmap*               eventIcon() const;
		int                    lastColumnWidth() const   { return mLastColumnWidth; }
		EventListViewItemBase* nextSibling() const       { return (EventListViewItemBase*)QListViewItem::nextSibling(); }
		static int             iconWidth();

	protected:
		void                   setLastColumnText();
		virtual QString        lastColumnText() const = 0;   // get the text to display in the last column

		KAEvent*               mEvent;            // the event for this item

	private:
		static QPixmap*        mTextIcon;
		static QPixmap*        mFileIcon;
		static QPixmap*        mCommandIcon;
		static QPixmap*        mEmailIcon;
		static int             mIconWidth;        // maximum width of any icon

		AlarmResource*         mResource;         // the resource for this item
		int                    mLastColumnWidth;  // width required to display message column
};

#endif // EVENTLISTVIEWBASE_H

