/*
 *  functions.cpp  -  miscellaneous functions
 *  Program:  kalarm
 *  Copyright © 2001-2008 by David Jarvie <software@astrojar.org.uk>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "kalarm.h"
#include "functions.h"

#include "alarmcalendar.h"
#include "alarmevent.h"
#include "alarmlistview.h"
#include "alarmresources.h"
#include "editdlg.h"
#include "kalarmapp.h"
#include "kamail.h"
#include "mainwindow.h"
#include "messagewin.h"
#include "preferences.h"
#include "shellprocess.h"
#include "templatelistview.h"
#include "templatemenuaction.h"

#include <qdeepcopy.h>
#include <qdir.h>
#include <qregexp.h>

#include <kconfig.h>
#include <kaction.h>
#include <kglobal.h>
#include <klocale.h>
#include <kstandarddirs.h>
#include <ksystemtimezone.h>
#include <kstdguiitem.h>
#include <kstdaccel.h>
#include <kmessagebox.h>
#include <kfiledialog.h>
#include <dcopclient.h>
#include <kdebug.h>

#include <libkcal/event.h>
#include <libkcal/icalformat.h>
#include <libkpimidentities/identitymanager.h>
#include <libkpimidentities/identity.h>
#include <libkcal/person.h>
#include <libkholidays/kholidays.h>


namespace
{
bool          refreshAlarmsQueued = false;
QCString      korganizerName    = "korganizer";
QString       korgStartError;
const char*   KORG_DCOP_OBJECT  = "KOrganizerIface";
const char*   KORG_DCOP_WINDOW  = "KOrganizer MainWindow";
const char*   KMAIL_DCOP_WINDOW = "kmail-mainwindow#1";
const QString KORGANIZER_UID    = QString::fromLatin1("-korg");

const char*   ALARM_OPTS_FILE       = "alarmopts";
const char*   DONT_SHOW_ERRORS_GROUP = "DontShowErrors";

bool sendToKOrganizer(const KAEvent*);
bool deleteFromKOrganizer(const QString& eventID);
QString uidKOrganizer(const QString& eventID);
inline bool runKOrganizer()   { return KAlarm::runProgram("korganizer", KORG_DCOP_WINDOW, korganizerName, korgStartError); }
}


namespace KAlarm
{

void doEditNewAlarm(EditAlarmDlg*, AlarmListView*);


/******************************************************************************
*  Display a main window with the specified event selected.
*/
MainWindow* displayMainWindowSelected(const QString& eventID)
{
	MainWindow* win = MainWindow::firstWindow();
	if (!win)
	{
		if (theApp()->checkCalendar())    // ensure calendar is open
		{
			win = MainWindow::create();
			win->show();
		}
	}
	else
	{
		// There is already a main window, so make it the active window
		bool visible = win->isVisible();
		if (visible)
			win->hide();        // in case it's on a different desktop
		if (!visible  ||  win->isMinimized())
			win->showNormal();
		win->raise();
		win->setActiveWindow();
	}
	if (win  &&  !eventID.isEmpty())
		win->selectEvent(eventID);
	return win;
}

/******************************************************************************
* Create an "Alarms Enabled/Enable Alarms" action.
*/
KToggleAction* createAlarmEnableAction(KActionCollection* actions, const char* name)
{
	KToggleAction* action = new KToggleAction(i18n("Enable &Alarms"), 0, actions, name);
	action->setCheckedState(i18n("Disable &Alarms"));
	action->setChecked(theApp()->alarmsEnabled());
	QObject::connect(action, SIGNAL(toggled(bool)), theApp(), SLOT(setAlarmsEnabled(bool)));
	// The following line ensures that all instances are kept in the same state
	QObject::connect(theApp(), SIGNAL(alarmEnabledToggled(bool)), action, SLOT(setChecked(bool)));
	return action;
}

/******************************************************************************
* Create a New From Template KAction.
*/
TemplateMenuAction* createNewFromTemplateAction(const QString& label, QObject* receiver, const char* slot, KActionCollection* actions, const char* name)
{
	return new TemplateMenuAction(label, "new_from_template", receiver, slot, actions, name);
}

/******************************************************************************
* Add a new active (non-archived) alarm.
* Save it in the calendar file and add it to every main window instance.
* If 'selectionView' is non-null, the selection highlight is moved to the new
* event in that listView instance.
* Parameters: parent = parent widget for any resource selection prompt.
* 'event' is updated with the actual event ID.
*/
UpdateStatus addEvent(KAEvent& event, AlarmListView* selectionView, AlarmResource* resource, QWidget* errmsgParent,
                      int options, bool showKOrgErr)
{
	kdDebug(5950) << "KAlarm::addEvent(): " << event.id() << endl;
	bool cancelled = false;
	UpdateStatus status = UPDATE_OK;
	if (!theApp()->checkCalendar())    // ensure calendar is open
		status = UPDATE_FAILED;
	else
	{
		// Save the event details in the calendar file, and get the new event ID
		AlarmCalendar* cal = AlarmCalendar::resources();
		KAEvent* newev = new KAEvent(event);
		if (!cal->addEvent(newev, selectionView, (options & USE_EVENT_ID), resource, (options & NO_RESOURCE_PROMPT), &cancelled))
		{
			delete newev;
			status = UPDATE_FAILED;
		}
		else
		{
			event = *newev;   // update event ID etc.
			if (!cal->save())
				status = SAVE_FAILED;
		}
		if (status == UPDATE_OK)
		{
			if ((options & ALLOW_KORG_UPDATE)  &&  event.copyToKOrganizer())
			{
				if (!sendToKOrganizer(newev))    // tell KOrganizer to show the event
					status = UPDATE_KORG_ERR;
			}

			// Update the window lists
			AlarmListView::addEvent(newev, selectionView);
		}
	}

	if (status != UPDATE_OK  &&  !cancelled  &&  errmsgParent)
		displayUpdateError(errmsgParent, status, ERR_ADD, 1, 1, showKOrgErr);
	return status;
}

/******************************************************************************
* Add a list of new active (non-archived) alarms.
* Save them in the calendar file and add them to every main window instance.
* If 'selectionView' is non-null, the selection highlight is moved to the last
* new event in that listView instance.
* The events are updated with their actual event IDs.
*/
UpdateStatus addEvents(QValueList<KAEvent>& events, AlarmListView* selectionView, QWidget* errmsgParent,
                       bool allowKOrgUpdate, bool showKOrgErr)
{
	kdDebug(5950) << "KAlarm::addEvents(" << events.count() << ")\n";
	if (events.isEmpty())
		return UPDATE_OK;
	int warnErr = 0;
	int warnKOrg = 0;
	UpdateStatus status = UPDATE_OK;
	AlarmResource* resource;
	if (!theApp()->checkCalendar())    // ensure calendar is open
		status = UPDATE_FAILED;
	else
	{
		resource = AlarmResources::instance()->destination(KCalEvent::ACTIVE, selectionView);
		if (!resource)
		{
			kdDebug(5950) << "KAlarm::addEvents(): no resource" << endl;
			status = UPDATE_FAILED;
		}
	}
	if (status == UPDATE_OK)
	{
		QString selectID;
		AlarmCalendar* cal = AlarmCalendar::resources();
		for (QValueList<KAEvent>::Iterator it = events.begin();  it != events.end();  ++it)
		{
			// Save the event details in the calendar file, and get the new event ID
			KAEvent* newev = new KAEvent(*it);
			if (!cal->addEvent(newev, selectionView, false, resource))
			{
				delete newev;
				status = UPDATE_ERROR;
				++warnErr;
				continue;
			}
			*it = *newev;   // update event ID etc.
			if (allowKOrgUpdate  &&  newev->copyToKOrganizer())
			{
				if (!sendToKOrganizer(newev))    // tell KOrganizer to show the event
				{
					++warnKOrg;
					if (status == UPDATE_OK)
						status = UPDATE_KORG_ERR;
				}
			}

			// Update the window lists, but not yet which item is selected
			AlarmListView::addEvent(newev, 0);
			selectID = newev->id();
		}
		if (warnErr == static_cast<int>(events.count()))
			status = UPDATE_FAILED;
		else if (!cal->save())
		{
			status = SAVE_FAILED;
			warnErr = 0;    // everything failed
		}
		else if (selectionView  &&  !selectID.isEmpty())
			selectionView->select(selectID);    // select the last added event
	}

	if (status != UPDATE_OK  &&  errmsgParent)
		displayUpdateError(errmsgParent, status, ERR_ADD, (warnErr ? warnErr : events.count()), warnKOrg, showKOrgErr);
	return status;
}

/******************************************************************************
* Save the event in the archived resource and adjust every main window instance.
* The event's ID is changed to an archived ID if necessary.
*/
bool addArchivedEvent(KAEvent& event, AlarmResource* resource)
{
	kdDebug(5950) << "KAlarm::addArchivedEvent(" << event.id() << ")\n";
	QString oldid = event.id();
	bool archiving = (event.category() == KCalEvent::ACTIVE);
	if (archiving  &&  !Preferences::archivedKeepDays())
		return false;   // expired alarms aren't being kept
	KAEvent* newev = new KAEvent(event);
	if (archiving)
	{
		newev->setCategory(KCalEvent::ARCHIVED);    // this changes the event ID
		newev->setSaveDateTime(KDateTime::currentUtcDateTime());   // time stamp to control purging
	}
	AlarmCalendar* cal = AlarmCalendar::resources();
	// Note that archived resources are automatically saved after changes are made
	if (!cal->addEvent(newev, 0, false, resource))
	{
		delete newev;   // failed to add to calendar - leave event in its original state
		return false;
	}
	event = *newev;   // update event ID etc.

	// Update window lists
	if (!archiving)
		AlarmListView::addEvent(newev, 0);
	else
		AlarmListView::modifyEvent(oldid, newev, 0);
	return true;
}

/******************************************************************************
* Add a new template.
* Save it in the calendar file and add it to every template list view.
* If 'selectionView' is non-null, the selection highlight is moved to the new
* event in that listView instance.
* 'event' is updated with the actual event ID.
* Parameters: promptParent = parent widget for any resource selection prompt.
*/
UpdateStatus addTemplate(KAEvent& event, TemplateListView* selectionView, QWidget* promptParent,
                         AlarmResource* resource, QWidget* errmsgParent)
{
	kdDebug(5950) << "KAlarm::addTemplate(): " << event.id() << endl;
	UpdateStatus status = UPDATE_OK;

	// Add the template to the calendar file
	KAEvent* newev = new KAEvent(event);
	AlarmCalendar* cal = AlarmCalendar::resources();
	if (!cal->addEvent(newev, promptParent, false, resource))
	{
		delete newev;
		status = UPDATE_FAILED;
	}
	else
	{
		event = *newev;   // update event ID etc.
		if (!cal->save())
			status = SAVE_FAILED;
		else
		{
			cal->emitEmptyStatus();

			// Update the window lists
			TemplateListView::addEvent(newev, selectionView);
			return UPDATE_OK;
		}
	}

	if (errmsgParent)
		displayUpdateError(errmsgParent, status, ERR_TEMPLATE, 1);
	return status;
}

/******************************************************************************
* Modify an active (non-archived) alarm in the calendar file and in every main
* window instance.
* The new event must have a different event ID from the old one.
* If 'selectionView' is non-null, the selection highlight is moved to the
* modified event in that listView instance.
*/
UpdateStatus modifyEvent(KAEvent& oldEvent, KAEvent& newEvent, AlarmListView* selectionView,
                         QWidget* errmsgParent, bool showKOrgErr)
{
	kdDebug(5950) << "KAlarm::modifyEvent(): '" << oldEvent.id() << endl;

	UpdateStatus status = UPDATE_OK;
	if (!newEvent.valid())
	{
		deleteEvent(oldEvent, true);
		status = UPDATE_FAILED;
	}
	else
	{
		QString oldId = oldEvent.id();
		if (oldEvent.copyToKOrganizer())
		{
			// Tell KOrganizer to delete its old event.
			// But ignore errors, because the user could have manually
			// deleted it since KAlarm asked KOrganizer to set it up.
			deleteFromKOrganizer(oldId);
		}
		// Delete from the window lists to prevent the event's invalid
		// pointer being accessed.
		AlarmListView::deleteEvent(oldId);

		// Update the event in the calendar file, and get the new event ID
		KAEvent* newev = new KAEvent(newEvent);
		AlarmCalendar* cal = AlarmCalendar::resources();
		if (!cal->modifyEvent(oldId, newev))
		{
			delete newev;
			AlarmListView::refreshAll();
			status = UPDATE_FAILED;
		}
		else
		{
			newEvent = *newev;
			if (!cal->save())
				status = SAVE_FAILED;
			if (status == UPDATE_OK)
			{
				if (newev->copyToKOrganizer())
				{
					if (!sendToKOrganizer(newev))    // tell KOrganizer to show the new event
						status = UPDATE_KORG_ERR;
				}

				// Remove "Don't show error messages again" for the old alarm
				setDontShowErrors(oldId);

				// Update the window lists
				AlarmListView::addEvent(newev, selectionView);
			}
		}
	}

	if (status != UPDATE_OK  &&  errmsgParent)
		displayUpdateError(errmsgParent, status, ERR_MODIFY, 1, 1, showKOrgErr);
	return status;
}

/******************************************************************************
* Update an active (non-archived) alarm from the calendar file and from every
* main window instance.
* The new event will have the same event ID as the old one.
* If 'selectionView' is non-null, the selection highlight is moved to the
* updated event in that listView instance.
* The event is not updated in KOrganizer, since this function is called when an
* existing alarm is rescheduled (due to recurrence or deferral).
*/
UpdateStatus updateEvent(KAEvent& event, AlarmListView* selectionView, QWidget* errmsgParent, bool archiveOnDelete)
{
	kdDebug(5950) << "KAlarm::updateEvent(): " << event.id() << endl;

	if (!event.valid())
		deleteEvent(event, archiveOnDelete);
	else
	{
		// Update the event in the calendar file.
		AlarmCalendar* cal = AlarmCalendar::resources();
		KAEvent* newEvent = cal->updateEvent(event);
		if (!cal->save())
		{
			if (errmsgParent)
				displayUpdateError(errmsgParent, SAVE_FAILED, ERR_ADD, 1);
			return SAVE_FAILED;
		}

		// Update the window lists
		AlarmListView::modifyEvent(newEvent, selectionView);
	}
	return UPDATE_OK;
}

/******************************************************************************
* Update a template in the calendar file and in every template list view.
* If 'selectionView' is non-null, the selection highlight is moved to the
* updated event in that listView instance.
*/
UpdateStatus updateTemplate(KAEvent& event, TemplateListView* selectionView, QWidget* errmsgParent)
{
	AlarmCalendar* cal = AlarmCalendar::resources();
	KAEvent* newEvent = cal->updateEvent(event);
	UpdateStatus status = UPDATE_OK;
	if (!newEvent)
		status = UPDATE_FAILED;
	else if (!cal->save())
		status = SAVE_FAILED;
	if (status != UPDATE_OK)
	{
		if (errmsgParent)
			displayUpdateError(errmsgParent, status, ERR_TEMPLATE, 1);
		return status;
	}

	TemplateListView::modifyEvent(newEvent->id(), newEvent, selectionView);
	return UPDATE_OK;
}

/******************************************************************************
* Delete alarms from the calendar file and from every main window instance.
* If the events are archived, the events' IDs are changed to archived IDs if necessary.
*/
UpdateStatus deleteEvent(KAEvent& event, bool archive, QWidget* errmsgParent, bool showKOrgErr)
{
	KAEvent::List events;
	events += &event;
	return deleteEvents(events, archive, errmsgParent, showKOrgErr);
}

UpdateStatus deleteEvents(KAEvent::List& events, bool archive, QWidget* errmsgParent, bool showKOrgErr)
{
	kdDebug(5950) << "KAlarm::deleteEvents(" << events.count() << ")\n";
	if (events.isEmpty())
		return UPDATE_OK;
	int warnErr = 0;
	int warnKOrg = 0;
	UpdateStatus status = UPDATE_OK;
	AlarmCalendar* cal = AlarmCalendar::resources();
	for (KAEvent::List::Iterator ev = events.begin();  ev != events.end();  ++ev)
	{
		// Save the event details in the calendar file, and get the new event ID
		KAEvent* event = *ev;
		QString id = event->id();

		// Update the window lists
		AlarmListView::deleteEvent(id);

		// Delete the event from the calendar file
		if (event->category() != KCalEvent::ARCHIVED)
		{
			if (event->copyToKOrganizer())
			{
				// The event was shown in KOrganizer, so tell KOrganizer to
				// delete it. But ignore errors, because the user could have
				// manually deleted it from KOrganizer since it was set up.
				if (!deleteFromKOrganizer(id))
				{
					++warnKOrg;
					if (status == UPDATE_OK)
						status = UPDATE_KORG_ERR;
				}
			}
			if (archive  &&  event->toBeArchived())
				addArchivedEvent(*event);     // this changes the event ID to an archived ID
		}
		if (!cal->deleteEvent(id, false))   // don't save calendar after deleting
		{
			status = UPDATE_ERROR;
			++warnErr;
		}

		// Remove "Don't show error messages again" for this alarm
		setDontShowErrors(id);
	}

	if (warnErr == static_cast<int>(events.count()))
		status = UPDATE_FAILED;
	else if (!cal->save())      // save the calendars now
	{
		status = SAVE_FAILED;
		warnErr = events.count();
	}
	if (status != UPDATE_OK  &&  errmsgParent)
		displayUpdateError(errmsgParent, status, ERR_DELETE, warnErr, warnKOrg, showKOrgErr);
	return status;
}

/******************************************************************************
* Delete templates from the calendar file and from every template list view.
*/
UpdateStatus deleteTemplates(const QStringList& eventIDs, QWidget* errmsgParent)
{
	kdDebug(5950) << "KAlarm::deleteTemplates(" << eventIDs.count() << ")\n";
	if (eventIDs.isEmpty())
		return UPDATE_OK;
	int warnErr = 0;
	UpdateStatus status = UPDATE_OK;
	AlarmCalendar* cal = AlarmCalendar::resources();
	for (QStringList::ConstIterator it = eventIDs.constBegin();  it != eventIDs.constEnd();  ++it)
	{
		// Update the window lists
		QString id = *it;
		TemplateListView::deleteEvent(id);

		// Delete the template from the calendar file
		AlarmCalendar* cal = AlarmCalendar::resources();
		if (!cal->deleteEvent(id, false))    // don't save calendar after deleting
		{
			status = UPDATE_ERROR;
			++warnErr;
		}
	}

	if (warnErr == static_cast<int>(eventIDs.count()))
		status = UPDATE_FAILED;
	else if (!cal->save())      // save the calendars now
	{
		status = SAVE_FAILED;
		warnErr = eventIDs.count();
	}
	cal->emitEmptyStatus();
	if (status != UPDATE_OK  &&  errmsgParent)
		displayUpdateError(errmsgParent, status, ERR_TEMPLATE, warnErr);
	return status;
}

/******************************************************************************
* Delete an alarm from the display calendar.
*/
void deleteDisplayEvent(const QString& eventID)
{
	kdDebug(5950) << "KAlarm::deleteDisplayEvent(" << eventID << ")\n";
	AlarmCalendar* cal = AlarmCalendar::displayCalendarOpen();
	if (cal)
		cal->deleteEvent(eventID, true);   // save calendar after deleting
}

/******************************************************************************
* Undelete archived alarms, and update every main window instance.
* The archive bit is set to ensure that they get re-archived if deleted again.
* If 'selectionView' is non-null, the selection highlight is moved to the
* restored event in that listView instance.
* 'ineligibleIDs' is filled in with the IDs of any ineligible events.
*/
UpdateStatus reactivateEvent(KAEvent& event, AlarmListView* selectionView,
                             AlarmResource* resource, QWidget* errmsgParent, bool showKOrgErr)
{
	QStringList ids;
	KAEvent::List events;
	events += &event;
	return reactivateEvents(events, ids, selectionView, resource, errmsgParent, showKOrgErr);
}

UpdateStatus reactivateEvents(KAEvent::List& events, QStringList& ineligibleIDs, AlarmListView* selectionView,
                              AlarmResource* resource, QWidget* errmsgParent, bool showKOrgErr)
{
	kdDebug(5950) << "KAlarm::reactivateEvents(" << events.count() << ")\n";
	ineligibleIDs.clear();
	if (events.isEmpty())
		return UPDATE_OK;
	int warnErr = 0;
	int warnKOrg = 0;
	UpdateStatus status = UPDATE_OK;
	if (!resource)
		resource = AlarmResources::instance()->destination(KCalEvent::ACTIVE, selectionView);
	if (!resource)
	{
		kdDebug(5950) << "KAlarm::reactivateEvents(): no resource" << endl;
		status = UPDATE_FAILED;
		warnErr = events.count();
	}
	else
	{
		QString selectID;
		int count = 0;
		AlarmCalendar* cal = AlarmCalendar::resources();
		KDateTime now = KDateTime::currentUtcDateTime();
		for (KAEvent::List::Iterator it = events.begin();  it != events.end();  ++it)
		{
			// Delete the event from the archived resource
			KAEvent* event = *it;
			if (event->category() != KCalEvent::ARCHIVED
			||  !event->occursAfter(now, true))
			{
				ineligibleIDs += event->id();
				continue;
			}
			++count;

			KAEvent* newev = new KAEvent(*event);
			QString oldid = event->id();
			newev->setCategory(KCalEvent::ACTIVE);    // this changes the event ID
			if (newev->recurs()  ||  newev->repeatCount())
				newev->setNextOccurrence(now);   // skip any recurrences in the past
			newev->setArchive();    // ensure that it gets re-archived if it is deleted

			// Save the event details in the calendar file.
			// This converts the event ID.
			if (!cal->addEvent(newev, selectionView, true, resource))
			{
				delete newev;
				status = UPDATE_ERROR;
				++warnErr;
				continue;
			}
			if (newev->copyToKOrganizer())
			{
				if (!sendToKOrganizer(newev))    // tell KOrganizer to show the event
				{
					++warnKOrg;
					if (status == UPDATE_OK)
						status = UPDATE_KORG_ERR;
				}
			}

			// Update the window lists
			AlarmListView::undeleteEvent(oldid, newev, 0);
			selectID = newev->id();

			if (cal->event(oldid)    // no error if event doesn't exist in archived resource
			&&  !cal->deleteEvent(oldid, false))   // don't save calendar after deleting
			{
				status = UPDATE_ERROR;
				++warnErr;
			}
			*it = newev;
		}
		if (selectionView  &&  !selectID.isEmpty())
			selectionView->select(selectID);    // select the last added event

		if (warnErr == count)
			status = UPDATE_FAILED;
		// Save the calendars, even if all events failed, since more than one calendar was updated
		if (!cal->save()  &&  status != UPDATE_FAILED)
		{
			status = SAVE_FAILED;
			warnErr = count;
		}
	}
	if (status != UPDATE_OK  &&  errmsgParent)
		displayUpdateError(errmsgParent, status, ERR_REACTIVATE, warnErr, warnKOrg, showKOrgErr);
	return status;
}

/******************************************************************************
* Enable or disable alarms in the calendar file and in every main window instance.
* The new events will have the same event IDs as the old ones.
* If 'selectionView' is non-null, the selection highlight is moved to the
* updated event in that listView instance.
*/
UpdateStatus enableEvents(KAEvent::List& events, AlarmListView* selectionView, bool enable, QWidget* errmsgParent)
{
	kdDebug(5950) << "KAlarm::enableEvents(" << events.count() << ")\n";
	if (events.isEmpty())
		return UPDATE_OK;
	UpdateStatus status = UPDATE_OK;
	AlarmCalendar* cal = AlarmCalendar::resources();
	for (KAEvent::List::Iterator it = events.begin();  it != events.end();  ++it)
	{
		KAEvent* event = *it;
		if (enable != event->enabled())
		{
			event->setEnabled(enable);

			// Update the event in the calendar file
			KAEvent* newev = cal->updateEvent(event);

			// If we're disabling a display alarm, close any message window
			if (!enable  &&  event->displayAction())
			{
				MessageWin* win = MessageWin::findEvent(event->id());
				delete win;
			}

			// Update the window lists
			AlarmListView::modifyEvent(newev, selectionView);
		}
	}

	if (!cal->save())
		status = SAVE_FAILED;
	if (status != UPDATE_OK  &&  errmsgParent)
		displayUpdateError(errmsgParent, status, ERR_ADD, events.count(), 0);
	return status;
}

/******************************************************************************
* This method must only be called from the main KAlarm queue processing loop,
* to prevent asynchronous calendar operations interfering with one another.
*
* Purge all archived events from the default archived alarm resource whose end
* time is longer ago than 'purgeDays'. All events are deleted if 'purgeDays'
* is zero.
*/
void purgeArchive(int purgeDays)
{
	if (purgeDays < 0)
		return;
	kdDebug(5950) << "KAlarm::purgeArchive(" << purgeDays << ")" << endl;
	QDate cutoff = KDateTime::currentLocalDate().addDays(-purgeDays);
	AlarmResource* resource = AlarmResources::instance()->getStandardResource(AlarmResource::ARCHIVED);
	if (!resource)
		return;
	KAEvent::List events = AlarmCalendar::resources()->events(resource);
	for (KAEvent::List::Iterator it = events.begin();  it != events.end();  )
	{
		KAEvent* event = *it;
		KCal::Incidence* kcalIncidence = resource->incidence(event->id());
		if ((purgeDays  &&  kcalIncidence  &&  kcalIncidence->created().date() >= cutoff)
		||  event->category() != KCalEvent::ARCHIVED)
			it = events.erase(it);
		else
		{
			AlarmListView::deleteEvent(event->id());   // update the window lists
			++it;
		}
	}
	if (!events.isEmpty())
		AlarmCalendar::resources()->purgeEvents(events);   // delete the events and save the calendar
}

/******************************************************************************
* Display an error message about an error when saving an event.
*/
void displayUpdateError(QWidget* parent, UpdateStatus status, UpdateError code, int nAlarms, int nKOrgAlarms, bool showKOrgError)
{
	QString errmsg;
	if (status > UPDATE_KORG_ERR)
	{
		switch (code)
		{
			case ERR_ADD:
			case ERR_MODIFY:
				errmsg = (nAlarms > 1) ? i18n("Error saving alarms")
				                       : i18n("Error saving alarm");
				break;
			case ERR_DELETE:
				errmsg = (nAlarms > 1) ? i18n("Error deleting alarms")
				                       : i18n("Error deleting alarm");
				break;
			case ERR_REACTIVATE:
				errmsg = (nAlarms > 1) ? i18n("Error saving reactivated alarms")
				                       : i18n("Error saving reactivated alarm");
				break;
			case ERR_TEMPLATE:
				errmsg = (nAlarms > 1) ? i18n("Error saving alarm templates")
				                       : i18n("Error saving alarm template");
				break;
		}
		KMessageBox::error(parent, errmsg);
	}
	else if (showKOrgError)
		displayKOrgUpdateError(parent, code, nKOrgAlarms);
}

/******************************************************************************
* Display an error message corresponding to a specified alarm update error code.
*/
void displayKOrgUpdateError(QWidget* parent, UpdateError code, int nAlarms)
{
	QString errmsg;
	switch (code)
	{
		case ERR_ADD:
		case ERR_REACTIVATE:
			errmsg = (nAlarms > 1) ? i18n("Unable to show alarms in KOrganizer")
			                       : i18n("Unable to show alarm in KOrganizer");
			break;
		case ERR_MODIFY:
			errmsg = i18n("Unable to update alarm in KOrganizer");
			break;
		case ERR_DELETE:
			errmsg = (nAlarms > 1) ? i18n("Unable to delete alarms from KOrganizer")
			                       : i18n("Unable to delete alarm from KOrganizer");
			break;
		case ERR_TEMPLATE:
			return;
	}
	KMessageBox::error(parent, errmsg);
}

/******************************************************************************
* Execute a New Alarm dialog for the specified alarm type.
*/
void editNewAlarm(EditAlarmDlg::Type type, QWidget* parent, AlarmListView* view)
{
	EditAlarmDlg* editDlg = EditAlarmDlg::create(false, type, true, parent);
	doEditNewAlarm(editDlg, view);
	delete editDlg;
}

/******************************************************************************
* Execute a New Alarm dialog for the specified alarm type.
*/
void editNewAlarm(KAEvent::Action action, QWidget* parent, AlarmListView* view, const AlarmText* text)
{
	bool setAction = false;
	EditAlarmDlg::Type type;
	switch (action)
	{
		case KAEvent::MESSAGE:
		case KAEvent::FILE:
			type = EditAlarmDlg::DISPLAY;
			setAction = true;
			break;
		case KAEvent::COMMAND:
			type = EditAlarmDlg::COMMAND;
			break;
		case KAEvent::EMAIL:
			type = EditAlarmDlg::EMAIL;
			break;
		default:
			return;
	}
	EditAlarmDlg* editDlg = EditAlarmDlg::create(false, type, true, parent);
	if (setAction  ||  text)
		editDlg->setAction(action, *text);
	doEditNewAlarm(editDlg, view);
	delete editDlg;
}

/******************************************************************************
* Execute a New Alarm dialog, optionally either presetting it to the supplied
* event, or setting the action and text.
*/
void editNewAlarm(const KAEvent* preset, QWidget* parent, AlarmListView* view)
{
	EditAlarmDlg* editDlg = EditAlarmDlg::create(false, preset, true, parent);
	doEditNewAlarm(editDlg, view);
	delete editDlg;
}

/******************************************************************************
* Common code for editNewAlarm() variants.
*/
void doEditNewAlarm(EditAlarmDlg* editDlg, AlarmListView* view)
{
	if (editDlg->exec() == QDialog::Accepted)
	{
		KAEvent event;
		AlarmResource* resource;
		editDlg->getEvent(event, resource);

		// Add the alarm to the displayed lists and to the calendar file
		switch (addEvent(event, view, resource, editDlg))
		{
			case UPDATE_FAILED:
				return;
			case UPDATE_KORG_ERR:
				displayKOrgUpdateError(editDlg, ERR_ADD, 1);
				break;
			default:
				break;
		}
		Undo::saveAdd(event, resource);

		outputAlarmWarnings(editDlg, &event);
	}
}

/******************************************************************************
* Display the alarm edit dialogue to edit a new alarm, optionally preset with
* a template.
*/
bool editNewAlarm(const QString& templateName, QWidget* parent, AlarmListView* view)
{
	if (!templateName.isEmpty())
	{
		KAEvent* templateEvent = AlarmCalendar::resources()->templateEvent(templateName);
		if (templateEvent->valid())
		{
			editNewAlarm(templateEvent, parent, view);
			return true;
		}
		kdWarning(5950) << "KAlarm::editNewAlarm(" << templateName << "): template not found" << endl;
	}
	return false;
}

/******************************************************************************
* Create a new template.
*/
void editNewTemplate(EditAlarmDlg::Type type, QWidget* parent, TemplateListView* view)
{
	EditAlarmDlg* editDlg = EditAlarmDlg::create(true, type, true, parent);
	if (editDlg->exec() == QDialog::Accepted)
	{
		KAEvent event;
		AlarmResource* resource;
		editDlg->getEvent(event, resource);

		// Add the template to the displayed lists and to the calendar file
		addTemplate(event, view, parent, resource, editDlg);
		Undo::saveAdd(event, resource);
	}
	delete editDlg;
}

/******************************************************************************
* Create a new template, based on an existing event or template.
*/
void editNewTemplate(const KAEvent* preset, QWidget* parent, TemplateListView* view)
{
	EditAlarmDlg* editDlg = EditAlarmDlg::create(true, preset, true, parent);
	if (editDlg->exec() == QDialog::Accepted)
	{
		KAEvent event;
		AlarmResource* resource;
		editDlg->getEvent(event, resource);

		// Add the template to the displayed lists and to the calendar file
		addTemplate(event, view, parent, resource, editDlg);
		Undo::saveAdd(event, resource);
	}
	delete editDlg;
}

/******************************************************************************
* Open the Edit Alarm dialogue to edit the specified alarm.
* If the alarm is read-only or archived, the dialogue is opened read-only.
*/
void editAlarm(KAEvent* event, QWidget* parent, AlarmListView* view)
{
	if (event->expired()  ||  AlarmCalendar::resources()->eventReadOnly(event->id()))
	{
		viewAlarm(event, parent);
		return;
	}
	EditAlarmDlg* editDlg = EditAlarmDlg::create(false, event, false, parent, 0, EditAlarmDlg::RES_USE_EVENT_ID);
	if (editDlg->exec() == QDialog::Accepted)
	{
		KAEvent newEvent;
		AlarmResource* resource;
		bool changeDeferral = !editDlg->getEvent(newEvent, resource);

		// Update the event in the displays and in the calendar file
		Undo::Event undo(*event, resource);
		if (changeDeferral)
		{
			// The only change has been to an existing deferral
			if (updateEvent(newEvent, view, editDlg, true) != UPDATE_OK)   // keep the same event ID
				return;   // failed to save event
		}
		else
		{
			if (modifyEvent(*event, newEvent, view, editDlg) == UPDATE_KORG_ERR)
				displayKOrgUpdateError(editDlg, ERR_MODIFY, 1);
		}
		Undo::saveEdit(undo, newEvent);

		outputAlarmWarnings(editDlg, &newEvent);
	}
	delete editDlg;
}

/******************************************************************************
* Display the alarm edit dialogue to edit a specified alarm.
* An error occurs if the alarm is read-only or expired.
*/
bool editAlarm(const QString& eventID, QWidget* parent)
{
	KAEvent* event = AlarmCalendar::resources()->event(eventID);
	if (!event)
	{
		kdError(5950) << "KAlarm::editAlarm(): event ID not found: " << eventID << endl;
		return false;
	}
	if (AlarmCalendar::resources()->eventReadOnly(eventID))
	{
		kdError(5950) << "KAlarm::editAlarm(" << eventID << "): read-only" << endl;
		return false;
	}
	switch (event->category())
	{
		case KCalEvent::ACTIVE:
		case KCalEvent::TEMPLATE:
			break;
		default:
			kdError(5950) << "KAlarm::editAlarm(" << eventID << "): event not active or template" << endl;
			return false;
	}
	editAlarm(event, parent);
	return true;
}

/******************************************************************************
* Open the Edit Alarm dialogue to edit the specified template.
* If the template is read-only, the dialogue is opened read-only.
*/
void editTemplate(KAEvent* event, QWidget* parent, TemplateListView* view)
{
	if (AlarmCalendar::resources()->eventReadOnly(event->id()))
	{
		// The template is read-only, so make the dialogue read-only
		EditAlarmDlg* editDlg = EditAlarmDlg::create(true, event, false, parent, 0, EditAlarmDlg::RES_PROMPT, true);
		editDlg->exec();
		delete editDlg;
		return;
	}
	EditAlarmDlg* editDlg = EditAlarmDlg::create(true, event, false, parent, 0, EditAlarmDlg::RES_USE_EVENT_ID);
	if (editDlg->exec() == QDialog::Accepted)
	{
		KAEvent newEvent;
		AlarmResource* resource;
		editDlg->getEvent(newEvent, resource);
		QString id = event->id();
		newEvent.setEventID(id);

		// Update the event in the displays and in the calendar file
		Undo::Event undo(*event, resource);
		updateTemplate(newEvent, view, editDlg);
		Undo::saveEdit(undo, newEvent);
	}
	delete editDlg;
}

/******************************************************************************
* Open the Edit Alarm dialogue to view the specified alarm (read-only).
*/
void viewAlarm(const KAEvent* event, QWidget* parent)
{
	EditAlarmDlg* editDlg = EditAlarmDlg::create(false, event, false, parent, 0, EditAlarmDlg::RES_PROMPT, true);
	editDlg->exec();
	delete editDlg;
}

/******************************************************************************
*  Returns a list of all alarm templates.
*  If shell commands are disabled, command alarm templates are omitted.
*/
KAEvent::List templateList()
{
	KAEvent::List templates;
	bool includeCmdAlarms = ShellProcess::authorised();
	KAEvent::List events = AlarmCalendar::resources()->events(KCalEvent::TEMPLATE);
	for (KAEvent::List::ConstIterator it = events.constBegin();  it != events.constEnd();  ++it)
	{
		KAEvent* event = *it;
		if (includeCmdAlarms  ||  event->action() != KAEvent::COMMAND)
			templates.append(event);
	}
	return templates;
}

/******************************************************************************
* To be called after an alarm has been edited.
* Prompt the user to re-enable alarms if they are currently disabled, and if
* it's an email alarm, warn if no 'From' email address is configured.
*/
void outputAlarmWarnings(QWidget* parent, const KAEvent* event)
{
	if (event  &&  event->action() == KAEvent::EMAIL
	&&  Preferences::emailAddress().isEmpty())
		KMessageBox::information(parent, i18n("Please set the 'From' email address...",
		                                      "%1\nPlease set it in the Configuration dialog.").arg(KAMail::i18n_NeedFromEmailAddress()));

	if (!theApp()->alarmsEnabled())
	{
		if (KMessageBox::warningYesNo(parent, i18n("Alarms are currently disabled.\nDo you want to enable alarms now?"),
		                              QString::null, i18n("Enable"), i18n("Keep Disabled"),
		                              QString::fromLatin1("EditEnableAlarms"))
		                == KMessageBox::Yes)
			theApp()->setAlarmsEnabled(true);
	}
}

/******************************************************************************
* Reload the calendar.
*/
void refreshAlarms()
{
	kdDebug(5950) << "KAlarm::refreshAlarms()" << endl;
	if (!refreshAlarmsQueued)
	{
		refreshAlarmsQueued = true;
		theApp()->processQueue();
	}
}

/******************************************************************************
* This method must only be called from the main KAlarm queue processing loop,
* to prevent asynchronous calendar operations interfering with one another.
*
* If refreshAlarms() has been called, reload the calendars.
*/
void refreshAlarmsIfQueued()
{
	if (refreshAlarmsQueued)
	{
		kdDebug(5950) << "KAlarm::refreshAlarmsIfQueued()" << endl;
		AlarmCalendar::resources()->reload();

		// Close any message windows for alarms which are now disabled
		KAEvent::List events = AlarmCalendar::resources()->events(KCalEvent::ACTIVE);
		for (KAEvent::List::ConstIterator it = events.constBegin();  it != events.constEnd();  ++it)
		{
			KAEvent* event = *it;
			if (!event->enabled()  &&  event->displayAction())
			{
				MessageWin* win = MessageWin::findEvent(event->id());
				delete win;
			}
		}

		// AlarmResources::reload() causes a display refresh for each resource
		refreshAlarmsQueued = false;
	}
}

/******************************************************************************
*  Start KMail if it isn't already running, and optionally iconise it.
*  Reply = reason for failure to run KMail (which may be the empty string)
*        = null string if success.
*/
QString runKMail(bool minimise)
{
	QCString dcopName;
	QString errmsg;
	if (!runProgram("kmail", (minimise ? KMAIL_DCOP_WINDOW : ""), dcopName, errmsg))
		return i18n("Unable to start KMail\n(%1)").arg(errmsg);
	return QString::null;
}

/******************************************************************************
*  Start another program for DCOP access if it isn't already running.
*  If 'windowName' is not empty, the program's window of that name is iconised.
*  On exit, 'dcopName' contains the DCOP name to access the application, and
*  'errorMessage' contains an error message if failure.
*  Reply = true if the program is now running.
*/
bool runProgram(const QCString& program, const QCString& windowName, QCString& dcopName, QString& errorMessage)
{
	if (!kapp->dcopClient()->isApplicationRegistered(program))
	{
		// KOrganizer is not already running, so start it
		if (KApplication::startServiceByDesktopName(QString::fromLatin1(program), QString::null, &errorMessage, &dcopName))
		{
			kdError(5950) << "runProgram(): couldn't start " << program << " (" << errorMessage << ")\n";
			return false;
		}
		// Minimise its window - don't use hide() since this would remove all
		// trace of it from the panel if it is not configured to be docked in
		// the system tray.
		kapp->dcopClient()->send(dcopName, windowName, "minimize()", QString::null);
	}
	else if (dcopName.isEmpty())
		dcopName = program;
	errorMessage = QString::null;
	return true;
}

/******************************************************************************
* The "Don't show again" option for error messages is personal to the user on a
* particular computer. For example, he may want to inhibit error messages only
* on his laptop. So the status is not stored in the alarm calendar, but in the
* user's local KAlarm data directory.
******************************************************************************/
#warning Provide a clean-up mechanism for out-of-date entries

/******************************************************************************
* Return the Don't-show-again error message tags set for a specified alarm ID.
*/
QStringList dontShowErrors(const QString& eventId)
{
	if (eventId.isEmpty())
		return QStringList();
	KConfig config(locateLocal("appdata", ALARM_OPTS_FILE));
	config.setGroup(QString::fromLatin1(DONT_SHOW_ERRORS_GROUP));
	return config.readListEntry(eventId);
}

/******************************************************************************
* Check whether the specified Don't-show-again error message tag is set for an
* alarm ID.
*/
bool dontShowErrors(const QString& eventId, const QString& tag)
{
	if (tag.isEmpty()  ||  eventId.isEmpty())
		return false;
	QStringList tags = dontShowErrors(eventId);
	return tags.find(tag) != tags.end();
}

/******************************************************************************
* Reset the Don't-show-again error message tags for an alarm ID.
* If 'tags' is empty, the config entry is deleted.
*/
void setDontShowErrors(const QString& eventId, const QStringList& tags)
{
	if (eventId.isEmpty())
		return;
	KConfig config(locateLocal("appdata", ALARM_OPTS_FILE));
	config.setGroup(QString::fromLatin1(DONT_SHOW_ERRORS_GROUP));
	if (tags.isEmpty())
		config.deleteEntry(eventId);
	else
		config.writeEntry(eventId, tags);
	config.sync();
}

/******************************************************************************
* Set the specified Don't-show-again error message tag for an alarm ID.
* Existing tags are unaffected.
*/
void setDontShowErrors(const QString& eventId, const QString& tag)
{
	if (tag.isEmpty()  ||  eventId.isEmpty())
		return;
	KConfig config(locateLocal("appdata", ALARM_OPTS_FILE));
	config.setGroup(QString::fromLatin1(DONT_SHOW_ERRORS_GROUP));
	QStringList tags = config.readListEntry(eventId);
	if (tags.find(tag) == tags.end())
	{
		tags += tag;
		config.writeEntry(eventId, tags);
		config.sync();
	}
}

/******************************************************************************
*  Read the size for the specified window from the config file, for the
*  current screen resolution.
*  Reply = true if size set in the config file, in which case 'result' is set
*        = false if no size is set, in which case 'result' is unchanged.
*/
bool readConfigWindowSize(const char* window, QSize& result, int* splitterWidth)
{
	KConfig* config = KGlobal::config();
	config->setGroup(QString::fromLatin1(window));
	QWidget* desktop = KApplication::desktop();
	QSize s = QSize(config->readNumEntry(QString::fromLatin1("Width %1").arg(desktop->width()), 0),
	                config->readNumEntry(QString::fromLatin1("Height %1").arg(desktop->height()), 0));
	if (s.isEmpty())
		return false;
	result = s;
	if (splitterWidth)
		*splitterWidth = config->readNumEntry(QString::fromLatin1("Splitter %1").arg(desktop->width()), -1);
	return true;
}

/******************************************************************************
*  Write the size for the specified window to the config file, for the
*  current screen resolution.
*/
void writeConfigWindowSize(const char* window, const QSize& size, int splitterWidth)
{
	KConfig* config = KGlobal::config();
	config->setGroup(QString::fromLatin1(window));
	QWidget* desktop = KApplication::desktop();
	config->writeEntry(QString::fromLatin1("Width %1").arg(desktop->width()), size.width());
	config->writeEntry(QString::fromLatin1("Height %1").arg(desktop->height()), size.height());
	if (splitterWidth >= 0)
		config->writeEntry(QString::fromLatin1("Splitter %1").arg(desktop->width()), splitterWidth);
	config->sync();
}

/******************************************************************************
* Return the current KAlarm version number.
*/
int Version()
{
	static int version = 0;
	if (!version)
		version = getVersionNumber(KALARM_VERSION);
	return version;
}

/******************************************************************************
* Convert the supplied KAlarm version string to a version number.
* Reply = version number (double digit for each of major, minor & issue number,
*         e.g. 010203 for 1.2.3
*       = 0 if invalid version string.
*/
int getVersionNumber(const QString& version, QString* subVersion)
{
	// N.B. Remember to change  Version(int major, int minor, int rev)
	//      if the representation returned by this method changes.
	if (subVersion)
		*subVersion = QString::null;
	int count = version.contains('.') + 1;
	if (count < 2)
		return 0;
	bool ok;
	unsigned vernum = version.section('.', 0, 0).toUInt(&ok) * 10000;  // major version
	if (!ok)
		return 0;
	unsigned v = version.section('.', 1, 1).toUInt(&ok);               // minor version
	if (!ok)
		return 0;
	vernum += (v < 99 ? v : 99) * 100;
	if (count >= 3)
	{
		// Issue number: allow other characters to follow the last digit
		QString issue = version.section('.', 2);
		if (!issue.at(0).isDigit())
			return 0;
		int n = issue.length();
		int i;
		for (i = 0;  i < n && issue.at(i).isDigit();  ++i) ;
		if (subVersion)
			*subVersion = issue.mid(i);
		v = issue.left(i).toUInt();   // issue number
		vernum += (v < 99 ? v : 99);
	}
	return vernum;
}

/******************************************************************************
* Check from its mime type whether a file appears to be a text or image file.
* If a text file, its type is distinguished.
* Reply = file type.
*/
FileType fileType(const KMimeType::Ptr& mimetype)
{
	if (mimetype->is("text/html"))
		return TextFormatted;
	if (mimetype->is("application/x-executable"))
		return TextApplication;
	if (mimetype->is("text/plain"))
		return TextPlain;
	if (mimetype->name().startsWith(QString::fromLatin1("image/")))
		return Image;
	return Unknown;
}

/******************************************************************************
* Display a modal dialogue to choose an existing file, initially highlighting
* any specified file.
* @param initialFile The file to initially highlight - must be a full path name or URL.
* @param defaultDir The directory to start in if @p initialFile is empty. If empty,
*                   the user's home directory will be used. Updated to the
*                   directory containing the selected file, if a file is chosen.
* @param mode OR of KFile::Mode values, e.g. ExistingOnly, LocalOnly.
* Reply = URL selected. If none is selected, URL.isEmpty() is true.
*/
QString browseFile(const QString& caption, QString& defaultDir, const QString& initialFile,
                   const QString& filter, int mode, QWidget* parent, const char* name)
{
	QString initialDir = !initialFile.isEmpty() ? QString(initialFile).remove(QRegExp("/[^/]*$"))
	                   : !defaultDir.isEmpty()  ? defaultDir
	                   :                          QDir::homeDirPath();
	KFileDialog fileDlg(initialDir, filter, parent, name, true);
	fileDlg.setOperationMode(mode & KFile::ExistingOnly ? KFileDialog::Opening : KFileDialog::Saving);
	fileDlg.setMode(KFile::File | mode);
	fileDlg.setCaption(caption);
	if (!initialFile.isEmpty())
		fileDlg.setSelection(initialFile);
	if (fileDlg.exec() != QDialog::Accepted)
		return QString::null;
	KURL url = fileDlg.selectedURL();
	defaultDir = url.path();
	return (mode & KFile::LocalOnly) ? url.path() : url.prettyURL();
}

/******************************************************************************
* Check whether a date/time is during working hours and/or holidays, depending
* on the flags set for the specified event.
* Working time is interpreted in the default KAlarm time zone.
*/
bool isWorkingTime(const KDateTime& dt, const KAEvent* event)
{
	bool workOnly = event && event->workTimeOnly();
	bool holidays = event && event->holidaysExcluded();
#warning Working time should be interpreted in preferred time zone
#ifdef WORKING_TIME_USE_TZ
	KDateTime kdt = dt.toZone(Preferences::timeZone());
	if ((workOnly  &&  !Preferences::workDays().testBit(kdt.date().dayOfWeek() - 1))
	||  (holidays  &&  Preferences::holidays().isHoliday(kdt.date()))) 
		return false;
	if (!workOnly)
		return true;
	return kdt.isDateOnly()
	   ||  (kdt.time() >= Preferences::workDayStart()  &&  kdt.time() < Preferences::workDayEnd());
#else
	if ((workOnly  &&  !Preferences::workDays().testBit(dt.date().dayOfWeek() - 1))
	||  (holidays  &&  Preferences::holidays().isHoliday(dt.date()))) 
		return false;
	if (!workOnly)
		return true;
	return dt.isDateOnly()
	   ||  (dt.time() >= Preferences::workDayStart()  &&  dt.time() < Preferences::workDayEnd());
#endif
}

/******************************************************************************
*  Return the first day of the week for the user's locale.
*  Reply = 1 (Mon) .. 7 (Sun).
*/
int localeFirstDayOfWeek()
{
	static int firstDay = 0;
	if (!firstDay)
		firstDay = KGlobal::locale()->weekStartDay();
	return firstDay;
}

/******************************************************************************
* Return the week day name (Monday = 1).
*/
QString weekDayName(int day, const KLocale* locale)
{
	switch (day)
	{
		case 1: return locale->translate("Monday");
		case 2: return locale->translate("Tuesday");
		case 3: return locale->translate("Wednesday");
		case 4: return locale->translate("Thursday");
		case 5: return locale->translate("Friday");
		case 6: return locale->translate("Saturday");
		case 7: return locale->translate("Sunday");
	}
	return QString();
}

/******************************************************************************
*  Convert a date/time specification string into a local date/time or date value.
*  Parameters:
*    tzString  = in the form [[[yyyy-]mm-]dd-]hh:mm [TZ] or yyyy-mm-dd [TZ].
*    dateTime  = receives converted date/time value.
*    defaultDt = default date/time used for missing parts of tzString, or null
*                to use current date/time.
*    allowTZ   = whether to allow a time zone specifier in tzString.
*  Reply = true if successful.
*/
bool convTimeString(const QCString& tzString, KDateTime& dateTime, const KDateTime& defaultDt, bool allowTZ)
{
#define MAX_DT_LEN 19
	int i = tzString.find(' ');
	if (i > MAX_DT_LEN  ||  (i >= 0 && !allowTZ))
		return false;
	QString zone = (i >= 0) ? QString::fromLatin1(tzString.mid(i)) : QString::null;
	char timeStr[MAX_DT_LEN+1];
	strcpy(timeStr, tzString.left(i >= 0 ? i : MAX_DT_LEN));
	int dt[5] = { -1, -1, -1, -1, -1 };
	char* s;
	char* end;
	bool noTime;
	// Get the minute value
	if ((s = strchr(timeStr, ':')) == 0)
		noTime = true;
	else
	{
		noTime = false;
		*s++ = 0;
		dt[4] = strtoul(s, &end, 10);
		if (end == s  ||  *end  ||  dt[4] >= 60)
			return false;
		// Get the hour value
		if ((s = strrchr(timeStr, '-')) == 0)
			s = timeStr;
		else
			*s++ = 0;
		dt[3] = strtoul(s, &end, 10);
		if (end == s  ||  *end  ||  dt[3] >= 24)
			return false;
	}
	bool noDate = true;
	if (s != timeStr)
	{
		noDate = false;
		// Get the day value
		if ((s = strrchr(timeStr, '-')) == 0)
			s = timeStr;
		else
			*s++ = 0;
		dt[2] = strtoul(s, &end, 10);
		if (end == s  ||  *end  ||  dt[2] == 0  ||  dt[2] > 31)
			return false;
		if (s != timeStr)
		{
			// Get the month value
			if ((s = strrchr(timeStr, '-')) == 0)
				s = timeStr;
			else
				*s++ = 0;
			dt[1] = strtoul(s, &end, 10);
			if (end == s  ||  *end  ||  dt[1] == 0  ||  dt[1] > 12)
				return false;
			if (s != timeStr)
			{
				// Get the year value
				dt[0] = strtoul(timeStr, &end, 10);
				if (end == timeStr  ||  *end)
					return false;
			}
		}
	}

	QDate date;
	if (dt[0] >= 0)
		date = QDate(dt[0], dt[1], dt[2]);
	QTime time(0, 0, 0);
	if (noTime)
	{
		// No time was specified, so the full date must have been specified
		if (dt[0] < 0  ||  !date.isValid())
			return false;
		dateTime = KAlarm::applyTimeZone(zone, date, time, false, defaultDt);
	}
	else
	{
		// Compile the values into a date/time structure
		time.setHMS(dt[3], dt[4], 0);
		if (dt[0] < 0)
		{
			// Some or all of the date was omitted.
			// Use the default date/time if provided.
			if (defaultDt.isValid())
			{
				dt[0] = defaultDt.date().year();
				date.setYMD(dt[0],
				            (dt[1] < 0 ? defaultDt.date().month() : dt[1]),
				            (dt[2] < 0 ? defaultDt.date().day() : dt[2]));
			}
			else
				date.setYMD(2000, 1, 1);  // temporary substitute for date
		}
		dateTime = KAlarm::applyTimeZone(zone, date, time, true, defaultDt);
		if (!dateTime.isValid())
			return false;
		if (dt[0] < 0)
		{
			// Some or all of the date was omitted.
			// Use the current date in the specified time zone as default.
			KDateTime now = KDateTime::currentDateTime(dateTime.timeSpec());
			date = dateTime.date();
			date.setYMD(now.date().year(),
			            (dt[1] < 0 ? now.date().month() : dt[1]),
			            (dt[2] < 0 ? now.date().day() : dt[2]));
			if (!date.isValid())
				return false;
			if (noDate  &&  time < now.time())
				date = date.addDays(1);
			dateTime.setDate(date);
		}
	}
	return dateTime.isValid();
}

/******************************************************************************
* Convert a time zone specifier string and apply it to a given date and/or time.
* The time zone specifier is a system time zone name, e.g. "Europe/London",
* "UTC" or "Clock". If no time zone is specified, it defaults to the local time
* zone.
* If 'defaultDt' is valid, it supplies the time spec and default date.
*/
KDateTime applyTimeZone(const QString& tzstring, const QDate& date, const QTime& time,
                        bool haveTime, const KDateTime& defaultDt)
{
	bool error = false;
	KDateTime::Spec spec = KDateTime::LocalZone;
	QString zone = tzstring.stripWhiteSpace();
	if (defaultDt.isValid())
	{
		// Time spec is supplied - time zone specifier is not allowed
		if (!zone.isEmpty())
			error = true;
		else
			spec = defaultDt.timeSpec();
	}
	else if (!zone.isEmpty())
	{
		if (zone == QString::fromLatin1("Clock"))
			spec = KDateTime::ClockTime;
		else if (zone == QString::fromLatin1("UTC"))
			spec = KDateTime::UTC;
		else
		{
			KTimeZone tz = KSystemTimeZones::zone(zone);
			error = !tz.isValid();
			if (!error)
				spec = tz;
		}
	}

	KDateTime result;
	if (!error)
	{
		if (!date.isValid())
		{
			// It's a time without a date
			if (defaultDt.isValid())
			       result = KDateTime(defaultDt.date(), time, spec);
			else if (spec == KDateTime::LocalZone  ||  spec == KDateTime::ClockTime)
				result = KDateTime(KDateTime::currentLocalDate(), time, spec);
		}
		else if (haveTime)
		{
			// It's a date and time
			result = KDateTime(date, time, spec);
		}
		else
		{
			// It's a date without a time
			result = KDateTime(date, spec);
		}
	}
	return result;
}

/******************************************************************************
*  Return the supplied string with any accelerator code stripped out.
*/
QString stripAccel(const QString& text)
{
	unsigned len = text.length();
	QString out = QDeepCopy<QString>(text);
	QChar *corig = (QChar*)out.unicode();
	QChar *cout  = corig;
	QChar *cin   = cout;
	while (len)
	{
		if ( *cin == '&' )
		{
			++cin;
			--len;
			if ( !len )
				break;
		}
		*cout = *cin;
		++cout;
		++cin;
		--len;
	}
	unsigned newlen = cout - corig;
	if (newlen != out.length())
		out.truncate(newlen);
	return out;
}

#ifndef NDEBUG
Q_INT64 testTimeOffset = -1;  // test mode offset to apply to current system time

/******************************************************************************
* Set up KAlarm test conditions based on environment variables.
* KALARM_TIME: specifies current system time (format [[[yyyy-]mm-]dd-]hh:mm).
*/
void setTestModeConditions()
{
	testTimeOffset = 0;
	const char* offset = ::getenv("KALARM_TIME");
	if (offset && offset[0])
	{
		KDateTime dt;
		KDateTime now = KDateTime::currentLocalDateTime();
		QTime t = now.time();
		now.setTime(QTime(t.hour(), t.minute(), 0));
		if (convTimeString(offset, dt, now, false))
		{
			testTimeOffset = now.secsTo_long(dt);
			KDateTime::setSystemTimeAdjustment(testTimeOffset);
		}
	}
}
#endif

} // namespace KAlarm


namespace {

/******************************************************************************
*  Tell KOrganizer to put an alarm in its calendar.
*  It will be held by KOrganizer as a simple event, without alarms - KAlarm
*  is still responsible for alarming.
*/
bool sendToKOrganizer(const KAEvent* event)
{
	KCal::Event* kcalEvent = AlarmCalendar::resources()->createKCalEvent(event);
	// Change the event ID to avoid duplicating the same unique ID as the original event
	QString uid = uidKOrganizer(event->id());
	kcalEvent->setUid(uid);
	kcalEvent->clearAlarms();
	QString userEmail;
	switch (event->action())
	{
		case KAEvent::MESSAGE:
		case KAEvent::FILE:
		case KAEvent::COMMAND:
			kcalEvent->setSummary(event->cleanText());
			userEmail = Preferences::emailAddress();
			break;
		case KAEvent::EMAIL:
		{
			QString from = event->emailFromId()
			             ? KAMail::identityManager()->identityForUoid(event->emailFromId()).fullEmailAddr()
			             : Preferences::emailAddress();
			AlarmText atext;
			atext.setEmail(event->emailAddresses(", "), from, QString::null, QString::null, event->emailSubject(), QString::null);
			kcalEvent->setSummary(atext.displayText());
			userEmail = from;
			break;
		}
	}
	kcalEvent->setOrganizer(KCal::Person(QString::null, userEmail));

	// Translate the event into string format
	KCal::ICalFormat format;
	format.setTimeSpec(Preferences::timeZone(true));
	QString iCal = format.toICalString(kcalEvent);
	delete kcalEvent;

	// Send the event to KOrganizer
	if (!runKOrganizer())     // start KOrganizer if it isn't already running
		return false;
	QByteArray  data, replyData;
	QCString    replyType;
	QDataStream arg(data, IO_WriteOnly);
	arg << iCal;
	if (kapp->dcopClient()->call(korganizerName, KORG_DCOP_OBJECT, "addIncidence(QString)", data, replyType, replyData)
	&&  replyType == "bool")
	{
		bool result;
		QDataStream reply(replyData, IO_ReadOnly);
		reply >> result;
		if (result)
		{
			kdDebug(5950) << "sendToKOrganizer(" << uid << "): success\n";
			return true;
		}
	}
	kdError(5950) << "sendToKOrganizer(): KOrganizer addEvent(" << uid << ") dcop call failed\n";
	return false;
}

/******************************************************************************
*  Tell KOrganizer to delete an event from its calendar.
*/
bool deleteFromKOrganizer(const QString& eventID)
{
	if (!runKOrganizer())     // start KOrganizer if it isn't already running
		return false;
	QString newID = uidKOrganizer(eventID);
	QByteArray  data, replyData;
	QCString    replyType;
	QDataStream arg(data, IO_WriteOnly);
	arg << newID << true;
	if (kapp->dcopClient()->call(korganizerName, KORG_DCOP_OBJECT, "deleteIncidence(QString,bool)", data, replyType, replyData)
	&&  replyType == "bool")
	{
		bool result;
		QDataStream reply(replyData, IO_ReadOnly);
		reply >> result;
		if (result)
		{
			kdDebug(5950) << "deleteFromKOrganizer(" << newID << "): success\n";
			return true;
		}
	}
	kdError(5950) << "deleteFromKOrganizer(): KOrganizer deleteEvent(" << newID << ") dcop call failed\n";
	return false;
}

/******************************************************************************
* Insert a KOrganizer string after the hyphen in the supplied event ID.
*/
QString uidKOrganizer(const QString& id)
{
	QString result = id;
	int i = result.findRev('-');
	if (i < 0)
		i = result.length();
	return result.insert(i, KORGANIZER_UID);
}

} // namespace
