/*
 *  packedlayout.cpp  -  layout to pack items into rows
 *  Program:  kalarm
 *  Copyright © 2007 by David Jarvie <software@astrojar.org.uk>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
#include "kalarm.h"

#include <qvaluevector.h>
#include "packedlayout.h"


class PackedLayoutIterator : public QGLayoutIterator
{
	public:
		PackedLayoutIterator(QPtrList<QLayoutItem>* items)
			: index(0), list(items) {}
		uint count() const         { return list->count(); }
		QLayoutItem* current()     { return (index < static_cast<int>(list->count())) ? list->at(index) : 0; }
		QLayoutItem* next()        { ++index;  return current(); }
		QLayoutItem* takeCurrent() { return (index < static_cast<int>(list->count())) ? list->take(index) : 0; }

	private:
		int index;
		QPtrList<QLayoutItem>* list;
};


PackedLayout::PackedLayout(QWidget* parent, Qt::AlignmentFlags alignment, int margin, int spacing, const char* name)
	: QLayout(parent, margin, spacing, name),
	  mAlignment(alignment),
	  mWidthCached(0)
{
	mItems.setAutoDelete(true);
}

PackedLayout::PackedLayout(QLayout* parent, Qt::AlignmentFlags alignment, int spacing, const char* name)
	: QLayout(parent, spacing, name),
	  mAlignment(alignment),
	  mWidthCached(0)
{
	mItems.setAutoDelete(true);
}

PackedLayout::PackedLayout(Qt::AlignmentFlags alignment, int spacing, const char* name)
	: QLayout(spacing, name),
	  mAlignment(alignment),
	  mWidthCached(0)
{
	mItems.setAutoDelete(true);
}

/******************************************************************************
 * Inserts a button into the group.
 */
void PackedLayout::addItem(QLayoutItem* item)
{
	mWidthCached = 0;
	mItems.append(item);
}

QLayoutIterator PackedLayout::iterator()
{
	return QLayoutIterator(new PackedLayoutIterator(&mItems));
}

int PackedLayout::heightForWidth(int w) const
{
	if (w != mWidthCached)
	{
		mHeightCached = arrange(QRect(0, 0, w, 0), false);
		mWidthCached = w;
	}
	return mHeightCached;
}

void PackedLayout::setGeometry(const QRect& rect)
{
	QLayout::setGeometry(rect);
	arrange(rect, true);
}

// Return the minimum size of any widget.
QSize PackedLayout::minimumSize() const
{
	QSize size;
	for (QLayoutItem* item = mItems.first();  item;  item = mItems.next())
		size = size.expandedTo(item->minimumSize());
	int m = margin() * 2;
	return QSize(size.width() + m, size.height() + m);
}

// Arranges widgets and returns height required.
int PackedLayout::arrange(const QRect& rect, bool set) const
{
	int x = rect.x();
	int y = rect.y();
	int yrow = 0;
	int count = 0;
	QValueVector<QLayoutItem*> items(mItems.count());
	QValueVector<QRect> posn(mItems.count());
	for (QLayoutItem* item = mItems.first();  item;  item = mItems.next())
	{
		if (item->isEmpty())
			continue;   // ignore hidden widgets
		QSize size = item->sizeHint();
		int right = x + size.width();
		if (right > rect.right()  &&  x > rect.x())
		{
			x = rect.x();
			y = y + yrow + spacing();
			right = x + size.width();
			yrow = size.height();
		}
		else
			yrow = QMAX(yrow, size.height());
		items[count] = item;
		posn[count++] = QRect(QPoint(x, y), size);
		x = right + spacing();
	}
	if (set)
	{
		if (mAlignment == Qt::AlignLeft)
		{
			// Left aligned: no position adjustment needed
			// Set the positions of all the layout items
			for (int i = 0;  i < count;  ++i)
				items[i]->setGeometry(posn[i]);
		}
		else
		{
			// Set the positions of all the layout items
			for (int i = 0;  i < count; )
			{
				// Adjust item positions a row at a time
				y = posn[i].y();
				int last;   // after last item in this row
				for (last = i + 1;  last < count && posn[last].y() == y;  ++last) ;
				int n = last - i;   // number of items in this row
				int free = rect.right() - posn[last - 1].right();
				switch (mAlignment)
				{
					case Qt::AlignJustify:
						if (n == 1)
						{
							items[i]->setGeometry(posn[i]);
							++i;
						}
						else if (n > 1)
						{
							for (int j = 0;  i < last;  ++j, ++i)
								items.at(i)->setGeometry(QRect(QPoint(posn[i].x() + (free * j)/(n - 1), y), posn[i].size()));
						}
						break;
					case Qt::AlignHCenter:
						free /= 2;
						// fall through to AlignRight
					case Qt::AlignRight:
						for ( ;  i < last;  ++i)
							items.at(i)->setGeometry(QRect(QPoint(posn[i].x() + free, y), posn[i].size()));
						break;
					default:
						break;
				}
			}
		}
	}
	return y + yrow - rect.y();
}
