/*
 *  packedlayout.h  -  layout to pack items into rows
 *  Program:  kalarm
 *  Copyright © 2007 by David Jarvie <software@astrojar.org.uk>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
#ifndef PACKEDLAYOUT_H
#define PACKEDLAYOUT_H

#include <qlayout.h>
#include <qptrlist.h>


/**
 *  The PackedLayout class packs a group of widgets into rows.
 *  The widgets are arranged according to the total width available.
 *
 *  @author David Jarvie <software@astrojar.org.uk>
 */
class PackedLayout : public QLayout
{
	public:
		/** Constructor.
		 *  @param parent the parent widget
		 *  @param alignment how to align the widgets horizontally within the layout
		 *  @param margin margin round its contents
		 *  @param spacing spacing between layout items
		 *  @param name the name of this layout
		 */
		PackedLayout(QWidget* parent, Qt::AlignmentFlags alignment, int margin = 0, int spacing = -1, const char* name = 0);
		/** Constructor.
		 *  @param parent the parent layout
		 *  @param alignment how to align the widgets horizontally within the layout
		 *  @param spacing spacing between layout items
		 *  @param name the name of this layout
		 */
		PackedLayout(QLayout* parent, Qt::AlignmentFlags alignment, int spacing = -1, const char* name = 0);
		/** Constructor.
		 *  @param alignment how to align the widgets horizontally within the layout
		 *  @param spacing spacing between layout items
		 *  @param name the name of this layout
		 */
		explicit PackedLayout(Qt::AlignmentFlags alignment, int spacing = -1, const char* name = 0);
		// Override QLayout methods
		virtual bool hasHeightForWidth() const  { return true; }
		virtual int heightForWidth(int w) const;
		virtual void addItem(QLayoutItem* item);
		virtual void setGeometry(const QRect& r);
		virtual QSize sizeHint() const  { return minimumSize(); }
		virtual QSize minimumSize() const;
		virtual QLayoutIterator iterator();
		virtual QSizePolicy::ExpandData expanding() const  { return QSizePolicy::BothDirections; }
		virtual void invalidate()  { mWidthCached = mHeightCached = false; }

	private:
		int arrange(const QRect&, bool set) const;
		mutable QPtrList<QLayoutItem> mItems;
		Qt::AlignmentFlags mAlignment;
		mutable int mWidthCached;
		mutable int mHeightCached;
};

#endif // PACKEDLAYOUT_H
