/*
 *  stackedwidgets.cpp  -  group of stacked widgets
 *  Program:  kalarm
 *  Copyright © 2008 by David Jarvie <software@astrojar.org.uk>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "stackedwidgets.h"
#include "desktop.h"

#include <kdialogbase.h>
#include <kdebug.h>

#include <qstyle.h>


StackedScrollWidget::StackedScrollWidget(StackedScrollGroup* group, QWidget* parent)
	: StackedWidgetT<QScrollView>(group, parent)
{
	setFrameStyle(QFrame::NoFrame);
	setHScrollBarMode(QScrollView::AlwaysOff);
	setVScrollBarMode(QScrollView::Auto);
	setResizePolicy(QScrollView::AutoOneFit);
	setSizePolicy(QSizePolicy::MinimumExpanding, QSizePolicy::MinimumExpanding);
}

StackedScrollGroup::StackedScrollGroup(KDialogBase* dlg, QObject* tabParent)
	: StackedGroupT<QScrollView>(tabParent),
	  mDialog(dlg),
	  mMinHeight(-1),
	  mHeightReduction(0),
	  mSized(false)
{
}

/******************************************************************************
* Return the minimum size for the tab, adjusted if necessary to a height that
* fits the screen.
* In order to make the QStackedWidget containing the tabs take the correct
* size, the value returned is actually the minimum size of the largest tab.
* Otherwise, only the currently visible tab would be taken into account with
* the result that the dialog would initially be displayed too small.
*/
QSize StackedScrollGroup::minimumSizeHint() const
{
	QSize s = maxMinimumSizeHint();
	if (!s.isEmpty()  &&  mMinHeight > 0  &&  mMinHeight < s.height())
		return QSize(s.width() + mWidgets[0]->style().pixelMetric(QStyle::PM_ScrollBarExtent), mMinHeight);
	return s;
}

/******************************************************************************
* Return the maximum minimum size for any instance.
*/
QSize StackedScrollGroup::maxMinimumSizeHint() const
{
	QSize sz;
	for (QValueList<StackedWidgetT<QScrollView>*>::ConstIterator it = mWidgets.constBegin();  it != mWidgets.constEnd();  ++it)
	{
		QWidget* w = static_cast<StackedScrollWidget*>(*it)->widget();
		if (!w)
			return QSize();
		QSize s = w->minimumSizeHint();
		if (!s.isValid())
			return QSize();
		sz = sz.expandedTo(s);
	}
	return sz;
}

/******************************************************************************
* Return the minimum size for the dialog.
* If the minimum size would be too high to fit the desktop, the tab contents
* are made scrollable.
*/
QSize StackedScrollGroup::adjustSize(bool force)
{
	if (force)
		mSized = false;
	if (mSized)
		return QSize();
	// Cancel any previous minimum height
	mMinHeight = -1;
	mHeightReduction = 0;

	QSize s = maxMinimumSizeHint();
	if (s.isEmpty())
		return QSize();
	int maxTabHeight = s.height();
	for (QValueList<StackedWidgetT<QScrollView>*>::ConstIterator it = mWidgets.constBegin();  it != mWidgets.constEnd();  ++it)
	{
		(*it)->setMinimumHeight(0);
		QWidget* w = static_cast<StackedScrollWidget*>(*it)->widget();
		if (w)
			w->resize(s);
	}
	for (QWidget* w = mWidgets[0]->parentWidget();  w && w != mDialog;  w = w->parentWidget())
		w->adjustSize();
	int decoration = mDialog->frameGeometry().height() - mDialog->geometry().height();
	if (!decoration)
	{
		// On X11 at least, the window decoration height may not be
		// available, so use a guess of 25 pixels.
		decoration = 25;
	}
	int desk = KAlarm::desktopWorkArea().height();
	// There is no stored size, or the deferral group is visible.
	// Allow the tab contents to be scrolled vertically if that is necessary
	// to avoid the dialog exceeding the screen height.
	s = mDialog->KDialog::sizeHint();
	int y = s.height() + decoration - desk;
	if (y > 0)
	{
		mHeightReduction = y;
		mMinHeight = maxTabHeight - y;
		kdDebug(5950) << "Scrolling: max tab height=" << maxTabHeight << ", reduction=" << mHeightReduction << "-> min tab height=" << mMinHeight << endl;
		if (mMinHeight > 0)
		{
			for (QValueList<StackedWidgetT<QScrollView>*>::ConstIterator it = mWidgets.constBegin();  it != mWidgets.constEnd();  ++it)
			{
				(*it)->setMinimumHeight(mMinHeight);
				(*it)->resize(QSize((*it)->width(), mMinHeight));
			}
		}
		mSized = true;
		for (QWidget* w = mWidgets[0]->parentWidget();  w && w != mDialog;  w = w->parentWidget())
			w->adjustSize();
		mDialog->adjustSize();
		s = mDialog->KDialogBase::minimumSizeHint();
		if (mDialog->sizeHint().height() < s.height())
			s.setHeight(mDialog->sizeHint().height());
	}
	if (!mHeightReduction)
		mSized = true;
	mDialog->resize(s);
	return s;
}
