/*
 *  mainwindow.cpp  -  main application window
 *  Program:  kalarm
 *  Copyright © 2001-2008 by David Jarvie <software@astrojar.org.uk>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "kalarm.h"

#include <qiconset.h>
#include <qdragobject.h>
#include <qheader.h>
#include <qsplitter.h>

#include <kmenubar.h>
#include <ktoolbar.h>
#include <kpopupmenu.h>
#include <kaccel.h>
#include <kaction.h>
#include <kactionclasses.h>
#include <kstdaction.h>
#include <kiconloader.h>
#include <kmessagebox.h>
#include <kinputdialog.h>
#include <kurldrag.h>
#include <klocale.h>
#include <kglobalsettings.h>
#include <kconfig.h>
#include <kkeydialog.h>
#include <kedittoolbar.h>
#include <kaboutdata.h>
#include <dcopclient.h>
#include <kdebug.h>

#include <libkdepim/maillistdrag.h>
#include <libkmime/kmime_content.h>
#include <libkcal/calendarlocal.h>
#include <libkcal/icaldrag.h>

#ifdef LIKEBACK
#include <likeback/likeback.h>
#endif

#include "alarmcalendar.h"
#include "alarmevent.h"
#include "alarmlistview.h"
#include "alarmresources.h"
#include "alarmtext.h"
#include "birthdaydlg.h"
#include "editdlg.h"
#include "functions.h"
#include "kalarmapp.h"
#include "kamail.h"
#include "newalarmaction.h"
#include "prefdlg.h"
#include "preferences.h"
#include "resourceselector.h"
#include "synchtimer.h"
#include "templatepickdlg.h"
#include "templatedlg.h"
#include "templatemenuaction.h"
#include "traywindow.h"
#include "mainwindow.moc"

using namespace KCal;

static const char* UI_FILE     = "kalarmui.rc";
static const char* WINDOW_NAME = "MainWindow";

static const QString VIEW_GROUP         = QString::fromLatin1("View");
static const QString SHOW_TIME_KEY      = QString::fromLatin1("ShowAlarmTime");
static const QString SHOW_TIME_TO_KEY   = QString::fromLatin1("ShowTimeToAlarm");
static const QString SHOW_ARCHIVED_KEY  = QString::fromLatin1("ShowArchivedAlarms");
static const QString SHOW_RESOURCES_KEY = QString::fromLatin1("ShowResources");

static QString   undoText;
static QString   undoTextStripped;
static QString   undoIcon;
static KShortcut undoShortcut;
static QString   redoText;
static QString   redoTextStripped;
static QString   redoIcon;
static KShortcut redoShortcut;


/*=============================================================================
=  Class: MainWindow
=============================================================================*/

MainWindow::WindowList   MainWindow::mWindowList;
TemplateDlg*             MainWindow::mTemplateDlg = 0;

// Collect these widget labels together to ensure consistent wording and
// translations across different modules.
QString MainWindow::i18n_a_ShowAlarmTimes()     { return i18n("Show &Alarm Times"); }
QString MainWindow::i18n_m_ShowAlarmTime()      { return i18n("Show alarm ti&me"); }
QString MainWindow::i18n_o_ShowTimeToAlarms()   { return i18n("Show Time t&o Alarms"); }
QString MainWindow::i18n_l_ShowTimeToAlarm()    { return i18n("Show time unti&l alarm"); }
QString MainWindow::i18n_ShowArchivedAlarms()   { return i18n("Show Archived Alarms"); }
QString MainWindow::i18n_e_ShowArchivedAlarms() { return i18n("Show &Archived Alarms"); }
QString MainWindow::i18n_HideArchivedAlarms()   { return i18n("Hide Archived Alarms"); }
QString MainWindow::i18n_e_HideArchivedAlarms() { return i18n("Hide &Archived Alarms"); }
QString MainWindow::i18n_r_ShowResources()      { return i18n("Show &Resources");}

/******************************************************************************
* Construct an instance.
* To avoid resize() events occurring while still opening the calendar (and
* resultant crashes), the calendar is opened before constructing the instance.
*/
MainWindow* MainWindow::create(bool restored)
{
	theApp()->checkCalendar();    // ensure calendar is open
	return new MainWindow(restored);
}

MainWindow::MainWindow(bool restored)
	: MainWindowBase(0, "MainWin", WGroupLeader | WStyle_ContextHelp | WDestructiveClose),
	  mResourcesWidth(-1),
	  mMinuteTimerActive(false),
	  mHiddenTrayParent(false)
{
	kdDebug(5950) << "MainWindow::MainWindow()\n";
	setAutoSaveSettings(QString::fromLatin1(WINDOW_NAME));    // save window sizes etc.
	setPlainCaption(kapp->aboutData()->programName());
	KConfig* config = KGlobal::config();
	config->setGroup(VIEW_GROUP);
	mShowResources = config->readBoolEntry(SHOW_RESOURCES_KEY, false);
	mShowArchived  = config->readBoolEntry(SHOW_ARCHIVED_KEY, false);
	mShowTime      = config->readBoolEntry(SHOW_TIME_KEY, true);
	mShowTimeTo    = config->readBoolEntry(SHOW_TIME_TO_KEY, false);
	if (!restored)
	{
		QSize s;
		if (KAlarm::readConfigWindowSize(WINDOW_NAME, s, &mResourcesWidth))
			resize(s);
	}
	config->setGroup(QString::fromLatin1(WINDOW_NAME));
	QValueList<int> order = config->readIntListEntry(QString::fromLatin1("ColumnOrder"));

	setAcceptDrops(true);         // allow drag-and-drop onto this window
	if (!mShowTimeTo)
		mShowTime = true;     // ensure at least one time column is visible

	mSplitter = new QSplitter(QSplitter::Horizontal, this, "splitter");
	mSplitter->setChildrenCollapsible(false);
	setCentralWidget(mSplitter);

	// Create the calendar resource selector widget
	AlarmResources* resources = AlarmResources::instance();
	mResourceSelector = new ResourceSelector(resources, mSplitter);
	mSplitter->setResizeMode(mResourceSelector, QSplitter::KeepSize);
	connect(resources, SIGNAL(signalErrorMessage(const QString&)), SLOT(showErrorMessage(const QString&)));

	// Create the alarm list widget
	mListView = new AlarmListView(order, mSplitter, "listView");
	mListView->selectTimeColumns(mShowTime, mShowTimeTo);
	mListView->showArchived(mShowArchived);
	mListView->refresh();          // populate the alarm list
	mListView->clearSelection();

	connect(mListView, SIGNAL(itemDeleted()), SLOT(slotDeletion()));
	connect(mListView, SIGNAL(selectionChanged()), SLOT(slotSelection()));
	connect(mListView, SIGNAL(contextMenuRequested(QListViewItem*, const QPoint&, int)),
	                   SLOT(slotContextMenuRequested(QListViewItem*, const QPoint&, int)));
	connect(mListView, SIGNAL(mouseButtonClicked(int, QListViewItem*, const QPoint&, int)),
	                   SLOT(slotMouseClicked(int, QListViewItem*, const QPoint&, int)));
	connect(mListView, SIGNAL(executed(QListViewItem*)), SLOT(slotDoubleClicked(QListViewItem*)));
	connect(mListView->header(), SIGNAL(indexChange(int, int, int)), SLOT(columnsReordered()));
	connect(mResourceSelector, SIGNAL(resized(const QSize&, const QSize&)), SLOT(resourcesResized()));
	connect(resources, SIGNAL(resourceStatusChanged(AlarmResource*, AlarmResources::Change)),
	                   SLOT(slotResourceStatusChanged()));

	// Create the menu, toolbar, etc.
	initActions();

	mWindowList.append(this);
	if (mWindowList.count() == 1)
	{
		// It's the first main window
		if (theApp()->wantShowInSystemTray())
			theApp()->displayTrayIcon(true, this);     // create system tray icon for run-in-system-tray mode
		else if (theApp()->trayWindow())
			theApp()->trayWindow()->setAssocMainWindow(this);    // associate this window with the system tray icon
	}
	setUpdateTimer();
	slotResourceStatusChanged();   // initialise action states now that window is registered
}

MainWindow::~MainWindow()
{
	kdDebug(5950) << "MainWindow::~MainWindow()\n";
	bool trayParent = isTrayParent();   // must call before removing from window list
	mWindowList.remove(this);
	if (theApp()->trayWindow())
	{
		if (trayParent)
			delete theApp()->trayWindow();
		else
			theApp()->trayWindow()->removeWindow(this);
	}
	MinuteTimer::disconnect(this);
	mMinuteTimerActive = false;    // to ensure that setUpdateTimer() works correctly
	setUpdateTimer();
	MainWindow* main = mainMainWindow();
	if (main)
		KAlarm::writeConfigWindowSize(WINDOW_NAME, main->size(), mResourcesWidth);
	KToolBar* tb = toolBar();
	if (tb)
		tb->saveSettings(KGlobal::config(), "Toolbars");
	KGlobal::config()->sync();    // save any new window size to disc
	theApp()->quitIf();
}

/******************************************************************************
* Save settings to the session managed config file, for restoration
* when the program is restored.
*/
void MainWindow::saveProperties(KConfig* config)
{
	config->writeEntry(QString::fromLatin1("HiddenTrayParent"), isTrayParent() && isHidden());
	config->writeEntry(QString::fromLatin1("ShowArchived"), mShowArchived);
	config->writeEntry(QString::fromLatin1("ShowTime"), mShowTime);
	config->writeEntry(QString::fromLatin1("ShowTimeTo"), mShowTimeTo);
	config->writeEntry(QString::fromLatin1("ResourcesWidth"), mResourceSelector->isHidden() ? 0 : mResourceSelector->width());
}

/******************************************************************************
* Read settings from the session managed config file.
* This function is automatically called whenever the app is being
* restored. Read in whatever was saved in saveProperties().
*/
void MainWindow::readProperties(KConfig* config)
{
	mHiddenTrayParent = config->readBoolEntry(QString::fromLatin1("HiddenTrayParent"));
	mShowArchived     = config->readBoolEntry(QString::fromLatin1("ShowArchived"));
	mShowTime         = config->readBoolEntry(QString::fromLatin1("ShowTime"));
	mShowTimeTo       = config->readBoolEntry(QString::fromLatin1("ShowTimeTo"));
	mResourcesWidth   = config->readNumEntry(QString::fromLatin1("ResourcesWidth"));
	mShowResources    = (mResourcesWidth > 0);
}

/******************************************************************************
* Get the main main window, i.e. the parent of the system tray icon, or if
* none, the first main window to be created. Visible windows take precedence
* over hidden ones.
*/
MainWindow* MainWindow::mainMainWindow()
{
	MainWindow* tray = theApp()->trayWindow() ? theApp()->trayWindow()->assocMainWindow() : 0;
	if (tray  &&  tray->isVisible())
		return tray;
	for (WindowList::Iterator it = mWindowList.begin();  it != mWindowList.end();  ++it)
		if ((*it)->isVisible())
			return *it;
	if (tray)
		return tray;
	if (mWindowList.isEmpty())
		return 0;
	return mWindowList.first();
}

/******************************************************************************
* Check whether this main window is effectively the parent of the system tray icon.
*/
bool MainWindow::isTrayParent() const
{
	TrayWindow* tray = theApp()->trayWindow();
	if (!tray  ||  !theApp()->haveSystemTray())
		return false;
	if (tray->assocMainWindow() == this)
		return true;
	return mWindowList.count() == 1;
}

/******************************************************************************
*  Close all main windows.
*/
void MainWindow::closeAll()
{
	while (!mWindowList.isEmpty())
		delete mWindowList.first();    // N.B. the destructor removes the window from the list
}

/******************************************************************************
*  Called when the window's size has changed (before it is painted).
*  Sets the last column in the list view to extend at least to the right hand
*  edge of the list view.
*  Records the new size in the config file.
*/
void MainWindow::resizeEvent(QResizeEvent* re)
{
	// Save the window's new size only if it's the first main window
	if (mainMainWindow() == this)
		KAlarm::writeConfigWindowSize(WINDOW_NAME, re->size(), mResourcesWidth);
	MainWindowBase::resizeEvent(re);
}

void MainWindow::resourcesResized()
{
	QValueList<int> widths = mSplitter->sizes();
	if (widths.count() > 1)
	{
		mResourcesWidth = widths.first();
		// Width is reported as non-zero when resource selector is
		// actually invisible, so note a zero width in these circumstances.
		if (mResourcesWidth <= 5)
			mResourcesWidth = 0;
	}
}

/******************************************************************************
*  Called when the window is first displayed.
*  Sets the last column in the list view to extend at least to the right hand
*  edge of the list view.
*/
void MainWindow::showEvent(QShowEvent* se)
{
	if (mResourcesWidth > 0)
	{
		QValueList<int> widths;
		widths.append(mResourcesWidth);
		mSplitter->setSizes(widths);
	}
	setUpdateTimer();
	slotUpdateTimeTo();
	MainWindowBase::showEvent(se);
}

/******************************************************************************
*  Display the window.
*/
void MainWindow::show()
{
	MainWindowBase::show();
	if (mMenuError)
	{
		// Show error message now that the main window has been displayed.
		// Waiting until now lets the user easily associate the message with
		// the main window which is faulty.
		KMessageBox::error(this, i18n("Failure to create menus\n(perhaps %1 missing or corrupted)").arg(QString::fromLatin1(UI_FILE)));
		mMenuError = false;
	}
}

/******************************************************************************
*  Called after the window is hidden.
*/
void MainWindow::hideEvent(QHideEvent* he)
{
	setUpdateTimer();
	MainWindowBase::hideEvent(he);
}

/******************************************************************************
*  Called when the list's column order is changed.
*  Save the new column order as the default the next time the program is run.
*/
void MainWindow::columnsReordered()
{
	KConfig* config = KGlobal::config();
	config->setGroup(QString::fromLatin1(WINDOW_NAME));
	config->writeEntry(QString::fromLatin1("ColumnOrder"), mListView->columnOrder());
	config->sync();
}

/******************************************************************************
*  Initialise the menu, toolbar and main window actions.
*/
void MainWindow::initActions()
{
	KActionCollection* actions = actionCollection();
	mActionTemplates       = new KAction(i18n("&Templates..."), 0, this, SLOT(slotTemplates()), actions, "templates");
	mActionNew             = new NewAlarmAction(false, i18n("&New"), actions, "new");
	connect(mActionNew, SIGNAL(selected(EditAlarmDlg::Type)), this, SLOT(slotNew(EditAlarmDlg::Type)));
	mActionNewDisplay      = NewAlarmAction::newDisplayAlarmAction(actions, this, SLOT(slotNewDisplay()));
	mActionNewCommand      = NewAlarmAction::newCommandAlarmAction(actions, this, SLOT(slotNewCommand()));
	mActionNewEmail        = NewAlarmAction::newEmailAlarmAction(actions, this, SLOT(slotNewEmail()));
	mActionNewFromTemplate = KAlarm::createNewFromTemplateAction(i18n("New &From Template"), this, SLOT(slotNewFromTemplate(const KAEvent*)), actions, "newFromTempl");
	mActionCreateTemplate  = new KAction(i18n("Create Tem&plate..."), 0, this, SLOT(slotNewTemplate()), actions, "createTemplate");
	mActionCopy            = new KAction(i18n("&Copy..."), "editcopy", Qt::SHIFT+Qt::Key_Insert, this, SLOT(slotCopy()), actions, "copy");
	mActionModify          = new KAction(i18n("&Edit..."), "edit", Qt::CTRL+Qt::Key_E, this, SLOT(slotModify()), actions, "modify");
	mActionDelete          = new KAction(i18n("&Delete"), "editdelete", Qt::Key_Delete, this, SLOT(slotDelete()), actions, "delete");
	mActionReactivate      = new KAction(i18n("Reac&tivate"), 0, Qt::CTRL+Qt::Key_R, this, SLOT(slotReactivate()), actions, "undelete");
	mActionEnable          = new KAction(QString::null, 0, Qt::CTRL+Qt::Key_B, this, SLOT(slotEnable()), actions, "disable");
	mActionShowTime        = new KToggleAction(i18n_a_ShowAlarmTimes(), 0, this, SLOT(slotShowTime()), actions, "showAlarmTimes");
	mActionShowTime->setCheckedState(i18n("Hide &Alarm Times"));
	mActionShowTimeTo      = new KToggleAction(i18n_o_ShowTimeToAlarms(), Qt::CTRL+Qt::Key_I, this, SLOT(slotShowTimeTo()), actions, "showTimeToAlarms");
	mActionShowTimeTo->setCheckedState(i18n("Hide Time t&o Alarms"));
	mActionShowArchived    = new KToggleAction(i18n_e_ShowArchivedAlarms(), "history", Qt::CTRL+Qt::Key_P, this, SLOT(slotShowArchived()), actions, "showArchivedAlarms");
	mActionShowArchived->setCheckedState(i18n_e_HideArchivedAlarms());
	mActionToggleTrayIcon  = new KToggleAction(i18n("Show in System &Tray"), 0, this, SLOT(slotToggleTrayIcon()), actions, "showInSystemTray");
	mActionToggleTrayIcon->setCheckedState(i18n("Hide From System &Tray"));
	mActionToggleResourceSel = new KToggleAction(i18n_r_ShowResources(), "view_choose", 0, this, SLOT(slotToggleResourceSelector()), actions, "showResources");
	mActionToggleResourceSel->setCheckedState(i18n("Hide &Resources"));
	mActionImportAlarms    = new KAction(i18n("Import &Alarms..."), 0, this, SLOT(slotImportAlarms()), actions, "importAlarms");
	mActionImportBirthdays = new KAction(i18n("Import &Birthdays..."), 0, this, SLOT(slotBirthdays()), actions, "importBirthdays");
	new KAction(i18n("&Refresh Alarms"), "reload", 0, this, SLOT(slotRefreshAlarms()), actions, "refreshAlarms");
	KAlarm::createAlarmEnableAction(actions, "alarmsEnable");
	if (undoText.isNull())
	{
		// Get standard texts, etc., for Undo and Redo actions
		KAction* act = KStdAction::undo(this, 0, actions);
		undoIcon         = act->icon();
		undoShortcut     = act->shortcut();
		undoText         = act->text();
		undoTextStripped = KAlarm::stripAccel(undoText);
		delete act;
		act = KStdAction::redo(this, 0, actions);
		redoIcon         = act->icon();
		redoShortcut     = act->shortcut();
		redoText         = act->text();
		redoTextStripped = KAlarm::stripAccel(redoText);
		delete act;
	}
	mActionUndo = new KToolBarPopupAction(undoText, undoIcon, undoShortcut, this, SLOT(slotUndo()), actions, "edit_undo");
	mActionRedo = new KToolBarPopupAction(redoText, redoIcon, redoShortcut, this, SLOT(slotRedo()), actions, "edit_redo");
	KStdAction::find(mListView, SLOT(slotFind()), actions);
	mActionFindNext = KStdAction::findNext(mListView, SLOT(slotFindNext()), actions);
	mActionFindPrev = KStdAction::findPrev(mListView, SLOT(slotFindPrev()), actions);
	KStdAction::selectAll(mListView, SLOT(slotSelectAll()), actions);
	KStdAction::deselect(mListView, SLOT(slotDeselect()), actions);
	KStdAction::quit(this, SLOT(slotQuit()), actions);
	KStdAction::keyBindings(this, SLOT(slotConfigureKeys()), actions);
	KStdAction::configureToolbars(this, SLOT(slotConfigureToolbar()), actions);
	KStdAction::preferences(this, SLOT(slotPreferences()), actions);
#ifdef LIKEBACK
	theApp()->likeBack()->sendACommentAction(actions);
#endif
	mResourceSelector->initActions(actions);
	setStandardToolBarMenuEnabled(true);
	createGUI(UI_FILE);

	mContextMenu = static_cast<KPopupMenu*>(factory()->container("listContext", this));
	mActionsMenu = static_cast<KPopupMenu*>(factory()->container("actions", this));
	KPopupMenu* resourceMenu = static_cast<KPopupMenu*>(factory()->container("resourceContext", this));
	mResourceSelector->setContextMenu(resourceMenu);
	mMenuError = (!mContextMenu  ||  !mActionsMenu  ||  !resourceMenu);
	connect(mActionUndo->popupMenu(), SIGNAL(aboutToShow()), SLOT(slotInitUndoMenu()));
	connect(mActionUndo->popupMenu(), SIGNAL(activated(int)), SLOT(slotUndoItem(int)));
	connect(mActionRedo->popupMenu(), SIGNAL(aboutToShow()), SLOT(slotInitRedoMenu()));
	connect(mActionRedo->popupMenu(), SIGNAL(activated(int)), SLOT(slotRedoItem(int)));
	connect(Undo::instance(), SIGNAL(changed(const QString&, const QString&)), SLOT(slotUndoStatus(const QString&, const QString&)));
	connect(mListView, SIGNAL(findActive(bool)), SLOT(slotFindActive(bool)));
	Preferences::connect(SIGNAL(preferencesChanged()), this, SLOT(slotPrefsChanged()));
	connect(theApp(), SIGNAL(trayIconToggled()), SLOT(updateTrayIconAction()));

	// Set menu item states
	setEnableText(true);
	mActionShowTime->setChecked(mShowTime);
	mActionShowTimeTo->setChecked(mShowTimeTo);
	mActionShowArchived->setChecked(mShowArchived);
	mActionToggleResourceSel->setChecked(mShowResources);
	slotToggleResourceSelector();
	slotPrefsChanged();         // set the correct text for this action
	mActionUndo->setEnabled(Undo::haveUndo());
	mActionRedo->setEnabled(Undo::haveRedo());
	mActionFindNext->setEnabled(false);
	mActionFindPrev->setEnabled(false);

	mActionCopy->setEnabled(false);
	mActionModify->setEnabled(false);
	mActionDelete->setEnabled(false);
	mActionReactivate->setEnabled(false);
	mActionEnable->setEnabled(false);
	mActionCreateTemplate->setEnabled(false);

	KToolBar* tb = toolBar();
	if (tb)
		tb->applySettings(KGlobal::config(), "Toolbars");

	Undo::emitChanged();     // set the Undo/Redo menu texts
//	Daemon::monitoringAlarms();
}

/******************************************************************************
* Enable or disable the Templates menu item in every main window instance.
*/
void MainWindow::enableTemplateMenuItem(bool enable)
{
	for (WindowList::Iterator it = mWindowList.begin();  it != mWindowList.end();  ++it)
		(*it)->mActionTemplates->setEnabled(enable);
}

/******************************************************************************
* Refresh the alarm list in every main window instance which is displaying
* archived alarms.
* Called when an archived alarm setting changes in the user preferences, or
* when archive alarms have been purged.
*/
void MainWindow::updateArchived()
{
	kdDebug(5950) << "MainWindow::updateArchived()\n";
	int days = Preferences::archivedKeepDays();
	for (WindowList::Iterator it = mWindowList.begin();  it != mWindowList.end();  ++it)
	{
		MainWindow* w = *it;
		if (w->mShowArchived)
		{
			if (!days)
				w->slotShowArchived();
			else
				w->mListView->updateArchived();
		}
		w->mActionShowArchived->setEnabled(days);
		w->mResourceSelector->archiveDaysChanged(days);
	}
}

/******************************************************************************
* Start or stop the timer which updates the time-to-alarm values every minute.
* Should be called whenever a main window is created or destroyed, or shown or
* hidden.
*/
void MainWindow::setUpdateTimer()
{
	// Check whether any windows need to be updated
	MainWindow* needTimer = 0;
	MainWindow* timerWindow = 0;
	for (WindowList::Iterator it = mWindowList.begin();  it != mWindowList.end();  ++it)
	{
		MainWindow* w = *it;
		if (w->isVisible()  &&  !w->isMinimized()  &&  w->mListView->showingTimeTo())
			needTimer = w;
		if (w->mMinuteTimerActive)
			timerWindow = w;
	}

	// Start or stop the update timer if necessary
	if (needTimer  &&  !timerWindow)
	{
		// Timeout every minute.
		needTimer->mMinuteTimerActive = true;
		MinuteTimer::connect(needTimer, SLOT(slotUpdateTimeTo()));
		kdDebug(5950) << "MainWindow::setUpdateTimer(): started timer" << endl;
	}
	else if (!needTimer  &&  timerWindow)
	{
		timerWindow->mMinuteTimerActive = false;
		MinuteTimer::disconnect(timerWindow);
		kdDebug(5950) << "MainWindow::setUpdateTimer(): stopped timer" << endl;
	}
}
/******************************************************************************
* Update the time-to-alarm values for each main window which is displaying them.
*/
void MainWindow::slotUpdateTimeTo()
{
	kdDebug(5950) << "MainWindow::slotUpdateTimeTo()" << endl;
	for (WindowList::Iterator it = mWindowList.begin();  it != mWindowList.end();  ++it)
	{
		MainWindow* w = *it;
		if (w->isVisible()  &&  w->mListView->showingTimeTo())
			w->mListView->updateTimeToAlarms();
	}
}

/******************************************************************************
* Select an alarm in the displayed list.
*/
void MainWindow::selectEvent(const QString& eventID)
{
	mListView->clearSelection();
	AlarmListViewItem* item = mListView->getEntry(eventID);
	if (item)
	{
		mListView->setSelected(item, true);
		mListView->setCurrentItem(item);
		mListView->ensureItemVisible(item);
	}
}

/******************************************************************************
*  Called when the New button is clicked to edit a new alarm to add to the list.
*/
void MainWindow::slotNew(EditAlarmDlg::Type type)
{
	KAlarm::editNewAlarm(type, this, mListView);
}

/******************************************************************************
*  Called when a template is selected from the New From Template popup menu.
*  Executes a New Alarm dialog, preset from the selected template.
*/
void MainWindow::slotNewFromTemplate(const KAEvent* tmplate)
{
	KAlarm::editNewAlarm(tmplate, this, mListView);
}

/******************************************************************************
*  Called when the New Template button is clicked to create a new template
*  based on the currently selected alarm.
*/
void MainWindow::slotNewTemplate()
{
	AlarmListViewItem* item = mListView->selectedItem();
	if (item)
		KAlarm::editNewTemplate(item->event(), this);
}

/******************************************************************************
*  Called when the Copy button is clicked to edit a copy of an existing alarm,
*  to add to the list.
*/
void MainWindow::slotCopy()
{
	AlarmListViewItem* item = mListView->selectedItem();
	if (item)
		KAlarm::editNewAlarm(item->event(), this, mListView);
}

/******************************************************************************
*  Called when the Modify button is clicked to edit the currently highlighted
*  alarm in the list.
*/
void MainWindow::slotModify()
{
	AlarmListViewItem* item = mListView->selectedItem();
	if (item)
		KAlarm::editAlarm(item->event(), this, mListView);
}

/******************************************************************************
*  Called when the Delete button is clicked to delete the currently highlighted
*  alarms in the list.
*/
void MainWindow::slotDelete()
{
	KAEvent::List events = mListView->selectedEvents();
	// Copy the events to be deleted, in case any are deleted by being
	// triggered while the confirmation prompt is displayed.
	KAEvent::List eventCopies;
	Undo::EventList undos;
	AlarmCalendar* resources = AlarmCalendar::resources();
	for (KAEvent::List::ConstIterator it = events.constBegin();  it != events.constEnd();  ++it)
	{
		KAEvent* event = *it;
		eventCopies.append(event);
		undos.append(*event, resources->resourceForEvent(event->id()));
	}
	if (Preferences::confirmAlarmDeletion())
	{
		int n = events.count();
		if (KMessageBox::warningContinueCancel(this, i18n("Do you really want to delete the selected alarm?",
		                                                  "Do you really want to delete the %n selected alarms?", n),
		                                       i18n("Delete Alarm", "Delete Alarms", n),
		                                       KGuiItem(i18n("&Delete"), "editdelete"),
		                                       Preferences::CONFIRM_ALARM_DELETION)
		    != KMessageBox::Continue)
			return;
	}

	// Delete the events from the calendar and displays
	KAlarm::deleteEvents(eventCopies, true, this);
	Undo::saveDeletes(undos);
}

/******************************************************************************
*  Called when the Reactivate button is clicked to reinstate the currently
*  highlighted archived alarms in the list.
*/
void MainWindow::slotReactivate()
{
	KAEvent::List events = mListView->selectedEvents();
	mListView->clearSelection();

	// Add the alarms to the displayed lists and to the calendar file
	Undo::EventList undos;
	QStringList ineligibleIDs;
	KAlarm::reactivateEvents(events, ineligibleIDs, mListView, 0, this);

	// Create the undo list, excluding ineligible events
	AlarmCalendar* resources = AlarmCalendar::resources();
	for (KAEvent::List::ConstIterator itev = events.constBegin();  itev != events.constEnd();  ++itev)
	{
		QString id = (*itev)->id();
		if (ineligibleIDs.find(id) == ineligibleIDs.end())
			undos.append(**itev, resources->resourceForEvent(id));
	}
	Undo::saveReactivates(undos);
}

/******************************************************************************
*  Called when the Enable/Disable button is clicked to enable or disable the
*  currently highlighted alarms in the list.
*/
void MainWindow::slotEnable()
{
	bool enable = mActionEnableEnable;    // save since changed in response to KAlarm::enableEvent()
	KAEvent::List events = mListView->selectedEvents();
	KAEvent::List eventCopies;
	for (KAEvent::List::ConstIterator it = events.constBegin();  it != events.constEnd();  ++it)
		eventCopies += *it;
	KAlarm::enableEvents(eventCopies, mListView, enable, this);
}

/******************************************************************************
*  Called when the Show Alarm Times menu item is selected or deselected.
*/
void MainWindow::slotShowTime()
{
	mShowTime = !mShowTime;
	mActionShowTime->setChecked(mShowTime);
	if (!mShowTime  &&  !mShowTimeTo)
		slotShowTimeTo();    // at least one time column must be displayed
	else
	{
		mListView->selectTimeColumns(mShowTime, mShowTimeTo);
		KConfig* config = KGlobal::config();
		config->setGroup(VIEW_GROUP);
		config->writeEntry(SHOW_TIME_KEY, mShowTime);
		config->writeEntry(SHOW_TIME_TO_KEY, mShowTimeTo);
	}
}

/******************************************************************************
*  Called when the Show Time To Alarms menu item is selected or deselected.
*/
void MainWindow::slotShowTimeTo()
{
	mShowTimeTo = !mShowTimeTo;
	mActionShowTimeTo->setChecked(mShowTimeTo);
	if (!mShowTimeTo  &&  !mShowTime)
		slotShowTime();    // at least one time column must be displayed
	else
	{
		mListView->selectTimeColumns(mShowTime, mShowTimeTo);
		KConfig* config = KGlobal::config();
		config->setGroup(VIEW_GROUP);
		config->writeEntry(SHOW_TIME_KEY, mShowTime);
		config->writeEntry(SHOW_TIME_TO_KEY, mShowTimeTo);
	}
	setUpdateTimer();
}

/******************************************************************************
*  Called when the Show Archived Alarms menu item is selected or deselected.
*/
void MainWindow::slotShowArchived()
{
	mShowArchived = !mShowArchived;
	mActionShowArchived->setChecked(mShowArchived);
	mActionShowArchived->setToolTip(mShowArchived ? i18n_HideArchivedAlarms() : i18n_ShowArchivedAlarms());
	mListView->showArchived(mShowArchived);
	mListView->updateArchived();
	KConfig* config = KGlobal::config();
	config->setGroup(VIEW_GROUP);
	config->writeEntry(SHOW_ARCHIVED_KEY, mShowArchived);
}

/******************************************************************************
*  Called when the Import Alarms menu item is selected, to merge alarms from an
*  external calendar into the current calendars.
*/
void MainWindow::slotImportAlarms()
{
	AlarmCalendar::importAlarms(this);
}

/******************************************************************************
*  Called when the Import Birthdays menu item is selected, to display birthdays
*  from the address book for selection as alarms.
*/
void MainWindow::slotBirthdays()
{
	BirthdayDlg dlg(this);
	if (dlg.exec() == QDialog::Accepted)
	{
		QValueList<KAEvent> events = dlg.events();
		if (!events.isEmpty())
		{
			mListView->clearSelection();
			// Add alarm to the displayed lists and to the calendar file
			KAlarm::UpdateStatus status = KAlarm::addEvents(events, mListView, &dlg, true, true);

			Undo::EventList undos;
			AlarmCalendar* resources = AlarmCalendar::resources();
			for (QValueList<KAEvent>::ConstIterator it = events.constBegin();  it != events.constEnd();  ++it)
				undos.append(*it, resources->resourceForEvent((*it).id()));
			Undo::saveAdds(undos, i18n("Import birthdays"));

			if (status != KAlarm::UPDATE_FAILED)
				KAlarm::outputAlarmWarnings(&dlg);
		}
	}
}

/******************************************************************************
*  Called when the Templates menu item is selected, to display the alarm
*  template editing dialogue.
*/
void MainWindow::slotTemplates()
{
	if (!mTemplateDlg)
	{
		mTemplateDlg = TemplateDlg::create(this);
		enableTemplateMenuItem(false);     // disable menu item in all windows
		connect(mTemplateDlg, SIGNAL(finished()), SLOT(slotTemplatesEnd()));
		mTemplateDlg->show();
	}
}

/******************************************************************************
*  Called when the alarm template editing dialogue has exited.
*/
void MainWindow::slotTemplatesEnd()
{
	if (mTemplateDlg)
	{
		mTemplateDlg->delayedDestruct();   // this deletes the dialogue once it is safe to do so
		mTemplateDlg = 0;
		enableTemplateMenuItem(true);      // re-enable menu item in all windows
	}
}

/******************************************************************************
*  Called when the Display System Tray Icon menu item is selected.
*/
void MainWindow::slotToggleTrayIcon()
{
	theApp()->displayTrayIcon(!theApp()->trayIconDisplayed(), this);
}

/******************************************************************************
*  Called when the Show Resource Selector menu item is selected.
*/
void MainWindow::slotToggleResourceSelector()
{
	mShowResources = mActionToggleResourceSel->isChecked();
	if (mShowResources)
	{
		if (mResourcesWidth <= 0)
		{
			mResourcesWidth = mResourceSelector->sizeHint().width();
			mResourceSelector->resize(mResourcesWidth, mResourceSelector->height());
			QValueList<int> widths = mSplitter->sizes();
			if (widths.count() == 1)
			{
				int listwidth = widths.first() - mSplitter->handleWidth() - mResourcesWidth;
				mListView->resize(listwidth, mListView->height());
				widths.append(listwidth);
				widths.first() = mResourcesWidth;
			}
			mSplitter->setSizes(widths);
		}
		mResourceSelector->show();
	}
	else
		mResourceSelector->hide();

	KConfig* config = KGlobal::config();
	config->setGroup(VIEW_GROUP);
	config->writeEntry(SHOW_RESOURCES_KEY, mShowResources);
}

/******************************************************************************
* Called when an error occurs in the resource calendar, to display a message.
*/
void MainWindow::showErrorMessage(const QString& msg)
{
	KMessageBox::error(this, msg);
}

/******************************************************************************
* Called when the user preferences have changed.
*/
void MainWindow::slotPrefsChanged()
{
	mActionShowArchived->setEnabled(Preferences::archivedKeepDays());
	updateTrayIconAction();
}

/******************************************************************************
* Called when the system tray icon is created or destroyed.
* Set the system tray icon menu text according to whether or not the system
* tray icon is currently visible.
*/
void MainWindow::updateTrayIconAction()
{
	mActionToggleTrayIcon->setEnabled(theApp()->haveSystemTray());
	mActionToggleTrayIcon->setChecked(theApp()->trayIconDisplayed());
}

/******************************************************************************
*  Called when the active status of Find changes.
*/
void MainWindow::slotFindActive(bool active)
{
	mActionFindNext->setEnabled(active);
	mActionFindPrev->setEnabled(active);
}

/******************************************************************************
*  Called when the Undo action is selected.
*/
void MainWindow::slotUndo()
{
	Undo::undo(this, KAlarm::stripAccel(mActionUndo->text()));
}

/******************************************************************************
*  Called when the Redo action is selected.
*/
void MainWindow::slotRedo()
{
	Undo::redo(this, KAlarm::stripAccel(mActionRedo->text()));
}

/******************************************************************************
*  Called when an Undo item is selected.
*/
void MainWindow::slotUndoItem(int id)
{
	Undo::undo(id, this, Undo::actionText(Undo::UNDO, id));
}

/******************************************************************************
*  Called when a Redo item is selected.
*/
void MainWindow::slotRedoItem(int id)
{
	Undo::redo(id, this, Undo::actionText(Undo::REDO, id));
}

/******************************************************************************
*  Called when the Undo menu is about to show.
*  Populates the menu.
*/
void MainWindow::slotInitUndoMenu()
{
	initUndoMenu(mActionUndo->popupMenu(), Undo::UNDO);
}

/******************************************************************************
*  Called when the Redo menu is about to show.
*  Populates the menu.
*/
void MainWindow::slotInitRedoMenu()
{
	initUndoMenu(mActionRedo->popupMenu(), Undo::REDO);
}

/******************************************************************************
*  Populate the undo or redo menu.
*/
void MainWindow::initUndoMenu(KPopupMenu* menu, Undo::Type type)
{
	menu->clear();
	const QString& action = (type == Undo::UNDO) ? undoTextStripped : redoTextStripped;
	QValueList<int> ids = Undo::ids(type);
	for (QValueList<int>::ConstIterator it = ids.constBegin();  it != ids.constEnd();  ++it)
	{
		int id = *it;
		QString actText = Undo::actionText(type, id);
		QString descrip = Undo::description(type, id);
		QString text = descrip.isEmpty()
		             ? i18n("Undo/Redo [action]", "%1 %2").arg(action).arg(actText)
		             : i18n("Undo [action]: message", "%1 %2: %3").arg(action).arg(actText).arg(descrip);
		menu->insertItem(text, id);
	}
}

/******************************************************************************
*  Called when the status of the Undo or Redo list changes.
*  Change the Undo or Redo text to include the action which would be undone/redone.
*/
void MainWindow::slotUndoStatus(const QString& undo, const QString& redo)
{
	if (undo.isNull())
	{
		mActionUndo->setEnabled(false);
		mActionUndo->setText(undoText);
	}
	else
	{
		mActionUndo->setEnabled(true);
		mActionUndo->setText(QString("%1 %2").arg(undoText).arg(undo));
	}
	if (redo.isNull())
	{
		mActionRedo->setEnabled(false);
		mActionRedo->setText(redoText);
	}
	else
	{
		mActionRedo->setEnabled(true);
		mActionRedo->setText(QString("%1 %2").arg(redoText).arg(redo));
	}
}

/******************************************************************************
*  Called when the Refresh Alarms menu item is selected.
*/
void MainWindow::slotRefreshAlarms()
{
	KAlarm::refreshAlarms();
}

/******************************************************************************
*  Called when the "Configure KAlarm" menu item is selected.
*/
void MainWindow::slotPreferences()
{
	KAlarmPrefDlg::display();
}

/******************************************************************************
*  Called when the Configure Keys menu item is selected.
*/
void MainWindow::slotConfigureKeys()
{
	KKeyDialog::configure(actionCollection(), this);
}

/******************************************************************************
*  Called when the Configure Toolbars menu item is selected.
*/
void MainWindow::slotConfigureToolbar()
{
	saveMainWindowSettings(KGlobal::config(), WINDOW_NAME);
	KEditToolbar dlg(factory());
	connect(&dlg, SIGNAL(newToolbarConfig()), this, SLOT(slotNewToolbarConfig()));
	dlg.exec();
}

/******************************************************************************
*  Called when OK or Apply is clicked in the Configure Toolbars dialog, to save
*  the new configuration.
*/
void MainWindow::slotNewToolbarConfig()
{
	createGUI(UI_FILE);
	applyMainWindowSettings(KGlobal::config(), WINDOW_NAME);
}

/******************************************************************************
*  Called when the Quit menu item is selected.
*/
void MainWindow::slotQuit()
{
	theApp()->doQuit(this);
}

/******************************************************************************
*  Called when the user or the session manager attempts to close the window.
*/
void MainWindow::closeEvent(QCloseEvent* ce)
{
	if (!theApp()->sessionClosingDown())
	{
		// The user (not the session manager) wants to close the window.
		if (isTrayParent())
		{
			// It's the parent window of the system tray icon, so just hide
			// it to prevent the system tray icon closing.
			hide();
			theApp()->quitIf();
			ce->ignore();
			return;
		}
		else if (!theApp()->trayIconDisplayed()  &&  count() == 1)
		{
			// There is no system tray icon, and this is the last main
			// window, so closing it will quit the application.
			theApp()->doQuit(this);
			ce->ignore();
			return;
		}
	}
	ce->accept();
}

/******************************************************************************
*  Called when an item is deleted from the ListView.
*  Disables the actions if no item is still selected.
*/
void MainWindow::slotDeletion()
{
	if (!mListView->selectedCount())
	{
		kdDebug(5950) << "MainWindow::slotDeletion(true)\n";
		selectionCleared();    // disable actions
	}
}

/******************************************************************************
*  Called when the drag cursor enters the window.
*/
void MainWindow::dragEnterEvent(QDragEnterEvent* e)
{
	executeDragEnterEvent(e);
}

/******************************************************************************
*  Called when the drag cursor enters a main or system tray window, to accept
*  or reject the dragged object.
*/
void MainWindow::executeDragEnterEvent(QDragEnterEvent* e)
{
	if (KCal::ICalDrag::canDecode(e))
		e->accept(!AlarmListView::dragging());   // don't accept "text/calendar" objects from KAlarm
	else
		e->accept(QTextDrag::canDecode(e)
		       || KURLDrag::canDecode(e)
		       || KPIM::MailListDrag::canDecode(e));
}

/******************************************************************************
*  Called when an object is dropped on the window.
*  If the object is recognised, the edit alarm dialog is opened appropriately.
*/
void MainWindow::dropEvent(QDropEvent* e)
{
	executeDropEvent(this, e);
}

static QString getMailHeader(const char* header, KMime::Content& content)
{
	KMime::Headers::Base* hd = content.getHeaderByType(header);
	return hd ? hd->asUnicodeString() : QString::null;
}

/******************************************************************************
*  Called when an object is dropped on a main or system tray window, to
*  evaluate the action required and extract the text.
*/
void MainWindow::executeDropEvent(MainWindow* win, QDropEvent* e)
{
	KAEvent::Action action = KAEvent::MESSAGE;
	QString        text;
	QByteArray     bytes;
	AlarmText      alarmText;
	KPIM::MailList mailList;
	KURL::List     files;
	KCal::CalendarLocal calendar(Preferences::timeZone(true));
#ifndef NDEBUG
	QCString fmts;
	for (int idbg = 0;  e->format(idbg);  ++idbg)
	{
		if (idbg) fmts += ", ";
		fmts += e->format(idbg);
	}
	kdDebug(5950) << "MainWindow::executeDropEvent(): " << fmts << endl;
#endif

	/* The order of the tests below matters, since some dropped objects
	 * provide more than one mime type.
	 * Don't change them without careful thought !!
	 */
	if (e->provides("message/rfc822")
	&&  !(bytes = e->encodedData("message/rfc822")).isEmpty())
	{
		// Email message(s). Ignore all but the first.
		kdDebug(5950) << "MainWindow::executeDropEvent(email)" << endl;
		QCString mails(bytes.data(), bytes.size());
		KMime::Content content;
		content.setContent(mails);
		content.parse();
		QString body;
		if (content.textContent())
			content.textContent()->decodedText(body, true, true);    // strip trailing newlines & spaces
		unsigned long sernum = 0;
		if (e->provides(KPIM::MailListDrag::format())
		&&  KPIM::MailListDrag::decode(e, mailList)
		&&  mailList.count())
		{
			// Get its KMail serial number to allow the KMail message
			// to be called up from the alarm message window.
			sernum = mailList.first().serialNumber();
		}
		alarmText.setEmail(getMailHeader("To", content),
		                   getMailHeader("From", content),
		                   getMailHeader("Cc", content),
		                   getMailHeader("Date", content),
		                   getMailHeader("Subject", content),
				   body, sernum);
	}
	else if (KURLDrag::decode(e, files)  &&  files.count())
	{
		kdDebug(5950) << "MainWindow::executeDropEvent(URL)" << endl;
		action = KAEvent::FILE;
		alarmText.setText(files.first().prettyURL());
	}
	else if (e->provides(KPIM::MailListDrag::format())
	&&       KPIM::MailListDrag::decode(e, mailList))
	{
		// KMail message(s). Ignore all but the first.
		kdDebug(5950) << "MainWindow::executeDropEvent(KMail_list)" << endl;
		if (!mailList.count())
			return;
		KPIM::MailSummary& summary = mailList.first();
		QDateTime dt;
		dt.setTime_t(summary.date());
		QString body = KAMail::getMailBody(summary.serialNumber());
		alarmText.setEmail(summary.to(), summary.from(), QString::null,
		                   KGlobal::locale()->formatDateTime(dt), summary.subject(),
		                   body, summary.serialNumber());
	}
	else if (KCal::ICalDrag::decode(e, &calendar))
	{
		// iCalendar - ignore all but the first event
		kdDebug(5950) << "MainWindow::executeDropEvent(iCalendar)" << endl;
		KCal::Event::List events = calendar.rawEvents();
		if (!events.isEmpty())
		{
			KAEvent ev(events.first());
			KAlarm::editNewAlarm(&ev, win, (win ? win->mListView : 0));
		}
		return;
	}
	else if (QTextDrag::decode(e, text))
	{
		kdDebug(5950) << "MainWindow::executeDropEvent(text)" << endl;
		alarmText.setText(text);
	}
	else
		return;

	if (!alarmText.isEmpty())
	{
		if (action == KAEvent::MESSAGE
		&&  (alarmText.isEmail() || alarmText.isScript()))
		{
			// If the alarm text could be interpreted as an email or command script,
			// prompt for which type of alarm to create.
			QStringList types;
			types += i18n("Display Alarm");
			if (alarmText.isEmail())
				types += i18n("Email Alarm");
			else if (alarmText.isScript())
				types += i18n("Command Alarm");
			bool ok = false;
			QString type = KInputDialog::getItem(i18n("Alarm Type"),
			                                     i18n("Choose alarm type to create:"), types, 0, false, &ok, mainMainWindow());
			if (!ok)
				return;   // user didn't press OK
			int i = types.findIndex(type);
			if (i == 1)
				action = alarmText.isEmail() ? KAEvent::EMAIL : KAEvent::COMMAND;
		}
		KAlarm::editNewAlarm(action, win, (win ? win->mListView : 0), &alarmText);
	}
}

/******************************************************************************
* Called when the status of a resource has changed.
* Enable or disable actions appropriately.
*/
void MainWindow::slotResourceStatusChanged()
{
	// Find whether there are any writable resources
	AlarmResources* resources = AlarmResources::instance();
	bool active  = resources->activeCount(AlarmResource::ACTIVE, true);
	bool templat = resources->activeCount(AlarmResource::TEMPLATE, true);
	for (WindowList::Iterator it = mWindowList.begin();  it != mWindowList.end();  ++it)
	{
		MainWindow* w = *it;
		w->mActionImportAlarms->setEnabled(active || templat);
		w->mActionImportBirthdays->setEnabled(active);
		w->mActionNew->setEnabled(active);
		w->mActionNewFromTemplate->setEnabled(active);
		w->mActionCreateTemplate->setEnabled(templat);
		w->slotSelection();
	}
}

/******************************************************************************
*  Called when the selected items in the ListView change.
*  Enables the actions appropriately.
*/
void MainWindow::slotSelection()
{
	// Find which items have been selected
	QValueList<EventListViewItemBase*> items = mListView->selectedItems();
	int count = items.count();
	if (!count)
	{
		selectionCleared();    // disable actions
		return;
	}

	// Find whether there are any writable resources
	bool active = mActionNew->isEnabled();

	bool readOnly = false;
	bool allArchived = true;
	bool enableReactivate = true;
	bool enableEnableDisable = true;
	bool enableEnable = false;
	bool enableDisable = false;
	AlarmCalendar* resources = AlarmCalendar::resources();
	KDateTime now = KDateTime::currentUtcDateTime();
	for (QValueList<EventListViewItemBase*>::Iterator it = items.begin();  it != items.end();  ++it)
	{
		KAEvent* event = ((AlarmListViewItem*)(*it))->event();
		bool expired = event->expired();
		if (!expired)
			allArchived = false;
		if (resources->eventReadOnly(event->id()))
			readOnly = true;
		if (enableReactivate
		&&  (!expired  ||  !event->occursAfter(now, true)))
			enableReactivate = false;
		if (enableEnableDisable)
		{
			if (expired)
				enableEnableDisable = enableEnable = enableDisable = false;
			else
			{
				if (!enableEnable  &&  !event->enabled())
					enableEnable = true;
				if (!enableDisable  &&  event->enabled())
					enableDisable = true;
			}
		}
	}

	kdDebug(5950) << "MainWindow::slotSelection(true)\n";
	mActionCreateTemplate->setEnabled((count == 1) && (AlarmResources::instance()->activeCount(AlarmResource::TEMPLATE, true) > 0));
	mActionCopy->setEnabled(active && count == 1);
	mActionModify->setEnabled(count == 1);
	mActionDelete->setEnabled(!readOnly && (active || allArchived));
	mActionReactivate->setEnabled(active && enableReactivate);
	mActionEnable->setEnabled(active && !readOnly && (enableEnable || enableDisable));
	if (enableEnable || enableDisable)
		setEnableText(enableEnable);
}

/******************************************************************************
* Called when a context menu is requested either by a mouse click or by a
* key press.
*/
void MainWindow::slotContextMenuRequested(QListViewItem*, const QPoint& pt, int)
{
	kdDebug(5950) << "MainWindow::slotContextMenuRequested()" << endl;
	if (mContextMenu)
		mContextMenu->popup(pt);
}

/******************************************************************************
* Called when the mouse is clicked on the ListView.
* Deselects the current item and disables the actions if appropriate.
* Note that if a right button click is handled by slotContextMenuRequested().
*/
void MainWindow::slotMouseClicked(int button, QListViewItem* item, const QPoint&, int)
{
	if (button != Qt::RightButton  &&  !item)
	{
		kdDebug(5950) << "MainWindow::slotMouseClicked(left)" << endl;
		mListView->clearSelection();
		selectionCleared();    // disable actions
	}
}

/******************************************************************************
*  Disables actions when no item is selected.
*/
void MainWindow::selectionCleared()
{
	mActionCreateTemplate->setEnabled(false);
	mActionCopy->setEnabled(false);
	mActionModify->setEnabled(false);
	mActionDelete->setEnabled(false);
	mActionReactivate->setEnabled(false);
	mActionEnable->setEnabled(false);
}

/******************************************************************************
*  Called when the mouse is double clicked on the ListView.
*  Displays the Edit Alarm dialog, for the clicked item.
*/
void MainWindow::slotDoubleClicked(QListViewItem* item)
{
	kdDebug(5950) << "MainWindow::slotDoubleClicked()\n";
	if (item)
	{
		if (mActionModify->isEnabled())
			slotModify();
	}
}

/******************************************************************************
*  Set the text of the Enable/Disable menu action.
*/
void MainWindow::setEnableText(bool enable)
{
	mActionEnableEnable = enable;
	mActionEnable->setText(enable ? i18n("Ena&ble") : i18n("Disa&ble"));
}

/******************************************************************************
*  Display or hide the specified main window.
*  This should only be called when the application doesn't run in the system tray.
*/
MainWindow* MainWindow::toggleWindow(MainWindow* win)
{
	if (win  &&  mWindowList.find(win) != mWindowList.end())
	{
		// A window is specified (and it exists)
		if (win->isVisible())
		{
			// The window is visible, so close it
			win->close();
			return 0;
		}
		else
		{
			// The window is hidden, so display it
			win->hide();        // in case it's on a different desktop
			win->showNormal();
			win->raise();
			win->setActiveWindow();
			return win;
		}
	}

	// No window is specified, or the window doesn't exist. Open a new one.
	win = create();
	win->show();
	return win;
}
