/*
 *  mainwindow.h  -  main application window
 *  Program:  kalarm
 *  Copyright © 2001-2008 by David Jarvie <software@astrojar.org.uk>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#ifndef MAINWINDOW_H
#define MAINWINDOW_H

/** @file mainwindow.h - main application window */

#include <libkcal/calendar.h>

#include "alarmevent.h"
#include "alarmresources.h"
#include "alarmtext.h"
#include "editdlg.h"
#include "mainwindowbase.h"
#include "undo.h"

class QListViewItem;
class QSplitter;
class KAction;
class KToggleAction;
class KToolBarPopupAction;
class KPopupMenu;
class ActionAlarmsEnabled;
class AlarmListView;
class TemplateDlg;
class TemplateMenuAction;
class ResourceSelector;
class AlarmResources;


class MainWindow : public MainWindowBase, public KCal::Calendar::Observer
{
		Q_OBJECT

	public:
		static MainWindow* create(bool restored = false);
		~MainWindow();
		bool               isTrayParent() const;
		bool               isHiddenTrayParent() const   { return mHiddenTrayParent; }
		bool               showingArchived() const      { return mShowArchived; }
		void               selectEvent(const QString& eventID);

		static void        updateArchived();
		static void        addEvent(const KAEvent&, MainWindow*);
		static void        executeDragEnterEvent(QDragEnterEvent*);
		static void        executeDropEvent(MainWindow*, QDropEvent*);
		static void        closeAll();
		static MainWindow* toggleWindow(MainWindow*);
		static MainWindow* mainMainWindow();
		static MainWindow* firstWindow()      { return mWindowList.first(); }
		static int         count()            { return mWindowList.count(); }

		static QString i18n_a_ShowAlarmTimes();     // text of 'Show Alarm Times' checkbox, with 'A' shortcut
		static QString i18n_m_ShowAlarmTime();      // text of 'Show alarm time' checkbox, with 'M' shortcut
		static QString i18n_o_ShowTimeToAlarms();   // text of 'Show Time to Alarms' checkbox, with 'O' shortcut
		static QString i18n_l_ShowTimeToAlarm();    // text of 'Show time until alarm' checkbox, with 'L' shortcut
		static QString i18n_ShowArchivedAlarms();   // plain text of 'Show Archived Alarms' action
		static QString i18n_e_ShowArchivedAlarms(); // text of 'Show Archived Alarms' checkbox, with 'E' shortcut
		static QString i18n_HideArchivedAlarms();   // plain text of 'Hide Archived Alarms' action
		static QString i18n_e_HideArchivedAlarms(); // text of 'Hide Archived Alarms' action, with 'E' shortcut
		static QString i18n_r_ShowResources();      // text of 'Show Resources' action, with 'R' shortcut

	public slots:
		virtual void   show();

	protected:
		virtual void   resizeEvent(QResizeEvent*);
		virtual void   showEvent(QShowEvent*);
		virtual void   hideEvent(QHideEvent*);
		virtual void   closeEvent(QCloseEvent*);
		virtual void   dragEnterEvent(QDragEnterEvent*);
		virtual void   dropEvent(QDropEvent*);
		virtual void   saveProperties(KConfig*);
		virtual void   readProperties(KConfig*);

	private slots:
		void           slotNew(EditAlarmDlg::Type);
		void           slotNewFromTemplate(const KAEvent*);
		void           slotNewTemplate();
		void           slotNewDisplay()   { slotNew(EditAlarmDlg::DISPLAY); }
		void           slotNewCommand()   { slotNew(EditAlarmDlg::COMMAND); }
		void           slotNewEmail()     { slotNew(EditAlarmDlg::EMAIL); }
		void           slotCopy();
		void           slotModify();
		void           slotDelete();
		void           slotReactivate();
		void           slotEnable();
		void           slotToggleTrayIcon();
		void           slotRefreshAlarms();
		void           slotImportAlarms();
		void           slotBirthdays();
		void           slotTemplates();
		void           slotTemplatesEnd();
		void           slotPreferences();
		void           slotConfigureKeys();
		void           slotConfigureToolbar();
		void           slotNewToolbarConfig();
		void           slotQuit();
		void           slotDeletion();
		void           slotSelection();
		void           slotContextMenuRequested(QListViewItem*, const QPoint&, int);
		void           slotMouseClicked(int button, QListViewItem*, const QPoint&, int);
		void           slotDoubleClicked(QListViewItem*);
		void           slotShowTime();
		void           slotShowTimeTo();
		void           slotShowArchived();
		void           slotUpdateTimeTo();
		void           slotUndo();
		void           slotUndoItem(int id);
		void           slotRedo();
		void           slotRedoItem(int id);
		void           slotInitUndoMenu();
		void           slotInitRedoMenu();
		void           slotUndoStatus(const QString&, const QString&);
		void           slotFindActive(bool);
		void           slotPrefsChanged();
		void           updateTrayIconAction();
		void           columnsReordered();
		void           slotToggleResourceSelector();
		void           slotResourceStatusChanged();
		void           resourcesResized();
		void           showErrorMessage(const QString&);

	private:
		typedef QValueList<MainWindow*> WindowList;

		MainWindow(bool restored);
		void           createListView(bool recreate);
		void           initActions();
		void           initCalendarResources();
		void           selectionCleared();
		void           setEnableText(bool enable);
		static KAEvent::Action  getDropAction(QDropEvent*, QString& text);
		static void    initUndoMenu(KPopupMenu*, Undo::Type);
		static void    setUpdateTimer();
		static void    enableTemplateMenuItem(bool);

		static WindowList    mWindowList;   // active main windows
		static TemplateDlg*  mTemplateDlg;  // the one and only template dialogue

		AlarmListView*       mListView;
		ResourceSelector*    mResourceSelector;    // resource selector widget
		QSplitter*           mSplitter;            // splits window into list and resource selector
		AlarmResources*      mAlarmResources;      // calendar resources to use for this window
		KToggleAction*       mActionToggleResourceSel;
		KAction*             mActionImportAlarms;
		KAction*             mActionImportBirthdays;
		KAction*             mActionTemplates;
		KAction*             mActionNew;
		KAction*             mActionNewDisplay;
		KAction*             mActionNewCommand;
		KAction*             mActionNewEmail;
		TemplateMenuAction*  mActionNewFromTemplate;
		KAction*             mActionCreateTemplate;
		KAction*             mActionCopy;
		KAction*             mActionModify;
		KAction*             mActionDelete;
		KAction*             mActionReactivate;
		KAction*             mActionEnable;
		KAction*             mActionFindNext;
		KAction*             mActionFindPrev;
		KToolBarPopupAction* mActionUndo;
		KToolBarPopupAction* mActionRedo;
		KToggleAction*       mActionToggleTrayIcon;
		KToggleAction*       mActionShowTime;
		KToggleAction*       mActionShowTimeTo;
		KToggleAction*       mActionShowArchived;
		KPopupMenu*          mActionsMenu;
		KPopupMenu*          mContextMenu;
		int                  mResourcesWidth;      // width of resource selector widget
		bool                 mMinuteTimerActive;   // minute timer is active
		bool                 mHiddenTrayParent;    // on session restoration, hide this window
		bool                 mShowResources;       // show resource selector
		bool                 mShowArchived;        // include archived alarms in the displayed list
		bool                 mShowTime;            // show alarm times
		bool                 mShowTimeTo;          // show time-to-alarms
		bool                 mActionEnableEnable;  // Enable/Disable action is set to "Enable"
		bool                 mMenuError;           // error occurred creating menus: need to show error message
};

#endif // MAINWINDOW_H

